
;;===========================================================================
;; Deduce for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; This file contains all the structure definitions used in the NEITHER 2.0
;; code. It also contains any special routines used to update these
;; structures. 
;;
;; CHANGE HISTORY
;;
;; 23-DEC-92: (ptb) added compile-time conditionals to make the code work for
;;            common lisp version 2.0 (cltl2).
;; 03-JAN-93: (ptb) added a new structure to represent the candidate 
;;            revisions considered by abduce and retract-rules.
;; 05-JAN-93: (ptb) added a new field to the candidate structure to allow 
;;            retraction (and abduction possibly in the future) return info
;;            regarding how the threshold for the rule should change.
;; 03-MAR-93: (ptb) added number elements to the rule structure for automatic
;;            numbering of rules (see debug.lisp)
;; 12-MAR-93: (ptb) added new structures for student models. Updated rule 
;;            structure to include value for rule consequent.
;; 16-MAR-93: (ptb) expanded the vertex structure to optimize abduction and
;;            rule retraction.
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; GLOBAL VARIABLE REFERENCES
;;
;; The code here references three global variables which it assumes are
;; defined elsewhere in the code.
;;===========================================================================

#+:cltl2(declaim (special *neither-theory*)
                 (special *student-model*))
#-:cltl2(progn
          (proclaim '(special *neither-theory*))    ;; in io.lisp
          (proclaim '(special *student-model*))     ;; in model.lisp
          )


;;===========================================================================
;; NEITHER TREE STRUCTUES
;;===========================================================================

(defstruct vertex
  ;;-------------------------------------------------------------------------
  ;; The vertex forms the basis of the tree structure which is used to hold
  ;; the rules of the domain. The overall theory is a list of "category"
  ;; trees. Each category tree is a vertex which is the head of a tree that
  ;; holds the rules associated with proving an example in the given
  ;; category. Thus, each vertex forms the head of a sub tree. Each field of
  ;; a vertex is described below.
  ;;
  ;; (1) "prop" means "proposition" because we using propositional
  ;; logic. Thus the "prop" is the consequent of the rule or, if no rule,
  ;; just the assertion of the vertex (eg operational propositions). For
  ;; operational vertices, the prop field is the name of the feature (the
  ;; prop field of the antecedent structure has the particular instance of
  ;; the feature with its value).
  ;;
  ;; (2) Each example is converted to a vector (array) and the
  ;; "example-index" here refers to a location in the array where the value
  ;; for the proposition associated with the vertex can be found.
  ;;
  ;; (3) Children is a list of rules (see rule structure below). For a vertex
  ;; to be considered "satisfied", any ONE of its children must be satisfied.
  ;; 
  ;; (4) "no-rules?" is used to indicate an intermediate concept that
  ;; currently has no rules defined that support it. Such concepts are
  ;; different from concepts which are operational and are satisfied by
  ;; values stored in the examples (and also have no children).
  ;;
  ;; (ptb 10-29-92) Added two elements to the structure to make deduction a
  ;; a linear function (ie, linear in the size of the theory). The idea is to
  ;; mark each vertex as it is proved, thereby avoiding reproving. When a
  ;; vertex is proved, its value is stored here as well, so that the result
  ;; of the proof for the vertex is available witout reproof.
  ;;
  ;; 16-MAR-93 (ptb) Added two fields, idential in function to the Dvisited
  ;; and Dvalue fields used to optimize deduction. The new fields, Fvisited
  ;; and Fvalue, are used to make fix computation faster (where a "fix" is
  ;; either a set of abductions or a set of retractions). The reason for 
  ;; have two completely analagous sets of fields here arises from the fact
  ;; that the fix computations call deduction. Without separate fields, the
  ;; two operations collide.
  ;;-------------------------------------------------------------------------
  prop             ; property name
  example-index    ; index into example vector with object properties
  (children ())    ; list of children (rules)
  no-rules?        ; t if vertex is an antecedent but not a consequent and
		   ; not operational, ie, an undefined intermediate.
  (Dvisited 0)     ; used to keep from reproving same vertex in deduction
  Dvalue           ; holds value of vertex in deduction
  (Fvisited 0)     ; same as Dvisited but for abduction and retraction
  Fvalue           ; same as Dvalue but for abduction and retraction
  )


(defstruct rule
  ;;-------------------------------------------------------------------------
  ;; A rule structure is a component of a vertex (see above) that indicates
  ;; the antecedents that make up one of the rules defining the consequent
  ;; stored in the vertex structure. In other words, since the same
  ;; consequent can be defined by different rules, the consequent of a rule
  ;; is stored in the vertex structure, and the rest of the rule is stored
  ;; here. A rule consists of three components:
  ;;
  ;; (1) used flag. This flag indicates whether or not the rule is in use in
  ;; the current theory. By setting this flag to nil, the rule is effectively
  ;; deleted from the theory.
  ;;
  ;; (2) threshold. Threshold is an integer that indicates how many of the
  ;; antecedents of the rule (see antecedent structure below) can FAIL and 
  ;; have the rule still be considered satisfied.
  ;;
  ;; (3) antecedents. A list of the antecedents that make up the rule.
  ;;
  ;; PTB 3-2-93: I added a "number" field so I could have NEITHER number the
  ;; rules automatically in the same order that it prints them out. I use
  ;; the number later in a routine called "rules-satisfied" (see deduce.lisp)
  ;; to print out the rules which an example satisfies. The "number-theory"
  ;; routine is in "debug.lisp".
  ;;
  ;; PTB 3-12-93: added a consequent field to store the consequent of the 
  ;; rule. Strictly speaking, this is redundant information since the parent
  ;; vertex has the consequent, but it was the most painless way to deal with
  ;; saving a whole rule associated with a revision for the student model. 
  ;; The other option was to set a pointer back to the parent vertex 
  ;; structure, but keeping all the pointers set correctly could get to be a
  ;; nightmare.
  ;;-------------------------------------------------------------------------
  used
  threshold
  consequent
  (antecedents ())
  (number -1)
  )


(defstruct antecedent
  ;;-------------------------------------------------------------------------
  ;; A structure holding an antecedent for a rule.
  ;;
  ;; The need for this structure arose from the existence of undefined
  ;; intermediates. Originally, the code treated these the same as
  ;; operational concepts which was wrong. Each antecedent structure now
  ;; points to a vertex (which may be shared among several different
  ;; antecedents). Thus, vertices which are roots of rules may be deleted
  ;; from different rules by changing flags *here* in this structure, leaving
  ;; the shared information in the vertex intact. The fields of the
  ;; antecedent are explained as follows:
  ;;
  ;; (1) The "prop" field holds the particular usage of the vertex
  ;; proposition in the given rule. Thus, a feature like "(birth live)" would
  ;; store "birth" as the prop field of the vertex and "(birth live)" as the
  ;; value of the prop field here. Singleton propositions are stored the same
  ;; way here as they are for vertices.
  ;; (2) When an antecedent is abduced, (see abduce.lisp) the "abduced?" flag
  ;; is set to "t". Normally, it defaults to "nil" meaning that the
  ;; antecedent must be satisfied for the rule to be satisfied.
  ;; (3) The "vertex" field has a pointer to the associated vertex.
  ;;-------------------------------------------------------------------------
  prop       ;; holds the use of the proposition as it appears in the rule.
  abduced?   ;; nil as default. Set to "t" when antecedent is abduced.
  vertex     ;; pointer to the vertex associated with the prop.
  )


(defstruct example
  ;;-------------------------------------------------------------------------
  ;; Example structures hold the category for which the example should be
  ;; provable (name) and a feature vector of values for the example (vectors)
  ;; now stored as a simple vector (1-d Lisp array). Note that at this point
  ;; the "name" can only have a single value, ie, the example can only prove
  ;; one category.
  ;;-------------------------------------------------------------------------
  name
  values)


;;===========================================================================
;; NEITHER REVISION STRUCTUES
;;===========================================================================

(defstruct deletion
  ;;-------------------------------------------------------------------------
  ;; A deletion is used during generalization and specialization to determine
  ;; candidate revisions to the theory to enable it to prove an example that
  ;; should be proved or to avoid proving an example which should not be
  ;; proved (see comments in generalize.lisp and specialize.lisp). Each
  ;; deletion is of the form 
  ;;
  ;;    deletion = (parent
  ;;                child
  ;;                antes      ;; ie "(ante-1 ante-2 ... ante-n)"
  ;;                )
  ;;
  ;; For generalization, these fields are interpreted as follows. "parent" is
  ;; the vertex to be changed, "child" is a rule structure, representing
  ;; which of the children of the parent vertex is modified in the abduction,
  ;; and "antes" is a list of the antecedent structures which should be
  ;; removed from the child to make the unprovable positive provable. There
  ;; may be any number of rules involved, and each may be linked with any
  ;; number of antecedents to be abduced. Note that each abduction is
  ;; INDEPENDENT. Thus, we could have several abductions like 
  ;;
  ;;      ((parent-x child-2 ante-3 ante-17)
  ;;       (parent-x child-2 ante-1 ante-3 ante-4) ... )
  ;;
  ;; which shows overlapping abductions from the same child of the same rule
  ;; (ie, each removes the third antecedent of the second child of parent-x).
  ;;
  ;; For specialization, these fields are interpreted as follows. "parent" is
  ;; still the vertex to be changed, but now "child" represents a definition
  ;; of the parent (ie, one of the rules defining the consequent stored in
  ;; the parent) which should be deleted from the theory. The "antes" field
  ;; is not used. Note that multiple rule deletions from the same parent
  ;; vertex would be listed separately.
  ;;
  ;; 27-OCT-92 (ptb): Added a fourth element to this structure to hold the
  ;; change in the child rule's threshold value that would be necessary to
  ;; generalize/specialize the rule. For example, if three antecedents were
  ;; returned by abduction (ie, all three needed to be removed to make the
  ;; rule provable for a given example) and the threshold for the rule were
  ;; currently at 1, the threshold-delta value would be set to 3-1=2. For 
  ;; specialization, if the threshold were currently at 2 and the number of
  ;; naturally occurring false antecedents (ie, antecedents which are false
  ;; with no rule deletions) was 1, the new threshold would be 
  ;;    2 + (1 - 2 - 1) = 2 + (-2) = 0.
  ;; (note: formula for the above is old + (number-false - old - 1) which 
  ;; ensures that the number of false antecedents exceeds the threshold.
  ;;-------------------------------------------------------------------------
  parent
  child
  (antes ())
  (threshold-delta 0)
  )


(defstruct candidate
  ;;-------------------------------------------------------------------------
  ;; Generalization and specialization depend upon the abduce routine and the
  ;; retract-rules routine to compute the deletions necessary (if any) to fix
  ;; a given example. To compute these fixes, both abduce and retract-rules
  ;; consider several candidate solutions. This structue is used by those two
  ;; routines to hold these candidate solutions. Each of the fields of the 
  ;; used as described below.
  ;;
  ;; For abduce, this structure replaces a list of deletions with a flag 
  ;; consed to the front of the list indicating whether or not a given rule
  ;; was provable without deletions. Thus, the "provable" flag is set to the
  ;; flag from the old list, and "deletions" is set to the list of deletions.
  ;; The last field is not used by abduce.
  ;;
  ;; For retract-rules, this structure again replaces a list of deletions and
  ;; a provable flag. So, "provable" indicates whether the rule is provable 
  ;; and "deletions" holds the list of subrule deletions necessary to make
  ;; the given rule provable (these subrule deletions are rule deletions that
  ;; make enough antecedents of the rule false that the rule becomes unprov-
  ;; able).
  ;;
  ;; 05-JAN-93 (ptb): adding another field so that the retract-rules routines
  ;; can pass back information about how a rule's threshold should change. 
  ;; Since rule retraction was written to take threshold information into 
  ;; account, I needed a way to pass around new threshold information in the
  ;; candidate structure since this is the structure that all the routines 
  ;; return. Probably, I'll want to update the abduction routines to make use
  ;; use of this field too (some day!).
  ;;-------------------------------------------------------------------------
  provable
  (deletions ())
  (threshold-delta 0)
  )


(defstruct solution
  ;;-------------------------------------------------------------------------
  ;; Both generalization and specialization must keep track of potential
  ;; solutions so that they may be compared. Each example which is not proved
  ;; correctly by the theory has an associated "fix" which, if applied to the
  ;; theory, will correct the theory for that particular example. Thus, a
  ;; solution has two slots:
  ;;
  ;; A "fix" is a list of deletions. These must all be performed to fix the
  ;; theory for a given example.
  ;;
  ;; An "example" which is the example fixed by "fix". Note that there are
  ;; TWO KINDS OF EXAMPLES which are fixed by NEITHER: unprovable positives 
  ;; or provable negatives. Unprovable positives are stored simply using the
  ;; example structure for the example (see above) but provable negatives are
  ;; more complicated. Since a given example can be a provable negative for
  ;; MANY DIFFERENT categories, the same example can act as many provable
  ;; negatives. Thus a "PROVABLE-NEG" STRUCTURE is used to store examples
  ;; here when the solution is for a provable negative (see below).
  ;;-------------------------------------------------------------------------
  (fix ())
  example
  )


(defstruct provable-neg
  ;;-------------------------------------------------------------------------
  ;; NEITHER must keep track of the remaining "negative" examples for which
  ;; the theory must be specialized. These have a special structure, since
  ;; each example may act as several different negative examples (called
  ;; provable negatives). The information which must be kept is:
  ;;
  ;; (1) The category for which the example was proven but should not have
  ;; been.
  ;; (2) A flag, indicating whether or not the example was provable for its
  ;; CORRECT category. This is essential for keeping track of the right
  ;; information to pass to the induction stage (see comments in the routines
  ;; "init-retracton-lists" and "provable-without-rule").
  ;; (3) The example in question.
  ;;-------------------------------------------------------------------------
  category
  flag
  example
  )

  
(defstruct revision
  ;;-------------------------------------------------------------------------
  ;; Once the fixes for the various examples have been compared, one is
  ;; selected to be applied to the theory (see comments in generalization and
  ;; specialization). The best fix is called a "revision" and it may correct
  ;; the theory for MORE THAN ONE example. Thus a revision has to keep two
  ;; pieces of information:
  ;;
  ;; (1) The best "fix" (ie, list of deletions).
  ;;
  ;; (2) The "solutions" which were covered by the fix stored here.
  ;;
  ;; I added a third (and right now superfluous structure component) called
  ;; "threshold-solutions". The motivation behind it is that a given fix can
  ;; be applied two ways at present: either as a change to the threshold or 
  ;; as a deletion. The two applications will likely cover different sets of
  ;; solutions. So, I wanted the option of being able to collect both types
  ;; of information for a given fix.
  ;;-------------------------------------------------------------------------
  (fix ())
  (solutions ())
  (threshold-solutions ())
  )


;;===========================================================================
;; MODEL STRUCTURES
;;
;; The structures below are used for manipulation of all the revisions made 
;; for a given student. 
;;===========================================================================

(defstruct model-change
  ;;-------------------------------------------------------------------------
  ;; A structure which holds a revision made for a student.
  ;;
  ;; The fields of this structure are very similar to those of a revision, 
  ;; but with the information stored in a format more conducive to 
  ;; constrution of a common student model. They are:
  ;;
  ;; TYPE: holds the type of revision. Valid types are (for generalization):
  ;; antecedent-deletion, rule-addition, threshold-increment; (for speciali-
  ;; zation): rule-deletion, antecedent-addition, threshold-decrement. The 
  ;; actual type names used are "del-ante", "add-rule", "del-rule", "add-ante"
  ;; and "thresh-inc", "thresh-dec".
  ;;
  ;; RULE: a pointer to the rule structure being updated.
  ;;
  ;; DELTA-SUM: the benefit/cost of implementing this rule change in the base
  ;; theory. Delta-sums are calculated by summing (not surprisingly) all the
  ;; delta values for all student models. A delta for a student model is
  ;; calculated by finding the difference in distance from the base theory
  ;; for the student model when the revision is implemented.
  ;;
  ;; ANTECEDENTS: the antecedents of the change being made to the rule. For a
  ;; deleted rule, this field is empty. For deleted antecedents, this has a 
  ;; list of the antecedents being deleted. For an added rule it holds the 
  ;; antecedents for the new rule and for added antecedents, it holds what's
  ;; being added.
  ;;
  ;; GENSYM-ANTES: -- right now, unused. May be used later for split rules.
  ;;
  ;; THRESHOLD: -- right now, unused.
  ;;
  ;; Note that each revision here is assumed to be a change to one rule. This
  ;; is done to make the computation of LGG values easier (see model.lisp).
  ;;-------------------------------------------------------------------------
  type
  rule
  (delta-sum 0)
  (antecedents ())
  (gensym-antes ())
  (threshold 0)
  )


(defstruct model-apply
  ;;-------------------------------------------------------------------------
  ;; A structure used to sort a list of model change structures by rule. Used
  ;; to create a new theory which implements the model-change revisions. Has
  ;; a rule field which should have the same entry as the rule field of all
  ;; the model-change structures stored here, followed by a field FOR EACH
  ;; TYPE of model-change structure allowed. All the model-change revisions
  ;; of the same type for a given rule are stored together in one of these
  ;; fields.
  ;;-------------------------------------------------------------------------
  rule
  (add-rules ())
  (del-rules ())
  (add-antes ())
  (del-antes ())
  )


;;===========================================================================
;; SUPPORT ROUTINES
;;
;; Below are a variety of routines for manipulation and testing of the above
;; structures that don't really belong in any other section of the code. They
;; are mostly test, search, and reset routines.
;;===========================================================================

(defun vertex-member-test (item vertex)
  ;;-------------------------------------------------------------------------
  ;; The "prop" field of the vertex structure is checked against the incoming
  ;; "item". Since eq is used, this routine will not work for propositions
  ;; such as "(man socrates)" which are not singletons.
  ;;-------------------------------------------------------------------------
  (eq item (vertex-prop vertex)))


(defmacro leaf-vertex (v)
  ;;-------------------------------------------------------------------------
  ;; Checks to see if the vertex is a leaf by (1) testing that it is a vertex
  ;; and (2) seeing if it has any children. Note that both operational and
  ;; intermediate vertices with no children will show up as leaf vertices.
  ;;-------------------------------------------------------------------------
  `(and (vertex-p ,v)
    (not (vertex-children ,v))))


(defun find-category (name &optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Given an input name (probably from an example) this routine searches
  ;; through the list of vertices in `theory' looking for one which has, as
  ;; its consequent, a proposition which matches `name'. Note that this only
  ;; checks the top level rules, as it should, which contain consequents for
  ;; the rules of each category. Thus, what is returned is the top level
  ;; "category tree" which holds all the rules associated with proving a
  ;; given category.
  ;;-------------------------------------------------------------------------
  (first (member name theory
		 :test #'vertex-member-test)))


(defun find-vertex (name &optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; Uses "find-vertex-1" to search for the vertex that matches `name' and
  ;; catches the result here. Uses throw/catch so the recursive search of
  ;; "find-vertex-1" can quit as soon as possible.
  ;;-------------------------------------------------------------------------
  (catch 'found-vertex
    (loop for th in theory
	  do (find-vertex-1 name th))))


(defun find-vertex-1 (name vert)
  ;;-------------------------------------------------------------------------
  ;; Recursively (depth first) searches the verticies of `vert' and its
  ;; children until it finds a vertex whose `prop' field matches `name'.
  ;; Throws to "find-vertex" when found.
  ;;-------------------------------------------------------------------------
  (if (eq (vertex-prop vert) name)
      (throw 'found-vertex vert)
      (loop for c in (vertex-children vert)
	    if (rule-used c)
	    do (loop for a in (rule-antecedents c)
		     do (find-vertex-1 name (antecedent-vertex a))))))


(defun set-abduced?-flags (deletion abduced?-val)
  "Given a deletion, which has a rule and a list of antecedents to be abduced
from the rule, change the antecedent-abduced? flag of the antecedents to
abduced?-val"
  ;;-------------------------------------------------------------------------
  ;; Sets the "abduced" flag of a antecedent to the "abduced?-val" value.
  ;;
  ;; INPUTS: a deletion, which has a child rule and the antecedents which are
  ;; to be deleted (abduced) from it. Note that the threshold of the rule
  ;; must be changed to reflect this
  ;;
  ;; Use abduced?-val = t to temporarily remove the antecedents, and use
  ;; abduced?-val = nil to put the antecedents back into the theory.
  ;;-------------------------------------------------------------------------
  (loop for ante in (deletion-antes deletion)
	do (setf (antecedent-abduced? ante) abduced?-val)))


(defun set-rule-used (rule use-val)
  "Given a rule, of the form (flag rest-of-rule), changes the flag
associated to flag-val to retract the rule (nil = retracted)."
  ;;-------------------------------------------------------------------------
  ;; Sets the "rule-used" flag of a rule to the "use-val" value.
  ;;
  ;; Use use-val = nil to remove the rule from the theory, and use
  ;; use-val = t to put the rule back into the theory.
  ;;-------------------------------------------------------------------------
  (setf (rule-used rule) use-val))


(defun incf-threshold (deletion)
  "Given a rule, of the form (flag threshold rest-of-rule), increments the
threshold by the value indicated in the threshold-delta field of the
deletion."
  ;;-------------------------------------------------------------------------
  ;; Increments the threshold value of the rule by threshold-delta field of 
  ;; "deletion".
  ;;-------------------------------------------------------------------------
  (incf (rule-threshold (deletion-child deletion))
	(deletion-threshold-delta deletion)))


(defun decf-threshold (deletion)
  "Given a rule, of the form (flag threshold rest-of-rule), decrements the
threshold by the value indicated in the threshold-delta field of the
deletion."
  ;;-------------------------------------------------------------------------
  ;; Increments the threshold value of the rule by threshold-delta.
  ;;-------------------------------------------------------------------------
  (decf (rule-threshold (deletion-child deletion))
	(deletion-threshold-delta deletion)))


(defun reset-threshold (rule)
  "Given a rule, this routine will set the threshold of that rule to 0 (ie,
no failures allowed)."
  (setf (rule-threshold rule) 0))


(defun count-antes (rule)
  ;;-------------------------------------------------------------------------
  ;; counts how many antecedents (unabduced) are left in a rule
  ;;-------------------------------------------------------------------------
  (loop for ante in (rule-antecedents rule)
   	sum (if (antecedent-abduced? ante) 0 1)))


(defun original-rule (rule)
  "Retrieves the original rule, in standard list format, from the rule 
structure passed in."
  ;;-------------------------------------------------------------------------
  ;; Retrieves the original rule from the "rule" rule structure by listing 
  ;; the consequents and antecedents together. Note this routine does NOT 
  ;; print out the threshold value associated with the rule. I make the
  ;; assumption that the consequent and antecedents are enough to uniquely
  ;; indentify the rule. This assumption may have to be changed.
  ;;-------------------------------------------------------------------------
  `(,(rule-consequent rule) 
    <- 
    ,@(loop for ante in (rule-antecedents rule) 
            collect (antecedent-prop ante))))
