
;;===========================================================================
;; Specialization algorithm for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; The code here is used to alter the theory until all provable negative
;; examples become unprovable. The result is a specialization of the theory,
;; thus the main routine here is "specialize".
;;
;; CHANGE HISTORY
;;
;; 06-JAN-93: (ptb) modified the code to work for m-of-n threshold rules.
;; 12-MAR-93: (ptb) modified the perm-deletion code to call temp-deletion 
;;            instead so that the student modeling would work.
;; 09-APR-93: (ptb) found a bug in the way the best deletion is selected for
;;            applying to the theory. During rule retraction, I am very care-
;;            ful to make sure that any fix for an example does NOT cause the
;;            same example to be a new failing positive. However, during the
;;            computation to see which fix covers the most failing negatives
;;            there is not check to see if a fix creates this problem for a
;;            DIFFRENT failing negative. So, I put in a fix between the two
;;            routines which calculate the negative fix value (see the code
;;            for calc-neg-fix-value).
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

#+:cltl2(declaim (special *trace-spec*))
#-:cltl2(proclaim '(special *trace-spec*))

;;---------------------------------------------------------------------------
;; GLOBAL VARIABLES
;;

;; I use two globals here, one to keep track of all the original provable
;; negative examples, the other to keep track of how many remain "uncovered"
;; for a given set of fixes applied to the theory. Not surprizingly, the 
;; variable *provable-negatives* has a list of the provable positive 
;; examples. The other global, *remaining-negatives*, is used to avoid 
;; changing *provable-negatives* directly. "init-retraction-list" sets this 
;; second global by calling "retract-rules" first on the elements of 
;; *provable-negatives* and then subsequently on the elements of
;; *remaining-negatives* The rest of the routines loop to generalize the 
;; theory, quiting when *remaining-negatives* is nil.
;;
;; The format of *remaining-negatives* is a list of SOLUTIONS (see
;; structures.lisp). Each solution is a provable negative example paired with
;; the NEGATIVE fix that would allow the theory to avoid proving the example,
;; eg: 
;;
;;    *remaining-negatives* = ( (fix-1 pn-1)
;;                              (fix-2 pn-2) ... )
;;
;; Here, a "pn" is a provable-neg structure and includes: the erroneous
;; category, t or nil indicating if the correct category was provable for the
;; example, and the actual example.
;;
;; A negative fix is a LIST of DELETIONS (see the deletion structure). Note
;; that the deletion-antes component of the structure is not used for
;; specialization. Also, multiple rule deletions from the same parent vertex
;; would be listed separately in the fix.
;;
;; The *neg-induction-alist* global is used to keep all induction information
;; in one place until the end of specialization. It holds all the deletions
;; (plus their associated pn's) which caused new failing positives. It is of
;; the form: 
;;
;;   *neg-induction-alist* = ((rule-1
;;                              (deletion pn1 pn2 ...)
;;                              (deletion pn1 pn2 ...)
;;                              ...)
;;                            (rule-2
;;                              (deletion pn1 pn2 ...)
;;                              (deletion pn1 pn2 ...)
;;                              ...) )
;;
;; So, each deletion is assoc'ed with the rule to which it applies.
;; Furthermore, the deletion (in the same format as above) is stored with
;; all the provable negative examples that require the deletion. Note that
;; the "rules" in the above alist are NOT vertex structures but members of
;; the "children" field of a vertex (see structures.lisp).
;;---------------------------------------------------------------------------
(defvar *remaining-negatives* nil
  "A list of solutions for the provable negative examples")

(defvar *neg-induction-alist* nil
  "An alist of the deletions which caused new unprovable positives and thus
must be fixed by inducing a new rule.")



(defun specialize (&optional (m-of-n nil) (theory *neither-theory*))
  "Given a theory, makes incremental modifications to the theory until all
provable negatives (ie, false positives) are unprovable (assumes
label-examples and generalize have been called)."
  ;;-------------------------------------------------------------------------
  ;; Makes incremental changes to the theory as follows: (1) Find from among
  ;; the "fixes" for the remaining provable negatives that fix which is the
  ;; "best." Then (2) modify the theory  DESTRUCTIVELY with that fix and
  ;; repeat until no provable negatives are left.
  ;;-------------------------------------------------------------------------
  ;; first, get a fix for every unprovable positive
  (init-retraction-lists)

  ;; then, redefine the is-covered-by? routine for subset matching (see
  ;; subset.lisp)
  (redef-for-deletions)

  ;; continue until all remaining negatives are covered. 
  (loop while *remaining-negatives*
	for revision = (best-deletion theory m-of-n)
        for pure-deletions = (try-neg-fix revision theory m-of-n)
	do
	(update-negatives revision pure-deletions theory m-of-n))

  ;; last, induce any new rules for revisions which could not be made to the
  ;; theory.
  (add-neg-inductions m-of-n theory))


(defun init-retraction-lists (&optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; This routine is called by specialize to ensure that the four global
  ;; varibles in "label.lisp" are correctly reset for specialization. It
  ;; assumes that generalization has just completed and that the
  ;; generalization process has not altered the *provable-negatives* nor the
  ;; *provable-negative-alist* variables (this is provably true for the
  ;; generalization algorithm). However, since generalization guarantees that
  ;; all unprovable positives are now provable, the *unprovable-positives*
  ;; and *provable-positives* variables must be updated. This is done by
  ;; moving everything on the two lists to *provable-positives*.
  ;;
  ;; Once these globals are updated, this routine can then initialize the
  ;; value of *remaining-negatives*, which is a list of SOLUTIONS (see the
  ;; file "structures.lisp"). This list contains a fix for each example
  ;; where a fix is a LIST of rule deletions: (parent rule). Note that each
  ;; "rule" here is an element of the "children" field of the "parent"
  ;; vertex.
  ;;
  ;; BUG FIX: I added code to change the "provable-neg-flag" of the provable
  ;; negative examples to reflect any fixes from generalization.
  ;; Specifically, for every example moved from the unprovable positives to
  ;; the provable positives, the flag is changed from false to true if the
  ;; example is present in the provable negatives list.
  ;;-------------------------------------------------------------------------
  (loop for unp in *unprovable-positives*
	for pn = (car (member unp *provable-negatives*
			      :test #'(lambda (x y)
					(eq x (provable-neg-example y)))))
	if pn
	do (setf (provable-neg-flag pn) t))
  (setf *provable-positives*
	(nconc *provable-positives* *unprovable-positives*))
  (setf *unprovable-positives* nil)
  (setf *neg-induction-alist* nil)
  (setf *remaining-negatives*
	(mapcar
	 #'(lambda (pn)
	     (make-solution :fix (retract-rules
				  (provable-neg-example pn)
				  (provable-neg-category pn)
				  (provable-neg-flag pn) theory)
			    :example pn))
	 *provable-negatives*)))


(defun redef-for-deletions ()
  "Redefines the is-covered-by? routine to work for rule deletions."
  ;;-------------------------------------------------------------------------
  ;; This routine redefines the "is-covered-by?" routine in the file
  ;; subset.lisp to compare two deletions. Here the deletion d1 is
  ;; considered to be "covered" by d2 in the routine below if the two
  ;; deletions are identical (ie, refer to exactly the same component of the
  ;; NEITHER decision tree structure).
  ;;-------------------------------------------------------------------------
  (setf (symbol-function 'is-covered-by?)
	#'(lambda (d1 d2)
	    (and (eq (deletion-child d1) (deletion-child d2))))))


;;===========================================================================
;; FIX COMPARISON ROUTINES
;;
;; Used to select the best fix from among all the candidate fixes for the
;; provable negative examples in *remaining-negatives*
;;===========================================================================

(defun best-deletion (&optional (theory *neither-theory*) (m-of-n nil))
  "Finds the provable negative example in *remaining-negatives* whose fix
covers the most provable negatives, creates the fewest new unprovable
positives, and requires the shortest rule deletions. Returns a revision: the
best fix found and a sublist of *remaining-negatives* containing the
negatives covered."
  ;;-------------------------------------------------------------------------
  ;; The idea here is to loop through all the provable negatives, computing
  ;; some numerical measure of how "useful" it would be to delete the
  ;; rules of the fix associated with the pn. This metric combines three
  ;; elements: (1) how many provable negatives are covered (2) the size and
  ;; number of rules that must be abduced and (3) how many unprovable 
  ;; positives are created. 
  ;;
  ;; Returns a revision (see structures.lisp), which is the fix (ie, list of
  ;; deletions) for the best pn found paired with a LIST of the provable
  ;; negatives covered by the fix, ie:  
  ;;
  ;;    (fix ((fix pn) (fix pn) ...) )
  ;;          \______________________/
  ;;          sublist of *remaining-negatives*
  ;;
  ;; NOTE: while this routine is operating, it stores a modified version of a
  ;; revision in "best" which has an extra element at the front for the
  ;; "score" of the revision. This score is compared for picking the best
  ;; revision.
  ;;-------------------------------------------------------------------------
  (loop for soln in (cdr *remaining-negatives*)
	with best = (neg-fix-value (solution-fix (car *remaining-negatives*))
				   theory m-of-n)
	with temp = nil
	finally (return (cdr best))
	do
	(unless (equal (solution-fix soln) (revision-fix (cdr best)))
	  (setf temp (neg-fix-value (solution-fix soln) theory m-of-n))
	  (if (> (car temp) (car best))
	      (setf best temp)))))


(defun neg-fix-value (fix &optional (theory *neither-theory*) (m-of-n nil))
  ;;-------------------------------------------------------------------------
  ;; Computes a "value" for the given fix according to a prescribed formula
  ;; (see calc-neg-fix-value) that is an attempt to measure the "goodness" of
  ;; the fix.
  ;;
  ;; inputs: "fix" is a list of deletion structures (ie, the first field of 
  ;; a solution structure). "examples" and "theory" are what you would expect
  ;; and "m-of-n" is a flag indicating whether or not to try changing 
  ;; threshold values as a revision.
  ;;
  ;; Works in a fashion similar to "pos-fix-value" (see generalize.lisp) but
  ;; with a twist. In specialization, threshold changes are LESS drastic than
  ;; rule deletions. Specifically, the deductive closure of a deleted rule is
  ;; always smaller than that of the threshold change for the same rule. As a
  ;; result, deletions are tried before threshold changes in specialization, 
  ;; and so we must rank here according to deletions to maintain the 
  ;; optimistic heuristic used in pos-fix-value.
  ;;-------------------------------------------------------------------------
  (let ((rev (make-revision :fix fix))
	fix-value)
    (when m-of-n
      (dolist (del fix) (incf-threshold del))
      (calc-neg-fix-value fix rev theory t)
      (dolist (del fix) (decf-threshold del)))
    (dolist (del fix) (set-rule-used (deletion-child del) nil))
    (setf fix-value (calc-neg-fix-value fix rev theory nil))
    (dolist (del fix) (set-rule-used (deletion-child del) t))
    (cons fix-value rev)))


(defun calc-neg-fix-value (fix rev &optional (theory *neither-theory*) 
                               (m-of-n nil))
  "Divides the number of provable negatives covered by the fix, by the
product of the size of the fix and the number of new unprovable positives.
Returns this ratio."
  ;;-------------------------------------------------------------------------
  ;; Here "fix" is a list of deletions, showing what should be retracted for
  ;; a particular pn. These retractions are applied temporarily to the
  ;; theory, and then a ratio is computed which is: 
  ;;
  ;;                       #-provable-negatives-covered
  ;;            -------------------------------------------------
  ;;           (#-new-unprovable-positives * size-of-rules-deleted)
  ;;
  ;; The value of the above ratio is returned.
  ;;
  ;; 09-APR-93 (ptb): found a bug here. When I determine which examples are
  ;; covered by a given fix, I must ELIMINATE any such examples which ALSO
  ;; turn out to be new failing positives. To fix things, I make the variable
  ;; which holds the covered examples dynamically scoped, and then delete
  ;; unwanted elements in the "new-positive-count" routine.
  ;;-------------------------------------------------------------------------
  (let (*covered-examples* num-positives)
    (declare (special *covered-examples*))
    (setf *covered-examples* (negatives-covered theory))
    (setf num-positives (new-positive-count theory))
    (if m-of-n
      (setf (revision-threshold-solutions rev) *covered-examples*)
      (setf (revision-solutions rev) *covered-examples*))
    (/ (length *covered-examples*)
       (* (1+ num-positives)
          (if m-of-n
            (neg-threshold-changes fix)
            (deleted-rules-size fix))))))


(defun negatives-covered (&optional (theory *neither-theory*))
  ;;-------------------------------------------------------------------------
  ;; This routine assumes that *remaining-negatives* has been converted to a
  ;; list of (fix pn) lists.  Each provable negative in *remaining-negatives*
  ;; is checked, and if its bad category (ie provable-neg-category) is NOT
  ;; provable, then the example is collected into a list of all covered
  ;; examples.
  ;;-------------------------------------------------------------------------
  (loop for soln in *remaining-negatives*
	for pn = (solution-example soln)
	unless (member (provable-neg-category pn)
		       (prove-categories (provable-neg-example pn) theory))
	collect soln))


(defun deleted-rules-size (fix)
  ;;-------------------------------------------------------------------------
  ;; Here `fix' is a list of deletions, showing the rules to be deleted for a
  ;; particular example. This routine returns the sum of the lengths of the
  ;; rules deleted in the fix. This is done by summing the length of the
  ;; rules of each element of the fix.
  ;;-------------------------------------------------------------------------
  (loop for del in fix sum (count-antes (deletion-child del))))


(defun neg-threshold-changes (fix)
  ;;-------------------------------------------------------------------------
  ;; Sums all the changes to all the rule thresholds in the fix. Multiply by
  ;; -1 to get a positive number (all threshold changes should be negative).
  ;;-------------------------------------------------------------------------
  (* -1 (loop for del in fix sum (deletion-threshold-delta del))))


(defun new-positive-count (&optional (theory *neither-theory*))
  "Counts the number of NEW unprovable positives created by theory. Typically
called after some rules have been deleted from the theory."
  ;;-------------------------------------------------------------------------
  ;; This routine goes through all the positive examples (in *provable-
  ;; positives*) and checks if they are still provable. If it finds any which
  ;; are NOT, it increments count. 
  ;;
  ;; Works as follows. For each provable positive, computes the categories
  ;; provable. If the desired category is NOT one of these, count is
  ;; incremented. 
  ;;
  ;; 09-APR-93 (pb): This routine updated to fix a bug in the way the best
  ;; fix is found. A dynamically scoped variable is set up in calc-neg-fix-
  ;; value above, which holds all the examples covered by a fix. Here, if the
  ;; same example turns up as a new failing positive, I remove it from that
  ;; dynamically scoped variable. It's kind of a hack, but it was the first
  ;; fix I could think of that was garbage-less, a key consideration since
  ;; this is one of those routines that is called a LOT.
  ;;-------------------------------------------------------------------------
  (declare (special *covered-examples*))
  (loop for ex in *provable-positives*
	for cats = (prove-categories ex theory)
	with count = 0
	do 
        (unless (member (example-name ex) cats) 
          (incf count)
          (setf *covered-examples*
                (delete ex *covered-examples* 
                   :test #'(lambda (x l) 
                             (eq x (provable-neg-example 
                                    (solution-example l)))))))
	finally (return count)))


(defun find-new-positives (&optional (theory *neither-theory*))
  "Collects the NEW unprovable positives created by the theory."
  ;;-------------------------------------------------------------------------
  ;; Similar to new-positive-count, but collects the positives instead.
  ;;-------------------------------------------------------------------------
  (loop for ex in *provable-positives*
	unless (member (example-name ex) (prove-categories ex theory))
	collect ex))


(defun any-new-positives? (&optional (theory *neither-theory*))
  "Tests to see if there are any new UNprovable positives. Quits as soon as
any are found and returns T, else returns NIL."
  ;;-------------------------------------------------------------------------
  ;; Identical to find-new-posities, but returns T/NIL and quits as soon as
  ;; it finds the FIRST unprovable positive.
  ;;-------------------------------------------------------------------------
  (loop for ex in *provable-positives*
	when (not (member (example-name ex) (prove-categories ex theory)))
	return t
	finally (return nil)))

   
;;===========================================================================
;; FIX IMPLEMENTATION ROUTINES
;;
;; Used to implement a fix in a theory. Destructively modifies the theory.
;;===========================================================================


(defun try-neg-fix (revision &optional (theory *neither-theory*) (m-of-n nil))
  "Applies the fix, the first element of the first argument, to the theory.
If the result proves new unprovable positives, the fix is stored for
induction."
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "revision" is a PAIR where the first element is the best fix
  ;; found and the second element is a list of the examples covered.
  ;;
  ;; The idea here is to apply the deletions of the fix one at a time. As
  ;; each is applied, a check is made to see if any new unprovable positives
  ;; are created. If not, the deletion is made permanently. If so, then we
  ;; must induce a rule to replace the deleted rule in the theory. However,
  ;; we wait until the end of specialization to make all the inductions at
  ;; once (see add-neg-inductions).
  ;;
  ;; 18-JAN-93 (ptb) wanted to make a note here that the call to set the
  ;; value of fixed-negs will try to use the negatives covered by a threshold
  ;; fix if at all possible. This is because the fixed-negs value is not used
  ;; unless this code falls through to the induction phase. Since induction
  ;; prefers to use threshold changes, this routine prefers to give it the
  ;; fixed-negs found using threshold changes.
  ;;-------------------------------------------------------------------------
  (trace-print *trace-spec* 
               "~%~%*Attempting to make the following specializations:")
  (loop for del in (revision-fix revision)
	for fixed-negs in (sort-neg-examples revision m-of-n)
        with pure-deletions = t
        finally (return pure-deletions)
        do (when *trace-spec*
             (format t "~%  To make the following examples UNprovable:")
	     (loop for pn in fixed-negs
		   do (format t "~%    example ~D, Bad category = ~A"
			      (position (provable-neg-example pn)
					*neither-examples*)
			      (provable-neg-category pn))))
        unless (try-deletion-fix del theory)
	do (setf pure-deletions nil)
           (unless (try-neg-threshold-fix del m-of-n theory)
             (save-for-neg-induction del fixed-negs))))


(defun try-deletion-fix (deletion theory)
  "Tries to delete a rule from the theory. If successful, the rule is left 
out and T is returned. Otherwise, the theory is reset and NIL is returned."
  ;;-------------------------------------------------------------------------
  ;; For specialization, deletion changes are tried first. This routine is 
  ;; considered successful if it can delete the rule in "deletion" without
  ;; causing any new failing positives. T or Nil is returned, depending upon
  ;; the success of the deletion.
  ;;-------------------------------------------------------------------------
  (let (rval)
    (when *trace-spec*
      (format t "~%  attempting to delete the rule:")
      (pprule deletion))
    (temp-deletion deletion)
    (setf rval (any-new-positives? theory))
    (if rval
      (progn (undo-deletion deletion)
             (trace-print *trace-spec* " ... failed."))
      (progn (perm-deletion deletion)
             (trace-print *trace-spec* " ... succeeded.")
             (record-rule-deletions deletion)))
    (not rval)))


(defun try-neg-threshold-fix (deletion m-of-n theory)
  "Tries to specialize a rule deletion by changing the threshold. If this
succeeds, the threshold is left at its new value and T is returned. Otherwise
the threshold is restored to its old value and NIL is returned."
  ;;-------------------------------------------------------------------------
  ;; Tries to specialize a rule deletion by changing the threshold. If this
  ;; succeeds, the threshold is left at its new value and T is returned. 
  ;; Otherwise the threshold is restored and NIL is returned.
  ;;
  ;; NOTE: If the threshold change works, it is still possible that the new
  ;; rule threshold is < 0, effectively deleting the rule. In this event, the
  ;; rule is deleted from the theory.
  ;;-------------------------------------------------------------------------
  (if m-of-n
      (let (rval)
	(when *trace-spec*
	  (format t "~%  in rule:~%    ")
	  (pprule deletion nil)
	  (format t "~%  change threshold by: ~D"
		  (deletion-threshold-delta deletion)))
	(incf-threshold deletion)
	(setf rval (any-new-positives? theory))
	(if rval
          (progn
            (trace-print *trace-spec* " ... failed.")
            (decf-threshold deletion))
          (progn
            (trace-print *trace-spec* " ... succeeded.")
            (if (< (rule-threshold (deletion-child deletion)) 0)
                (perm-deletion deletion))))
	(not rval))
      nil))


(defun sort-neg-examples (revision m-of-n)
  ;;-------------------------------------------------------------------------
  ;; INPUTS: revision is a revision structure (see structures.lisp) which
  ;; contains solutions showing which example(s) is (are) covered by the fix.
  ;;
  ;; The idea is to check here to see if a given example can be covered using
  ;; only a subset of fix. If so, then only those deletions of fix need a
  ;; reference to the example. Thus the examples are pidgeon-holed into a
  ;; slots, one slot for each deletion in the fix. 
  ;;
  ;; RETURN: a list of lists, equal in length to the number of deletions in
  ;; the best fix (the car of "revision"). Each sublist is a list of provable
  ;; negatives that require the given deletion. This information is required
  ;; by the induction phase (if reached) where we must have the provable
  ;; negatives related to the deletion.
  ;;
  ;; 18-JAN-93 (ptb): added an argument so this routine can try to use the
  ;; examples covered by threshold changes. Since the only use of the example
  ;; list returned by this routine is in induction, and since induction 
  ;; prefers to use threshold changes, this routine tries to use the examples
  ;; covered by the threshold change (if any). Otherwise, it uses those
  ;; examples covered by rule deletion.
  ;;-------------------------------------------------------------------------

  ;; Initialize the return value to be a list of empty lists. Also, create a
  ;; blank list for the `marks' used by `covered-by-subset?' 
  (let* ((fix (revision-fix revision))
	 (sorted-negatives (make-sequence 'list (length fix)))
	 (marks (make-sequence 'list (length fix) :initial-element 0)))

    ;; loop through the examples covered by the fix, inserting the example in
    ;; the appropriate element(s) of `sorted-negatives'.
    (loop for soln in (if m-of-n (revision-threshold-solutions revision)
                          (revision-solutions revision))
	  for ex-fix = (solution-fix soln)
	  for pn = (solution-example soln)
	  do
	  (if (covered-by-subset? ex-fix fix marks)
	      
	      ;; if the fix for the example needs only a subset of the best
	      ;; fix, add example to only those sublists of sorted-negatives
	      (do ((sublist sorted-negatives (cdr sublist))
		   (m marks (cdr m)))
		  ((null sublist) nil)
		(if (= 1 (car m))
		    (rplaca sublist (cons pn (car sublist)))))
	      
	      ;; otherwise, add example to all sublists of sorted-negatives
	      (do ((sublist sorted-negatives (cdr sublist)))
		  ((null sublist) nil)
		(rplaca sublist (cons pn (car sublist))))))
    sorted-negatives))


(defun temp-deletion (deletion)
  "Given a deletion, sets flags such that the rule of the deletion is
temporarily removed from the theory."
  ;;-------------------------------------------------------------------------
  ;; Calls "set-rule-used" to do all the work.
  ;;-------------------------------------------------------------------------
  (set-rule-used (deletion-child deletion) nil))


(defun undo-deletion (deletion)
  "Given a deletion, sets flags such that the rule of the deletion which was
temporarily removed from the theory is restored."
  ;;-------------------------------------------------------------------------
  ;; Calls "set-rule-used" to do all the work.
  ;;-------------------------------------------------------------------------
  (set-rule-used (deletion-child deletion) t))


(defun perm-deletion (deletion)
  "Given a deletion, permanently (destructively) removes the rule of the
deletion from the theory."
  ;;-------------------------------------------------------------------------
  ;; Since the deletion has the parent vertex and the rule to be deleted from
  ;; the children of that vertex, this routine is straightforward. Note,
  ;; however, that we must be sure to RESET the value of the children field
  ;; of the parent vertex in case we are deleting the first child. Also, note
  ;; that the "eq" function can be used since we are comparing pointers.
  ;;
  ;; 12-MAR-93 (ptb): modified to call temp-deletion instead of making a 
  ;; permanent deletion. I did this to allow the student modeling code to 
  ;; work, which needs a copy of the original rule to store with the revision
  ;; related to the rule. I left the old code commented out, so it can be
  ;; easily restored if necessary.
  ;;-------------------------------------------------------------------------
  ;;(let ((vert (deletion-parent deletion)))
  ;;  (setf (vertex-children vert)
  ;;        (delete (deletion-child deletion) (vertex-children vert)
  ;;                :test #'eq)))
  (temp-deletion deletion)
  )


(defun save-for-neg-induction (deletion negatives)
  "Puts the deletion on a list for later induction of new rules.
Stores the negatives covered with the deletion."
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "deletion" is a deletion in the same format as described in the
  ;; file "structures.lisp". "negatives" is a list of provable negative
  ;; structures that require the deletion (see sort-neg-examples routine).
  ;;
  ;; OUTPUT: destructively modifies *neg-induction-alist* to include the new
  ;; deletion and its provable negatives. Each is associated with the rule
  ;; which was deleted. If there is already a list for that rule, the
  ;; deletion is added to that list. Otherwise, a new list is created and
  ;; consed to the front of *neg-induction-alist*.
  ;;-------------------------------------------------------------------------
  (let* ((rule (deletion-child deletion))
	 (loc (assoc rule *neg-induction-alist* :test #'eq)))
    (if loc
	(rplacd loc (cons (cons deletion negatives) (cdr loc)))
	(setf *neg-induction-alist*
	      (cons (list rule (cons deletion negatives))
		    *neg-induction-alist*)))))


(defun update-negatives (revision pure-deletions
                                  &optional (theory *neither-theory*)
                                  (m-of-n nil))
  "Removes negatives from *remaining-negatives* that are covered by revision,
and recomputes fixes for the negatives remaining using theory."
  ;;-------------------------------------------------------------------------
  ;; Needs to (1) remove the covered negatives from *remaining-negatives* and
  ;; then (2) recompute the fix for each remaining negative as done in the
  ;; init-retraction-lists routine.
  ;;
  ;; Removing the covered negatives is easy since these are stored as part of
  ;; the revision. We can loop through the covered negatives and use "delete"
  ;; (with the eq function) since we have direct pointers to elements of the
  ;; *remaining-negatives* list. Note, however, that since the first element
  ;; of the list may be deleted, we must re-setf *remaining-negatives*.
  ;;
  ;; Recomputing fixes is done destructively in place to save on garbage. We
  ;; simply loop through whatever is left on *remaining-negatives*, replacing
  ;; the old fix with a new fix computed for the example.
  ;;
  ;; RETURN: since *remaining-negatives* is destructively modified, no return
  ;; value is required. 
  ;;
  ;; 08-JAN-93 (ptb): changed in much the same fashion that its counterpart
  ;; "update-positives" was changed (see generalize.lisp file). Here however
  ;; the set of examples covered by threshold changes is a SUBSET of the set
  ;; of examples covered by deletion. Yet, if the program is being run 
  ;; without m-of-n changes, we must default to the deletion set of examples
  ;; (since the threshold set was not computed). 
  ;;-------------------------------------------------------------------------
  (let ((covered-negatives
         (if (and m-of-n (not pure-deletions))
             (revision-threshold-solutions revision)
             (revision-solutions revision))))
    (loop for soln in covered-negatives
	  do (setf *remaining-negatives*
		   (delete soln *remaining-negatives* :test #'eq)))
    (loop for soln in *remaining-negatives*
	  for pn = (solution-example soln)
	  do (setf (solution-fix soln)
		   (retract-rules (provable-neg-example pn)
				  (provable-neg-category pn)
				  (provable-neg-flag pn) theory)))))


(defun add-neg-inductions (&optional (m-of-n nil) (theory *neither-theory*))
  "Performs induction for all deletions which could not be directly applied
to the theory and replaces them in the theory with new rules."
  ;;-------------------------------------------------------------------------
  ;; This routine loops through all the elements of *neg-induction-alist*
  ;; and forms new rules (by induction) to cover the negatives in this
  ;; alist. The method is as follows: (1) delete the rule in question and
  ;; then (2) induce new rules to replace it using the examples stored with
  ;; the deletion.
  ;;
  ;; Inducing the new rules is done by a series of nested subroutine calls
  ;; that work as follows (from the inside out). First, the positive and
  ;; negative examples are converted into a form for induction. These are
  ;; passed to a routine that builds new rules. The new rules are
  ;; then converted into the internal NEITHER format, which is gets "patched"
  ;; with the consequent of the rule in the deletion. Finally, the resulting
  ;; patched theory (in NEITHER format) is merged with the existing theory
  ;; stored in *neither-theory*.
  ;;
  ;; HACK ADDED: I needed the same hack as was used in add-pos-inductions
  ;; (see documentation in generalize.lisp).
  ;;
  ;; (ptb 9-24-92): I changed this routine somewhat so I could get a printout
  ;; during a trace of specialization. All I did was to break off the final
  ;; call to merge-with-current-theory so I could get the form of the rules
  ;; after being translated into NEITHER's internal decision tree format.
  ;;
  ;; 15-JAN-93 (ptb): modified to work with m-of-n rules. Induction for 
  ;; specialization reveals much the same problem as was exhibited for 
  ;; generalization (see documentation for "add-pos-induction" and "induce-
  ;; rules"). Again, we end up with a gensym consequent we must replace in 
  ;; the original rule. Consider the following ("-" indicates failed 
  ;; antecedent, "+" indicates successful antecedent):
  ;;
  ;;    3: a <- b c d e f
  ;;   e1:      + + + + -  (negative example), provable
  ;;   e2:      + + - - -  (positive example), provable
  ;;
  ;; Here, example e1 is a provable negative which must be eliminated by 
  ;; specializing the rule. In the old specialization algorithm, we would 
  ;; delete the rule, see which positives were no longer provable, induce, 
  ;; and add the results of induction to the original rule. So, if induction
  ;; yielded (x y z) we'd get
  ;;
  ;;    3: a <- b c d e f x y z
  ;;   e1:      + + + + - ? ? ?
  ;;   e2:      + + - - - + + +
  ;;
  ;; But here we get a problem. Since only one "?" need be negative, the 
  ;; negative example is guaranteed to have only 2 failures. Since the 
  ;; positive example (e2) has 3 failures, we can't simply set the threshold 
  ;; to 4 to make e2 provable (since e1 might still be provable). What we 
  ;; really need is something like:
  ;;
  ;;    3: G <- b c d e f
  ;;    0: a <- G x y z
  ;;
  ;; Now, both e1 and e2 satisfy the first rule, but only e2 satisfies the 
  ;; second rule.
  ;;
  ;; One other note is that we can actually do better at finding the new 
  ;; unprovable positives than by deleting the rule. Instead, we can decrease 
  ;; the number of failures allowed until the negative becomes unprovable and 
  ;; then compute the positives. This should give a tighter fit for the 
  ;; induction (though in some cases the threshold may actually be 
  ;; decremented below 0 which is the same as deleting the rule).
  ;;
  ;; SPLITTING: The split-rule argument to induce-rules can be set to nil in
  ;; either of two cases: (1) the original threshold is 0 or (2) the change 
  ;; needed in the threshold is only 1. Otherwise, the splitting must take
  ;; place, even if threshold changes aren't being used. In fact, if no 
  ;; threshold information is available, then any rule with a non-zero 
  ;; threshold must be split.
  ;;-------------------------------------------------------------------------
  (push 'positive *categories*)
  (loop for ind in *neg-induction-alist*
	for rule = (first ind)
	for del-pn-pairs = (rest ind)
        for del = (first (first del-pn-pairs))
        for old-threshold = (rule-threshold rule)
	with new-tree = nil
	with temp = nil
	do

	;; temporarily remove rule from the theory
        (if m-of-n
            (incf-threshold del)
	    (temp-deletion del))
	
	(setf temp
	      (convert-theory
	       (induce-rules
		(cnvt-to-standard-ex
		 (find-new-positives theory)
		 (collect-induction-negatives del-pn-pairs))
		old-threshold
		(if (or (= 0 old-threshold)
			(= -1 (deletion-threshold-delta del)))
                    nil t)
		(loop for ante in (rule-antecedents rule)
                      unless (antecedent-abduced? ante)
		      collect (antecedent-prop ante)))))
	;; (format t "~%induced rules are ")
	;; (dolist (r temp) (format t "~%   ~A" r))
	(setf new-tree
	      (patch-decision-tree
	       (vertex-prop (deletion-parent del))
	       (theory-to-decision-tree temp)))

	(when *trace-spec*
	  (format t "~%~%+Replacing the rule")
	  (pprule (first (first del-pn-pairs)))
	  (format t "~% with the following rule(s):")
	  (decompile-theory nil nil new-tree))

        ;; record the changes in the student model
        (record-ante-additions rule new-tree)

	;; Note: must put rule back before merging so any lower level rules
	;; from the original rule are reused.
        (if m-of-n
            (decf-threshold (first (first del-pn-pairs)))
	    (undo-deletion (first (first del-pn-pairs))))
	(merge-with-current-theory new-tree)

	;; Remove the rule from the theory
	(perm-deletion (first (first del-pn-pairs))))
  (pop *categories*))


(defun collect-induction-negatives (del-pn-pairs)
  ;;-------------------------------------------------------------------------
  ;; Collects all the provable negative examples stored in the list of
  ;; deletion/provable-negative lists assoc'ed with a given rule. The input
  ;; "del-pn-pairs" is assumed to be in the following form:
  ;;
  ;;   ((deletion pn1 pn2 ...) (deletion pn1 pn2 ...) ...)
  ;;-------------------------------------------------------------------------
  (loop for pair in del-pn-pairs
	with ret-val = nil
	do (loop for pn in (rest pair)
		 for ex = (provable-neg-example pn)
		 unless (member ex ret-val :test #'eq)
		 do (setf ret-val (cons ex ret-val)))
	finally (return ret-val)))
