
;;===========================================================================
;; PFOIL utility routines for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; This file provides a few routines useful for making the correct calls to
;; PFOIL to generate new rules and incorporate them into the theory.
;;
;; PFOIL STRUCTURE: the return value from "train-pfoil" is a list of a
;; default category (for those examples which cannot be labeled) followed by
;; a DNF-list for each category. The DNF-list consists of three items: the
;; category, the number of examples in that category, and the DNF definition
;; for the category. The category is the consequent of the rules formed by
;; the DNF section. The number of examples in the category is used by PFOIL
;; to define default behavior.
;;
;; Note that the "NEGATIVE" category, if it exists, does not have a DNF-list
;; associated with it. Instead, it is used as the default if none of the
;; other DNF-lists can satisfy a given example (thus implementing
;; negation-as-failure).
;;
;; The transformation process, then, is relatively straightforward. The rules
;; are created by looping through the DNF-lists, creating one rule for each
;; conjunct of each clause of the DNF.
;;
;; CHANGE HISTORY
;;
;; 23-FEB-93: (ptb) first version of code.
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; GLOBAL VARIABLE REFERENCE
;;
;; As far as I can figure from Ray's code, PFOIL requires that the value of
;; the *categories* global be exactly the same as the number of categories
;; that you are trying to induce rules for. This is different from ID3, which
;; only required that the desired categories be present on that global.
;; Anyway, if you take this fact plust the fact that I have a hack in another
;; part of the code that plays with the value of *categories*, you get a
;; mess. So, what I do here is simple. I save the value of *categories*, set
;; it to what PFOIL needs (based on the assumption that I'm always going to
;; be inducing a rule to tell POSITIVE from NEGATIVE examples) and then
;; restore the value of *categories* before I leave.
;;===========================================================================
#+:cltl2(declaim (special *categories*))
#-:cltl2(proclaim '(special *categories*))    ;; in data file.


(defun pfoil-induce (examples desired-category threshold base-antes)
  ;;-------------------------------------------------------------------------
  ;; A wrapper routine to make the correct internal calls here to do a pfoil
  ;; type induction.
  ;;-------------------------------------------------------------------------
  (let ((cats-save *categories*) rval)
    (setf *categories* '(positive negative))
    (setf rval (pf2th (train-pfoil examples) desired-category
		      threshold base-antes))
    (setf *categories* cats-save)
    ;;(format t "~%Returning from pfoil with: ~A" rval)
    rval))
			      
    
(defun pf2th (pfoil-result desired-category threshold base-antes
			   &optional (keep-all-rules nil))
  "Converts the result of calling PFOIL to a set of rules, ie, a theory.
Keeps only those rules whose consequent matches the desired-category. The
optional base-antes argument consists of antecedents to add to each rule."
  ;;-------------------------------------------------------------------------
  ;; Loops through the DNF-list sections of the pfoil-result (see comments at
  ;; the top of this file) creating a rule for each conjunct of each DNF.
  ;; Each rule is given a threshold of "threshold" and has the contents of
  ;; "base-antes" added to its antecedents.
  ;;-------------------------------------------------------------------------
  (loop for DNF-list in (rest pfoil-result)
	for conseq = (first DNF-list)
	if (or keep-all-rules (eq conseq desired-category))
	nconc
	(loop for conjunct in (third DNF-list)
	      collect `(,threshold ,conseq <- ,@base-antes
			           ,@(xlate-pfoil-rule conjunct)))))


(defun xlate-pfoil-rule (conjunct)
  ;;-------------------------------------------------------------------------
  ;; For a given conjunct (ie, a list of feature-value pairs), this routine
  ;; will return a list of the feature NAMES paired with the values (rather
  ;; than the POSITION of the feature name in the *feature-names* global).
  ;;-------------------------------------------------------------------------
  (loop for ante in conjunct
	collect (list (nth (first ante) *feature-names*) (second ante))))

