
;;===========================================================================
;; Main code for NEITHER
;;
;; -------------------------------------------------------------------------
;; authors: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; This file contains the main routines for running NEITHER.
;;
;; CHANGE HISTORY
;;
;; 26-FEB-93: (ptb) had to recreate this file because I deleted it (shit!).
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")


;;===========================================================================
;; STANDARD INTERFACE ROUTINES
;;===========================================================================

(setf *print-circle* t) ;; tells the printer not to hang when printing self
                        ;; referential structures


;; use LOAD-NEITHER-DATA to load in the file you want to train/test
;; (see the file io.lisp) for a description of "load-neither-data"


(defun train-neither (theory examples &optional (m-of-n nil) (silent nil))
  ;;-------------------------------------------------------------------------
  ;; A standard routine for running NEITHER. This routine makes the following
  ;; assumptions:
  ;; (1) that the "load-neither-data" routine has been called to load the
  ;;     file with the data for training and testing
  ;; (2) that the incoming "theory" is of the correct form and
  ;; (3) that the incoming "examples" are of the correct form.
  ;;
  ;; The easiest way to find out what the format for a data file is like is
  ;; to look at the examples data file that comes with the NEITHER source
  ;; code. It's also useful to read through the comments at the top of the
  ;; file "io.lisp" since they explain what NEITHER does in terms of I/O.
  ;;
  ;; This routine returns a structure which contains a representation of the
  ;; revised theory. If you want to look at the theory, you can call the
  ;; "decompile-theory" routine (in the "debug.lisp" file). This structure is
  ;; what gets passed to the "test-neither" routine to do testing.
  ;;-------------------------------------------------------------------------
  (prepare-theory-and-examples theory examples silent)
  (label-examples)
  (unless silent (format t "~%~%Generalizing Theory..."))
  (generalize m-of-n)
  (unless silent (format t "~%~%Specializing Theory..."))
  (specialize m-of-n)
  *neither-theory*)


(defun test-neither (examples training-result
			      &optional (silent t) (use-floating-point nil))
  ;;-------------------------------------------------------------------------
  ;; A test routie for NEITHER. This code assumes that the "training-result"
  ;; is a representation of the theory stored in NEITHER's internal format,
  ;; such as what is returned from "train-neither". It also assumes the
  ;; examples are in the same format as used in "train-neither".
  ;;
  ;; This routine returns three values: the overall accuracy on the test
  ;; data, the number of "failing positives" (examples not provable in the
  ;; correct category) and the number of "failing negatives" (examples which
  ;; prove a non-nil category which is not correct).
  ;;
  ;; You can set the "silent" optional variable to "nil" and this routine
  ;; will show you what categories were proved for each example as it calculate
  ;;-------------------------------------------------------------------------
  (setf *neither-theory* training-result)
  (set-examples examples t)
  (test-deduce silent use-floating-point))


;;===========================================================================
;; OTHER VERY USEFUL ROUTINES:
;;
;; ROUTINE               FILE             PURPOSE
;; -------               ----             -------
;; decompile-theory      debug.lisp       prints out theory
;; number-theory         debug.lisp       numbers the rules in the theory
;; set-theory            io.lisp          resets theory
;; set-examples          io.lisp          resets examples
;; ppexample-num         debug.lisp       prints example based on position num
;; trace-neither         debug.lisp       traces neither execution
;; trace-neither-status  debug.lisp       shows status of trace flags
;; 
;; END STANDARD INTERFACE ROUTINES
;;===========================================================================

(defun run-neither (&optional (m-of-n nil))
  "Executes the NEITHER algorithm. Assumes a data file has been loaded using
a call to neither-load-data. Attempts m-of-n revisions depending upon the
value of the optional argument."
  ;;-------------------------------------------------------------------------
  ;; Assumes that the examples and data have been loaded into NEITHER and all
  ;; that needs to be done is generalization and specialization.
  ;; Specifically, assumes that calls have been made to "neither-load-data"
  ;; and to "prepare-theory-and-examples".
  ;;-------------------------------------------------------------------------
  (label-examples)
  (generalize m-of-n)
  (specialize m-of-n))


(defun leave-one-out-test (theory examples stud-pos &optional (m-of-n nil))
  ;;-------------------------------------------------------------------------
  ;; Runs NEITHER once for each element of the examples, using all the OTHER
  ;; elements of the examples as the training set. Computes accuracy by
  ;; counting the number of accuracte predictions made by the revised
  ;; theories and dividing by the length of "examples". NOTE that this
  ;; routine assumes the file with all the data has already been loaded by a
  ;; call to "neither-load-data".
  ;;-------------------------------------------------------------------------
  (set-examples examples t)  ;; xlate examples into NEITHER format
  (let ((ex-list (copy-list *neither-examples*)));; make copy of examples
    (loop for ex in ex-list
	  with num-pos = 0
	  and  num-neg = 0
	  and  cats = nil
	  and  max-negs = (nth stud-pos *nurse-error-maxes*)
	  for train = (remove ex ex-list) ;; train with rest of examples
	  for name = (example-name ex)
	  finally
	  (return (values (float (/ (- max-negs num-pos num-neg) max-negs))
			  num-pos num-neg))
	  do
	  (set-theory theory t)
	  (setf *neither-examples* train)
	  (run-neither m-of-n)
	  (setf cats (prove-categories ex))
	  if (and (or cats
		      (not (eq name *negative-category*)))
		  (or (not (= (length cats) 1))
		      (not (member name cats :test #'eq))))
	  do (if (eq name *negative-category*)
		 (incf num-neg (length cats))
		 (if (member (example-name ex) cats :test #'eq)
		     (incf num-neg (1- (length cats)))
		     (progn (incf num-pos)
			    (incf num-neg (length cats))))))))
	      

(defun no-train-stats (theory examples)
  ;;-------------------------------------------------------------------------
  ;; collects stats similar to those of leave-one-out-test for the unmodified
  ;; theory.
  ;;-------------------------------------------------------------------------
  (set-examples examples);; xlate examples into NEITHER format
  (set-theory theory)
  (loop for ex in *neither-examples*
	  with correct = 0
	  with h-pos = 0
	  with c-pos = 0
	  with v-pos = 0
	  with num-neg = 0
	  with n-pos = 0
	  with cats = nil
	  for name = (example-name ex)
	  finally
	  (progn (format t "~%cor=~D  neg=~D  h-pos=~D  c-pos=~D  v-pos=~D  n-pos=~D"
			correct num-neg h-pos c-pos v-pos n-pos)
		 (format t "~%accuracy=~D"
			 (round (* 100 (/ correct (length examples))))))
	  do
	  (setf cats (prove-categories ex))
	  (if (or (and (member name cats :test #'eq)
		       (= (length cats) 1))
		  (and (not cats)
		       (eq name *negative-category*)))
	      (incf correct)
	      (if (eq name *negative-category*)
		  (progn (incf n-pos)
			 (incf num-neg (length cats)))
		  (if (member name cats :test #'eq)
		      (incf num-neg (1- (length cats)))
		      (progn (if (eq 'hypovolemic name) (incf h-pos))
			     (if (eq 'cardiogenic name) (incf c-pos))
			     (if (eq 'vascular-tone name) (incf v-pos))
			     (incf num-neg (length cats))))))
	  ))


(defun run-pfoil (examples)
  ;;-------------------------------------------------------------------------
  ;; Routine for using just the inductive component of neither.
  ;;
  ;; Given a set of examples in neither format, used pfoil to induce a set of
  ;; rules to cover the examples. Destructively modifies the *neither-theory*
  ;; variable by calling "set-theory" using the induced rules. Note that this
  ;; routine assumes that 'NEGATIVE has been pushed onto *categories*.
  ;;-------------------------------------------------------------------------
  (let ((pf-result
	 (train-pfoil
	  (loop for ex in examples
		collect (list (example-name ex) (example-values ex)))))
	temp)
    (setf temp (append temp (pf2th pf-result 'hypovolemic 0 nil)))
    (setf temp (append temp (pf2th pf-result 'cardiogenic 0 nil)))
    (setf temp (append temp (pf2th pf-result 'vascular-tone 0 nil)))
    (set-theory temp)))
    
