
;;===========================================================================
;; Modeling code for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; The routines here are used to store revisions to a theory as they are made
;; and to print out the revisions in a standard list format. These lists can
;; be read in at a later time and several can be combined to form a common
;; student model (stereotypical model).
;;
;; CHANGE HISTORY
;;
;; 12-MAR-93: (ptb) creation date. Note that at this time, I did not design
;;            any of the routines to handle threshold rules, thus while the
;;            NEITHER code will revise threshold rules, the modeling code 
;;            will not work with theories that have any thresholds greater 
;;            than 0 (zero).
;; 23-MAR-93: (ptb) altered the record-ante-additions routine to record ante
;;            additions to rules which were added during generalization in
;;            the added rule model change structure (rather than creating a
;;            new structure). 
;; 02-APR-93: (ptb) altered the record-ante-additions routine again to make
;;            sure that multiple antecedent additions to a rule which were
;;            part of different revisions end up in the same model-change
;;            structure.
;; 08-APR-93: (ptb) completed the code for calculating the common student
;;            model. Have not yet tested it completely.
;; 15-APR-93: (ptb) fixed two bugs in the altered-rules routine.
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;---------------------------------------------------------------------------
;; GLOBAL VARIABLE REFERENCES
;;
;; The code here references three global variables which it assumes are
;; defined elsewhere in the code.
;;---------------------------------------------------------------------------
#-:cltl2(proclaim '(special *trace-common-model*))    ;; in debug.lisp
#+:cltl2(declaim (special *trace-common-model*))


;;===========================================================================
;; GLOBAL VARIABLE DEFINITIONS
;;===========================================================================

(defvar *student-model* nil
  "Used to store all the revisions associated with a student")


;;===========================================================================
;; MODEL ADDITION ROUTINES
;;
;; These routines are used for adding a new element to the *student-model*
;; list.
;;===========================================================================

(defun add-to-model (new-model-change)
  (setf *student-model* (cons new-model-change *student-model*)))


(defun record-ante-deletions (deletion)
  "Records the changes stored in input argument as antecedent deletion type
revisions. Destructively modifies *student-model*."
  ;;-------------------------------------------------------------------------
  ;; Updates the *student-model* variable to reflect the antecedent deletions
  ;; stored in 'deletion'. Note that all deletions from a given rule can be 
  ;; considered together for the student model, even though their source may 
  ;; be different examples (i.e., the various deletions fix separate training 
  ;; examples). Thus this routine first checks to see if antecedent deletions
  ;; have already been recorded for this rule. If so, it adds these deletions
  ;; to that list. If not, a new model-change structure is created.
  ;;-------------------------------------------------------------------------
  ;;(format t "~%->~A" (original-rule (deletion-child deletion)))
  (let* ((orig-rule (original-rule (deletion-child deletion)))
         (rule-present (member orig-rule *student-model*
                               :test #'(lambda (o m) 
                                         (equal o (model-change-rule m)))))
         (antes (loop for ante in (deletion-antes deletion)
                      collect (antecedent-prop ante))))
    (if rule-present
        (setf (model-change-antecedents (car rule-present))
              (append antes (model-change-antecedents (car rule-present))))
        (add-to-model (make-model-change :type 'del-ante :rule orig-rule
                                       :antecedents antes)))))


(defun record-rule-additions (base-rule new-tree)
  "Records the changes stored in its input argument as rule addition type
revisions. Destructively modifies *student-model*."
  ;;-------------------------------------------------------------------------
  ;; This routine is given two inputs: the first being the original rule that
  ;; is being generalized, and the second being the list of rules to add to
  ;; the theory. A new model-change structure is created for each new rule 
  ;; and added to *student-model*.
  ;;
  ;; assumptions: the "base-rule" passed in is a rule structure and refers to
  ;; the rule being generalized by the addition of the new rules in new-tree.
  ;; "new-tree" is a list of vertex structures. Each vertex in new-tree is
  ;; assumed to have zero or more children, and all children are assumed to
  ;; be non-heirarchical definitions for the consequent of base-rule. Thus,
  ;; each child of each vertex struture is visited in turn, and a new model-
  ;; change structure is created for it.
  ;;
  ;; Note that the first version of this routine does NOT handle additions to
  ;; rules with thresholds greater than 0. Thus, no split rules are dealt 
  ;; with here (yet). I did this to avoid trying to figure out how to do an
  ;; lgg on threshold rules. Eventually, I need to add this.
  ;;-------------------------------------------------------------------------
  (let ((orig-rule (original-rule base-rule)))
    (loop for vert in new-tree
          do (loop for child in (vertex-children vert)
                   do (add-to-model
                       (make-model-change 
                        :type 'add-rule :rule orig-rule
                        :antecedents (loop for a in (rule-antecedents child)
                                           collect (antecedent-prop a))))))))


(defun record-threshold-increase ())
;; not implemented yet.
;; NOTE I may need to keep track of which rules have already been altered to
;; work the specializations correctly. Some changes can be compounded and I
;; don't know what the devil to do with them yet.


(defun record-rule-deletions (deletion)
  "Given a deletion structure, picks out the rule being deleted and records 
it in the student model. Destructively modifies *student-model*."
  ;;-------------------------------------------------------------------------
  ;; This is perhaps the easiest of the recording routines here. All it does
  ;; is extract the rule using "original-rule" and create model-change struct
  ;; for it. The *student-model* is then updated using "add-to-model".
  ;;
  ;; Note that it would be weird to have a rule which was added during the
  ;; generalization phase be deleted here, but I suppose it's possible. In
  ;; any event, both modifications are left in, since the either or both may
  ;; be more common singly than in tandem.
  ;;-------------------------------------------------------------------------
  (add-to-model (make-model-change 
                 :type 'del-rule 
                 :rule (original-rule (deletion-child deletion)))))


(defun record-ante-additions (base-rule new-tree)
  "Records the changes stored in its input arguments as antecedent addition 
type revisions. Destructively modifies *student-model*."
  ;;-------------------------------------------------------------------------
  ;; This routine is given two inputs: the first being the original rule that
  ;; is being specialized, and the second being the list of rules which would
  ;; replace the original rule. Each new rule has one or more antecedents.
  ;; The object of this routine, then, is to pick out the antecedents that
  ;; are added in each new rule, create a model-change structure for them, 
  ;; and add it to *student-model*.
  ;;
  ;; assumptions: the "base-rule" passed in is a rule structure and refers to
  ;; the rule being specialized by the replacing it with the new rules in 
  ;; new-tree. "new-tree" is a list of vertex structures. Each vertex in 
  ;; new-tree is assumed to have zero or more children, and all children are 
  ;; assumed to be non-heirarchical definitions for the new rules. Thus,
  ;; each child of each vertex struture is visited in turn, and a new model-
  ;; change structure is created for it.
  ;;
  ;; There is one quirk which this routine must address because
  ;; specialization comes after generalization. In short, if I get notice of
  ;; an antecedent addition to a rule which was added during generalization,
  ;; the additional antecedent is simply added to the list of antecedents of
  ;; the added rule. This check is implemented by the appended-to-added-rules
  ;; routine below.
  ;;
  ;; Note that the first version of this routine does NOT handle additions to
  ;; rules with thresholds greater than 0. Thus, no split rules are dealt 
  ;; with here (yet). I did this to avoid trying to figure out how to do an
  ;; lgg on threshold rules. Eventually, I need to add this.
  ;;
  ;; 02-APR-93 (pbt): put in another check here to append any added antes to
  ;; previous specializations which have added antecedents to a given rule.
  ;; Thus, if two specializations add antecedents to the same rule, both 
  ;; revisions are combined. This is done by checking to see if the rule to
  ;; be modified is an exact copy of THE RESULT of a previous antecedent 
  ;; addition revision (see appended-to-added-antes routine). Both this and
  ;; the previous change are now incorporated with the original design of 
  ;; this routine by setting up variables which are passed to "record-tree"
  ;; which creates the model-change structure(s). Note that if this routine 
  ;; finds itself modifying a previous revision, the old version of that 
  ;; revision is deleted (after its contents are used to set up the call to
  ;; record-tree).
  ;;-------------------------------------------------------------------------
  (let ((orig-rule (original-rule base-rule))
        (orig-antes (loop for ante in (rule-antecedents base-rule)
                          unless (antecedent-abduced? ante)
                          collect (antecedent-prop ante)))
        old-mc base-rule base-antes new-type)
    (if (setf old-mc (base-is-added-rule orig-rule))
        (setf base-rule (model-change-rule old-mc)
              base-antes (model-change-antecedents old-mc)
              new-type 'add-rule)
        (if (setf old-mc (base-from-adding-antes orig-rule))
            (setf base-rule (model-change-rule old-mc)
                  base-antes (model-change-antecedents old-mc)
                  new-type 'add-ante)
            (setf base-rule orig-rule
                  base-antes nil
                  new-type 'add-ante)))
    (if old-mc (setf *student-model* (delete old-mc *student-model*)))
    (record-tree base-rule base-antes new-type new-tree orig-antes)))

              

(defun record-tree (base-rule base-antes new-mc-type new-tree old-antes)
  ;;-------------------------------------------------------------------------
  ;; This routine is a special subroutine for revisions which add antecedents
  ;; to a base rule. Since the code must handle three different types of add
  ;; antecedent cases, this code is the essence of the addition, with the
  ;; the decisions about what goes in each field of the model-change struct
  ;; made by the caller. What this routine does is create a new model-change
  ;; struct for every child of the new-tree. The "type" field is set to new-
  ;; mc-type, the "rule" field is set to base-rule, and the "antecedents" 
  ;; field is set by appending base-antes with anything in the child that is
  ;; not part of old-antes.
  ;;-------------------------------------------------------------------------
  (loop for vert in new-tree do
        (loop for child in (vertex-children vert) do
              (add-to-model
               (make-model-change
                :type new-mc-type :rule base-rule
                :antecedents
                  (append base-antes
                    (loop for a in (rule-antecedents child)
                          for prop = (antecedent-prop a)
                          unless (member prop old-antes :test #'equal)
                          collect prop)))))))


(defun base-is-added-rule (base)
  "Takes a base rule (in list format, with no threshold) and determines if it
is present in the student model in the form of an added rule. If so, that 
entry of the student model is returned. Otherwise, nil is returned."
  ;;-------------------------------------------------------------------------
  ;; The antecedents of the base rule passed in are checked against the 
  ;; antecedents field of all the structures in the student model. If there 
  ;; is ever an exact match, then the antes being added by the calling 
  ;; routine are being added to a rule which was generated during 
  ;; generalization. In this case, return that mc; else return nil.
  ;;-------------------------------------------------------------------------
  (loop for mc in *student-model*
	with rule-antes = (cddr base)
	finally (return nil)
	if (and (eq (model-change-type mc) 'add-rule)
                (equal rule-antes (model-change-antecedents mc)))
	return mc))


(defun base-from-adding-antes (base)
  "Takes a base rule (in list format, with no threshold), and determines if 
the rule is present in the student model in the form of a rule with added 
antecedents. If so, that entry is returned. Otherwise, nil is returned."
  ;;-------------------------------------------------------------------------
  ;; The antecedents of the base rule passed in are checked against the the 
  ;; COMBINED antecedents in the original rule and the antecedents field for 
  ;; all the structures in the student model. If there is ever an exact 
  ;; match, then the antes being added by the calling routine are being added
  ;; to a rule which already has added antecedents. In this case, return that
  ;; mc. Otherwise nil is returned.
  ;;-------------------------------------------------------------------------
  (loop for mc in *student-model*
        with rule-antes = (cddr base)
	finally (return nil)
	if (and (eq (model-change-type mc) 'add-ante)
                (rule-plus-antes-match mc rule-antes))
        return mc))


(defun rule-plus-antes-match (mc ante-list)
  ;;-------------------------------------------------------------------------
  ;; returns t if the sequence of antecedents in the rule in mc FOLLOWED BY 
  ;; the sequence of antecedents in the antecedents list of the mc exactly 
  ;; matches the sequence of antecedents in ante-list.
  ;;-------------------------------------------------------------------------
  (if (loop for ante in (cddr (model-change-rule mc))
            finally (return t)
            if (not (equal ante (car ante-list))) return nil
            do (setf ante-list (cdr ante-list)))
      (loop for ante in (model-change-antecedents mc)
            finally (return t)
            if (not (equal ante (car ante-list))) return nil
            do (setf ante-list (cdr ante-list)))
      nil))


(defun record-threshold-decrease ())
;; not implemented yet. 


;;===========================================================================
;; STUDENT MODEL PRINTING ROUTINES
;;===========================================================================

(defun ppmodel (&optional (model *student-model*))
  ;;-------------------------------------------------------------------------
  ;; Prints the model passed in to the screen, one revision at a time.
  ;;-------------------------------------------------------------------------
  (loop for m in model
        for type = (model-change-type m)
        do
        (format t "~%~A:"
                (ecase type
                  (del-ante "DELETE ANTECEDENT(S)")
                  (add-rule "ADD RULE")
                  (del-rule "DELETE RULE")
                  (add-ante "ADD ANTECEDENT(S)")))
        (format t "~%  original rule = ~A" (model-change-rule m))
        (format t "~%  antecedents = ~A" (model-change-antecedents m))))


(defun list-model (&optional (model *student-model*))
  "Returns the student model in a list format. Each element of the list is a
list of the elements in the model-change structure, listed in the order they
appear in the structures.lisp specification."
  ;;-------------------------------------------------------------------------
  ;; Prints out the structures in model by simply returning lists of lists,
  ;; where each element of the outer list is a list representing the contents
  ;; of a model-change structure. Each sublist shows the contents.
  ;;-------------------------------------------------------------------------
  (loop for mc in model
	collect (list (model-change-type mc)
		      (model-change-rule mc)
		      (model-change-delta-sum mc)
		      (model-change-antecedents mc)
		      (model-change-gensym-antes mc)
		      (model-change-threshold mc))))


(defun write-student-model (model-name file-name
				       &optional (model *student-model*))
  ;;-------------------------------------------------------------------------
  ;; Writes the contents of *student-model* to the file "file-name" using the
  ;; symbol in "model-name" as the name of the model. If the file does not
  ;; yet exist, this routine will also print the appropriate header so that
  ;; the file is automatically setup with the common lisp user package. Also
  ;; writes out a "setf" instruction" to update the value of a global var-
  ;; iable in the file called "*models-list*" which, when the file is loaded,
  ;; will hold all the student models.
  ;;-------------------------------------------------------------------------
  (let ((file-exists (probe-file file-name))
	outfile)
    (setf outfile (open file-name :direction :output :if-exists :append
			:if-does-not-exist :create))
    (when (not file-exists)
      (format outfile
              "~%(in-package #+:cltl2 ~cCL-USER~c #-:cltl2 ~cUSER~c)"
              #\" #\" #\" #\")
      (format outfile "~%~%(defvar *models-list* nil)")
      (format outfile "~%(setf *models-list* nil)"))
    (format outfile "~%~%(setf ~A~%      '(" model-name)
    (loop for mc in (list-model model) do
	  (format outfile "~%        ~A" mc))
    (format outfile "~%        ))")
    (format outfile "~%(setf *models-list* (cons ~A *models-list*))"
            model-name)
    (close outfile)))


;;===========================================================================
;; COMMON MODEL ROUTINES
;;
;; The code below is used for finding a subset of retractions from among all 
;; the retractions given in a list of student models. The idea is to find the
;; largest most common set of antecedents that will move the common student 
;; model closest to the centroid position (ie, will minimize the sum of all 
;; model deltas). 
;;
;; VOCABULARY DEFINITIONS
;; ----------------------
;;
;; - REVISION: this is a change applied to a theory. A revision can fix one 
;;   or more examples.
;;
;; - FIX: a revision that fixes at least one example.
;;
;; - STUDENT MODEL or MODEL: a set of fixes applied to some starting theory 
;;   T that make the revised theory T' consistent with the examples taken 
;;   from the student's interaction with the system.
;;
;; - STUDENT MODEL DISTANCE, MODEL DISTANCE, or DISTANCE: In the space of all 
;;   possible theories, the distance between two theories can be measured in 
;;   terms of the number of literal additions and deletions required to 
;;   transform one theory into another.
;;
;; - BASE THEORY: for a given set of student models, the base theory is the 
;;   theory used to compute all the student model distances. 
;;
;; - DISTANCE SUM: the sum of all student model distances.
;;
;; - COMMON STUDENT MODEL, CENTROID MODEL, STEREOTYPICAL STUDENT MODEL: the 
;;   base theory that minimizes sum distance. 
;;
;; - STUDENT MODEL DELTA, MODEL DELTA, DISTANCE DELTA, or DELTA: the change 
;;   in student model distance from a theory T when that theory is modified 
;;   by adding a revision. Let T be the original theory and T' be a modified 
;;   theory with the revision added. Let M be the theory associated with the 
;;   student model. Model delta is defined as 
;;
;;        model delta = distance(M, T') - distance(M, T)
;;
;;   where "distance" returns the distance between its two arguments.
;;
;; - DELTA SUM: the sum of all student model deltas for a given revision. 
;;
;; - BENEFICIAL REVISION: a revision which has a negative delta sum, i.e., 
;;   decreases the distance sum.
;;
;; GENERAL ALGORITHM
;; -----------------
;; Given a set of additions for a given rule, we want the revision which is a 
;; subset of the set of all additions that minimizes sum distance. One 
;; obvious approach is to compute the delta for each individual rule. However,
;; it is quite likely that no rule addition will be a beneficial revision; it
;; may be that only subsets of each rule are beneficial. The idea, then, is 
;; to find some rule to add that is beneficial, update the base theory with 
;; it, and repeat. This is your basic hill-climbing search, just like ID3, 
;; FOIL and other systems use. After each beneficial revision is found, the 
;; base theory is modified with it, which changes the deltas for each 
;; remaining revision.
;;
;; The trick is to find a good method for constructing a rule to add for each
;; iteration of the hill-climbing. One way to do this is to use hill-climbing
;; search again. The idea is to take all potential revisions, sort them by 
;; delta sum, and start with the best of the lot. Call that the current 
;; hypothesis revision (ie, the one we're currently hypothesizing as an 
;; addition to the base theory). Enter a loop which (a) forms the lgg of the
;; hypothesis with all revisions (b) recomputes the delta sum for each new 
;; lgg (c) tests to see if any new lgg has a lower delta sum than the 
;; hypothesis. If so, update the hypothesis. Repeat until no cycle produces
;; any lgg with a better delta sum.
;;
;; Once out of the above loop, check if the hypothesis has a negative delta 
;; sum (it may not, even after all the lgg's). If so, then add the hypothesis
;; to the theory, update the delta sum values for all the revisions and 
;; repeat. Continue until no revision produces a hypothesis with a negative
;; delta sum.
;;===========================================================================

;;===========================================================================
;; GLOBAL VARIABLES
;;
;; Below are the global variable definitions and references used by the model
;; building code. They are each defined in turn.
;;
;; DEFINITIONS
;; -----------
;; *revisions* 
;;   used by the code to build the common model. It is a list of all the 
;;   revisions from all the student models stored in a particular student 
;;   model file. Each revision is stored as a model-change structure.
;;
;; *models*
;;   a list of the student models. Each element of this list is a list of
;;   model-change structures representing the revisions for that student.
;;
;; REFERENCES
;; ----------
;; *models-list*
;;   This variable is defined and set by the file which contains the student
;;   models. They are stored in list format (see write-student-model routine)
;;===========================================================================
(defvar *revisions* nil
  "List of all revisions from all student models. Stored as a list of model
change structures.")

(defvar *models* nil
  "List of the student models. Stored as a list of model-change structure 
lists.")

#-:cltl2
(progn (proclaim '(special *models-list*)))
#+:cltl2
(declaim (special *models-list*))


(defun prepare-models-and-revisions (file-name &optional (pick-models nil))
  ;;-------------------------------------------------------------------------
  ;; Loads the file "file-name" which is assumed to be a file consisting of
  ;; student models written in the format shown in the "write-student-model"
  ;; routine above. Note that this file is assumed to have set the *models-
  ;; list* variable when loaded.
  ;; The idea is to loop through the models in *models-list*, creating a 
  ;; model-change structure for each revision in each model. These are linked
  ;; into one list and stored in *revisions-list*
  ;;-------------------------------------------------------------------------
  (setf *revisions* nil)
  (setf *models* nil)
  (load file-name)
  (if pick-models
      (setf *models-list*
	    (loop for i in pick-models collect (nth i *models-list*))))
  (loop for model in *models-list* do
        (push 
         (loop for rev in model
               collect
               (progn
                 (unless (revision-present rev)
                   (push (make-model-change 
                          :type (first rev) :rule (second rev)
                          :delta-sum (third rev) :antecedents (fourth rev)
                          :gensym-antes (fifth rev) :threshold (sixth rev))
                         *revisions*))
                 (make-model-change 
                  :type (first rev) :rule (second rev)
                  :delta-sum (third rev) :antecedents (fourth rev)
                  :gensym-antes (fifth rev) :threshold (sixth rev))))
         *models*)))


(defun revision-present (rev)
  ;;-------------------------------------------------------------------------
  ;; Checks to see if "rev" is already present in the *revisions* list of
  ;; model-change structures. If the rule, type and antecedents of rev match
  ;; some element of *revisions*, then it is already present.
  ;;-------------------------------------------------------------------------
  (member rev *revisions*
          :test #'(lambda (r l)
                    (and (eq (first r) (model-change-type l))
                         (equal (second r) (model-change-rule l))
                         (equal (fourth r) (model-change-antecedents l))))))


(defun common-revisions (&optional (match-percentage .50))
  ;;-------------------------------------------------------------------------
  ;; The return value for this routine is a list of all the revisions which,
  ;; if added to the theory from which all the student models in "models"
  ;; were formed, would create the centroid model. Said another way, the list
  ;; of revisions returned is one which represents the typical revisions of
  ;; all the models. Only those revisions which are beneficial will be added
  ;; to the return value.
  ;;
  ;; Below I have the pseudo code for this routine, which outlines the
  ;; algorithm in a slightly different way than is actually done here. The
  ;; pseudo code is easier to understand conceptually. The main difference
  ;; between the pseudo code and the real code is that I do not actually keep
  ;; track of the original theory and the student theories; instead, I just
  ;; keep the revisions from each model and compute the delta-sum based on
  ;; how it differs from the revisions for a given model. The result is the
  ;; same, but I store a lot less. Also, I do not have a "correct theory"
  ;; anywhere in the code. Instead, I just return the revisions which should
  ;; be applied to the correct theory to convert it into the centroid model.
  ;;
  ;; PSEUDO CODE FOR CENTROID MODEL CONSTRUCTION
  ;; -------------------------------------------
  ;; Let R = set of all revisions from all student models
  ;;     M = list of all student model theories
  ;;     C = common student model
  ;;     H = current best hypothesis for updating C
  ;;
  ;; 1. initialize C = correct theory
  ;; 2. sort R by DELTA-SUM(r,C,M) for all r in R 
  ;; 3. for all r in R
  ;;      set H = r
  ;;      repeat while H still improving
  ;;        set L = LGG-LIST(H,R)
  ;;        compute DELTA-SUM(l,C,M) for all l in L
  ;;        if best of L has lower delta sum than H then
  ;;          H = best of L
  ;;        end-if
  ;;      end-repeat
  ;;      if delta sum of H is negative then
  ;;        update C by adding H
  ;;        quit step 3; backup to step 2.
  ;;      end-if
  ;;    end-for
  ;; 4. return C
  ;;-------------------------------------------------------------------------
  (let ((N (length *models*))
        centroid-revisions hypo)
    (loop with adding-to-common-model = t
	  while adding-to-common-model
	  do
	  ;; compute the delta sum for each revision
          (compute-delta-sums)

	  ;; sort revision by LOWEST (i.e., best) delta sum
          (sort-by-delta-sums)

	  ;;(if (y-or-n-p "show all revision delta sums?")
	  ;;    (loop for r in *revisions* do
	  ;;	    (format t "~%<R> type:~A  delta-sum:~D~%    rule:~A~%    antecedents:~A"
	  ;;		    (model-change-type r) (model-change-delta-sum r)
	  ;;		    (model-change-rule r) (model-change-antecedents r))))
	  ;;(when (y-or-n-p "show all models?")
	  ;;  (format t "~%~% MODELS ARE:")
	  ;;  (loop for m in *models* do
	  ;;	  (loop for r in m
	  ;;		with count = 0
	  ;;		do
	  ;;		(format t "~%<R~D> type:~A  delta-sum:~D~%    rule:~A~%    antecedents:~A"
	  ;;			count
	  ;;			(model-change-type r) (model-change-delta-sum r)
	  ;;			(model-change-rule r) (model-change-antecedents r))
	  ;;		(incf count))))

	  (loop with testing-revisions = t
		while testing-revisions
		for r in *revisions*
		finally (if testing-revisions
			    (setf adding-to-common-model nil))
		do
		(setf hypo
		      (make-model-change :type (model-change-type r)
					 :rule (model-change-rule r)
					 :delta-sum (model-change-delta-sum r)
					 :antecedents (model-change-antecedents r)))
		(when *trace-common-model*
		  (format t "~%~%~%Seed Hypothesis:")
		  (format t "~%  rule:~A~%  delta sum:~D~%  type:~A~%  antes:~A"
			  (model-change-rule hypo) (model-change-delta-sum hypo)
			  (model-change-type hypo) (model-change-antecedents hypo)))
                (setf hypo (find-best-hypo hypo))
                (when (good-hypothesis hypo N match-percentage)
                  (push hypo centroid-revisions)
		  (update-models hypo)
		  (setf testing-revisions nil))))
    centroid-revisions))


(defun compute-delta-sums ()
  (loop for r in *revisions* do
        (setf (model-change-delta-sum r) (delta-sum r))))


(defun sort-by-delta-sums ()
  (setf *revisions* (sort *revisions* #'< :key #'model-change-delta-sum)))


(defun good-hypothesis (hypo N match-percentage)
  ;;-------------------------------------------------------------------------
  (let ((L (if (eq 'del-rule (model-change-type hypo))
               (1- (length (model-change-rule hypo)))
               (length (model-change-antecedents hypo)))))
    (>= (/ (- (* N L) (model-change-delta-sum hypo)) (* L (- (* 2 N) 2)))
        match-percentage)))


(defun find-best-hypo (cur-hypo)
  ;;-------------------------------------------------------------------------
  ;;-------------------------------------------------------------------------
  (loop with hypo-improving = t
        while hypo-improving
        for temp-hypo = (best-lgg cur-hypo)
        finally (return cur-hypo)
        do
        ;;(format t "~%  temp hypo: ~A" temp-hypo)
        (if (and temp-hypo (< (model-change-delta-sum temp-hypo)
                              (model-change-delta-sum cur-hypo)))
            (setf cur-hypo temp-hypo)
            (setf hypo-improving nil))
        (when *trace-common-model*
          (format t "~%New Hypothesis:")
          (format t "~%  rule:~A~%  delta sum:~D~%  type:~A~%  antes:~A"
                  (model-change-rule cur-hypo) 
                  (model-change-delta-sum cur-hypo)
                  (model-change-type cur-hypo) 
                  (model-change-antecedents cur-hypo)))))


(defun delta-sum (revision)
  ;;-------------------------------------------------------------------------
  ;; *** NOTE that the "revision" passed in here is a MODEL-CHANGE structure,
  ;; NOT a revision structure. Computes the benefit (or cost!) of putting the
  ;; "revision" into the centroid model. A beneficial revision is one that 
  ;; results in a reduction of cost, and would be a negative number returned
  ;; from this routine. Positive return values mean that the revision would
  ;; push the centroid model farther away from the average student model.
  ;;-------------------------------------------------------------------------
  (loop for m in *models*
        for overlaps = (collect-like-revisions revision m)
        sum 
        (ecase (model-change-type revision)
          (add-ante (best-match revision overlaps))
          (del-ante (del-ante-delta revision overlaps))
          (add-rule (best-match revision overlaps))
          (del-rule (del-rule-delta revision overlaps)))))


(defun collect-like-revisions (revision model)
  ;;-------------------------------------------------------------------------
  ;; Collects the revisions in "model" which have the same type as "revision"
  ;; and operate on the same base rule.
  ;;-------------------------------------------------------------------------
  (loop for rev in model
        with type = (model-change-type revision)
        with rule = (model-change-rule revision)
        if (and (eq (model-change-type rev) type)
                (equal (model-change-rule rev) rule))
        collect rev))


(defun best-match (revision overlaps)
  ;;-------------------------------------------------------------------------
  ;; Given a revision and a list of overlapping revisions from a student
  ;; model, this routine will find the single overlap which produces the best
  ;; "match" to the revision. Match values are based on how much savings the
  ;; match will yield in the delta sum. Savings are negative values, costs 
  ;; are positive. For every antecedent in the revision that matches an ante
  ;; in an overlap, the savings is increased by one (ie, + -1). For every
  ;; ante in the revision NOT in the overlap, there is a cost of one. 
  ;;
  ;; Returns two values: the cost savings and the overlap found, if any. Note
  ;; that the default return value is all cost (ie, no match) which is just
  ;; the length of the revision's antecedent list.
  ;;-------------------------------------------------------------------------
  (loop for ovlp in overlaps
        with proposed = (model-change-antecedents revision)
        with bestvalue = (length proposed)
        with bestmatch = nil
        with temp = 0
        for needed = (model-change-antecedents ovlp)
        finally (return (values bestvalue bestmatch))
        do
        (setf temp (loop for p in proposed
                         sum (if (member p needed :test #'equal) -1 1)))
        (when (< temp bestvalue)
          (setf bestvalue temp
                bestmatch ovlp))))


(defun del-ante-delta (revision overlaps)
  ;;-------------------------------------------------------------------------
  ;; For antecedent deletions, let the deletions in the proposed revision be
  ;; called proposed deletions (PD). Let the corresponding deletions from the
  ;; overlaps in the student model be called the needed deletions (ND). Note 
  ;; that ND is the COMBINED deletions for all revisions in the student model
  ;; that delete antecedents from the given rule. For every element of PD 
  ;; which is also a member of ND, the delta improves by one (ie, -1). For 
  ;; every member of PD deletions NOT in ND, the delta gets one worse (+1). 
  ;; Any elements of ND which are not in proposed deletions are a wash. The 
  ;; base delta is worse by (length PD).
  ;;
  ;; Here, PD is the list of antecedents from the revision, and the ND is the
  ;; combined list of antecedents from all the elements of the overlaps.
  ;;-------------------------------------------------------------------------
  (if overlaps
    (loop for ante in (model-change-antecedents revision)
          with needed = (loop for ovlp in overlaps
                              append (model-change-antecedents ovlp))
          sum (if (member ante needed :test #'equal) -1 1))
    (length (model-change-antecedents revision))))


(defun del-rule-delta (revision overlaps)
  ;;-------------------------------------------------------------------------
  ;; For deleted rules, cost is easy. If the student model also deletes the
  ;; rule, the cost benefit is the length of the deleted rule. Otherwise the
  ;; cost is worse by the length. Recall that each revision stores a copy of 
  ;; the rule being deleted in the "rule" field of the model-change struct.
  ;; The rule is guaranteed NOT to have a threshold, but it does have the 
  ;; "<-" implication. So, the easiest way to find rule length is to take the
  ;; length of the rule field and subtract 1.
  ;;-------------------------------------------------------------------------
  (if overlaps 
      (* -1 (1- (length (model-change-rule revision))))
      (1- (length (model-change-rule revision)))))


(defun update-models (revision)
  ;;-------------------------------------------------------------------------
  ;; Updates student model "model" to reflect the fact that the centroid 
  ;; model has been altered by adding "revision". This amounts to deleting 
  ;; anything from among the revisions of the model that overlaps with the 
  ;; elements of revision. This must be done to avoid counting the same 
  ;; revision of a given model in multiple delta-sum calculations.
  ;;
  ;; Assumes the models to be updated are stored in the *models* global
  ;; variable.
  ;;-------------------------------------------------------------------------
  (setf *models*
	(loop for m in *models* collect
	      (ecase (model-change-type revision)
		(add-ante (remove-best-match revision m))
		(del-ante (del-ante-modify revision m))
		(add-rule (remove-best-match revision m))
		(del-rule (del-rule-modify revision m)))))
  (setf *revisions*
        (delete revision *revisions*
                :test #'(lambda (r l) 
                          (and (eq (model-change-type r)
                                   (model-change-type l))
                               (equal (model-change-rule r)
                                      (model-change-rule l))
                               (equal (model-change-antecedents r)
                                      (model-change-antecedents l)))))))


(defun remove-best-match (revision model)
  ;;-------------------------------------------------------------------------
  ;; Loops through all the revisions of the model which overlap with revision
  ;; finding the one which is the best match to the revision. If the best 
  ;; match is better than the default cost for adding the revision (ie, cost 
  ;; with no match) then this routine removes that revision from the model.
  ;; Returns the revised version of the model, which changes only the local
  ;; copy of model (so caller will most likely need to do a setf).
  ;;-------------------------------------------------------------------------
  (multiple-value-bind (delta bestmatch)
                       (best-match revision
                          (collect-like-revisions revision model))
    (declare (ignore delta))
    ;;(format t "~%** delta ~D bestmatch = ~A" delta bestmatch)
    ;;(format t "~%  member test = ~A" (if (member bestmatch model :test #'eq) t nil))
    
    (when bestmatch
      (setf model (delete bestmatch model :test #'eq)))
    
    ;;(when (y-or-n-p "See the model being returned?")
    ;;  (loop for r in model
    ;;    with count = 0
    ;;	    do
    ;;	    (format t "~%<R~D> type:~A  delta-sum:~D~%    rule:~A~%    antecedents:~A"
    ;;		    count
    ;;		    (model-change-type r) (model-change-delta-sum r)
    ;;		    (model-change-rule r) (model-change-antecedents r))
    ;;	    (incf count)))
    
    model))


(defun del-ante-modify (revision model)
  ;;-------------------------------------------------------------------------
  ;; Loops through all revisions in "model" which overlap with "revision" and
  ;; removes any matching antecedents.
  ;;-------------------------------------------------------------------------
  (loop for ovlp in (collect-like-revisions revision model)
	with proposed = (model-change-antecedents revision)
	finally (return model)
	do
	(loop for p in proposed do
	      (setf (model-change-antecedents ovlp)
		    (delete p (model-change-antecedents ovlp))))
        (when (null (model-change-antecedents ovlp))
          (setf model (delete ovlp model :test #'eq)))
        ))
	

(defun del-rule-modify (revision model)
  ;;-------------------------------------------------------------------------
  ;; Loop through the revisions in the model, deleting any which are
  ;; revisions that match "revision" in both type and base rule.
  ;;
  ;; Returns a revised version of the model, but does NOT destructively the
  ;; model. 
  ;;-------------------------------------------------------------------------
  (loop for ovlp in (collect-like-revisions revision model)
	finally (return model)
	do
	(setf model (delete ovlp model :test #'eq))))


(defun best-lgg (hypothesis)
  ;;-------------------------------------------------------------------------
  ;; Finds the best "lgg" of the hypothesis and one of the elements of
  ;; revisions. "lgg" stands for "least general generalization" which, simply
  ;; put, is the most specific set of components that the two revisions have
  ;; in common. For propositional logic, this is easy to form: simply take
  ;; the subset of the two items in question. This work is done by the "calc-
  ;; lgg" routine below. From all possible lgg's (of the hypothesis with one
  ;; of the elements of revisions) this routine selects the best and returns
  ;; it. If no lgg's can be found at all (i.e., all calls to calc-lgg return
  ;; nil), then nil is returned.
  ;;-------------------------------------------------------------------------
  ;;(format t "~%<in best-lgg>~%  for hypo: ~A~%" hypothesis)
  (loop for r in *revisions*
	for lgg = (calc-lgg hypothesis r)
	with best-lgg = nil
	finally (return best-lgg)
	do
	(when lgg
	  (setf (model-change-delta-sum lgg) (delta-sum lgg))
	  ;;(format t "~%  Found lgg = ~A" lgg)
	  (if (or (null best-lgg) (< (model-change-delta-sum lgg)
				     (model-change-delta-sum best-lgg)))
	      (setf best-lgg lgg))
	  ;;(format t "~%  best-lgg  = ~A" best-lgg)
	  )))


(defun calc-lgg (rev1 rev2)
  ;;-------------------------------------------------------------------------
  ;; Computes the lgg of the two revisions, returning a new model-change
  ;; structure or NIL. If the two revisions do not match in both type and
  ;; base rule, then NIL is returned. Otherwise, the two revisions are
  ;; searched for any common elements in their "antecedents" field. If any
  ;; exist, a new model-change structure is created. If not, NIL is returned.
  ;;-------------------------------------------------------------------------
  (if (and (eq (model-change-type rev1)
		(model-change-type rev2))
	   (equal (model-change-rule rev1)
		  (model-change-rule rev2)))
      (loop for ante in (model-change-antecedents rev1)
	    with others = (model-change-antecedents rev2)
	    with result = nil
	    finally
	    (return (if (or result (eq (model-change-type rev1) 'del-rule))
			(make-model-change :type (model-change-type rev1)
					   :rule (model-change-rule rev1)
					   :antecedents result)
			nil))
	    do
	    (if (member ante others :test #'equal)
		(push ante result)))
      nil))
				 

;;===========================================================================
;; MODEL INCORPORATION ROUTINES
;;===========================================================================

(defun add-model-to-theory (revisions &optional (theory *theory*))
  "Takes a list of model change structures and an optional theory and creates
a new theory which incorporates the revisions."
  ;;-------------------------------------------------------------------------
  ;; Creates a new theory by looping through the old theory, copying any
  ;; rules which are unchanged, and replacing any rules with the new versions
  ;; suggested by the revisions.
  ;;
  ;; Works as follows. First, sort all the revisions by rule and type. These
  ;; get stored in model-apply structures (see structures.lisp). Then, loop
  ;; through each rule of the old theory, copying it straight out if there is
  ;; no corresponding rule in the model-apply structures. If there is a
  ;; matching rule, then write out the rule(s) based on each possible type of
  ;; revision.
  ;;-------------------------------------------------------------------------
  (loop for r in theory
	with changes = (group-revisions revisions)
	for alter = (member r changes :test #'equal :key #'model-apply-rule)
	append
	(if alter
	    (altered-rules (car alter) changes)
	    (list (copy-list r)))))


(defun group-revisions (revisions)
  ;;-------------------------------------------------------------------------
  ;; Loops through the revisions creating a list of model-apply structures
  ;; with all revisions for a given rule grouped together by type.
  ;;-------------------------------------------------------------------------
  (loop for r in revisions
	for rule = (model-change-rule r)
	for type = (model-change-type r)
	for c = (member rule changes :test #'equal :key #'model-apply-rule)
	with changes = nil
	finally (return changes)
	do
	(ecase type
	  (add-rule
	   (if c
	       (push r (model-apply-add-rules (car c)))
	       (push (make-model-apply :rule rule :add-rules (list r))
		     changes)))
	  (del-rule
	   (if c
	       (push r (model-apply-del-rules (car c)))
	       (push (make-model-apply :rule rule :del-rules (list r))
		     changes)))
	  (add-ante
	   (if c
	       (push r (model-apply-add-antes (car c)))
	       (push (make-model-apply :rule rule :add-antes (list r))
		     changes)))
	  (del-ante
	   (if c
	       (push r (model-apply-del-antes (car c)))
	       (push (make-model-apply :rule rule :del-antes (list r))
		     changes))))))


(defun altered-rules (alteration all-changes)
  ;;-------------------------------------------------------------------------
  ;; The incoming argument is a model-apply structure (see structures.lisp)
  ;; which contains one or more revisions to be applied to a given rule. This
  ;; routine loops through the types of revisions (by looking at each field
  ;; of the model-apply structure) and creates the appropriate rule for the
  ;; revisions stored in that field. The return value is a list of all the
  ;; rules created for the alteration.
  ;;
  ;; 15-APR-93 (ptb) I needed to change this routine to fix a bug. Note that 
  ;; since generalization comes before specialization, I needed to check to
  ;; see if any antecedents were deleted, and make those deletions, before 
  ;; processing any add-antecedent type modifications. Otherwise, I could
  ;; get rules that were too specific (ie, needed an extra antecedent deleted
  ;; from the generalization phase).
  ;;
  ;; 15-APR-93 (ptb) Another bug found. I must check to see that a rule being
  ;; added is not later deleted (see comments below). To make this check, I
  ;; had to pass in the list of all alterations via "all-changes".
  ;;-------------------------------------------------------------------------
  (let ((rule (copy-list (model-apply-rule alteration))))
    (append
     ;; add in all added rules. Note, however, that is is possible to have a
     ;; student model which says to add a rule and then later says to delete
     ;; that rule. I leave both such alterations in the model (see comments
     ;; in record rule deletions) to give the best chances of finding common
     ;; revisions. Here, I must check to see that the rule being added is not
     ;; also being deleted.
     (when (model-apply-add-rules alteration)
       (loop for mc in (model-apply-add-rules alteration)
	     for ante-list = (copy-list (model-change-antecedents mc))
	     with conseq = (first (model-apply-rule alteration))
             for rule-to-add = `(,conseq <- ,@ante-list)
             unless (rule-is-deleted rule-to-add all-changes)
	     collect rule-to-add))

     ;; modify the base rule defined locally to reflect any ante deletions
     (when (model-apply-del-antes alteration)
       (loop for mc in (model-apply-del-antes alteration)
	     for ante-list = (model-change-antecedents mc)
	     do
	     (loop for ante in ante-list do
		   (setf rule (delete ante rule :test #'equal)))))

     ;; check if there are new rules with added antecedents. If so, add in 
     ;; each new one using the rule defined locally as the base. If not, then
     ;; just copy out the base rule as is, as long as it was not deleted.
     (if (model-apply-add-antes alteration)
         (loop for mc in (model-apply-add-antes alteration)
	       for ante-list = (copy-list (model-change-antecedents mc))
	       for base-rule = (copy-list rule)
	       collect
	       `(,@base-rule ,@ante-list))
         (unless (model-apply-del-rules alteration)
           (list rule))))))


(defun rule-is-deleted (rule all-changes)
  ;;-------------------------------------------------------------------------
  ;; Written to fix a bug when deciding which added rules to include in the
  ;; new theory. Called by altered-rules to check if a rule to be added, 
  ;; passed in here as "rule", is not also deleted.
  ;;-------------------------------------------------------------------------
  (loop for alt in all-changes
        for alt-rule = (model-apply-rule alt)
        finally (return nil)
        if (and (model-apply-del-rules alt) (equal rule alt-rule))
        return t))
