
;;===========================================================================
;; Covering algorithm for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; This file sets up four global variables, which are used by the ante-cover
;; and rule-cover code to build the antecedent and rule covers for NEITHER.
;; Expects *neither-theory* and *neither-examples* to have been initialized
;; by "prepare-theory-and-examples".
;;
;; There are many ways to label example, either as "false-positives"
;; "failing-negatives" or "unprovable-positives" etc. The various labeling
;; schemes can be confusing, so here is an outline of what we use here.
;;
;; (1) POSITIVE/NEGATIVE: the term "positive" refers to how the EXAMPLE is
;; labeled, not to whether or not the current theory proves the example. Thus
;; if we had three categories, "a" "b" and "c", and an example like (ex-1 a),
;; we would label example "ex-1" POSITIVE for "a" and NEGATIVE for "b" and
;; "c". 
;;
;; (2) PROVABLE/UNPROVABLE: "provable" means the theory can prove the
;; category given the example, "unprovable" means it cannot.
;;
;; So, we have four types of labels we can use. "provable positives" are
;; examples which should and do prove a particular category. "unprovable
;; positives" are examples which should, but do NOT, prove a particular
;; category (same as false negatives). "provable negatives" are examples
;; which do, but should NOT, prove a category for an example (same as false
;; positives). Finally, "unprovable negatives" are examples which should not
;; and do not prove a category with an example.
;;
;; CHANGE HISTORY
;;
;; 6-23-92: (ptb) Changed some of the terminology in this file to make it
;;          clearer. 
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;---------------------------------------------------------------------------
;; GLOBAL VARIABLES
;;---------------------------------------------------------------------------
;; The four globals below are used by most of the routines that build the
;; rule and antecedent covers. The format of each (after the "label-examples"
;; routine is called) is as follows:
;;
;; *provable-positives*
;;   A list of examples. Each example is the same format as stored in
;;   *neither-examples* (see "prepare-theory-and-examples")
;;
;; *uprovable-positives*
;;   A list of examples. These are the positives which could not be proven.
;;
;; *provable-negatives*
;;   A list of lists, each of which is a triple of the form
;;         (bad-category good-cat-provable-flag example)
;;   where the first element is the erroneously proved category, the second
;;   element indicates if the right category was provable, and the last
;;   element is the example.
;;
;; *provable-negative-alist*
;;   An alist, associating the false negatives by category.
;;---------------------------------------------------------------------------
#+:cltl2(declaim (special *negative-category*))
#-:cltl2(proclaim '(special *negative-category*))    ;; set up by io.lisp

(defvar *provable-positives* nil
  "A list of examples which should be, and are, provable by the theory")

(defvar *unprovable-positives* nil
  "The list of false negatives, ie, examples that could not be proved but
should be")

(defvar *provable-negatives* nil
  "The list of false positives, ie, examples that could be proved but should
not be") 

(defvar *provable-negative-alist* nil
  "An alist of the false positives, associated with their erroneous
categories")


;;===========================================================================
;; ROUTINES
;;===========================================================================


(defun label-examples (&optional (examples *neither-examples*) 
		       (theory *neither-theory*))
  "Proves `examples' using `theory' and initializes lists of false positives
and false negatives required by the antecedent-cover and rule-cover routines" 
  ;;-------------------------------------------------------------------------
  ;; Updates (initializes) the four global variables above to account for how
  ;; the input `theory' performs on the input `examples'. False positives are
  ;; examples for which the theory cannont prove the desired category, and
  ;; false negatives are examples for which the theory proves additional
  ;; undesirable categories.  
  ;;
  ;; In addition to the lists of false positives and false negatives, two
  ;; other globals are initialized. One is the list of true positives (used
  ;; later in the rule covering code) and the other is an alist for the false
  ;; negatives where categories are the keys and the datum is a list of all
  ;; the failing negatives which erroneously proved that category. 
  ;;
  ;; After initialization, this routine calls two "mapping" functions that
  ;; abduce the antecedent(s) to be deleted for each false positive, and the
  ;; rule(s) to be deleted for each false negative. 
  ;;-------------------------------------------------------------------------
  (initialize-cover-globals)
  (loop for ex in examples
	for desired-cat = (example-name ex)
	for proved-cats = (prove-categories ex theory)
	for bad-cats = (remove desired-cat proved-cats)
	for proved-good-cat = (if (member desired-cat proved-cats) t)
	do

	;; First, check to see if the positive part of the example (the
	;; desired category) was proved. Only put on unprovable positives
	;; list if examples was supposed to prove some non "'NEGATIVE"
	(if proved-good-cat
	    (setf *provable-positives* (cons ex *provable-positives*))
	    (unless (eq desired-cat *negative-category*)
	      (setf *unprovable-positives*
		    (cons ex *unprovable-positives*))))

	;; Next, for every remaining other category (the bad cats) put on the
	;; provable negatives list and alist.
	(unless (null bad-cats)
	  (loop for bc in bad-cats
		for loc = (assoc bc *provable-negative-alist*)
		do
		(setf *provable-negatives*
		      (cons (make-provable-neg :category bc
					       :flag proved-good-cat
					       :example ex)
			    *provable-negatives*))
		(rplacd loc (cons ex (cdr loc)))))))


(defun initialize-cover-globals ()
  "Used to initialize the global variables"
  ;;-------------------------------------------------------------------------
  ;; Initialization
  ;;-------------------------------------------------------------------------
  (setf *unprovable-positives* nil)
  (setf *provable-positives* nil)
  (setf *provable-negatives* nil)
  (setf *provable-negative-alist* (init-negative-alist)))


(defun init-negative-alist ()
  "Creates an initial alist for the failing negatives using the *categories*
global variable"
  ;;-------------------------------------------------------------------------
  ;; Used by `label-examples' to set up the initial alist for the provable
  ;; negatives.
  ;;------------------------------------------------------------------------- 
  (loop for cat in *categories*
	collect (list cat)))
