
;;===========================================================================
;; Generalization algorithm for NEITHER
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof.
;; -------------------------------------------------------------------------
;;
;; The code here is used to alter the theory until all unprovable positive
;; examples become provable. The result is a generalization of the theory,
;; thus the main routine here is "generalize". See structure definitions in
;; "structure.lisp".
;;
;; CHANGE HISTORY
;;
;; 08-SEP-92: (ptb) I moved the subset code into a separate file subset.lisp
;;            and modified this routine to redefine the is-covered-by? code.
;; 09-SEP-92: (ptb) Added code to postpone all inductions until the end of
;;            generalization.
;; 19-NOV-92: (ptb) Modified to work for m-of-n rules.
;; 03-JAN-93: (ptb) rewrote m-of-n stuff. Now uses thresholds directly which
;;            avoids post filters.
;; 14-JAN-93: (ptb) Modified induction to work with m-of-n rules.
;; 12-MAR-93: (ptb) modified to handle new rule structure field. Also changed
;;            the perrm-abductions routine to avoid making permanent changes.
;; 09-APR-93: (ptb) found a bug in the way the best deletion is selected for
;;            applying to the theory. During abduction, I am very careful to
;;            ensure that any fix for an example does NOT cause the same 
;;            example to be a new failing negative However, during the
;;            computation to see which fix covers the most failing positives
;;            there is not check to see if a fix creates this problem for a
;;            DIFFRENT failing positive So, I put in a fix between the two
;;            routines which calculate the positive fix value (see the code
;;            for calc-pos-fix-value).
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")
#+:cltl2(declaim (special *trace-genl*))
#-:cltl2(proclaim '(special *trace-genl*))

;;---------------------------------------------------------------------------
;; GLOBAL VARIABLES
;;
;; I use three globals here, one to keep track of all the original unprovable
;; positive examples, another to keep track of how many remain "uncovered"
;; for a given set of fixes applied to the theory. Not surprizingly, the 
;; variable *unprovable-positives* has a list of the unprovable positive 
;; examples. The other global, *remaining-positives*, is used to avoid 
;; changing *unprovable-positives* directly. "init-abduction-list" sets this 
;; second global by calling "abduce" first on the elements of 
;; *unprovable-positives* and then subsequently on the elements of
;; *remaining-positives*. The rest of the routines loop to generalize the 
;; theory, quiting when *remaining-positives* is nil.
;;
;; The format of *remaining-positives* is a list of SOLUTIONS. Each solution
;; is an example paired with the POSITIVE FIX that would allow the theory to
;; prove the example, eg:
;;
;;    *remaining-positives* = ( (fix-1 example-1)
;;                              (fix-2 example-2) ... )
;;
;; A positive fix is a LIST of DELETIONS (see structures.lisp).
;;
;; The *pos-induction-alist* global is used to keep all induction information
;; in one place until the end of generalization. It holds all the deletions
;; (plus their associated examples) which caused new failing negatives. It is
;; of the form: 
;;
;;   *pos-induction-alist* = ((rule-1
;;                              (deletion ex1 ex2 ...)
;;                              (deletion ex1 ex2 ...)
;;                              ...)
;;                            (rule-2
;;                              (deletion ex1 ex2 ...)
;;                              (deletion ex1 ex2 ...)
;;                              ...) )
;;
;; So, each deletion is assoc'ed with the rule to which it applies.
;; Furthermore, the deletion is stored with all the unprovable positive
;; examples that rely on the deletion. Note that the "rules" in the above 
;; alist are NOT vertex structures but members of the "children" field of a 
;; vertex (see structures.lisp).
;;---------------------------------------------------------------------------
(defvar *remaining-positives* nil
  "A list of solutions for the unprovable positive examples")

(defvar *pos-induction-alist* nil
  "An alist of the abductions which caused new negatives and thus must be
fixed by inducing a new rule.")



(defun generalize (&optional (m-of-n nil) (examples *neither-examples*)
			     (theory *neither-theory*))
  "Given examples and a theory, makes incremental modifications to the theory
until all unprovable positives (ie, false negatives) are provable (assumes
the label-examples routine has been called)."
  ;;-------------------------------------------------------------------------
  ;; Writing the algorithm as per discussion with Ray on 8/17/92. The new
  ;; idea is to generalize without building an antecedent cover or, if you
  ;; like, to generalize and build the cover at the same time. Here's the
  ;; basic idea: (1) Find from among the "fixes" for the remaining unprovable
  ;; positives that fix which is the "best." Then (2) modify the theory
  ;; DESTRUCTIVELY with that fix and (3) repeat until no unprovable positives 
  ;; are left.
  ;;-------------------------------------------------------------------------
  ;; first, get a fix for every unprovable positive
  (init-abduction-lists theory)

  ;; then, redefine the is-covered-by? routine for subset matching (see
  ;; subset.lisp)
  (redef-for-abductions)

  ;; continue until all remaining positives are covered. 
  (loop while *remaining-positives*
	for revision = (best-abduction examples theory m-of-n)
	for pure-thresholds = (try-pos-fix revision examples theory m-of-n)
	do
	(update-positives revision pure-thresholds theory m-of-n))

  ;; last, induce any new rules for revisions which could not be made to the
  ;; theory.
  (add-pos-inductions examples theory))


(defun init-abduction-lists (&optional (theory *neither-theory*))
  "Takes a list of unprovable positive examples and constructs a list of
those positives coupled with the antecedents which should be removed to make
each SINGLE example provable." 
  ;;-------------------------------------------------------------------------
  ;; Assumes the incoming *unprovable-positives* list is a list of examples
  ;; stored in the same format as *neither-examples*. Indeed, this list is a
  ;; subset of the *neither-examples*, thus its members should not be
  ;; altered. Builds a list of SOLUTIONS (see structures.lisp) each of which
  ;; is of the form (fix example) where a "fix" is a list of DELETIONS. The
  ;; new list is stored in *remaining-positives*.
  ;;-------------------------------------------------------------------------
  (setf *pos-induction-alist* nil)
  (setf *remaining-positives*
	(mapcar #'(lambda (unp)
		    (make-solution
		     :fix (abduce unp (example-name unp) theory)
		     :example unp))
		*unprovable-positives*)))


(defun redef-for-abductions ()
  "Redefines the is-covered-by? routine to work for abductions."
  ;;-------------------------------------------------------------------------
  ;; This routine redefines the "is-covered-by?" routine in the file
  ;; subset.lisp to compare two deletions. Here the deletion a1 is
  ;; considered to be "covered" by a2 in the routine below if two conditions
  ;; hold: (1) both deletions are applied to the same rule (ie, the first
  ;; elements of the two deletions are the same) and (2) all the antecedents
  ;; in a1 are members of the antecedent list for a2.
  ;;
  ;; NOTE: I can use "eq" here instead of the slower "equal" because the test
  ;; is whether each deletion refers to the same rule or antecedent. When we
  ;; compare rules or antecedents, we are looking for exact matches (ie,
  ;; references to the same structures in the tree, ie, the same memory
  ;; location).
  ;;-------------------------------------------------------------------------
  (setf (symbol-function 'is-covered-by?)
	#'(lambda (a1 a2)
	    (and (eq (deletion-parent a1) (deletion-parent a2))
		 (loop for ante in (deletion-antes a1)
		       if (not (member ante (deletion-antes a2) :test #'eq))
		       do (return nil)
		       finally (return t))))))


;;===========================================================================
;; FIX COMPARISON ROUTINES
;;
;; Used to select the best fix from among all the candidate fixes for the
;; unprovable positive examples in *remaining-positives*
;;===========================================================================

(defun best-abduction (&optional (examples *neither-examples*)
				 (theory *neither-theory*) (m-of-n nil))
  "Finds the example in *remaining-positives* whose fix covers the most
unprovable positives, creates the fewest new provable negatives, and requires
the fewest antecedent abductions. Returns a revision: the best fix found and
a sublist of *remaining-positives* containing the positives covered."
  ;;-------------------------------------------------------------------------
  ;; The idea here is to loop through all the unprovable positives, computing
  ;; some numerical measure of how "useful" it would be to abduce the
  ;; antecedents of the fix associated with the example. This metric combines
  ;; three elements: (1) how many unprovable positives are covered (2) how
  ;; many antecedents must be abduced and (3) how many new provable negatives
  ;; are created. 
  ;;
  ;; Returns a revision, which is the fix (see structures.lisp) for the
  ;; best example found grouped with a LIST of the unprovable positives
  ;; covered by the fix, ie: 
  ;;
  ;;    (fix ((fix example) (fix example) ...) )
  ;;          \_____________________________/
  ;;          sublist of *remaining-positives*
  ;;
  ;; NOTE: while this routine is operating, it stores a modified version of a
  ;; revision in "best" which has an extra element at the front for the
  ;; "score" of the revision. This score is compared for picking the best
  ;; revision.
  ;;-------------------------------------------------------------------------
  (loop for soln in (cdr *remaining-positives*)
	with best = (pos-fix-value (solution-fix (car *remaining-positives*))
				   examples theory m-of-n)
	with temp = nil
	finally (return (cdr best))
	do
	(unless (equal (solution-fix soln) (revision-fix (cdr best)))
	  (setf temp
		(pos-fix-value (solution-fix soln) examples theory m-of-n))
	  (if (> (car temp) (car best))
	      (setf best temp)))))


(defun pos-fix-value (fix &optional (examples *neither-examples*)
			  (theory *neither-theory*) (m-of-n nil))
  ;;-------------------------------------------------------------------------
  ;; Computes a "value" for the given fix according to a prescribed formula
  ;; (see calc-pos-fix-value) that is an attempt to measure the "goodness" of
  ;; the fix.
  ;;
  ;; inputs: "fix" is a list of deletion structures (ie, the first field of 
  ;; a solution structure). "examples" and "theory" are what you would expect
  ;; and "m-of-n" is a flag indicating whether or not to try changing 
  ;; threshold values as a revision.
  ;;
  ;; Works as follows. The basic idea is to make a temporary change to the
  ;; theory, compute the "goodness" of the fix by calling calc-pos-fix-value,
  ;; and then undo the changes to the theory. Now, there are two ways the 
  ;; theory can be changed: by changing thresholds or by performing the 
  ;; abductions. The former method is only computed if m-of-n changes are
  ;; turned on (as represented by the m-of-n flag). Thus this routine makes
  ;; the temporary abductions and calls calc-pos-fix-value which computes
  ;; the fix value and sets the "solutions" field of the revision structure.
  ;; Then, if m-of-n is set, the thresholds are changed and compute-fix-
  ;; value is invoked again, this time setting the "threshold-solutions" 
  ;; field of the revision structure.
  ;;
  ;; NOTE that the threshold changes are done AFTER the abduction changes. 
  ;; This ordering is important. The value returned by calc-pos-fix-value may
  ;; be different for threshold and abduction changes. Since this value is 
  ;; used by "best-abduction" (see above) to select the best fix, we need to
  ;; use the appropriate value. Because generalization must try threshold
  ;; revisions BEFORE abduction revisions, I take the optimistic view that
  ;; the threshold revisions will work and rank according to the threshold
  ;; change value. This means that abduction revisions may not be done in the
  ;; best order, but it is a heuristic decision. Note that if m-of-n revisons
  ;; are not used, then the second call to calc-pos-fix-value is never made 
  ;; and the abduction changes are ranked in the proper order.
  ;;-------------------------------------------------------------------------
  (let ((rev (make-revision :fix fix))
	fix-value)
    (dolist (ab fix) (set-abduced?-flags ab t))
    (setf fix-value (calc-pos-fix-value fix rev examples theory nil))
    (dolist (ab fix) (set-abduced?-flags ab nil))
    (when m-of-n
      (dolist (ab fix) (incf-threshold ab))
      (setf fix-value (calc-pos-fix-value fix rev examples theory t))
      (dolist (ab fix) (decf-threshold ab)))
    (cons fix-value rev)))
  

(defun calc-pos-fix-value (fix rev &optional (examples *neither-examples*) 
			       (theory *neither-theory*) (m-of-n nil))
  "Divides the number of unprovable positives covered by the fix, by the
product of the size of the fix and the number of new provable negatives.
Returns this ratio."
  ;;-------------------------------------------------------------------------
  ;; Here `fix' is a list of deletions, showing what should be abduced for a
  ;; particular example. These abductions are applied temporarily to the
  ;; theory, and then a ratio is computed which is: 
  ;;
  ;;                       #-unprovable-positives-covered
  ;;            -------------------------------------------------
  ;;            (#-new-provable-negatives * #-antecedents-abduced)
  ;;
  ;; The value of the above ratio is returned.
  ;;
  ;; 09-APR-93 (ptb): found a bug here. When I determine which examples are
  ;; covered by a given fix, I must ELIMINATE any such examples which ALSO
  ;; turn out to be new failing negatives. To fix things, I make the variable
  ;; which holds the covered examples dynamically scoped, and then delete
  ;; unwanted elements in the "new-negative-count" routine.
  ;;-------------------------------------------------------------------------
  (let (*covered-examples* num-negatives)
    (declare (special *covered-examples*))
    (setf *covered-examples* (positives-covered theory))
    (setf num-negatives (new-negative-count examples theory))
    (if m-of-n
	(setf (revision-threshold-solutions rev) *covered-examples*)
	(setf (revision-solutions rev) *covered-examples*))
    (/ (length *covered-examples*)
       (* (1+ num-negatives)
	  (if m-of-n
	      (pos-threshold-changes fix)
	      (number-antes-abduced fix))))))


(defun positives-covered (&optional (theory *neither-theory*))
  "Returns a list of the unprovable positives provable by theory. Typically
called after some antecededents or rules have been abduced from the theory."
  ;;-------------------------------------------------------------------------
  ;; This routine assumes that *remaining-positives* has been converted to a
  ;; list of (fix example) lists.  Each unprovable positive in *remaining-
  ;; positives* is checked, and if its category (ie "example-name") IS
  ;; provable, then the example is collected into a list of all covered
  ;; examples in *remaining-positives* to be returned.
  ;;
  ;; NOTE THIS ROUTINE ALLOWS AMBIGUITY. That is, a unprovable positive is
  ;; considered `covered' if the correct category is provable, even if other
  ;; categories are also provable.
  ;;-------------------------------------------------------------------------
  (loop for soln in *remaining-positives*
	for ex = (solution-example soln)
	if (member (example-name ex) (prove-categories ex theory))
	collect soln))


(defun number-antes-abduced (fix)
  ;;-------------------------------------------------------------------------
  ;; Here `fix' is a list of deletions, showing what should be abduced for a
  ;; particular example. This routine returns the sum of the number of
  ;; antecedents abduced for each rule. This is done by summing the length of
  ;; the antecedents of each element of the fix.
  ;;-------------------------------------------------------------------------
  (loop for ab in fix sum (length (deletion-antes ab))))


(defun pos-threshold-changes (fix)
  ;;-------------------------------------------------------------------------
  ;; Sums all the changes to all the rule thresholds in the fix.
  ;;-------------------------------------------------------------------------
  (loop for ab in fix sum (deletion-threshold-delta ab)))


(defun new-negative-count (&optional (examples *neither-examples*)
				     (theory *neither-theory*))
  "Counts the number of NEW negatives created by theory. Typically called
after some antecedents or rules have been abduced from the theory."
  ;;-------------------------------------------------------------------------
  ;; This routine goes through all the ORIGINAL examples (in `examples') and
  ;; checks for provable negatives. If it finds any which were not already
  ;; recorded, it increments count. To be a new provable negative, the
  ;; examples must (1) prove something outside the desired category which (2)
  ;; was not already provable with original theory. 
  ;;
  ;; Works as follows. For each provable negative, computes the categories
  ;; provable. Removes the desired category from this list. Then, for each
  ;; category left in the list, checks the *provable-negative-alist* to see
  ;; if the example was formerly detected as a provable negative. If not, it
  ;; counts as a new provable negative and `count' is incremented.
  ;;
  ;; 09-APR-93 (pb): This routine updated to fix a bug in the way the best
  ;; fix is found. A dynamically scoped variable is set up in calc-pos-fix-
  ;; value above, which holds all the examples covered by a fix. Here, if the
  ;; same example turns up as a new failing negative, I remove it from that
  ;; dynamically scoped variable. It's kind of a hack, but it was the first
  ;; fix I could think of that was garbage-less, a key consideration since
  ;; this is one of those routines that is called a LOT.
  ;;-------------------------------------------------------------------------
  (declare (special *covered-examples*))
  (let ((count 0))
    (dolist (pn examples count)
      (dolist (cat (remove (example-name pn) (prove-categories pn theory)))
	(unless (member pn (cdr (assoc cat *provable-negative-alist*)))
	  (incf count)
          (setf *covered-examples*
                (delete pn *covered-examples*
                        :test #'(lambda (x l)
                                  (eq x (solution-example l))))))))))


(defun find-new-negatives (&optional (examples *neither-examples*)
				     (theory *neither-theory*))
  "Returns the NEW negatives created by theory. Called after some antecedents
have been abduced from the theory."
  ;;-------------------------------------------------------------------------
  ;; This routine goes through all the ORIGINAL examples (in `examples') and
  ;; checks for provable negatives. It collects any which were not already
  ;; recorded. To be a new provable negative, an example must (1) prove
  ;; something outside the desired category which (2) was not already
  ;; provable with original theory. 
  ;;
  ;; Works as follows. For each provable negative, computes the categories
  ;; provable. Removes the desired category from this list. Then, for each
  ;; category left in the list, checks the *provable-negative-alist* to see
  ;; if the example was formerly detected as a provable negative. If not, it
  ;; counts as a new provable negative and collects it.
  ;;-------------------------------------------------------------------------
  (loop for ex in examples
	when
	(loop for cat in (remove (example-name ex)
				 (prove-categories ex theory))
	      finally (return nil)
	      unless (member ex (cdr (assoc cat *provable-negative-alist*))
			     :test #'eq)
	      return t)
	collect ex))


(defun any-new-negatives? (&optional (examples *neither-examples*)
				     (theory *neither-theory*))
  "Returns T if NEW negatives are created by theory, else returns nil."
  ;;-------------------------------------------------------------------------
  ;; Works as follows. For each provable negative, computes the categories
  ;; provable. Removes the desired category from this list. Then, for each
  ;; category left in the list, checks the *provable-negative-alist* to see
  ;; if the example was formerly detected as a provable negative. If not, it
  ;; QUITS IMMEDIATELY and returns T. Default return is NIL.
  ;;-------------------------------------------------------------------------
  (loop for ex in examples
	when
	(loop for cat in (remove (example-name ex)
				 (prove-categories ex theory))
	      finally (return nil)
	      unless (member ex (cdr (assoc cat *provable-negative-alist*))
			     :test #'eq)
	      return t)
	do (return t)
	finally (return nil)))


;;===========================================================================
;; FIX IMPLEMENTATION ROUTINES
;;
;; Used to implement a fix in a theory. Destructively modifies the theory.
;;===========================================================================


(defun try-pos-fix (revision &optional (examples *neither-examples*)
			     (theory *neither-theory*) (m-of-n nil))
  "Applies the fix, the first element of the first argument, to the theory.
If the result proves new false positives, induction is called to build a new
rule."
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "revision" is a pair where the first element is the best fix
  ;; found and the second element is a list of the examples covered.
  ;;
  ;; The idea here is to apply the deletions of the fix one at a time. As
  ;; each is applied, a check is made to see if any new negatives are
  ;; created. If not, the deletion is made permanently. If so, then we must
  ;; induce a rule to add to the theory. However, we wait until the end of
  ;; generalization to make all the inductions at once (see
  ;; add-pos-inductions).
  ;;
  ;; RETURNS: t or nil, indicating whether all changes were made using
  ;; threshold revisions.
  ;;-------------------------------------------------------------------------
  ;;  (format t "~%--> Best fix found:")
  ;;  (loop for del in (revision-fix revision) do
  ;;        (pprule del)
  ;;        (loop for a in (deletion-antes del)
  ;;              do (format t "~%~A" (antecedent-prop a)))
  ;;        (format t "~%new Delta=~D~%" (deletion-threshold-delta del)))
  ;;  (format t "~%covered examples:")
  ;;  (loop for soln in (revision-solutions revision)
  ;;        for ex = (solution-example soln)
  ;;        do (format t " ~D" (position ex examples)))
  (trace-print *trace-genl*
	       "~%~%*Attempting to make the following generalizations")
  (loop for ab in (revision-fix revision)
	for fixed-pos in (sort-pos-examples revision)
	with pure-thresholds = t
	finally (return pure-thresholds)
	do (when *trace-genl*
	     (format t "~%  To make the following examples provable:")
	     (loop for p in fixed-pos
		   do (format t "~%    example ~D"
			      (position p *neither-examples*))))
	unless (try-pos-threshold-fix ab m-of-n examples theory)
	do (setf pure-thresholds nil)
           (unless (try-abduction-fix ab examples theory)
             (save-for-pos-induction ab fixed-pos))))


(defun try-pos-threshold-fix (abduction m-of-n examples theory)
  "Tries to use a threshold change to satisfy the abduction. If successful,
the threshold is left at its new value and T is returned. Otherwise, the
threshold is restored and NIL is returned."
  ;;-------------------------------------------------------------------------
  ;; Tries to use a threshold change to satisfy the abduction. If successful,
  ;; the threshold is left at its new value and T is returned. Otherwise, the
  ;; threshold is restored and NIL is returned. If m-of-n is set to nil, then
  ;; no threshold changes are tried and nil is immediately returned.
  ;;-------------------------------------------------------------------------
  (if m-of-n
    (let (rval)
      (when *trace-genl*
        (format t "~%  in rule:~%    ")
        (pprule abduction nil)
        (format t "~%  increase threshold by: ~D"
                (deletion-threshold-delta abduction)))
      (incf-threshold abduction)
      (setf rval (any-new-negatives? examples theory))
      (if rval
          (progn
            (decf-threshold abduction)
            (trace-print *trace-genl* " ... failed."))
	  (progn
	    (trace-print *trace-genl* " ... succeeded.")))
      (not rval))
    nil))


(defun try-abduction-fix (abduction examples theory)
  "Tries to delete the antecedents of the abduction. If successful, the 
antecedents are left out and T is returned. Otherwise, the rule is restored 
and NIL is returned."
  ;;-------------------------------------------------------------------------
  ;; Tries implementing the abduction by deleting the antecedents. Returns T
  ;; if this can be done without causing any new negatives to be formed. If
  ;; new negatives do occur, returns NIL and resets the theory.
  ;;-------------------------------------------------------------------------
  (let (rval)
    (temp-abductions abduction)
    (when *trace-genl* (ppabduction abduction))
    (setf rval (any-new-negatives? examples theory))
    (if rval
      (progn
        (trace-print *trace-genl* " ... failed.")
        (undo-abductions abduction))
      (progn
        (trace-print *trace-genl* " ... succeeded.")
        (perm-abductions abduction)
        (record-ante-deletions abduction)))
    (not rval)))


(defun sort-pos-examples (revision)
  ;;-------------------------------------------------------------------------
  ;; INPUTS: revision is a revision structure (see structures.lisp) which
  ;; contains solutions showing which example(s) is (are) covered by the fix.
  ;;
  ;; The idea is to check here to see if a given example can be covered using
  ;; only a subset of fix. If so, then only those deletions of fix need a
  ;; reference to the example. Thus the examples are pidgeon-holed into a
  ;; slots, one slot for each deletion in the fix. 
  ;;
  ;; RETURN: a list of lists, equal in length to the number of deletions in
  ;; the best fix. Each sublist is a list of examples that require the given
  ;; deletion. This information is required by the induction phase (if
  ;; reached) where we must have the unproved positives related to the
  ;; deletion.
  ;;-------------------------------------------------------------------------

  ;; Initialize the return value to be a list of empty lists. Also, create a
  ;; blank list for the `marks' used by `covered-by-subset?' 
  (let* ((fix (revision-fix revision))
	 (sorted-positives (make-sequence 'list (length fix)))
	 (marks (make-sequence 'list (length fix) :initial-element 0)))

    ;; loop through the examples covered by the fix, inserting the example in
    ;; the appropriate element(s) of `sorted-positives'.
    (loop for soln in (revision-solutions revision)
	  for ex-fix = (solution-fix soln)
	  for ex = (solution-example soln)
	  do
	  (if (covered-by-subset? ex-fix fix marks)
	      
	      ;; if the fix for the example needs only a subset of the best
	      ;; fix, add example to only those sublists of sorted-positives
	      (do ((sublist sorted-positives (cdr sublist))
		   (m marks (cdr m)))
		  ((null sublist) nil)
		(if (= 1 (car m))
		    (rplaca sublist (cons ex (car sublist)))))
	      
	      ;; otherwise, add example to all sublists of sorted-positives
	      (do ((sublist sorted-positives (cdr sublist)))
		  ((null sublist) nil)
		(rplaca sublist (cons ex (car sublist))))))
    sorted-positives))


(defun temp-abductions (abduction)
  "Given an abduction, sets flags such that the antecedents of the abduction
are temporarily removed from the theory."
  ;;-------------------------------------------------------------------------
  ;; Calls "set-abduced?-flags" to do all the work.
  ;;-------------------------------------------------------------------------
  (set-abduced?-flags abduction t))


(defun undo-abductions (abduction)
  "Given an abduction, sets flags such that the antecedents of the abduction
which were temporarily removed from the theory are restored."
  ;;-------------------------------------------------------------------------
  ;; Calls "set-abduced?-flags" to do all the work.
  ;;-------------------------------------------------------------------------
  (set-abduced?-flags abduction nil))


(defvar *leave-abduced-rule* nil)

(defun perm-abductions (abduction)
  "Given an abduction, permanently (destructively) removes the antecedents
from the rule associated with the abductions."
  ;;-------------------------------------------------------------------------
  ;; Simply loops through the list of antecedents in "abduction" and removes
  ;; them from the "rule" of the abduction. Recall that "abduction" is a
  ;; deletion structure (see structures.lisp).
  ;;
  ;; DESTRUCTIVELY modifies rules in the theory.
  ;;
  ;; 19-NOV-92 (ptb): Modified to use the global variable above to copy the
  ;; rule and make abductions in the copy. This was just a hack so I could 
  ;; try an experimental comparison. By setting the above flag to t, NEITHER
  ;; will leave the old rule around.
  ;;
  ;; 12-MAR-93 (ptb): added consequent to call to make-rule for new field in 
  ;; the rule structure. Also, modified routine to avoid any deletions from 
  ;; the theory. Instead, the antecedents are simply left as marked-deleted.
  ;; This is necessary for printing out the student model; specifically, I 
  ;; need to be able to print out the rule in its original form which I can't
  ;; do if I start deleting antecedents.
  ;;-------------------------------------------------------------------------
  (let ((child (deletion-child abduction)))
    (when *leave-abduced-rule*
      (undo-abductions abduction)
      (let ((parent (deletion-parent abduction))
	    (ante-copy
	     (loop for ante in (rule-antecedents child)
		   collect
		   (make-antecedent :prop (antecedent-prop ante)
				    :abduced? (antecedent-abduced? ante)
				    :vertex (antecedent-vertex ante)))))
	(setf (vertex-children parent)
	      (cons
	       (make-rule :used (rule-used child)
			  :threshold (rule-threshold child)
			  :antecedents ante-copy
                          :consequent (vertex-prop parent))
	       (vertex-children parent)))))
    ;; The loop below can be used to make permanent changes to the theory,
    ;; i.e., do delete the antecedents. It is commented out so that the
    ;; modeling code will work. Instead, I just hand off to temp-abductions.
    ;;(loop for ante in (deletion-antes abduction)
    ;;      for ante-list = (rule-antecedents child)
    ;;      do
    ;;      (setf (rule-antecedents child)
    ;;            (delete ante ante-list :test #'eq)))
    (temp-abductions abduction)
    ))


(defun save-for-pos-induction (abduction positives)
  "Puts the abduction on a list for later induction of new rules.
Stores the positives covered with the abduction."
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "abduction" is a deletion in the same format as described in the
  ;; file "structures.lisp". "positives" is a list of examples that require
  ;; the abduction (see sort-pos-examples routine). 
  ;;
  ;; OUTPUT: destructively modifies *pos-induction-alist* to include the new
  ;; abduction and its positives. Each is associated with the child rule
  ;; where the changes were made (ie, the child of the vertex for that
  ;; consequent from which the antecedents are deleted--see deduce.lisp)
  ;; If there is already a list for that rule, the abduction is added to that
  ;; list. Otherwise, a new list is created and consed to the front of
  ;; *pos-induction-alist*.
  ;;-------------------------------------------------------------------------
  (let* ((rule (deletion-child abduction))
	 (loc (assoc rule *pos-induction-alist* :test #'eq)))
    (if loc
	(rplacd loc (cons (cons abduction positives) (cdr loc)))
	(setf *pos-induction-alist*
	      (cons (list rule (cons abduction positives))
		    *pos-induction-alist*)))))


(defun update-positives (revision pure-thresholds
				  &optional (theory *neither-theory*)
				  (m-of-n nil))
  "Removes positives from *remaining-positives* that are covered by revision,
and recomputes fixes for the positives remaining using theory."
  ;;-------------------------------------------------------------------------
  ;; Needs to (1) remove the covered positives from *remaining-positives* and
  ;; then (2) recompute the fix for each remaining positive as done in the
  ;; init-abduction-list routine.
  ;;
  ;; Removing the covered positives is easy since these are stored as part of
  ;; the revision. We can loop through the covered positives and use "delete"
  ;; (with the eq function) since we have direct pointers to elements of the
  ;; *remaining-positives* list. Note, however, that since the first element
  ;; of the list may be deleted, we must re-setf *remaining-positives*.
  ;;
  ;; Recomputing fixes is done destructively in place to save on garbage. We
  ;; simply loop through whatever is left on *remaining-positives*, replacing
  ;; the old fix with a new fix computed for the example.
  ;;
  ;; RETURN: since *remaining-positives* is destructively modified, no return
  ;; value is required. 
  ;;
  ;; 19-NOV-92 (ptb): Now that changes can be made by altering thresholds or
  ;; by abduction, there are two possible sets of positive solutions covered
  ;; by the update. The way that "try-pos-fix" works, some of the revisions
  ;; could be threshold changes while others could be abduction or induction
  ;; repairs. So, the trick here is to pick the right set of covered positive
  ;; examples, based upon what happened in the try-pos-fix code. Since a 
  ;; threshold change will have a strictly larger deductive closure, the 
  ;; positive examples covered by threshold changes is a SUPERSET of those
  ;; covered by abduction changes. Thus, the try-pos-fix routine keeps track
  ;; of whether or not ALL of its changes are made using thresholds. If so,
  ;; we can use the (possibly larger) threshold set of covered positives. If
  ;; not, we must use the abduction set of positives to ensure that NEITHER
  ;; remains consistent.
  ;;-------------------------------------------------------------------------
  (let ((covered-positives
	 (if (and m-of-n pure-thresholds)
	     (revision-threshold-solutions revision)
	     (revision-solutions revision))))
    (loop for soln in covered-positives
	    do (setf *remaining-positives*
		     (delete soln *remaining-positives* :test #'eq)))
    (loop for soln in *remaining-positives*
	  for ex = (solution-example soln)
	  do (setf (solution-fix soln)
		   (abduce ex (example-name ex) theory)))))


(defun add-pos-inductions (&optional (examples *neither-examples*)
				     (theory *neither-theory*))
  "Performs induction for all abductions which could not be directly applied
to the theory and adds the new rules to the theory."
  ;;-------------------------------------------------------------------------
  ;; This routine loops through all the elements of *pos-induction-alist*
  ;; and forms new rules (by induction) to cover the positives in this
  ;; alist. The method is as follows: (1) temporarily delete all antecedents
  ;; that were proposed abductions from a rule (2) induce and add new rules
  ;; to account for the positive examples, then (3) "undelete" all the
  ;; antecedents in the original rule (leaving it intact).
  ;;
  ;; Inducing and adding the new rules is done by a series of nested
  ;; subroutine calls that work as follows (from the inside out). First, the
  ;; positive and negative examples are converted into a form for induction.
  ;; These, plus any remaining antecedents not deleted from the original
  ;; rule, are passed to a routine to build new rules. The new
  ;; rules are then converted into the internal NEITHER format, which is gets 
  ;; "patched" with the name of the rule in the abduction. Finally, the 
  ;; resulting patched theory (in NEITHER format) is merged with the existing 
  ;; theory stored in *neither-theory*.
  ;;
  ;; The patching process may seem obscure. Here's the reason for doing it.
  ;; Induction gets examples labeled "positive" or "negative". The resulting
  ;; rules thus have "positive" or "negative" as their consequent. The patch,
  ;; then, replaces the consequents of the rules coming out with the
  ;; consequent of the rule in abduction which is being updated.
  ;;
  ;; HACK ADDED: I needed to add a hack to make this code work; specifically,
  ;; to make the call to "theory-to-decision-tree" work. When I collect
  ;; positive and negative examples, I label them as "positive" and
  ;; "negative" respectively (see induce.lisp). However, these categories
  ;; are not guaranteed to be on the *categories* global, and that presents a
  ;; problem for translating the rules learned into NEITHER's internal
  ;; decision tree format. So, as a total hack, I push "positive" on to this
  ;; list and pop it off when I'm done with abduction. I don't have to push
  ;; "negative" because I'm not interested in translating the negative rules.
  ;;
  ;; 14-JAN-93 (ptb): Added a check to this code to use the new version of
  ;; induce-rules (see induce.lisp). During generalization, we have the 
  ;; case where some rule does not prove a positive. So, for example, the
  ;; following rule and positive example ("-" indicates failed antecedent, 
  ;; "+" indicates successful antecedent):
  ;;
  ;;    1: a <- b c d e f
  ;;   e1:      - - - - +  (positive example), not provable
  ;;   e2:      - - - + +  (negative example), not provable
  ;;
  ;; Here example e1 is an unprovable positive (ie, can't satisfy the above 
  ;; rule) where e2 is currently an unprovable negative. Now, if we copy the 
  ;; rule and change it by performing some antecedent deletions (say, delete
  ;; b c d) we'd get:
  ;;
  ;;    1: a <- b c d e f
  ;;    1: a <- e f
  ;;   e1:      - +  (positive example), provable
  ;;   e2:      + +  (negative example), provable
  ;;
  ;; Normally, we then pass of all the covered positives and new provable 
  ;; negatives (which now includes e2) to induction. The result of induction 
  ;; is then appended to the altered rule. So, say induction returned (x y z) 
  ;; as conditions. The old way of doing generalize would do:
  ;;
  ;;    1: a <- b c d e f
  ;;    1: a <- e f x y z
  ;;   e1:      - + + + +  (positive)
  ;;   e2:      + + ? ? ?  (negative)
  ;;
  ;; where the "?" need only contain a SINGLE failure! That means we will not 
  ;; necessarily eliminate the negatives we want to avoid proving since e2 
  ;; would be provable if only one "?" were a failure.
  ;;
  ;; The real answer is to produce the following rule set:
  ;;
  ;;    1: a <- b c d e f
  ;;    1: G <- e f
  ;;    0: a <- G x y z
  ;;
  ;; Now, the second rule has a new consequent, and the third forces the 
  ;; examples to satisfy BOTH rules. Only e1 can satisfy rules 2 and 3; e2 
  ;; will succeed for rule 2 but fail on rule 3 since it must have at least
  ;; one of (x y z) false.
  ;;
  ;; It turns out that this "splitting" of the rule into two parts is only 
  ;; necessary if more than one antecedent is deleted from the original rule 
  ;; and if the rule has a non-zero threshold. So, this routine checks how
  ;; many deletions are needed and passes T or NIL to "induce-rules"
  ;; accordingly.
  ;;-------------------------------------------------------------------------
  (push 'positive *categories*)
  (loop for ind in *pos-induction-alist*
	for rule = (first ind)
        for old-count = (count-antes rule)
	for ab-ex-pairs = (rest ind)
	with new-tree = nil
        with new-count = 0
	with temp = nil
	do

	;; temporarily remove all antecedents
	(loop for pair in ab-ex-pairs
	      do (temp-abductions (first pair)))
        (setf new-count (count-antes rule))

	(setf temp
	      (convert-theory
	       (induce-rules 
		(cnvt-to-standard-ex
		 (collect-induction-positives ab-ex-pairs)
		 (find-new-negatives examples theory))
		(rule-threshold rule)
		(if (or (= 0 (rule-threshold rule))
			(= 1 (- old-count new-count))) nil t)
		(remaining-antecedents rule))))
	;;(format t "~%induced rules are ") 
	;;(dolist (r temp) (format t "~%   ~A" r))

	(setf new-tree
	      (patch-decision-tree
	       (vertex-prop (deletion-parent (first (first ab-ex-pairs))))
	       (theory-to-decision-tree temp)))

	(when *trace-genl*
	  (format t "~%~%+Adding the following rule(s) to the theory")
          (decompile-theory nil nil new-tree))

        ;; record changes in student model
        (record-rule-additions rule new-tree)
	
	;; reset all antecedents
	(loop for pair in ab-ex-pairs
	      do (undo-abductions (first pair)))

	(merge-with-current-theory new-tree))
  (pop *categories*))


(defun collect-induction-positives (ab-ex-pairs)
  ;;-------------------------------------------------------------------------
  ;; Collects all the examples stored in the list of abduction-example lists
  ;; assoc'ed with a given rule. The input "ab-ex-pairs" is assumed to be in
  ;; the following form:
  ;;   ((abduction ex1 ex2 ...) (abduction ex1 ex2 ...) ...)
  ;; Recall that each abduction is a deletion structure (see structures.lisp)
  ;;-------------------------------------------------------------------------
  (loop for pair in ab-ex-pairs
	with ret-val = nil
	do (loop for ex in (rest pair)
		 unless (member ex ret-val :test #'eq)
		 do (setf ret-val (cons ex ret-val)))
	finally (return ret-val)))


(defun remaining-antecedents (rule)
  ;;-------------------------------------------------------------------------
  ;; Called after all antecedents in all abductions assoc'ed with a given
  ;; rule have been temporarily "deleted" from the rule. The idea is to loop
  ;; through the rule, collecting anything not marked as abduced.
  ;;
  ;; INPUTS: "rule" is a rule (see structures.lisp) of the
  ;; *pos-induction-alist* list. 
  ;;-------------------------------------------------------------------------
  (loop for ante in (rule-antecedents rule)
	unless (antecedent-abduced? ante)
	collect (antecedent-prop ante)))
