
;;===========================================================================
;; Abduction system for NEITHER 
;;
;; -------------------------------------------------------------------------
;; AUTHORS: Christopher M. Whatley, Paul T. Baffes.
;; Copyright (c) 1992 by AUTHORS. This program may be freely copied, used,
;; or modified provided that this copyright notice is included in each copy
;; of this code and parts thereof. 
;; -------------------------------------------------------------------------
;;
;; Note that this code depends heavily upon the data structures and
;; supporting routines defined in "structures.lisp" (see file for comments).
;;
;;
;; CHANGE HISTORY
;;
;; 08-APR-92: (cmw) Redid abduce to return parents along with the leaves to 
;;            be deleted.
;; 01-JUN-92: (ptb) Wrote comments for all routines.
;; 03-JUN-92: (ptb) Fixed bug in shortest-list routine to count number of
;;            abduced antecedents rather than number of rules (fixed by 
;;            adding abduction-length routine).
;; 17-DEC-92: (ptb) recording an earlier change to the abduce routine that 
;;            covers the event that no abductions are possible.
;; 17-DEC-92: (ptb) recording change made to routine "abduction-length" to
;;            subtract out the threshold.
;; 17-DEC-92: (ptb) added comments describing change to "choose-deletions" to
;;            deal with the interchange of abduction length and provability.
;; 03-JAN-92: (ptb) updated code to use the new candidate structure.
;; 07-JAN-92: (ptb) changed code to return only the "n best" abductions which
;;            are necessary to make each rule provable. That is, I removed 
;;            the assumption that all rules have a threshold of 0.
;; 17-MAR-93: (ptb) Added a global variable called *fix-marker* and updated
;;            the abduce, abduce-1 and abduce-child routines to avoid 
;;            revisiting parts of the theory which have already been searched
;;            for abductions. In theories with shared concepts, this saves 
;;            time. Note that, as with retract-rules, I needed to use a
;;            separate marking scheme from that used by prove-categories so
;;            the marks wouldn't overwrite eachother.
;;===========================================================================

(in-package #+:cltl2 "CL-USER" #-:cltl2 "USER")

;;===========================================================================
;; GLOBAL VARIABLE REFERENCE
;;
;; Four global variables are referenced here.
;; *trace-pos-fix* 
;;    used to print out messages as to how an abduction is found for a given 
;;    example.
;; *neither-examples* 
;;    used as a default for printout out tracing info.
;; *neither-theory*
;;    the default theory used to compute abductions.
;; *fix-marker*
;;     Used to make abduction more efficient. The principle is the same as 
;;     in deduce.lisp: avoid revisiting parts of the theory which have 
;;     already been traversed.
;;===========================================================================
(defvar *fix-marker* 0
  "Used in finding abductions and retractions")

#+:cltl2(declaim (special *trace-pos-fix*)
                 (special *neither-examples*)
                 (special *neither-theory*))
#-:cltl2(progn
          (proclaim '(special *trace-pos-fix*))
          (proclaim '(special *neither-examples*))
          (proclaim '(special *neither-theory*)))


(defun abduce (example category &optional (theory *neither-theory*))
  "Returns a fix for the example representing the antecedents which, if
abduced from theory, allow theory to prove category from example."
  ;;-------------------------------------------------------------------------
  ;; Returns a list of deletions, ie a FIX (see structures.lisp) which, if 
  ;; deleted, would make prove-categories return the correct category for the 
  ;; example.
  ;;
  ;; (ptb 12-17-92) Adding a comment to describe "pop up" phenomenon. The 
  ;; essential need for this extra code is to cover the event that no 
  ;; abductions are returned from the call to abduce-1. In this case, the 
  ;; theory is made maximally general and returns an abduction which makes
  ;; the category trivially provable.
  ;;
  ;; 17-MAR-93 (ptb) added an increment to *fix-marker* so each example has
  ;; a unique number for abduction.
  ;;-------------------------------------------------------------------------
  (incf *fix-marker*)
  (let ((*category* category)
	(*example*  example)
	(vert (find-category category theory))
	rval)
    (declare (special *category* *example*))

    ;; do a printout for tracing purposes
    (when (and *trace-pos-fix*
               (member (position example *neither-examples*) *trace-pos-fix*))
      (push 'yes *trace-pos-fix*)
      (format t "~%~%Abductions to prove the following as ~A" *category*)
      (format t "~%  example number: ~D"
	      (position example *neither-examples*))
      (ppexample *example*))

    ;; perform the actual retraction. Save results
    (setf rval (candidate-deletions
                (abduce-1 (example-values example) vert)))
    (unless rval
      (trace-print (and *trace-pos-fix* (eq (car *trace-pos-fix*) 'yes))
                   "~%*** must abduce all; POPPED ALL THE WAY UP")
      (setf rval
	    (loop for child in (vertex-children vert)
		  collect
		  (make-deletion
		   :parent vert :child child
		   :antes (copy-list (rule-antecedents child))
                   :threshold-delta (length (rule-antecedents child))))))

    ;; do another printout for tracing purposes before returning
    (when (and *trace-pos-fix* (eq (car *trace-pos-fix*) 'yes))
      (format t "~%~%Total abductions for example ~D"
	      (position example *neither-examples*))
      (loop for a in rval do (ppabduction a))
      (pop *trace-pos-fix*))
    rval))


(defun abduce-1 (exvals vertex)
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "exvals" is a vector part of an example (see file deduce.lisp)
  ;; which contains the values for the example. "vertex" is the top of a
  ;; tree from *neither-theory* 
  ;;
  ;; Finds a list of antecedents which should be abduced to make "vertex"
  ;; satisfiable for the given example vector in "exvals". Works as follows.
  ;; First, all children of "vertex" are passed to "abduce-child" to
  ;; determine their candidate deletions, ie, the minimum number of 
  ;; antecedents that need to be abduced to make the child satisfiable given 
  ;; the example. These are collected for all children of "vertex" and passed
  ;; to "choose-deletions" which selects the smallest (since we only need one
  ;; for the proof). 
  ;;
  ;; RETURNS the deletions (abductions) which make the vertex provable with
  ;; exvals. The format of the return is a CANDIDATE structure (see the file
  ;; structures.lisp) which contains a flag and a list of deletions. The 
  ;; "flag" indicates whether the parent was provable with NO deletions
  ;; (t=no deletions required, nil=at least one deletion required).
  ;;
  ;; 17-MAR-93 (ptb) modified routine to check first to see if vertex has 
  ;; already been visited. If so, the old value for the vertex is returned;
  ;; if not, the normal abduction is followed and the result is stored as the
  ;; Fvalue (ie, fix value) for the vertex. The vertex is marked as visited
  ;; with the current value of *fix-marker*.
  ;;-------------------------------------------------------------------------
  (when (= *fix-marker* (vertex-Fvisited vertex))
    (return-from abduce-1 (vertex-Fvalue vertex)))
  (setf (vertex-Fvisited vertex) *fix-marker*)
  (setf (vertex-Fvalue vertex)
        (choose-deletions
         (loop for child in (vertex-children vertex)
	       if (rule-used child)
	       collect (abduce-child vertex exvals child)))))


(defun abduce-child (parent exvals child)
  ;;-------------------------------------------------------------------------
  ;; INPUTS: "parent" is the vertex for which "child" is one of its children.
  ;; "child" is a rule structure (see structures.lisp). "exvals" is a vector
  ;; of values for the example.
  ;;
  ;; Collects lists of potential deletions to satisfy a given parent. Note
  ;; that since the child is an antecedent list which is assumed to be a
  ;; conjunction, this routine must collect ALL vertices which must be
  ;; abduced into its return list. All vertices are thus tested, and any 
  ;; which are not satisfiable are saved for the return value (see comments
  ;; below).
  ;;
  ;; Return value is in the same format as abduce-1 (see above).
  ;;
  ;; 07-JAN-93 (ptb) Removed the assumption that all children need to be 
  ;; satisfied for the rule to be true. Now, the threshold of the child is 
  ;; used directly to indicate how many false antecedents are allowed. All 
  ;; failing antecedents are stored in a local variable. Once all antecedents
  ;; are checked, the number of needed abductions is determined by taking the
  ;; difference between the number of failed antecedents and the threshold. 
  ;; If this difference is <= 0, the threshold is large enough to cover the
  ;; failures and the rule is provable. If not, some antes must be abduced.
  ;; If all antecedents must be abduced, this routine will "pop up" to the
  ;; caller, and the parent will be abduced from its rule. Note that any sub-
  ;; abductions are kept on a separate list and returned with the abductions
  ;; (if any) for this rule.
  ;;
  ;; 17-MAR-93 (ptb) made a small change to how "sub-abs" is updated. Since I
  ;; have moved to a scheme that reuses abductions when a vertex is visited
  ;; multiple times, I updated the modification of sub-abs to avoid duplicate
  ;; entries. Thus, if a shared concept has abductions as part of the fix for
  ;; two different antecedents, only one copy of the shared abductions gets
  ;; included in "sub-abs".
  ;;-------------------------------------------------------------------------
  (loop for ante in (rule-antecedents child)
	for vert = (antecedent-vertex ante)
	for leaf? = (leaf-vertex vert)
	with cand = nil
	with leaves = nil
	with sub-abs = nil
        with abs-count = 0
        with threshold = (rule-threshold child)
	if (and leaf?                            ;; if a leaf,
		(not (antecedent-abduced? ante)) ;; and shouldn't ignore,
		(or (vertex-no-rules? vert) ;; and either an empty intermed,
		    (not                    ;; or example cant satis. vertex
		     (leaf-test (antecedent-prop ante)
				(aref exvals (vertex-example-index vert))))))
          do (setf leaves (cons ante leaves))    ;; then add to ante lst

        ;; if not a leaf, recurse and collect
	if (and (not leaf?) (not (antecedent-abduced? ante)))
	  do (setf cand (abduce-1 exvals vert)) ;; find any sub abs.
	     (when (not (candidate-provable cand)) ;; if some abs required
               (incf abs-count)
	       (if (valid-deletions cand)     ;; check if sub abs valid
                 (dolist (ab (candidate-deletions cand)) ;; y, add to sub-abs
                   (unless (member ab sub-abs :test #'eq)
                     (setf sub-abs (cons ab sub-abs))))
		   (setf leaves (cons ante leaves)))) ;;else add to ante lst

	finally
        (let ((num-to-abduce (- (length leaves) (rule-threshold child))))
          (if (<= num-to-abduce 0)        ;; determine if rule still provable
              (setf num-to-abduce 0))     ;; set number to abduce accordingly
	  (when (and *trace-pos-fix* (eq (car *trace-pos-fix*) 'yes))
	    (format t "~%  for rule:")
	    (pprule (make-deletion :parent parent :child child))
	    (format t "~%   returning:")
	    (format t "~%   provable : ~A"
                    (>= threshold (+ (length leaves) abs-count)))
            (format t "~%   any sub-abs: ~A" (if sub-abs t nil))
	    (format t "~%   antes deleted:")
	    (if (= num-to-abduce (count-antes child))
		(format t " ALL; POP UP")
		(loop for a in (subseq leaves 0 num-to-abduce) do
		      (format t " ~A" (antecedent-prop a)))))
	  (return
           (make-candidate 
            :provable (>= threshold (+ (length leaves) abs-count))
            :deletions (if (> num-to-abduce 0)    ;; if any abductions
		         (if (>= num-to-abduce 
                                 (- (count-antes child) threshold))
			   nil   ;; dont use ante list if whole rule deleted
			   (cons (make-deletion :parent parent :child child
                                  :antes (subseq leaves 0 num-to-abduce)
                                  :threshold-delta num-to-abduce)
				 sub-abs))
		         sub-abs))))))


(defun valid-deletions (candidate)
  ;;-------------------------------------------------------------------------
  ;; INPUT: the "candidate" here is a CANDIDATE structure, (see comments in 
  ;; structures.lisp). This routine is unconcerned with the value of the flag
  ;; in this structure. It focuses only on the list of deletions.
  ;;
  ;; A set of deletions is only considered valid if it passes the following
  ;; two tests: (1) there are some deletions and (2) when the deletions are
  ;; made, the given example DOES NOT also become a new negative. The problem
  ;; is, if the example is both a positive and a negative, induction will
  ;; ignore it (because it will appear as noise) and will thus produce an
  ;; inaccurate rule.  
  ;;
  ;; NOTE: I use two globals set up in "abduce" (see above) rather than
  ;; passing them around as parameters.
  ;;-------------------------------------------------------------------------
  (declare (special *category* *example* *provable-negative-alist*))
  (if (candidate-deletions candidate)

      ;; if there are abductions
      (let (cats)

	;; temporarily perform the abductions
	(dolist (ab (candidate-deletions candidate)) 
          (set-abduced?-flags ab t))
	(setf cats (prove-categories *example*))
	(dolist (ab (candidate-deletions candidate)) 
          (set-abduced?-flags ab nil))

	;; check to see if abductions cause new negatives
	(loop for c in (remove *category* cats) ;; ret. t if no new negs
	      finally (return t)
	      unless (member *example*
			     (cdr (assoc c *provable-negative-alist*)))
	      return nil))   ;; return nil, ie invalid, if new neg found
      nil))


(defun choose-deletions (candidate-list)
  ;;-------------------------------------------------------------------------
  ;; Since abduction is only required to find ONE solution which will prove
  ;; an example, this routine is used to select the smallest solution from
  ;; among a list. The input is a list of candidates, each candidate has a 
  ;; list of deletions. Each deletion pairs a parent vertex with the children 
  ;; antecedents that were abduced. Each candidate also contains a flag, 
  ;; indicating whether or not the parent vertex was provable with NO 
  ;; deletions (see "structues.lisp" and comments above in "abduce-1").
  ;;
  ;; Fixed, 6-3-92. Bug in the way "length" was computed. Length counted the
  ;; number of rules which had deletions, whereas what we really wanted was
  ;; the number of antecedents abduced, regardless of how many rules they
  ;; came from. Changed to call new routine "abduction-length".
  ;;
  ;; (ptb 12-17-92) This code got tricky once I added the flags to indicate
  ;; whether or not the example could be proven with no deletions. The result
  ;; made the test for updating the best more complicated. Now, the best is 
  ;; updated under a compound condition (1) only update if your current best
  ;; is not provable with no abductions (if it is provable with no abductions
  ;; there can be nothing better). The second condition is a disjunct that 
  ;; must satisfy either (2a) the new instance is provable with no abductions
  ;; (2b) or if not, then there are some abductions in the new example and 
  ;; none in the current (ie, you just can't compare lengths since the one 
  ;; which is not provable but can't have abductions would win with a length 
  ;; of zero) or (2c) both are not provable, and both have some abductions, 
  ;; but the current example has fewer. If (1) and any of (2a) (2b) or (2c) 
  ;; are true, then we update.
  ;;-------------------------------------------------------------------------
  (loop for cand in candidate-list
	for provable = (candidate-provable cand)
	for dels = (candidate-deletions cand)
	for length = (abduction-length dels)
	with best = (first candidate-list)
	with bestdels = (candidate-deletions best)
	with bestlength = (abduction-length bestdels)
	with bestprovable = (candidate-provable best)
	if (and (not bestprovable)
		(or provable
		    (and dels (not bestdels))
		    (and dels bestdels (< length bestlength))))
	do (setf best cand
                 bestdels dels
		 bestlength length
		 bestprovable provable)
	finally 
        (when (and *trace-pos-fix* (eq (car *trace-pos-fix*) 'yes))
          (format t "~%~%  Best Chosen:")
          (dolist (del bestdels) 
            (pprule del) 
            (format t ": abduce ~D" (length (deletion-antes del))))
          (format t "~%  length=~D" bestlength)
          (format t "~%  provable=~A" bestprovable)
          (format t "~%"))
        (return best)))	


(defun abduction-length (deletion-list)
  ;;-------------------------------------------------------------------------
  ;; INPUT: "deletion-list" is a list of deletions. This routine returns
  ;; the total number of ANTECEDENTS abduced by looping through the
  ;; deletion-list, counting lengths of the antecedent lists (see the file
  ;; structure.lisp)
  ;;-------------------------------------------------------------------------
  (loop for del in deletion-list
	sum (length (deletion-antes del))))
