;;;____________________________________________________________________________________
;;;                           ANIMATED LEARNING DEMO
;;;
;;; The code in this file is designed to work in conjunction with grapher.lisp.
;;; It handles the initialization of all the learning process animation windows.
;;;
;;;  Created and designed by Clifford A. Brunk 05/20/91
;;;                                            07/14/91 modiifed
;;;
;;;  Problems:
;;;____________________________________________________________________________________

(require 'grapher)

(in-package :user)

(setf *backtrace-on-break* nil)

;;;____________________________________________________________________________________
;;;  close-learning-windows

(defun close-learning-windows ()
  (if (window-p *EBL-WINDOW*) (window-close *EBL-WINDOW*))
  (if (window-p *CURRENT-GAIN-WINDOW*) (window-close *CURRENT-GAIN-WINDOW*))
  (if (window-p *BEST-GAIN-WINDOW*) (window-close *BEST-GAIN-WINDOW*))
  (if (window-p *COVERAGE-AND-WORK-WINDOW*) (window-close *COVERAGE-AND-WORK-WINDOW*))
  (if (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*) (window-close *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))
  (close-pause-box))


;;;____________________________________________________________________________________
;;;  some-learning-window-is-open

(defun some-learning-window-is-open ()
  (or (window-p *COVERAGE-AND-WORK-WINDOW*)
      (window-p *CURRENT-GAIN-WINDOW*)
      (window-p *BEST-GAIN-WINDOW*)
      (window-p *EBL-WINDOW*)
      (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*)
      (window-p *PAUSE-BOX*)))

;;;____________________________________________________________________________________
;;;  init-learning-graph-windows

(defun init-learning-graph-windows (graph-learning
                                    prolog-literal-being-learned
                                    goal-concept)
  (close-learning-windows)
  (when (member :ebl graph-learning) (setup-EBL-WINDOW goal-concept))
  (when (member :coverage-and-work graph-learning) (setup-COVERAGE-AND-WORK-WINDOW))
  (when (member :best-gain graph-learning) (setup-BEST-GAIN-WINDOW))
  (when (member :current-gain graph-learning) (setup-CURRENT-GAIN-WINDOW))
  (when (member :learned-concept-description graph-learning) (setup-LCD-WINDOW prolog-literal-being-learned))

  ;; Reposition Windows when some are missing here

  (if (window-p *EBL-WINDOW*) (window-select *EBL-WINDOW*))
  (if (window-p *COVERAGE-AND-WORK-WINDOW*) (window-select *COVERAGE-AND-WORK-WINDOW*))
  (if (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*) (window-select *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))
  (if (window-p *CURRENT-GAIN-WINDOW*) (window-select *CURRENT-GAIN-WINDOW*))
  (if (window-p *BEST-GAIN-WINDOW*) (window-select *BEST-GAIN-WINDOW*))

  (break))


;;;__________________________________________________________________________________
;;; modify-parameters
;;;
;;; This huge function creates an equally huge dialog box which allows the user
;;; to change almost all of the parameters to FOCL. Currently clauses and 
;;; goal-concept-name can not be changes (although the use theory option can force
;;; these to nil if they are present.


(defun modify-focl-parameters (goal-concept-name
                               clauses
                               reset-hash-tables
                               max-new-variables
                               reset-statistics)
  (setf *focl-trace-level* nil)
  (let* ((window-h 600)
         (window-v 370)
         (offset 20)
         (x 20)
         (x-o 40)
         (x2 220)
         (x2-o 240)
         (x3 420)
         (x3-o 440)
         (y 5)
         (y2 5)
         (y3 5)
         (gain-function-cluster 0)
         (refinement-cluster 1)
         (noise-tolerance-cluster 2)
         (display-list *graph-learning*)
         (intensional-preds? (if *intensional-preds* t nil))
         (theory? (if (and intensional-preds?
                           (or goal-concept-name clauses)) t nil))
         (learnable? (and *extensional-preds* (pred-p (get *predicate-being-learned* 'pred))))
         (modify-parameters-dialog 
          (make-instance 'dialog
                         :window-show nil
                         :window-type :double-edge-box
                         :view-position :centered
                         :view-size (make-point 600 370)
                         :close-box-p nil
                         :view-font '("Chicago" 12 :srcor :plain)
                         :view-subviews
                         (list 
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point 15 y) #@(250 16)
                                            "FOCL Parameter Settings..." nil)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point 20 (incf y 25)) #@(70 16)
                                            "Learning:" nil)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point 90 y) #@(210 16)
                                            (format nil "~A" *predicate-being-learned*) nil)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point 300 y) #@(100 16)
                                            "Goal Concept:" nil
                                            :dialog-item-enabled-p theory?
                                            :view-nick-name :goal-concept-title)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point 400 y) #@(200 16)
                                            (format nil "~A" goal-concept-name) nil
                                            :dialog-item-enabled-p theory?
                                            :view-nick-name :goal-concept-title)

                          ;;; Empirical Learning
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x (setf y3 (setf y2 (incf y 25)))) #@(200 16)
                                            "Empirical Learning" nil)
                          (make-dialog-item 'editable-text-dialog-item 
                                            (make-point x-o (incf y 19)) #@(20 16) 
                                            (format nil "~a" max-new-variables)
                                            nil
                                            :allow-returns nil
                                            :view-nick-name :max-new-variables)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point (+ x 25 offset) y) #@(200 16)
                                            "max-new-variables" nil)
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 20)) #@(200 16)
                                            "intensional induction"
                                            nil
                                            :dialog-item-enabled-p intensional-preds?
                                            :check-box-checked-p 
                                            (if intensional-preds? *intensional-induction*)
                                            :view-nick-name :intensional-induction)
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "operationalize results"
                                            nil
                                            :dialog-item-enabled-p intensional-preds?
                                            :check-box-checked-p 
                                            (if intensional-preds? *operationalize-intensional*)
                                            :view-nick-name :operationalize-intensional)
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "use determinate literals"
                                            nil
                                            :check-box-checked-p *enable-determinate-literals*
                                            :view-nick-name :enable-determinate-literals)
                          (make-dialog-item 'editable-text-dialog-item 
                                            (make-point x-o (incf y 19)) #@(20 16) 
                                            (format nil "~a" *max-determinate-literals*)
                                            nil
                                            :allow-returns nil
                                            :view-nick-name :max-determinate-literals)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point (+ x 25 offset) y) #@(200 16)
                                            "max determinate lits" nil)


                          ;;; Relational Cliche's
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x2 y2) #@(200 16)
                                            "Relational Clichs" nil)

                          (make-dialog-item 'editable-text-dialog-item 
                                            (make-point x2-o (incf y2 19)) #@(20 16) 
                                            (format nil "~a" *max-new-cliche-vars*)
                                            nil
                                            :allow-returns nil
                                            :view-nick-name :max-new-cliche-vars)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point (+ x2-o 25) y2) #@(200 16)
                                            "max-new-cliche-vars" nil)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 20)) #@(200 16)
                                            "use clichs"
                                            nil
                                            :check-box-checked-p *use-relational-cliches*
                                            :view-nick-name :use-cliches)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "always try clichs"
                                            nil
                                            :check-box-checked-p *always-try-cliches*
                                            :view-nick-name :always-try-cliches)
                          
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "builtin threshold only"
                                            nil
                                            :check-box-checked-p *builtin-threshold-only*
                                            :view-nick-name :builtin-threshold-only)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "create clich preds"
                                            nil
                                            :check-box-checked-p *create-preds-from-cliches*
                                            :view-nick-name :create-preds-from-cliches)
                          
                          ;;; Theory Based Learning
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x3 y3) #@(200 16)
                                            "Theory Based Learning" nil
                                            :dialog-item-enabled-p theory?)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x3-o (incf y3 16)) #@(200 16)
                                            "use theory"
                                            nil
                                            :dialog-item-enabled-p theory?
                                            :check-box-checked-p theory?
                                            :view-nick-name :use-theory)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x3-o (incf y3 16)) #@(200 16)
                                            "prefer theory"
                                            nil
                                            :dialog-item-enabled-p theory?
                                            :check-box-checked-p 
                                            (if theory? *theory-mode* nil)
                                            :view-nick-name :theory-mode)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(200 16)
                                            "partial-dt-0-gain"
                                            nil
                                            :dialog-item-enabled-p theory?
                                            :check-box-checked-p 
                                            (if theory? *partial-dt-0-gain* nil)
                                            :view-nick-name :partial-dt-0-gain)
        
                          ;;; Gain-Function    { :information, :ratio }
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x (setf y (+ (max y y2 y3) 28) 
                                                                y2 y
                                                                y3 y))
                                            #@(200 16)
                                            "Gain Function" nil)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x-o (incf y 16)) #@(150 16)
                                            "information" nil
                                            :radio-button-cluster gain-function-cluster
                                            :radio-button-pushed-p (eql *gain-function* :information)
                                            :view-nick-name :information)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x-o (incf y 15)) #@(150 16)
                                            "ratio" nil
                                            :radio-button-cluster gain-function-cluster
                                            :radio-button-pushed-p (eql *gain-function* :ratio)
                                            :view-nick-name :ratio)

                          ;;; Simplify
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x2 y2) #@(200 16)
                                            "Simplify" nil)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 16)) #@(200 16)
                                            "clauses"
                                            nil
                                            :check-box-checked-p *simplify-clauses*
                                            :view-nick-name :simplify-clauses)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "operationalizations"
                                            nil
                                            :dialog-item-enabled-p intensional-preds?
                                            :check-box-checked-p 
                                            (if intensional-preds? *simplify-operationalizations* nil)
                                            :view-nick-name :simplify-operationalizations)
        

                          ;;; Refinement    { :leaves, :frontier }
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x3 (decf y3 30)) #@(200 16)
                                            "Refinement" nil
                                            :dialog-item-enabled-p intensional-preds?)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(150 16)
                                            "leaves" nil
                                            :radio-button-cluster refinement-cluster
                                            :radio-button-pushed-p 
                                            (if intensional-preds? (eql *refinement* :leaves) nil)
                                            :view-nick-name :leaves
                                            :dialog-item-enabled-p intensional-preds?)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(150 16)
                                            "frontier" nil
                                            :radio-button-cluster refinement-cluster
                                            :radio-button-pushed-p 
                                            (if intensional-preds? (eql *refinement* :frontier) nil)
                                            :view-nick-name :frontier
                                            :dialog-item-enabled-p intensional-preds?)
                          
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(200 16)
                                            "prefer children"
                                            nil
                                            :check-box-checked-p
                                            (if intensional-preds? *prefer-children*)
                                            :view-nick-name :prefer-children
                                            :dialog-item-enabled-p intensional-preds?)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(200 16)
                                            "prefer deletions"
                                            nil
                                            :check-box-checked-p 
                                            (if intensional-preds? *prefer-deletions*)
                                            :view-nick-name :prefer-deletions
                                            :dialog-item-enabled-p intensional-preds?)


                          ;;; Graph Learning
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x (setf y (+ 30 (max y y2 y3))
                                                         y2 y
                                                         y3 y))
                                            #@(200 16)
                                            "Display" nil)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 16)) #@(200 16)
                                            "current gain"
                                            nil
                                            :check-box-checked-p 
                                            (if display-list 
                                              (member :current-gain display-list)
                                              t)
                                            :view-nick-name :graph-current-gain)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "best gain"
                                            nil
                                            :check-box-checked-p 
                                            (if display-list 
                                              (member :best-gain display-list)
                                              t)
                                            :view-nick-name :graph-best-gain)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "ebl"
                                            nil
                                            :check-box-checked-p 
                                            (if theory?
                                              (if display-list 
                                                (member :ebl display-list)
                                                t)
                                              nil)
                                            :dialog-item-enabled-p theory?
                                            :view-nick-name :graph-ebl)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "learned definition"
                                            nil
                                            :check-box-checked-p 
                                            (if display-list 
                                              (member :learned-concept-description display-list)
                                              t)
                                            :view-nick-name :graph-learned-concept-description)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x-o (incf y 15)) #@(200 16)
                                            "coverage and work"
                                            nil
                                            :check-box-checked-p 
                                            (if display-list 
                                              (member :coverage-and-work display-list)
                                              t)
                                            :view-nick-name :graph-coverage-and-work)

                          ;;; Low Level Parameters
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x2 y2) #@(200 16)
                                            "Low Level" nil)

                          (make-dialog-item 'editable-text-dialog-item 
                                            (make-point x2-o (incf y2 19)) #@(20 16) 
                                            (format nil "~a" *max-winners*)
                                            nil
                                            :allow-returns nil
                                            :view-nick-name :max-winners)
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point (+ x2-o 25) y2) #@(200 16)
                                            "max winners" nil)

                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 20)) #@(200 16)
                                            "save examples"
                                            nil
                                            :check-box-checked-p *save-examples*
                                            :view-nick-name :save-examples)
        
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "proof vars available"
                                            nil
                                            :check-box-checked-p *proof-vars-available*
                                            :view-nick-name :proof-vars-available)
        
                          (make-dialog-item 'check-box-dialog-item
                                            (make-point x2-o (incf y2 15)) #@(200 16)
                                            "stop when all pos covered"
                                            nil
                                            :check-box-checked-p *stop-when-all-pos-covered*
                                            :view-nick-name :stop-when-all-pos-covered)
        
        
                          ;;; Noise-Tolerance    { :FOIL, :REP or NIL (everything else) }
                          (make-dialog-item 'static-text-dialog-item
                                            (make-point x3 y3) #@(200 16)
                                            "Noise Tolerance" nil)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 16)) #@(150 16)
                                            "foil's" nil
                                            :radio-button-cluster noise-tolerance-cluster
                                            :radio-button-pushed-p (eql *stopping-criteria-enabled* :FOIL)
                                            :view-nick-name :FOIL)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(150 16)
                                            "recursive" nil
                                            :radio-button-cluster noise-tolerance-cluster
                                            :radio-button-pushed-p (eql *stopping-criteria-enabled* :RECURSIVE)
                                            :view-nick-name :RECURSIVE)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(150 16)
                                            "rep" nil
                                            :radio-button-cluster noise-tolerance-cluster
                                            :radio-button-pushed-p (eql *stopping-criteria-enabled* :REP)
                                            :view-nick-name :REP)

                          (make-dialog-item 'radio-button-dialog-item
                                            (make-point x3-o (incf y3 15)) #@(150 16)
                                            "none" nil
                                            :radio-button-cluster noise-tolerance-cluster
                                            :radio-button-pushed-p 
                                            (not (or (eql *stopping-criteria-enabled* :FOIL)
                                                     (eql *stopping-criteria-enabled* :RECURSIVE)
                                                     (eql *stopping-criteria-enabled* :REP)))
                                            :view-nick-name nil)
                           


                          ;;; Dialog Exit - Button Cluster
                           (make-dialog-item 'button-dialog-item
                                            (make-point (- window-h 145)  (- window-v 27))
                                            #@(60 20) " Learn "
                                            #'(lambda (item)
                                                (window-hide (view-container item))

                                                ;;; Graph Learning
                                                (setf *graph-learning* nil)
                                                  (if (check-box-checked-p (find-named-sibling item :graph-learned-concept-description))
                                                    (setf *graph-learning* (push :learned-concept-description *graph-learning*)))
                                                  (if (check-box-checked-p (find-named-sibling item :graph-current-gain))
                                                    (setf *graph-learning* (push :current-gain *graph-learning*)))
                                                  (if (check-box-checked-p (find-named-sibling item :graph-best-gain))
                                                    (setf *graph-learning* (push :best-gain *graph-learning*)))
                                                  (if (check-box-checked-p (find-named-sibling item :graph-ebl))
                                                    (setf *graph-learning* (push :ebl *graph-learning*)))
                                                  (if (check-box-checked-p (find-named-sibling item :graph-coverage-and-work))
                                                    (setf *graph-learning* (push :coverage-and-work *graph-learning*)))
                                                  (when *graph-learning*
                                                    (setf *graph-learning* (push :pause *graph-learning*)))
                     
                     
                                                ;;; Gain-Function    { :information, :ratio }
                                                (setf *gain-function*
                                                      (pushed-radio-button (view-container item) gain-function-cluster))
                                                (when *gain-function*
                                                  (setf *gain-function* (view-nick-name *gain-function*)))
  
                                                ;;; Empirical Learning
                                                (setf max-new-variables
                                                      (read-from-string (dialog-item-text (find-named-sibling item :max-new-variables))))
                                                (setf *intensional-induction* 
                                                      (check-box-checked-p (find-named-sibling item :intensional-induction)))
                                                (setf *operationalize-intensional*
                                                      (check-box-checked-p (find-named-sibling item :operationalize-intensional)))
                                                (setf *enable-determinate-literals*
                                                      (check-box-checked-p (find-named-sibling item :enable-determinate-literals)))
                                                (setf *max-determinate-literals*
                                                      (read-from-string (dialog-item-text (find-named-sibling item :max-determinate-literals))))


                                                ;;; Constructive Induction - Cliche's
                                                (setf *max-new-cliche-vars*
                                                      (read-from-string (dialog-item-text (find-named-sibling item :max-new-cliche-vars))))
                                                (setf *use-relational-cliches* 
                                                      (check-box-checked-p (find-named-sibling item :use-cliches)))
                                                (setf *builtin-threshold-only* 
                                                      (check-box-checked-p (find-named-sibling item :builtin-threshold-only)))
                                                (setf *always-try-cliches* 
                                                      (check-box-checked-p (find-named-sibling item :always-try-cliches)))
                                                (setf *create-preds-from-cliches*
                                                      (check-box-checked-p (find-named-sibling item :create-preds-from-cliches)))
                                               
                                                ;;; Theory Based Learning
                                                (unless (check-box-checked-p (find-named-sibling item :use-theory))
                                                  (setf clauses nil
                                                        goal-concept-name nil))
                                                (setf *theory-mode*
                                                      (check-box-checked-p (find-named-sibling item :theory-mode)))
                                                (setf *refinement*
                                                      (pushed-radio-button (view-container item) refinement-cluster))
                                                (when *refinement*
                                                  (setf *refinement* (view-nick-name *refinement*)))

                                                (setf *prefer-children*
                                                      (check-box-checked-p (find-named-sibling item :prefer-children)))
                                                (setf *prefer-deletions* 
                                                      (check-box-checked-p (find-named-sibling item :prefer-deletions)))

                                                (setf *partial-dt-0-gain* 
                                                      (check-box-checked-p (find-named-sibling item :partial-dt-0-gain)))
                                                
                                                ;;; Simplicifation
                                                (setf *simplify-operationalizations* 
                                                      (check-box-checked-p (find-named-sibling item :simplify-operationalizations)))
                                                (setf *simplify-clauses* 
                                                      (check-box-checked-p (find-named-sibling item :simplify-clauses)))
                                                
                                                ;;; Noise-Tolerance
                                                (setf *stopping-criteria-enabled*
                                                      (pushed-radio-button (view-container item) noise-tolerance-cluster))
                                                (when *stopping-criteria-enabled*
                                                  (setf *stopping-criteria-enabled* (view-nick-name *stopping-criteria-enabled*)))
                                                
                                                ;;; Low Level
                                                (setf *max-winners*
                                                      (read-from-string (dialog-item-text (find-named-sibling item :max-winners))))
                                                (setf *save-examples* 
                                                      (check-box-checked-p (find-named-sibling item :save-examples)))
                                                (setf *proof-vars-available* 
                                                      (check-box-checked-p (find-named-sibling item :proof-vars-available)))
                                                (setf *stop-when-all-pos-covered* 
                                                      (check-box-checked-p (find-named-sibling item :stop-when-all-pos-covered)))

                                                (return-from-modal-dialog nil))
                      
                                            :view-nick-name :LEARN
                                            :default-button learnable? 
                                            :dialog-item-enabled-p learnable?)
                         
                         (make-dialog-item 'button-dialog-item
                                           (make-point (- window-h 70)  (- window-v 27))
                                           #@(60 20) " Cancel "
                                           #'(lambda (item) 
                                               (declare (ignore item))
                                               (return-from-modal-dialog :cancel))
                                           :default-button (not learnable?))
                         
                         ))))
          
    (modal-dialog modify-parameters-dialog t)
    (values goal-concept-name
            clauses
            reset-hash-tables
            max-new-variables
            reset-statistics)

    ))
  

;;;__________________________________________________________________________________
;;; learn

(defun learn ()
  (when (and (fboundp 'focl)
             (pred-p (get-pstruct *answer-pred*)))
    (let ((parameters *focl-parameters*))
      (setf (getf parameters :goal-concept-name) *top-level-call*)
      (setf *error-output* *terminal-io*)
      (window-hide *top-listener*)
      (catch-cancel (apply #'focl *answer-pred* parameters))
      (setf *error-output* CCL::*pop-up-terminal-io*)
      (window-select *top-listener*))
    (print *learned-concept-description*))
  (values))


;;;____________________________________________________________________________________
;;;  Learn Menu

(defparameter *learn-menu*
  (let ((learn-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Learn"
                        :disabled t
                        :menu-item-action #'(lambda () (eval-enqueue '(learn)))
                        :update-function #'(lambda (item) 
                                             (if (and (fboundp 'learn)
                                                      (get *answer-pred* 'pred))
                                               (menu-item-enable item)
                                               (menu-item-disable item)))))
        (set-parameters-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Learning Parameters"
                        :disabled t
                        :menu-item-action #'(lambda () (eval-enqueue '(learn)))
                        :update-function #'(lambda (item) 
                                             (if (get *answer-pred* 'pred)
                                               (menu-item-enable item)
                                               (menu-item-disable item))
                                             (menu-item-disable item))))
        (apply-heuristics-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Revise"
                        :menu-item-action #'(lambda () (eval-enqueue '(apply-heuristics)))
                        :update-function #'(lambda (item) (if (and (fboundp 'apply-heuristics)
                                                                   *learned-concept-description*
                                                                   *GOAL-CONCEPT*
                                                                   *KR-APPLICABLE*)
                                                            (menu-item-enable item)
                                                            (menu-item-disable item)))))
        (inspect-relations-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Inspect Relations"
                        :menu-item-action #'(lambda () (eval-enqueue '(inspect-relations)))
                        :update-function #'(lambda (item) (if (or *intensional-preds*
                                                                  *extensional-preds*
                                                                  *builtin-preds*)
                                                            (menu-item-enable item)
                                                            (menu-item-disable item)))))
        (inspect-cliches-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Inspect Clichs"
                        :menu-item-action #'(lambda () (eval-enqueue '(inspect-cliches)))
                        :update-function #'(lambda (item) (if *available-relational-cliches*
                                                            (menu-item-enable item)
                                                            (menu-item-disable item)))))
        )
    (make-instance 'menu
                   :menu-title "Learn"
                   :menu-items (list learn-menu-item
                                     set-parameters-menu-item
                                     (make-instance 'menu-item :menu-item-title "-" :disabled t)
                                     apply-heuristics-menu-item
                                     (make-instance 'menu-item :menu-item-title "-" :disabled t)
                                     inspect-relations-menu-item
                                     inspect-cliches-menu-item
                                     ))
    ))

(provide :animate-learning)
