;;;____________________________________________________________________________________
;;; The code in this file is designed to work in conjunction with grapher.lisp
;;; to animate the learned-concept-description.
;;;
;;;  This file also contains the graph generation code needed to construct gui-graphs
;;;  from literal structures.
;;;
;;;  Created and designed by Clifford A. Brunk 05/20/91
;;;
;;;  Problems:
;;;____________________________________________________________________________________


(require 'grapher)

(in-package :user)

;;;____________________________________________________________________________________
;;; return-literal-struct-corresponding-to-node

(defun return-literal-struct-corresponding-to-node (root node)
  (let* ((conjunction (conjunction-containing-node node))
         (literal-position (position node conjunction))
         (clause-position (position conjunction (node-children root)))
         (literal (if (eq *status* :finished) 
                    (elt *learned-concept-description* clause-position)
                    (car (last *learned-concept-description* (+ clause-position 1))))))
    (dotimes  (i literal-position literal)
      (setf literal (literal-next literal)))))


;;;____________________________________________________________________________________
;;;  inspect-last-node-selected-as-literal

(defmethod inspect-last-node-selected-as-literal ((view graph-view))
  (let ((selected-node (last-node-selected view))
        (root (graph-root view)))
    (when selected-node
      (if (eq (node-parent selected-node) root)
        (inspect (return-literal-struct-corresponding-to-node root selected-node))
        (inspect-last-node-selected-as-pred  view)))))

;;;____________________________________________________________________________________
;;; Learned Concept Description Graph Generation
;;;____________________________________________________________________________________


;;;____________________________________________________________________________________
;;; generate-learned-definition-graph

(defmethod generate-learned-definition-graph ((view graph-view) root-literal theory)
  (when (and root-literal theory)
    (setf (graph-root view) (create-node view
                                         :literal root-literal
                                         :kind :intensional)
          (node-children (graph-root view))
          (connect-clauses view (graph-root view) theory :normal t))
    (if (eql (graph-expand view) :first-use)
      (display-only-first-use view))
    (position-nodes view)))

;;;____________________________________________________________________________________
;;;  create-learned-definition-graph-window

(defun create-learned-definition-graph-window (head learned-definition)
  (let* ((window (make-instance 'graph-window :window-show nil :kind :learned-concept))
         (view (view-named :graph-view (view-named :graph-scroller window))))
    (generate-learned-definition-graph view head learned-definition)
    (resize-window window)
    (position-graph view :centered t)
    (auto-position-window window :centered t)
    (set-window-title window " Learned Concept Description ")
    (window-select window)))

;;;____________________________________________________________________________________
;;; Learned Concept Description Graph Animation Code
;;;____________________________________________________________________________________

;;;____________________________________________________________________________________
;;;  add-new-clause-to-lcd-graph

(defun add-new-clause-to-lcd-graph (literal)
  (when (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*)
    (let ((view (view-named :graph-view (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))))
      (setf (node-children (graph-root view))
            (append (node-children (graph-root view))
                    (list (connect-clause view (graph-root view) literal :normal t))))
      (position-nodes view)
      (reset-scroll-bars (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))
      (force-graph-redraw view))))

;;;____________________________________________________________________________________
;;; replace-last-clause-in-lcd-graph

(defun replace-last-clause-in-lcd-graph (new-clause)
  (when (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*)
    (let ((view (view-named :graph-view (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))))
      (setf (node-children (graph-root view)) (butlast (node-children (graph-root view)))))
    (add-new-clause-to-lcd-graph new-clause)))

;;;____________________________________________________________________________________
;;; add-literal-to-last-clause-in-lcd-graph

(defun add-literal-to-last-clause-in-lcd-graph (literal)
  (when (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*)
    (let ((view (view-named :graph-view (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))))
         (setf (car (last (node-children (graph-root view))))
               (append (car (last (node-children (graph-root view)))) 
                       (connect-clause view (graph-root view) literal :normal t)))
      (position-nodes view)           ;;   <----    Perhaps this could be done better
      (reset-scroll-bars (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))
      (force-graph-redraw view))))


;;;____________________________________________________________________________________
;;; setup-LCD-WINDOW

(defun setup-LCD-WINDOW (prolog-literal-being-learned)
  (unless (window-p *LEARNED-CONCEPT-DESCRIPTION-WINDOW*)
    (setf *LEARNED-CONCEPT-DESCRIPTION-WINDOW* (make-instance 'graph-window
                                      :window-show nil
                                      :view-size #@(450 275)
                                      :view-position (make-point (- *screen-width* 453) (- *screen-height* 278))
                                      :window-title "Learned Concept Description"
                                      :kind :learned-concept)))
  (let ((view (view-named :graph-view (view-named :graph-scroller *LEARNED-CONCEPT-DESCRIPTION-WINDOW*))))
    (setf (graph-root view) (create-node view 
                                         :parent nil
                                         :literal prolog-literal-being-learned)) 
    (setf (node-kind (graph-root view)) :intensional)
    (position-nodes view)
    (position-graph view :centered t)))

(provide :animate-learned-definition)
