(in-package :user)

(defmacro get-text-from-user (&rest x)
 (cons 'get-string-from-user x))

(defmacro catch-error-quietly (&body body)
  `(multiple-value-bind (valval ee) 
                        (ignore-errors (let ((catch-error-quietly-value (progn .,body)))
                          catch-error-quietly-value))
     (cond (ee (values nil ee))
           (t valval))))
                       

(defvar *answer-pred* nil)
(defvar *=* '(= . =))
(defvar *top-level-call* nil)
(defvar *fact-file* nil)
(defvar *Rule-file* nil)
(defparameter *type-files* nil)
(setq *facts-changed* nil)
(defvar *rules-changed* nil)
(defvar *use-menu* t)
(defvar *max-l* 6)
(setq *compile-with-rule-trace* t)
(setq *compile-each-clause* (fboundp 'focl))  ;; CB 07/27/91
(defvar *last-explanation* nil)
 

(defun trim-list (x &optional (m *max-l*) (end nil) &aux (length (length x) ))
  (if m
    (if (> length m)
      (append (butlast x (- length m)) end)
      x)
    x))


(defmacro notify-error(&rest message)
  `(message-dialog (format nil . ,message)
                 :size #@(480 200)))

(defun dump-pred (pred stream)
  (format stream "~%(def-pred ~s~%" (pred-name pred))
  (format stream "   :pos ~s~%" (pred-pos pred))
  (format stream "   :neg ~s~%" (pred-neg pred))
  (format stream "   :type ~s~%" (pred-type pred))
  (format stream "   :constraint ~s~%" (pred-constraint pred))
  (format stream "   :mode ~s~%" (pred-mode pred))
  (format stream "   :commutative ~s~%" (pred-commutative pred))
  (format stream "   :induction ~s~%" (pred-induction pred))
  (format stream "   :infix ~s~%" (pred-infix pred))
  (format stream "   :vars ~s~%" (pred-vars pred))
  (format stream "   :questions (~{~s~%     ~})" (pred-questions pred))
  (format stream ")" ) 
)


(defun dump-rule(p stream)
  (let  ((clauses (get (rule-name p) 'clauses)))
    (format stream "~%(def-rule ~s~%" (rule-name p))
    (format stream "   :clauses~%  (")
    (mapc #'(lambda(x)(print x stream)) clauses)
    (format stream "  )~%")    
    (format stream "   :type ~s~%" (rule-type p))
    (format stream "   :constraint ~s~%" (rule-constraint p))
    (format stream "   :mode ~s~%" (rule-mode p))
    (format stream "   :vars ~s~%" (rule-vars p))
    (format stream "   :questions ~s~%" (rule-questions p))
    (format stream "   :commutative ~s~%" (rule-commutative p))
    (format stream "   :induction ~s~%" (rule-induction p))
    (format stream ")" )
    ))


(defun re-def-rule (r &key (name (rule-name r)) 
                      (clauses nil) 
                      (type (rule-type r))
                      (constraint (rule-constraint r))
                      (mode (rule-mode r))
                      (commutative (rule-commutative r))
                      (induction (rule-induction r))
                      (vars nil)
                      (questions nil))
  (if (null clauses)
    (setf clauses (if (null vars)
                    (get name 'clauses)
                    (mapcar #'(lambda(c) (cons (cons name vars) 
                                               (nsublis (mapcar #'cons (cdr(car c)) vars)
                                                        (cdr c)
                                                        :test #'equalp)))
                            (get name 'clauses)))))
  (if (null questions)
    (setq questions (if (null vars)
                      (rule-questions r)
                      (nsublis (mapcar #'cons (rule-vars r) vars)
                               (rule-questions r)
                               :test #'equalp))))
  (if (null vars)
    (setq vars (rule-vars r)))
                      
  
  (eval `(def-rule ,name 
           :clauses ,clauses
           :type ,type
           :constraint ,constraint
           :mode ,mode
           :commutative ,commutative
           :induction ,induction
           :vars ,vars
           :questions ,questions))
  (setq *rules-changed* t)
  (show-rule-def name)
  name)



(defun dump-rules(&optional (f *rule-file*))
  (setq *rule-file* f)
  (with-open-file (s f :direction :output :if-exists :supersede)
    (format s "(reset-rules)~%")
    (format s "(def-es-focl-problem ~s ~s)~%" *answer-pred*  *top-level-call* )
    (mapc #'(lambda(name.rule)
              (dump-rule (cdr name.rule) s)
              (format t "~a~%" (car name.rule)))
          (reverse *intensional-preds* )))
  (set-mac-file-type f (if *type-files* 
                         'rule
                         'text))
  (setq *rules-changed* nil)
  (format t "Saved Rules in ~s" f))

(defun dump-facts (&optional (f *fact-file*))
  (setq *fact-file* f)
  (with-open-file (s f :direction :output :if-exists :supersede)
    (format s "(reset-facts)~%")
    (mapc #'(lambda(name.rule)
              (dump-pred   (cdr name.rule) s)
              (format t "~a~%" (car name.rule)))
          (reverse *extensional-preds*))
    (format s "~%")
    (mapc #'(lambda(type)
              (format s "(def-type ~a ~{~a ~})~%" type (get-type-instances type))
              (format t "TYPE: ~a~%" type))
          (reverse *all-types*)))
  (set-mac-file-type f (if *type-files* 
                         'fact
                         'text))
  (setq *facts-changed* nil)
  (format t "Saved Facts in ~s" f))


                   

(defun dump-facts-as()
  (catch-cancel (dump-facts (choose-new-file-dialog 
                             :button-string "Save Facts"))))

(defun dump-rules-as()
  (catch-cancel (dump-rules (choose-new-file-dialog 
                                                    :button-string "Save Rules"))))

(defun abort-operation(l1)
  (y-or-n-dialog (format nil "~a~%Abort?" l1)
                 :yes-text "Yes, Abort"
                 :no-text  "No"
                 :size #@(480 200)
                 :cancel-text nil))
  


(defun load-rules(&aux f)
  (unless (or (and *rules-changed* 
                   (abort-operation "Rules have been modified and not saved"))
              (eq :cancel (setq f (catch-cancel (choose-file-dialog :mac-file-type (if *type-files* 
                                                                                     'rule
                                                                                     'text)
                                                                    :button-string "Load Rules")))))
    (multiple-value-bind (value error) 
                         (catch-error-quietly
                           (cond ((with-open-file (s f)
                                    (equal (read s) '(reset-rules)))
                                  (load f :verbose t :print t))
                                 (t (notify-error "Not a valid rule file~%") :error)))
    (cond((eq value :error))
         ((null error)
            (format t "~%Loaded.~%")
            (setq *rules-changed* nil)
            (setq *rule-file* f))
           (t (notify-error "~%An error occured: ~a No rules are loaded~%" error)
              (reset-rules))))))

(defun initialize-everything()
  (unless (and (or *top-level-call* *intensional-preds* *extensional-preds* *answer-pred* )
             (y-or-n-dialog "The rule base has already been initialized.  Abort?"
                                :yes-text "Yes"
                                :no-text  "No"
                                :cancel-text nil))
    (reset-rules)
    (reset-facts)
    (let* ((p (get-atom-from-user "What is the name of the top level goal to prove?")))
      (when p
        (let ((a (get-args "What arguments should be used. Suggestion: (?example) for binary classifications or (?example ?class) for multi-class classifications" nil))
              (ap (intern (format nil "~a-FACT" p))))
          (when a
            (let ((types (get-types p a)))
              (when types
            (let ((q (get-question (cons p a) nil)))
              (when q
                
                (eval `(def-rule ,p :clauses nil :questions ,q :vars ,a :type ,types))
                (eval `(def-pred ,ap :questions ((:FACT ,q)) :vars ,a :type ,types))
                (eval `(def-pred example :vars ,a :type ,types))
                (format t "~%")
                (eval `(def-es-focl-problem ,ap ,p))
                (setq *rules-changed* t)
                (setq *facts-changed* t)
                (setq *fact-file* nil)
                (setq *rule-file* nil)
                ))))))))))
          

(defun load-facts(&aux f)
  (unless (or (and *facts-changed* 
                  (abort-operation "Facts have been modified and not saved"))
              (eq :cancel (setq f (catch-cancel (choose-file-dialog :mac-file-type (if *type-files* 
                                                                                     'fact
                                                                                     'text)
                                                                    :button-string "Load Facts")))))
    (multiple-value-bind (value error) 
                         (catch-error-quietly
                           (cond ((with-open-file (s f)
                                    (equal (read s) '(reset-facts)))
                                  (load f :verbose t :print t))
                                 (t (notify-error "Not a valid fact file") :error)))
      (cond((eq value :error))
            ((null error)
            (format t "~%Loaded.~%")
            (setq *facts-changed* nil)
            (setq *fact-file* f))
           (t (notify-error "~%An error occured: ~a No facts are loaded~%" error)
              (reset-facts))))))

(defun load-something(&aux f)
  (unless (eq :cancel (setq f (catch-cancel (choose-file-dialog :mac-file-type (if *type-files* 
                                                                                 'fact
                                                                                 'text)
                                                                :button-string "Import"))))
    (multiple-value-bind (value error) 
                         (catch-error-quietly
                           (with-open-file (s f)
                             (do ((e (read s nil :end-of-file)(read s nil :end-of-file)))
                                 ((eq e :end-of-file))
                               (unless (and (consp e)
                                            (or (member (car e)
                                                    '(reset-facts reset-rules def-es-focl-problem))
                                                (eq (cadr e) 'example)))
                                 (print (eval e))))))
      (declare (ignore value))
      (cond((null error)
            (format t "~%Loaded.~%"))
           (t (notify-error "An error occured ~a" error)
              )))))

(defun reset-facts()  ;;used to change problems
  (remove-pred-defs *extensional-preds* 'pred)
  (setq *extensional-preds* nil)
  (setq *all-types* nil)
  'RESETTING_FACTS)

(defun reset-rules() 
(remove-pred-defs *intensional-preds* 'rule)
  (remove-pred-defs *intensional-preds* 'brules)
  (remove-pred-defs *intensional-preds* 'clauses)
  (setq *intensional-preds* nil)
  'RESETTING_RULES)


(def-builtin < #'<  
  :treat-as-commutative nil
  :one-variable-comp nil
  :questions (?x is less than ?y)
  :vars (?x ?y)
  :induction t)

(def-builtin <= #'<=  
  :treat-as-commutative nil
  :one-variable-comp nil
  :questions (?x is less than or equal to ?y)
  :vars (?x ?y)
  :induction t)

(def-builtin >= #'>=  
  :treat-as-commutative nil
  :one-variable-comp nil
  :questions (?x is greater than or equal to ?y)
  :vars (?x ?y)
  :induction t)

(def-builtin math-= #'=  
  :treat-as-commutative nil
  :one-variable-comp nil
  :questions (?x is equal to ?y)
  :vars (?x ?y)
  :induction t)

(def-builtin > #'>
  :treat-as-commutative nil
  :one-variable-comp nil
  :questions (?x is greater than ?y)
  :vars (?x ?y)
  :induction t)

(defvar *original-file-menu* (find-menu "File"))
(defvar *original-edit-menu* (find-menu "Edit"))
(defvar *original-eval-menu* (find-menu "Eval"))
(defvar *original-tools-menu* (find-menu "Tools"))
(defvar *original-windows-menu* (find-menu "Windows"))

(defun save-menus-to-globals()
       (unless *original-file-menu*
         (setq *original-file-menu* (find-menu "File"))
         (setq *original-edit-menu* (find-menu "Edit"))
         (setq *original-eval-menu* (find-menu "Eval"))
         (setq *original-tools-menu* (find-menu "Tools"))
         (setq *original-windows-menu* (find-menu "Windows"))))



(defun reset-menubar()
       (remove-menu-items *es-file-menu* *original-file-menu* *original-eval-menu* *original-tools-menu*)
       (menu-deinstall *es-file-menu*)
       (menu-deinstall *edit-menu*)
       (menu-deinstall *es-rules-menu*)
       (menu-deinstall *es-facts-menu*)
       (menu-deinstall *es-type-menu*)
       (menu-deinstall *es-run-menu*)
       (when (boundp '*focl-windows-menu*)
         (menu-deinstall *focl-windows-menu*)
         (menu-deinstall *grapher-menu*)
         (menu-deinstall *learn-menu*))
       (menu-install *original-file-menu*)
       (menu-install *original-edit-menu*)
       (menu-install *original-eval-menu*)
       (menu-install *original-tools-menu*)
       (menu-install *original-windows-menu*)
       nil)
      

        

(defparameter *save-rules-item* (make-instance 'menu-item
                                 :menu-item-title "Save Rules"
                                 :menu-item-action #'dump-rules))

(defparameter *save-rules-as-item* (make-instance 'menu-item
                                    :menu-item-title "Save Rules As"
                                    :menu-item-action #'dump-rules-as))

(defparameter *save-facts-item* (make-instance 'menu-item
                                 :menu-item-title "Save Facts"
                                 :menu-item-action #'dump-facts))


(defparameter *save-facts-as-item* (make-instance 'menu-item
                                    :menu-item-title "Save Facts As"
                                    :menu-item-action #'dump-facts-as))
                                    

(defparameter *load-facts-item* (make-instance 'menu-item
                                 :menu-item-title "Load Facts"
                                 :menu-item-action #'load-facts))

(defparameter *import-facts-item* (make-instance 'menu-item
                                 :menu-item-title "Import Rules Or Facts"
                                 :menu-item-action #'load-something))

(defparameter *load-rules-item* (make-instance 'menu-item
                                 :menu-item-title "Load Rules"
                                 :menu-item-action #'load-rules))

(defparameter *initialize-item* (make-instance 'menu-item
                                 :menu-item-title "Initialize Rules and Facts"
                                 :menu-item-action #'(lambda()(eval-enqueue '(initialize-everything)))))




(defun quit-es()
  (catch-cancel 
    (when (and *rules-changed* 
               (y-or-n-dialog "Do you want to save the rule file?"))
      (dump-rules (or *rule-file* (choose-new-file-dialog 
                                   :button-string "Save Rules"))))
    (when (and *facts-changed* 
               (y-or-n-dialog  "Do you want to save the fact file?"))
      (dump-facts (or *fact-file* (choose-new-file-dialog 
                                   :button-string "Save Facts"))))
    (ccl:quit)))
    


(defparameter *es-file-menu*             ;; CB(02)  Reordered Menu
 (make-instance 'menu 
               :Menu-title "File"
               :menu-items (list *save-facts-item*
                                 *save-facts-as-item* 
                                 (make-instance 'menu-item :menu-item-title "-")
                                 *save-rules-item*
                                 *save-rules-as-item* 
                                 (make-instance 'menu-item :menu-item-title "-")
                                 *load-facts-item*
                                 *load-rules-item*
                                 (make-instance 'menu-item :menu-item-title "-")
                                 *import-facts-item*
                                 (make-instance 'menu-item :menu-item-title "-")
                                 (find-menu-item *file-menu* "Print")
                                 (make-instance 'menu-item
                                        :menu-item-title "Quit"
                                        :menu-item-action 
                                        #'quit-es)
                                 (make-instance 'menu-item :menu-item-title "-")
                                 (make-instance 'menu-item
                                    :menu-item-title "Return To Lisp"
                                    :menu-item-action #'reset-menubar))
                                 ))

(defparameter *judge-item*
  (make-instance 'menu-item
         :menu-item-title "Test On All Examples"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(judge-rules-on-examples)))))

(defparameter *run-new-item*
  (make-instance 'menu-item
         :menu-item-title "Run On New Example"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(run-rules-on-examples t)))))

(defparameter *run-old-item*
  (make-instance 'menu-item
         :menu-item-title "Run On Old Example"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(run-rules-on-examples)))))

(defparameter *check-rules*
  (make-instance 'menu-item
         :menu-item-title "Check Rules"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(check-rules)))))


(defparameter *no-menu-item*
  (make-instance 'menu-item
         :menu-item-title "Type In New Clauses"
         :menu-item-action 
         #'(lambda()(setq *use-menu* nil))))
(defparameter *yes-menu-item*
  (make-instance 'menu-item
         :menu-item-title "Use Menus For New Clauses"
         :menu-item-action 
         #'(lambda()(setq *use-menu* t))))


(defparameter *who-calls*
  (make-instance 'menu-item
         :menu-item-title "Who Calls"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(who-calls)))))

(defparameter *prove-goal*
  (make-instance 'menu-item
         :menu-item-title "Prove Goal"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(prove-goal-typed-in)))))

(defparameter *trace-all*
  (make-instance 'menu-item
         :menu-item-title "Trace All"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(setq *traced-predicates* :all)))))

(defparameter *untrace-all*
  (make-instance 'menu-item
         :menu-item-title "Untrace All"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(setq *traced-predicates* nil)))))

(defparameter *add-trace*
  (make-instance 'menu-item
         :menu-item-title "Add Trace"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(add-trace)))))

(defparameter *remove-trace*
  (make-instance 'menu-item
         :menu-item-title "Remove Trace"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(remove-trace)))))

(defparameter *explain-item*
  (make-instance 'menu-item
         :menu-item-title "Explain Last Proof"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(explain-last)))))

(defun explain-last()
  (print-trace *last-explanation*))

(defvar *new-type-item*
  (make-instance 'menu-item
         :menu-item-title "New Type"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(new-type)))))

(defun new-type()
  (let ((name (get-atom-from-user "Enter a name for the new type: ")))
    (when name
      (pushnew name *all-types*)
      (setq *facts-changed* t)
      (do ((new (get-new-instance-for-type name)
                (get-new-instance-for-type name)))
          ((null new)))
      (list-type name))))

(defun get-new-instance-for-type(type)
  (let ((name (get-atom-from-user (format nil "Enter a new instance of the type ~a" type)
                                  "No More")))
    (when name 
      (setq *facts-changed* t)
      (pushnew name (get-type-instances type)))))

(defun select-type(&aux c)
  (unless (eq :cancel (setq c (catch-cancel (select-item-from-list *all-types* :window-title "Select a type"))))
    (car c)))

(defun remove-trace(&aux c)
  (unless (eq :cancel (setq c (catch-cancel 
                              (select-item-from-list *traced-predicates*  :selection-type :disjoint :window-title "Select Predicates to Untrace"))))
    (setq *traced-predicates* (set-difference *traced-predicates* c))))

(defun add-trace(&aux c p)
  (setq p (sort (delete-if #'(lambda(x)(member x *traced-predicates*))
                           (nconc (mapcar #'car *builtin-preds*)
                       (mapcar #'car *extensional-preds*)
                       (mapcar #'car *intensional-preds*)
                       (list 'not '= 'is)))
                      #'(lambda(x y)(string< (symbol-name x) (symbol-name y)))))
  (unless (eq :cancel (setq c (catch-cancel 
                              (select-item-from-list p  :selection-type :disjoint :window-title "Select Predicates to Untrace"))))
    (setq *traced-predicates* (nconc *traced-predicates* c))))

(defvar *list-types-item*
  (make-instance 'menu-item
         :menu-item-title "List All Types"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(list-types)))))

(defun list-types()
  (mapc #'(lambda(X)(list-type x)(format t "~%~%")) *all-types*))
(defun list-type(x)
  (when x
    (format t "~%TYPE ~a:" x)
    (mapc #'(lambda(X)(format t "~%     ~a" x))
          (get-type-instances x))))




(defvar *delete-from-type-item*
  (make-instance 'menu-item
         :menu-item-title "Delete From Type"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(delete-from-type)))))

(defun delete-from-type()
  (let ((type (select-type)) c)
    (when (and type 
               (not(eq :cancel (setq c (catch-cancel (select-item-from-list (get-type-instances type)
                                                                            :window-title
                                                                            (format nil "Select element(s) of ~a to be deleted" type)
                                                                            :selection-type :disjoint)))))
               c)
      (setf (get-type-instances type) (set-difference (get-type-instances type) c))
      (list-type type)
      (setq *facts-changed* t))))

(defvar *delete-type-item*
  (make-instance 'menu-item
         :menu-item-title "Delete Type"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(delete-type)))))

(defun delete-type()
  (let ((type (select-type)))
    (when (and type (y-or-n-dialog (format nil "Do you really want to delete ~a?" type)
               :yes-text "Yes, Delete"
               :no-text  "No"
               :cancel-text nil))
      (setf (get-type-instances type) nil)
      (setq *facts-changed* t)
      (setf *all-types* (delete type *all-types*)))))

(defvar *add-to-type-item*
  (make-instance 'menu-item
         :menu-item-title "Add To Type"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(add-to-type)))))

(defun add-to-type()
  (let ((name (select-type)))
    (when name
      (do ((new (get-new-instance-for-type name)
                (get-new-instance-for-type name)))
          ((null new)))
      (list-type name))))

(defvar *list-a-type-item*
  (make-instance 'menu-item
         :menu-item-title "List Type"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(list-a-type)))))

(defun list-a-type()
  (list-type (select-type)))


(defparameter *es-run-menu*
 (make-instance 'menu 
               :Menu-title "Run"
               :menu-items (list *run-new-item* *run-old-item* *judge-item* *prove-goal*
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *check-rules* *who-calls*
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *yes-menu-item* *no-menu-item*
               (make-instance 'menu-item
                      :menu-item-title "-")
               *explain-item*
               (make-instance 'menu-item
                     :menu-item-title "-")
              *trace-all*
              *untrace-all*
              *add-trace*
              *remove-trace*
              )))


(defparameter *es-type-menu*
 (make-instance 'menu 
               :Menu-title "Types"
               :menu-items (list *new-type-item* 
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *list-a-type-item* *list-types-item* 
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *add-to-type-item* *delete-from-type-item* *delete-type-item*)))




(defun select-rule(&optional (title  "Select A Rule") (exceptions nil) (alist *intensional-preds*) &aux v)
  (setq v (catch-cancel (select-item-from-list (delete-if #'(lambda(x)
                                                              (member x exceptions))
                                                          (sort (mapcar #'car alist)
                                                                #'(lambda(x y)(string< (symbol-name x)
                                                                                       (symbol-name y)))))
                                               
                                               :window-title title)))
  (unless (eq v :cancel)
    (first v)))

(defun select-clause(c &optional (title  "Select A Clause") &aux v)
  (setq v (catch-cancel (select-item-from-list c 
                                               :table-print-function #'(lambda(p s)
                                                                         (princ (cdr p) s))
                                               :window-title title)))
  (unless (eq v :cancel)
    (first v)))

(defun select-literal( c &optional (title  "Select A Literal") &aux v)
    (setq v (catch-cancel (select-item-from-list  c :window-title title)))
  (unless (eq v :cancel)
    (first v)))

(defun new-rule()
  (let ((p (progn (Format t "Type name of predicate to be concluded~%")
                  (type-in-predicate))))
    (when p 
      (if (get-pstruct p)
        (notify-error "There is already a predicate named ~a~%Delete it first, or use a different name.~%~%" p)
        (let ((a (get-arguments p)))
          (when a
            (let ((types (get-types p a)))
              (when types
            (let ((q (get-question (cons p a) nil)))
              (when q
                (let ((c (create-new-clause p a)))
                  (when c
                    (eval `(def-rule ,p :clauses (,c) :questions ,q :vars ,a :type ,types))
                    (setq *rules-changed* t) 
                    (show-rule-def p)))))))))))))


(defun copy-a-rule()
  (let ((r (select-rule "Select A Rule To Be Copied")))
    (when r 
      (let ((p (progn (Format t "Type a name for the copy of ~a~%" r)
                      (type-in-predicate)))
            (s (get-pstruct r)))
        (when p 
          (cond((get-pstruct p)
                (notify-error "There is already a predicate named ~a~%Delete it first, or use a different name.~%~%" p))
               (t (eval `(def-rule ,p :clauses ,(subst p r (get r 'clauses))
                           :type ,(rule-type s) :constraint ,(rule-constraint s)
                           :mode ,(rule-mode s) :commutative ,(rule-commutative s)
                           :induction ,(rule-induction s) :vars ,(rule-vars s)
                           :questions ,(rule-questions s)))
                  (setq *rules-changed* t) 
                  (show-rule-def p))))))))


(defun rename-a-rule()
  (let ((r (select-rule "Select A Rule To Be Renamed")))
    (when r 
      (let ((p (progn (Format t "Type a new name for ~a~%" r)
                      (type-in-predicate)))
            (s (get-pstruct r)))
        (when p 
          (cond((get-pstruct p)
                (notify-error "There is already a predicate named ~a~%Delete it first, or use a different name.~%~%" p))
               (t (eval `(def-rule ,p :clauses ,(subst p r (get r 'clauses))
                           :type ,(rule-type s) :constraint ,(rule-constraint s)
                           :mode ,(rule-mode s) :commutative ,(rule-commutative s)
                           :induction ,(rule-induction s) :vars ,(rule-vars s)
                           :questions ,(rule-questions s)))
                  (setq *rules-changed* t)
                  (delete-rule r)
                  (show-rule-def p)
                  (if (eq r *top-level-call*)
                    (setq *top-level-call* p))
                  (revise-callers r p))))))))


(defun rename-a-fact()
  (let ((r (select-rule "Select A Fact To Be Renamed" (list *answer-pred* 'example) *extensional-preds*)))
    (when r 
      (let ((p (progn (Format t "Type a new name for ~a~%" r)
                      (type-in-predicate)))
            (s (get-pstruct r)))
        (when p 
          (cond((get-pstruct p)
                (notify-error "There is already a predicate named ~a~%Delete it first, or use a different name.~%~%" p))
               (t (eval `(def-pred ,p :pos ,(pred-pos s) :neg ,(pred-neg s)
                           :type ,(pred-type s) :constraint ,(pred-constraint s)
                           :mode ,(pred-mode s) :commutative ,(pred-commutative s)
                           :induction ,(pred-induction s) :vars ,(pred-vars s)
                           :infix ,(pred-infix s)
                           :questions ,(pred-questions s)))
                  
                  (setf (get r 'pred) nil *facts-changed* t
                        *extensional-preds* (delete r *extensional-preds* :key #'car))
                  (revise-callers r p))))))))
(defun new-fact()  
  (let ((p (progn (Format t "Type name of a new fact~%")
                  (type-in-predicate))))
    (when p 
      (if (get-pstruct p)
        (notify-error "There is already a predicate named ~a~%Delete it first, or use a different name.~%~%" p)
        (let ((a (get-arguments p)))
          (when a
            (let ((types (get-types p a)))
              (when types
                (let* ((r (cons p a))
                       (s (get-question r nil)))
                  (when s 
                    (let ((q (get-question r nil 1)))
                      (when q
                        (let((qs (when (cdr a)
                                   (mapcar #'(lambda(x)(cons x (get-variable-questions r x (remove x a))))
                                           a))))
                          (eval `(def-pred ,p :questions ((:fact ,s) (:question ,q) .,qs)
                                   :vars ,a :type ,types))
                          (setq *facts-changed* t)
                          (format t "~%~a defined~%" p))))))))))))))

(defun show-fact()
  (let ((f (select-rule "Select a fact name" nil *extensional-preds* ))
        (*max-l* nil))
    (when f
      (describe-fact f))))

(defun change-var-type()
  (let ((p (select-rule "Select a Fact to be Modified (Change Variable Type)" '(example)
                        *extensional-preds*)))
    (when p
      (let* ((s (get-pstruct p))
             (vars (pred-vars s))
             (types (pred-type s))
              n v new)
        (setq v (catch-cancel (car (select-item-from-list  vars 
                                                           :window-title "What variable do you want to change the type of"
                                                           ))))
        (when (and v (not (eq v :cancel)))
          (setq n (position v vars))
          (setq new (catch-cancel (get-type-from-user p v)))
          (when (and n (not (eq new :cancel)))
            (setf (elt types n) new)
            (setq *facts-changed* t)))))))
          
(defun describe-fact(F)
  (let ((s (get-pstruct f)))
    (format t "~%(~a~{ ~a~}) Variable Types: ~a" f (pred-vars s)(pred-type s))
    (when (pred-pos s)
      (format t "~%Positive Examples")
      (mapc  #'(lambda(x)(format t "~%   (~a~{ ~a~})" f x)) (trim-list (pred-pos s)))
      (when (and *max-l* (> (length (pred-pos s)) *max-l*))
        (format t "~%...")))
    (when (pred-neg s)
      (format t "~%~%Negative Examples")
      (mapc #'(lambda(x)(format t "~%   (~a~{ ~a~})" f x)) (trim-list(pred-neg s)))
      (when (and *max-l* (> (length (pred-neg s)) *max-l*))
        (format t "~%...")))
    (when (pred-questions s)
      (format t "~%English Translations")
      (mapc #'(lambda(x)
                (cond ((pcvar-p (car x))
                       (format t "~%   To ask about ~a~:{~%~6T if ~a is known, use ~a ~a~}"
                               (car x) (cdr x)))
                      (t (format t "~%  ~a ~a" (car x)(cadr x)))))
            (pred-questions s))
      (format t "~%~%"))))

(defun delete-entire-fact()
  (let ((f (select-rule "Select a fact to delete" (list *answer-pred* 'example) *extensional-preds* )))
    (when (and f (y-or-n-dialog (format nil "Do you really want to predicate ~a with the following positive examples: ~{ ~a~}" f
                                        (trim-list (pred-pos (get-pstruct f)) *max-l* '(|...|)))
                                      :yes-text "Yes, Delete"
                                      :no-text  "No"
                                      :cancel-text nil
                                      :size #@(480 200)))
      (setq *facts-changed* t)
      (setf (get f 'pred) nil
            *extensional-preds* (delete f *extensional-preds* :key #'car))
      (format t "~%~a deleted~%" f))))

(defun delete-fact()
  (let ((f (select-rule "Select a fact name (to delete an example)" nil *extensional-preds* )))
    (when f
      (let* ((s (get-pstruct f))
            (pos (pred-pos s))
            (v (catch-cancel (select-item-from-list  pos :window-title "Select Examples" 
                                                     :selection-type :disjoint))))
        (unless (or (null v)(eq v :cancel)
                    (not (y-or-n-dialog (format nil "Do you really want to delete~{ ~a~} from ~a" v f)
                                      :yes-text "Yes, Delete"
                                      :no-text  "No"
                                      :cancel-text nil
                                      :size #@(480 200))))
          (setq *facts-changed* t)
          (mapc #'(lambda(v)(destroy-fact f v)) v)
          (describe-fact f)
        )))))


(defun delete-negative-fact()
  (let ((f (select-rule "Select a fact name (to delete an example)" nil *extensional-preds* )))
    (when f
      (let* ((s (get-pstruct f))
            (pos (pred-neg s))
            (v (catch-cancel (select-item-from-list  pos :window-title "Select Examples" 
                                                     :selection-type :disjoint))))
        (unless (or (null v)(eq v :cancel)
                    (not (y-or-n-dialog (format nil "Do you really want to delete~{ ~a~} from negative examples of ~a" v f)
                                      :yes-text "Yes, Delete"
                                      :no-text  "No"
                                      :cancel-text nil
                                      :size #@(480 200))))
          (setq *facts-changed* t)
          (setf (pred-neg s) (set-difference  (pred-neg s) v))
          (describe-fact f)
        )))))      
      

(defun get-variable-questions(f var vars &optional (r nil))
  (unless (and  r
           (null (cdr vars)))
  (when (y-or-n-dialog (format nil "Are there any (more) ways to ask the user about ~a" var)
                         :yes-text "Yes"
                         :no-text  "No"
                         :cancel-text nil)
    (let ((q (get-var-q f var vars)))
      (when q
        (cons q (get-variable-questions f var vars t)))))))

(defun get-var-q(f v vars)
  (let ((kvars (catch-cancel (if (cdr vars)
                               (select-item-from-list  vars :window-title (format nil "Which variable(s) must be known to ask the value of ~a" v)
                                                    :selection-type :disjoint)
                               vars))))
    (unless (eq kvars :cancel)
      (let ((multi (y-or-n-dialog (format nil "For a given value of the variable(s) ~a can there be more than one value of ~a" kvars v)
                         :yes-text "Yes"
                         :no-text  "No"
                         :cancel-text nil))
            (q (get-question f nil 3 kvars v)))
        (when q (list kvars q (if multi :multi-valued :single-valued)))))))


(defun changes-vars-and-questions()
  (let ((p (select-rule "Select a Rule to be Modified" )))
    (when p
      (let* ((s (get-pstruct p))
             (a (get-arguments p nil (p-vars s))))
        (when a
          (let ((types (if (not (= (length (pred-vars s))
                                   (length a)))
                         (get-types p a)
                         (pred-type s))))
            (when types
              (let ((q (get-question (cons p a) (p-questions s))))
                (when q
                  (cond ((and (eq p *top-level-call*)
                              (not (= (length (pred-vars s))
                                      (length a))))
                         (unless(abort-operation (format nil
                                                         "You have changed the number of variables used by the top level predicate.  If you do not abort, the definitions
of example and ~a will be reset.  This means the ~a example(s) you have defined will be deleted
since they are not valid. " *answer-pred* (length (pred-pos (get-pstruct *answer-pred*)))))
                           (eval `(def-pred , *answer-pred*  :questions ((:fact ,q))
                                    :vars ,a :type ,types))
                           (eval `(def-pred example :vars ,a :type ,types))
                           (re-def-rule s :questions q :vars a :type types :mode nil)))
                        (t (re-def-rule s :questions q :vars a :type types :mode nil))))))))))))


(defun change-questions-fact()
  (let ((p (select-rule "Select a Fact to be Modified (Change English)" '(example) *extensional-preds*)))
    (when p
      (let* ((s (get-pstruct p))
             (q (get-vars-for-question (cons p (p-vars s))
                                       (mapcar #'car (p-questions s)) (p-questions s))))
        (when q 
          (setq *facts-changed* t)
          (describe-fact p))))))


(defun delete-questions-fact()
  (let ((p (select-rule "Select a Fact to be Modified (Delete English)" '(example) 
                        (remove-if-not #'(lambda(x)(cdr(p-vars(cdr x))))
                                *extensional-preds*))))
    (when p
      (let* ((s (get-pstruct p))
             (q (delete-vars-for-question  (delete-if-not #'pcvar-p (mapcar #'car (p-questions s)))
                                           (p-questions s)
                                           )))
        (when q 
          (setq *facts-changed* t)
          (describe-fact p))))))




(defun delete-vars-for-question(vars qs &aux x)
  (setq x (catch-cancel
            (let ((v (car (select-item-from-list  vars 
                                                  :window-title "What do you want to delete the English of?"
                                                  ))))
                (let* ((q (assoc v qs :test #'equalp))
                       (kvars (car(select-item-from-list (cdr q)
                                                         :window-title "Which Question do you want to delete?"))))
                         
                         (when kvars
                           (setf (cdr q)(delete kvars (cdr q)))
                           t)))))
  (unless (eq x :cancel) x))

(defun add-questions-fact()
  (let ((p (select-rule "Select a Fact to be Modified (Add English)" '(example) 
                        (remove-if-not #'(lambda(x)(cdr(p-vars(cdr x))))
                                *extensional-preds*))))
    (when p
      (let* ((s (get-pstruct p))
             (q (add-vars-for-question  p (p-vars s)
                                        (p-questions s)
                                           )))
        (when q 
          (setq *facts-changed* t)
          (describe-fact p))))))




(defun change-var-type-rule()
  (let ((p (select-rule "Select a Rule to be Modified (Change Variable Type)" nil
                        *intensional-preds*)))
    (when p
      (let* ((s (get-pstruct p))
             (vars (rule-vars s))
             (types (rule-type s))
              n v new)
        (setq v (catch-cancel (car (select-item-from-list  vars 
                                                           :window-title "What variable do you want to change the type of"
                                                           ))))
        (when (and v (not (eq v :cancel)))
          (setq n (position v vars))
          (setq new (catch-cancel (get-type-from-user p v)))
          (when (and n (not (eq new :cancel)))
            (setf (elt types n) new)
            (setq *rules-changed* t)))))))


(defun add-vars-for-question(p vars qs &aux x b)
  (setq x (catch-cancel
            (let* ((v (car (select-item-from-list  vars 
                                                  :window-title "What variable do you want to add a question for?"
                                                  )))
                  (q (get-var-q (cons p vars) v (remove v vars))))
              (when q
                (setq b (assoc v qs :test #'equalp))
                (cond ((null b)(nconc qs (list (cons v (list q)))))
                      (t (nconc b (list q))))))))
  (unless (eq x :cancel) x))


(defun get-vars-for-question(r vars qs &aux x)
  (setq x (catch-cancel
            (let ((v (car (select-item-from-list  vars 
                                                  :window-title "What variable do you want to change the English for?"
                                                  ))))
              (unless (eq v :cancel)
                (let* ((q (assoc v qs :test #'equalp))
                       (kvars nil))
                  (cond ((pcvar-p v)
                         (setq q 
                               (car(select-item-from-list (cdr q)
                                                    :window-title "Which Question do you want to Change")))
                         (setq kvars (remove-if-not #'pcvar-p (cadr q)))
                         (let ((multi 
                           (y-or-n-dialog (format nil "For a given value of the variable(s) ~a can there be more than one value of ~a" 
                                                  kvars v)
                                          :yes-text "Yes"
                                          :no-text  "No"
                                          :cancel-text nil))
                                 (new-q (get-question r (cadr q) 3 kvars v)))
                             (when new-q
                               (setf (cadr q) new-q)
                               (setf (caddr q) (if multi :multi-valued :single-valued)))))
                        
                        (t (let ((new-q (get-question r  (cadr q) (if (eq v :fact) 2 1))))
                             (when new-q
                               (setf (cadr q) new-q))))))))))
  (unless (eq x :cancel) x))


(defun get-question (r d &optional (question 2) (a (cdr r))(v nil) &aux
                       (string (case  question
                                 (1 (format nil 
                                            "How should a question about ~a be expressed in english?~
                                             ~%For example, (AGE ?PERSON ?N) is~
                                             ~%(is ?PERSON ?N years old)" r))
                                 (2 (format nil "How should ~a be expressed in english?~
                                                 ~%For example, (AGE ?PERSON ?N) is~
                                                 ~%(?PERSON is ?N years old)" r))
                                 (3 (format nil "How should a question for ~a of ~a be expressed using ~a in english~
                                                 ~%For example, To ask about ?N of (AGE ?PERSON ?N):~
                                                 ~%(How old is ?PERSON)" v r a)))))
  (let ((s (catch-cancel (get-text-from-user string
                                               :initial-string (if d (if (stringp d) d (format nil "~s" d)) "")
                                               ))))
    (unless (eq s :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string s))
        (cond  ((or error
                    (not (consp value)))
                (notify-error "~%~a ill-formed question" value)
                (get-question r s question a v))
               ((some #'(lambda(x)(and (pcvar-p x)
                                            (not (member x value :test #'equalp))))
                      a)
                (notify-error "~%~a: Every variable in ~a must be used.~%~%" value a)
                (get-question r s question a v))
               ((some #'(lambda(x)(and (pcvar-p x)
                                       (not (member x a :test #'equalp))))
                      value)
                (notify-error "~%An unknown variable cannot be used in ~a~
                               ~%Only variables in ~a can be used.~%~%" value a)
                (get-question r s question a v))
               (t value)))))
  )

(defun add-clause-to-rule()
  (let ((r (select-rule "Select A Rule To Be Modified" )))
    (when r 
      (let ((position (select-clause (cons '(head . |New clause should be first|)
                                           (get r 'clauses)) 
                                     "Select A Clause To Insert New Clause After")))
        (when position
          (let ((c (create-new-clause r)) new-clauses rest)
            (when c
              (cond ((eq (cdr position) '|New clause should be first|)
                     (setf new-clauses (cons c (get r 'clauses))))
                    (t (setf rest (cdr (member position (get r 'clauses))))
                       (setf new-clauses
                             (nconc (ldiff (get r 'clauses) rest)
                                    (list c)
                                    rest))))
              (re-def-rule (get r 'rule) 
                           :clauses new-clauses))))
        ))))
  
(defun add-literal-to-clause()
  (let ((r (select-rule "Select A Rule To Be Modified" )))
    (when r 
      (let ((c (select-clause (get r 'clauses) "Select A Clause To Be Modified")))
        (when c
          (let ((position (select-literal (cons '|New literal should be first literal|
                                                (cdr c))
                                   "Select A Literal To Insert New Literal After"
                                   )))
            (when position
              (let ((l (create-new-literal c)) rest new-clause)
                (when l
                  (cond ((eq position '|New literal should be first literal|)
                         (setf new-clause
                               (cons (car c)
                                     (cons l (cdr c)))))
                        (t (setf rest (cdr (member position c)))
                         (setf new-clause
                                 (nconc (ldiff c rest)
                                        (list l)
                                        rest))))
                  (re-def-rule (get r 'rule) 
                               :clauses (nsubstitute new-clause c (get r 'clauses))))))
            ))))))
                  
                                        
(defun create-new-literal(c  &optional (negation nil))
  (unless negation 
    (format t "Create a new literal for~%IF ~a~%THEN ~a~%~%"
            (cdr c)(car c)))
  (let ((l (catch-cancel
             (let ((p (prompt-for-predicate negation)))
               (when p
                 (let* ((all-vars (collect-vars c))
                        (s (get-pstruct p))
                        (p-vars (when s
                                  (p-vars s)))
                        (default-vars (if (and p-vars (subsetp p-vars all-vars))
                                        p-vars))
                        (a (get-arguments p negation default-vars c)))
                   (when a
                     (cons p a))))))))
    (if (eq l :cancel)
      nil
      l)))

(defun create-new-clause (r &optional (a nil))
  (let ((v (or a
               (p-vars (get r 'rule))
               (progn (format t "~%Type arguments for THEN part~%") 
                      (get-arguments r)))))
    (when v
      (let ((c (list (cons r v))))
        (cond (*use-menu* 
               (let ((l (create-new-literal c))
                     (next-l))
                 (when l
                   (setq c (nconc c (list l)))
                   (do ()
                       ((y-or-n-dialog (format nil "Is ~a the last literal?" l)
                                       :yes-text "Yes"
                                       :no-text  "No"
                                       :cancel-text nil)
                        c)
                     (setq next-l (create-new-literal c))
                     (when next-l 
                       (setq l next-l )
                       (setq c (nconc c (list l))))))))
              (t (get-new-clause c)))))))


(defun get-arguments(p &optional (negation nil) (d nil)(c nil))
  (cond ((eq p 'not) (let ((l (create-new-literal c t)))
                       (when l (list l))))
        ((eq p 'is)
         (get-args  "Type 2 argument(s) for is. For example (?x (+ ?y 7))"
                    2
                    d))
        (t (let ((s (get-pstruct p)))
            (get-args (format nil "Type ~a argument(s) for ~a~a~%For example ~a" 
                     (if (or (null s) d) "" (p-arity s))
                     (if negation "not " "")
                     p 
                     (if (or (null s)
                             (null (p-vars s)))
                       "(?x ...)" 
                       (p-vars s)))
                      (if s (p-arity s))
                      d)))))


(defun get-types(p a &aux c)
  (setq c (catch-cancel
  (mapcar #'(lambda(a)
              (get-type-from-user p a))
          a)))
  (unless (eq c :cancel) c))

;; CB (6/5/91)  get-type changed to get-type-from-user to avoid conflict with DT mutator.
(defun get-type-from-user (p a &aux x)
  (setq x (car (select-item-from-list (append '(|type in| :numeric :anything) *all-types*)
                                           :window-title (format nil "Select a type for ~a of ~a" a p))))
              (cond((null x) :anything)
                   ((eq x '|type in|)
                    (setq x (get-atom-from-user "Enter the name of a new type"))
                    (cond (x (pushnew x *all-types*)
                             (format t "~%~a: New Type Defined" x) x)
                          (t (cancel)))
                    )
                   (t x)))

(defun prompt-for-predicate(&optional (negation nil))
  (if negation
    (let((type (car (select-item-from-list '(rule fact primitive |type in|) 
                                         :window-title "Select a type of predicate (to be negated)"))))
    (when type
      (case type
        (rule (select-rule "Select a rule name (to be negated)" ) )
        (fact (select-rule "Select a fact name (to be negated)" (list *answer-pred* 'example) *extensional-preds* ))
        (primitive (select-rule "Select a primitive name (to be negated)" nil (cons *=* *builtin-preds*)))
        (|type in| (type-in-predicate t))
        )))
    
  (let((type (car (select-item-from-list '(rule fact primitive not is |type in|) 
                                         :window-title "Select a type of predicate"))))
    (when type
      (case type
        (rule (select-rule "Select a rule name" ) )
        (fact (select-rule "Select a fact name" (list *answer-pred* 'example) *extensional-preds* ))
        (primitive (select-rule "Select a primitive name" nil (cons *=* *builtin-preds*)))
        (|type in| (type-in-predicate))
        (not 'not)
        (is 'is))))))

(defun get-args(string arity &optional (d nil))
  (let ((v (catch-cancel (get-string-from-user string :initial-string (if d (format nil "~s" d) "")))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (consp value)))
                (notify-error "~%~a ill-formed arguments" value)
                nil)
               ((and arity (null d) (not(eq arity (length value))))
                (notify-error "You typed ~a.  ~a argument(s) are required~%" value arity)
                nil)
               (t value))))))

(defun type-in-predicate(&optional (negation nil))
  (let ((v (catch-cancel (get-string-from-user (if negation "Type a predicate name (to be negated)"
                                                   "Type a predicate name")))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (symbolp value)))
                (notify-error "~%~a ill-formed predicate" value)
                nil)
               (t value))))))


                
          
(defun delete-literal-from-clause()
  (let ((r (select-rule "Select A Rule To Be Modified" )))
    (when r 
      (let ((c (select-clause (get r 'clauses) "Select A Clause To Be Modified")))
        (when c
          (let ((l (select-literal (cdr c) "Select A Literal To Be Deleted")))
            (when (and l
                       (y-or-n-dialog (format nil "Do you really want to delete~% ~a from~%IF ~a~%THEN ~a"
                                              l (cdr c) (car c))
                                      :yes-text "Yes, Delete"
                                      :no-text  "No"
                                      :cancel-text nil
                                      :size #@(480 200)))
              
              (setq *rules-changed* t)
              (delete l c) ;note destructive delete always works because car is head of list
              (re-def-rule (get r 'rule) :clauses (get r 'clauses))
              )))))))

(defun delete-clause-from-rule()
  (let ((r (select-rule "Select A Rule To Be Modified" )))
    (when r 
      (let ((c (select-clause (get r 'clauses) "Select A Clause To Be Deleted")))
        (when (and c
                   (y-or-n-dialog (format nil "Do you really want to delete~%IF ~a~%THEN ~a"  (cdr c) (car c))
                                  :yes-text "Yes, Delete"
                                  :no-text  "No"
                                  :cancel-text nil
                                  :size #@(480 200)))
          
          (setq *rules-changed* t)
          (re-def-rule (get r 'rule) :clauses (delete c (get r 'clauses)))
          )))))


(defun text-edit-clause()
  (let ((r (select-rule "Select A Rule To Be Modified" )))
    (when r 
      (prog* ((c (select-clause (get r 'clauses) "Select A Clause To Be Edited"))
              (newc c))
        (unless c (return))
        retry (setq newc (get-new-clause newc))
        (if (null newc)(return))
        (unless (y-or-n-dialog (format nil "Do you really want to change~%IF ~a~%THEN ~a~%to~%IF ~a~%THEN ~a"  
                                     (cdr c) (car c)
                                     (cdr newc)(car newc))
                             :yes-text "Yes, Change"
                             :no-text  "No"
                             :cancel-text nil
                             :size #@(580 300))
          (go retry))
        (setq *rules-changed* t)
        (re-def-rule (get r 'rule) :clauses (substitute newc c (get r 'clauses)))
        ))))

(defun delete-the-rule()
  (let ((r (select-rule "Select A Rule To Be Deleted" (list *top-level-call*))))
    (when (and r
               (y-or-n-dialog (format nil "Do you really want to delete ~a" r)
                              :yes-text "Yes, Delete"
                              :no-text  "No"
                              :cancel-text nil))
      (setq *rules-changed* t)
      (setf (get r 'clauses) nil
            (get r 'brules) nil
            (get r 'rule) nil
            *intensional-preds* (delete r *intensional-preds* :key #'car))
      (Format t "~%~a deleted" r))))

(defun show-rule()
  (let ((r (select-rule "Select A Rule To Be Displayed")))
    (when r (show-rule-def r))))
            

  
(defparameter *new-rule-item* (make-instance 'menu-item
                                 :menu-item-title "New Rule"
                                 :menu-item-action #'(lambda()(eval-enqueue '(new-rule)))))

(defparameter *delete-clause-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Clause From Rule"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-clause-from-rule)))))

(defparameter *delete-literal-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Literal From Clause"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-literal-from-clause)))))

(defparameter *add-clause-item* (make-instance 'menu-item
                                 :menu-item-title "Add Clause To Rule"
                                 :menu-item-action #'(lambda()(eval-enqueue '(add-clause-to-rule)))))

(defparameter *add-literal-item* (make-instance 'menu-item
                                 :menu-item-title "Add Literal To Clause"
                                 :menu-item-action #'(lambda()(eval-enqueue '(add-literal-to-clause)))))

(defparameter *delete-rule-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Rule"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-the-rule)))))

(defparameter *show-rule-item* (make-instance 'menu-item
                                 :menu-item-title "Show Rule"
                                 :menu-item-action #'(lambda()(eval-enqueue '(show-rule)))))

(defparameter *show-rules-item* (make-instance 'menu-item
                                 :menu-item-title "Show All Rules"
                                 :menu-item-action #'(lambda()(eval-enqueue '(show-rules)))))

(defparameter *change-vq-item* (make-instance 'menu-item
                                 :menu-item-title "Change Vars/English"
                                 :menu-item-action #'(lambda()(eval-enqueue '(changes-vars-and-questions)))))

(defparameter *text-edit-clause-item* (make-instance 'menu-item
                                 :menu-item-title "Text Edit Clause"
                                 :menu-item-action #'(lambda()(eval-enqueue '(text-edit-clause)))))


(defparameter *change-v-item-rule* (make-instance 'menu-item
                                 :menu-item-title "Change Variable Type"
                                 :menu-item-action #'(lambda()(eval-enqueue '(change-var-type-rule)))))

(defparameter *Copy-rule*
  (make-instance 'menu-item
         :menu-item-title "Copy Rule"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(copy-a-rule)))))

(defparameter *Rename-rule*
  (make-instance 'menu-item
         :menu-item-title "Rename Rule"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(rename-a-rule)))))

(defparameter *Rename-fact*
  (make-instance 'menu-item
         :menu-item-title "Rename Fact"
         :menu-item-action 
         #'(lambda()(eval-enqueue '(rename-a-fact)))))

(defparameter *es-rules-menu*
 (make-instance 'menu 
               :Menu-title "Rules"
               :menu-items (list *show-rule-item* *show-rules-item* (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *new-rule-item*
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *delete-clause-item* *delete-literal-item*
                                 (make-instance 'menu-item
                                        :menu-item-title "-")
                                 *add-clause-item* *add-literal-item*
                                (make-instance 'menu-item
                                        :menu-item-title "-")
                                  *delete-rule-item*
                                  (make-instance 'menu-item
                                        :menu-item-title "-")
                                  *change-vq-item*
                                  *change-v-item-rule* 
                                  *text-edit-clause-item*
                                  *rename-rule*
                                  *copy-rule*
                                  (make-instance 'menu-item
                                        :menu-item-title "-")
                                  
                                 *initialize-item*)))

(defparameter *new-fact-item* (make-instance 'menu-item
                                 :menu-item-title "New Fact"
                                 :menu-item-action #'(lambda()(eval-enqueue '(new-fact)))))



(defparameter *delete-entire-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Fact Predicate"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-entire-fact)))))



(defparameter *delete-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Positive Fact"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-fact)))))

(defparameter *delete-neg-item* (make-instance 'menu-item
                                 :menu-item-title "Delete Negative Fact"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-negative-fact)))))



(defparameter *show-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Show Fact"
                                 :menu-item-action #'(lambda()(eval-enqueue '(show-fact)))))

(defparameter *show-facts-item* (make-instance 'menu-item
                                 :menu-item-title "Show All Facts"
                                 :menu-item-action #'(lambda()(eval-enqueue '(show-facts)))))

(defparameter *show-example-item* (make-instance 'menu-item
                                 :menu-item-title "Show Example"
                                 :menu-item-action #'(lambda()(eval-enqueue '(show-facts-for-example)))))



(defparameter *change-vq-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Change English"
                                 :menu-item-action #'(lambda()(eval-enqueue '(change-questions-fact)))))

(defparameter *change-v-item* (make-instance 'menu-item
                                 :menu-item-title "Change Variable Type"
                                 :menu-item-action #'(lambda()(eval-enqueue '(change-var-type)))))

(defparameter *add-vq-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Add English"
                                 :menu-item-action #'(lambda()(eval-enqueue '(add-questions-fact)))))
(defparameter *delete-vq-fact-item* (make-instance 'menu-item
                                 :menu-item-title "Delete English"
                                 :menu-item-action #'(lambda()(eval-enqueue '(delete-questions-fact)))))

(defparameter *es-facts-menu*
  (make-instance 'menu 
         :Menu-title "Facts"
         :menu-items (list *show-fact-item* *show-facts-item* *show-example-item* (make-instance 'menu-item
                                                                     :menu-item-title "-")
                           *new-fact-item*
                           (make-instance 'menu-item
                                  :menu-item-title "-")
                           *delete-entire-fact-item* 
                           *delete-fact-item*
                           *delete-neg-item* 
                           (make-instance 'menu-item
                                  :menu-item-title "-")
                           *add-vq-fact-item*
                           *delete-vq-fact-item*
                           *change-vq-fact-item*
                           *change-v-item*
                           *rename-fact*
                           (make-instance 'menu-item
                                  :menu-item-title "-")
                           *initialize-item*)))




(set-menu-item-update-function *es-file-menu* 
                               #'(lambda(x)(declare(ignore x))
                                  (cond (*intensional-preds*
                                         (menu-item-enable *save-rules-item*)
                                         (menu-item-enable *save-rules-as-item*))
                                        (t
                                         (menu-item-disable *save-rules-item*)
                                         (menu-item-disable *save-rules-as-item*)))
                                  (cond (*extensional-preds*
                                         (menu-item-enable *save-facts-item*)
                                         (menu-item-enable *save-facts-as-item*))
                                        (t
                                         (menu-item-disable *save-facts-item*)
                                         (menu-item-disable *save-facts-as-item*)))
                                  (unless *rules-changed* 
                                    (menu-item-disable *save-rules-item*))
                                  (unless *facts-changed* 
                                    (menu-item-disable *save-facts-item*))
                                  (unless *rule-file* 
                                    (menu-item-disable *save-rules-item*))
                                  (unless *fact-file* 
                                    (menu-item-disable *save-facts-item*))))


(set-menu-item-update-function *es-run-menu*
                           #'(lambda(x)(declare(ignore x))
                              (cond ((and *intensional-preds* *extensional-preds*)
                                     (menu-item-enable *judge-item*)
                                     (menu-item-enable *run-new-item*)
                                     (menu-item-enable *check-rules*)
                                     (menu-item-enable *who-calls*)
                                     (menu-item-enable *run-old-item*)
                                     (menu-item-enable *prove-goal*)
                                     (menu-item-enable *trace-all*)
                                     (menu-item-enable *untrace-all*)
                                     (menu-item-enable *add-trace*)
                                     (menu-item-enable *remove-trace*)
                                     (menu-item-enable *explain-item*)
                                     )
                                    
                                    (t   
                                     (menu-item-disable *prove-goal*)
                                     (menu-item-disable *run-new-item*)
                                     (menu-item-disable *run-old-item*)
                                     (menu-item-disable *check-rules*)
                                     (menu-item-disable *who-calls*)
                                     (menu-item-disable *judge-item*)
                                     (menu-item-disable *trace-all*)
                                     (menu-item-disable *untrace-all*)
                                     (menu-item-disable *add-trace*)
                                     (menu-item-disable *remove-trace*)
                                     (menu-item-disable *explain-item*)))
                              (unless *last-explanation*  (menu-item-disable *explain-item*))
                              (cond(*use-menu* (menu-item-disable *yes-menu-item*)
                                               (menu-item-enable *no-menu-item*))
                                   (t (menu-item-enable *yes-menu-item*)
                                      (menu-item-disable *no-menu-item*)))
                              (cond ((null *traced-predicates*)
                                     (menu-item-disable *untrace-all*)
                                     (menu-item-disable *remove-trace*))
                                    ((eq *traced-predicates* :all)
                                     (menu-item-disable *trace-all*)
                                     (menu-item-disable *remove-trace*)
                                     (menu-item-disable *add-trace*)))))


(set-menu-item-update-function  *es-type-menu*
                           #'(lambda(x)(declare(ignore x))
                              (cond ((and *intensional-preds* *extensional-preds*)
                                     (menu-item-enable *add-to-type-item*)
                                     (menu-item-enable *delete-from-type-item*)
                                     (menu-item-enable *delete-type-item*)
                                     (menu-item-enable *list-a-type-item*)
                                     (menu-item-enable *list-types-item*)
                                     (menu-item-enable *new-type-item*))
                                    (t    (menu-item-disable *add-to-type-item*)
                                          (menu-item-disable *delete-from-type-item*)
                                          (menu-item-disable *delete-type-item*)
                                          (menu-item-disable *list-a-type-item*)
                                          (menu-item-disable *list-types-item*)
                                          (menu-item-disable *new-type-item*)))))


(set-menu-item-update-function  *es-rules-menu*
                           #'(lambda(x)(declare(ignore x))
                              (cond (*top-level-call*  (menu-item-enable *new-rule-item*))
                                    (t  (menu-item-disable *new-rule-item*)))                          
                              (cond (*intensional-preds* 
                                     (menu-item-enable *delete-clause-item*) 
                                     (menu-item-enable *show-rule-item*)
                                     (menu-item-enable *show-rules-item*)
                                     (menu-item-enable *change-vq-item*) 
                                     (menu-item-enable *change-v-item-rule*) 
                                     (menu-item-enable *text-edit-clause-item*) 
                                     (menu-item-enable *delete-literal-item*) 
                                     (menu-item-enable *add-clause-item*) 
                                     (menu-item-enable *add-literal-item*) 
                                     (menu-item-enable *copy-rule*) 
                                     (menu-item-enable *rename-rule*) 
                                     (menu-item-enable *delete-rule-item*))
                                    (t 
                                     (menu-item-disable *delete-clause-item*) 
                                     (menu-item-disable *show-rule-item*) 
                                     (menu-item-disable *show-rules-item*)
                                     (menu-item-disable *change-vq-item*) 
                                     (menu-item-disable *text-edit-clause-item*) 
                                     (menu-item-disable *delete-literal-item*) 
                                     (menu-item-disable *change-v-item-rule*) 
                                     (menu-item-disable *add-clause-item*) 
                                     (menu-item-disable *copy-rule*) 
                                     (menu-item-disable *rename-rule*) 
                                     (menu-item-disable *add-literal-item*) 
                                     (menu-item-disable *delete-rule-item*)))))

(set-menu-item-update-function  *es-facts-menu*
                           #'(lambda(x)(declare(ignore x))
                              (cond (*top-level-call*  (menu-item-enable *new-fact-item*))
                                    (t  (menu-item-disable *new-fact-item*)))                          
                              (cond (*extensional-preds* 
                                     (menu-item-enable *delete-fact-item*)
                                     (menu-item-enable *delete-neg-item*)
                                     (menu-item-enable *show-fact-item*)
                                     (menu-item-enable *show-facts-item*)
                                     (menu-item-enable *show-example-item*)
                                     (menu-item-enable *change-vq-fact-item*)
                                     (menu-item-enable *rename-fact*) 
                                     (menu-item-enable *change-v-item*)
                                     (menu-item-enable *add-vq-fact-item*)
                                     (menu-item-enable *delete-vq-fact-item*)
                                     (menu-item-enable *delete-entire-fact-item*) 
                                     )
                                    (t 
                                     (menu-item-disable *delete-fact-item*) 
                                     (menu-item-disable *delete-neg-item*) 
                                     (menu-item-disable *show-fact-item*)
                                     (menu-item-disable *show-facts-item*)
                                     (menu-item-disable *show-example-item*)
                                     (menu-item-disable *change-vq-fact-item*) 
                                     (menu-item-disable *change-v-item*) 
                                     (menu-item-disable *add-vq-fact-item*) 
                                     (menu-item-disable *delete-vq-fact-item*) 
                                     (menu-item-disable *rename-fact*) 
                                     (menu-item-disable *delete-entire-fact-item*)
                                     ))))

(defun kr-es()
       (es)
       (menu-install *focl-windows-menu*)
       (menu-install *grapher-menu*)
       (menu-install *learn-menu*)
       nil)

(defun deinstall-named-menu(x)
       (let ((x (find-menu x)))
         (when x (menu-deinstall x))))

(defun deinstall-everything ()
  (deinstall-named-menu "File")
  (deinstall-named-menu "Files")
  (deinstall-named-menu "Edit")
  (deinstall-named-menu "Eval")
  (deinstall-named-menu "Tools")
  (deinstall-named-menu "Windows")
  (deinstall-named-menu "Rules")
  (deinstall-named-menu "Facts")
  (deinstall-named-menu "Types")
  (deinstall-named-menu "Run")
  (deinstall-named-menu "Grapher")
  (deinstall-named-menu "Learn"))
   
(defun es()
       (save-menus-to-globals)
       (deinstall-everything)
       (add-menu-items *es-file-menu* *original-file-menu* *original-eval-menu* *original-tools-menu*)
       (menu-install *es-file-menu*)    
       (menu-install *edit-menu*)
       (menu-install *es-rules-menu*)
       (menu-install *es-facts-menu*)
       (menu-install *es-type-menu*)
       (menu-install *es-run-menu*)
       nil)

(defun show-rule-def(r)
  (let ((c (get r 'clauses))(s (get-pstruct r)))
    (format t "~%Rule Definition: ")
    (format t "(~a~{ ~a~}) Variable Types: ~a~%" r (rule-vars s)(rule-type s))
    (mapc #'(lambda(x)
              (format t "IF ~a~%THEN ~a~%~%" (cdr x) (car x)))
          c)
    (when (pred-questions s)
      (format t "English Translation of conclusion: ~a~%~%" (pred-questions s)))))

(defun show-rules()
  (mapc #'(lambda(x)(show-rule-def (car x))) *intensional-preds*))

(defun show-facts()
  (let ((*max-l* nil))
  (mapc #'(lambda(x)(describe-fact (car x))) *extensional-preds*)))

(defun show-facts-for-example( &aux (ft t))
  (catch-cancel 
    (let ((examples (mapcar #'car(pred-pos (get-pstruct 'example))))
          e)
      (setq e (car (select-item-from-list (cons '|type in| examples)
                                          :window-title "Select an example")))
      (when (eq e '|type in|)
        (setq e (get-atom-from-user "Type a symbol")))
      (when e
        (mapc #'(lambda(s.a)
                  (mapc #'(lambda(p)
                            (when (member e p)
                              (when ft
                                (format T "~%~%Positive Facts for ~a" e)
                                (setq ft nil))
                              (format t "~%~a" (cons (car s.a) p))))
                        (pred-pos (cdr s.a))))
              *extensional-preds*)
        (setq ft t)
        (mapc #'(lambda(s.a)
                  (mapc #'(lambda(p)
                            (when (member e p)
                              (when ft
                                (format T "~%~%Negative Facts for ~a" e)
                                (setq ft nil))
                              (format t "~%~a" (cons (car s.a) p))))
                        (pred-neg (cdr s.a))))
              *extensional-preds*)
        (values))))) 


(defun judge-rules-on-examples()
  (let ((examples (pred-pos (get-pstruct 'example)))
        (answers  (pred-pos (get-pstruct *answer-pred*)))
        (*batch* nil)
        (*maintain-prolog-rule-trace* nil)
        (oerrors nil)
        (cerrors nil)
        (a nil))
    (setq *last-explanation* nil)
    (format T "~%Testing Examples")
    (dolist (e examples)
      
      (format t "~% ~a" e)
      (setq e (add-extra-variables (car e) *top-level-call*))
      (setq a (cdr(prove-goal `(,*top-level-call*  . ,e))))
      (format t " ~a" a)
      (cond ((null a)
             (when (setq a (member   e answers :test #'unify))
               (push (car a) oerrors)))
            (t (unless (member a answers :test #'equal)
                 (push  a cerrors)))))
    (when oerrors
      (format t "~%The following inferences(s) should be true, but are not supported by the rules:~%~
                 ~{~a~%~}~%~%" oerrors))
    (when cerrors
      (format t "~%The following inference(s) should not be true, but are supported by the rules:~%~
                 ~{~a~%~}~%~%" cerrors))))

(defun check-rules(&aux (errors 0) v s type)
  (mapc #'(lambda(n.r)
            (setq s (get-pstruct (car n.r)))
            (mapc #'(lambda(clause)
                      (setq v (mapcar #' cons (rule-vars s)(rule-type s)))
                      (mapc #'(lambda(literal)
                                (cond ((listp literal)
                                       (if (eq (car literal) 'not)
                                         (setq literal (cadr literal)))
                                       (cond ((eq (car literal) '=))
                                             ((eq (car literal) 'is)
                                              (cond ((pcvar-p (second literal))
                                                     (if (not (setq type (assoc (second literal) v :test #'equalp)))
                                                       (push (cons (second literal) :numeric) v)
                                                       (unless (member (cdr type)
                                                                       '(:numeric :anything))
                                                         (incf errors)
                                                         (format t "~%~%IS should use a numeric variable: ~a in ~a" literal (car n.r)))))
                                                    (t (incf errors)
                                                       (format t "~%~%IS should use variable for first argument: ~a in ~a" literal (car n.r))
                                                       )))
                                             ((not (get-pstruct (car literal)))
                                              (unless (equal literal '(fail))
                                                (incf errors)
                                                (format t "~%~%Undefined Term: ~a in ~a" literal (car n.r))))
                                             ((not (= (length (cdr literal))
                                                      (length (p-type (get-pstruct (car literal))))))
                                              (incf errors)
                                              (format t "~%~%Wrong number of arguments: ~a in ~a uses ~a" literal (car n.r)
                                                      (p-type (get-pstruct (car literal)))))                                                 
                                             (t (mapc #'(lambda(arg vtype)
                                                          (when (pcvar-p arg)
                                                            (cond ((not (setq type (cdr(assoc arg v :test #'equalp))))
                                                                   (push (cons arg type) v))
                                                                  ((eq type :anything))
                                                                  ((eq vtype :anything))
                                                                  ((eq type vtype))
                                                                  (t (incf errors)
                                                                     (format t "~%~%Variable type mismatch ~a should be a ~a: ~a in ~a" arg vtype literal (car n.r))))))
                                                      (cdr literal)
                                                      (p-type (get-pstruct (car literal)))))))
                                      ((not(eq literal '!))
                                       (format t "~%~%Undefined Literal: ~a in ~a" literal (car n.r))
                                       (incf errors))))
                            (cdr clause))
                      
                      (mapc #'(lambda(v)
                                (incf errors)
                                (format t "~%~%Singleton var ~a  in ~a" v clause))
                            (singleton-vars clause)))
                  (get (car n.r) 'clauses)))
        *intensional-preds*)
  (Format t "~%~a problems found" errors))


(defun who-calls(&optional (p nil) &aux c (found 0))
  (unless p (setq p (sort (nconc (mapcar #'car *builtin-preds*)
                                 (mapcar #'car *extensional-preds*)
                                 (mapcar #'car *intensional-preds*)
                                 (list 'not '= 'is))
                          #'(lambda(x y)(string< (symbol-name x) (symbol-name y)))))
          (cond  ((eq :cancel (setq c (catch-cancel 
                                        (select-item-from-list p :window-title "Select Predicates"))))
                  (setq p nil))
                 (t (setq p (car c)))))
  (when p
    (mapc #'(lambda(n.r)
              (mapc #'(lambda(clause)
                        (mapc #'(lambda(literal)
                                  (cond ((listp literal)
                                         (cond((eq (car literal) 'not)
                                               (when (eq p 'not)
                                                 (incf found)
                                                 (format t "~%Found ~a in ~a of ~a" p (cdr clause) (car n.r)))
                                               (setq literal (cadr literal))))
                                         (cond ((eq (car literal) p)
                                                (incf found)
                                                (format t "~%Found ~a in ~a of ~a" p (cdr clause) (car n.r))))
                                         )))
                              (cdr clause)))
                    (get (car n.r) 'clauses)))
          *intensional-preds*)
    (Format t "~%~a calls found" found)))



(defun revise-callers(p new-p)
  (mapc #'(lambda(n.r)
            (when (some #'(lambda(clause)
                            (some #'(lambda(literal)
                                      (cond ((listp literal)
                                             (cond((eq (car literal) 'not)
                                                   (setq literal (cadr literal))))
                                             (eq (car literal) p))))
                                  (cdr clause)))
                        (get (car n.r) 'clauses))
              (Format t "~%Changing ~a to use ~a instead of ~a ~%" (car n.r) new-p p)
              (re-def-rule (cdr n.r) :clauses (subst new-p p (get (car n.r) 'clauses)))))
        *intensional-preds*))

(defun singleton-vars(clause )
  (let* ((vars (collect-vars clause))
        (uvars (remove-duplicates vars)))
    (delete-if #'(lambda(v)(or (not (= (count v vars :test #'equalp) 1))
                               (eq #\_ (char (symbol-name (pcvar-id v)) 0))))
                   uvars)))

(defun collect-vars(l)
  (if (consp l)
    (nconc (collect-vars (car l))(collect-vars (cdr l)))
    (if (pcvar-p l) (list l))))
    
                           
(defun run-rules-on-examples(&optional (type-in))
  (catch-cancel 
    (let ((examples (pred-pos (get-pstruct 'example)))
          (answers  (pred-pos (get-pstruct *answer-pred*)))
          (*batch* nil)
          (*maintain-prolog-rule-trace* t)
          a
          e)
      (setq *new-facts* (setq *new-negs* nil))
      (setq e (if type-in 
                (get-atom-from-user "Type a name for the example")
                (car (first (select-item-from-list examples
                                                   :window-title "Select an old example")))))
      (when e
        
        (setq e (add-extra-variables e *top-level-call*))
        (cond ((and type-in (member e examples :test #'equal))
               (notify-error "There is already an example named ~a" e))
              (t
               (when e
                 (pushnew (car e) (get-type-instances (car (pred-type (get-pstruct *top-level-call*)))))
                 (multiple-value-setq  (a *last-explanation*)
                                       (prove-goal `(,*top-level-call*  . ,e)))
                 (if type-in
                   (cond ((null a)
                          (insert-pos (list 'example (car e)))
                          (supposed-to-fail? e))
                         (t (insert-pos (list 'example (car e)))
                            (supposed-to-succeed? e a)))
                   (cond ((null a)
                          (format t "~%~a was not proved to be true~%" (cons *top-level-call*  e))
                          (when (setq a (car (member e answers :test #'equal)))
                            (Format t "Previously, it was indicated that ~a should be inferred" a)))
                         (t (format t "~%~a was infered~%" a)
                            (unless (car (member (cdr a) answers :test #'equal))
                              (Format t "Previously, it was not indicated that ~a should be inferred" a)))))
                 (record-changes-to-facts?))))))))


(defun prove-goal-typed-in (&aux (g(get-goal)) a)
  (let ((*batch* nil)
        (*maintain-prolog-rule-trace* t))
    (when g
      (multiple-value-setq  (a *last-explanation*) (prove g))
      (if a (format t "~%~a was proved: ~a" g a)
          (format t "~%~a was not proved" g))
      (record-changes-to-facts?))))

(defun add-extra-variables (e p &aux (s (get-pstruct p)))
  (cons e (cdr (pred-vars s))))

(defun supposed-to-fail? (e)
  (format t "~%~a was not proved to be true~%" (cons *top-level-call*  e))
  (cond ((y-or-n-dialog (format nil  "~a failed.~% Was this intended?"  (cons *top-level-call*  e))
                     :cancel-text nil
                     :size #@(480 200))
         (insert-neg (get-pstruct *answer-pred*)
                     (cons *answer-pred* e)))
        (t (destroy-fact 'example (list(car e)))
           
           (setq *new-facts* (delete 'example *new-facts* :key #'car)))))

(defun supposed-to-succeed? (e a)
  (format t  "~%~a was infered for ~a.~%" a e)
  (cond ((y-or-n-dialog (format nil  "~a was infered for ~a.~% Is this the intended answer?" a e)
                     :cancel-text nil
                     :size #@(480 200))
         (insert-pos (cons *answer-pred* (cdr a))))
        (t (destroy-fact 'example (list(car e)))
           (setq *new-facts* (delete 'example *new-facts* :key #'car)))))

(defun record-changes-to-facts?(&aux s)
  (when (or *new-facts* *new-negs*)
    (when *new-facts*
      (Format t "~%New Facts:~%")
      (mapc #'print *new-facts*))
    (when *new-Negs*
      (Format t "~%New negative examples:~%")
      (mapc #'print *new-negs*))
    (unless (y-or-n-dialog "Do you want the new facts to be permanently recorded?"
                     :cancel-text nil)
      (mapc #'(lambda(x)(destroy-fact (car x)(cdr x))) *new-facts*)
      (mapc #'(lambda(F)
                (setq s (get-pstruct (car f)))
                (setf (pred-neg s) (delete (cdr f) (pred-neg s) :test #'equalp)))
            *new-negs* ))
      (setq *new-facts* (setq *new-negs* nil))))
      
    



(defmacro def-es-focl-problem(x y)
  `(progn (format t "\"The correct classification is stored as a fact by ~a\"" (setq *answer-pred* ',x))
          (format nil "The classification of examples is inferred by ~a" (setq *top-level-call* ',y))
           ))

(defun get-atom-from-user(string &optional cancel)
  (let ((v (catch-cancel (if cancel (get-string-from-user string :cancel-text cancel)
                         (get-string-from-user string)))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (atom value)))
                (notify-error "~%~a ill-formed symbol" value)
                nil)
               (t value))))))


(defun get-new-clause(c)
  (let ((v (catch-cancel (if (cdr c)
                           (get-text-from-user  (format nil "Type a new body for ~a" (car c))
                                                 :initial-string (format nil "~a" (cdr c))
                                                 )
                           (get-text-from-user  (format nil "Type a new clause for ~a" (car c))
                                                 )))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (consp value)))
                (notify-error "~%~a ill-formed clause" value)
                nil)
               (t (cons (car c) value)))))))

(defvar *last-goal-typed-in* "")
(defun get-goal()
  (let ((v (catch-cancel (get-text-from-user  "Type a goal to prove. For example ((= ?x 1)(is ?y (+ 3 ?x)))"
                                              :initial-string *last-goal-typed-in*))))
    
    (unless (eq v :cancel)
      (setq *last-goal-typed-in* v)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (consp value))
                    (not (consp (car value))))
                (notify-error "~%~a ill-formed goal" value)
                nil)
               (t value))))))


(provide :es)

