
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff  3/13/91   implement to table look up and automatic table extension
;;;  01  cliff  3/23/91   modify computations to work with builtins
;;;                       comments modified to include input and return values
;;;  02  cliff  4/29/91   utilize commutative constraint when counting variablizations
;;;
;;;  Note:  The number of variablization of a predicate which is computed here
;;;         is independent of the setting of max-new-variables.  Thus, the 
;;;         computation here tell the number of bits to encode a literal of a
;;;         predicate when considering the complete space of variablizations,
;;;         and does not reflect the fact that we might only be searching a 
;;;         portion of that space.
;;;
;;; [XXXX How do "functions" effect the number of predicates? ]
;;;       This code doesn't handle function or constants yet!


;;;________________________________________________________________________________
;;; init-stopping-tables (goal-concept pos-tuples neg-tuples)
;;;
;;;  Sets number-of-predicates which is used by bits-to-encode-literal-using-pred, and
;;;  grows variablizations-of-new-slots-table to (- max-arity 1)

(defun init-stopping-tables (goal-concept pos-tuples neg-tuples)

  (set-initial-training-set-values-stopping-criteria pos-tuples neg-tuples)
  (set-number-of-predicates goal-concept)
   
  (let ((max-arity 0))
    (dolist (pred *extensional-preds*)
      (setq max-arity (max max-arity (p-arity (cdr pred)))))
    
    (dolist (pred *intensional-preds*)
      (setq max-arity (max max-arity (p-arity (cdr pred)))))
    (grow-variablizations-of-new-slots-table (- max-arity 1))))


;;;________________________________________________________________________________
;;; factorial (input) 
;;;
;;;  Given:    An integer, input.
;;;  Returns:  An integer, the factorial of input.
;;;
;;;  Looks up factorial of input in factorial-table.  If the input is greater than
;;;  the highest stored value factorial-table grow-factorial-table is called.
;;;  grow-fatorial-table extended factorial table to include all the factorials
;;;  upto the current input.
;;;
;;;   input        0   1   2   3   4    5 ...
;;;   factorial    1   1   2   6  24  120 ...

(let ((factorial-table (make-array  1 :adjustable t :initial-element 1))
      (max-factorial-table-index 0))

  (defun grow-factorial-table (new-max-index)
    (adjust-array factorial-table (+ 1 new-max-index))
    (do* ((i max-factorial-table-index (incf i))
          (fact (aref factorial-table i) (* fact i)))
         ((> i new-max-index))
      (setf (aref factorial-table i) fact))
    (setf max-factorial-table-index new-max-index)
    (aref factorial-table new-max-index))
  
  (defun factorial (input)
    (if (> input max-factorial-table-index)
      (grow-factorial-table input)
      (aref factorial-table input)))
)

;;;________________________________________________________________________________
;;; binomial (n k)
;;;
;;;  Given:    Two integers, n and k.
;;;  Returns:  An integer, the n choose k.
;;;
;;;   n          0   5   5   5   5   5   5 
;;;   k          X   0   1   2   3   4   6  
;;;   binomial   1   1   5  10  10   5   0

(defun binomial (n k)
   (if (= n 0)
     1
     (/ (factorial n) (* (factorial (- n k)) (factorial k)))))

;;;________________________________________________________________________________
;;; lg-factorial (input) 
;;;
;;;  Given:    An integers, input.
;;;  Returns:  A floating point number, the log base 2 the factorial of input.
;;;
;;;  Looks up the closest lg-factorial of input in lg-factorial-table.  
;;;  lg-factorial-table stores every thenth lg-factorial. If input is more than
;;;  ten larget than the the highest stored value lg-factorial-table 
;;;  grow-lg-factorial-table is called.  grow-lg-factorial-table extended 
;;;  lg-factorial table to include every tenth lg-factorial upto the current input.
;;;                              
;;;   input         0     1     2      3      4       5 ...
;;;   factorial     1     1     2      6     24     120 ...
;;;   lg-factorial  0.0   0.0   1.0    2.58.. 4.58..  6.90..

(let ((lg-factorial-table (make-array  1 :adjustable t :initial-element 0))
      (max-lg-factorial-table-index 0))

  (defun grow-lg-factorial-table (new-max-index)
    (setf lg-factorial-table
          (adjust-array lg-factorial-table (+ 1 new-max-index)))
    (do* ((i max-lg-factorial-table-index (incf i))
          (start (+ (* i 10) 1) (+ (* i 10) 1))
          (end (+ start 9) (+ start 9)))
         ((= i new-max-index))
      (setf (aref lg-factorial-table (+ i 1))
            (+ (aref lg-factorial-table i)
               (do* ((j start (incf j))
                     (sum (log j 2) (+ sum (log j 2))))
                    ((= j end) sum)))))
    (setq max-lg-factorial-table-index new-max-index)
    (aref lg-factorial-table new-max-index))
  
  (defun lg-factorial (input)
    (let* ((index (truncate input 10))
           (start (+ (* index 10) 1))
           (base-lg-fact (if (> index max-lg-factorial-table-index)
                           (grow-lg-factorial-table index)
                           (aref lg-factorial-table index))))
      (do* ((i start (incf i))
            (sum 0))
           ((> i input) (+ base-lg-fact sum))
        (incf sum (log i 2)))))
)

;;;________________________________________________________________________________
;;; lg-binomial (n k)      
;;;
;;;  Given:    Two integers, n and k.
;;;  Returns:  A floating point number, log base 2 n choose k.

(defun lg-binomial (n k)
  (if (= n 0)
    0
    (- (lg-factorial n) (+ (lg-factorial (- n k)) (lg-factorial k)))))


;;;________________________________________________________________________________
;;; number-of-variablizations-of-new-slots (n constraint)
;;;
;;; Given:   The number of slots to contain new vaiables, n, and the constraints
;;;          on those slots, constraint.
;;; Returns: The number of distinct variablizations of n slots using new variables.
;;;          If constraint = :unique-vars there is only 1 such variablization.
;;;
;;; Assumptions:
;;;    1. There is always 1 and only 1 way of variablizing 0 slots.
;;;    2. There is an unlimited supply of new variables
;;;
;;; number-of-variablizations-of-new-slots looks up the number of distinct 
;;; variablizations of n new slots in the variablizations-of-new-slots-table.
;;; If the n is greater than the highest stored value then
;;; grow-variablizations-of-new-slots-table is called to extend the table to
;;; include the number of variablization for any number of new slots up to n.
;;; grow-variablizations-of-new-slots-table uses frequency-list and 
;;; next-iteration to extend the table.
;;;
;;;________________________________________________________________________________
;;; next-iteration (current-state)
;;;
;;; Given:   A frequency list [description below], current-state.
;;; Returns: A frequency list which reprents the next state of current-state
;;;
;;;  New (unbound) variables behave differently than old (bound) variables in terms
;;;  the number of unique variablalizations of slots they can produce.
;;;
;;;     Let N1, N2, and N3 be new variables.
;;;     Let O1, O2, and O3 be old variables.
;;;
;;;     How many unique ways are there of variablizing 2 slots with:
;;;
;;;          New Variables                   Old Variables
;;;
;;;     1.  (N1 N1) = (N2 N2) = (N3 N3)        1 (O1 O1)
;;;                                            2 (O2 O2)
;;;                                            3 (O3 O3)
;;;     2.  (N1 N2) = (N1 N3) =                4 (O1 O2) 
;;;         (N2 N1) = (N2 N3) =                5 (O1 O3)
;;;         (N3 N1) = (N3 N2)                  6 (O2 O1)
;;;                                            7 (O2 O3)
;;;                                            8 (O3 O1)
;;;                                            9 (O3 O2)
;;;
;;;  next-iteration implements a recurance which allows the number of
;;;  variablization of new slots to be determined.
;;;
;;;   S    Explicit List                     Frequency List    Variablizations
;;;   1    (1)                               (1)                     1
;;;   2    (2)                               (0 1)                   2
;;;   3    (2 3)                             (0 1 1)                 5
;;;   4    (2 3 3 3 4)                       (0 1 3 1)              15
;;;   5    (2 3 3 3 4 3 3 4 3 3 4 4 4 4 5)   (0 1 7 6 1)            52
;;;
;;;  Summing the element in the explicit list yields the number of ways
;;;  variablizing S slots with new variables.  This is equivalent
;;;  to the sum of the elements in frequency list each multiplied by
;;;  its position in the list.


(let ((variablizations-of-new-slots-table (make-array 2 :adjustable t :initial-contents '(1 1)))
      (max-variablizations-of-new-slots-table-index 1)
      (frequency-list '(1)))

  (defun grow-variablizations-of-new-slots-table (new-slots)
    (labels 
      ((next-iteration (current-state) ; current frequency-list
                       (let ((position 0)        ; the current position in the frequency-list
                             (previous 0)        ; value stored in the previous position
                             (next-state nil))   ; next state of the frequency-list (return value)
                         (dolist (element current-state (nreverse (push previous next-state)))
                           (let* 
                             ((current (+ (* element position) previous)))
                             (setq previous element)
                             (incf position)
                             (push current next-state))))))
      
      (adjust-array variablizations-of-new-slots-table (+ new-slots 1))
      (do ((i (+ max-variablizations-of-new-slots-table-index 1) (incf i)))
          ((> i new-slots))
        (setq frequency-list (next-iteration frequency-list))
        (let ((position 0)
              (variablizations 0))
          (setf (aref variablizations-of-new-slots-table i) 
                (dolist (element frequency-list variablizations)
                  (incf variablizations (* element (incf position)))))))
      (setf max-variablizations-of-new-slots-table-index new-slots)
      (aref variablizations-of-new-slots-table new-slots)))
  
  (defun number-of-variablizations-of-new-slots 
         (n                            ; number of new slots
          constraint)                  ; either :unique-vars or nil
    
    (if (equal constraint :unique-vars)
      1
      (if (> n max-variablizations-of-new-slots-table-index)
        (grow-variablizations-of-new-slots-table n)
        (aref variablizations-of-new-slots-table n))))
  )

;;;________________________________________________________________________________
;;; number-of-variablizations-of-old-slots (s b constraint)
;;;
;;; Given:   The number of slots to contain old variables, s; the number of old
;;;          variables, b; and constaints on the slots, constraint.
;;; Returns: The number of distinct variablizations of s slots using b old 
;;;          variables. Handles constraint = :unique-vars
;;;
;;; Assumptions:
;;;    1. There is always 1 and only 1 way of variablizing 0 slots.
;;;    2. There is no way of variablizing 1 or more slots with no old variables.

(defun number-of-variablizations-of-old-slots 
       (s                           ; number of old slots
        b                           ; number of bound variables
        constraint)                 ; either :unique-vars or nil
  (cond
   ((= s 0) 1)
   ((= b 0) 0) 
   ((equal constraint :unique-vars) (if (> s b)
                                      0 
                                      (/ (factorial b) (factorial (- b s)))))
   (t (expt b s))))


;;;________________________________________________________________________________
;;; number-of-variablizations-with-n-new-slots (n slots p-freq old-freq 
;;;                                             mode-new mode-old constraint)
;;;
;;; Given:   The number of new slots, n;
;;;          The total number of slots, slots;
;;;          A list of the number of slots of each type, p-freq;
;;;          A list of the number of old variales of each type, old-freq;
;;;          A list of the number of slots of each type which must contain new variables, mode-new;
;;;          A list of the number of slots of each type which must contain old variables, mode-old;
;;;          and the constraints on the slots, constraint.
;;; Returns: The number of distinct variablizations of s slots of which exactly n  
;;;          are distiguished as new (contain new variables).
;;;
;;; [XXXX] This function could be sped up using dynamic programming.

(defun number-of-variablizations-with-n-new-slots 
       (n                 ; number of new slots
        slots             ; total number of slots
        p-freq            ; list of the number of slots of each type
        old-freq          ; list of the number of new variables of each type
        mode-new          ; list of the number of slots of each type which must be new
        mode-old          ; list of the number of slots of each type which must be old
        constraint        ; either :all-unique or nil

        &aux (result 0)                   ; number of variablizations (return value)
             (type-slots (car p-freq))    ; number of slots of current type
             (type-vars (car old-freq))   ; number of bound variables of current type
             (new-slots (car mode-new))   ; number of slots of current type which must be new
             (old-slots (car mode-old))   ; number of slots of current type which must be old
             (either-slots (- type-slots  ; number of slots of current type which could be either
                              new-slots 
                              old-slots))
                                                 ; remainders of the list originally passed in
             (next-slots (- slots type-slots))   ; set up for recursing down the lists
             (next-p-freq (cdr p-freq))
             (next-old-freq (cdr old-freq))
             (next-mode-new (cdr mode-new))
             (next-mode-old (cdr mode-old)))
 (cond
   ((> n slots) 0)                ;  more new-slots than total slots  => 0 variablizations

   ((< n new-slots) 0)            ;  more new-slots of this type than total new-slots => 0

   ((= type-slots slots)          ;  if this is the last set of typed-slots don't recurse
    (*(binomial either-slots  
                (- n new-slots))
      (number-of-variablizations-of-new-slots n 
                                              constraint)
      (number-of-variablizations-of-old-slots (- type-slots n)
                                              type-vars
                                              constraint)))

   ((= either-slots 0)            ; if there are no either-slots don't try all partitions
    (* (number-of-variablizations-of-new-slots new-slots  
                                               constraint)
       (number-of-variablizations-of-old-slots old-slots
                                               type-vars
                                               constraint)
       (number-of-variablizations-with-n-new-slots (- n new-slots) 
                                                   next-slots
                                                   next-p-freq
                                                   next-old-freq
                                                   next-mode-new
                                                   next-mode-old
                                                   constraint)))

   (t                             ; if there's no short cut, do it all
    (dotimes (i either-slots result)
      (let*
        ((new-slots-used (+ new-slots i)))
        (setq result
              (+ result
                 (* (binomial either-slots i) 
                    (number-of-variablizations-of-new-slots new-slots-used
                                                            constraint)
                    (number-of-variablizations-of-old-slots (- type-slots new-slots-used)
                                                            type-vars
                                                            constraint)
                    (number-of-variablizations-with-n-new-slots (- n new-slots-used) 
                                                                next-slots
                                                                next-p-freq
                                                                next-old-freq
                                                                next-mode-new
                                                                next-mode-old
                                                                constraint)))))))))




(let ((number-of-predicates 0))

;;;________________________________________________________________________________
;;; set-number-of-predicates (goal-concept)
;;;
;;; Given:   Goal concept, and the ability to access the global lists of predicates.
;;; Returns: The function determines the number of predicate in the representation
;;;          language used to express the learned concept description.
;;;          
;;; Note:  This function has the side effect of setting  the non-local variable
;;;        number-of-predicates, and should be called to set once before 
;;;        attempting to use any of the bits-to-enocode... functions.
;;;
;;; Assumptions:
;;;  1. The number of predicates will not change as FOCL runs. 
;;;
;;; [XXXX] Any introduction of new predicates or functions will violate this
;;;        assumption.  When a new predicate is introduced does it mean than
;;;        the clauses learned earlier really required more bits to encode?
;;;
;;; [XXXX] This function does not currently count builtin functions.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff  3/23/91   extended to include built-in predicates 

  (defun set-number-of-predicates (goal-concept)
    (let ((i-induction 0)
          (i-not 0)
          (e-induction 0)
          (e-not 0)
          (b-induction 0)
          (b-not 0))
      
      (dolist (pair *intensional-preds*)
        (if (p-induction (cdr pair))
          (incf i-induction)
          (incf i-not)))
      
      (dolist (pair *extensional-preds*)
        (if (p-induction (cdr pair))
          (incf e-induction)
          (incf e-not)))

      (dolist (pair *builtin-preds*)
        (if (p-induction (cdr pair))
          (incf b-induction)
          (incf b-not)))
      
      (setf number-of-predicates
            (cond                                               ;; EXTENSIONAL INDUCTION ONLY
             ((or (null *intensional-preds*)                    ;; Only extensional and built-in
                  (null *constructive-induction*)               ;; preds available for induction can 
                  (and (eq i-induction 0) (null goal-concept))) ;; appear in learned definition.
              (+ e-induction b-induction))
                                                                ;; INTESIONAL INDUCTION
             ((and *constructive-induction*                     ;; Extensional, built-in, and 
                   (null *operationalize-constructive*)         ;; intensional preds avaiable for
                   (null goal-concept))                         ;; induction can appear.
              (+ e-induction b-induction i-induction))      
                                                                ;; ANY METHOD (BUT OPERATIONALIZE)
             (*operationalize-constructive*                     ;; Extensional and built-in preds...
              (+ e-induction e-not b-induction b-not))

             (t                                                 ;; ANY METHOD 
              (+ i-induction i-not                              ;; Extensional, built-in, and
                 e-induction e-not                              ;; intensional preds can appear
                 b-induction b-not))))))                        ;; can appear in learned definition.


  ;;;________________________________________________________________________________
  ;;; bits-to-encode-literal-using-pred (p old-types) 
  ;;;
  ;;; Given:   An extended p-structure, p, (a predicate, a rule or a builtin), and
  ;;;          A list of the types of the old variables, old-types.
  ;;; Returns: The number of bits required to encode a literal using p.
  ;;;          Accounts for type, mode, and constraint.
  ;;;
  ;;; NOTE: If there are no valid variablizations, bits-to-encode-literal-using-pred is
  ;;;       meaningless and the function returns 0 [XXXX perhaps this should be nil].
  ;;;
  ;;; NOTE: bits-to-encode-literal-using-pred Is uneffected by max-new-variables.  First, it is
  ;;;       difficult to incorporate max-new-variables into the way I have set up 
  ;;;       the counting.  Second, it doesn't make sense to say that the bits to
  ;;;       needed encode a literal are depended on the (the degree to/rate at) which
  ;;;       one is willing to allow the search space to expand.
  ;;;
  ;;; [XXXX] How many variablizations are there of a function?
  ;;;
  ;;; [XXXX] Perhaps old-freq, mode-new and mode-old should be computed once and stored
  ;;;        as part of the pred structure.  Additionally, bits-to-encode-literal-using-pred can
  ;;;        be computed and stored once when old = the set of variable in the head of
  ;;;        the clause.  This will speed things along when max-new-variables is 0 and
  ;;;        at the beginning of each clause for as long as no "new" variables are 
  ;;;        introduced.

  (defun bits-to-encode-literal-using-pred (p old-types)
    (if (eql (p-name p) 'is)
      (progn
        (format t "~%Warning: bits-to-encode-literal-using-pred a builtin-fn is not defined correctly!")
        (format t "~%         Results obtained using stopping criteria on builtin-fns may")
        (format t "~%         be in error.")
        (+ 1                                        ;; [XXXX] Can functions be negated?
           (log number-of-predicates 2)           ;; [XXXX] Can functions be defined dynamically?
        ));;  (number-of-variablizations)           ;; [XXXX] How many variablizations are there?
                                                    ;; [XXXX] Obviously this is incomplete!!!

      (let*                                           ;; An Example:  old-types (:c :a :a :a :a :b :c)
        ((type (p-type p))                            ;;  (:a :a :a :b :c :b)
         (arity (p-arity p))                          ;;  6
         (mode (p-mode p))                            ;;  (:+ :+ :? :- :+ :+)
         (constraint (p-constraint p))                ;;  :all-unique, or nil
         (type-mode-list (mapcar #'list type mode))   ;;  ((:a :+)(:a :+)(:a :?)(:b :-)(:c :+)(:b :+))              
         (type-list (remove-duplicates type))         ;;  (:a :b :c)
         (p-freq nil)                                 ;;  (3 2 1)
         (old-freq nil)                               ;;  (4 1 2)
         (mode-new nil)                               ;;  (0 1 0)
         (mode-old nil)                               ;;  (2 1 1)
         (number-of-variablizations 0))               ;;
        
        (dolist (e type-list)
          (push (count e type) p-freq)                                      ;; These are being
          (push (count e old-types) old-freq)                               ;; built in reverse.
          (push (count (list e :-) type-mode-list :test #'equal) mode-new)  ;; They are reversed
          (push (count (list e :+) type-mode-list :test #'equal) mode-old)) ;; at the call.
        
        (dotimes (n arity)
          (incf number-of-variablizations
                (number-of-variablizations-with-n-new-slots n
                                                            arity
                                                            (nreverse p-freq)
                                                            (nreverse old-freq)
                                                            (nreverse mode-new)
                                                            (nreverse mode-old)
                                                            constraint)))
        (if (p-commutative p)                         ;; commutative is defined 
          (setf number-of-variablizations             ;; for binary relations only.
                (/ number-of-variablizations 2)))
        
        (if (= number-of-variablizations 0)   ; No literals of using this predicate are valid !!!
          0
          (+ 1                                       ; to indicate whether negated
             (log number-of-predicates 2)          ; to specify which predicate
             (log number-of-variablizations 2))))))  ; to specify which variablization
)

;;;________________________________________________________________________________
;;; GET-LITERAL-P 
;;;
;;; returns the p-structure accociated with the a literal structures predicate name.
;;;
;;;  rv  who    date      reason
;;;  00  cliff  05/20/91  moved from stopping.lisp

(defun get-literal-p (L)
  (let ((L-name (literal-predicate-name L)))
    (get-pstruct L-name)))

;;;________________________________________________________________________________
;;; bits-to-encode-literal (L old-types) 

(defun bits-to-encode-literal (L old-types)
  (cond ((literal-deleted? L) 0)
        ((literal-negated? L)
         (bits-to-encode-literal-using-pred (get-literal-p (literal-negated-literals L)) old-types))
        (t (bits-to-encode-literal-using-pred (get-literal-p L) old-types))))


;;;________________________________________________________________________________
;;; bits-to-encode-ordered-literals (L old-types) 
;;;
;;; Given:   A linked list of literal, L, and the set of old-types
;;; Returns: The number of bits required to encode an ordered set of literals.
;;;          (i.e., the sum of the number of bits to encode each literal.)
;;;
;;; Note:  For the most part order is not important, thus this function over
;;;        estimates the number of bits required to encode the set of literals. 
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff  3/23/91   extended to work correctly on built-in predicates 

(defun bits-to-encode-ordered-literals (L old-types)
  (cond 
   ((null L) 0)

   ((literal-deleted? L) 
    (bits-to-encode-ordered-literals (literal-next L) old-types))

   ((literal-negated? L) 
    (+ (bits-to-encode-ordered-literals (literal-negated-literals L) old-types)
       (bits-to-encode-ordered-literals (literal-next L) old-types)))
   
   (t 
    (+ (bits-to-encode-literal L old-types)
       (bits-to-encode-ordered-literals (literal-next L) old-types)))))


;;;________________________________________________________________________________
;;; bits-to-encode-clause (literals old-types) 

(defun bits-to-encode-clause (literals old-types &optional (literals-in-clause 1))
  (cond 
   ((null literals) 0.0)

   ((literal-deleted? literals) 
    (bits-to-encode-clause (literal-next literals) old-types literals-in-clause))

   ((literal-negated? literals) 
    (+ (bits-to-encode-clause (literal-negated-literals literals) old-types)
       (bits-to-encode-clause (literal-next literals) old-types (+ literals-in-clause 1))))
   
   (t 
    (+ (- (bits-to-encode-literal literals old-types) (log literals-in-clause 2))
       (bits-to-encode-clause (literal-next literals) old-types (+ literals-in-clause 1))))))


;;;________________________________________________________________________________
;;; bits-to-encode-conjunction-of-literals (L old-types) 
;;;
;;; Given:   A linked-list of literal structures L (i.e.,a clause), and
;;;          A list of the types of the old variables, old-types.
;;; Returns: The number of bits required to encode clause L.

(defun bits-to-encode-conjunction-of-literals (L old-types)
  (- (bits-to-encode-ordered-literals L old-types)
     (lg-factorial (count-literals L))))


;;;________________________________________________________________________________
;;; COUNT-LITERALS
;;;
;;;  returns the number of non-deleted literals in a conjunction of literal stuctures.
;;;
;;;  rv  who    date      reason

(defun count-literals (L)
  (cond ((null L) 0)
        ((literal-deleted? L) (count-literals (literal-next L)))
        ((literal-negated? L) (+ (count-literals (literal-negated-literals L))
                                 (count-literals (literal-next L))))
        (t (+ 1 (count-literals (literal-next L))))))


;;;________________________________________________________________________________
;;; POS-COV
;;;
;;;  given a set of possibly extended positive tuples and the number of terms in
;;;  an original (unextended) tuple.  Returns the number of unique progenators
;;;  represented in the extended positive tuples.
;;;
;;;  rv  who    date      reason

(defun pos-cov (pos-tuples number-of-initial-terms)
  (let ((number-of-added-terms (- (length (car pos-tuples)) number-of-initial-terms)))
    (length (remove-duplicates
             (mapcar #'(lambda(x)
                         (butlast x number-of-added-terms)) pos-tuples) :test #'equal))))






;;_________________________________________________________________________________
;; Encapsulating environment for redefining info-gain to handle stopping criteria

(let ((initial-training-set-pos nil)
      (initial-training-set-neg nil)
      (arity-of-pred-being-learned nil)
      (clause-training-set-pos nil)
      (clause-training-set-neg nil)
      (clause-training-set-total nil)
      (pos-covered-so-far nil)
      (old-types nil)
      (bits-to-encode-clause-so-far nil)
      (literals-in-clause-so-far  nil)
      (maximum-bits-needed-to-indicate-pos-tuples nil)
      (bits-available nil)
      (some-literal-required-too-many-bits nil))

  (setf (symbol-function 'real-info-gain) (symbol-function 'info-gain))


  ;;_________________________________________________________________________________
  ;; set-initial-training-set-values-stopping-criteria
  
  (defun set-initial-training-set-values-stopping-criteria (pos-tuples neg-tuples)
    (setf initial-training-set-pos (length pos-tuples)
          initial-training-set-neg (length neg-tuples)
	  arity-of-pred-being-learned (length (car (or pos-tuples
                                                       neg-tuples)))))


  ;;_________________________________________________________________________________
  ;; set-clause-training-set-values-for-stopping-criteria
  
  (defun set-clause-training-set-values-for-stopping-criteria (pos-tuples neg-tuples)
    (setf clause-training-set-pos (length pos-tuples)
          clause-training-set-neg (length neg-tuples)
          clause-training-set-total (+ clause-training-set-pos clause-training-set-neg)

          some-literal-required-too-many-bits nil
          
          bits-to-encode-clause-so-far 0.0

          literals-in-clause-so-far 0
          
          maximum-bits-needed-to-indicate-pos-tuples
          (+ (log clause-training-set-total 2)
             (lg-binomial clause-training-set-total
                          (min clause-training-set-pos (/ clause-training-set-total 2))))

          bits-available maximum-bits-needed-to-indicate-pos-tuples)

    (when (member :b *focl-trace-level*)
      (format t "~%Clause Training Set:    Pos : ~a     Total : ~a"
              clause-training-set-pos clause-training-set-total)
      (format t "~%Maximum bits needed to indicate positive tuples ~6f bits"
              maximum-bits-needed-to-indicate-pos-tuples)))


  ;;_________________________________________________________________________________
  ;; update-clause-values-for-stopping-criteria

  (defun update-clause-values-for-stopping-criteria (literal-added pos-tuples types)

    (incf literals-in-clause-so-far 1)

    (let* ((literal-bits (bits-to-encode-literal literal-added types))
           (added-literal-bits (- literal-bits (log literals-in-clause-so-far 2)))
           (pos-covered (pos-cov pos-tuples arity-of-pred-being-learned)))

      (incf bits-to-encode-clause-so-far added-literal-bits)

      (setf pos-covered-so-far pos-covered

            old-types types
            maximum-bits-needed-to-indicate-pos-tuples
            (+ (log clause-training-set-total 2)
               (lg-binomial clause-training-set-total
                            (min pos-covered-so-far 
                                 (/ clause-training-set-total 2))))

                    
               bits-available (- maximum-bits-needed-to-indicate-pos-tuples
                                 bits-to-encode-clause-so-far))
    
      (when (member :b *focl-trace-level*)
        (format t "~%Bits to encode added literal :  ~6f" literal-bits)
        (format t "~%Bits required to indicate new literal in clause : ~6f" added-literal-bits)
        (format t "~%Bits to encode clause so far :  ~6f" bits-to-encode-clause-so-far)
        (format t "~%Bits available to encode the rest of the clause :  ~6f~%" bits-available))))
  


  ;;_________________________________________________________________________________
  ;; some-literal-required-too-many-bits

  (defun some-literal-required-too-many-bits ()
    some-literal-required-too-many-bits)


  ;;_________________________________________________________________________________
  ;; predicate-requires-too-many-bits

  (defun predicate-requires-too-many-bits (pred)
    (let ((literal-bits (bits-to-encode-literal-using-pred pred old-types)))

    (cond ((> literal-bits bits-available)
           (if (member :b *focl-trace-level*)
             (format t "~%NOT ENOUGH BITS LEFT! Adding any literal using ~a requires ~6f bits"
                     (p-name pred) literal-bits))
           (setf some-literal-required-too-many-bits t)
           t)
          (t nil))))
 

  ;;_________________________________________________________________________________
  ;; bits-needed-to-indicate-pos-tuples

  (defun bits-needed-to-indicate-pos-tuples (old-variables
                                             pos-tuples
                                             pred
                                             variabilization
                                             negative?)
    (+ (log clause-training-set-total 2)
       (lg-binomial clause-training-set-total
                     (pos-cov (extend-pos-tuples old-variables     ;; This could be
						 pos-tuples        ;; more efficient!
						 pred              ;;
						 variabilization   ;;
						 negative?)        ;;
			      arity-of-pred-being-learned))))

;;  00  glenn  5/2/91  got rid of conjunction-stuff and cliche info for printing

  ;;_________________________________________________________________________________
  ;; info-gain
  ;;
  ;;  Redefine info-gain to determine if the stopping criteria will allow the literal
  ;;  to be added.  If it will compute the information gain of the literal if not
  ;;  return 0 gain.
  ;;
  ;; returns four values
  ;;  1. Info gain
  ;;  2. Max possible gain for any specialized version
  ;;  3. Stop- a flag to determine if this is the best you can do

  
  (defun info-gain (pred
                    variablization
                    negative?
                    pos-tuples
                    neg-tuples
                    
                    &optional 
                    (original-info nil)
                    (use-negative-hash-table nil)
                    
                    &key 
                    instantiated-cliche
                    position-in-cliche)
    
    (multiple-value-bind (gain max-possible-gain covered-all-pos-tuples)
                         (real-info-gain pred
                                         variablization
                                         negative?
                                         pos-tuples
                                         neg-tuples
                                         original-info
                                         use-negative-hash-table
                                         :instantiated-cliche instantiated-cliche
                                         :position-in-cliche position-in-cliche)
      (case *stopping-criteria-enabled*

        ((:FOIL)
         (values gain
                 max-possible-gain
                 covered-all-pos-tuples))

        
        ((:RECURSIVE)
         nil)
#|
         (if (> bits-to-encode-new-clause         ;;  This is no longer computed!
                (bits-needed-to-indicate-pos-tuples old-variables
                                                    pos-tuples
                                                    pred
                                                    variablization
                                                    negative?))
           
           ;;  The positive examples covered do not provide enough bits to encode new clause
           ;;  Return 0 gain and a suitble max-possible-gain to avoid pruning.
           (values 0                     ;; gain
                   max-possible-gain     ;; max-possible-gain
                   nil)                  ;; covered-all-pos-tuples
           
           ;;  The positive examples covered provide enough bits to encode new clause
           ;;  Return real gain, max-possible-gain, covered-all-pos-tuples, 
           (values gain
                   max-possible-gain
                   covered-all-pos-tuples)))
|#

        (otherwise 
         (values gain
                 max-possible-gain
                 covered-all-pos-tuples))))))


;;______________________________________________________________________________________________
;; extend-pos-tuples

(defun extend-pos-tuples (variables
                          pos-tuples 
                          pred 
                          variabilization 
                          negated?)
  (multiple-value-bind (literal-vars
                        new-vars
                        new-types
                        alist-ignore)
                       (transfer-literal-vars variabilization
                                              (p-type pred)
                                              variables
                                              (length variables))
    (declare (ignore new-types alist-ignore))
    (cond
     ((pred-p pred)
      (if negated?
        (extend-negation-tuples pred pos-tuples literal-vars new-vars)
        (extend-tuples-extensional pred pos-tuples literal-vars new-vars)))
     ((builtin-p pred)	
      (if negated?
        (extend-negation-tuples-builtin pred pos-tuples variabilization)
        (extend-tuples-builtin pred pos-tuples variabilization)))
     (t (extend-tuples-intensional pred pos-tuples literal-vars negated? new-vars)))))
