
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

;;;  WARNING: 'concept-description is a special rule name used for judging accuracy
;;;            if you use this name for any rule (or pred?) in your domain theory
;;;            or set of extensional predicates, the tests conducted with this
;;;            code may be invalid.
;;;
;;;  Modified 5/02/91
;;;  Modified just before MLW91 and again 7/25/91
;;;
;;; Things to modify
;;;  1. delete rule should remove rel from property list of pred and brule
;;;  2. garbage colllection should be envoked after each iteration for each system
;;;  3. reduced error pruning should be modified to test drop evey-literal (perhaps a flag to select)
;;;  4. code should be cleaned up by using other-keys.
;;;  5. reduced-error-pruning and illegal-noise-test should be divided into smaller functions



(proclaim '(inline classify-example
                   generate-example
                   return-negative-example
                   return-positive-example
                   make-prolog-rule))

;;;______________________________________________________________________________
;;; GARBAGE-COLLECT
(defun garbage-collect ()
  #+:ccl (gc)
  #+:allegro (gc t)
  #+:kcl (gc)       ;;  <-  I have no idea if this is right
  #+:lucid (gc)
  )

;;;______________________________________________________________________________
;;; MAKE-PROLOG-RULE
;;;                 
;;;   Convert result of focl to executable prolog rule
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Added induction keyword

(defun make-prolog-rule (pred arity clauses &key (induction nil))
  (let* ((old-vars (do ((i (- arity 1) (decf i))
                        (result nil))
                       ((< i 0) result)
                     (push (make-pcvar :id i) result) ))
         (prolog-def (mapcar #'(lambda(x) (cons (cons pred old-vars)
                                                (convert-to-prolog x)))
                             clauses)))
    (setf (get pred 'pred) nil)
    (eval `(def-rule ,pred 
             :clauses ,prolog-def
             :induction ,induction))))


;;;______________________________________________________________________________
;;; JUDGE-CONCEPT
;;;
;;;   This function can be used two ways:
;;;      1.  To determine the accuracy of concept-description at classifying
;;;          a set of correctly classified positive and negative examples.
;;;      2.  To determine the accuracy of a set of positive and negative examples
;;;          given a correct and complete concept-description.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Modified to work with the rule compiler
;;;  01  MP      7/10/91  allow concept function to be passed in
;;;                       useful for judging domain theory
(defun judge-concept (concept-description pos-examples neg-examples 
					  &optional (concept-function
						     (make-concept-function 'concept-description 
									    (length (or (first pos-examples) 
											(first neg-examples)))
									    concept-description)))
  (let ((pos-correct 0)			; number of pos-examples concept-description correctly classifies
        (neg-correct 0)			; number of neg-examples concept-description correctly classifies
        (pos-missed 0)			; number of pos-examples concept-description incorrectly classifies
        (neg-missed 0)			; number of neg-examples concept-description incorrectly classifies
        )
    (dolist (p pos-examples)
	    (if (prove-function? concept-function nil nil p)
		(incf pos-correct)
	      (incf pos-missed)))
    (dolist (n neg-examples)
	    (if (prove-function? concept-function nil nil n)
		(incf neg-missed)
	      (incf neg-correct)))
    (values pos-correct neg-correct pos-missed neg-missed)))


;;;______________________________________________________________________________
;;; QUALITATIVE-FEATURES
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   6/13/91  

(defun qualitative-features (concept-description)
  (let ((number-of-clauses (length concept-description))
        (number-of-non-negated-literals 0)
        (number-of-negated-literals 0))
    (dolist (clause concept-description)
      (do* ((literal clause (literal-next literal)))
           ((null literal))
        (if (literal-negated? literal)
          (incf number-of-negated-literals
                (count-literals (literal-negated-literals literal)))
          (incf number-of-non-negated-literals 1))))
    (values number-of-clauses 
            (+ number-of-non-negated-literals number-of-negated-literals)
            number-of-negated-literals)))


;;;______________________________________________________________________________
;;; GENERATE-EXAMPLE-SET
;;;
;;;    generates or augments a set of positive (in) and negative (out) examples
;;;    which contain percent-class-noise percent noise and which are composed of
;;;    percent-positive postive examples (:from-domain uses the distribution
;;;    obtained from the example generator).
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   3/3/91   Correctly implement noise (used to be error rate)
;;;                       now its 50% error rate.

(defun generate-example-set (n          ; number of examples to generate
                             &key 
                             (in nil)   ; list of positive examples (possibly containing noise)
                             (out nil)  ; list of negative examples (possibly containing noise)
                             (percent-positive :from-domain)
                             (percent-class-noise 0)
                             (percent-tuple-noise 0))

  (let ((p-class-noise percent-class-noise)
        (n-class-noise percent-class-noise)
        (tuple-noise-level (* percent-tuple-noise 1000))
        (domain-range 8))         ;; [XXXX] Hacked for illegal

    (labels 
      ((add-t-noise (example)
                        (cond ((null example) nil)
                              ((< (random 100000) tuple-noise-level)
                               (rplaca example (+ (random domain-range) 1))
                               (add-t-noise (cdr example)))
                              (t (add-t-noise (cdr example)))))
       (add-tuple-noise (example)
                        (if (> tuple-noise-level 0)
                          (add-t-noise example))
                        example))

      (if (equal percent-positive :from-domain)
        (dotimes (i n)                            ; Select examples from the domain at random
          (let* ((example (generate-example)))
            (if (classify-example example)
              (if (< (random 100) p-class-noise)  ; Introduce class noise into positive examples
                (if (< (random 100) 50)           ;   Introduce error with probablity .5
                  (push (add-tuple-noise example) out)              
                  (push (add-tuple-noise example) in))
                (push (add-tuple-noise example) in))
              (if (< (random 100) n-class-noise)  ; Introduce class noise into negative examples
                (if (< (random 100) 50)           ;   Introduce error with probablity .5   
                  (push (add-tuple-noise example) out)
                  (push (add-tuple-noise example) in))
                (push (add-tuple-noise example) out)))))
        
        (dotimes (i  n)                              ; Select examples to conform to percent-positive
          (if (< (random 100) percent-positive)      ; Choose class of example (pos or neg)
            (if (< (random 100) p-class-noise)       ; Introduce class noise into positive examples
              (if (< (random 100) 50)                ;   Introduce error with probablity .5
                (push (add-tuple-noise (return-positive-example)) out)              
                (push (add-tuple-noise (return-positive-example)) in))
              (push (add-tuple-noise (return-positive-example)) in))
            (if (< (random 100) n-class-noise)       ; Introduce class noise into negative examples
              (if (< (random 100) 50)                ;   Introduce error with probablity .5   
                (push (add-tuple-noise (return-negative-example)) out)
                (push (add-tuple-noise (return-negative-example)) in))
              (push (add-tuple-noise (return-negative-example)) out)))))
      
      (values in out))))







(defconstant scale-factor 100)
(defconstant max-level (* 100 scale-factor))

;;;______________________________________________________________________________
;;; SWAP  (swap the pos1 element with the pos2 element in a list)

(defun swap (list pos1 pos2)
  (let ((cons1 (nthcdr pos1 list))
        (cons2 (nthcdr pos2 list))
        (temp nil))
    (setf temp (car cons1))
    (rplaca cons1 (car cons2))
    (rplaca cons2 temp))
  list)

;;;______________________________________________________________________________
;;; INTRODUCE-CLASS-NOISE
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   4/10/91  Created

(defun introduce-class-noise (real-pos 
                              real-neg
                              &key (percent-noise 0))
  (let ((noise-level (* percent-noise scale-factor))
        (noisy-pos nil)
        (noisy-neg nil))

    (dolist (p real-pos)                        ;; Misclassify some of the positives
      (if (< (random max-level) noise-level)
        (if (< (random 100) 50)
          (push p noisy-neg)
          (push p noisy-pos))
        (push p noisy-pos)))

    (dolist (n real-neg)                        ;; Misclassify some of the negatives
      (if (< (random max-level) noise-level)
        (if (< (random 100) 50)  
          (push n noisy-pos)
          (push n noisy-neg))
        (push n noisy-neg)))

    (let ((l-noisy-pos (length noisy-pos))      ;; Scramble Misclassified examples
          (l-noisy-neg (length noisy-neg)))
      (dotimes (i l-noisy-pos)
        (swap noisy-pos i (random l-noisy-pos))
      (dotimes (i l-noisy-neg)
        (swap noisy-neg i (random l-noisy-neg)))))

    (values noisy-pos noisy-neg)))




;;;______________________________________________________________________________
;;; RETURN-POSITIVE-EXAMPLE
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun return-positive-example ()
  (do ((example (generate-example)))
      ((illegal-position example) example)
    (setq example (generate-example))))


;;;______________________________________________________________________________
;;; RETURN-NEGATIVE-EXAMPLE
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun return-negative-example ()
  (do ((example (generate-example))) 
      ((not (illegal-position example)) example)
    (setq example (generate-example))))


;;;______________________________________________________________________________
;;; GENERATE-EXAMPLE
;;;
;;;    generate a random board position
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun generate-example ()
  (let ((result nil))
    (dotimes (i 6 result) (push (+ 1 (random 8)) result))))

;;;______________________________________________________________________________
;;; CLASSIFY-EXAMPLE

(defun classify-example (example)
    (illegal-position example))


;;;______________________________________________________________________________
;;; judge-example-set
;;;     judges the accuracy of a set of examples with respect to illegal-position.
;;;
;;; Note:  This could be replaced by a call to judge with a correct and complete
;;;        concept-description
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun judge-example-set (in out)
  (let ((pos-correct 0)   ; number of in which are actually positive examples
        (neg-correct 0)   ; number of out which are actually negative examples
        (pos-missed 0)    ; number of in which are actually negative examples
        (neg-missed 0))   ; number of out which are actually positive examples
    (dolist (example in)
      (if (illegal-position example)
        (incf pos-correct)
        (incf pos-missed)))
    (dolist (example out)
      (if (illegal-position example)
        (incf neg-missed)
        (incf neg-correct)))
    (values pos-correct neg-correct pos-missed neg-missed)))



;;;______________________________________________________________________________
;;; LISP DOMAIN CONCEPT DESCRIPTION                             (Domain Specific)
;;;______________________________________________________________________________
;;; illegal-position (L)
;;;     returns T if position is illegal.
;;;     L is a vector of peices (white king, white rook, and black king)
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun adjacent (i j k l)
  (or (and (eq i k) 
           (near j l)) 
      (and (eq j l)
           (near i k))
      (and (near i k) 
           (near j l))))

(defun near (i j)
  (or (eq i (+ 1 j)) 
      (eq i (- j 1))))

(defun between (a b c d e f)
  (or (and (= a c e) 
           (or (< d b f)
               (> d b f)))
      (and (= b d f) 
           (or (< c a e) 
               (> c a e)))))

(defun same-line (i j k l)
  (or (eq i k) 
      (eq j l)))

(defun illegal-position (L)
   (cond ((and (eq (first L) (third L))          ;; white king and white rook at same position
               (eq (second L) (fourth L))) t)
         ((and (eq (first L) (fifth L))          ;; white king and black king at same position
               (eq (second L) (sixth L))) t)
         ((and (eq (third L) (fifth L))          ;; white rool and black king at same position
               (eq (fourth L) (sixth L))) t)
         ((adjacent (first L) (second L)         ;; white king capture black king
                    (fifth L) (sixth L)) t) 
         ((and (same-line (third L) (fourth L)   ;; white rook captures black king
                          (fifth L) (sixth L))
               (not (between (first L) (second L) 
                             (third L) (fourth L) 
                             (fifth L) (sixth L))) t))))


;;;______________________________________________________________________________
;;; GENERATOR-ILLEGAL-DEF-PRED
;;;
;;;     generates and evaluated a def-pred for illegal containing in and out
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun generator-illegal-def-pred (in
                                   out
                                   &key (typed? nil))
  (if typed? 
    (eval `(def-pred illegal 
             :induction nil 
             :type (:row :column :row :column :row :column)
             :pos ,in 
             :neg ,out))
    (eval `(def-pred illegal 
             :induction nil
             :type (:thing :thing :thing :thing :thing :thing)
             :pos ,in 
             :neg ,out))))


;;;_____________________________________________________________________________
;;; PRINT-HEADER

(defun print-header ()
  (format t "~%~%;P P +~AN N +~AMCP -~AMCN -~A Cs ~A Ls ~AN-Ls~ASize~A%Pos~A%Noise~ANVs~A Alogithm"
          #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab)
  (format t "~%;_____~A_____~A_____~A_____~A____~A____~A____~A____~A____~A______~A___~A__________"
          #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab))

;;;_____________________________________________________________________________
;;; PRINT-DATUM

(defun print-datum (PC NC MCP MCN test-size Cs Ls N-Ls
                    size percent-positive noise max-new-vars alogithm)
  (format t "~%~5f~A~5f~A~5f~A~5f~A~A~A~A~A~A~A~A~A~A~A~A~A~A~A~A"
          (/ pc test-size) #\tab 
          (/ nc test-size) #\tab
          (/ mcp test-size) #\tab 
          (/ mcn test-size) #\tab
          Cs #\tab
          Ls #\tab
          N-Ls #\tab
          size #\tab
          percent-positive #\tab
          noise #\tab
          max-new-vars #\tab
          alogithm))


(defun noise-level-test ()
  (reset-preds)
  (load-source-file "illegal-preds-untyped")
  (noise-test :noise-levels '(2 5 10 20 50)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun prune-set-size-test ()
  (reset-preds)
  (load-source-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(5 10 20 30 33.33 40 50 70 90)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun maximum-new-variables-test ()
  (reset-preds)
  (load-source-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(0 1 2)
              :iterations 10
              :starting-state t))

(defun example-set-size-test ()
  (reset-preds)
  (load-source-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(10 20 40 80 160 320)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun distribution-test ()
  (reset-preds)
  (load-source-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(10 20 :from-domain 50 70 90)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))



;;;_____________________________________________________________________________
;;; NOISE-TEST

(defun noise-test (&key (noise-levels '(20))
                        (percent-positive-values '(50))
                        (example-set-sizes '(250))
                        (test-set-size 1000)
                        (nt-alogorithms '(:STANDARD :FOIL :REP))
                        (percent-pruning-set-sizes '(33.33))
                        (maximum-new-variables-values '(0))
                        (iterations 10)
                        (starting-state t)
                        )
  (let ((learn-pred-name 'illegal)
        (focl-parameters nil)
        (current-example-set-size 0)
        (pos nil)
        (neg nil)
        (in nil)
        (out nil))
    (setf (getf focl-parameters :clauses) nil
          (getf focl-parameters :reset-hash-tables) t
          (getf focl-parameters :operationalize-constructive) t 
          (getf focl-parameters :constructive-induction) nil
          (getf focl-parameters :trace) nil
          (getf focl-parameters :simplify-clauses) t
          (getf focl-parameters :reset-statistics) t
          (getf focl-parameters :save-examples) nil
          (getf focl-parameters :prefer-theory) t
          (getf focl-parameters :proof-vars-available) t
          (getf focl-parameters :partial-dt-0-gain) nil
          (getf focl-parameters :simplify-operationalizations) nil
          (getf focl-parameters :batch-mode) nil

          *random-state* (make-random-state starting-state))

    (format t "~%~%#|")
    (format t "~%~%;;;ILLEGAL NOISE-TEST")
    (format t "~%;;;")
    (format t "~%;;;  NOISE-LEVELS :            ~A" noise-levels)
    (format t "~%;;;  PERCENT POSITIVE :        ~A" percent-positive-values)
    (format t "~%;;;  EXAMPLE-SET-SIZES :       ~A" example-set-sizes)
    (format t "~%;;;  TEST-SET-SIZES :          ~A" test-set-size)
    (format t "~%;;;  ALGORITHMS :              ~A" nt-alogorithms)
    (format t "~%;;;  PERCENT-PRUNE-SET-SIZES : ~A" percent-pruning-set-sizes)
    (format t "~%;;;  MAX-NEW-VARIABLES :       ~A" maximum-new-variables-values)
    (format t "~%;;;")    
    (format t "~%;;;  ITERATIONS :              ~A" iterations)
    (format t "~%;;;  INITIAL RANDOM STATE :    ~A" *random-state*)
    (format t "~%~%|#")

    (dotimes (i iterations)
      (garbage-collect)
      (dolist (percent-positive percent-positive-values)
        (multiple-value-bind (pos-test neg-test)
                             (generate-example-set test-set-size 
                                                   :percent-positive percent-positive
                                                   :percent-tuple-noise 0
                                                   :percent-class-noise 0)
          (setf current-example-set-size 0
                pos nil
                neg nil)
          (dolist (example-set-size example-set-sizes)        
            (multiple-value-setq (pos neg)
                                 (generate-example-set (- example-set-size current-example-set-size)
                                                       :in pos
                                                       :out neg
                                                       :percent-positive percent-positive
                                                       :percent-tuple-noise 0
                                                       :percent-class-noise 0))
            (setf current-example-set-size example-set-size)
                                 
            (dolist (noise-level noise-levels)
              (multiple-value-setq (in out)
                                   (introduce-class-noise pos neg :percent-noise noise-level))
              
              (let ((length-in (length in))
                    (length-out (length out)))
                
                (print-header)
                (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                     (judge-example-set in out)
                  (print-datum pos-correct neg-correct pos-missed neg-missed
                               example-set-size "" "" ""
                               example-set-size percent-positive noise-level
                               "" "EXAMPLE SET"))

                (dolist (max-new-variables maximum-new-variables-values)
                  (dolist (nt-alogorithm nt-alogorithms)
                    (case nt-alogorithm
                      
                      ((:STANDARD :RECURSIVE :FOIL)
                       (generator-illegal-def-pred in out)
                       (setf (getf focl-parameters :noise-tolerance) (if (eql :STANDARD nt-alogorithm)
                                                                       nil
                                                                       nt-alogorithm)
                             (getf focl-parameters :max-new-variables) max-new-variables)
                       (let ((learned-concept-description 
                              (apply #'focl learn-pred-name focl-parameters)))
                         
                         (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                              (judge-concept learned-concept-description 
                                                             pos-test 
                                                             neg-test)
                           (multiple-value-bind (clauses literals negated-literals)
                                                (qualitative-features learned-concept-description)

                             (print-datum pos-correct neg-correct pos-missed neg-missed
                                          test-set-size clauses literals negated-literals
                                          example-set-size percent-positive noise-level 
                                          max-new-variables nt-alogorithm)))))

                      (:REP
                       (dolist (percent-pruning-set-size percent-pruning-set-sizes)
                         (let* 
                           ((pos-pruning-set-size (round (* percent-pruning-set-size length-in) 100))
                            (neg-pruning-set-size (round (* percent-pruning-set-size length-out) 100))
                            (pos-training-set (butlast in pos-pruning-set-size))
                            (pos-pruning-set (nthcdr (- length-in pos-pruning-set-size) in))
                            (neg-training-set (butlast out neg-pruning-set-size))
                            (neg-pruning-set (nthcdr (- length-out neg-pruning-set-size) out))
                            (pruning-set-size (+ pos-pruning-set-size neg-pruning-set-size))
                            (percent-training-set-size (- 100 percent-pruning-set-size))
                            (training-set-size (- example-set-size pruning-set-size)))
                           
                           (when (> pruning-set-size 0)
                          
                             (generator-illegal-def-pred pos-training-set neg-training-set)
                             
                             (setf (getf focl-parameters :noise-tolerance) nil
                                   (getf focl-parameters :max-new-variables) max-new-variables)
                             (let ((learned-concept-description 
                                    (apply #'focl learn-pred-name focl-parameters)))
                               
                               (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                                    (judge-concept learned-concept-description 
                                                                   pos-test 
                                                                   neg-test)
                                 (multiple-value-bind (clauses literals negated-literals)
                                                      (qualitative-features learned-concept-description)
                                   
                                   (print-datum pos-correct neg-correct pos-missed neg-missed
                                          test-set-size clauses literals negated-literals
                                          training-set-size percent-positive noise-level 
                                          max-new-variables 
                                          (format nil "~A% STANDARD" percent-training-set-size))))
                               
                               (setf learned-concept-description 
                                     (Reduced-Error-Pruning 
                                      :concept-description learned-concept-description
                                      :pos-set pos-pruning-set
                                      :neg-set neg-pruning-set))
                               
                               (case learned-concept-description
                                 ((:failed-true :true)
                                  (print-datum (length pos-test) 0 0 (length neg-test)
                                               test-set-size 1 1 0
                                               example-set-size percent-positive noise-level 
                                               max-new-variables 
                                               (format nil "~A% ~A" percent-pruning-set-size
                                                       nt-alogorithm)))

                                 ((:failed-empty :empty)
                                  (print-datum  0 (length neg-test) (length pos-test) 0 
                                               test-set-size 1 1 1
                                               example-set-size percent-positive noise-level 
                                               max-new-variables 
                                               (format nil "~A% ~A" percent-pruning-set-size
                                                       nt-alogorithm)))
                                 
                                 (otherwise
                                  (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                                       (judge-concept learned-concept-description 
                                                                      pos-test
                                                                      neg-test)
                                    (multiple-value-bind (clauses literals negated-literals)
                                                         (qualitative-features learned-concept-description)
                                      
                                      (print-datum  pos-correct neg-correct pos-missed neg-missed
                                                    test-set-size clauses literals negated-literals
                                                    example-set-size percent-positive noise-level 
                                                    max-new-variables 
                                                    (format nil "~A% ~A" percent-pruning-set-size
                                                       nt-alogorithm)))))
                                 ))))))
                      )))))))))))
  

#|
;;;  phax-function to test control flow

(defun focl (pred &key noise-tolerance 
                       max-new-variables
                  &allow-other-keys)
  (declare (ignore pred))
  (declare (ignore noise-tolerance))
  (declare (ignore max-new-variables))
  nil)

(defun Reduced-Error-Pruning (&rest keys)
  (declare (ignore keys))
  nil)

(defun judge-concept (&rest keys)
  keys
  (values 100 100 100 100))

(defun qualitative-features (&rest keys)
  keys
  (values 0 0 0))
|#

