
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)
;;;  move the structure to structs.lisp

(defstruct cliche
  name
  pred-restrictions
  var-restrictions
  cache?)

(defstruct literal-info 
  pred
  variabilization
  negated?
  pos
  neg
  vars
  types
  gain
  new-pos
  new-neg
  new-vars
  new-types)




;;defines a new cliche
(defmacro def-cliche (name &key pred-restrictions var-restrictions (cache? 'named))
  `(let*  ((name ',name)
           (pred-restrictions ',pred-restrictions)
           (var-restrictions ',var-restrictions)
           (cache? ',cache?)
           (bucket (assoc name *all-cliches*))
           (cliche (make-cliche :name name :pred-restrictions pred-restrictions
                                :var-restrictions var-restrictions
                                :cache? cache?)))
     (setf (get name 'cliche) cliche)
     (if bucket 
       (setf (cdr bucket) cliche)
       (push (cons name cliche) *all-cliches*))))

(defun get-cliche-struct (cliche-name)
  (get cliche-name 'cliche))

;;; runs through all *available-relational-cliches*

(defun find-literal-cliches (original-info 
                             predicate-being-learned 
                             variables 
                             variable-types
                             maximum-new-vars
                             pos-tuples 
                             neg-tuples
                             original-vars 
                             use-hash-tables
                             gain-to-beat)
  (let ((cliche nil)
        (cliche-max-gain 0)
        (best-instantiated-cliche nil)
        (best-instantiated-cliche-name nil)
        (instantiated-cliche-with-max-gain nil)
        max-gain 
        covered-all-pos-tuples
        (varzns-checked-before-cliches *variablizations-checked*))
    (do* ((cliche-names *available-relational-cliches* (cdr cliche-names))
          (cliche-name (car cliche-names) (car cliche-names)))
         ((null cliche-name))

      (when *graph-learning*                                      ;;  CB (7/25/91)
        (display-cliche-name-in-induction-graphic cliche-name))   ;;

      (setq cliche (get-cliche-struct cliche-name))
      ; maybe can be a little smarted don't always have to deallocate
      (deallocate-literal-info-structs instantiated-cliche-with-max-gain)
      (setq instantiated-cliche-with-max-gain
            (make-literal-info-structs (length (cliche-pred-restrictions cliche))))
      (multiple-value-setq
        (instantiated-cliche-with-max-gain max-gain covered-all-pos-tuples)
        ; note figure out how to get covered-all-pos-tuples right (must make sure all pos-tuples are covered)
        (instantiate-cliche (cliche-pred-restrictions cliche) 
                            (cliche-var-restrictions cliche)
                            original-info predicate-being-learned variables variable-types 
                            *max-new-cliche-vars* pos-tuples
                            neg-tuples original-vars use-hash-tables gain-to-beat
                            instantiated-cliche-with-max-gain 0)) ; added 5/17
      (when (and instantiated-cliche-with-max-gain 
                 (not (eql instantiated-cliche-with-max-gain :fail))
                 (> max-gain gain-to-beat))
        (setq best-instantiated-cliche (copy-cliche-info instantiated-cliche-with-max-gain 
                                                         best-instantiated-cliche)
              best-instantiated-cliche-name cliche-name
              cliche-max-gain max-gain)
        ;(format t "~%setting best-cliche with gain ~a to " max-gain)
        ;(print-cliche best-instantiated-cliche)
))
    (incf *cliche-variabilizations-checked* 
        (- *variablizations-checked* varzns-checked-before-cliches))
    (cond ((null best-instantiated-cliche) nil)
          ((eql best-instantiated-cliche :fail) :fail)
          (t (create-literal-conjunction-from-structs best-instantiated-cliche 
                                                      cliche-max-gain
                                                      best-instantiated-cliche-name
                                                      variables)))))
                ; note cliche-max-gain should not be necessary



;;; return a list of num literal-info structs to be reused while instantiating cliches

(defun make-literal-info-structs (num)
  (let ((literal-info-structs nil))
    (dotimes (i num literal-info-structs)
      (push (allocate-literal-info-struct) literal-info-structs))))

; basically look at predicate restriction to decide whether to call 
; find-maximum-literal, find-literal-builtin, or both. also need to decide
; what predicates to pass to each (e.g., may pass arithmetic operators to
; implement arithmetic operator cliche, or = to implement non-numeric constant
; cliche.  Also need to be more careful about updating pos-tuples when 
; conjunctions are passed around.  Note can have conjunctions of arbitrary
; length.  Current thought is to accumulate necessary info in a list of lists
; and then and link the literals from the list.  Note need to be careful
; about updating the lists properly.  Important to complete each cliche and 
; measure info gain in a depth-first fashion so that we don't have a lot of
; junk lying around

(defun instantiate-cliche (pred-restrictions var-restrictions original-info
                           predicate-being-learned variables variable-types maximum-new-vars 
                           pos-tuples neg-tuples original-vars use-hash-tables gain-to-beat 
                           instantiated-cliche position-in-cliche)
  ;(format t "~%calling instantiate-cliche with vars ~a types ~a" variables variable-types)
  (let ((current-literal-info-struct (nth position-in-cliche instantiated-cliche))
        (max-gain gain-to-beat)
        (max-negated? nil)
        (covered-all-pos-tuples nil)
        (best-instantiated-cliche nil)
        (current-instantiated-cliche nil)
        (variabilization nil)
        (pred-max-gain nil)
        (pred-restr (car pred-restrictions))
        (var-restr (car var-restrictions))
        negated? ; for equality constant varzns
        (new-vars nil)
        (new-types nil)
        (all-vars nil)
        (all-types nil)
        (literal-vars nil)
        (alist-ignore nil)
        (new-pos-tuples nil)
        (new-neg-tuples nil)
        (pred-type-restr nil))
    (setq pred-type-restr (get-pred-type-restr pred-restr))
    (cond ((null pred-restrictions) nil); probably shouldn't get here
          ((null (cdr pred-restrictions)) ; compute info gain only on last one
           ;;; compute
           (let ((overall-pred-with-max-gain nil)
                 (overall-varzn-with-max-gain nil)) ; for info-gain
             (when (or (eql 'ext-pred pred-type-restr) (eql 'pred pred-type-restr))
               (expand-last-cliche-pred
                (find-maximum-literal original-info predicate-being-learned variables 
                                      variable-types maximum-new-vars pos-tuples 
                                      neg-tuples original-vars use-hash-tables 
                                      *extensional-preds* max-gain ;; gain-to-beat 
                                      :try-cliches t
                                      :pred-restrictions pred-restr
                                      :var-restrictions var-restr
                                       :instantiated-cliche instantiated-cliche
                                      :position-in-cliche position-in-cliche)))
             (when (or (eql 'pred pred-type-restr) (eql 'var-comp pred-type-restr))
               (expand-last-cliche-pred
                (find-maximum-literal original-info predicate-being-learned variables 
                                      variable-types maximum-new-vars pos-tuples 
                                      neg-tuples original-vars use-hash-tables 
                                      *builtin-preds* max-gain ;; gain-to-beat 
                                      :try-cliches t
                                      :pred-restrictions pred-restr
                                      :var-restrictions var-restr 
                                      :instantiated-cliche instantiated-cliche
                                      :position-in-cliche position-in-cliche)))
             (when (or (eql 'comp pred-type-restr) (eql 'pred pred-type-restr))
               (expand-last-cliche-pred
                (find-literal-builtin-thresh original-info variables variable-types pos-tuples 
                                      neg-tuples *builtin-preds* max-gain ;; gain-to-beat 
                                      :pred-restrictions pred-restr
                                      :var-restrictions var-restr
                                      :instantiated-cliche instantiated-cliche
                                      :position-in-cliche position-in-cliche)))
             (cond ((and overall-pred-with-max-gain
                        (not (eql overall-pred-with-max-gain :fail)))
                    (update-literal-info current-literal-info-struct overall-pred-with-max-gain 
                                         overall-varzn-with-max-gain pos-tuples neg-tuples 
                                         max-negated? variables variable-types max-gain)
                    (values instantiated-cliche max-gain))
      
                   (t (values nil 0)))
             ))
          (t 
           (do* ((preds (filter-pred-restrictions pred-restr maximum-new-vars) (cdr preds))
                 (pred (cdr (car preds)) (cdr (car preds))))
                ((null preds)
                 (cond ((and best-instantiated-cliche 
                             (not (eql best-instantiated-cliche :fail))
                             (> max-gain gain-to-beat))
                        (deallocate-literal-info-structs instantiated-cliche)
                        (values best-instantiated-cliche max-gain))
                       (t (deallocate-literal-info-structs instantiated-cliche)
                          (deallocate-literal-info-structs best-instantiated-cliche)
                          (values nil 0))))
             ;; handle positive variabilizations
             (setq max-negated? nil)
             (process-variabilizations 
              var-restr
              (process-cliche-variabilizations nil pred)) ; 
             ;;; deal with threshold variabilizations equality constants
             (when (pred-type-restriction-supports-threshold pred-type-restr)
               (process-threshold-variabilizations
                 (relational-thresh-builtins preds)
                 var-restrictions
                 variables
                 variable-types
                 pos-tuples
                 neg-tuples
                 variabilization
                 pred
                 instantiated-cliche
                 nil
                 (process-cliche-variabilizations nil comp) 
                 (process-cliche-variabilizations t comp))
               (process-equality-constant-varzns
                 (equality-constant-builtins preds)
                 var-restrictions
                 variables
                 variable-types
                 pos-tuples
                 neg-tuples
                 variabilization
                 pred
                 negated?
                 instantiated-cliche
                 nil
                 (process-cliche-variabilizations negated? comp))
               ) ; do negative at the same time for thresholds
             ; may want to handle this through varzn restrictions
             (when (process-neg-varzns-for-cliche? pred-restr)
               (setq max-negated? t)
               (process-variabilizations
                var-restr
                (process-cliche-variabilizations t pred))))))))


;;; by default skip negatives unless explicitly stated
(defun process-neg-varzns-for-cliche? (pred-restr)
  (or *try-all-conjunctions* 
      (member 'negative-varzns pred-restr)))


(defun update-literal-info (literal-info pred literal-vars pos-tuples neg-tuples max-negated? 
                            vars types gain &optional new-pos-tuples new-neg-tuples new-vars new-types)
  (setf (literal-info-pred literal-info) pred)
  (setf (literal-info-variabilization literal-info) literal-vars)
  (setf (literal-info-negated? literal-info) max-negated?)
  (setf (literal-info-vars literal-info) vars)
  (setf (literal-info-types literal-info) types)
  (setf (literal-info-gain literal-info) gain)
  (setf (literal-info-new-vars literal-info) new-vars)
  (setf (literal-info-new-types literal-info) new-types)
  (setf (literal-info-pos literal-info) pos-tuples)
  (setf (literal-info-neg literal-info) neg-tuples)
  (setf (literal-info-new-pos literal-info) new-pos-tuples)
  (setf (literal-info-new-neg literal-info) new-neg-tuples))

;;; computes the old-variables used in the instantiation of the cliche which will
;;; become the head vars of the cliche

(defun compute-cliche-head-vars (literal-conj old-vars)
  (let ((clause (convert-to-prolog literal-conj)))
    (remove-if-not
     #'(lambda (var)
         (some 
          #'(lambda (lit)
              (member var lit :test #'var-eq))
          clause))
         old-vars)))




;;; creates a conjunction of literals from an instantiated cliche which is in the form
;;; of a list of literal info structs.

(defun create-literal-conjunction-from-structs (literal-info-structs gain cliche-name old-vars)
  (let ((conjunction nil)
        (literal nil)
        (last-literal nil)
        (new-vars nil)
        (new-types nil)
        (new-pos-tuples nil)
        (new-neg-tuples nil)
        (first-literal nil)
        (max-gain nil)
        (last-literal-vars nil)
        (last-literal-types nil))
    (do* ((li-structs literal-info-structs (cdr li-structs))
          (li-struct (car li-structs) (car li-structs)))
         ((null li-structs) conjunction)
      (setq literal
            (cond ((cdr li-structs)
                   (make-literal-from-literal-info li-struct))
                  (t (setq last-literal-vars (literal-info-vars li-struct))
                     (setq last-literal-types (literal-info-types li-struct))
                     (multiple-value-setq 
                         (last-literal new-vars new-types new-pos-tuples new-neg-tuples
                          max-gain)
                         (create-literal-from-literal-info li-struct))
                     last-literal)))
      (push-last literal conjunction))
    (setq first-literal (car conjunction))
    (create-literal-linked-list conjunction)
    (let ((first-vars (literal-info-vars (car literal-info-structs)))
          (all-new-vars nil)
          (all-new-types nil))
      (mapc #'(lambda (var type) 
                (when (not (member var first-vars :test #'var-eq))
                  (push-last var all-new-vars)
                  (push-last type all-new-types)))
            last-literal-vars
            last-literal-types)
      (setq all-new-vars (nconc all-new-vars new-vars))
      (setq all-new-types (nconc all-new-types new-types))
      (values first-literal
              all-new-vars
              all-new-types
              new-pos-tuples
              new-neg-tuples
              gain
              cliche-name
              (compute-cliche-head-vars first-literal old-vars)))))

(defun create-literal-linked-list (literal-list)
  (do* ((literals literal-list (cdr literals))
        (prev nil current)
        (current (car literals) next)
        (next (second literals) (second literals)))
       ((null current) (car literal-list))
    (setf (literal-prev current) prev)
    (setf (literal-next current) next)))
              
(defvar *literal-info-structs* nil)

(defun make-literal-from-literal-info (literal-info)
  (let ((new-literal
         (construct-literal (literal-info-negated? literal-info)
                            (literal-info-pred literal-info)
                            (literal-info-variabilization literal-info)
                            :constructive-induction)))
    (when *save-examples*    
      (setf (literal-pos new-literal) (literal-info-pos literal-info))
      (setf (literal-neg new-literal) (literal-info-neg literal-info))
      (setf (literal-new-pos new-literal) (literal-info-new-pos literal-info))
      (setf (literal-new-neg new-literal) (literal-info-new-neg literal-info)))
    new-literal))

(defun create-literal-from-literal-info (literal-info)
;;; call create-literal here
  (create-literal (literal-info-vars literal-info) 
                  (literal-info-pos literal-info)
                  (literal-info-neg literal-info)
                  (literal-info-pred literal-info)
                  (literal-info-variabilization literal-info)
                  (literal-info-negated? literal-info)
                  (literal-info-gain literal-info)))

(defun copy-cliche-info (from-cliche to-cliche)
  (if (null to-cliche)
    (setq to-cliche (make-literal-info-structs (length from-cliche))))
  (mapcar #' copy-literal-info-struct from-cliche to-cliche))

(defun copy-literal-info-struct (from-literal-info to-literal-info)
  (setf (literal-info-pred to-literal-info) (literal-info-pred from-literal-info))
  (setf (literal-info-variabilization to-literal-info) 
        (literal-info-variabilization from-literal-info))
  (setf (literal-info-negated? to-literal-info) (literal-info-negated? from-literal-info))
  (setf (literal-info-pos to-literal-info) (literal-info-pos from-literal-info))
  (setf (literal-info-neg to-literal-info) (literal-info-neg from-literal-info))
  (setf (literal-info-new-pos to-literal-info) (literal-info-new-pos from-literal-info))
  (setf (literal-info-new-neg to-literal-info) (literal-info-new-neg from-literal-info))
  (setf (literal-info-vars to-literal-info) (literal-info-vars from-literal-info))
  (setf (literal-info-types to-literal-info) (literal-info-types from-literal-info))
  (setf (literal-info-new-vars to-literal-info) (literal-info-new-vars from-literal-info))
  (setf (literal-info-new-types to-literal-info) (literal-info-new-types from-literal-info))
  to-literal-info)

(defun allocate-literal-info-struct ()
  (if *literal-info-structs*
    (pop *literal-info-structs*)
    (make-literal-info)))

(defun deallocate-literal-info-struct (struct)
  (pushnew struct *literal-info-structs*))

(defun deallocate-literal-info-structs (structs)
  (mapc #'deallocate-literal-info-struct structs))

;;; misc

(defun print-cliche (cliche &optional position)
  (terpri)
  (when cliche
    (if position 
      (dotimes (i position)
        (print-cliche-element (nth i cliche)))
      (dolist (l-i cliche)
        (print-cliche-element l-i)))))


(defun print-cliche-element (l-i)
  (if (literal-info-negated? l-i) (princ "~"))
  (if (literal-info-pred l-i)
    (format t "~a~a " (p-name (literal-info-pred l-i)) (literal-info-variabilization l-i))
    (format t "<Empty Pred> ")))


(defun create-pred-from-cliche (literal-conj cliche-name cliche-head-vars)
  (let* ((cliche (get-cliche-struct cliche-name))
         (cache? (cliche-cache? cliche)))
    (when cache?
      (let* ((new-pred-name (gensym (format nil "~a-CLICHE" cliche-name)))
             (clauses (list (cons (cons new-pred-name cliche-head-vars)
                                  (convert-to-prolog literal-conj))))
             (new-pred
              (eval `(def-rule ,new-pred-name
                       :clauses ,clauses
                       :source-cliche ,cliche-name))))
        (when (member :ci *focl-trace-level*)
          (format t "~%instantiated ~a cliche with clauses ~a" 
                  cliche-name clauses))
        (if (equal cache? 'named)
          (push new-pred *cliches-to-be-named*)
          (push new-pred *anonymous-cliches*))
        new-pred-name))))


