
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

;;;_______________________________________
;;;   Parameters
;;;_______________________________________

(defconstant *pi* 3.414213562)

(defconstant *size-box-inner-dimension* 13)
(defconstant *scroll-bar-width* 15)
(defparameter *scroll-bar-point* #@(15 15))

(defconstant *graph-boarder* 20)
(defconstant *graph-start-h* 20)
(defconstant *graph-start-v* 20)
(defconstant *and-arc-ratio* .75)
(defvar *horizontal-v-gap* 2)

(defparameter *default-node-selection-constraint* nil)
(defparameter *default-edit-kind* nil)
(defparameter *default-font* '("monaco" 9 :srcor :plain))
(defparameter *default-orientation* :horizontal)

;;;_______________________________________
;;;   Classes
;;;_______________________________________

(defclass GRAPH-WINDOW (window)
  ((kind :initarg :kind :initform :graph :accessor kind)
   (edit-kind :initarg :edit-kind :initform *default-edit-kind* :accessor edit-kind)
   (controller :initarg :controller :initform nil :accessor controller)
   ))

(defclass GRAPH-SCROLLER (view)
  ())

(defclass GRAPH-VIEW (view)
  ((view-left :initarg :view-left :initform 0 :accessor view-left)
   (view-top :initarg :view-top :initform 0 :accessor view-top)
   (used-cells :initarg :used-cells :initform nil :accessor used-cells)
   (free-cells :initarg :free-cells :initform nil :accessor free-cells)
   (graph :initarg :graph :initform nil :accessor graph)
   (root :initarg :root :initform nil :accessor root)
   (graph-left :initarg :graph-left :initform *graph-start-h* :accessor graph-left)
   (graph-top :initarg :graph-top :initform *graph-start-v* :accessor graph-top)
   (graph-right :initarg :graph-right :initform *graph-start-h*  :accessor graph-right)
   (graph-bottom :initarg :graph-bottom :initform *graph-start-v* :accessor graph-bottom)
   (orientation :initarg :orientation :initform *default-orientation* :accessor orientation)
   (expand :initarg :expand :initform *default-expand* :accessor expand)
   (expand-depth :initarg :expand-depth :initform *default-expand-depth* :accessor expand-depth)
   (cell-height :initarg :cell-height :initform 20 :accessor cell-height)
   (cell-text-offset :initarg :cell-text-offset :initform nil :accessor cell-text-offset)
   (corner :initarg :corner :initform 2 :accessor corner)
   (node-selection-constraint :initarg :node-selection-constraint :initform *default-node-selection-constraint* :accessor node-selection-constraint)
   (last-node-selected :initarg :last-node-selected :initform nil :accessor last-node-selected)
   ))

;;;_______________________________________
;;;   GRAPH-WINDOW Methods
;;;_______________________________________

;;;_______________________________________
;;;  component accessors

(defun graph-scroller (window) (when window (view-named :graph-scroller window)))
(defun graph-view (window) (when window (view-named :graph-view (view-named :graph-scroller window))))
(defun window-nodes (window) (when window (graph-used-nodes (graph (graph-view window)))))
(defun window-graph (window) (when window (graph (graph-view window))))
(defun graph-window (view) (when view (view-container (view-container view))))


;;;_______________________________________
;;;   initialize-instance

(defmethod initialize-instance ((window graph-window) &rest initargs)
  (setf (getf initargs :color-p) t)
  (apply #'call-next-method window initargs )
  (set-view-scroll-position window 0 0)
  (set-view-font window *default-font*)
  (let* ((window-size-h (point-h (view-size window)))
         (window-size-v (point-v (view-size window))))
    (add-subviews window
                  (make-instance 'graph-scroller
                    :view-position #@(0 0)
                    :view-size (make-point window-size-h window-size-v)
                    :view-nick-name :graph-scroller)))
  (set-font-dependent-attributes (graph-view window)))

;;;_______________________________________
;;;  set-view-size

(defmethod set-view-size ((window graph-window) h &optional (v nil))
  (call-next-method)
  (reset-view-size (graph-scroller window) h v))

;;;_______________________________________
;;;   resize-window

(defmethod resize-window ((window graph-window) &optional (h nil) (v nil))
  (general-resize-window window h v 0 0))

(defun general-resize-window (window h v extra-h extra-v)
  (let* ((view (graph-view window))
         (real-h (if h h (min (+ (- (graph-right view) (graph-left view))
                                 *graph-boarder* *graph-boarder* *scroll-bar-width* extra-h)
                              (- *screen-width* 6))))
         (real-v (if v v (min (+ (- (graph-bottom view) (graph-top view))
                                 *graph-boarder* *graph-boarder* *scroll-bar-width* extra-v)
                              (- *screen-height* 44)))))
    (set-view-size window real-h real-v)))

;;;_______________________________________
;;;   grow-window-if-needed

(defmethod grow-window-if-needed ((window graph-window))
  (general-grow-window-if-needed window 0 0))

(defun general-grow-window-if-needed (window extra-h extra-v)
  (let* ((view (graph-view window))
         (window-size (view-size window))
         (current-h (point-h window-size))
         (current-v (point-v window-size))
         (new-h (max current-h
                     (min (+ (- (graph-right view) (graph-left view))
                             *graph-boarder* *graph-boarder* *scroll-bar-width* extra-h)
                          (- *screen-width* 6))))
         (new-v (max current-v
                     (min (+ (- (graph-bottom view) (graph-top view))
                             *graph-boarder* *graph-boarder* *scroll-bar-width* extra-v)
                          (- *screen-height* 44)))))
    (when (or (> new-h current-h)
              (> new-v current-v))
      (set-view-size window new-h new-v)
      t)))

;;;_______________________________________
;;;   window-zoom-event-handler

(defmethod window-zoom-event-handler ((window graph-window) message)
  (declare (ignore message))
  (call-next-method)
  (reset-view-size (graph-scroller window) (view-size window))
  )

;;;_______________________________________
;;; auto-position-window

(defun auto-position-window (graph-window &key (centered nil))
  (let* ((window-size (view-size graph-window))
         (window-position (view-position graph-window))
         (old-h (point-h window-position))
         (old-v (point-v window-position))
         (view-size-h (point-h window-size))
         (view-size-v (point-v window-size)))
    (if centered
      (set-view-position graph-window
                         (max 3 (round (- *screen-width* view-size-h) 2))
                         (max 41 (round (- *screen-height* view-size-v) 2)))
      (set-view-position graph-window
                         (if (> (+ old-h view-size-h 3) *screen-width*)
                           (max 3 (- *screen-width* view-size-h 3))
                           old-h)
                         (if (> (+ old-v view-size-v 3) *screen-height*)
                           (max 41 (- *screen-height* view-size-v 3))
                           old-v)))))


;;;_______________________________________
;;;   view-key-event-handler

(defmethod view-key-event-handler ((window graph-window) char)
  ;  (format t "CHARACTER : ~A      CHAR-CODE :~A~%" char  (char-code char))
  (let ((view (graph-view window)))
    (case (char-code char)
      ((69 101) (if (fboundp 'edit-last-node-selected)          ;;; E e
                  (edit-last-node-selected view)))
      ((71 103) (redisplay-last-node-selected view))            ;;; G g
      ((72 104) (hide-antecedents-of-selected-cells view))      ;;; H h
      ((73 105) (inspect-last-node-selected view))              ;;; I i
      ((82 114) (force-graph-redraw view))                      ;;; R r
      ((83 115) (show-antecedents-of-selected-cells view))      ;;; S s
      ((84 116) (show-tuples-last-node-selected view))          ;;; T t
      (others nil))))

;;;_______________________________________
;;;  window-setup

(defun window-setup (window)
  (when (graph-scroller window)
    (let* ((view (graph-view window))
           (title (window-title window))
           (font (view-font window))
           (orientation (orientation view))
           (expand (expand view))
           (expand-depth (expand-depth view))
           (resize-graph? nil)
           (reposition-cells? nil))
      (catch-cancel 
        (multiple-value-bind (new-font new-orientation new-expand new-expand-depth)
                             (values-list (get-window-setup title font orientation expand expand-depth))
          (when (user-monitor-p *user-monitor*)
            (incf (user-monitor-window-setup *user-monitor*)))
          (without-interrupts
           (with-cursor *watch-cursor*
             (unless (equal new-font font)
               (set-view-font window '("monaco" 9 :plain))
               (set-view-font window new-font)
               (set-font-dependent-attributes view)
               (size-all-cells view)
               (setf resize-graph? t
                     reposition-cells? t))
             (when (or new-expand new-expand-depth)
               (setf (expand view) new-expand
                     (expand-depth view) new-expand-depth)
               (let* ((class (class-of window))
                      (graph (window-graph window))
                      (base (graph-base graph))
                      (source (node-state base))
                      (root (graph-root graph))
                      (literal (cons (r-name (node-r-struct base)) (node-vars base))))
                 (cond ((string-equal (window-title window) "Learned Description"))
                       ((or (eq class (find-class 'rule-edit-window))
                            (eq class (find-class 'theory-edit-window))
                            (eq class (find-class 'graph-window)))
                        (dispose-node (node-antecedents root) graph t)
                        (setf (node-antecedents root) nil
                              (node-antecedents root) (list (list (connect-literal graph root literal nil source new-expand new-expand-depth)))
                              reposition-cells? t)
                        (display-tree-cells view)))))
             (unless (eql new-orientation orientation)
               (setf (orientation view) new-orientation
                     reposition-cells? t))
             (when reposition-cells?
               (position-cells view))
             (when (or resize-graph? reposition-cells?)
               (resize-window window)
               (auto-position-window window)
               (with-focused-view view
                 (re-position-graph view :centered t)))
             (invalidate-view window t))))))))


;;;_______________________________________
;;; copy

(defmethod copy ((window graph-window))
  (copy-graph-to window))


;;;_______________________________________
;;;   window-close

(defmethod window-close ((window graph-window))
  (dispose-graph-view (graph-view window))
  (call-next-method))


;;;_______________________________________
;;;   GRAPH-SCROLLER Methods
;;;_______________________________________

;;;_______________________________________
;;;   initialize-instance

(defmethod initialize-instance ((view graph-scroller) &rest initargs)
  (apply #'call-next-method view initargs )
  (set-view-scroll-position view 0 0)
  (let* ((scrolling-view-size-h (point-h (view-size view)))
         (scrolling-view-size-v (point-v (view-size view)))
         (view-size-h (- scrolling-view-size-h *scroll-bar-width*))
         (view-size-v (- scrolling-view-size-v *scroll-bar-width*)))
    (add-subviews
     view
     (make-instance 'scroll-bar-dialog-item
       :view-position (make-point -1 view-size-v)
       :direction :horizontal
       :length (+ view-size-h 2)
       :scroll-size 20
       :page-size (round view-size-h 2)
       :view-nick-name :h-scroll-bar
       :dialog-item-action
       #'(lambda (item) (let ((view (find-named-sibling item :graph-view)))
                          (set-view-left view (+ (- (graph-left view) *graph-boarder*) (scroll-bar-setting item)))
                          (reset-scroll-bars (view-container item)))))
     (make-instance 'scroll-bar-dialog-item
       :view-position (make-point view-size-h -1)
       :direction :vertical
       :length (+ view-size-v 2)
       :scroll-size 20
       :page-size (round view-size-v 2)
       :view-nick-name :v-scroll-bar
       :dialog-item-action
       #'(lambda (item) (let ((view (find-named-sibling item :graph-view)))
                          (set-view-top view (+ (- (graph-top view) *graph-boarder*) (scroll-bar-setting item)))
                          (reset-scroll-bars (view-container item)))))
     (make-instance 'graph-view
       :view-position #@(0 0)
       :view-size (make-point view-size-h view-size-v)
       :view-nick-name :graph-view)) ))


;;;_______________________________________
;;;  reset-view-size

(defmethod reset-view-size ((view graph-scroller) h &optional (v nil))
  (if (null v)
    (setf v (point-v h)
          h (point-h h)))
  (let ((graph-view (view-named :graph-view view))
        (h-scroll-bar (view-named :h-scroll-bar view))
        (v-scroll-bar (view-named :v-scroll-bar view))
        (graph-view-size-h (- h *scroll-bar-width*))
        (graph-view-size-v (- v *scroll-bar-width*)))
    (set-scroll-bar-length h-scroll-bar (+ graph-view-size-h 2))
    (set-scroll-bar-length v-scroll-bar (+ graph-view-size-v 2))
    (set-view-position h-scroll-bar -1 graph-view-size-v)
    (set-view-position v-scroll-bar graph-view-size-h -1)
    (set-view-size graph-view graph-view-size-h graph-view-size-v)
    (set-view-size view h v)
    (re-position-graph graph-view)
    ))

;;;_______________________________________
;;;  reset-scroll-bars

(defmethod reset-scroll-bars ((view graph-scroller))
  (let* ((graph-view (view-named :graph-view view))
         (v-left (view-left graph-view))
         (v-top (view-top graph-view))
         (v-right (+ v-left (point-h (view-size graph-view))))
         (v-bottom (+ v-top (point-v (view-size graph-view))))
         (g-left (- (graph-left graph-view) *graph-boarder*))
         (g-top (- (graph-top graph-view) *graph-boarder*))
         (g-right (+ (graph-right graph-view) *graph-boarder*))
         (g-bottom (+ (graph-bottom graph-view) *graph-boarder*))
         (v1 (max (- v-top g-top) 0))
         (v2 (max (- g-bottom v-bottom) (- g-top v-top) 0))
         (h1 (max (- v-left g-left) 0))
         (h2 (max (- g-right v-right )  (- g-left v-left) 0))
         (h-scroll-bar (view-named :h-scroll-bar view))
         (v-scroll-bar (view-named :v-scroll-bar view)))

    (setf (scroll-bar-page-size h-scroll-bar) (round (- v-right v-left) 2))
    (set-scroll-bar-min h-scroll-bar 0)
    (set-scroll-bar-max h-scroll-bar (+ h1 h2))
    (set-scroll-bar-setting h-scroll-bar h1)
    (if (and (= 0 h1) (= 0 h2))
      (dialog-item-disable h-scroll-bar)
      (dialog-item-enable h-scroll-bar))
    
    (setf (scroll-bar-page-size v-scroll-bar) (round (- v-bottom v-top) 2))
    (set-scroll-bar-min v-scroll-bar 0)
    (set-scroll-bar-max v-scroll-bar (+ v1 v2))
    (set-scroll-bar-setting v-scroll-bar v1)
    (if (and (= 0 v1) (= 0 v2))
      (dialog-item-disable v-scroll-bar)
      (dialog-item-enable v-scroll-bar))
    ))


;;;_______________________________________
;;;   GRAPH-VIEW Methods
;;;_______________________________________

;;;_______________________________________
;;; get-cell

(defmethod get-cell ((view graph-view)
                        &key (left 0)
                             (top 0)
                             (text-h 4)
                             (text-v 8)
                             (text "")
                             (external-text nil)
                             (on-screen? nil)
                             (hidden? nil)
                             (node nil))
  (let (cell)
    (if (free-cells view)
      (setf cell (first (free-cells view))
            (free-cells view) (rest (free-cells view)))
      (setf cell (make-cell)))
    (setf (node-cells node) (push cell (node-cells node))
          (used-cells view) (push cell (used-cells view))
          (cell-left cell) left
          (cell-top cell) top
          (cell-text-h cell) text-h
          (cell-text-v cell) text-v
          (cell-text cell) text
          (cell-external-text cell) external-text
          (cell-on-screen? cell) on-screen?
          (cell-hidden? cell) hidden?
          (cell-node cell) node
          (cell-view cell) view)
    (size-cell cell)
    cell))

;;;_______________________________________
;;; free-cell

(defun free-cell (cell)
  (when (cell-p cell)
    (let ((node (cell-node cell))
          (view (cell-view cell)))
      (setf (free-cells view) (push cell (free-cells view))
            (used-cells view) (delete cell (used-cells view))
            (node-cells node) (delete cell (node-cells node))
            (cell-node cell) nil
            (cell-external-text cell) nil
            (cell-text cell) nil)))
    cell)

;;;_______________________________________
;;; free-node-cell 

(defun free-node-cells (node-or-collection view &optional (recursive nil))
  (cond ((node-p node-or-collection)
         (free-cell (node-cell view node-or-collection))
         (when recursive
           (free-node-cells (node-antecedents node-or-collection) view recursive)))
        ((consp node-or-collection)
         (free-node-cells (first node-or-collection) view recursive)
         (free-node-cells (rest node-or-collection) view recursive))))

;;;_______________________________________
;;; dispose-view-cells

(defmethod dispose-view-cells ((view graph-view))
  (labels
    ((dispose-cell (cell node)
       (when (node-p node)
         (setf (node-cells node) (delete cell (node-cells node))))
       (setf (cell-node cell) nil
             (cell-external-text cell) nil
             (cell-text cell) nil
             (cell-view cell) nil)))
    (dolist (cell (free-cells view))
      (dispose-cell cell (cell-node cell)))
    (dolist (cell (used-cells view))
      (dispose-cell cell (cell-node cell)))
    (setf (free-cells view) nil
          (used-cells view) nil)))

;;;_______________________________________
;;; dispose-graph-view

(defmethod dispose-graph-view ((view graph-view))
  (dispose-view-cells view)
  (let ((graph (graph view)))
    (when (graph-p graph)
      (setf (graph-views graph) (delete view (graph-views graph)))
      (unless (or (graph-views graph)
                  (graph-permanent? graph))
        (dispose-graph graph)))
  (setf (last-node-selected view) nil
        (node-selection-constraint view) nil
        (view-left view) nil
        (view-top view) nil
        (root view) nil
        (graph-left view) nil
        (graph-top view) nil
        (graph-right view) nil
        (graph-bottom view) nil
        (orientation view) nil
        (expand view) nil
        (expand-depth view) nil
        (cell-height view) nil
        (cell-text-offset view) nil
        (corner view) nil)))


;;;_______________________________________
;;; export-graph-picture

(defmethod export-graph-picture ((view graph-view) &optional (picture-color-level :bw))
  (let* ((color-level *color-level*)
         (view-left (view-left view))
         (view-top (view-top view))
         (port-h (- (rref (wptr view) :WindowRecord.portRect.right)
                    (rref (wptr view) :WindowRecord.portRect.left)))
         (port-v (- (rref (wptr view) :WindowRecord.portRect.bottom)
                    (rref (wptr view) :WindowRecord.portRect.top)))
         (g-left (- (graph-left view) 1))
         (g-top (- (graph-top view) 1))
         (frame-h (+ (- (graph-right view) (graph-left view)) 2))
         (frame-v (+ (- (graph-bottom view) (graph-top view)) 2))
         (picture nil))
    (setf *color-level* picture-color-level)
    (if (or (> frame-h 32000) (> frame-v 32000))
      (message-dialog "The graph is too large to be exported." :position :centered)
      (without-interrupts
       (set-cursor *watch-cursor*)
       (rlet ((picture-frame :rect 
                             :left 0
                             :top 0
                             :right frame-h
                             :bottom frame-v))
         (with-focused-view view
           (#_portsize frame-h frame-v)
           (#_cliprect picture-frame)
           (setf (view-left view) g-left
                 (view-top view) g-top
                 picture (#_openpicture picture-frame))
           (draw-graph view :fast nil :clipped nil)
           (#_closepicture)
           (setf (view-left view) view-left
                 (view-top view) view-top)
           (#_portsize port-h port-v)))))
    (setf *color-level* color-level)
    picture))


;;;_______________________________________
;;;   position-graph

(defmethod position-graph ((view graph-view) &key (centered nil))
  (without-interrupts
   (let* ((g-left (- (graph-left view) *graph-boarder*))
          (g-top (- (graph-top view) *graph-boarder*))
          (g-right (+ (graph-right view) *graph-boarder*))
          (g-bottom (+ (graph-bottom view) *graph-boarder*))
          (view-size (view-size view))
          (view-left (view-left view))
          (view-top (view-top view))
          (view-h (point-h view-size))
          (view-v (point-v view-size)))
     (if centered
       (let ((center-cell (first-visible-cell view (root view)))
             (g-h (- g-right g-left))
             (g-v (- g-bottom g-top)))
         (setf (view-left view)
               (if (> g-h view-h)
                 (case (orientation view)
                   (:horizontal g-left)
                   (:diagonal g-left)
                   (:vertical (max (- (cell-center-h center-cell) (round view-h 2)) g-left)))
                 g-left)
               
               (view-top view)
               (if (> g-v view-v)
                 (case (orientation view)
                   (:horizontal (max (- (cell-center-v center-cell) (round view-v 2)) g-top))
                   (:vertical g-top)
                   (:diagonal g-top))
                 g-top)))
       
       (setf (view-left view)
             (if (and (< g-left view-left) (< g-right (+ view-left view-h)))
               (- view-left (min (- view-left g-left ) (- (+ view-left view-h) g-right )))
               view-left)
             
             (view-top view)
             (if (and (< g-top view-top) (< g-bottom (+ view-top view-v)))
               (- view-top (min (- view-top g-top) (- (+ view-top view-v) g-bottom)))
               view-top)
             ))
     (force-graph-redraw view)
     (reset-scroll-bars (view-container view))
     )))

;;;_______________________________________
;;;   re-position-graph

(defmethod re-position-graph ((view graph-view) &key (centered nil))
  (declare (ignore centered))
  (force-graph-redraw view)
  (reset-scroll-bars (view-container view)))

;;;_______________________________________
;;;  set-view-left

(defmethod set-view-left ((view graph-view) left)
  (without-interrupts
   (let ((delta-h (- (view-left view) left))
         (reg (#_newrgn)))
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size view)))
       (#_scrollrect view-rect delta-h 0 reg)
       (#_invalrgn reg)
       (#_disposergn reg)
       (setf (view-left view) left)
       (#_beginupdate (wptr view))
       (draw-graph view :fast nil :clipped t)
       (#_endupdate (wptr view))
       (#_validrect (rref (wptr view) :WindowRecord.portRect))))))

;;;_______________________________________
;;;  set-view-top

(defmethod set-view-top ((view graph-view) top)
  (without-interrupts
   (let ((delta-v (- (view-top view) top))
         (reg (#_newrgn)))
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size view)))
       (#_scrollrect view-rect 0 delta-v reg)
       (#_invalrgn reg)
       (#_disposergn reg)
       (setf (view-top view) top)
       (#_beginupdate (wptr view))
       (draw-graph view :fast nil :clipped t)
       (#_endupdate (wptr view))
       (#_validrect (rref (wptr view) :WindowRecord.portRect))))))

;;;_______________________________________
;;;   set-view-left-top

(defmethod set-view-left-top ((view graph-view) left top)
  (without-interrupts
   (let ((delta-h (- (view-left view) left))
         (delta-v (- (view-top view) top))
         (reg (#_newrgn)))
     (setf (view-left view) left
           (view-top view) top)
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size view)))
       (cond ((or (> (abs delta-h) (point-h (view-size view)))
                  (> (abs delta-v) (point-v (view-size view))))
              (#_eraserect view-rect)
              (#_invalrect view-rect))
             (t
              (#_scrollrect view-rect delta-h delta-v reg)
              (#_invalrgn reg)
              (#_disposergn reg)
              (#_beginupdate (wptr view))
              (draw-graph view :fast nil :clipped t)
              (#_endupdate (wptr view))
              (#_validrect (rref (wptr view) :WindowRecord.portRect))))))))

;;;_______________________________________
;;;  view-draw-contents

(defmethod view-draw-contents ((view graph-view))
  (without-interrupts
   (with-focused-view view
     (draw-graph view :fast nil :clipped t))))

;;;_______________________________________
;;;  force-graph-redraw

(defmethod force-graph-redraw ((view graph-view))
    (with-focused-view view
      (invalidate-view view t)))

;;;_______________________________________
;;;  draw-graph

(defmethod draw-graph ((view graph-view) &key (fast t) (clipped t))
  (without-interrupts
   (with-focused-view view
     (set-view-font (graph-window view) (view-font view))
     (let ((root (root view)))
       (when root
         (if fast
           (fast-draw-cell-and-children view root)
           (draw-cell-and-children view root :clipped clipped)))))))

;;;_______________________________________
;;;  draw-cell-and-children

(defmethod draw-cell-and-children ((view graph-view) node &key (clipped t))
  (let ((cell (node-cell view node)))
    (when cell
      (draw-antecedent-connectors cell :clipped clipped)
      (draw-cell cell :clipped clipped)
      (dolist (conjunction (node-antecedents node))
        (dolist (n conjunction)
          (draw-cell-and-children view n :clipped clipped))))))

;;;_______________________________________
;;;  fast-draw-cell-and-children

(defmethod fast-draw-cell-and-children ((view graph-view) node &key (clipped t))
  (let ((cell (node-cell view node)))
    (when cell
      (draw-antecedent-connectors cell :clipped clipped)
      (if (or (cell-on-screen? cell)
              (node-selected? node))
        (draw-cell cell :clipped clipped))
      (dolist (conjunction (node-antecedents node))
        (dolist (n conjunction)
          (fast-draw-cell-and-children view n :clipped clipped))))))

;;;_______________________________________
;;;  set-font-dependent-attributes

(defmethod set-font-dependent-attributes ((view graph-view))
  (let ((h-text-gap 4) (v-text-gap 1) (max-v 2) (max-h 2))
    (multiple-value-bind (ascent descent widmax leading)
                         (font-info (view-font view))
      (declare (ignore widmax leading))
      (setf (cell-height view) (+ (* 2 (+ max-v v-text-gap)) ascent descent)
            (cell-text-offset view) (make-point (+ max-h h-text-gap) (+ max-v v-text-gap ascent))
            (corner view) (round (cell-height view) 1.5)))))

;;;_______________________________________
;;;  find-cell-clicked-on

(defun find-cell-clicked-on (cell h v &optional (cell-clicked-on nil))
  (when cell
    (if (and (cell-on-screen? cell)
             (not (cell-hidden? cell))
             (< (cell-left cell) h)
             (< h (cell-right cell))
             (< (cell-top cell) v)
             (< v (cell-bottom cell)))
      (setf cell-clicked-on cell))
    (dolist (conjunction (node-antecedents (cell-node cell)))
      (dolist (node conjunction)
        (setf cell-clicked-on (find-cell-clicked-on (node-cell (cell-view cell) node)
                                                    h v cell-clicked-on)))))
  cell-clicked-on)

;;;_______________________________________
;;; set-node-as-last-selected

(defun set-node-as-last-selected (node)
  (mapc #'(lambda (cell) (setf (last-node-selected (cell-view cell)) node)) (node-cells node)))

;;;_______________________________________
;;; clear-last-selected-node-from-views-of

(defun clear-last-selected-node-from-views-of (graph)
  (map-windows #'(lambda (window)
                   (let ((view (graph-view window)))
                     (if (eq (graph view) graph)
                       (setf (last-node-selected view) nil))))
               :class 'graph-window))
                     

;;;_______________________________________
;;;  view-click-event-handler

(let ((number-of-double-clicks 0))
  
  (defmethod view-click-event-handler ((view graph-view) where)
    (let* ((cells-where-moved nil)
           (cell-selected (find-cell-clicked-on (node-cell view (root view))
                                                (+ (point-h where) (view-left view))
                                                (+ (point-v where) (view-top view))))
           (node-selected (if cell-selected (cell-node cell-selected) nil)))
      (if (double-click-p)
        (incf number-of-double-clicks)
        (setf number-of-double-clicks 0))
      (case (node-selection-constraint view)
        ((nil :no-drag)
         (if node-selected
           (case number-of-double-clicks
             (0 (cond ((shift-key-p)
                       (cond ((option-key-p)
                              (cond ((node-selected? node-selected)
                                     (deselect-node node-selected t))
                                    (t
                                     (select-node node-selected t)
                                     (setf cells-where-moved (drag-selected-cells view where)))))
                             (t
                              (cond ((node-selected? node-selected)
                                     (deselect-node node-selected))
                                    (t
                                     (select-node node-selected)
                                     (setf cells-where-moved (drag-selected-cells view where)))))))
                      ((option-key-p)
                       (cond ((node-selected? node-selected)
                              (select-node node-selected t)
                              (setf cells-where-moved (drag-selected-cells view where)))
                             (t
                              (deselect-node (graph-root (graph view)) t)
                              (select-node node-selected t)
                              (setf cells-where-moved (drag-selected-cells view where)))))
                      (t 
                       (cond ((node-selected? node-selected)
                              (setf cells-where-moved (drag-selected-cells view where)))
                             (t
                              (deselect-node (graph-root (graph view)) t)
                              (select-node node-selected)
                              (setf cells-where-moved (drag-selected-cells view where)))))))
             (1 (cond ((shift-key-p)
                       (cond ((option-key-p) 
                              (cond ((node-selected? node-selected)
                                     (select-node (conjunction-containing-node node-selected) t)
                                     (setf cells-where-moved (drag-selected-cells view where)))
                                    (t
                                     (deselect-node (conjunction-containing-node node-selected) t))))
                             (t
                              (cond ((node-selected? node-selected)
                                     (select-node (conjunction-containing-node node-selected))
                                     (setf cells-where-moved (drag-selected-cells view where)))
                                    (t
                                     (dolist (node (conjunction-containing-node node-selected))
                                       (deselect-node node)))))))
                      ((option-key-p)
                       (select-node (conjunction-containing-node node-selected) t)
                       (setf cells-where-moved (drag-selected-cells view where)))
                      (t
                       (select-node (conjunction-containing-node node-selected))
                       (setf cells-where-moved (drag-selected-cells view where)))))
             (otherwise 
              (cond ((shift-key-p)
                     (cond ((option-key-p) 
                            (cond ((node-selected? node-selected)
                                   (select-node (node-antecedents (node-consequent node-selected)) t)
                                   (setf cells-where-moved (drag-selected-cells view where)))
                                  (t
                                   (deselect-node (node-antecedents node-selected) t))))
                           (t
                            (cond ((node-selected? node-selected)
                                   (select-node (node-antecedents (node-consequent node-selected)))
                                   (setf cells-where-moved (drag-selected-cells view where)))
                                  (t
                                   (dolist (node (node-antecedents (node-consequent node-selected)))
                                     (deselect-node node)))))))
                    ((option-key-p)
                     (select-node (node-antecedents (node-consequent node-selected)) t)
                     (setf cells-where-moved (drag-selected-cells view where)))
                    (t
                     (select-node (node-antecedents (node-consequent node-selected)))
                     (setf cells-where-moved (drag-selected-cells view where)))))
             )
           (deselect-node (graph-root (graph view)) t))
         (clear-last-selected-node-from-views-of (graph view))
         (when (and (node-p node-selected) (node-selected? node-selected))
           (set-node-as-last-selected node-selected)))
        
        (:no-selection nil)
        )
      
      (let ((graph-window (graph-window view)))
        (when (ccl::inherit-from-p graph-window 'edit-window)
          (configure-controls graph-window)
          (when cells-where-moved
            (reorder-nodes graph-window))))
      
      (call-next-method)
      (if cells-where-moved :cells-where-moved nil)))
  )

;;;_______________________________________
;;; draw-cell                                                      

(defun draw-cell (cell &key (clipped t))
  (let* ((view (cell-view cell))
         (v-left (view-left view))
         (v-top (view-top view))
         (v-size (view-size view)))
    (cond ((cell-hidden? cell) (setf (cell-on-screen? cell) nil))
          (clipped (if (and
                        (> (cell-bottom cell) v-top)
                        (< (cell-top cell) (+ v-top (point-v v-size)))
                        (> (cell-right cell) v-left)
                        (< (cell-left cell) (+ v-left (point-h v-size))))
                     (setf (cell-on-screen? cell) (really-draw-cell cell))
                     (setf (cell-on-screen? cell) nil)))
          (t (really-draw-cell cell)))))

;;;_______________________________________
;;; focus-and-draw-cell

(defun focus-and-draw-cell (cell)
  (let ((view (cell-view cell)))
    (with-focused-view view
      (set-view-font (graph-window view) (view-font view))
      (really-draw-cell cell))))                                       

;;;_______________________________________
;;; really-draw-cell                                                      

(defun really-draw-cell (cell)
  (let* ((view (cell-view cell))
         (v-left (view-left view))
         (v-top (view-top view))
         (left (- (cell-left cell) v-left))
         (top (- (cell-top cell) v-top))
         (right (- (cell-right cell) v-left))
         (bottom (- (cell-bottom cell) v-top))
         (text-h (- (cell-text-h cell) v-left))
         (text-v (- (cell-text-v cell) v-top))
         (text (cell-text cell))
         (external-text (cell-external-text cell))
         (corner (corner view))
         (selected? (cell-selected? cell)))
    (multiple-value-bind (fill-pattern frame-color fill-color) (set-pen-given-state (cell-state cell))
      (when selected?
        (setf fill-pattern *black-pattern*
              fill-color *black-color*))
      (rlet ((rect :rect :left left :top top :right right :bottom bottom))
        (setf (cell-on-screen? cell) t)
        (case (cell-kind cell)
          (:cut (draw-cut-cell cell fill-pattern frame-color fill-color))
          ((:or :and :not) (draw-second-order-cell rect corner fill-pattern frame-color fill-color))
          (:extensional (draw-extensional-cell rect fill-pattern frame-color fill-color))
          ((:builtin :is := :fail) (draw-builtin-cell rect corner fill-pattern frame-color fill-color))
          ((:intensional :special :undefined)
           (draw-intensional-cell rect corner fill-pattern frame-color fill-color)
           (when (or (cell-recursive? cell)
                     (cell-unexpanded? cell))
             (#_pensize 1 1)
             (with-pstrs ((pascal-string ""))
               (if (eql (orientation view) :vertical)
                 (#_moveto (- (cell-center-h cell) v-left
                              (round (with-focused-view view 
                                       (#_StringWidth pascal-string)) 2)) (+ bottom 4))
                 (#_moveto (+ right 3) (+ (- (cell-center-v cell) v-top) 1)))
               (#_drawstring pascal-string))))
          (otherwise
           (multiple-value-setq (fill-pattern frame-color fill-color) (set-pen-given-state :undefined))
           (draw-undefined-cell rect fill-pattern frame-color fill-color)))
        
        (#_pensize 1 1)
        
        (when (cell-deleted? cell)
          (if selected?
            (#_penpat *white-pattern*)  
            (#_penpat *black-pattern*))
          (let* ((c (round corner 3))
                 (d-right (- right c 2))
                 (d-bottom (- bottom c 2))
                 (d-left (+ left c))
                 (d-top (+ top c)))
            (#_moveto d-left d-top)
            (#_lineto d-right d-bottom)
            (#_moveto d-left d-bottom)
            (#_lineto d-right d-top)) )
        
        (#_penpat *black-pattern*)
        
        (#_moveto text-h text-v)
        (if selected?
          (#_textmode 3)  ;; srcBic
          (#_textmode 1)) ;; scrOr
        (with-pstrs ((pascal-string text))
          (#_drawstring pascal-string))
        
        (#_textmode 1)
        (when external-text
          (if (or (cell-expanded? cell)
                  (eq (node-kind (node-consequent (cell-node cell))) :root))
            (draw-external-text external-text text-h (+ bottom 10) (- bottom top 6))
            (draw-external-text external-text (+ right 15) text-v (- bottom top 6))))
        
        t))))

;;;_______________________________________
;;;  draw-external-text                                                      

(defun draw-external-text (external-text h v v-step)
  (#_moveto h v)
  (dolist (string (partition-string external-text)) 
    (with-pstrs ((pascal-string string))
      (#_drawstring pascal-string)
      (#_moveto h (incf v v-step)))))

;;;_______________________________________
;;;  update-external-text                                                    

(defun update-external-text (cell new-external-text)
  (let* ((view (cell-view cell))
         (v-left (view-left view))
         (v-top (view-top view))
         (left (- (cell-left cell) v-left))
         (top (- (cell-top cell) v-top))
         (right (- (cell-right cell) v-left))
         (bottom (- (cell-bottom cell) v-top))
         (text-h (- (cell-text-h cell) v-left))
         (text-v (- (cell-text-v cell) v-top))
         (external-text (cell-external-text cell))
         (step (- (cell-bottom cell) (cell-top cell) 6))
         e-top e-bottom e-right e-left h v)
    (setf (cell-external-text cell) new-external-text)
    (with-focused-view view 
      (if (or (cell-expanded? cell)
              (node-root?  (node-consequent (cell-node cell))))
        (setf e-top (+ bottom 1)
              e-bottom e-top
              e-left text-h
              e-right right 
              h text-h
              v (+ bottom 10))
        (setf e-top top
              e-bottom top
              e-left (+ right 10)
              e-right left
              h (+ right 15)
              v text-v))
      
      (when external-text
        (dolist (string (partition-string external-text))
          (incf e-bottom step)
          (with-pstrs ((pascal-string string))
            (setf e-right (max e-right (+ e-left (#_stringwidth pascal-string)))))))
      
      (rlet ((rect :rect :left e-left :top e-top :right e-right :bottom e-bottom))
        (#_eraserect rect))
      (when new-external-text
        (draw-external-text new-external-text h v step)))))

;;;_______________________________________
;;;  partition-string                                                      

(defun partition-string (string)
  (let ((position (position #\return string)))
    (if position
      (cons (subseq string 0 position) (partition-string (subseq string (+ position 1))))
      (list string))))

;;;_______________________________________
;;;  set-color-level  (level :bw :color nil)

(defun set-color-level (level)
  (setf *color-level* level)
  (map-windows #'invalidate-view))

;;;_______________________________________
;;; set-pen-given-state
;;;
;;; values returned:  fill-pattern frame-color fill-color
                                            

(defun set-pen-given-state (cell-state)
  (case cell-state
    
    ((nil :ebl)
     (#_pensize 1 1)
     (#_penpat *black-pattern*)
     (values *white-pattern* *black-color* *white-color*))

    (:unoperationalized
     (#_pensize 2 2)
     (#_penpat (case *color-level* (:bw *gray-pattern*) (:color *black-pattern*) (otherwise *gray-pattern*)))
     (case *color-level*
       (:bw (values *light-gray-pattern* *black-color* *black-color*))
       (:color (values *black-pattern* *gray-color* *light-gray-color*))
       (otherwise (values *light-gray-pattern* *gray-color* *light-gray-color*)) ))

    (:extensional
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *red-color*))
       (otherwise (values *white-pattern* *red-color* *white-color*)) ))

    (:builtin
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *red-color*))
       (otherwise (values *white-pattern* *pink-color* *white-color*)) ))

    (:intensional
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *orange-color*))
       (otherwise (values *white-pattern* *orange-color* *white-color*)) ))
    
    (:cliche
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *light-blue-color*))
       (otherwise (values *white-pattern* *light-blue-color* *white-color*)) ))

    (:determinate
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *green-color*))
       (otherwise (values *white-pattern* *green-color* *white-color*)) ))
    
    ((:blank :blank-no-lines)
     (#_pensize 1 1)
     (#_penpat *white-pattern*)
     (values *white-pattern* *white-color* *white-color*))

    (:covers-neg
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *light-gray-pattern* *black-color* *black-color*))
       (:color (values *black-pattern* *red-color* *light-gray-color*))
       (otherwise (values *light-gray-pattern* *red-color* *dark-gray-color*)) ))

    (:covers-pos-and-neg
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *white-pattern* *red-color* *white-color*))
       (otherwise (values *white-pattern* *red-color* *white-color*)) ))

    (:covers-pos
     (#_pensize 1 1)
     (#_penpat *black-pattern*)
     (values *white-pattern* *black-color* *white-color*))

    (:covers-none
     (#_pensize 2 2)
     (#_penpat (case *color-level* (:bw *gray-pattern*) (:color *black-pattern*) (otherwise *gray-pattern*)))
     (case *color-level*
       (:bw (values *light-gray-pattern* *black-color* *black-color*))
       (:color (values *black-pattern* *gray-color* *light-gray-color*))
       (otherwise (values *light-gray-pattern* *dark-gray-color* *dark-gray-color*)) ))

    ((:undefined :error)
      (#_pensize 2 2)
      (#_penpat *black-pattern*)
      (case *color-level*
        (:bw (values *light-gray-pattern* *black-color* *black-color*))
        (:color (values *black-pattern* *red-color* *yellow-color*))
        (otherwise (values *light-gray-pattern* *red-color* *yellow-color*) )))

    (otherwise
     (#_pensize 2 2)
     (#_penpat *black-pattern*)
     (case *color-level*
       (:bw (values *white-pattern* *black-color* *white-color*))
       (:color (values *black-pattern* *black-color* *white-color*))
       (otherwise (values *white-pattern* *black-color* *white-color*)) ))
    ))

;;;_______________________________________
;;; draw-undefined-cell                                                     

(defun draw-undefined-cell (rect fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (#_fillrect rect fill-pattern))
  (with-fore-color frame-color
    (#_framerect rect)))
          
;;;_______________________________________
;;; draw-cut-cell                                                     

(defun draw-cut-cell (cell fill-pattern frame-color fill-color)
  (let* ((view (cell-view cell))
         (v-left (view-left view))
         (v-top (view-top view))
         (left (- (cell-left cell) v-left))
         (top (- (cell-top cell) v-top))
         (right (- (cell-right cell) v-left))
         (bottom (- (cell-bottom cell) v-top))
         (base (round (cell-height view) *pi*)))
    (#_pensize 1 1)
    (let ((poly (#_openpoly)))
      (#_moveto (+ left base) top)
      (#_lineto (- right base) top)
      (#_lineto right (+ top base))
      (#_lineto right (- bottom base))
      (#_lineto (- right base) bottom)
      (#_lineto (+ left base) bottom)
      (#_lineto left (- bottom base))
      (#_lineto left (+ top base))
      (#_lineto (+ left base) top)
      (#_closepoly)
      (with-fore-color fill-color
        (#_fillpoly poly fill-pattern))
      (with-fore-color frame-color
        (#_framepoly poly))
      (#_killpoly poly))
    
    (let ((poly (#_openpoly)))
      (#_moveto (+ left base 1) (+ top 2))
      (#_lineto (- right base 1) (+ top 2))
      (#_lineto (- right 2)(+ top base 1))
      (#_lineto (- right 2) (- bottom base 1))
      (#_lineto (- right base 1) (- bottom 2))
      (#_lineto (+ left base 1) (- bottom 2))
      (#_lineto (+ left 2) (- bottom base 1))
      (#_lineto (+ left 2) (+ top base 1))
      (#_lineto (+ left base 1) (+ top 2))
      (#_closepoly)
      (with-fore-color frame-color
        (#_framepoly poly))
      (#_killpoly poly))))

;;;_______________________________________
;;; draw-extensional-cell                                                     

(defun draw-extensional-cell (rect fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (#_fillrect rect fill-pattern))
  (with-fore-color frame-color
    (#_framerect rect)))

;;;_______________________________________
;;; draw-intensional-cell                                                     

(defun draw-intensional-cell (rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (#_fillroundrect rect corner corner fill-pattern))
  (with-fore-color frame-color
    (#_frameroundrect rect corner corner)))

;;;_______________________________________
;;; draw-builtin-cell                                                     

(defun draw-builtin-cell (rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (#_fillroundrect rect corner corner fill-pattern))
  (with-fore-color frame-color
    (#_frameroundrect rect corner corner)))

;;;_______________________________________
;;; draw-second-order-cell                                                     

(defun draw-second-order-cell (rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (#_fillroundrect rect corner corner fill-pattern))
  (with-fore-color frame-color
    (#_frameroundrect rect corner corner)))


;;;_______________________________________
;;; outline-cut-cell                                                    

(defun outline-cut-cell (cell)
  (let* ((view (cell-view cell))
         (v-left (view-left view))
         (v-top (view-top view))
         (left (- (cell-left cell) v-left))
         (top (- (cell-top cell) v-top))
         (right (- (cell-right cell) v-left))
         (bottom (- (cell-bottom cell) v-top))
         (base (round (cell-height view) *pi*)))
    (#_pensize 1 1)
    (with-focused-view view
      (let ((poly (#_openpoly)))
        (#_moveto (+ left base) top)
        (#_lineto (- right base) top)
        (#_lineto right (+ top base))
        (#_lineto right (- bottom base))
        (#_lineto (- right base) bottom)
        (#_lineto (+ left base) bottom)
        (#_lineto left (- bottom base))
        (#_lineto left (+ top base))
        (#_lineto (+ left base) top)
        (#_closepoly)
        (#_framepoly poly)
        (#_killpoly poly)))))

;;;_______________________________________
;;; draw-outline-of-selected-cells                                                     

(defmethod draw-outline-of-selected-cells ((view graph-view))
  (let ((corner (corner view))
        (v-left (view-left view))
        (v-top (view-top view)))
    (with-focused-view view
      (mapc #'(lambda (cell)
                (when (cell-selected? cell)
                  (rlet ((rect :rect
                               :left (- (cell-left cell) v-left)
                               :top (- (cell-top cell)  v-top)
                               :right (- (cell-right cell) v-left)
                               :bottom (- (cell-bottom cell) v-top)))
                    (case (cell-kind cell)
                      (:cut (outline-cut-cell cell))
                      ((:extensional nil) (#_framerect rect))
                      (otherwise (#_frameroundrect rect corner corner))))))
            (used-cells view)))))

;;;_______________________________________
;;; drag-selected-cells

(defmethod drag-selected-cells ((view graph-view) where)
  (unless (eql (node-selection-constraint view) :no-drag)
    (with-focused-view view
      (let ((view-size-h (point-h (view-size view)))
            (view-size-v (point-v (view-size view)))
            offset)
        
        (rlet ((limit-rect :rect :left 0 :top 0 :right view-size-h :bottom view-size-v)
               (slop-rect :rect :left -30 :top -30 :right (+ view-size-h 30) :bottom (+ view-size-v 30)))
          
          (let ((moving-region (#_newrgn)))
            (#_openrgn)
            (draw-outline-of-selected-cells view)
            (#_closergn moving-region)
            
            (#_penpat *black-pattern*)
            (#_pensize 1 1)
            
            (setf offset (#_draggrayrgn  moving-region where limit-rect slop-rect 0 (%null-ptr)))
            (#_disposergn moving-region)))
        
        (let ((h-offset (point-h offset))
              (v-offset (point-v offset)))
          (if (and (> h-offset -32768)
                   (or (> h-offset 2)
                       (< h-offset -2)
                       (> v-offset 2)
                       (< v-offset -2)))
            (progn
              (move-selected-cells view h-offset v-offset)
              (with-focused-view view
                (invalidate-view view t)
                (reset-graph-size view)
                (reset-scroll-bars (view-container view))
                t))
            nil))))))

;;;_______________________________________
;;; move-selected-cells

(defmethod move-selected-cells ((view graph-view) h-offset v-offset)
  (mapc #'(lambda (cell) 
            (when (cell-selected? cell) (offset-cell cell h-offset v-offset)))
        (used-cells view)))

;;;_______________________________________
;;; first-visible-cell

(defmethod first-visible-cell ((view graph-view) thing)
  (cond ((cell-p thing) (if (cell-hidden? thing)
                          (first-visible-cell view (node-antecedents (cell-node thing)))
                          thing))
        ((node-p thing) (first-visible-cell view (node-cell view thing)))
        ((consp thing) (or (first-visible-cell view (first thing))
                           (first-visible-cell view (rest thing))))
        (t nil)))

;;;_______________________________________
;;;  reset-graph-size

(defmethod reset-graph-size ((view graph-view) &optional (extra-width 0) (extra-height 0))
  (let ((cell (first-visible-cell view (root view))))
    (setf (graph-left view) (cell-left cell)
          (graph-top view) (cell-top cell)
          (graph-right view) (+ (cell-right cell) extra-width)
          (graph-bottom view) (+ (cell-bottom cell) extra-height))
    (set-graph-size view extra-width extra-height)))

;;;_______________________________________
;;; set-graph-size

(defmethod set-graph-size ((view graph-view) &optional (extra-width 0) (extra-height 0))
  (when (or (eq (class-of (graph-window view)) (find-class 'analyze-window))
            (eq (class-of (graph-window view)) (find-class 'learning-window)))
    (setf extra-width 70))
  (mapc #'(lambda (cell)
            (unless (cell-hidden? cell)
              (setf (graph-left view) (min (graph-left view) (cell-left cell))
                    (graph-top view) (min (graph-top view) (cell-top cell))
                    (graph-right view) (max (graph-right view) (+ (cell-right cell) extra-width))
                    (graph-bottom view) (max (graph-bottom view) (+ (cell-bottom cell) extra-height)))))
        (used-cells view)))

;;;_______________________________________
;;; inspect-last-node-selected

(defmethod inspect-last-node-selected ((view graph-view))
  (let ((node (last-node-selected view)))
    (when node (inspect node))))

;;;_______________________________________
;;;  draw-antecedent-connectors

(defun draw-antecedent-connectors (cell &key (clipped t))
  (when (and (not (cell-hidden? cell))
             (not (eq (cell-state cell) :blank-no-lines))
             (cell-expanded? cell))
    (let* ((node (cell-node cell))
           (view (cell-view cell))
           (view-left (view-left view))
           (view-top (view-top view))
           (out-h (- (cell-out-h cell) view-left))
           (out-v (- (cell-out-v cell) view-top))
           antecedent-cell in-h in-v arc-h last-arc-h arc-v last-arc-v state)
      (dolist (conjunction (node-antecedents node))
        (setf last-arc-h nil
              last-arc-v nil)
        (dolist (antecedent conjunction)
          (setf antecedent-cell (node-cell (cell-view cell) antecedent)
                state (node-state antecedent))
          (unless (cell-hidden? antecedent-cell)
            (setf in-h (- (cell-in-h antecedent-cell) view-left)
                  in-v (- (cell-in-v antecedent-cell) view-top)
                  arc-h (round (+ (* (- in-h out-h) *and-arc-ratio*) out-h))
                  arc-v (round (+ (* (- in-v out-v) *and-arc-ratio*) out-v)))
            (draw-line view out-h out-v in-h in-v :state state :clipped clipped)
            (when (and last-arc-h last-arc-v)
              (draw-line view last-arc-h last-arc-v arc-h arc-v :state state :clipped clipped))
            (setf last-arc-h arc-h
                  last-arc-v arc-v)))))))

;;;_______________________________________
;;;  move-first-point-on-screen

(defun  move-first-point-on-screen (x1 y1 x2 y2 left top right bottom)
  (let ((delta-x (- x2 x1))
        (delta-y (- y2 y1)))

    (cond ((= delta-x 0)
           (values x1 (if (< y1 y2) top bottom)))
          ((= delta-y 0)
           (values (if (< x1 x2) left right) y1))
          (t (let ((left-intercept nil)
                   (top-intercept nil)
                   (right-intercept nil)
                   (bottom-intercept nil))

               (if (< x1 x2)
                 (setf left-intercept (+ (round (* delta-y (- left x1)) delta-x) y1))
                 (setf right-intercept (+ (round (* delta-y (- right x1)) delta-x) y1))
                 )

               (if (< y1 y2)
                 (setf top-intercept (+ (round (* delta-x (- top y1)) delta-y) x1))
                 (setf bottom-intercept (+ (round (* delta-x (- bottom y1)) delta-y) x1))
                 )

               (cond
                ((and left-intercept (<= top left-intercept) (<= left-intercept bottom))
                 (values left left-intercept))
                ((and top-intercept (< left top-intercept) (< top-intercept right))
                 (values top-intercept top))
                ((and right-intercept (<= top right-intercept) (<= right-intercept bottom))
                 (values right right-intercept))
                ((and bottom-intercept (< left bottom-intercept) (< bottom-intercept right))
                 (values bottom-intercept bottom))))))))

;;;_______________________________________
;;;  draw-line

(defmethod draw-line ((v view) x1 y1 x2 y2 &key (state nil) (clipped t))
  (if clipped
    (let* ((left 0)
           (top 0)
           (right (+ left (point-h (view-size v))))
           (bottom (+ top (point-v (view-size v))))
           (x1-left (< x1 left))
           (x2-left (< x2 left))
           (x1-right (> x1 right))
           (x2-right (> x2 right))
           (y1-top (< y1 top))
           (y2-top (< y2 top))
           (y1-bottom (> y1 bottom))
           (y2-bottom (> y2 bottom))
           (P1-on-screen (not (or x1-left x1-right y1-top y1-bottom)))
           (P2-on-screen (not (or x2-left x2-right y2-top y2-bottom))))
      
      (unless (or (and x1-left x2-left)
                  (and x1-right x2-right)
                  (and y1-top y2-top)
                  (and y1-bottom y2-bottom))
        (unless P1-on-screen
          (multiple-value-setq (x1 y1)
            (move-first-point-on-screen x1 y1 x2 y2 left top right bottom)))
        (when (and x1 y1)
          (unless (or P2-on-screen)
            (multiple-value-setq (x2 y2)
              (move-first-point-on-screen x2 y2 x1 y1 left top right bottom)))
          (when (and x2 y2)
            (really-draw-line x1 y1 x2 y2 state)))))
    (really-draw-line x1 y1 x2 y2 state)))
                 

;;;_______________________________________
;;;  really-draw-line

(defun really-draw-line (x1 y1 x2 y2 &optional (state nil))
  (multiple-value-bind (fill-pattern frame-color fill-color)
                       (set-pen-given-state state)
    fill-pattern fill-color
    (with-fore-color frame-color
      (#_moveto x1 y1)
      (#_lineto x2 y2)))
  t)

;;;_______________________________________
;;;  size-cell

(defun size-cell (cell)
  (let* ((view (cell-view cell))
         (kind (cell-kind cell))
         (left (cell-left cell))
         (top (cell-top cell))
         (text-width (string-width (cell-text cell) (view-font view)))
         (text-h-offset (point-h (cell-text-offset view)))
         (text-v-offset (point-v (cell-text-offset view)))
         (right (+ left (case kind
                          (:cut (cell-height view))
                          (otherwise (+ text-width text-h-offset text-h-offset)))))
         (bottom (+ top (cell-height view))))
    (setf (cell-left cell) left
          (cell-top cell) top
          (cell-right cell) right
          (cell-bottom cell) bottom
          (cell-text-h cell) (+ left (case kind
                                       (:cut (floor (- (cell-height view) text-width) 2))
                                       (otherwise text-h-offset)))
          (cell-text-v cell) (+ top text-v-offset)))
  (orient-cell cell)
  cell)

;;;_______________________________________
;;;  size-all-cells

(defmethod size-all-cells ((view graph-view))
  (mapc #'size-cell (used-cells view)))


;;;_______________________________________
;;;  orient-cell

(defun orient-cell (cell)
  (let* ((left (cell-left cell))
         (top (cell-top cell))
         (right (cell-right cell))
         (bottom (cell-bottom cell))
         (center-h (floor (+ left right) 2))
         (center-v (floor (+ top bottom) 2)))
    (case (orientation (cell-view cell))
      (:horizontal
       (setf (cell-in-h cell) left
             (cell-in-v cell) center-v 
             (cell-out-h cell) right
             (cell-out-v cell) center-v))
      (:diagonal
       (setf (cell-in-h cell) left
             (cell-in-v cell) center-v)
       (case (cell-kind cell)
         ((:or :and :not)
          (setf (cell-out-h cell) right
                (cell-out-v cell) center-v))
         (otherwise (setf (cell-out-h cell) center-h
                          (cell-out-v cell) bottom))))
      (:vertical
       (setf (cell-in-h cell) center-h
             (cell-in-v cell) top
             (cell-out-h cell) center-h
             (cell-out-v cell) bottom))))
  cell)

;;;_______________________________________
;;;  orient-all-cells

(defmethod orient-all-cells ((view graph-view))
  (mapc #'orient-cell (used-cells view)))


;;;_______________________________________
;;;  place-cell

(defun place-cell (cell new-h new-v)
  (place-cell-h-components cell new-h)
  (place-cell-v-components cell new-v))

(defun place-cell-h-components (cell new-h)
  (let ((h-offset (- new-h (cell-left cell))))
    (offset-cell-h-components cell h-offset)))

(defun place-cell-v-components (cell new-v)
  (let ((v-offset (- new-v (cell-top cell))))
    (offset-cell-v-components cell v-offset)))

(defun offset-cell (cell h-offset v-offset)
  (offset-cell-h-components cell h-offset)
  (offset-cell-v-components cell v-offset))

(defun offset-cell-h-components (cell h-offset)
  (incf (cell-left cell) h-offset)
  (incf (cell-right cell) h-offset)
  (incf (cell-in-h cell) h-offset)
  (incf (cell-out-h cell) h-offset)
  (incf (cell-text-h cell) h-offset))

(defun offset-cell-v-components (cell v-offset)
  (incf (cell-top cell) v-offset)
  (incf (cell-bottom cell) v-offset)
  (incf (cell-in-v cell) v-offset)
  (incf (cell-out-v cell) v-offset)
  (incf (cell-text-v cell) v-offset))
  
;;;_______________________________________
;;; position-cells

(defmethod position-cells ((view graph-view))
  (orient-all-cells view)
  (case (orientation view)
    (:horizontal (position-horizontal-graph view))
    (:diagonal (position-diagonal-graph view))
    (:vertical (position-vertical-graph view)))
  (reset-graph-size view))


;;;_______________________________________
;;; position-horizontal-graph

(defmethod position-horizontal-graph ((view graph-view))
  (let* ((v-gap *horizontal-v-gap*)        ;;(max (round (cell-height view) 5) 1))
         (next-cell-h-position *graph-start-h*)
         (next-cell-v-position  *graph-start-v*))

    (labels ((top-child (node)
               (first (first (node-antecedents node))))
             
             (bottom-child (node)
               (first (last (first (last (node-antecedents node))))))
             
             (position-graph-v (root)
               (let ((cell (node-cell view root)))
                 (cond ((cell-expanded? cell)
                        (do* ((conjunctions (node-antecedents root) (rest conjunctions))
                              (conjunction (first conjunctions) (first conjunctions)))
                             ((null conjunctions))
                          (dolist (node conjunction)
                            (position-graph-v node))
                          (when (rest conjunctions)
                            (incf next-cell-v-position v-gap)))
                        (place-cell-v-components 
                         cell
                         (+ (floor (+ (cell-top (node-cell view (top-child root)))
                                      (cell-bottom (node-cell view (bottom-child root)))) 2)
                            (- (cell-top cell)
                               (cell-center-v cell)))))
                       (t
                        (place-cell-v-components cell next-cell-v-position)
                        (setf next-cell-v-position (+ (cell-bottom cell) v-gap))))))
             
             (position-graph-h (root h-position)
               (let ((cell (node-cell view root)))
                 (place-cell-h-components cell h-position)
                 (when (cell-expanded? cell)
                   (let* ((top-child (top-child root))
                          (bottom-child (bottom-child root))
                          (childrens-h-position 
                           (+ (cell-right cell)
                              (cond ((node-not? root) 10)
                                    ((eq top-child bottom-child) 15)
                                    ((max 15 (floor (- (cell-center-v (node-cell view bottom-child))
                                                       (cell-center-v (node-cell view top-child)))
                                                    4)))))))
                     (dolist (conjunction (node-antecedents root))
                       (dolist (node conjunction)
                         (position-graph-h node childrens-h-position)))))))
             )

      (position-graph-v (root view))
      (position-graph-h (root view) next-cell-h-position)
      )))

;;;_______________________________________
;;; position-diagonal-graph

(defmethod position-diagonal-graph ((view graph-view))
  (let* ((v-gap (max (round (cell-height view) 5) 1))
         (conjunction-gap v-gap)
         (next-cell-h-position *graph-start-h*)
         (next-cell-v-position  *graph-start-v*))

    (labels ((position-graph-v (root)
               (let ((cell (node-cell view root)))
                 (when (cell-p cell)
                   (place-cell-v-components cell next-cell-v-position)
                   (unless (node-second-order? root)
                     (setf next-cell-v-position (+ (cell-bottom cell) v-gap)))
                   (when (cell-expanded? cell)
                     (dolist (conjunction (node-antecedents root))
                       (dolist (node conjunction)
                         (position-graph-v node))
                       (incf next-cell-v-position conjunction-gap))))))
       
             (position-graph-h (root)
               (let ((cell (node-cell view root)))
                 (when (cell-expanded? cell)
                   (let* ((top (cell-bottom cell))
                          (bottom (cell-center-v 
                                   (node-cell
                                    view 
                                    (first (last (first (last (node-antecedents root))))))))
                          (child-h-position (+ (cell-center-h cell)
                                               (max 30 (round (- bottom top) 4)))))
                     (dolist (conjunction (node-antecedents root))
                       (dolist (node conjunction)
                         (place-cell-h-components (node-cell view node) child-h-position)
                         (position-graph-h node)))))))
             )
       
      (place-cell (node-cell view (root view)) next-cell-h-position next-cell-v-position)
      (position-graph-v (root view))
      (position-graph-h (root view))
      )))

;;;_______________________________________
;;; position-vertical-graph

(defmethod position-vertical-graph ((view graph-view))
  (let* ((height (cell-height view))
         (v-gap (max (+ height height) 20))
         (h-gap (max (round height 5) 2))
         (level-offset (+ height v-gap)) 
         (conjunction-gap (* h-gap 5))
         (next-cell-h-position *graph-start-h*)
         (next-cell-v-position  *graph-start-v*)
         (h-table (make-array (tree-depth (root view)) :initial-element next-cell-h-position)))
    
    (labels ((left-child (node)
               (first (first (node-antecedents node))))
             
             (right-child (node)
               (first (last (first (last (node-antecedents node))))))
             
             (offset-all-descendants-h (root h-offset)
               (let ((cell (node-cell view root)))
                 (when (cell-p cell)
                   (offset-cell-h-components cell h-offset)
                   (setf (aref h-table (round (- (cell-top cell) *graph-start-v*) level-offset))
                         (+ (cell-right cell) h-gap))
                   (dolist (conjunction (node-antecedents root))
                     (dolist (node conjunction)
                       (offset-all-descendants-h node h-offset))
                     (incf (aref h-table (+ (round (- (cell-top cell) *graph-start-v*) level-offset) 1))
                           conjunction-gap)))))
             
             (position-graph (root v-position)
               (let ((cell (node-cell view root)))
                 (when (cell-p cell)
                   (place-cell-v-components cell v-position)
                   (let ((antecedents-v-position (+ (cell-bottom cell) v-gap))
                         (cell-expanded? (cell-expanded? cell)))
                     (when cell-expanded?
                       (let ((h-index (round (- antecedents-v-position *graph-start-v*) level-offset)))
                         (dolist (conjunction (node-antecedents root))
                           (dolist (node conjunction)
                             (position-graph node antecedents-v-position))
                           (incf (aref h-table h-index) conjunction-gap))))
                     (let ((h-index (round (- (cell-top cell) *graph-start-v*) level-offset)))
                       (cond
                        (cell-expanded?
                         (let ((x-pos (+ (round (+ (cell-left (node-cell view (left-child root)))
                                                   (cell-right (node-cell view (right-child root))))
                                                2)
                                         (-  (cell-left cell) (cell-center-h cell)))))
                           (place-cell-h-components cell x-pos)
                           (when (< x-pos (aref h-table h-index))
                             (offset-all-descendants-h root (- (aref h-table h-index) x-pos)))
                           (setf (aref h-table h-index) (+ (cell-right cell) h-gap))))
                        (t
                         (place-cell-h-components cell (aref h-table h-index))
                         (setf (aref h-table h-index) (+ (cell-right cell) h-gap)))))))))
             )
      
      (position-graph (root view) next-cell-v-position)
      )))

;;;_______________________________________
;;;  display-tree-cells

(defmethod display-tree-cells ((view graph-view) &optional (node-or-collection (root view)))
  (cond ((node-p node-or-collection)
         (unless (node-cell view node-or-collection)
           (get-cell view :text (node-string node-or-collection) :node node-or-collection :hidden? (node-root? node-or-collection)))
         (display-tree-cells view (node-antecedents node-or-collection)))
        ((consp node-or-collection)
         (display-tree-cells view (first node-or-collection))
         (display-tree-cells view (rest node-or-collection)))))

;;;_______________________________________
;;;  show-antecedents-of-selected-cells

(defmethod show-antecedents-of-selected-cells ((view graph-view))
  (let ((graph (graph view))
         (window (graph-window view)))
    (when (expand-selected-nodes graph (root view))
      (display-tree-cells view)
      (position-cells view)
      (when (grow-window-if-needed window)
        (auto-position-window window))
      (invalidate-view view t))))

;;;_______________________________________
;;; expand-selected-nodes

(defun expand-selected-nodes (graph tree)
  (let ((some-node-expanded? nil))
    (labels ((expand (tree)
               (cond ((consp tree)
                      (expand (first tree))
                      (expand (rest tree)))
                     ((node-p tree)
                      (expand (node-antecedents tree))
                      (when (node-selected? tree)
                        (when (and (node-recursive? tree) (node-r-struct tree))
                          (setf (node-antecedents tree) (connect-clauses graph tree (editor-all-antecedents (node-r-struct tree) (node-vars tree)) (node-state tree) :never 0)
                                (node-recursive? tree) nil
                                some-node-expanded? t))
                        (select-node (node-antecedents tree)))))))
    (expand tree)
    some-node-expanded?)))

;;;_______________________________________
;;;  hide-antecedents-of-selected-cells

(defmethod hide-antecedents-of-selected-cells ((view graph-view))
  (let ((graph (graph view)))
    (when (truncate-selected-nodes graph (root view))
      (position-cells view)
      (reset-scroll-bars (view-container view))
      (invalidate-view view t))))

;;;_______________________________________
;;; truncate-selected-nodes

(defun truncate-selected-nodes (graph tree)
  (let ((some-node-truncated? nil))
    (labels ((truncate (tree)
               (cond ((consp tree)
                      (truncate (first tree))
                      (truncate (rest tree)))
                     ((node-p tree)
                      (if (and (node-selected? tree)
                               (node-intensional? tree))
                        (let ((antecedents (node-antecedents tree)))
                          (setf (node-antecedents tree) nil
                                (node-recursive? tree) t
                                some-node-truncated? t)
                          (dispose-node antecedents graph t))
                        (truncate (node-antecedents tree)))))))
      (truncate tree)
      some-node-truncated?)))

;;;_______________________________________
;;; hide-node

(defmethod hide-node ((view graph-view) node-or-collection &optional (recursive nil))
  (cond ((node-p node-or-collection)
         (let ((cell (node-cell view node-or-collection)))
           (when cell
             (setf (cell-hidden? cell) t
                   (node-selected? node-or-collection) nil)))
         (if (or recursive (node-second-order? node-or-collection) )
           (hide-node view (node-antecedents node-or-collection) recursive)))
        ((consp node-or-collection)
         (hide-node view (first node-or-collection) recursive)
         (hide-node view (rest node-or-collection) recursive))))


;;;_______________________________________
;;; show-node

(defmethod show-node ((view graph-view) node-or-collection &optional (select nil) (recursive nil))
  (cond ((node-p node-or-collection)
         (let ((cell (node-cell view node-or-collection)))
           (cond (cell (setf (cell-hidden? cell) nil
                             (node-selected? node-or-collection) select))
                 (t (setf cell (get-cell view :node node-or-collection :text (node-string node-or-collection)))
                    (size-cell cell))))
         (if (or recursive (node-second-order? node-or-collection) )
           (show-node view (node-antecedents node-or-collection) select recursive)))
        ((consp node-or-collection)
         (show-node view (first node-or-collection) select recursive)
         (show-node view (rest node-or-collection) select recursive))))

;;;_______________________________________
;;;  display-in-window

(defun display-in-window (root-or-graph &optional window)
  (when root-or-graph
    (let (root graph title)
      (cond ((node-p root-or-graph) (setf root root-or-graph
                                          graph (node-graph root)
                                          title (case (node-kind (graph-base graph))
                                                  (:cut "CUT")
                                                  (:undefined "Undefined")
                                                  (otherwise (node-string root))) ))
            ((graph-p root-or-graph) (setf graph root-or-graph
                                           root (graph-root graph)
                                           title (case (node-kind (graph-base graph))
                                                   (:cut "CUT")
                                                   (:undefined "Undefined")
                                                   (otherwise (node-string (graph-base graph)))))))
      (unless window
        (setf window (make-instance 'graph-window
                                    :window-show nil
                                    :window-title title)))
      (without-interrupts
       (let ((view (graph-view window)))
         (setf (root view) root
               (graph view) graph
               (graph-views graph) (push view (graph-views graph)))
         (display-tree-cells view root)
         (size-all-cells view)
         (position-cells view)
         (resize-window window)
         (position-graph view :centered t)
         (auto-position-window window :centered t)
         (force-graph-redraw view)
         (window-select window)
         window)))))

;;;_______________________________________
;;;  display-r-structs

(defun display-r-structs (r-structs &optional (source nil) (expand *default-expand*) (title nil))
  (without-interrupts
   (set-cursor *watch-cursor*)
   (let ((graph (generate-graph r-structs source expand ))
         (r-struct (first r-structs)))
     (when graph
       (let ((window (make-instance 'graph-window
                       :window-show nil
                       :window-title (or title
                                         (format nil "~(~A~)~A"
                                                 (r-name r-struct)
                                                 (r-vars r-struct))))))
         (display-in-window graph window))))))

;;;_______________________________________
;;;  display-goal-concept

(defun display-goal-concept (&optional (source nil) (expand *default-expand*) (expand-depth *default-expand-depth*))
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-display-goal-concept *user-monitor*)))
  (without-interrupts
   (set-cursor *watch-cursor*)
   (let* ((name (goal-concept-name))
          (r-struct (get-rule name))
          (graph (when r-struct (generate-graph (list r-struct) source expand expand-depth))))
     (when graph
       (let ((window (make-instance 'graph-window :window-show nil :window-title "Goal Concept")))
         (display-in-window graph window))))))

;;;_______________________________________
;;;  display-domain-theory

(defun display-domain-theory (&optional (source nil) (expand *default-expand*) (expand-depth *default-expand-depth*))
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-display-domain-theory *user-monitor*)))
  (without-interrupts
   (set-cursor *watch-cursor*)
   (let* ((name (goal-concept-name))
          (r-struct (get-rule name))
          (vars (when r-struct (r-vars r-struct)))
          (goal-concept (when vars (cons name vars)))
          (graph (generate-domain-theory-graph goal-concept source expand expand-depth)))
     (when graph
       (let ((window (make-instance 'graph-window :window-show nil :window-title "Domain Theory")))
         (display-in-window graph window))))))

;;;_______________________________________
;;;  redisplay-last-node-selected

(defmethod redisplay-last-node-selected ((view graph-view))
  (let ((node (last-node-selected view)))
         (when node
           (display-in-window (generate-graph (list (convert-tree-to-prolog node)) (node-state node))))))

;;;_______________________________________
;;;  clear-external-text

(defmethod clear-external-text ((view graph-view))
  (dolist (cell (used-cells view))
    (setf (cell-external-text cell) nil)))

;;;_______________________________________
;;;   window-open?

(defun window-open? (window)
  (cond ((null window) nil)
        ((wptr window) t)
        (t nil)))

;;;_______________________________________
;;;  shrink-all-cells

(defmethod shrink-all-cells ((view graph-view))
  (let ((window (graph-window view)))
    (set-view-font window '("times" 1 :plain))
    (let ((new-height 5)
          (text-offset 3)
          (font (view-font view)))
    (setf (cell-height view) new-height
          (cell-text-offset view) (make-point text-offset text-offset))
    (dolist (cell (used-cells view))
      (let* ((left (cell-left cell))
             (top (cell-top cell))
             (right (+ left (case (cell-kind cell)
                              (:cut new-height)
                              (otherwise (+ (string-width (cell-text cell) font) text-offset text-offset)))))
             (bottom (+ top new-height)))
        (setf (cell-left cell) left
              (cell-top cell) top
              (cell-right cell) right
              (cell-bottom cell) bottom
              (cell-text-h cell) (+ left text-offset)
              (cell-text-v cell) (+ top text-offset))))

    (position-cells view)
    (grow-window-if-needed window)
    (auto-position-window window)
    (reset-scroll-bars (graph-scroller window))
    (invalidate-view window t))))

;;;_______________________________________
;;;  copy-graph-to

(defun copy-graph-to (window)
  (case *copy-graph-to*
    (:prolog
     (let ((prolog (convert-graph-to-prolog (window-graph window))))
       (when prolog
         (put-scrap :TEXT (format nil "~A" prolog)))))
    (:selected-nodes
     (let ((selected-nodes (selected-nodes (window-graph window) t)))
       (when selected-nodes
         (put-scrap :TEXT (with-output-to-string (out)
                            (format out "(")
                            (dolist (node selected-nodes)
                              (format out "~A" (convert-tree-to-prolog node)))
                            (format out ")"))))))
    (:pict
     (let ((graph-picture (export-graph-picture (graph-view window))))
       (when graph-picture
         (put-scrap :PICT graph-picture t))))))

;;;_______________________________________
;;; FIXUP-VIEWS-OF-GRAPH

(defun fixup-views-of-graph (graph)
  (without-interrupts 
   (dolist (view (graph-views graph))
     (display-tree-cells view)
     (position-cells view)
     (when (grow-window-if-needed (graph-window view))
       (auto-position-window (graph-window view)))
     (invalidate-view view t))))
