
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

;;;=========================================================
;;; THEORY EDITOR WINDOW

;;;_______________________________________
;;; theory-edit-window

(defclass theory-edit-window (edit-window)
  ((changed-nodes :initarg :changed-nodes :initform nil :accessor changed-nodes)
   (induced-node.states :initarg :induced-node.states :initform nil :accessor induced-node.states)
   (learned-description :initarg :learned-description :initform nil :accessor learned-description) 
   (induced-portions :initarg :induced-portions :initform nil :accessor induced-portions)
   (kr-state :initarg :kr-state :initform nil :accessor kr-state)
   (show-dialog :initarg :show-dialog :initform t :accessor show-dialog)

   (enable-EDIT-DEF :initarg :enable-EDIT-DEF :initform t :accessor enable-EDIT-DEF)
   (enable-EDIT-CALL :initarg :enable-EDIT-CALL :initform t :accessor enable-EDIT-CALL)
   (enable-CREATE :initarg :enable-CREATE :initform t :accessor enable-CREATE)
   (enable-COPY :initarg :enable-COPY :initform t :accessor enable-COPY)
   (enable-NEGATE :initarg :enable-NEGATE :initform t :accessor enable-NEGATE)
   (enable-CONJOIN :initarg :enable-CONJOIN :initform t :accessor enable-CONJOIN)
   (enable-DISJOIN :initarg :enable-DISJOIN :initform t :accessor enable-DISJOIN)
   (enable-REPLACE :initarg :enable-REPLACE :initform t :accessor enable-REPLACE)
   (enable-DELETE :initarg :enable-DELETE :initform t :accessor enable-DELETE)
   (enable-SHOW :initarg :enable-SHOW :initform t :accessor enable-SHOW)
   (enable-HIDE :initarg :enable-HIDE :initform t :accessor enable-HIDE)
   (enable-NEXT :initarg :enable-NEXT :initform nil :accessor enable-NEXT)
   (enable-CANCEL :initarg :enable-CANCEL :initform t :accessor enable-CANCEL)
   (enable-DEFINE :initarg :enable-DEFINE :initform t :accessor enable-DEFINE)))

;;;_________________________________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window theory-edit-window) &rest initargs)
  (setf (getf initargs :view-size) #@(540 360))
  (apply #'call-next-method window initargs)
  (let ((x 5) (y 30)
        (button-size #@(70 18))
        (button-step 22)
        (button-font '("Chicago" 12 :plain))
        (space-step 30))
    (add-subviews 
     window
     (make-dialog-item
      'button-dialog-item (make-point x y) button-size " Edit Def "
      #'(lambda (item) (editor-EDIT-DEF (view-container item)))
      :view-font button-font :view-nick-name :edit-def)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Edit Call "
      #'(lambda (item) (editor-EDIT-CALL (view-container item)))
      :view-font button-font :view-nick-name :edit-call)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Create "
      #'(lambda (item) (editor-CREATE (view-container item)))
      :view-font button-font :view-nick-name :create)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Copy "
      #'(lambda (item) (editor-COPY (view-container item)))
      :view-font button-font :view-nick-name :copy)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Negate "
      #'(lambda (item) (editor-NEGATE (view-container item)))
      :view-font button-font :view-nick-name :negate)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Conjoin "
      #'(lambda (item) (editor-CONJOIN (view-container item)))
      :view-font button-font :view-nick-name :conjoin)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Disjoin "
      #'(lambda (item) (editor-DISJOIN (view-container item)))
      :view-font button-font :view-nick-name :disjoin)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Replace "
      #'(lambda (item) (editor-REPLACE (view-container item)))
      :view-font button-font :view-nick-name :replace)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Delete "
      #'(lambda (item) (editor-DELETE (view-container item)))
      :view-font button-font :view-nick-name :delete)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Show "
      #'(lambda (item) (editor-SHOW (view-container item)))
      :view-font button-font :view-nick-name :show)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Hide "
      #'(lambda (item) (editor-HIDE (view-container item)))
      :view-font button-font :view-nick-name :hide)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Next "
      #'(lambda (item) (editor-NEXT (view-container item)))
      :view-font button-font :view-nick-name :next)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Cancel "
      #'(lambda (item) (editor-CANCEL (view-container item)))
      :view-font button-font :view-nick-name :cancel)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Define "
      #'(lambda (item) (editor-DEFINE (view-container item)))
      :view-font button-font :view-nick-name :define)
     )))


;;;_______________________________________
;;;  configure-controls

(defmethod configure-controls ((window theory-edit-window))
  (let* ((view (graph-view window))
         (graph (graph view))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph))
         (a-nodes (selected-nodes a-base t))
         (u-nodes (selected-nodes u-base t))
         (intensional-selected nil)
         (expanded-intensional-selected nil)
         
         (node (only-one a-nodes u-nodes))
         (not-a-base (not (node-selected? a-base)))
         )
    
    (dolist (node (graph-used-nodes graph))
      (when (node-selected? node)
        (when (node-intensional? node)
          (setf intensional-selected t)
          (when (and (node-antecedents node) (not (eq node a-base)))
            (setf expanded-intensional-selected t)))))
    
    (if (and node (not (node-not? node)) (enable-EDIT-DEF window))
      (dialog-item-enable (view-named :edit-def window))
      (dialog-item-disable (view-named :edit-def window)))
    
    (if (and node not-a-base (not (node-not? node)) (enable-EDIT-CALL window))
      (dialog-item-enable (view-named :edit-call window))
      (dialog-item-disable (view-named :edit-call window)))
    
    (if (enable-CREATE window)
      (dialog-item-enable (view-named :create window))
      (dialog-item-disable (view-named :create window)))

    (if (enable-COPY window)
      (dialog-item-enable (view-named :copy window))
      (dialog-item-disable (view-named :copy window)))
    
    (if (and (enable-NEGATE window)
             not-a-base
             (or node
                 (and (nodes-form-conjunction a-nodes) (null u-nodes))
                 (and (nodes-form-conjunction u-nodes) (null a-nodes))) )
      (dialog-item-enable (view-named :negate window))
      (dialog-item-disable (view-named :negate window)))
    
    (if (and not-a-base
             (nodes-form-conjunction a-nodes)
             u-nodes (enable-CONJOIN window))
      (dialog-item-enable (view-named :conjoin window))
      (dialog-item-disable (view-named :conjoin window)))
    
    (if (and (only-one a-nodes)
             (node-intensional? (first a-nodes))
             u-nodes (enable-DISJOIN window))
      (dialog-item-enable (view-named :disjoin window))
      (dialog-item-disable (view-named :disjoin window)))
    
    (if (and node not-a-base (node-antecedents node) (enable-REPLACE window))
      (dialog-item-enable (view-named :replace window))
      (dialog-item-disable (view-named :replace window)))
    
    (if (and not-a-base (or a-nodes u-nodes) (enable-DELETE window))
      (dialog-item-enable (view-named :delete window))
      (dialog-item-disable (view-named :delete window)))
    
    (if (and intensional-selected (enable-SHOW window))
      (dialog-item-enable (view-named :show window))
      (dialog-item-disable (view-named :show window)))
    
    (if (and expanded-intensional-selected (enable-HIDE window))
      (dialog-item-enable (view-named :hide window))
      (dialog-item-disable (view-named :hide window)))
    
    (if (enable-NEXT window)
      (dialog-item-enable (view-named :next window))
      (dialog-item-disable (view-named :next window)))
    
    (if (enable-CANCEL window)
      (dialog-item-enable (view-named :cancel window))
      (dialog-item-disable (view-named :cancel window)))
    
    (if (enable-DEFINE window)
      (dialog-item-enable (view-named :define window))
      (dialog-item-disable (view-named :define window)))
    ))

;;;_______________________________________
;;; reorder-nodes

(defmethod reorder-nodes ((window theory-edit-window))
  (let* ((view (graph-view window))
         (graph (window-graph window))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph)))
    (setf (node-antecedents a-base) (order-nodes-based-on-cell-positions (node-antecedents a-base) view t)
          (node-antecedents u-base) (order-nodes-based-on-cell-positions (node-antecedents u-base) view t))
    (position-cells view)
    (invalidate-view view t)))

;;;_______________________________________
;;;   view-key-event-handler

(defmethod view-key-event-handler ((window theory-edit-window) char)
  (let ((view (graph-view window)))
    (case (char-code char)
      ((67 99) (editor-EDIT-DEF window))               ;;; C c
      ((69 101) (editor-EDIT-DEF window))              ;;; E e
      ((72 104) (editor-HIDE window))                  ;;; H h
      ((73 105) (inspect-last-node-selected view))     ;;; I i
      ((82 114) (invalidate-view window t))            ;;; R r
      ((83 115) (editor-SHOW window))                  ;;; S s
      ((84 116) (show-tuples-last-node-selected view)) ;;; T t
      (others nil))))

;;;_______________________________________
;;;  store-induced-node-states

(defmethod store-induced-node-states ((window theory-edit-window) nodes)
  (when (kr-state window)
    (cond ((node-p nodes)
           (unless (member nodes (induced-node.states window) :key #'car)
             (let ((state (node-state nodes)))
               (unless (or (equalp state :unoperationalized)
                           (equalp state :ebl)
                           (equalp state nil))
                 (setf (induced-node.states window) (push (cons nodes state) (induced-node.states window)))
                 (store-induced-node-states window (node-antecedents nodes))))))
         ((consp nodes)
          (store-induced-node-states window (first nodes))
          (store-induced-node-states window (rest nodes))))))

;;;________________________________
;;; hilite-induced-nodes

(defmethod hilite-induced-nodes ((window theory-edit-window))
  (let ((some-node-hi-lited nil) node state)
    (dolist (node.state (induced-node.states window))
      (setf node (first node.state)
            state (rest node.state))
      (unless (or (equalp (node-state (node-consequent node)) :unoperationalized)
                  (equalp (node-state node) state))
        (setf (node-state node) state
              some-node-hi-lited t)))
    (when some-node-hi-lited (hilite-induced-nodes window))))



;;;_______________________________________
;;;  editor-EDIT-DEF

(defmethod editor-EDIT-DEF ((window theory-edit-window))
  window)

;;;_______________________________________
;;;  editor-EDIT-CALL

(defmethod editor-EDIT-CALL ((window theory-edit-window))
  window)

;;;_______________________________________
;;;  editor-CREATE

(defmethod editor-CREATE ((window theory-edit-window))
  (user-create-nodes window nil))
 
;;;_______________________________________
;;;  editor-COPY

(defmethod editor-COPY ((window theory-edit-window))
  window)

;;;_______________________________________
;;;  editor-NEGATE

(defmethod editor-NEGATE ((window theory-edit-window))
  window)

;;;_______________________________________
;;;  editor-CONJOIN

(defvar *allow-user-to-change-name* t)

(defmethod editor-CONJOIN ((window theory-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (a-nodes (selected-nodes a-base t))
         (u-nodes (selected-nodes (graph-u-base graph) t)))
    (catch-cancel
      (when (and (not (node-selected? a-base)) (nodes-form-conjunction a-nodes) u-nodes)
        (dolist (node u-nodes)
          (when (eq (node-kind node) :special)
            (let ((a-vars (compute-conequent-and-conjunction-nodes-and-types a-nodes))
                  (name (node-r-struct node))
                  (vars (node-vars node))
                  (types (node-aux node)))
              (do ((pvs nil vs)
                   (vs vars (rest vs))
                   (pts nil ts)
                   (ts types (rest ts)))
                  ((null vs))
                (unless (member (first vs) a-vars)
                  (if (null pvs)
                    (setq vars (rest vs)
                          types (rest ts))
                    (setf (rest pvs) (rest vs)
                          (rest pts) (rest ts)))))
              (when *allow-user-to-change-name*
                (let ((literal (create-new-relation "Change name and arguments of new rule:"  (format nil "(~(~S~)~{ ~S~})" name vars))))
                  (setq name (first literal))
                  (unless (equal vars (rest literal))
                    (setq vars (rest literal)
                          types (make-list (length vars) :initial-element :anything)))))
              (eval `(def-rule ,name  :vars ,vars :type ,types :clauses ,(convert-antecedents-to-clauses (node-antecedents node) name vars)))
              (setf (node-r-struct node) (get-r-struct name)
                    (node-vars node) (r-vars (node-r-struct node))
                    (cell-text (first (node-cells node))) (node-string node)
                    (changed-nodes window) (push node (changed-nodes window))))))
        (let* (;(old-conjunction (copy-list (conjunction-containing-node (first a-nodes))))
               (new-conjunction (conjoin a-nodes u-nodes graph))
               (consequent (node-consequent (first new-conjunction))))
          (when new-conjunction
            (store-induced-node-states window new-conjunction)
            (setf (changed-nodes window) (push consequent (changed-nodes window)))
            ;(conjoin-to-like-consequents graph consequent old-conjunction new-conjunction u-nodes)
            (window-fixup window t)))))))

;;;_______________________________________
;;;  editor-DISJOIN

(defmethod editor-DISJOIN ((window theory-edit-window))
  (let* ((graph (window-graph window))
         (a-nodes (selected-nodes (graph-a-base graph) t))
         (u-nodes (selected-nodes (graph-u-base graph) t))
         (consequent (first a-nodes)))
    (when (and (only-one a-nodes) (node-intensional? consequent) u-nodes)
      (let ((new-conjunction (disjoin consequent u-nodes graph)))
        (store-induced-node-states window new-conjunction)
        (setf (changed-nodes window) (push consequent (changed-nodes window)))
        ;(disjoin-to-like-consequents graph consequent new-conjunction)
        (window-fixup window t)))))

;;;_______________________________________
;;;  editor-REPLACE

(defmethod editor-REPLACE ((window theory-edit-window))
  window)

;;;_______________________________________
;;;  editor-DELETE

(defmethod editor-DELETE ((window theory-edit-window))
  (let* ((graph (window-graph window)))
    (dolist (node (selected-nodes graph t))
      (setf (changed-nodes window) (push (node-consequent node) (changed-nodes window)))
      (disconnect-and-free-tree node graph)))
  (window-fixup window t))

;;;_______________________________________
;;;  editor-SHOW

(defmethod editor-SHOW ((window theory-edit-window))
  (without-interrupts
   (let* ((view (graph-view window))
          (graph (graph view)))
     (setf (expand view) nil)
     (dolist (node (selected-nodes graph))
       (show-node view (node-antecedents node) t nil)
       (when (and (node-intensional? node)
                  (node-r-struct node)
                  (null (node-antecedents node)) )
         (setf (node-antecedents node) (connect-clauses graph node (editor-all-antecedents (node-r-struct node) (node-vars node)) nil :never 0)
               (node-recursive? node) nil)
         (select-node (node-antecedents node))))
     (window-fixup window t))))

;;;_______________________________________
;;;  editor-HIDE

(defmethod editor-HIDE ((window theory-edit-window))
  (let ((view (graph-view window)))
    (hide-antecedents-of-selected-cells view))
  (configure-controls window))

;;;_______________________________________
;;;  editor-NEXT
    
(defmethod editor-NEXT ((window theory-edit-window))
  (when (fboundp 'apply-heuristics)
    (apply-heuristics window)))

;;;_______________________________________
;;;  editor-CANCEL
    
(defmethod editor-CANCEL ((window theory-edit-window))
  (window-close window))

;;;_______________________________________
;;;  editor-DEFINE
    
(defmethod editor-DEFINE ((window theory-edit-window))
  (deselect-node (root (graph-view window)) t)
  (apply-function-to-tree (root (graph-view window)) 
                          #'(lambda (node)
                              (select-node node)
                              (when (member node (changed-nodes window))
                                (let* ((antecedents (node-antecedents node))
                                       (r-struct (node-r-struct node))
                                       (head (cons (r-name r-struct) (node-vars node))))
                                  (select-node antecedents)
                                  (setf (r-vars r-struct) (node-vars node)
                                        (r-clauses r-struct) (mapcar #'(lambda (conjunction)
                                                                         (let ((prolog (convert-tree-to-prolog conjunction)))
                                                                           (if (and (eql (first prolog) 'and)
                                                                                    (not (node-and? (first conjunction))))
                                                                             (rplaca prolog head)
                                                                             (list head prolog))))
                                                                     antecedents)
                                        r-struct (def-from-r-struct r-struct))
                                  (deselect-node antecedents t)))
                              (deselect-node node))
                          t)
  (setf *rules-changed* t)
  (set-status :finished-revising)
  (window-close window))


;;;____________________________________
;;;  revise

(defun revise (&optional (learned-description *learned-description*))
  (without-interrupts
   (set-cursor *watch-cursor*)
   (when (user-monitor-p *user-monitor*)
     (incf (user-monitor-revise *user-monitor*)))
   (let* ((name (first *goal-concept*))
          (rule (get-r-struct name))
          (vars (r-vars rule))
          (graph (create-graph))
          (root (graph-root graph))
          (a-base (connect-literal graph root (cons name vars) nil nil :completely 0))
          (u-base (get-node graph :consequent root :kind :intensional))
          (window (make-instance 'theory-edit-window
                    :learned-description learned-description
                    :window-title "Revise Theory"))
          (view (graph-view window)))
     (setf (node-antecedents root) (list (list a-base) (list u-base))
           (node-selection-constraint view) :no-drag
           (orientation view) :horizontal
           (expand view) :every-use
           (root view) root
           (graph view) graph
           (graph-views graph) (push view (graph-views graph)))
     (get-cell view :text "u-base" :node u-base :hidden? t)
     (mark-ebl-deletions learned-description graph)   
     (setf (kr-state window) :initialize-KR-Heuristic-1)
     (apply-heuristics window)
     (values))))

;;;____________________________________
;;;  mark-ebl-deletions

(defun mark-ebl-deletions (learned-description graph)
  (let (derivation)
    (dolist (clause learned-description)
      (do ((literal clause (literal-next literal)))
          ((null literal))
        (setf derivation (literal-derivation literal))
        (if (and (literal-deleted? literal)
                 (equalp (derivation-type derivation) :ebl))
          (setf (node-deleted? (find-corresponding-node (derivation-path derivation)
                                                        (derivation-graph derivation)
                                                        graph
                                                        t))
                t))))))

;;;____________________________________
;;; apply-heuristics

(defun apply-heuristics (window)
  (let* ((learned-description (learned-description window))
         (induced-portions (induced-portions window))
         (view (graph-view window))
         (graph (window-graph window))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph)))

    (case (kr-state window)

      (:initialize-KR-Heuristic-1
       (setf (induced-portions window) (collect-clauses-with-same-ebl-portions (clauses-learned-via-combination learned-description))
             (kr-state window) :apply-KR-Heuristic-1
             (enable-EDIT-DEF window) nil
             (enable-EDIT-CALL window) nil
             (enable-CREATE window) nil
             (enable-COPY window) nil
             (enable-NEGATE window) nil
             (enable-CONJOIN window) t
             (enable-DISJOIN window) nil
             (enable-REPLACE window) nil
             (enable-DELETE window) nil
             (enable-SHOW window) t
             (enable-HIDE window) t
             (enable-NEXT window) t
             (enable-CANCEL window) t
             (enable-DEFINE window) nil)
       (show-node view a-base nil t)
       (window-fixup window t)
       (when (show-dialog window) (setf (show-dialog window) t))
       (apply-heuristics window))

    (:apply-KR-Heuristic-1
     (let ((clauses-with-same-ebl (first induced-portions)))
       (setf (induced-portions window) (rest induced-portions))
       (cond (clauses-with-same-ebl
              (deselect-node a-base t)
              (disconnect-and-free-tree (node-antecedents u-base) graph)
              (let ((disjunction (mapcar #'(lambda (clause) (get-induced-portion clause u-base graph)) clauses-with-same-ebl)))
                (if (rest disjunction)
                  (let ((disjunction-node (get-node graph :kind :special :state :intensional :consequent u-base :antecedents disjunction))
                        (nl-r-name (unique-r-name 'new_rule)))
                    (multiple-value-bind (nl-vars nl-types) (collect-vars-and-types disjunction-node)
                      (setq nl-vars (nreverse nl-vars)
                            nl-types (nreverse nl-types))
                      (setf (node-vars disjunction-node) nl-vars
                            (node-aux disjunction-node) nl-types
                            (node-r-struct disjunction-node) nl-r-name)
                      (get-cell view :node disjunction-node :text (format nil "(~(~S~)~{ ~S~})" nl-r-name nl-vars))
                      (dolist (conjunction disjunction)
                        (dolist (node conjunction)
                          (setf (node-consequent node) disjunction-node)))
                      (setf (node-antecedents u-base)  (list (list disjunction-node)))))
                  (setf (node-antecedents u-base) disjunction)))
              (display-tree-cells view)
              (hilite-node a-base :unoperationalized t t)
              (hi-lite-clause-ebl-paths (first clauses-with-same-ebl) graph)
              (window-fixup window t)
              (window-select window)
              (select-frontier a-base)
              (select-frontier u-base)
              (configure-controls window)
              (invalidate-view window t)
              (when (eq (show-dialog window) t)
                (setf (show-dialog window) :shown)
                (attach-literals-message)) )
             (t
              (setf (kr-state window) :initialize-KR-Heuristic-2)
              (apply-heuristics window)))))

    (:initialize-KR-Heuristic-2
     (setf (induced-portions window) (clauses-learned-via-induction learned-description)
           (kr-state window) :apply-KR-Heuristic-2
           (enable-EDIT-DEF window) nil
           (enable-EDIT-CALL window) nil
           (enable-CREATE window) nil
           (enable-COPY window) nil
           (enable-NEGATE window) nil
           (enable-CONJOIN window) nil
           (enable-DISJOIN window) t
           (enable-REPLACE window) nil
           (enable-DELETE window) nil
           (enable-SHOW window) t
           (enable-HIDE window) t
           (enable-NEXT window) t
           (enable-CANCEL window) t
           (enable-DEFINE window) nil)
     (hilite-node a-base :unoperationalized t t)
     (hi-lite-all-ebl-paths learned-description graph)
     (hilite-induced-nodes window)
     (show-node view a-base nil t)
     (when (show-dialog window) (setf (show-dialog window) t))
     (apply-heuristics window))

    (:apply-KR-Heuristic-2
     (let ((induced-clause (first induced-portions)))
       (setf (induced-portions window) (rest induced-portions))
       (cond (induced-clause
              (deselect-node a-base t)
              (disconnect-and-free-tree (node-antecedents u-base) graph)
              (setf (node-antecedents u-base) (list (connect-clause graph u-base induced-clause nil nil :every-use 0)))
              (nicify-node-bindings u-base (generate-nice-binding-alist induced-clause graph) t)
              (display-tree-cells view)
              (window-fixup window t)
              (window-select window)
              (select-node a-base nil t)
              (select-frontier u-base)
              (configure-controls window)
              (invalidate-view window t)
              (when (eq (show-dialog window) t)
                (setf (show-dialog window) :shown)
                (attach-clause-message)) )
             (t 
              (setf (kr-state window) :initialize-KR-Heuristic-3)
              (apply-heuristics window)))))

    (:initialize-KR-Heuristic-3
     (setf (enable-EDIT-DEF window) nil
           (enable-EDIT-CALL window) nil
           (enable-CREATE window) nil
           (enable-COPY window) nil
           (enable-NEGATE window) nil
           (enable-CONJOIN window) nil
           (enable-DISJOIN window) nil
           (enable-REPLACE window) nil
           (enable-DELETE window) t
           (enable-SHOW window) t
           (enable-HIDE window) t
           (enable-NEXT window) nil
           (enable-CANCEL window) t
           (enable-DEFINE window) t
           (kr-state window) nil)
     (deselect-node a-base t)
     (disconnect-and-free-tree (node-antecedents u-base) graph)
     (setf (node-antecedents u-base) nil)
     (hilite-node a-base :unoperationalized t t)
     (hi-lite-all-ebl-paths learned-description graph)
     (hilite-induced-nodes window)
     (show-node view a-base nil t)
     (window-fixup window t)
     (window-select window)
     (select-nodes-to-delete graph)
     (configure-controls window)
     (invalidate-view window t)
     (when (show-dialog window) (delete-message))
     ))))

(defun attach-literals-message ()
  (message-dialog
"During learning the selected clause in the upper
graph was specialized by conjoining the selected
literals in the lower graph.  The same accuracy
can be obtained by conjoining these literals at
other points along the operationalized path.
Select the best attachment point for each literal." :size #@(440 150) :position :centered))

(defun attach-clause-message ()
  (message-dialog
"During learning the slected clause in the lower
graph was added as a disjunct at the top level.
The same accuracy can be obtained by attaching
it at other points in the knowledge base.
Select the best attachment point for the clause." :size #@(440 150) :position :centered))


(defun delete-message ()
  (message-dialog
"The selected literals and clauses were not needed
to accurately describe the goal concept.
Please delete those literals and clauses which are
not semantically meaningful." :size #@(440 150) :position :centered))

;;;_______________________________________________________________________________
;;;  get-induced-portion

(defun get-induced-portion (clause u-base graph)
  (let ((conjunction nil)
        (added-graphs nil)
        (binding-alist (generate-nice-binding-alist clause graph))
        derivation type d-graph node)
    (do ((literal clause (literal-next literal)))
        ((null literal) nil)
      (setf derivation (literal-derivation literal)
            type (derivation-type derivation)
            d-graph (derivation-graph derivation))
      (case type
        (:ebl nil)
        ((:extensional :builtin :determinate :cliche)
         (setf node (connect-literal graph u-base literal nil type :every-use 0))
         (nicify-node-bindings node binding-alist)
         (push node conjunction))
        (:intensional
         (unless (member d-graph added-graphs)
           (push d-graph added-graphs)
           (setf node (duplicate-node (first (first (node-antecedents (graph-root d-graph))))
                                      u-base graph t))
           (nicify-node-bindings node binding-alist t)
           (push node conjunction))
         )))
    (nreverse conjunction)))

;;;_______________________________________________________________________________
;;;  nicify-node-bindings

(defun nicify-node-bindings (node alist &optional (recursive nil))
  (when (node-p node)
    (when (node-r-struct node)
      (setf (node-vars node) (nicify-parameter-bindings (node-vars node) (r-vars (node-r-struct node)) alist)))
    (when (or recursive (node-not? node))
      (dolist (conjunction (node-antecedents node))
        (dolist (n conjunction)
          (nicify-node-bindings n alist recursive))))))

;;;_______________________________________________________________________________
;;;  nicify-parameter-bindings

(defun nicify-parameter-bindings (parameters defaults alist)
  (cond ((null parameters) nil)
        ((consp parameters)
         (mapcar 
          #'(lambda (parameter default)
              (cond ((consp parameter)
                     (mapcar #'(lambda (p d) (nicify-parameter-bindings p d alist))
                             parameter default))
                    ((not (pcvar-p parameter))  parameter)
                    ((rest (assoc parameter alist :test #'equalp)))
                    (t (nconc alist (list (cons parameter default)))
                       default)))
          parameters defaults))
        ((not (pcvar-p parameters))  parameters)
        ((rest (assoc parameters alist :test #'equalp)))
        (t (nconc alist (list (cons parameters defaults)))
           defaults)))

;;;_______________________________________________________________________________
;;;  generate-nice-binding-alist

(defun generate-nice-binding-alist (clause revision-graph)
  (let* ((binding-alist (mapcar #'cons (rest *goal-concept*) (r-vars (get-r-struct (first *goal-concept*)))))
         derivation)
    (do ((literal clause (literal-next literal)))
        ((null literal) nil)
      (setf derivation (literal-derivation literal))
      (when (eq (derivation-type derivation) :ebl)
        (dolist (binding (literal-bindings literal
                                           (find-corresponding-node (derivation-path derivation)
                                                                    (derivation-graph derivation)
                                                                    revision-graph t)))
          (pushnew binding binding-alist :test #'equalp :key #'car))))
    (delete-if-not #'(lambda (binding) (pcvar-p (first binding))) binding-alist)))
                  
;;;_______________________________________________________________________________
;;;  literal-bindings

(defun literal-bindings (literal node)
  (case (node-kind node)
    (:not (clause-bindings (literal-negated-literals literal) (first (node-antecedents node))))
    (:is (if (consp (first (node-vars node)))
           (mapcar #'cons (first (literal-variablization literal)) (first (node-vars node)))
           (list (cons (first (literal-variablization literal)) (first (node-vars node))))))
    (otherwise (mapcar #'cons (literal-variablization literal) (node-vars node)))))

;;;_______________________________________________________________________________
;;;  clause-bindings

(defun clause-bindings (clause conjunction)
  (let ((binding-alist nil))
    (do* ((literal clause (literal-next literal))
          (nodes conjunction (rest nodes)))
        ((null literal) binding-alist)
      (dolist (binding (literal-bindings literal (first nodes)))
        (pushnew binding binding-alist :test #'equalp)))
    binding-alist))


;;;_______________________________________________________________________________
;;;  clause-learned-via
;;;
;;;  Returns :COMBINATION if the clause was learned by a combination of ebl and induction.
;;;          :INDUCTION   if the clause was learned by induction alone
;;;          :EBL         if the clause was learned by ebl alone

(defun clause-learned-via (clause)
  (let ((induced nil)
        (operationalized nil))
    (do* ((literal clause (literal-next literal)))
         ((or (null literal) (and operationalized induced)))
      (unless (literal-deleted? literal)
        (if (eql (derivation-type (literal-derivation literal)) :EBL)
          (setf operationalized t)
          (setf induced t))))
    (cond ((and induced operationalized) :COMBINATION)
          (induced :INDUCTION)
          (operationalized :EBL)
          (t nil))))

;;;_______________________________________________________________________________
;;;  clauses-learned-via-combination

(defun clauses-learned-via-combination (clauses)
  (remove-if-not #'(lambda (clause) (eql (clause-learned-via clause) :COMBINATION)) clauses))

;;;_______________________________________________________________________________
;;;  clauses-learned-via-induction

(defun clauses-learned-via-induction (clauses)
  (remove-if-not #'(lambda (clause) (eql (clause-learned-via clause) :INDUCTION)) clauses))

;;;________________________________________________________________________________
;;; same-ebl-portions-p

(defun same-ebl-portions-p (clause-A clause-B)
  (let ((equal? t))
    (do ((literal-A clause-A (literal-next literal-A)))
        ((or (null literal-A) (null equal?)) equal?)
      (when (equalp (derivation-type (literal-derivation literal-A)) :EBL)
        (do ((literal-B clause-B (literal-next literal-B)))
            ((or (null literal-B)
                 (eq (derivation-path (literal-derivation literal-A))
                     (derivation-path (literal-derivation literal-B))))
             (unless literal-B (setf equal? nil))))))))

;;;_______________________________________________________________________________
;;;  collect-clauses-with-same-ebl-portions

(defun collect-clauses-with-same-ebl-portions (combination-clauses)
  (let ((collections nil)
        (collection nil))
    (do* ((clauses combination-clauses (rest clauses))
          (clause (first clauses) (first clauses)))
         ((null clauses) collections)
      (setf collection (list clause))
      (do* ((remaining clauses (rest remaining))
            (candidate (second remaining) (second remaining)))
           ((null (rest remaining)) (setf collections (nconc collections (list collection))))
        (when (same-ebl-portions-p clause candidate)
          (setf collection (nconc collection (list candidate)))
          (rplacd remaining (rest (rest remaining))))))))




;;;________________________________
;;; hi-lite-all-ebl-paths

(defun hi-lite-all-ebl-paths (clauses graph)
  (dolist (clause clauses)
    (hi-lite-clause-ebl-paths clause graph)))

;;;________________________________
;;; hi-lite-clause-ebl-paths

(defun hi-lite-clause-ebl-paths (clause graph)
  (let (derivation)
    (do ((literal clause (literal-next literal)))
        ((null literal))
    (setf derivation (literal-derivation literal))
    (when (equalp (derivation-type derivation) :ebl)
      (hi-lite-path-to-root (find-corresponding-node (derivation-path derivation)
                                                      (derivation-graph derivation) graph t)
                             :ebl)))))

;;;________________________________
;;; hi-lite-path-to-root

(defun hi-lite-path-to-root (node state)
  (when node
    (setf (node-state node) state)
    (hi-lite-path-to-root (node-consequent node) state)))

;;;________________________________
;;; select-frontier

(defun select-frontier (root)
  (apply-function-to-tree root #'(lambda (node)
                                   (if (and (not (equalp (node-state node) :unoperationalized))
                                            (or (null (node-antecedents node))
                                                (node-not? node)))
                                     (select-node node nil t)))))

;;;________________________________
;;; select-nodes-to-delete

(defun nodes-for-same-r-struct (node graph)
  (let ((r-struct (node-r-struct node)))
    (mapcan #'(lambda (n) (when (eq (node-r-struct n) r-struct) (list n))) (graph-used-nodes graph))))


;;;________________________________
;;; select-nodes-to-delete

(defun select-nodes-to-delete (graph)
  (mapc #'select-node
        (delete-if #'(lambda (node)
                       (some #'(lambda (n)
                                 (and (eq (node-kind node) (node-kind n))
                                      (eq (node-r-struct node) (node-r-struct n))
                                      (not (equalp (node-state n) :unoperationalized))
                                      (not (node-deleted? n))
                                      (eq (node-r-struct (node-consequent node)) (node-r-struct (node-consequent n)))
                                      (eq node (find-corresponding-node n (node-consequent n) (node-consequent node) t))
                                      ))
                             (graph-used-nodes graph)))
                   (deleted-and-unoperationalized-frontier (graph-a-base graph)))))

;;;________________________________
;;; collect-node-meeting-criteria

(defun collect-node-meeting-criteria (root criteria)
  (let ((collection nil))
    (apply-function-to-tree root #'(lambda (node)
                                     (when (funcall criteria node)
                                       (setf collection (push node collection)))))
    collection))

;;;________________________________
;;; deleted-and-unoperationalized-frontier

(defun deleted-and-unoperationalized-frontier (root)
  (collect-node-meeting-criteria root #'(lambda (node)
                                          (and (or (node-deleted? node)
                                                   (equalp (node-state node) :unoperationalized))
                                               (not (node-deleted? (node-consequent node)))
                                               (not (equalp (node-state (node-consequent node)) :unoperationalized))))))




#|
;;;_______________________________________
;;;  user-edit-theory

(defun user-edit-theory (r-struct &optional (create nil) (nodes nil))
  (let ((batch *batch*))
    (setf *batch* t)
    (unwind-protect
      (without-interrupts
       (set-cursor *watch-cursor*)
       (catch-cancel
         (let* ((name (r-name r-struct))
                (vars (r-vars r-struct))
                (graph (create-graph))
                (root (graph-root graph))
                a-base u-base antecedents window view title)
           (if create
             (setf a-base (get-node graph :consequent root :vars vars :r-struct r-struct :kind :intensional)
                   antecedents nil
                   title (format nil "Create ~(~A~)" name))
             (setf a-base (connect-literal graph root (cons name vars) nil nil nil)
                   antecedents (connect-clauses graph a-base (all-antecedents r-struct vars) nil nil)
                   title (format nil "Edit ~(~A~)" name)))
           (setf window (make-instance 'theory-edit-window :window-title title :nodes nodes)
                 view (graph-view window)
                 (expand view) :always
                 (root view) root
                 (graph view) graph
                 (graph-views graph) (push view (graph-views graph))
                 u-base (get-node graph :consequent root :kind :intensional)
                 (node-antecedents root) (list (list a-base) (list u-base))
                 (node-antecedents a-base) antecedents
                 (node-recursive? a-base) nil)
           (get-cell view :text "u-base" :node u-base :hidden? t)
           (get-cell view :text (name-vars-string name vars) :node a-base)
           (when (and nodes (some #'node-coverage nodes))
             (multiple-value-bind (pos neg old-vars) (create-editor-tuples a-base nodes)
               (declare (ignore pos neg))
               (setf (edit-window-old-vars window) old-vars)))
           (window-fixup window t)
           (window-select window))))
      (setf *batch* batch))))
|#