
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

;;;_______________________________________________
;;; GET-VARIABILIZATIONS

(defun get-variabilizations
       (r-struct
        max-new-vars
        old-vars
        predicate-being-learned
        old-types
        original-vars
        &key (cache? t))
  (let ((key (list old-vars old-types max-new-vars)))
    (or (and cache?
             (gethash key (r-variabilizations r-struct)))
        (setf (gethash key (r-variabilizations r-struct))
              (generate-variabilizations r-struct predicate-being-learned
                                         old-vars old-types max-new-vars original-vars)))))

;;;_______________________________________________
;;; GENERATE-VARIABILIZATIONS
;;;
;;; For each relation, this generates all the variabilizations of
;;; that relation consistent with type, mode, commutativity and other constraints.

(defun generate-variabilizations 
    (r-struct                      ;generate variabilizations for this r-struct
     predicate-being-learned       ;used to detect recursion
     old-vars                      ;list of old variables
     old-types                     ;types of old variables
     maximum-new-vars              ;maximum number of new variables allowed
     original-vars                 ;used to detect recursion
     )
      
  (let ((arity (r-arity r-struct))
        (type (r-type r-struct))
        (mode (r-mode r-struct))
        (constraint (r-constraint r-struct))
        (commutative (r-commutative r-struct))
        (induction (r-induction r-struct))
        (all-variabilizations nil)
        (struct-list nil))

    (when induction
      (setf all-variabilizations (get-typed-and-moded-variabilizations
                                  type mode old-vars old-types 
                                  (make-new-vars (min (- arity 1) maximum-new-vars))))

      (do* ((variabilizations all-variabilizations (rest variabilizations))
            (variabilization (first all-variabilizations) (first variabilizations)))
           ((null variabilization) (reverse struct-list))
        
        (if (and (not (infinite-recursive-call r-struct predicate-being-learned
                                               variabilization original-vars))
                 (or (null commutative)
                     (check-commutative variabilization))
                 (or (null constraint) ; otherwise constraint must be :unique-vars
                     (not (contains-repeated-var variabilization))))
          (push (make-variabilization-struct :variabilization variabilization)
                struct-list))))))

;;;_______________________________________________
;;; MAKE-NEW-VARS

(defun make-new-vars (number &optional (start -1))
  (do ((i (- start number -1) (incf i))
       (result nil))
      ((> i start) result)
    (push (make-pcvar :id i) result)))


;;;___________________________________________
;;;  MAKE-OLD-VARS

#|
(defun make-old-vars (number &optional (start 0))
  (do ((i (- (+ number start) 1) (decf i))
       (result nil))
      ((< i start) result)
    (push (make-pcvar :id i) result)))
|#

(defvar *old-vars-table* (make-hash-table :test #'equal :size 100))

(defun make-old-vars (number &optional (start 0))
  (if (zerop start)
    (or (gethash number *old-vars-table*)
        (setf (gethash number *old-vars-table*) (real-make-old-vars number start)))
    (real-make-old-vars number start)))

(defun real-make-old-vars (number &optional (start 0))
  (do ((i (- (+ number start) 1) (decf i))
       (result nil))
      ((< i start) result)
    (push (make-pcvar :id i) result)))

;;;_______________________________________________
;;; INFINITE-RECURSIVE-CALL

(defun infinite-recursive-call (r-struct predicate-being-learned variabilization original-vars)
       (and (eq r-struct predicate-being-learned)
            (every #'(lambda (v) (or (member v original-vars :test #'var-eq) (new-var? v)))
                   variabilization)))

;;;_______________________________________________
;;; CHECK-COMMUTATIVE
;;;
;;; assumes a total ordering derivable from the names of variables (var-greater-eq)
;;; then a pair of variables (v1,v2) passes this test if according to that ordering
;;; (>= v1 v2) so far, commutative is only checked for predicates of arity 2.

(defun check-commutative (variabilization)
  (cond ((not (= (length variabilization) 2)) t)
        (t (var-greater-eq (first variabilization) (second variabilization)))))

;;;_______________________________________________
;;; VAR-GREATER-EQ

(defun var-greater-eq (var1 var2)
  (string>= (princ-to-string (pcvar-id var1)) (princ-to-string (pcvar-id var2))))


;;;_______________________________________________
;;; VAR-LESS-THAN

(defun var-less-than (var1 var2)
  (string< (princ-to-string (pcvar-id var1)) (princ-to-string (pcvar-id var2))))


;;;_______________________________________________
;;; CONTAINS-REPEATED-VAR

(defun contains-repeated-var (list)
  (cond ((null (rest list)) nil)
        ((member (first list) (rest list) :test #'var-eq) t)
        (t (contains-repeated-var (rest list)))))

;;;_______________________________________________
;;; GET-TYPED-AND-MODED-VARIABILIZATIONS

(defun get-typed-and-moded-variabilizations (type mode old-vars old-types new-vars)
   (let ((key (list old-vars old-types new-vars type mode)))
    (or (gethash key *variabilization-hash-array*)
        (setf (gethash key *variabilization-hash-array*)
              (generate-typed-and-moded-variabilizations type mode old-vars old-types new-vars)))))

;;;_______________________________________________
;;; GENERATE-TYPED-AND-MODED-VARIABILIZATIONS

(defun generate-typed-and-moded-variabilizations (type mode old-vars old-types new-vars)
  (delete-if #'(lambda (variablization) (every #'new-var? variablization))
             (typed-and-moded-variablizations type mode old-vars old-types new-vars)))

;;;_______________________________________________
;;; TYPED-AND-MODED-VARIABILIZATIONS

(defun typed-and-moded-variablizations (type mode old-vars old-types new-vars)
  (if (null type)
    '(())
    (case (first mode)
      (:+ (vzs-with-first-slot-bound type mode old-vars old-types new-vars))
      (:- (vzs-with-first-slot-new type mode old-vars old-types new-vars))
      (otherwise 
       (append
        (vzs-with-first-slot-new type mode old-vars old-types new-vars)
        (vzs-with-first-slot-bound type mode old-vars old-types new-vars))))))

(defun vzs-with-first-slot-bound (type mode old-vars old-types new-vars)
  (let* ((slot-type (first type))
         (rest-type (rest type))
         (rest-mode (rest mode))
         (endings (typed-and-moded-variablizations rest-type rest-mode old-vars old-types new-vars)))
    (mapcan #'(lambda (var type)
                (when (type-eq type slot-type)    ;;; changed eq to type-eq
                  (mapcar #'(lambda (ending) (cons var ending)) endings)))
              old-vars old-types)))

(defun vzs-with-first-slot-new (type mode old-vars old-types new-vars)
  (if (null new-vars)
    nil
    (let* ((new-var (first new-vars))
           (new-new-vars (rest new-vars))
           (rest-type (rest type))
           (rest-mode (rest mode))
           (new-old-vars (append old-vars (list new-var)))
           (new-old-types (append old-types (list (first type)))))
      (mapcar #'(lambda (ending) (cons new-var ending))
              (typed-and-moded-variablizations
               rest-type rest-mode new-old-vars new-old-types new-new-vars)))))

;;;_______________________________________________
;;; TYPED-TUPLES

(defun typed-tuples (type vars var-types)
   (if (null type)
     '(())
     (let ((slot-type (first type))
           (endings (typed-tuples (rest type) vars var-types)))
       (mapcan #'(lambda (var var-type)
                   (when (type-eq var-type slot-type)     ;;; changed eq to type-eq
                     (mapcar #'(lambda (ending) (cons var ending)) endings)))
               vars var-types))))