;;;; copyright (c) 1990, 1991 by the university of california, irvine. 
;;;; this program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  this program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the regents of the university of california.
;;;; this program was written by michael pazzani, cliff brunk, glenn silverstein
;;;; and kamal ali.  

(in-package :user)

(defun concat2 (list)
  (apply 'concatenate (cons 'string (mapcar 'princ-to-string list))))

(defun string-concat (&rest list)
  (apply 'concatenate (cons 'string (mapcar 'princ-to-string list))))

(defun symbol-concat (&rest list)
  (intern (concat2 list)))

(defun previous (list element)
  (do* ((prev list next)
        (next (rest prev) (rest prev)))
       ((or (null prev) (equal element (first next))) (first prev))))

(defun contains? (container thing)
  (cond ((equal container thing) t)
        ((consp container)
         (or (contains? (first container) thing)
             (contains? (rest container) thing)))))

(defun goal-concept-name () (getf (rest *focl-problem*) :goal-concept-name))

(defun predicate-being-learned () (first *focl-problem*))

(defun r-name< (r-struct1 r-struct2)
  (string< (symbol-name (r-name r-struct1)) (symbol-name (r-name r-struct2))))

(defun universal< (x y)
  (cond ((and (numberp x) (numberp y)) (< x y))
        ((and (symbolp x) (symbolp y)) (string< (symbol-name x) (symbol-name y)))
        (t (string< (format nil "~S" x) (format nil "~S" y)))))

(defun add-r-struct (r-struct)
  (setf *r-structs* (delete-if #'(lambda (r) (eq (r-name r) (r-name r-struct))) *r-structs*)
        *r-structs* (insert-inorder r-struct *r-structs* #'r-name<))
  (when (fboundp 'update-relations) (update-relations)))

(defun user-defined-r-structs ()
  (remove-if #'(lambda (r-struct) (member r-struct *special-r-structs*)) *r-structs*))

(defun some-relation-is-user-defined ()
  (not (every #'(lambda (r-struct) (member r-struct *special-r-structs*)) *r-structs*)))

(defun insert-inorder (element ordered-list &optional (order-function #'universal<))
  (let ((inserted? nil))
    (do ((prev-cons nil current-cons)
         (current-cons ordered-list (rest current-cons)))
        (inserted?)
      (cond ((null current-cons)
             (setq ordered-list (nconc ordered-list (list element))
                   inserted? t))
            ((funcall order-function element (first current-cons))
             (if prev-cons
               (rplacd prev-cons (cons element current-cons))
               (setq ordered-list (cons element ordered-list)))
             (setq inserted? t))))
    ordered-list))

;;;_____________________________________
;;;  OBJECT-VARIARBLE-TYPE?

(defun object-variarble-type? (type) (get type :object))

;;;_____________________________________
;;;  COLLECT-TYPES

(defun collect-types (&optional (r-structs *r-structs*))
  (let ((collected-types (copy-list *all-types*)))
    (dolist (r r-structs)
      (dolist (type (r-type r))
        (pushnew type collected-types)))
    (sort collected-types #'universal< )))

;;;_____________________________________
;;;  USER-DEFINED-TYPES

(defun user-defined-types ()
  (delete-if #'(lambda (type) (member type *special-types*)) (collect-types)))

;;;_______________________________________
;;; DEFINE-SPECIAL-R-STRUCTS

(defun define-special-r-structs ()
  (setf *special-r-structs*
        (sort 
         (list
          (make-r :name '< :arity 2
                  :vars '(?number1 ?number2) :type '(:number :number) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?number1 is less than ?number2) :function #'<
                  :prolog-function #'</2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil :sort-fn #'< :infix t)
          
          (make-r :name '<= :arity 2
                  :vars '(?number1 ?number2) :type '(:number :number) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?number1 is less than or equal to ?number2) :function #'<=
                  :prolog-function #'<=/2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil :sort-fn #'< :infix t)
          
          (make-r :name 'math-= :arity 2
                  :vars '(?number1 ?number2) :type '(:number :number) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?number1 is equal to ?number2) :function #'=
                  :prolog-function #'math-=/2 :constraint :unique-vars :commutative t :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? t :sort-fn #'<)
          
          (make-r :name '> :arity 2
                  :vars '(?number1 ?number2) :type '(:number :number) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?number1 is greater than ?number2) :function #'>
                  :prolog-function #'>/2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil :sort-fn #'< :infix t)
          
          (make-r :name '>= :arity 2
                  :vars '(?number1 ?number2) :type '(:number :number) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?number1 is greater than or equal to ?number2) :function #'>=
                  :prolog-function #'>=/2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil :sort-fn #'< :infix t)
          
          (make-r :name 'eql :arity 2
                  :vars '(?variable1 ?variable2) :type '(:anything :anything) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?variable1 is equivalent to ?variable2) :function #'eql
                  :prolog-function #'eql/2 :constraint :unique-vars :commutative t :treat-as-commutative t
                  :induction nil :try-constants nil :equality? t :sort-fn #'universal<)
          
          (make-r :name '= :arity 2
                  :vars '(?variable1 ?variable2) :type '(:anything :anything) :mode '(:? :?)
                  :kind := :determinacy nil :questions '(?variable1 unifies with ?variable2)
                  :prolog-function #'=/2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? t :sort-fn #'universal< :infix t)
          
          (make-r :name 'is :arity 2
                  :vars '(?variable ?expression) :type '(:anything :expression) :mode '(:? :+)
                  :kind :is :determinacy nil :questions '(?variable1 is assigned the value of ?variable2)
                  :prolog-function #'is/2 :constraint :unique-vars :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil :sort-fn nil :infix t)
          
          (make-r :name 'string-lessp :arity 2
                  :vars '(?string1 ?string2) :type '(:string :string) :mode '(:+ :+)
                  :kind :builtin :determinacy nil :questions '(?string1 alphabetically preceeds ?string2) :function #'string-lessp
                  :prolog-function #'string-lessp/2 :constraint :unique-vars :commutative t :treat-as-commutative t
                  :induction nil :try-constants nil :equality? nil :sort-fn #'universal<)
          
          (make-r :name 'setof :arity 3
                  :vars '(?expression ?goal ?results) :type '(:expression :goal :anything) :mode '(:+ :+ :-)
                  :kind :is :determinacy nil :questions '(The solutions to ?goal using '?var are ?results)
                  :prolog-function #'setof/3 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'bagof :arity 3
                  :vars '(?expression ?goal ?results) :type '(:expression :goal :anything) :mode '(:+ :+ :-)
                  :kind :is :determinacy nil :questions '(The solutions to ?goal using '?var are ?results)
                  :prolog-function #'bagof/3 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'find-proofs :arity 2
                  :vars '(?goals ?results) :type '(:goal :anything ) :mode '(:+  :-)
                  :kind :is :determinacy nil :questions '(The satisfied conditions of ?goals are ?results)
                  :prolog-function #'find-proofs/2 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'multi-value-query :arity 1
                  :vars '(?expression) :type '(:expression) :mode '(:+ )
                  :kind :is :determinacy nil :questions '()
                  :prolog-function #'multi-value-query/1 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'call :arity 1
                  :vars '(?goals) :type '(:goal) :mode '(:+ )
                  :kind :is :determinacy nil :questions '()
                  :prolog-function #'call/1 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'not :arity -1
                  :vars nil :type '(:goals) :mode '(:+)
                  :kind :not :determinacy nil :questions '()
                  :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'and :arity -1
                  :vars nil :type '(:goals) :mode '(:+)
                  :kind :and :determinacy nil :questions '()
                  :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'or :arity -1
                  :vars nil :type '(:goals) :mode '(:+)
                  :kind :or :determinacy nil :questions '()
                  :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name '! :arity 0
                  :vars nil :type nil :mode nil
                  :kind :cut :determinacy nil :questions '()
                  :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          
          (make-r :name 'fail :arity 0
                  :vars nil :type nil :mode nil
                  :kind :fail :determinacy nil :questions '()
                  :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          #|
          (make-r :name 'call-and :arity 1
                  :vars '(?goals) :type '(:goals) :mode '(:+ )
                  :kind :is :determinacy nil :questions '()
                  :prolog-function #'call-and/1 :constraint nil :commutative nil :treat-as-commutative nil
                  :induction nil :try-constants nil :equality? nil)
          |#
          )
         #'r-name<)
        *special-types* (collect-types *special-r-structs*)
        ))

;;;_______________________________________
;;; INSTALL-SPECIAL-R-STRUCTS

(defun install-special-r-structs ()
  (mapc #'(lambda (r-struct)
            (set-r-struct (r-name r-struct) r-struct)
            (setf *r-structs* (insert-inorder r-struct *r-structs* #'r-name<)))
        *special-r-structs*)
  (setf *builtin-preds* (mapcan #'(lambda (r-struct)
                                    (when (eq (r-kind r-struct) :builtin)
                                      (setf (r-induction r-struct) nil
                                            (r-try-constants r-struct) nil)
                                      (list (cons (r-name r-struct) r-struct))))
                                *special-r-structs*)))

;;;_______________________________________
;;; DELETE-R-STRUCT

(defun delete-r-struct (name-or-struct)
  (let ((r-struct (if (r-p name-or-struct) name-or-struct (get-r-struct name-or-struct))))
    (when (and (r-p r-struct) (not (member r-struct *special-r-structs*)))
      (let* ((name (r-name r-struct))
             (special-r-struct (find name *special-r-structs* :key #'r-name)))
        (setf *intensional-preds* (delete name *intensional-preds* :key #'first)
              *extensional-preds* (delete name *extensional-preds* :key #'first)
              *builtin-preds* (cond ((and special-r-struct (builtin-p special-r-struct))
                                     (rplacd (find name *builtin-preds* :key #'first) special-r-struct)
                                     *builtin-preds*)
                                    (t (delete name *builtin-preds* :key #'first)))
              *r-structs* (if (and special-r-struct (builtin-p special-r-struct))
                            (nsubstitute special-r-struct r-struct *r-structs*)
                            (delete r-struct *r-structs*)))
        (when (fboundp 'update-relations) (update-relations))
        (reset-r-struct r-struct)))))

;;;_______________________________________
;;; RESET-R-STRUCT

(defun reset-r-struct (r-struct)
  (when (and (r-p r-struct) (not (member r-struct *special-r-structs*)))
    (set-r-struct (r-name r-struct) nil)
    (set-clauses (r-name r-struct) nil)
    (setf (get (r-name r-struct) 'prolog-predicate-name) nil)
    (setf (r-name r-struct) nil
          (r-arity r-struct) nil
          (r-vars r-struct) nil
          (r-type r-struct) nil
          (r-mode r-struct) nil
          (r-determinacy r-struct) nil
          (r-kind r-struct) nil
          (r-questions r-struct) nil
          (r-nodes r-struct) nil
          (r-equality? r-struct) nil
          (r-from-r-struct r-struct) nil
          (r-from-cliche r-struct) nil
          (r-pos r-struct) nil
          (r-neg r-struct) nil
          (r-pos-hash r-struct) nil
          (r-neg-hash r-struct) nil
          (r-neg-hash-for-negated r-struct) nil
          (r-slot-value-hash  r-struct) nil
          (r-clauses r-struct) nil
          (r-prolog-function r-struct) nil
          (r-function r-struct) nil
          (r-variabilizations r-struct) nil
          (r-sort-fn r-struct) nil
          (r-constraint r-struct) nil
          (r-commutative r-struct) nil
          (r-treat-as-commutative r-struct) nil
          (r-induction r-struct) nil
          (r-try-constants r-struct) nil)))

;;;_______________________________________
;;; RESET-FACTS

(defun reset-facts ()
  (dolist (bucket *extensional-preds*)
    (delete-r-struct (rest bucket)))
  (setf *extensional-preds* nil
        *predicate-being-learned* nil
        *all-types* nil)
  (when (fboundp 'update-types) (update-types))
  'FACTS_RESET)

;;;_______________________________________
;;; RESET-BUILTINS

(defun reset-builtins ()
  (dolist (bucket *builtin-preds*)
    (delete-r-struct (rest bucket)))
  (setf *builtin-preds* nil)
  'RULES_BUILTINS)

;;;_______________________________________
;;; RESET-RULES

(defun reset-rules ()
  (dolist (bucket *intensional-preds*)
    (delete-r-struct (rest bucket)))
  (setf *intensional-preds* nil
        *goal-concept* nil)
  'RULES_RESET)

;;;_______________________________________
;;; RESET-PREDS

(defun reset-preds () (reset-relations))

;;;_______________________________________
;;; RESET-RELATIONS

(defun reset-relations ()
    (when (user-monitor-p *user-monitor*)
      (setf *user-monitor* (make-user-monitor)))
    (setf *example-templates* nil)
    (dolist (r-struct *r-structs*)
      (reset-r-struct r-struct))
    (dolist (type *all-types*)
      (mapc #'(lambda(p) (setf (get type p) nil)) *type-properties*)
      (set-type-instances type nil))
    (mapc #'fmakunbound *named-prolog-functions*)
    (setf *extensional-preds* nil
          *builtin-preds* nil
          *intensional-preds* nil
          *all-types* nil
          *named-prolog-functions* nil
          *r-structs* nil
          *goal-concept* nil
          *predicate-being-learned* nil
          *cliches-to-be-named* nil
          *named-cliches* nil
          *anonymous-cliches* nil
          *focl-problem* nil
          *focl-problems* nil
          *example-templates* nil
          *rules-changed* nil
          *facts-changed* nil
          *kb-file* nil
          *learned-description* nil
          *learned-description-head* nil)
    (when (boundp '*analyze-pred*) (setq *analyze-pred* nil))
    (when (boundp '*analyze-rule*) (setq *analyze-rule* nil))
    (clear-variables)
    (install-special-r-structs)
    (when (fboundp 'update-types) (update-types))
    (when (fboundp 'update-relations) (update-relations))
    (when (fboundp 'update-templates) (update-templates))
    (when (fboundp 'close-display-windows) (close-display-windows))
    (values))

;;;_______________________________________
;;;  MAKE-PROLOG-RULE

(defun make-prolog-rule (name arity clauses &key (induction nil) (old-vars nil))
  (let* ((old-vars (or old-vars (make-old-vars arity)))
         (prolog-def (mapcar #'(lambda (x) (cons (cons name old-vars) (convert-literals-to-prolog x))) clauses)))
    (eval `(def-rule ,name 
             :clauses ,prolog-def
             :induction ,induction))))

;;;_______________________________________
;;;  DEF-FROM-R-STRUCT

(defun def-from-r-struct (r-struct)
  (let* ((name (r-name r-struct))
         (vars (r-vars r-struct))
         (type (r-type r-struct))
         (mode (r-mode r-struct))
         (determinacy (r-determinacy  r-struct))
         (questions (r-questions r-struct))
         (pos (r-pos r-struct))
         (neg (r-neg r-struct))
         (kind (r-kind r-struct))
         (clauses (r-clauses r-struct))
         (induction (r-induction r-struct))
         (commutative (r-commutative r-struct))
         (constraint (r-constraint r-struct))
         (arity (length vars)) )
    (unless (= (length mode) arity) (setf mode nil))
    (unless (= (length type) arity) (setf type (make-list arity :initial-element :anything)))
    (case kind
      (:intensional
       (eval `(def-rule ,name :vars ,vars :type ,type :mode ,mode :determinacy ,determinacy :questions ,questions
                :clauses ,clauses :induction ,induction :commutative ,commutative :constraint ,constraint))
       (setf *RULES-CHANGED* t))
      (:extensional
       (eval `(def-pred ,name :vars ,vars :type ,type :mode ,mode :determinacy ,determinacy :questions ,questions
                :pos ,pos :neg ,neg :induction ,induction :commutative ,commutative :constraint ,constraint))
       (setf *FACTS-CHANGED* t)))
    (get-r-struct name)))

;;;_______________________________________
;;; RE-DEF-RULE

(defun re-def-rule (rule &key (name (r-name rule)) 
                           (vars (r-vars rule))
                           (type (r-type rule))
                           (mode (r-mode rule))
                           (clauses (if (equal vars (r-vars rule))
                                      (get-clauses name)
                                      (mapcar #'(lambda (c) (cons (cons name vars) (nsublis (mapcar #'cons (rest (first c)) vars) (rest c) :test #'equalp)))
                                              (get-clauses name))))
                           (induction (r-induction rule))
                           (constraint (r-constraint rule))
                           (commutative (r-commutative rule))
                           (deterimancy (r-determinacy rule))
                           (questions (if (equal vars (r-vars rule))
                                        (r-questions rule)
                                        (nsublis (mapcar #'cons (r-vars rule) vars) (r-questions rule) :test #'equalp)))
                           (show t))
    (eval `(def-rule ,name 
             :vars ,vars
             :type ,type
             :mode ,mode
             :clauses ,clauses
             :induction ,induction
             :constraint ,constraint
             :commutative ,commutative
             :determinacy ,deterimancy
             :questions ,questions))
    (setf *rules-changed* t)
    (when (fboundp 'update-relations) (update-relations))
    (when show (show-rule-def name))
    name)

;;;_______________________________________
;;; SHOW-RULE-DEF

(defun show-rule-def (name-or-struct &optional (stream t))
  (let (rule name)
    (if (rule-p name-or-struct)
      (setf rule name-or-struct
            name (r-name rule))
      (setf name name-or-struct
            rule (get-r-struct name)))
    (when (rule-p rule)
      (format stream "~%RULE DEFINITION: ~a   Variable Types: ~s~%" (name-vars-string name (r-vars rule)) (r-type rule))
      (dolist (clause (get-clauses name))
        (format stream "   IF (~a" (literal-string (cadr clause)))
        (dolist (literal (cddr clause))
          (format stream "~%       ~a" (literal-string literal)))
        (format stream ")~%     THEN ~a~%~%" (literal-string (first clause))))
      (when (r-questions rule)
        (format stream "English translation of conclusion:~%   ~a~%~%" (r-questions rule))))))

;;;_______________________________________
;;;  LITERAL-STRING

(defun literal-string (literal)
  (if (consp literal)
    (name-vars-string (first literal) (rest literal))
    (name-vars-string literal nil)))

;;;_______________________________________
;;;  NAME-VARS-STRING

(defun name-vars-string (name vars)
  (cond ((eq name '!) "!")
        ((eq name 'fail) "fail")
        ((or (eq name 'and)
             (eq name 'or)
             (eq name 'not))
         (if vars
           (format nil "(~(~S~) ~A)" name (clause-body-string vars))
           (format nil "~(~S~)" name)))
        (t
         (format nil "(~(~S~)~{ ~S~})" name vars))))

;;;_______________________________________
;;; CLAUSE-BODY-STRING

(defun clause-body-string (clause-body)
  (with-output-to-string (out)
    (do* ((literals clause-body next)
          (literal (first literals) (first literals))
          (next (rest literals) (rest literals)))
         ((null literals))
      (if (consp literal) 
        (format out "~a" (literal-string literal))
        (format out "~(~S~)" literal))
      (when next (format out " ")))))

;;;_______________________________________
;;; PRINT-LITERAL

(defun print-literal (literal stream depth)
  depth
  (print-literals literal stream)
  (format stream "."))
       
(defun print-literals (literal stream)
  (let (deleted? name variablization next)
    (do ((L literal next))
        ((null L) nil)
      (setf deleted? (literal-deleted? L)
            name (literal-predicate-name L)
            variablization (literal-variablization L)
            next (literal-next L))
      (when deleted? (format stream "[DELETED "))
      (cond ((literal-negated? L)
             (format stream "(not ")
             (print-literals (literal-negated-literals L) stream)
             (format stream ")"))
            (t (format stream "~A" (name-vars-string name variablization))))
      (format stream "~A~A" (if deleted? "]" "") (if next ", " ""))
      )))

;;;_______________________________________
;;; PRINT-R-STRUCT

(defun print-r-struct (r-struct stream depth) depth
  (cond ((pred-p r-struct) (print-pred r-struct stream depth))
        ((rule-p r-struct) (print-rule r-struct stream depth))
        (t (format stream "{~A ~A }" (r-kind r-struct) (name-vars-string (r-name r-struct) (r-vars r-struct))))))

;;;_______________________________________
;;; PRINT-PRED

(defun print-pred (r-struct stream depth) depth
  (format stream "{PRED ~A }" (name-vars-string (r-name r-struct) (r-vars r-struct))))

;;;_______________________________________
;;; PRINT-RULE

(defun print-rule (r-struct stream depth) depth
  (format stream "{RULE ~A ~A }" (name-vars-string (r-name r-struct) (r-vars r-struct)) (r-clauses r-struct)))
 
;;;_______________________________________
;;; PRINT-CLAUSE

(defun print-clause (clause stream depth) depth
  (format stream "{CLAUSE ~A ~A :- ~A }"
          (clause-number clause)
          (name-vars-string (if (clause-of clause) (r-name (clause-of clause)) (clause-head clause)) (clause-parameters clause))
          (clause-body-string (clause-body clause))))

;;;_______________________________________
;;; PRINT-LEARNED-DESCRIPTION

(defun print-learned-description (&optional (head *learned-description-head*)
                                            (clauses *learned-description*))
  (let ((name (r-name head))
        (vars (r-vars head)))
    (format t "~%Learned Definition of ~A:~%" name)
    (dolist (clause clauses)
      (format t "~% (~(~A~)~{ ~A~}) :- ~A" name vars clause)))
  (format t "~%~%")
  (values))

;;;_______________________________________
;;; UNIQUE-LEARNED-DESCRIPTION-NAME

(defun unique-learned-description-name (head) (unique-r-name (intern (format nil "~S__LEARNED_RULE" (r-name head)))))

;;;_______________________________________
;;; DEFINE-RULE-FOR-LEARNED-DESCRIPTION

(defun define-rule-for-learned-description (&optional (head *learned-description-head*)
                                                      (clauses *learned-description*))
  (let* ((prolog-head (cons (r-name head) (r-vars head)))
         (prolog-clauses (direct-substitute (mapcar #'(lambda (c) (cons prolog-head c)) (convert-literals-to-prolog clauses))
                                            (list (list (predicate-being-learned) (r-name *learned-description-head*))))))
    (eval `(def-rule ,(r-name head) 
             :vars ,(r-vars head)
             :type ,(r-type head)
             :mode ,(r-mode head)
             :clauses ,prolog-clauses
             :induction nil
             :constraint ,(r-constraint head)
             :commutative ,(r-commutative head)
             :determinacy ,(r-determinacy head)
             :questions nil)))
  (setf *rules-changed* t))

;;;________________________________________________________________
;;; TRANSFER-LITERAL-VARS
;;;
;;;  makes a copy of the argument to a literal, replacing new vars 
;;;  (i.e., not member of vars) with new bound vars returns a the replaced
;;;  argument, new vars, types of new vars, and an alist of (free . bound)
;;;  name for variable- bound names is 1 + number of current bound variables

(defun transfer-literal-vars (variabilization types old-vars &optional (next-id (length old-vars)) (existing-alist nil))
  (let* ((new nil)
         (new-vars nil)
         (new-types nil)
         (alist nil)
         (new-variabilization (mapcar 
                               #'(lambda (parameter type)
                                   (cond
                                    ((not (pcvar-p parameter)) parameter)
                                    ((member parameter old-vars :test #'var-eq) parameter)
                                    ((rest (assoc parameter alist :test #'var-eq)))
                                    ((rest (assoc parameter existing-alist :test #'var-eq)))
                                    (t (setf new (make-pcvar :id next-id))
                                       (incf next-id)
                                       (push (cons parameter new) alist)
                                       (push new new-vars)
                                       (push type new-types)
                                       new)))
                               variabilization types)))
    (values new-variabilization (nreverse new-vars) (nreverse new-types) alist)))

;;;________________________________________________________________
;;; TRANSFER-NEGATED-LITERAL-VARS
;;;
;;;  like transfer-literal-vars only new variables are left alone (usually with negative ids)

(defun transfer-negated-literal-vars (variabilization types old-vars &optional (existing-alist nil))
  (let* ((new-vars nil)
         (new-types nil)
         (new-variabilization (mapcar
                          #'(lambda (parameter type)
                              (cond ((not (pcvar-p parameter)) parameter)
                                    ((member parameter old-vars :test #'equalp) parameter)
                                    ((rest (assoc parameter existing-alist :test #'var-eq)))
                                    (t (push parameter new-vars)
                                       (push type new-types)
                                       parameter)))
                          variabilization types)))
    (values new-variabilization (nreverse new-vars) (nreverse new-types))))


;;;________________________________________________________________
;;; CONSTRUCT-LITERAL

(defun construct-literal (negated? r-struct variablization derivation-type)
  (cond
   ((is-op-p r-struct)
    (construct-literal negated? (get-r-struct 'is)
                       (list (first variablization)
                             (cons (r-function r-struct) (rest variablization)))
                       (make-derivation :type derivation-type)))
   (negated?
    (make-literal
     :negated? t
     :negated-literals (make-literal
                        :predicate-name (r-name r-struct)
                        :variablization variablization
                        :derivation (make-derivation :type derivation-type))
     :derivation (make-derivation :type derivation-type)))
   (t
    (make-literal
     :predicate-name (r-name r-struct)
     :variablization variablization
     :derivation (make-derivation :type derivation-type)))))

;;;________________________________________________________________
;;; CONJOIN-NOT-=

(defun conjoin-not-= (bound-variable-types bound-variables new-vars new-types literal pos-tuples  neg-tuples  source)
  (let ((new nil)
        (old literal))
    (unless (eq source :ebl)
      (mapc #'(lambda(nvar ntype)
                (when (object-variarble-type? ntype)
                  
                  (mapcar #'(lambda(var type)
                              (when (type-eq type ntype)
                                (setq new (make-literal :negated? t :prev literal
                                                        :derivation (make-derivation :type :object)
                                                        :negated-literals 
                                                        
				                        (make-literal :predicate-name '= 
						                      :derivation (make-derivation :type :object)
						                      :variablization (list var nvar)
                                                                      
			                                              :derivation (make-derivation :type :object))))
                                
                                (setf (literal-next literal) new)
	                        (when *save-examples*    
                                  (setf (literal-pos new) pos-tuples
                                        (literal-neg new) neg-tuples
                                        (literal-new-pos new) pos-tuples
                                        (literal-new-neg new) neg-tuples))
                                (setq literal new)))
                          bound-variables bound-variable-types)
                  ))
            new-vars new-types))
    old))

;;;________________________________________________________________
;;; CREATE-LITERAL

(defun create-literal (r-struct variabilization negated? source bound-variables pos-tuples neg-tuples &optional (bound-variable-types nil))
  (let (new-vars       ;; the list of "new" variables renamed to old-vars
        new-types      ;; the types of new vars
        literal-vars   ;; the variables in call with "new" vars renamed to "old" vars
        new-literal 
        new-pos-tuples
        new-neg-tuples)
    (if negated?
      (multiple-value-setq 
        (literal-vars new-vars new-types)
        (transfer-negated-literal-vars variabilization (r-type r-struct) bound-variables))
      (multiple-value-setq 
        (literal-vars new-vars new-types)
        (transfer-literal-vars variabilization (r-type r-struct) bound-variables (length bound-variables))))
    (setf new-literal (construct-literal negated? r-struct literal-vars source)
          new-pos-tuples (generalized-extend-tuples r-struct pos-tuples literal-vars negated? new-vars bound-variables)
          new-neg-tuples (generalized-extend-tuples r-struct neg-tuples literal-vars negated? new-vars bound-variables))
    (when *save-examples*    
      (setf (literal-pos new-literal) pos-tuples
            (literal-neg new-literal) neg-tuples
            (literal-new-pos new-literal) new-pos-tuples
            (literal-new-neg new-literal) new-neg-tuples))
    (cond (negated?
           (setq new-vars nil
                 new-types nil))
          (new-vars
           (setq new-literal (conjoin-not-= bound-variable-types bound-variables new-vars new-types new-literal  pos-tuples  neg-tuples source))))
    (values new-literal new-vars new-types new-pos-tuples new-neg-tuples)))

            
;;;________________________________________________________________
;;; CREATE-WINNER-LITERAL
;;;
;;;  converts a winner in "pred" form to a winner in literal form

(defun create-winner-literal (variables pos-tuples neg-tuples winner source &optional (variable-types nil))
  (multiple-value-bind
    (new-literal new-vars new-types new-pos-tuples new-neg-tuples)
    (create-literal (winner-literal winner)
                    (winner-vars winner)
                    (winner-negated? winner)
                    source
                    variables
                    pos-tuples
                    neg-tuples variable-types)
    (setf (winner-literal winner) new-literal
          (winner-vars winner) new-vars
          (winner-types winner) new-types
          (winner-pos winner) new-pos-tuples
          (winner-neg winner) new-neg-tuples))
  winner)


;;;_______________________________________
;;; ALL-TYPED-TUPLES

(defun all-typed-tuples (type-list)
   (if (null type-list) '(())
      (let ((short (all-typed-tuples (rest type-list))))
      (mapcan #'(lambda(new)
            (mapcar #'(lambda (old)
                  (cons new old))
               short))
            (get-type-instances (first type-list))))))

;;;_______________________________________
;;; MAKE-CLAUSE-STRUCTURE
;;;
;;;  make-clause-structure converts a prolog style rule (i.e., a list of prolog clauses)
;;;  to a list of clause-structures

(defun make-clause-structure (clauses &aux (clauseno 0))
  (mapcar #'(lambda(clause &aux (new (uniquify-variables clause)))
             (prog1 (make-clause :body (rest new)
                                 :number clauseno
                                 :neg-tuples-hash (make-hash-table :test #'equal :size 2)
                                 ;the negative hash table is small, since there
                                 ;is actually only 1 variabilization tested for each clause
                                 :head (first (first new))
                                 :new-vars (compute-new-vars
                                            (rest new)
                                            (rest (first new)))
                                 :parameters (rest (first new)))
               (incf clauseno)))
          clauses))

;;;_______________________________________
;;; NEG-HASH-TABLE
;;;
;;;   returns the appropriate hash table to store # of matches

(defun neg-hash-table (pred negative?)
  (if (clause-p pred)
      (clause-neg-tuples-hash pred) ;;negative shouldn't be true for clauses
    (if negative?
      (r-neg-hash-for-negated pred)
      (r-neg-hash pred))
    ))

;;;_____________________________________________________
;;; MAKE-POS-HASH

(defun make-pos-hash (pos)
  (let*
    ((table (make-hash-table :test #'equal :size (+ (length pos) 1))))
    (dolist (tuple pos table)
      (add-to-pos-hash tuple table))))

;;;_____________________________________________________
;;; ADD-TO-POS-HASH

(defun add-to-pos-hash (tuple table &optional (delete nil))
  (setf (gethash tuple table) (if delete nil (list tuple))))

;;;_______________________________________
;;; MAKE-SLOT-VALUE-HASH

(defun make-slot-value-hash (arity pos)
  (let ((array (make-array (list arity))))
    (dotimes (i arity)
      (setf (aref array i) (make-hash-table :test #'equal  ;;changed to equal so that 1.0 equal 1
                                            :size *init-slot-table-size*
                                            :rehash-size *slot-table-rehash-size*
                                            :rehash-threshold *slot-table-rehash-threshold*)))
    (dolist (tuple pos)
      (add-to-slot-value-hash array tuple arity))
    array))

;;;_______________________________________
;;; ADD-TO-SLOT-VALUE-HASH

(defun add-to-slot-value-hash(array tuple arity &optional (delete nil))
  (dotimes (i arity)
      (if delete
        (setf (gethash (nth i tuple) (aref array i))
              (delete tuple (gethash (nth i tuple) (aref array i)) :test #'equal))
        (pushnew tuple (gethash (nth i tuple) (aref array i))))))

;;;_______________________________________
;;;  ASSERT-FACT

(defun assert-fact (fact &optional (negated? nil) (pred-name (first fact))  (tuple (rest fact)))
  (when (or (eq pred-name '-) (eq pred-name 'not))
    (setf fact (second fact)
          negated? (not negated?)
          pred-name (first fact)
          tuple (rest fact)))
  (let ((pred (get-pred pred-name)))
    (cond (pred
           (cond (negated?
                  (pushnew tuple (r-neg pred))
                  (push fact *new-facts-neg*))
                 (t
                  (pushnew tuple (r-pos pred))
                  (add-to-slot-value-hash (r-slot-value-hash pred) tuple (length tuple))
                  (add-to-pos-hash tuple (r-pos-hash pred))
                  (push fact *new-facts-pos*)) ))
          (t
           (format t "~%~%Warning: the predicate ~(~s~) was created using defaults." pred-name)
           (if negated?
             (push fact *new-facts-neg*)
             (push fact *new-facts-pos*))
           (eval `(def-pred ,pred-name
                    :induction t
                    :pos ,(unless negated? (list tuple))
                    :neg ,(when negated? (list tuple))))))
    (setf *FACTS-CHANGED* t)
    tuple))

;;;_______________________________________
;;;  RETRACT-FACT

(defun retract-fact (fact &optional (negated? nil) (pred-name (first fact)) (tuple (rest fact)))
  (when (or (eq pred-name '-) (eq pred-name 'not))
    (setf fact (second fact)
          negated? (not negated?)
          pred-name (first fact)
          tuple (rest fact)))
  (let ((pred (get-pred pred-name)))
    (when pred
      (cond (negated?
             (setf (r-neg pred) (delete tuple (r-neg pred) :test #'equal)))
            (t
             (setf (r-pos pred) (delete tuple (r-pos pred) :test #'equal))
             (add-to-slot-value-hash (r-slot-value-hash pred) tuple (length tuple) t)
             (add-to-pos-hash tuple (r-pos-hash pred) t)) ))
    (setf *FACTS-CHANGED* t)
    tuple))

;;;_______________________________________
;;; COMPUTE-NEW-VARS

(defun compute-new-vars (body old-vars &optional (new-vars nil))
  (cond ((pcvar-p body) (if (or (member body old-vars :test #'var-eq)
                                (member body new-vars :test #'var-eq))
                          new-vars
                          (cons body new-vars)))
        ((consp body) (compute-new-vars (first body) old-vars (compute-new-vars (rest body) old-vars new-vars)))
        ((node-p body) (compute-new-vars (node-vars body) old-vars new-vars))
        ((literal-p body) (compute-new-vars (literal-variablization body) old-vars new-vars))
        (t new-vars)))

;;;_______________________________________
;;; COMPUTE-NEW

#|
(defun compute-new (variablization)
  (sort (delete-duplicates (remove-if-not #'new-var? variablization)) #'< :key #'pcvar-id))    ;;; This isn't very robust!!!
|#

(defun compute-new (variablization)
  (let ((ordered-new-vars nil))
    (dolist (var variablization)
      (when (new-var? var)
        (unless (member var ordered-new-vars :test #'var-eq)
          (setq ordered-new-vars (insert-inorder var ordered-new-vars #'(lambda (v1 v2) (< (pcvar-id v1) (pcvar-id v2))))))))
    ordered-new-vars))

;;;_______________________________________
;;; NEW-VAR?
;;;
;;;  return T if var is new.  A variable is new if it is
;;;  a member of new-vars (count-matches), or when new-vars
;;;  isn't supplied if its id is a number < 0 (extend-tuples).

(defun new-var? (var &optional (new-vars nil) &aux id)
  (when (pcvar-p var)
    (if new-vars
      (member var new-vars :test #'var-eq)
      (and (numberp (setq id (pcvar-id var))) (> 0 id)))))

;;;_______________________________________
;;; CLASSIFY-ARGUMENTS-FOR-PROLOG
;;;
;;;  input    a argument of the form (?1 ?2 a b)
;;;  returns :all-bound (if no vars) or an integer representing the position of
;;;                     the first constants (where 0 is the first variable), or
;;;          :all-vars  (if there are no constants)

(defun classify-arguments-for-prolog  (vs &aux (position 0) (first-const nil) (var nil))
  (dolist (i vs)
    (if (pcvar-p i)
      (setf var t)
      (unless first-const (setf first-const position)))
    (incf position))
  (if (null var)
    :all-bound
    (if first-const
      first-const
      :all-vars)))

;;;_______________________________________
;;; CLASSIFY-VARIABLIZATIONS-FOR-INDUCTION
;;;
;;;  like above, but it works on variabilizatiosn rather than arguments
;;;  in a variabilization $n indicates an old var, and $-n indicates a new var
;;;  Therefore return :all-bound if no new vars, or the position of the first old-var
;;;  (if there are new) or :all-vars if all new vars (used by (six ?x) in loan examples)

(defun classify-variabilization-for-induction  (vs &optional (new-vars nil))
  (let ((position 0)
        (first-const nil)
        (new nil))
    (dolist (i vs)
      (if (new-var? i new-vars)
        (setf new t)
        (unless first-const (setf first-const position)))
      (incf position))
    (if (null new)
      :all-bound
      (if first-const
        first-const
        :all-vars))))

;;;_______________________________________
;;; RETURN-ARGUMENT-HASH-TABLE
;;;
;;;  return hash table for argument (position) of pred)

(defun return-argument-hash-table (position pred)
  (aref (r-slot-value-hash pred) position))


;;;_______________________________________
;;; RETRIEVE-SUPERSET-OF-MATCHING-TUPLES
;;;
;;;  pred-  a pred struct
;;;  vs-    an instantiated variabilization (a b ?-1 ?-1 %-3)
;;;  key-   an positional index into vs (for positional value), or
;;;         :all-vars (only for prolog), or
;;;         :all-bound
;;;
;;;  NOTE: that equal hash array for all bound returns a list of one tuple
;;;         returns tuples of pred that may unify with vs

(defun retrieve-superset-of-matching-tuples (pred vs key)
  (case key
    (:all-vars (r-pos pred))
    (:all-bound (gethash vs (r-pos-hash pred)))
    (otherwise (gethash (nth key vs) (return-argument-hash-table key pred)))))

;;;_______________________________________
;;; MEMBCAR
;;;
;;;  accepts an element (e) and a list l and returns true e is the car of any element of the list

(defun membcar (e l) (member e l :test #'(lambda (e1 e2) (and (consp e2) (eql e1 (first e2))))))


(defun undefined? (literal) (null (get-r-struct (first literal))))
(defun extensional? (literal) (get-pred (first literal)))
(defun intensional? (literal) (get-rule (first literal)))
(defun builtin? (literal) (get-builtin (first literal)))
(defun builtin-fn? (literal) (get-builtin-fn (first literal)))
(defun negation? (literal) (eq (first literal) 'not))
(defun equality? (literal) (eq (first literal) '=))
(defun is? (literal) (eq (first literal) 'is))
(defun cut? (literal) (eq literal '!))


;;;_______________________________________
;;; RESET-HASH-TABLES

(defun reset-hash-tables()
  (mapc #'(lambda (name.pred)
            (setf (r-neg-hash (rest name.pred)) (make-hash-table :test #'equal :size (* (r-arity (rest name.pred)) 10))
                  (r-neg-hash-for-negated (rest name.pred)) (make-hash-table :test #'equal :size (* (r-arity (rest name.pred)) 10))))
        *extensional-preds*)
  
  (mapc #'(lambda (name.rule)
            (setf (r-neg-hash (rest name.rule)) (make-hash-table :test #'equal :size (* (r-arity (rest name.rule)) 10))
                  (r-neg-hash-for-negated (rest name.rule)) (make-hash-table :test #'equal :size (* (r-arity (rest name.rule)) 10))))
        *intensional-preds*)
  
  (mapc #'(lambda(name.builtin)
            (setf (r-neg-hash (rest name.builtin)) (make-hash-table :test #'equal :size (* (r-arity (rest name.builtin)) 10))
                  (r-neg-hash-for-negated (rest name.builtin)) (make-hash-table :test #'equal :size (* (r-arity (rest name.builtin)) 10))))
        *builtin-preds*))


;;;_______________________________________
;;; COUNT-EXAMPLES

(defun count-examples (tuples &optional (arity (r-arity *learned-description-head*)))
  (length (remove-duplicates (mapcar #'(lambda (i) (subseq i 0 arity)) tuples) :test #'equal)))

;;;_______________________________________
;;;  LITERAL UTILITIES

(defun last-literal (literal)
  (let ((next (literal-next literal)))
    (if next (last-literal next) literal)))

(defun first-undeleted-literal (literal)
  (if (literal-deleted? literal) (next-literal literal) literal))

(defun last-undeleted-literal (literal)
  (let ((next (next-literal literal)))
    (if next (last-undeleted-literal next) literal)))

(defun next-literal (literal)
  (let ((next (literal-next literal)))
    (when next (if (literal-deleted? next) (next-literal next) next))))

(defun prev-literal (literal)
  (let ((prev (literal-prev literal)))
    (when prev (if (literal-deleted? prev) (prev-literal prev) prev))))

(defun first-undeleted-or-analytical-literal (literal)
  (if (or (not (literal-deleted? literal))
          (eq (derivation-type (literal-derivation literal)) :ebl))
    literal
    (next-undeleted-or-analytical-literal literal)))
           
(defun next-undeleted-or-analytical-literal (literal)
  (let ((next (literal-next literal)))
    (when next
      (if (or (not (literal-deleted? next))
              (eq (derivation-type (literal-derivation next)) :ebl))
        next
        (next-undeleted-or-analytical-literal next)))))


;;;_______________________________________
;;;  COPY-LITERALS
;;;
;;;  copies a clause of literal structs by copying each individual literal struct

(defun copy-literals (clause)
  (let ((copy (copy-literal clause)))
    (do* ((literal clause (literal-next literal))
          (prev-copy copy next-copy)
          (next-copy (if (literal-next literal) (copy-literal (literal-next literal)))
                     (if (literal-next literal) (copy-literal (literal-next literal)))))
         ((null next-copy) copy)
      (setf (literal-next prev-copy) next-copy)
      (if next-copy (setf (literal-prev next-copy) prev-copy)))))

;;;_______________________________________
;;;  REMOVE-DELETED-LITERALS

(defun remove-deleted-literals (clause pos-tuples neg-tuples old-variables)
  (let ((new-clause (first-undeleted-literal clause)))
    (do ((prev nil literal)
         (literal new-clause (literal-next literal)))
        ((null literal))
      (setf (literal-prev literal) prev
            (literal-next literal) (next-literal literal)))
    (when *save-examples*
      (insert-tuples new-clause pos-tuples neg-tuples old-variables))
    new-clause))

;;;_______________________________________
;;;  REMOVE-DELETED-EMPIRICAL-LITERALS

(defun remove-deleted-empirical-literals (clause pos-tuples neg-tuples old-variables)
  (let ((new-clause (first-undeleted-or-analytical-literal clause)))
    (do* ((prev nil literal)
          (literal new-clause (literal-next literal)))
         ((null literal))
      (setf (literal-prev literal) prev
            (literal-next literal) (next-undeleted-or-analytical-literal literal)))

    (when *save-examples*
      (insert-tuples new-clause pos-tuples neg-tuples old-variables))
     new-clause))

;;;______________________________________
;;; INSERT-TUPLES

(defun insert-tuples (nodes-or-literals pos-tuples neg-tuples old-vars &optional (from :learning))
  (cond ((null nodes-or-literals))
        ((or (node-p nodes-or-literals)
             (conjunction-p nodes-or-literals)
             (disjunction-p nodes-or-literals))
         (insert-node-tuples nodes-or-literals pos-tuples neg-tuples old-vars nil from nil t))
        ((literal-p nodes-or-literals)
         (insert-literal-tuples nodes-or-literals pos-tuples neg-tuples old-vars))))

;;;_______________________________________
;;; INSERT-LITERAL-TUPLES

(defun insert-literal-tuples (literal pos-tuples neg-tuples old-variables &optional (save-examples *save-examples*))
  (cond ((null literal) (values pos-tuples neg-tuples old-variables))
        ((literal-deleted? literal) (insert-literal-tuples (literal-next literal) pos-tuples neg-tuples old-variables save-examples))
        ((literal-negated? literal) (multiple-value-bind (extended-pos extended-neg) (insert-literal-tuples (literal-negated-literals literal) pos-tuples neg-tuples old-variables save-examples)
                                      (setf extended-pos (return-originals-not-extended pos-tuples extended-pos)
                                            extended-neg (return-originals-not-extended neg-tuples extended-neg))
                                      (when save-examples    
                                        (setf (literal-pos literal) pos-tuples
                                              (literal-neg literal) neg-tuples
                                              (literal-new-pos literal) extended-pos
                                              (literal-new-neg literal) extended-neg))
                                      (insert-literal-tuples (literal-next literal) extended-pos extended-neg old-variables save-examples)))
        (t
         (let ((new-vars (compute-new-vars literal old-variables)))
           (multiple-value-bind (extended-pos extended-neg) (extend-tuples literal pos-tuples neg-tuples old-variables nil new-vars)
             (when (eq extended-pos :uses-undefined-relation) (setf extended-pos nil))
             (when save-examples    
               (setf (literal-pos literal) pos-tuples
                     (literal-neg literal) neg-tuples
                     (literal-new-pos literal) extended-pos
                     (literal-new-neg literal) extended-neg))
             (insert-literal-tuples (literal-next literal) extended-pos extended-neg (append old-variables new-vars) save-examples))))))

;;;_______________________________________
;;;  CONSTANT-TYPE-EQ

(defun constant-type-eq (arg type)
  (if (or (eq type :anything)
          (eq type :experssion)
          (and (eq type :number) (numberp arg))
          (and (eq type :goal) (listp arg))
          (and (eq type :goals) (listp arg))
          (member arg (get-type-instances type) :test #'equalp))
    t
    nil))

;;;_______________________________________
;;;  TYPE-EQ

(defun type-eq (type1 type2)
  (or (equalp type1 type2)
      (equalp type1 :anything)
      (equalp type2 :anything)
      (equalp type1 :goal)    ;;; These are a temporary hack to avoid type clashes
      (equalp type2 :goal)
      (equalp type1 :goals)
      (equalp type2 :goals)
      (equalp type1 :expression)
      (equalp type2 :expression)))

;;;_______________________________________
;;;  TYPE-EQUAL

(defun type-equal (types1 types2)
  (every #'type-eq types1 types2))


;;;=====================================================
;;;  REPRESENTATION CONVERTORS
;;;=====================================================

;;;_______________________________________
;;;  CONVERT-TO-NODES

(defun convert-to-nodes (winner vars negated? instantiated-cliche source graph)
  (cond (instantiated-cliche (convert-cliche-to-nodes instantiated-cliche winner vars negated? graph))
        ((r-p winner) (convert-r-struct-to-node winner vars negated? source graph))
        ((node-p winner) winner)
        ((conjunction-p winner) winner)
        ((disjunction-p winner) winner)
        ((literal-p winner) (convert-literals-to-nodes winner graph))
        ((literal-disjunction-p winner) (convert-literals-to-nodes winner graph))
        ((consp winner) (convert-prolog-to-nodes winner source graph))
        (t nil)))

;;;_______________________________________
;;;  CONVERT-R-STRUCT-TO-NODE

(defun convert-r-struct-to-node (r-struct vars negated? source graph)
  (let ((node (get-node graph :r-struct r-struct :vars vars :kind (if (r-p r-struct) (r-kind r-struct) :undefined) :state source)))
    (when negated?
      (setf node (negate-node node graph)
            (node-state node) source))
    node))

;;;_______________________________________
;;;  CONVERT-CLICHE-TO-NODES

(defun convert-cliche-to-nodes (instantiated-cliche last-r-struct last-vars negated? graph)
  (nconc (mapcar #'(lambda (literal-info)
                     (convert-r-struct-to-node (literal-info-pred literal-info)
                                               (literal-info-variabilization literal-info)
                                               (literal-info-negated? literal-info) :cliche graph))
                 (butlast instantiated-cliche))
         (list (convert-r-struct-to-node last-r-struct last-vars negated? :cliche graph))))

;;;_______________________________________
;;;  CONVERT-PROLOG-TO-NODES

(defun convert-prolog-to-nodes (prolog source graph)
  (cond ((null prolog) nil)
        ((eql prolog '!) (get-node graph :kind :cut :state source))
        ((eql prolog 'fail) (get-node graph :kind :fail :state source))
        ((consp (first prolog)) (cons (convert-prolog-to-nodes (first prolog) source graph)
                                      (convert-prolog-to-nodes (rest prolog) source graph)))
        ((eql (first prolog) 'not) (let ((nodes (convert-prolog-to-nodes (rest prolog) source graph)))
                                     (cond ((node-p nodes) (negate-node nodes graph))
                                           ((conjunction-p nodes) (negate-conjunction nodes graph))
                                           (t nil))))
        ((eql (first prolog) 'and) (let ((nodes (convert-prolog-to-nodes (rest prolog) source graph)))
                                     (cond ((node-p nodes) (conjoin-nodes-with-and-node (list nodes) graph))
                                           ((conjunction-p nodes) (conjoin-nodes-with-and-node nodes graph))
                                           (t nil))))
        ((eql (first prolog) 'or) (let ((nodes (convert-prolog-to-nodes (rest prolog) source graph)))
                                     (cond ((node-p nodes) (disjoin-nodes-with-or-node (list nodes) graph))
                                           ((conjunction-p nodes) (disjoin-nodes-with-or-node nodes graph))
                                           (t nil))))
        (t (convert-r-struct-to-node (get-r-struct (first prolog)) (rest prolog) nil source graph))))

;;;_______________________________________
;;;  CONVERT-TREE-TO-PROLOG

(defun convert-tree-to-prolog (tree)
  (cond ((node-p tree)
         (unless (node-deleted? tree)
           (case (node-kind tree)
             ((:or :and)
              (convert-tree-to-prolog (node-antecedents tree)))
             (:not
              (let ((negated (convert-tree-to-prolog (node-antecedents tree))))
                (cond ((eq (first negated) 'and) (rplaca negated 'not))
                      (negated (list 'not negated))
                      (t nil))))
             (:cut (when (node-selected? tree) '!))
             (otherwise
              (cond ((node-selected? tree) (cons (node-relation-name tree) (node-vars tree)))
                    ((node-antecedents tree) (convert-tree-to-prolog (node-antecedents tree)))
                    (t nil))))))
        ((conjunction-p tree)
         (let ((conjunction (mapcan #'(lambda (node) 
                                        (let ((frontier (convert-tree-to-prolog node)))
                                          (when frontier (list frontier))))
                                    tree)))
           (cond ((rest conjunction) (cons 'and conjunction))
                 (conjunction (first conjunction))
                 (t nil))))
        ((disjunction-p tree)
         (let ((disjunction (mapcan #'(lambda (conjunction) 
                                        (let ((frontier (convert-tree-to-prolog conjunction)))
                                          (when frontier (list frontier))))
                                    tree)))
           (cond ((rest disjunction) (cons 'or disjunction))
                 (disjunction (first disjunction))
                 (t nil))))))

;;;_______________________________________
;;;  CONVERT-GRAPH-TO-PROLOG

(defun convert-graph-to-prolog (graph)
  (convert-tree-to-prolog (graph-base graph)))


;;;_______________________________________
;;;  TRANSFER-ATTRIBUTES-FROM-LITERAL-TO-NODE

(defun transfer-attributes-from-literal-to-node (literal node &optional (from :learning) (select nil) (source nil))
  (let ((derivation (literal-derivation literal)))
    (setf (node-aux node) literal
          (node-selected? node) select
          (node-deleted? node) (literal-deleted? literal)
          (node-state node) (if (derivation-p derivation) (derivation-type derivation) source))
    (when (or (literal-pos literal)
              (literal-neg literal))
      (insert-tuples-into-coverage-struct node from (literal-pos literal) (literal-neg literal) nil nil
                                          (literal-new-pos literal) (literal-new-neg literal) nil nil))
    (when (literal-negated? literal)
      (do ((L (literal-negated-literals literal) (literal-next L))
           (NS (first (node-antecedents node)) (rest NS)))
          ((null L))
        (transfer-attributes-from-literal-to-node L (first NS))))
    node))

;;;_______________________________________
;;;  CONVERT-LITERALS-TO-NODES

(defun convert-literals-to-nodes (literals graph)
  (cond ((null literals) nil)
        ((literal-p literals)
         (cons
          (if (derivation-path (literal-derivation literals))
            (transfer-attributes-from-literal-to-node literals (duplicate-node (derivation-path (literal-derivation literals)) nil graph))
            (if (literal-negated? literals)
              (let* ((nodes (convert-literals-to-nodes (literal-negated-literals literals) graph))
                     (not-node (cond ((node-p nodes) (negate-node nodes graph))
                                     ((conjunction-p nodes) (negate-conjunction nodes graph))
                                     (t nil))))
                (transfer-attributes-from-literal-to-node literals not-node))
              (let ((node (convert-r-struct-to-node (get-r-struct (literal-predicate-name literals))
                                                    (literal-variablization literals)
                                                    nil
                                                    (derivation-type (literal-derivation literals))
                                                    graph)))
                (transfer-attributes-from-literal-to-node literals node))))
          (convert-literals-to-nodes (literal-next literals) graph)))
        ((literal-disjunction-p literals) (cons (convert-literals-to-nodes (first literals) graph)
                                                (convert-literals-to-nodes (rest literals) graph)))))

;;;_______________________________________
;;; CONVERT-NODES-TO-LITERALS

(defun convert-nodes-to-literals (nodes graph &optional (coverage-from nil) (save-path t))
  (cond ((null nodes) nil)
        ((node-p nodes)
         (let ((coverage (when coverage-from (find coverage-from (node-coverage nodes) :key #'coverage-from))))
           (if (node-not? nodes)
             (make-literal :negated? t
                           :negated-literals (convert-nodes-to-literals (first (node-antecedents nodes)) graph coverage-from save-path)
                           :predicate-name nil
                           :variablization nil
                           :pos (when coverage (coverage-input-pos coverage))
                           :neg (when coverage (coverage-input-neg coverage))
                           :new-pos (when coverage (coverage-output-pos coverage))
                           :new-neg (when coverage (coverage-output-neg coverage))
                           :derivation (make-derivation :type (node-state nodes) :path (when save-path nodes) :graph graph)
                           :deleted? (node-deleted? nodes))
             (make-literal :negated? nil
                           :negated-literals nil
                           :predicate-name (r-name (node-r-struct nodes))
                           :variablization (node-vars nodes)
                           :pos (when coverage (coverage-input-pos coverage))
                           :neg (when coverage (coverage-input-neg coverage))
                           :new-pos (when coverage (coverage-output-pos coverage))
                           :new-neg (when coverage (coverage-output-neg coverage))
                           :derivation (make-derivation :type (node-state nodes) :path (when save-path nodes) :graph graph)
                           :deleted? (node-deleted? nodes))) ))
        ((conjunction-p nodes)
         (let ((first (convert-nodes-to-literals (first nodes) graph coverage-from save-path))
               (rest (convert-nodes-to-literals (rest nodes) graph coverage-from save-path)))
           (setf (literal-next first) rest)
           (when (literal-p rest) (setf (literal-prev rest) first))
           first))
        ((disjunction-p nodes)
         (cons (convert-nodes-to-literals (first nodes) graph coverage-from save-path)
               (convert-nodes-to-literals (rest nodes) graph coverage-from save-path)))))

;;;_______________________________________
;;; CONVERT-LITERAL-TO-PROLOG

(defun convert-literal-to-prolog (literal &optional (convert-deleted? nil))
  (when (literal-p literal)
    (when (or (not (literal-deleted? literal)) convert-deleted?)
      (if (literal-negated? literal)
        (cons 'not (convert-literals-to-prolog (literal-negated-literals literal) convert-deleted?))
        (cons (literal-predicate-name literal) (literal-variablization literal))))))

;;;_______________________________________
;;; CONVERT-LITERALS-TO-PROLOG

(defun convert-literals-to-prolog (literals &optional (convert-deleted? nil))
  (cond ((null literals) nil)
        ((literal-p literals)
         (let ((first (convert-literal-to-prolog literals convert-deleted?)))
           (if first
             (cons first (convert-literals-to-prolog (literal-next literals) convert-deleted?))
             (convert-literals-to-prolog (literal-next literals) convert-deleted?))))
        ((literal-disjunction-p literals)
         (let ((first (convert-literals-to-prolog (first literals) convert-deleted?)))
           (if first
             (cons first (convert-literals-to-prolog (rest literals) convert-deleted?))
             (convert-literals-to-prolog (rest literals) convert-deleted?))))))

;;;_______________________________________
;;; CONVERT-TO-PROLOG-FUNCTION

(defun convert-to-prolog-function (nodes-or-literals old-vars)
  (cond ((or (node-p nodes-or-literals)
             (conjunction-p nodes-or-literals)
             (disjunction-p nodes-or-literals))
         (select-node nodes-or-literals)
         (convert-tree-to-prolog-function nodes-or-literals old-vars))
        ((or (literal-p nodes-or-literals)
             (literal-disjunction-p nodes-or-literals))
         (convert-literals-to-prolog-function nodes-or-literals old-vars))))

;;;_______________________________________
;;; CONVERT-LITERALS-TO-PROLOG-FUNCTION

(defun convert-literals-to-prolog-function (literals old-vars)
  (cond ((literal-p literals)
         (focl-compile-clause-function (cons (cons 'dummy old-vars) (convert-literals-to-prolog literals)) (length old-vars)))
        ((literal-disjunction-p literals)
         (focl-compile-clauses-function
          (mapcar #'(lambda (clause) (cons (cons 'dummy old-vars) (convert-literals-to-prolog clause))) literals) (length old-vars)))))

;;;_______________________________________
;;;  CONVERT-TREE-TO-PROLOG-FUNCTION

(defun convert-tree-to-prolog-function (tree variables)
  (let ((frontier (convert-tree-to-prolog tree)))
    (values (when frontier 
              (focl-compile-clause-function (list (cons 'dummy variables) frontier) (length variables)))
            variables)))

;;;_______________________________________
;;;  CONVERT-GRAPH-TO-PROLOG-FUNCTION

(defun convert-graph-to-prolog-function (graph)
  (let ((base (graph-base graph)))
    (convert-tree-to-prolog-function base (node-vars base))))


;;;_______________________________________
;;; PATTERNS AND DIRECT MAPPING

(defparameter *filtered-indicator* '_)

(defun filtered? (term) (or (eq term *filtered-indicator*)
                            (new-var? term)))

(defun filtered-example-info (example pattern)
  (mapcan #'(lambda (e p) (if (filtered? p) (list e) nil)) example pattern))

(defun example-matches-pattern (example pattern)
  (every #'(lambda (e p) (or (filtered? p) (equalp e p))) example pattern))

(defun count-examples-matching-pattern (examples pattern)
  (count-if #'(lambda (example) (example-matches-pattern example pattern)) examples))

(defun return-examples-matching-pattern (examples pattern)
  (delete-duplicates (remove-if-not #'(lambda (example) (example-matches-pattern example pattern)) examples) :test #'equal))

(defun direct-mapping (list1 list2)
  (mapcar #'list list1 list2))

(defun direct-substitute (form mapping &optional (filter nil))
  (cond ((null form) nil)
        ((atom form)
         (let ((pair (assoc form mapping))) 
           (if pair
             (second pair)
             (if filter *filtered-indicator* form))))
        ((consp form)
         (mapcar #'(lambda (e) (direct-substitute e mapping filter)) form))))

(defun direct-substitute-args (form mapping &optional (filter nil))
  (cond ((null form) nil)
        ((atom form)
         (let ((pair (assoc form mapping))) 
           (if pair
             (second pair)
             (if filter *filtered-indicator* form))))
        ((consp form)
         (let ((first (first form)))
           (if (atom first)
             (cons first (mapcar #'(lambda (e) (direct-substitute-args e mapping filter)) (rest form)))
             (mapcar #'(lambda (e) (direct-substitute-args e mapping filter)) form))))))

(defun used-in (form object)
  (cond ((null object) nil)
        ((equal form object) t)
        ((null form) nil)
        ((consp form) (or (used-in (first form) object)
                          (used-in (rest form) object)))))

;;;_______________________________________
;;;  CONVERT-EXAMPLE-TO-FACT

(defun convert-example-to-fact (predicate-name example)
  (unless (consp example) (setf example (list example)))
  (if (consp predicate-name)
    (list 'not (cons (second predicate-name) example))
    (cons predicate-name example)))

;;;_______________________________________
;;;  SET-BUILTIN-FLAGS

(defun set-builtin-flags (names induction-flags &optional (try-constants-flags nil) &aux builtin)
  (mapc #'(lambda (name induction-flag)
            (when (setq builtin (rest (assoc name *builtin-preds*)))
              (setf (r-induction builtin) induction-flag)))
        names induction-flags)
  (mapc #'(lambda (name try-constants-flag)
            (when (setq builtin (rest (assoc name *builtin-preds*)))
              (setf (r-try-constants builtin) try-constants-flag)))
        names try-constants-flags)
  (values))

;;;______________________________________
;;; SET-COVERAGE

(defun set-coverage (coverage &key from input-pos input-neg input-vars input-type output-pos output-neg output-vars output-type)
  (when (coverage-p coverage)
    (setf (coverage-from coverage) from
          (coverage-input-pos coverage) input-pos
          (coverage-input-neg coverage) input-neg
          (coverage-input-vars coverage) input-vars
          (coverage-input-type coverage) input-type
          (coverage-output-pos coverage) output-pos
          (coverage-output-neg coverage) output-neg
          (coverage-output-vars coverage) output-vars
          (coverage-output-type coverage) output-type)
    coverage))

;;;______________________________________
;;; FREE-COVERAGE

(defun free-coverage (coverage)
  (when (coverage-p coverage)
    (set-coverage coverage)
    (setf *free-coverage-structs* (push coverage *free-coverage-structs*))))

;;;______________________________________
;;; GET-COVERAGE

(defun get-coverage (&rest keys &key the-ignored-key &allow-other-keys)
  (declare (ignore the-ignored-key))
  (let (coverage)
    (if *free-coverage-structs*
      (setf coverage (first *free-coverage-structs*)
            *free-coverage-structs* (rest *free-coverage-structs*))
      (setf coverage (make-coverage)))
    (apply #'set-coverage coverage keys)))

;;;______________________________________
;;; DUPLICATE-COVERAGE

(defun duplicate-coverage (coverage)
  (when (coverage-p coverage)
    (get-coverage :from (coverage-from coverage)
                  :input-pos (copy-list (coverage-input-pos coverage))
                  :input-neg (copy-list (coverage-input-neg coverage))
                  :input-vars (copy-list (coverage-input-vars coverage))
                  :input-type (copy-list (coverage-input-type coverage))
                  :output-pos (copy-list (coverage-output-pos coverage))
                  :output-neg (copy-list (coverage-output-neg coverage))
                  :output-vars (copy-list (coverage-output-vars coverage))
                  :output-type (copy-list (coverage-output-type coverage)))))

;;;_______________________________________
;;; INSERT-TUPLES-INTO-COVERAGE-STRUCT

(defun insert-tuples-into-coverage-struct (node from input-pos input-neg input-vars input-type 
                                                output-pos output-neg output-vars output-type &optional (save *save-examples*))
  (when save
    (let ((coverage (find from (node-coverage node) :key #'coverage-from)))
      (unless (coverage-p coverage)
        (setf coverage (get-coverage :from from)
              (node-coverage node) (push coverage (node-coverage node))))
      (set-coverage coverage :from from
                    :input-pos input-pos 
                    :input-neg input-neg
                    :input-vars input-vars
                    :input-type input-type
                    :output-pos output-pos
                    :output-neg output-neg
                    :output-vars output-vars
                    :output-type output-type)
      coverage)))

;;;_______________________________________
;;; REDUCE-TUPLES
;;;
;;; INPUT
;;;  reduced-args  a subset of the tuple-vars possibly augmented with constants       e.g. (?v4 5 ?v3)
;;;  tuples-vars   a list of the variables associated with each datum in the tuple    e.g. (?v1 ?v2 ?v3 ?v4) 
;;;  tuples        a list of tuples                                                   e.g. ((0 0 0 0) (0 0 0 1) (0 0 1 0) (0 0 1 1)) 
;;;
;;; OUTPUT
;;;  a list of tuples modified to conform to reduced-args                             e.g. ((0 5 0) (1 5 0) (0 5 1) (1 5 1)) 
;;;
;;;  This function is optimized for handle four cases: 
;;;  1 - reduced-args contains no constants and is a subsequence of tuple-vars        
;;;  2 - reduced-args contains no constants and is not a subsequence of tuple-vars
;;;  3 - reduced-args contains constants and is a subsequence of tuple-vars
;;;  4 - reduced-args contains constants and is not a subsequence of tuple-vars

(defun reduce-tuples (reduced-args tuple-vars tuples)
  (if (equal reduced-args tuple-vars)
    tuples
    (if (every #'variable-p reduced-args)
      (if (let ((remaining-tuple-vars tuple-vars))
            (every #'(lambda (arg) (when (setq remaining-tuple-vars (member arg remaining-tuple-vars)) t)) reduced-args))
        (mapcar #'(lambda (tuple)
                    (let ((vars tuple-vars)
                          (datums tuple))
                      (mapcar #'(lambda (arg)
                                  (do () ((var-eq (first vars) arg) (first datums))
                                    (setq vars (rest vars) datums (rest datums))))
                              reduced-args)))
                tuples)
        (let ((position-vector (mapcar #'(lambda (arg) (position arg tuple-vars :test #'var-eq)) reduced-args)))
          (mapcar #'(lambda (tuple) (mapcar #'(lambda (p) (nth p tuple)) position-vector)) tuples)))
      (if (let ((remaining-tuple-vars tuple-vars))
            (every #'(lambda (arg) (if (variable-p arg) (when (setf remaining-tuple-vars (member arg remaining-tuple-vars)) t) t)) reduced-args))
        (mapcar #'(lambda (tuple)
                    (let ((vars tuple-vars)
                          (datums tuple))
                      (mapcar #'(lambda (arg)
                                  (if (variable-p arg)
                                    (do () ((var-eq (first vars) arg) (first datums))
                                      (setq vars (rest vars) datums (rest datums)))
                                    arg))
                              reduced-args)))
                tuples)
        (mapcar #'(lambda (tuple) (mapcar #'(lambda (arg) (if (variable-p arg) (nth (position arg tuple-vars :test #'var-eq) tuple) arg)) reduced-args)) tuples)))))

;;;______________________________________
;;; compute-new-vars-and-types

(defun compute-new-vars-and-types (args arg-types old-vars new-vars new-types)
  (do* ((vars args (rest vars))
        (var (first vars) (first vars))
        (types arg-types (rest types))
        (type (first types) (first types)))
       ((null vars))
    (cond ((pcvar-p var)
           (unless (or (member var old-vars :test #'var-eq)
                       (member var new-vars :test #'var-eq))
             (push var new-vars)
             (push (or type :anything) new-types)))
          ((consp var)
           (multiple-value-bind (n-vars n-types) (compute-new-vars-and-types var nil (append old-vars new-vars) nil nil)
             (setq new-vars (nconc n-vars new-vars)
                   new-types (nconc n-types new-types))))))
  (values new-vars new-types))

(defun compute-node-new-vars-and-types (node old-vars &optional (new-vars nil) (new-types nil))
  (when (node-p node)
    (compute-new-vars-and-types (reverse (node-vars node)) (when (r-p (node-r-struct node)) (reverse (r-type (node-r-struct node)))) old-vars new-vars new-types)))

(defun compute-conjunction-nodes-and-types (conjunction)
  (let ((vars nil)
        (types nil))
    (dolist (node (reverse conjunction))
      (multiple-value-setq (vars types) (compute-node-new-vars-and-types node nil vars types)))
    (values vars types)))

(defun compute-conequent-and-conjunction-nodes-and-types (conjunction)
  (multiple-value-bind (vars types) (compute-node-new-vars-and-types (node-real-consequent (first conjunction)) nil nil nil)
    (dolist (node (reverse conjunction))
      (multiple-value-setq (vars types) (compute-node-new-vars-and-types node nil vars types)))
    (values vars types)))

(defun compute-r-struct-new-vars-and-types (r-struct arguments old-vars &optional (new-vars nil) (new-types nil))
  (when (r-p r-struct)
    (compute-new-vars-and-types (reverse arguments) (reverse (r-type r-struct)) old-vars new-vars new-types)))

;;;_______________________________________
;;; INSERT-NODE-TUPLES

(defun insert-node-tuples (nodes pos-tuples neg-tuples old-vars old-types from &optional (view nil) (recursive t) (save *save-examples*) (only-insert-into-first-cover t))
    (cond ((null nodes) (values pos-tuples neg-tuples old-vars old-types))
          
          ((node-p nodes)
           (if (node-deleted? nodes)
             (values pos-tuples neg-tuples old-vars old-types)
             (multiple-value-bind (new-vars new-types) (compute-node-new-vars-and-types nodes old-vars)
               (let ((antecedents (node-antecedents nodes))
                     (extended-pos-tuples nil)
                     (extended-neg-tuples nil)
                     (extended-variables nil)
                     (extended-types nil))
                 (when view (select-node-only nodes))
                 (case (node-kind nodes)
                   (:intensional
                    (setf extended-variables (if new-vars (append old-vars new-vars) old-vars)
                          extended-types (if new-vars (append old-types new-types) old-types))
                    (if (and recursive antecedents)
                      (let ((pos-not-covered pos-tuples)
                            (neg-not-covered neg-tuples)
                            (pos nil)
                            (neg nil)
                            (vars nil))
                        (dolist (conjunction antecedents)
                          (multiple-value-setq (pos neg vars)
                            (insert-node-tuples conjunction pos-not-covered neg-not-covered old-vars old-types from view recursive save only-insert-into-first-cover))
                          (if only-insert-into-first-cover
                            (setf extended-pos-tuples (nconc extended-pos-tuples (delete-duplicates (reduce-tuples extended-variables vars pos) :test #'equal))
                                  extended-neg-tuples (nconc extended-neg-tuples (delete-duplicates (reduce-tuples extended-variables vars neg) :test #'equal))
                                  pos-not-covered (return-originals-not-extended pos-not-covered pos)
                                  neg-not-covered (return-originals-not-extended neg-not-covered neg))
                            (setf extended-pos-tuples (delete-duplicates (nconc extended-pos-tuples (reduce-tuples extended-variables vars pos)) :test #'equal)
                                  extended-neg-tuples (delete-duplicates (nconc extended-neg-tuples (reduce-tuples extended-variables vars neg)) :test #'equal)
                                  pos-not-covered pos-tuples
                                  neg-not-covered neg-tuples))))
                      (multiple-value-setq (extended-pos-tuples extended-neg-tuples)
                        (extend-tuples nodes pos-tuples neg-tuples old-vars old-types new-vars))))
                   (:not
                    (multiple-value-bind (extended-pos extended-neg)
                                         (insert-node-tuples antecedents pos-tuples neg-tuples old-vars old-types from view recursive save only-insert-into-first-cover)
                      (setf extended-variables old-vars
                            extended-types old-types
                            extended-pos-tuples (return-originals-not-extended pos-tuples extended-pos)
                            extended-neg-tuples (return-originals-not-extended neg-tuples extended-neg))))
                   (:and
                    (multiple-value-setq (extended-pos-tuples extended-neg-tuples extended-variables extended-types) 
                      (insert-node-tuples (first antecedents) pos-tuples neg-tuples old-vars old-types from view recursive save only-insert-into-first-cover)))
                   (:or
                    (multiple-value-setq (extended-pos-tuples extended-neg-tuples extended-variables extended-types) 
                      (insert-node-tuples antecedents pos-tuples neg-tuples old-vars old-types from view recursive save only-insert-into-first-cover)))
                   (:cut
                    (setf extended-variables old-vars
                          extended-types old-types
                          extended-pos-tuples pos-tuples
                          extended-neg-tuples neg-tuples))
                   (otherwise
                    (setf extended-variables (if new-vars (append old-vars new-vars) old-vars)
                          extended-types (if new-vars (append old-types new-types) old-types))
                    (multiple-value-setq (extended-pos-tuples extended-neg-tuples)
                      (extend-tuples nodes pos-tuples neg-tuples old-vars old-types new-vars))))
                 (when view
                   (deselect-node-only nodes)
                   (hilight-node-coverage nodes extended-pos-tuples extended-neg-tuples)
                   (with-focused-view view
                     (draw-cell (node-cell view nodes))))
                 (when (eq extended-pos-tuples :uses-undefined-relation) (setq extended-pos-tuples nil))
                 (insert-tuples-into-coverage-struct nodes from pos-tuples neg-tuples old-vars old-types (copy-list extended-pos-tuples) (copy-list extended-neg-tuples)
                                                     extended-variables extended-types save)
                 (values extended-pos-tuples extended-neg-tuples extended-variables extended-types)))))
          
          ((conjunction-p nodes)
           (dolist (node nodes)
             (multiple-value-setq (pos-tuples neg-tuples old-vars old-types)
               (insert-node-tuples node pos-tuples neg-tuples old-vars old-types from view recursive save only-insert-into-first-cover)))
           (when view
             (let* ((last-node (first (last nodes)))
                    (cell (node-cell view last-node)))
               (when cell (update-external-text cell (format nil "~A+ ~A-" (length pos-tuples) (length neg-tuples))))))
           (values pos-tuples neg-tuples old-vars old-types))
          
          ((disjunction-p nodes)
           (let ((pos-not-covered pos-tuples)
                 (neg-not-covered neg-tuples)
                 (pos-covered nil)
                 (neg-covered nil)
                 (extended-pos nil)
                 (extended-neg nil))
             (dolist (conjunction nodes)
               (multiple-value-setq (extended-pos extended-neg)
                 (insert-node-tuples conjunction pos-not-covered neg-not-covered old-vars old-types from view recursive save only-insert-into-first-cover))
               (if only-insert-into-first-cover
                 (setf pos-covered (nconc pos-covered (return-originals-extended pos-tuples extended-pos))
                       neg-covered (nconc neg-covered (return-originals-extended neg-tuples extended-neg))
                       pos-not-covered (return-originals-not-extended pos-not-covered extended-pos) 
                       neg-not-covered (return-originals-not-extended neg-not-covered extended-neg))
                 (setf pos-covered (delete-duplicates (nconc pos-covered (return-originals-extended pos-tuples extended-pos)) :test #'equal)
                       neg-covered (delete-duplicates (nconc neg-covered (return-originals-extended neg-tuples extended-neg))  :test #'equal)
                       pos-not-covered pos-tuples
                       neg-not-covered neg-tuples)))
             (values pos-covered neg-covered old-vars old-types)))))

;;;_______________________________________
;;;  ADD-EXTRA-VARIABLES

(defun add-extra-variables (e p &aux (s (get-r-struct p)))
  (cons e (rest (r-vars s))))

;;;_______________________________________
;;;  JUDGE-RULES-ON-EXAMPLES

(defun judge-rules-on-examples ()
  (let* ((answers  (r-pos (get-r-struct *predicate-being-learned*)))
         (examples (remove-duplicates (nconc (mapcar #'(lambda(x) (list (first x))) answers)
                                             (mapcar #'(lambda(x) (list (first x))) (r-neg (get-r-struct *predicate-being-learned*)))) :test #'equal))
         
         (*batch* t)   ;;; CAB used to be nil ???
         (*maintain-prolog-rule-trace* nil)
         (oerrors nil)
         (cerrors nil)
         (a nil))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-test-on-all-examples *user-monitor*)))
    (setq *last-explanation* nil)
    (format T "~%Testing Examples")
    (dolist (e examples)
      (format t "~% ~a" e)
      (setq e (add-extra-variables (first e) (first *goal-concept*)))
      (setq a (rest (prove-goal `(,(first *goal-concept*) . ,e))))
      (format t " ~a" a)
      (if a
        (unless (member a answers :test #'equal)
          (format t " commission error")
          (push  a cerrors))
        (when (setq a (member e answers :test #'unify))
          (format t " omission error")
          (push (first a) oerrors))))
    (when oerrors (format t "~%The following inferences(s) should be true, but are not supported by the rules:~%~{~a~%~}~%~%" oerrors))
    (when cerrors (format t "~%The following inference(s) should not be true, but are supported by the rules:~%~{~a~%~}~%~%" cerrors))
    (format t "~%The Knowledge Base is ~4F% accurate on these examples~%" (- 100 (* 100 (/ (+ (length oerrors) (length cerrors)) (length examples)))))
    ))

(defun evaluate-prolog-function (prolog-function pos neg)
  (let ((pos-proved 0)
        (pos-not-proved 0)
        (neg-proved 0)
        (neg-not-proved 0)
        (cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))))
    (dolist (p pos)
      (if (catch 'reg-prove (apply prolog-function cont nil p))
        (incf pos-proved)
        (incf pos-not-proved)))
    (dolist (n neg)
      (if (catch 'reg-prove (apply prolog-function cont nil n))
        (incf neg-proved)
        (incf neg-not-proved)))
    (values pos-proved pos-not-proved neg-proved neg-not-proved)))

(defun evaluate-relation (r-struct pos neg)
  (when (and (r-p r-struct) (or pos neg))
    (let* ((*batch* t)
           (*maintain-prolog-rule-trace* nil)
           (*var-counter* 0))
      (evaluate-prolog-function (r-prolog-function r-struct) pos neg))))

;;;=======================================
;;; The following are for compatibility

;;;_______________________________________
;;;  CONVERT-NODES-TO-PROLOG

(defun convert-nodes-to-prolog (nodes)
  (select-node nodes)
  (convert-tree-to-prolog nodes))
 

;;;_______________________________________
;;; CONVERT-NODES-TO-PROLOG-FUNCTION

(defun convert-nodes-to-prolog-function (nodes old-vars)
  (select-node nodes)
  (convert-tree-to-prolog-function nodes old-vars))

;;;_______________________________________
;;; CONVERT-TO-PROLOG

(defun convert-to-prolog (nodes-or-literals)
  (cond ((or (node-p nodes-or-literals)
             (conjunction-p nodes-or-literals)
             (disjunction-p nodes-or-literals))
         (convert-tree-to-prolog nodes-or-literals))
        ((or (literal-p nodes-or-literals)
             (literal-disjunction-p nodes-or-literals))
         (convert-literals-to-prolog nodes-or-literals))))

;;;_______________________________________
;;; USES-CUT?

(defun uses-cut? (r-name &optional (already-checked nil))
  (unless (member r-name already-checked)
    (push r-name already-checked)
    (let ((clauses (get-clauses r-name)))
      (or (used-in clauses '!)
          (some #'(lambda (clause) (some #'(lambda (literal) (when (consp literal) (uses-cut? (first literal) already-checked))) clause)) clauses)))))

;;;_______________________________________
;;; USES-UNDEFINED-RELATION?

(defun uses-undefined-relation? (literal &optional (already-checked nil))
  (if (consp literal)
    (let ((r-name (first literal)))
      (unless (member r-name already-checked)
        (push r-name already-checked)
        (cond ((or (eq r-name 'and)
                   (eq r-name 'or)
                   (eq r-name 'not))
               (some #'(lambda (l) (uses-undefined-relation? l already-checked)) (rest literal)))
              ((get-rule r-name) (some #'(lambda (c) (some #'(lambda (l) (uses-undefined-relation? l already-checked)) (rest c))) (get-clauses r-name)))
              (t (let ((r-struct (get-r-struct r-name)))
                   (if (and (r-p r-struct) (not (eq (r-kind r-struct) :undefined))) nil t))))))
    (let ((r-struct (get-r-struct literal)))
      (if (and (r-p r-struct) (not (eq (r-kind r-struct) :undefined))) nil t))))

;;;_______________________________________
;;; GET-TIME

(defvar days-of-week '("Monday" "Tuesday" "Wednesday" "Thursday" "Friday" "Saturday" "Sunday"))

(defun get-time ()
  (multiple-value-bind (sec min hour date month year day-of-week savings-time time-zone) (get-decoded-time)
    (declare (ignore time-zone savings-time))
    (let ((date-str (format nil "~2,'0D/~2,'0D/~2,'0D" month date (- year 1900)))
	  (time-str (format nil "~2,'0D:~2,'0D:~2,'0D" hour min sec))
	  (day-str (string-capitalize (string-downcase (princ-to-string (elt days-of-week day-of-week))))))
      (concatenate 'string time-str " on " day-str ", " date-str))))

;;;_______________________________________
;;; PRINT-RULE-SUMMARY

(defun print-rule-summary (clauses)
  (mapc #'print-clause-summary clauses)
  (values))

;;;_______________________________________
;;; PRINT-CLAUSE-SUMMARY

(defun print-clause-summary (clause &aux c)
  (format t "~%---------New Clause---------~%")
  (do ((l clause (literal-next l)))
      ((null l))
    (setf c (copy-literal l))
    (setf (literal-next c) nil) ;;only print one literal
    (when  (literal-deleted? c) 
      (format t "~%DELETED DELETED DELETED DELETED DELETED"))
    (format t "~%~a in: ~20T+~a ~27T-~a out: ~40T+~a ~47T-~a"
            c (length (literal-pos c))(length (literal-neg c))
            (length (literal-new-pos c))(length (literal-new-neg c)))
    (format t "~%source ~a" (derivation-type (literal-derivation c)))
    (when (member (derivation-type (literal-derivation c)) '(:ebl :intensional))
      (pprint (mapcar #'(lambda(x)
                                 (if (and (consp x)
                                          (clause-p (car x)))
                                   (cdr x)
                                   x))
                             (derivation-path  (literal-derivation c)))))
    (format t "~%")))

;;;_______________________________________
;;; LITERAL-USES-RELATION

(defun literal-uses-relation (rule-name clause-index literal-index literal relation)
  (unless (literal-deleted rule-name clause-index literal-index)
    (if (consp literal)
      (let ((r (first literal)))
        (cond ((eql r relation) t)
              ((eql r 'not) (clause-uses-relation r 0 (rest literal) relation))
              ((eql r 'and) (clause-uses-relation r 0 (rest literal) relation))
              ((eql r 'or) (clause-uses-relation r 0 (rest literal) relation))
              ((eql r 'bagof) (literal-uses-relation r 0 0(second (rest literal)) relation))
              ((eql r 'call) (clause-uses-relation r 0 (rest literal) relation))
              ((eql r 'find-proofs) (literal-uses-relation r 0 0 (first (rest literal)) relation))
              ((eql r 'setof) (literal-uses-relation r 0 0 (second (rest literal)) relation))))
      (eql literal relation))))

;;;_______________________________________
;;; CLAUSE-USES-RELATION

(defun clause-uses-relation (rule-name clause-index clause relation)
  (unless (clause-deleted rule-name clause-index)
    (let ((literal-index -1))
      (some #'(lambda (literal) (literal-uses-relation rule-name clause-index (incf literal-index) literal relation)) clause))))

;;;_______________________________________
;;; RULES-USING-RELATION

(defun rules-using-relation (relation)
  (let (rule-name clause-index)
    (mapcan #'(lambda (r-struct)
                (when (rule-p r-struct)
                  (setq rule-name (r-name r-struct)
                        clause-index -1)
                  (when (some #'(lambda (clause) (clause-uses-relation rule-name (incf clause-index) (rest clause) relation)) (get-clauses rule-name))
                    (list r-struct))))
            *r-structs*)))

;;;_______________________________________
;;; RULES-DEPENDING-ON-RELATIONS

(defun rules-depending-on-relations (relations &optional (rules-depending-on-relations nil))
  (if relations
    (let* ((relation (first relations))
           (remaining-relations (rest relations))
           r-name clause-index)
      (dolist (r *r-structs*)
        (setq r-name (r-name r)
              clause-index -1)
        (when (and (rule-p r)
                   (some #'(lambda (clause) (clause-uses-relation r-name (incf clause-index) (rest clause) relation)) (get-clauses r-name)))
          (unless (or (member r-name relations)
                      (member r-name rules-depending-on-relations))
            (push r-name remaining-relations))))
      (rules-depending-on-relations remaining-relations (cons relation rules-depending-on-relations)))
    rules-depending-on-relations))

;;;_______________________________________
;;; RULE-MAKES-MULTIPLE-CALLS-TO-SAME-RULE?

(defun rule-makes-multiple-calls-to-same-rule? (name)
  (let ((rule (get-rule name)))
    (when rule
      (multiple-value-bind (relations-called multiple-calls)
                           (relations-called-by-clauses name (get-clauses name) nil nil)
        (declare (ignore relations-called))
        multiple-calls))))

;;;_______________________________________
;;; RELATIONS-CALLED-BY-RELATION

(defun relations-called-by-relation (name)
  (when (get-rule name)
    (relations-called-by-clauses name (get-clauses name) nil nil)))

;;;_______________________________________
;;; RELATIONS-CALLED-BY-LITERAL

(defun relations-called-by-literal (rule-name clause-index literal-index literal relations-called multiple-calls)
  (if (and (consp literal)
           (not (literal-deleted rule-name clause-index literal-index)))
    (let ((name (first literal)))
      (cond ((eql name 'not) (relations-called-by-clause name 0 (rest literal) relations-called multiple-calls))
            ((eql name 'and) (relations-called-by-clause name 0 (rest literal) relations-called multiple-calls))
            ((eql name 'or) (relations-called-by-clause name 0 (rest literal) relations-called multiple-calls))
            ((eql name 'bagof) (relations-called-by-literal name 0 0 (second (rest literal)) relations-called multiple-calls))
            ((eql name 'call) (relations-called-by-clause name 0 (rest literal) relations-called multiple-calls))
            ((eql name 'find-proofs) (relations-called-by-literal name 0 0 (first (rest literal)) relations-called multiple-calls))
            ((eql name 'setof) (relations-called-by-literal name 0 0 (second (rest literal)) relations-called multiple-calls))
            ((get-rule name) (if (member name relations-called :test #'equal)
                               (values relations-called t)
                               (relations-called-by-clauses name (get-clauses name) (push name relations-called) multiple-calls)))
            (t (values (pushnew name relations-called :test #'equal) multiple-calls))))
    (values relations-called multiple-calls)))

;;;_______________________________________
;;; RELATIONS-CALLED-BY-CLAUSE

(defun relations-called-by-clause (rule-name clause-index clause relations-called multiple-calls)
  (unless (clause-deleted rule-name clause-index)
    (let ((literal-index -1))
      (dolist (literal clause)
        (multiple-value-setq (relations-called multiple-calls)
          (relations-called-by-literal rule-name clause-index (incf literal-index) literal  relations-called multiple-calls)))))
  (values relations-called multiple-calls))
    
;;;_______________________________________
;;; RELATIONS-CALLED-BY-CLAUSES

(defun relations-called-by-clauses (rule-name clauses relations-called multiple-calls)
  (let ((clause-index -1))
    (dolist (clause clauses)
      (multiple-value-setq (relations-called multiple-calls)
        (relations-called-by-clause rule-name (incf clause-index) (rest clause) relations-called multiple-calls))))
  (values relations-called multiple-calls))


;;;_________________________________________________________
;;; learn

#+:ccl-2 (defun learn ()
	   (when (and (fboundp 'find-concept-description)
		      (get-pred (first *focl-problem*)))
		 (when (user-monitor-p *user-monitor*)
		       (incf (user-monitor-learn *user-monitor*)))
		 (apply #'set-learning-parameters (first *focl-problem*) (rest *focl-problem*))
		 ;;(window-hide *top-listener*)
		 (when *user-interface-available* (menu-install *es-learning-menu*))
		 (unwind-protect
		     (let ((*backtrace-on-break* nil)
			   (*error-output* ccl::*terminal-io*))
		       (find-concept-description)
		       (print-learned-description))
		   (when *user-interface-available*
			 (menu-deinstall *es-learning-menu*)
			 ;;(window-select *top-listener*)
			 )))
	   (values))


#-:ccl-2 (defun learn ()
           (when (and (fboundp 'find-concept-description)
                      (get-pred (first *focl-problem*)))
		 (apply #'set-learning-parameters (first *focl-problem*) (rest *focl-problem*))
		 (find-concept-description)
		 (print-learned-description))
           (values))

(defun random-element (list)
  (when list (nth (random (length list)) list)))