;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  


(in-package :user)

;;;_______________________________________
;;;   Node and Cell field Conversions
;;;_______________________________________

(defun node-or? (node) (eq (node-kind node) :OR))
(defun node-and? (node) (eq (node-kind node) :AND))
(defun node-not? (node) (eq (node-kind node) :NOT))
(defun node-cut? (node) (eq (node-kind node) :CUT))
(defun node-root? (node) (eq (node-kind node) :ROOT))
(defun node-intensional? (node) (eq (node-kind node) :INTENSIONAL))
(defun node-second-order? (node) (case (node-kind node)
                                  ((:or :and :not) t)
                                  (otherwise nil)))
       

(defun node-cell (view node)
  (when view 
    (find-if #'(lambda (cell) (eq (cell-view cell) view)) (node-cells node))))

(defun node-root (node)
  (if (node-consequent node) (node-root (node-consequent node))  node))

(defun node-depth (node &optional (depth -1))
  (if (node-consequent node)
    (node-depth (node-consequent node) (+ depth 1))
    depth))

(defun node-old-vars (node)
  (if (node-root? (node-consequent node))
    (copy-list (node-vars node))
    (let* ((old-vars (node-old-vars (node-consequent node)))
           new-vars)
      (do* ((nodes (conjunction-containing-node node) (rest nodes))
            (n (first nodes) (first nodes)))
           ((eq n node))
        (setf new-vars (mapcan #'(lambda (var) (when (and (variable-p var)
                                                          (not (member var old-vars :test #'equal)))
                                                 (list var)))
                               (node-vars n))
              old-vars (nconc old-vars new-vars)))
      old-vars)))


(defun node-graph (node) (find-if #'(lambda (graph) (member node (graph-used-nodes graph))) *graphs*))

(defun cell-consequent (cell) (node-consequent (cell-node cell)))
(defun cell-antecedents (cell) (node-antecedents (cell-node cell)))
(defun cell-r-struct (cell) (node-r-struct (cell-node cell)))
(defun cell-vars (cell) (node-vars (cell-node cell)))
(defun cell-deleted? (cell) (node-deleted? (cell-node cell)))
(defun cell-selected? (cell) (node-selected? (cell-node cell)))
(defun cell-recursive? (cell) (node-recursive? (cell-node cell)))
(defun cell-kind (cell) (node-kind (cell-node cell)))
(defun cell-state (cell) (node-state (cell-node cell)))
(defun cell-cells (cell) (node-cells (cell-node cell)))
(defun cell-coverage (cell) (node-coverage (cell-node cell)))
(defun cell-aux (cell) (node-aux (cell-node cell)))

(defun cell-not? (cell) (node-not? (cell-node cell)))
(defun cell-center-h (cell) (floor (+ (cell-left cell) (cell-right cell)) 2))
(defun cell-center-v (cell) (floor (+ (cell-top cell) (cell-bottom cell)) 2))

(defun cell-parent (cell)
  (let ((consequent (cell-consequent cell)))
    (when consequent (node-cell (cell-view cell) consequent))))

(defun cell-children (cell)
  (let ((view (cell-view cell)))
    (mapcan #'(lambda (conjunction)
                (mapcan #'(lambda (node) (node-cell view node))
                        conjunction))
            (cell-antecedents cell))))

(defun cell-expanded? (cell)
  (when (cell-p cell)
    (some #'(lambda (c)
              (some #'(lambda (n)
                        (let ((cell (node-cell (cell-view cell) n)))
                          (and cell (not (cell-hidden? cell)))))
                    c))
    (cell-antecedents cell))))

(defun cell-unexpanded? (cell) (not (cell-expanded? cell)))

;;;_______________________________________
;;; get-node

(defun get-node (graph &rest keys &key ignored &allow-other-keys)
  (declare (ignore ignored))
  (let (node)
    (if (and (graph-p graph) (graph-free-nodes graph))
      (setf node (first (graph-free-nodes graph))
            (graph-free-nodes graph) (rest (graph-free-nodes graph)))
      (setf node (make-node)))
    (setf (graph-used-nodes graph) (push node (graph-used-nodes graph)))
    (apply #'set-node node keys)
    node))

;;;_______________________________________
;;; set-node

(defun set-node (node &key
                      (consequent nil)
                      (antecedents nil)
                      (r-struct nil)
                      (vars nil)
                      (deleted? nil)
                      (selected? nil)
                      (recursive? nil)
                      (kind nil)
                      (state nil)
                      (cells nil)
                      (coverage nil)
                      (aux nil))
  (when (node-p node)
    (setf (node-consequent node) consequent
          (node-antecedents node) antecedents
          (node-r-struct node) r-struct
          (node-vars node) vars
          (node-deleted? node) deleted?
          (node-selected? node) selected?
          (node-recursive? node) recursive?
          (node-kind node) kind
          (node-state node) state
          (node-cells node) cells
          (node-coverage node) coverage
          (node-aux node) aux)
    (when (r-p r-struct)
      (setf (r-nodes r-struct) (push node (r-nodes r-struct))))
    node))

;;;_______________________________________
;;; reset-node

(defun reset-node (node)
  (let ((r-struct (node-r-struct node)))
    (when (r-p r-struct)
      (setf (r-nodes r-struct) (delete node (r-nodes r-struct)))))
  (when (node-cells node)
    (dolist (cell (node-cells node))
      (free-cell cell)))
  (set-node node))

;;;_______________________________________
;;; free-node

(defun free-node (node-or-collection graph &optional (recursive nil))
  (when (graph-p graph)
    (cond ((node-p node-or-collection)
           (setf (graph-used-nodes graph) (delete node-or-collection (graph-used-nodes graph))
                 (graph-free-nodes graph) (push node-or-collection (graph-free-nodes graph)))
           (let ((antecedents (node-antecedents node-or-collection)))
             (reset-node node-or-collection)
             (when recursive (free-node antecedents graph recursive))))
          ((consp node-or-collection)
           (free-node (first node-or-collection) graph recursive)
           (free-node (rest node-or-collection) graph recursive)))))

;;;_______________________________________
;;; dispose-node

(defun dispose-node (node-or-collection graph &optional (recursive nil))
  (when (graph-p graph)
    (cond ((node-p node-or-collection)
           (setf (graph-used-nodes graph) (delete node-or-collection (graph-used-nodes graph))
                 (graph-free-nodes graph) (delete node-or-collection (graph-free-nodes graph)))
           (let ((antecedents (node-antecedents node-or-collection)))
             (reset-node node-or-collection)
             (when recursive (dispose-node antecedents graph recursive))))
          ((consp node-or-collection)
           (dispose-node (first node-or-collection) graph recursive)
           (dispose-node (rest node-or-collection) graph recursive)))))



;;;_______________________________________
;;; create-graph

(defun create-graph (&key (free-nodes nil)
                          (used-nodes nil)
                          (root (make-node :kind :root))
                          (permanent? nil)
                          (views nil))
  (let ((graph (make-graph :free-nodes free-nodes
                           :used-nodes (pushnew root used-nodes)
                           :root root
                           :permanent? permanent?
                           :views views)))
    (setf *graphs* (push graph *graphs*))
    graph))

;;;_______________________________________
;;; dispose-graph

(defun dispose-graph (graph)
  (when (graph-p graph)
    (let (window)
      (dolist (view (graph-views graph))
        (setf window (graph-window view))
        (when (window-open? window)
          (window-close (graph-window view)))))
    (when (graph-used-nodes graph)
      (dispose-node (graph-used-nodes graph) graph nil))
    (setf (graph-free-nodes graph) nil
          (graph-used-nodes graph) nil
          (graph-root graph) nil
          (graph-permanent? graph) nil
          (graph-views graph) nil
          *graphs* (delete graph *graphs*))))


;;;_______________________________________
;;; graph-base

(defun graph-a-base (graph) (first (first (node-antecedents (graph-root graph)))))
(defun graph-u-base (graph) (first (second (node-antecedents (graph-root graph)))))
(defun graph-base (graph) (graph-a-base graph))
(defun set-graph-base (graph base) (setf (node-antecedents (graph-root graph)) (list (list base))))

;;;_______________________________________
;;; select-node-only

(defun select-node-only (node-or-collection)
  (cond ((node-p node-or-collection)
         (unless (node-selected? node-or-collection)
           (setf (node-selected? node-or-collection) t)
           (dolist (cell (node-cells node-or-collection))
             (unless (cell-hidden? cell)
               (focus-and-draw-cell cell)))))
        ((consp node-or-collection)
         (select-node-only (first node-or-collection))
         (select-node-only (rest node-or-collection)))))

;;;_______________________________________
;;; deselect-node-only

(defun deselect-node-only (node-or-collection)
  (cond ((node-p node-or-collection)
         (when (node-selected? node-or-collection)
           (setf (node-selected? node-or-collection) nil)
           (dolist (cell (node-cells node-or-collection))
             (unless (cell-hidden? cell)
               (focus-and-draw-cell cell)))))
        ((consp node-or-collection)
         (deselect-node-only (first node-or-collection))
         (deselect-node-only (rest node-or-collection)))))

;;;_______________________________________
;;; select-node

(defun select-node (node-or-collection &optional (recursive nil) (hidden nil))
  (cond ((node-p node-or-collection)
         (unless (node-selected? node-or-collection)
           (setf (node-selected? node-or-collection) t)
           (dolist (cell (node-cells node-or-collection))
             (unless (or hidden (cell-hidden? cell))
               (focus-and-draw-cell cell))))
         (if (or recursive (node-second-order? node-or-collection))
           (select-node (node-antecedents node-or-collection) recursive hidden)))
        ((consp node-or-collection)
         (select-node (first node-or-collection) recursive hidden)
         (select-node (rest node-or-collection) recursive hidden))))

;;;_______________________________________
;;; deselect-node

(defun deselect-node (node-or-collection &optional (recursive nil))
  (cond ((node-p node-or-collection)
         (when (node-selected? node-or-collection)
           (setf (node-selected? node-or-collection) nil)
           (dolist (cell (node-cells node-or-collection))
             (unless (cell-hidden? cell)
               (focus-and-draw-cell cell))))
         (if (or recursive (node-second-order? node-or-collection))
           (deselect-node (node-antecedents node-or-collection) recursive))
         (let ((consequent (node-consequent node-or-collection)))
           (when (and consequent 
                      (node-second-order? consequent)
                      (node-selected? consequent)
                      (all-antecedents-deselected? consequent))
             (deselect-node consequent nil))))
        ((consp node-or-collection)
         (deselect-node (first node-or-collection) recursive)
         (deselect-node (rest node-or-collection) recursive))))

;;;__________________________________
;;; delete-node

(defun delete-node (node-or-collection &optional (recursive nil) (hidden nil))
  (cond ((node-p node-or-collection)
         (unless (node-deleted? node-or-collection)
           (setf (node-deleted? node-or-collection) t)
           (dolist (cell (node-cells node-or-collection))
             (unless (or hidden (cell-hidden? cell))
               (focus-and-draw-cell cell))))
         (if (or recursive (node-second-order? node-or-collection))
           (delete-node (node-antecedents node-or-collection) recursive hidden)))
        ((consp node-or-collection)
         (delete-node (first node-or-collection) recursive hidden)
         (delete-node (rest node-or-collection) recursive hidden))))

;;;__________________________________
;;; undelete-node

(defun undelete-node (node-or-collection &optional (recursive nil))
  (cond ((node-p node-or-collection)
         (when (node-deleted? node-or-collection)
           (setf (node-deleted? node-or-collection) nil)
           (dolist (cell (node-cells node-or-collection))
             (unless (cell-hidden? cell)
               (focus-and-draw-cell cell))))
         (if (or recursive (node-second-order? node-or-collection) )
           (undelete-node (node-antecedents node-or-collection) recursive))
         (let ((consequent (node-consequent node-or-collection)))
           (when (and consequent 
                      (node-second-order? consequent)
                      (node-selected? consequent)
                      (all-antecedents-deselected? consequent))
             (undelete-node consequent nil))))
        ((consp node-or-collection)
         (undelete-node (first node-or-collection) recursive)
         (undelete-node (rest node-or-collection) recursive))))

;;;_______________________________________
;;; all-antecedents-deselected?

(defun all-antecedents-deselected? (node)
  (when node
    (every #'(lambda (conjunction)
               (every #'(lambda (n) (not (node-selected? n))) conjunction))
           (node-antecedents node))))

;;;_______________________________________
;;; self-ancestor-p

(defun self-ancestor-p (node &optional (ancestor (node-consequent node)))
  (cond ((null ancestor) nil)
        ((eq (node-r-struct ancestor) (node-r-struct node)) t)
        (t (self-ancestor-p node (node-consequent ancestor)))))

;;;_______________________________________
;;; new-var generator for ebl learning

(defvar *last-new-var-id* 0)

(defun reset-last-new-var-id ()
  (setf *last-new-var-id* 0))

(defun create-new-var ()
  (make-pcvar :id (decf *last-new-var-id*)))

(defun create-new-vars (number)
  (let ((result nil))
    (dotimes (i number) (push (create-new-var) result))
    (nreverse result)))


;;;_______________________________________
;;; all-antecedents

(defun all-antecedents (r-struct vars &optional (create-new-vars-for-learning nil))
  (when (rule-p r-struct)
    (mapcar #'(lambda (clause)
                (let* ((body (clause-body clause))
                       (parameters (clause-parameters clause))
                       (new-body (substitute-vars body (unify-list parameters vars)))
                       (new-vars (clause-new-vars clause)))
                  (if (and create-new-vars-for-learning new-vars)
                    (substitute-vars new-body (unify new-vars (create-new-vars (length new-vars))))
                    new-body)))
            (r-clauses r-struct))))

;;;_______________________________________
;;; editor-all-antecedents

(defun editor-all-antecedents (r-struct vars)
  (when (rule-p r-struct)
    (mapcar #'(lambda (clause)
                (let* ((body (rest clause))
                       (parameters (rest (first clause)))
                       (new-vars (compute-new-vars body parameters))
                       (collisions (intersection new-vars vars)))
                  (when collisions
                    (direct-substitute body (direct-mapping collisions (uniquify-variables collisions))))
                  (direct-substitute body (direct-mapping parameters vars))))
            (get-clauses (r-name r-struct)))))

;;;_______________________________________
;;;  connect-clauses

(defun connect-clauses (graph consequent clauses source expand expand-depth)
  (let ((number -1))
    (mapcan #'(lambda (clause) 
                (when clause 
                  (list (connect-clause graph consequent clause (incf number) source expand expand-depth))))
            clauses)))
;;;_______________________________________
;;;  connect-clause

(defun connect-clause (graph consequent clause number source expand expand-depth)
  (if (literal-p clause)
    (let ((conjunction nil))
      (do ((literal clause (literal-next literal)))
          ((null literal) conjunction)
        (setf conjunction 
              (nconc conjunction 
                     (list (connect-literal graph consequent literal number source expand expand-depth))))))
    (mapcar #'(lambda (literal) (connect-literal graph consequent literal number source expand expand-depth)) clause)))

;;;_______________________________________
;;;  connect-literal

(defun connect-literal (graph consequent literal number source expand expand-depth)
  (when literal
    (let ((deleted? nil) (state source) (aux nil) name vars kind node)
      (cond ((literal-p literal) (setf name (if (literal-negated? literal)
                                              'not
                                              (literal-predicate-name literal))
                                       vars (if (literal-negated? literal)
                                              (literal-negated-literals literal)
                                              (literal-variablization literal))
                                       deleted? (literal-deleted? literal)
                                       state (derivation-type (literal-derivation literal))
                                       aux literal))
            ((consp literal) (setf name (first literal)
                                   vars (rest literal)))
            ((eql literal '!) (setf name '!))
            ((eql literal 'fail) (setf name 'fail)))
      (let ((r-struct (get-r-struct name)))
        (cond 
         ((r-p r-struct)
          (setf kind (r-kind r-struct))
          (when (eq kind :undefined)
            (setf state :undefined
                  aux :undefined)))
         ((eql name 'or) (setf kind :OR))
         ((eql name 'and) (setf kind :AND))
         (t (let ((arity (length vars)))
              (setf r-struct (make-r :name name :vars vars :type (make-list arity :initial-element :anything) :arity arity :kind :undefined))
              (set-r-struct name r-struct)
              (add-r-struct r-struct)
              (setf kind :undefined
                    state :undefined
                    aux :undefined))))
        (setf node (get-node graph
                             :consequent consequent
                             :r-struct r-struct
                             :vars (case kind
                                     ((:or :and :not :cut :fail) nil)
                                     (otherwise vars))
                             :kind kind
                             :state state
                             :deleted? deleted?
                             :aux aux))
        (case kind
          (:or
           (setf (node-antecedents node)
                 (when vars
                   (let ((n -1))
                     (mapcar #'(lambda (l) (list (connect-literal graph node l (incf n) source expand expand-depth))) vars)))))
          (:and
           (setf (node-antecedents node) (when vars (list (connect-clause graph node vars number source expand expand-depth)))))
          (:not 
           (setf (node-antecedents node) (when vars (list (connect-clause graph node vars number source expand expand-depth)))))
          (:intensional
           (if (self-ancestor-p node)
             (setf (node-recursive? node) t)
             (let ((antecedents (editor-all-antecedents r-struct vars)))
               (case expand 
                 ((:never nil) (setf (node-recursive? node) t))
                 ((:completely t) (setf (node-antecedents node) (connect-clauses graph node antecedents state expand (decf expand-depth))))
                 (:first-use (if (and (> expand-depth 0) (< (count-if #'(lambda (n) (eq r-struct (node-r-struct n))) (graph-used-nodes graph)) 2))
                               (setf (node-antecedents node) (connect-clauses graph node antecedents state expand (decf expand-depth)))
                               (setf (node-recursive? node) t)))
                 (otherwise (if (> expand-depth 0) 
                              (setf (node-antecedents node) (connect-clauses graph node antecedents state expand (decf expand-depth)))
                              (setf (node-recursive? node) t)
                              ))))))
          (otherwise nil))
        (when deleted? (mark-tree-as-deleted node))
        node))))

;;;_______________________________________
;;; mark-tree-as-deleted

(defun mark-tree-as-deleted (root)
  (setf (node-deleted? root) t)
  (dolist (conjunction (node-antecedents root))
    (dolist (node conjunction)
      (mark-tree-as-deleted node))))


;;;_______________________________________
;;; node-relation-name

(defun node-relation-name (node)
  (let ((r-struct (node-r-struct node)))
    (cond ((r-p r-struct) (r-name r-struct))
          ((node-or? node) 'or)
          ((node-and? node) 'and)
          ((node-root? node) 'root)
          (t 'unknown))))

;;;_______________________________________
;;; node-string

(defun node-string (node)
  (when (node-p node)
    (name-vars-string (node-relation-name node) (node-vars node))))

;;;_______________________________________
;;;  conjunction-containing-node

(defun conjunction-containing-node (node)
  (when (node-p node)
    (let ((consequent (node-consequent node)))
      (when (node-p consequent)
        (do* ((conjunctions (node-antecedents consequent) (rest conjunctions))
              (conjunction (first conjunctions) (first conjunctions))
              (count 0 (incf count)))
             ((member node conjunction) (values conjunction count)))))))

;;;_______________________________________
;;; hilite-node

(defun hilite-node (node-or-collection state &optional (recursive nil) (deselect nil))
  (cond ((node-p node-or-collection)
         (setf (node-state node-or-collection) state)
         (when deselect (setf (node-selected? node-or-collection) nil))
         (when (or recursive (node-second-order? node-or-collection))
           (hilite-node (node-antecedents node-or-collection) state recursive deselect)))
        ((consp node-or-collection)
         (hilite-node (first node-or-collection) state recursive deselect)
         (hilite-node (rest node-or-collection) state recursive deselect))))

;;;________________________________
;;; duplicate-node

(defun duplicate-node (node-or-collection destination-consequent destination-graph &optional (recursive nil) (save-source-in-aux nil))
  (cond ((node-p node-or-collection)
         (let ((duplicate (get-node destination-graph
                                    :consequent destination-consequent
                                    :antecedents nil
                                    :r-struct (node-r-struct node-or-collection)
                                    :vars (node-vars node-or-collection)
                                    :deleted? (node-deleted? node-or-collection)
                                    :selected? (node-selected? node-or-collection)
                                    :recursive? (node-recursive? node-or-collection)
                                    :kind (node-kind node-or-collection)
                                    :state (node-state node-or-collection)
                                    :cells nil
                                    :coverage (mapcar #'duplicate-coverage (node-coverage node-or-collection))
                                    :aux (if save-source-in-aux node-or-collection (node-aux node-or-collection))
                                    )))
           (when (or recursive (node-second-order? node-or-collection))
             (setf (node-antecedents duplicate)
                   (duplicate-node (node-antecedents node-or-collection) duplicate destination-graph recursive save-source-in-aux)))
           duplicate))
         ((consp node-or-collection)
          (mapcar #'(lambda (n-or-c) (duplicate-node n-or-c destination-consequent destination-graph recursive save-source-in-aux))
                  node-or-collection))))

;;;________________________________
;;; find-corresponding-node

(defun find-corresponding-node (node thing corresponding-thing &optional (recursive t))
  (cond ((eq node thing)
         corresponding-thing)
        ((node-second-order? node)
         (node-consequent (find-corresponding-node (first (first (node-antecedents node))) thing corresponding-thing recursive)))
        ((node-p thing)
         (when recursive 
           (find-corresponding-node node (node-antecedents thing) (node-antecedents corresponding-thing) recursive)))
        ((consp thing)
         (or (find-corresponding-node node (first thing) (first corresponding-thing) recursive)
             (find-corresponding-node node (rest thing) (rest corresponding-thing) recursive)))
        ((graph-p thing)
         (find-corresponding-node node (graph-root thing) (graph-root corresponding-thing) recursive))
        (t nil)))


;;;_______________________________________
;;;  nodes-in-tree

(defun nodes-in-tree (graph-node-or-collection)
  (let ((nodes nil))
    (labels 
      ((process (node-or-collection)
         (cond ((node-p node-or-collection)
                (setf nodes (push node-or-collection nodes))
                (process (node-antecedents node-or-collection)))
               ((consp node-or-collection)
                (process (first node-or-collection))
                (process (rest node-or-collection))))))
      (cond ((graph-p graph-node-or-collection) (process (graph-root graph-node-or-collection)))
            ((node-p graph-node-or-collection) (process graph-node-or-collection))
            ((consp graph-node-or-collection) (process graph-node-or-collection)))
      nodes)))

;;;_______________________________________
;;;  selected-nodes

(defun selected-nodes (graph-node-or-collection &optional (only-first-along-branch nil))
  (let ((selected-nodes nil))
    (labels
      ((process (node-or-collection)
         (cond ((node-p node-or-collection)
                (if (node-selected? node-or-collection)
                  (push node-or-collection selected-nodes))
                (unless (and only-first-along-branch (node-selected? node-or-collection))
                  (process (node-antecedents node-or-collection))))
               ((consp node-or-collection)
                (process (first node-or-collection))
                (process (rest node-or-collection))))))
      (cond ((graph-p graph-node-or-collection) (process (graph-root graph-node-or-collection)))
            ((node-p graph-node-or-collection) (process graph-node-or-collection))
            ((consp graph-node-or-collection) (process graph-node-or-collection))))
    (nreverse selected-nodes)))

;;;_______________________________________
;;;  generate-graph

(defun generate-graph (list-of-literals-and-r-structs &optional (source nil) (expand *default-expand*) (expand-depth *default-expand-depth*))
  (when list-of-literals-and-r-structs
    (let* ((graph (create-graph))
           (root (graph-root graph))
           (new-node nil))
      (dolist (thing list-of-literals-and-r-structs)
        (cond ((r-p thing)
               (setf new-node (connect-literal graph root (cons (r-name thing) (r-vars thing)) nil source expand expand-depth)))
              ((consp thing)
               (setf new-node (connect-literal graph root thing nil source expand expand-depth)))
              (t
               (setf new-node nil)))
        (when (node-p new-node)
          (add-disjunctive-antecedent-to-node root (list new-node))))
      (unless (node-antecedents root)
        (dispose-graph graph)
        (setf graph nil))
      graph)))

;;;____________________________________________________________________________________
;;; generate-learned-description-graph

(defun generate-learned-description-graph (head learned-description &optional (expand *default-expand*) (expand-depth *default-expand-depth*))
  (when head
    (let* ((graph (create-graph))
           (root (graph-root graph))
           (head-node (get-node graph
                                :consequent root
                                :vars (r-vars head)
                                :r-struct head
                                :kind :intensional)))
      (setf (node-antecedents root) (list (list head-node))
            (node-antecedents head-node)
            (if learned-description (connect-clauses graph head-node learned-description nil expand expand-depth) nil))
      graph)))

;;;_______________________________________
;;;  generate-domain-theory-graph

(defun generate-domain-theory-graph (goal-concept &optional (source nil) (expand *default-expand*) (expand-depth *default-expand-depth*))
  (when (some-relation-is-user-defined)
    (let* ((graph (create-graph))
           (root (graph-root graph)))
      (when goal-concept
        (add-disjunctive-antecedent-to-node
         root (list (connect-literal graph root goal-concept nil source expand expand-depth))))
      (dolist (r-struct *r-structs*)
        (unless (or (member r-struct *special-r-structs*)
                    (some #'(lambda (n) (eq (node-root n) root)) (r-nodes r-struct)))
          (add-disjunctive-antecedent-to-node
           root (list (connect-literal graph root (cons (r-name r-struct) (r-vars r-struct)) nil source expand expand-depth)))))
      graph)))


;;;_______________________________________
;;; add-disjunctive-antecedent-to-node

(defun add-disjunctive-antecedent-to-node (node conjunction)
  (setf (node-antecedents node) (nconc (node-antecedents node) (list conjunction)))
  (dolist (antecedent conjunction)
    (setf (node-consequent antecedent) node))
  conjunction)

;;;_______________________________________
;;; tree-depth

(defun tree-depth (root)
  (case (node-kind root)
    ((:intensional :not :and :or :root)
     (let ((max-depth 0))
       (dolist (conjunction (node-antecedents root) max-depth)
         (dolist (node conjunction max-depth)
           (setf max-depth (max max-depth (+ 1 (tree-depth node))))))))
    (otherwise 1)))

;;;_______________________________________
;;;  node-real-consequent

(defun node-real-consequent (node)
  (when (node-p node)
    (let ((consequent (node-consequent node)))
      (cond ((node-second-order? consequent) (node-real-consequent consequent))
            ((node-root? consequent) (values node node))
            (t (values consequent node))))))

;;;_______________________________________
;;;  direct-antecedent-of

(defun direct-antecedent-of (consequent antecedent)
  (when (node-p antecedent)
    (eq (node-consequent antecedent) consequent)))

;;;_______________________________________
;;;  antecedent-of

(defun antecedent-of (consequent antecedent)
  (when (node-p antecedent)
    (let ((c (node-consequent antecedent)))
      (cond ((eq c consequent) t)
            ((node-second-order? c) (antecedent-of consequent c))))))

;;;_______________________________________
;;;  ancestor-of

(defun ancestor-of (consequent ancestor)
  (cond ((not (node-p ancestor)) nil)
        ((eq (node-consequent ancestor) consequent) t)
        (t (ancestor-of consequent (node-consequent ancestor)))))

;;;_______________________________________
;;;  nodes-form-conjunction

(defun nodes-form-conjunction (possible-conjunction)
  (let ((conjunction (conjunction-containing-node (first possible-conjunction))))
    (when (and (every #'(lambda (node) (member node conjunction)) possible-conjunction)
               (every #'(lambda (node) (member node possible-conjunction)) conjunction))
      possible-conjunction)))

;;;_______________________________________
;;;  antecedent-conjunction-of

(defun antecedent-conjunction-of (consequent possible-conjunction)
  (let ((conjunction (nodes-form-conjunction possible-conjunction)))
    (if (antecedent-of consequent (first conjunction))
      conjunction)))

;;;_______________________________________
;;;  contained-in

(defun contained-in (node collection)
  (cond ((node-p collection) (eq node collection))
        ((consp collection) (or (contained-in node (first collection))
                                (contained-in node (rest collection))))
        (t nil)))

;;;_______________________________________
;;;  conjunction-p

(defun conjunction-p (conjunction)
  (and (consp conjunction) (every #'node-p conjunction)))

;;;_______________________________________
;;;  disjunction-p

(defun disjunction-p (disjunction)
  (and (consp disjunction) (every #'conjunction-p disjunction)))

;;;_______________________________________
;;; negate-node

(defun negate-node (node graph &optional (selected? nil))
  (when (node-p node)
    (let* ((consequent (node-consequent node))
           (not-node (get-node graph :consequent consequent :antecedents (list (list node)) :r-struct (get-r-struct 'not) :kind :not :state nil :selected? selected?)))
      (when consequent
        (setf (node-antecedents consequent) (nsubst not-node node (node-antecedents consequent))))
      (setf (node-consequent node) not-node)
      not-node)))
 
;;;_______________________________________
;;; negate-conjunction

(defun negate-conjunction (conjunction graph &optional (selected? nil))
  (when conjunction
    (let* ((consequent (node-consequent (first conjunction)))
           (not-node (get-node graph :consequent consequent :antecedents (list conjunction) :r-struct (get-r-struct 'not)  :kind :not :state nil :selected? selected?)))
      (when consequent
        (setf (node-antecedents consequent) (nsubstitute (list not-node) conjunction (node-antecedents consequent))))
      (dolist (node conjunction)
        (setf (node-consequent node) not-node))
      not-node)))

;;;_______________________________________
;;; conjoin-nodes-with-and-node

(defun conjoin-nodes-with-and-node (conjunction graph &optional (selected? nil))
  (when conjunction
    (let* ((consequent (node-consequent (first conjunction)))
           (and-node (get-node graph :consequent consequent :antecedents (list conjunction) :r-struct (get-r-struct 'and) :kind :and :state nil :selected? selected?)))
      (when consequent
        (setf (node-antecedents consequent) (nsubstitute (list and-node) conjunction (node-antecedents consequent))))
      (dolist (node conjunction)
        (setf (node-consequent node) and-node))
      and-node)))

;;;_______________________________________
;;; disjoin-nodes-with-or-node

(defun disjoin-nodes-with-or-node (conjunction graph &optional (selected? nil))
  (when conjunction
    (let* ((consequent (node-consequent (first conjunction)))
           (or-node (get-node graph :consequent consequent :antecedents (mapcar #'list conjunction) :r-struct (get-r-struct 'or) :kind :or :state nil :selected? selected?)))
      (when consequent
        (setf (node-antecedents consequent) (nsubstitute (list or-node) conjunction (node-antecedents consequent))))
      (dolist (node conjunction)
        (setf (node-consequent node) or-node))
      or-node)))

;;;_______________________________
;;;  reduce-nodes

(defun reduce-nodes (nodes)
  (when (and (disjunction-p nodes) (null (rest nodes))) (setf nodes (first nodes)))
  (when (and (conjunction-p nodes) (null (rest nodes))) (setf nodes (first nodes)))
  nodes)

;;;_______________________________________
;;;  replace-node
;;;
;;;  replaces a node with its antecedents

(defun replace-node (node graph)
  (let* ((consequent (node-consequent node))
         (antecedents (node-antecedents node))
         (conjunction (conjunction-containing-node node))
         (replacement nil))
    (dolist (nodes (butlast antecedents))
      (push (mapcan #'(lambda (n) (if (eq n node) nodes (list (duplicate-node n consequent graph t)))) conjunction) replacement))
    (push (mapcan #'(lambda (n) (if (eq n node) (first (last antecedents)) (list n))) conjunction) replacement)
    (dolist (c antecedents)
      (dolist (n c)
        (setf (node-consequent n) consequent)
        (select-node n)))
    (setf (node-antecedents consequent) (mapcan #'(lambda (c) (if (eq c conjunction) (nreverse replacement) (list c))) (node-antecedents consequent)))
    (free-node node graph)
    (when (node-not? consequent)
      (convert-negated-disjunction consequent graph))
    antecedents))

;;;_______________________________________
;;;  convert-negated-disjunction

(defun convert-negated-disjunction (not-node graph)
  (let* ((conjunction (conjunction-containing-node not-node))
         (consequent (node-consequent not-node))
         (antecedents (node-antecedents not-node)))
    (when (rest antecedents)
      (let* ((replacement (mapcar #'(lambda (conjunction)
                                      (let ((not-node (get-node graph :consequent consequent
                                                                :antecedents (list conjunction)
                                                                :kind :not :state nil :selected? t)))
                                        (dolist (n conjunction)
                                          (setf (node-consequent n) not-node))
                                        not-node))
                                  antecedents))
             (new-conjunction (mapcan #'(lambda (n) (if (eq n not-node) replacement (list n))) conjunction)))
        (setf (node-antecedents consequent) (nsubstitute new-conjunction conjunction (node-antecedents consequent))))
      (free-node not-node graph))))


;;;_______________________________________
;;; disconnect-and-free-tree

(defun disconnect-and-free-tree (node-or-collection graph)
  (cond ((node-p node-or-collection)
         (let* ((conjunction (conjunction-containing-node node-or-collection))
                (consequent (node-consequent node-or-collection))
                (nodes-to-free (nodes-in-tree node-or-collection))
                (new-conjunction (delete node-or-collection conjunction)))
           (cond ((and (node-second-order? consequent) (null new-conjunction) (null (rest (node-antecedents consequent))))
                  (disconnect-and-free-tree consequent graph))
                 (t
                  (setf (node-antecedents consequent)
                        (if new-conjunction
                          (nsubstitute new-conjunction conjunction (node-antecedents consequent))
                          (delete conjunction (node-antecedents consequent))))
                  (free-node nodes-to-free graph)))))
        ((consp node-or-collection)
         (disconnect-and-free-tree (first node-or-collection) graph)
         (disconnect-and-free-tree (rest node-or-collection) graph))))

;;;_______________________________________
;;; replace-antecedent-conjunction

(defun replace-antecedent-conjunction (node old-conjunction new-conjunction)
  (if new-conjunction
    (setf (node-antecedents node) (nsubstitute new-conjunction old-conjunction (node-antecedents node)))
    (setf (node-antecedents node) (delete old-conjunction (node-antecedents node)))))

;;;_______________________________________
;;; conjoin

(defun conjoin (nodes unattached-nodes graph)
  (when (nodes-form-conjunction nodes)
    (let* ((conjunction (conjunction-containing-node (first nodes)))
           (consequent (node-consequent (first conjunction)))
           (u-base (graph-u-base graph)))
      (dolist (node unattached-nodes)
        (if (direct-antecedent-of u-base node)
          (let ((old-conjunction (conjunction-containing-node node)))
            (replace-antecedent-conjunction u-base old-conjunction (delete node old-conjunction))
            (setf (node-consequent node) consequent)
            (nconc conjunction (list node)))
          (nconc conjunction (list (duplicate-node node consequent graph t)))))
      conjunction)))

;;;_______________________________________
;;; disjoin

(defun disjoin (consequent unattached-nodes graph)
  (let* ((conjunction nil)
         (u-base (graph-u-base graph)))
    (dolist (node unattached-nodes)
      (if (direct-antecedent-of u-base node)
        (let ((old-conjunction (conjunction-containing-node node)))
          (replace-antecedent-conjunction u-base old-conjunction (delete node old-conjunction))
          (setf (node-consequent node) consequent
                conjunction (nconc conjunction (list node))))
        (setf conjunction (nconc conjunction (list (duplicate-node node consequent graph t))))))
    (setf (node-antecedents consequent) (nconc (node-antecedents consequent) (list conjunction)))
    conjunction))

;;;_______________________________________
;;; ANTECEDENT-LEADING-TO-NODE

(defun antecedent-leading-to-node (consequent node)
  (do* ((n node c)
        (c node (node-consequent n)))
       ((eq c consequent) n)))

;;;_______________________________________
;;; APPLY-FUNCTION-TO-TREE

(defun apply-function-to-tree (node-or-collection function &optional (postorder? nil))
  (cond ((consp node-or-collection)
         (apply-function-to-tree (first node-or-collection) function postorder?)
         (apply-function-to-tree (rest node-or-collection) function postorder?))
        ((node-p node-or-collection)
         (cond (postorder?
                (apply-function-to-tree (node-antecedents node-or-collection) function postorder?)
                (funcall function node-or-collection))
               (t
                (funcall function node-or-collection)
                (apply-function-to-tree (node-antecedents node-or-collection) function postorder?))))))

;;;_______________________________________
;;;  LOWEST-COMMON-ANCESTOR

(defun lowest-common-ancestor (node1 node2)
  (cond ((not (node-p node1)) nil)
        ((not (node-p node2)) nil)
        ((eq node1 node2) node1)
        ((ancestor-of node1 node2) node1)
        ((ancestor-of node2 node1) node2)
        (t (lowest-common-ancestor (node-consequent node1) node2))))

;;;_______________________________________
;;;  LOWEST-COMMON-ANCESTOR-OF-FRONTIER

(defun lowest-common-ancestor-of-frontier (graph)
  (do* ((node (next-selected-node (graph-base graph) :include-self t) (next-selected-node node))
        (lowest-common-ancestor node))
       ((null node) lowest-common-ancestor)
    (setq lowest-common-ancestor (lowest-common-ancestor lowest-common-ancestor node))))