
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

(defun set-focl-problem (pred-name &rest learning-parameters)
  (let* ((goal-concept-name (getf learning-parameters :goal-concept-name))
         (goal-concept-r-struct (get-r-struct goal-concept-name)))
    (when goal-concept-r-struct
      (setf *goal-concept* (cons goal-concept-name (r-vars goal-concept-r-struct))))
    (setf *predicate-being-learned* pred-name
          *focl-problem* (cons pred-name learning-parameters)
          *focl-problems* (delete-if #'(lambda (problem)
                                         (and (equal (first problem) pred-name)
                                              (equal (getf (rest problem) :goal-concept-name) goal-concept-name)))
                                     *focl-problems*)
          *focl-problems* (push *focl-problem* *focl-problems*)))
  (list 'focl-problem pred-name))

(defmacro def-focl-problem (pred-name &rest learning-parameters)
  `(apply #'set-focl-problem ',pred-name ',learning-parameters))


;;;  Retained for compatiblity
(defmacro def-es-focl-problem (pred-name rule-name)
  `(apply #'set-focl-problem ',pred-name (list :goal-concept-name ',rule-name)))


;; -- deduce ----------------------------------------

(defmacro delay (&body body)
  `(make-generator :closure #'(lambda () ,@body)))


;; -- count ----------------------------------------

;;;all-images (fn lis)- like  mapcar, only doesn't add nil to value
(defmacro all-images (fn lis)
    `(do* ((flist ,lis (cdr flist))
                    (little-result )
                    (result nil))  ((null flist) (nreverse result))
           (if (setq little-result (funcall ,fn (car flist)))
                        (push little-result result))))

;;;all-unique-images (fn lis)- like  mapcar, only doesn't add nil to value
;;;also doesn't add duplicates (while equal check) to value
(defmacro all-unique-images (fn lis)
  `(do* ((flist ,lis (cdr flist))
         (little-result )
         (result nil))
    ((null flist) (nreverse result))
    (if (setq little-result (funcall ,fn (car flist)))
        (unless (member little-result result :test #'equal)
          (push little-result result)))))


;;;_______________________________________________________________
;;;  def-pred         defines a new predicate

(defmacro def-pred (name &key
                         pos
                         (neg nil)
                         (arity nil)
                         (vars nil)
                         (type nil)
                         (mode nil)
                         (determinacy nil)
                         (questions nil)
                         (constraint nil)
                         (commutative nil)
                         (induction t)
                         (try-constants nil)
                         (sort-fn nil)
                         (infix nil)
                         )
  (setf arity (or arity (length (or (first pos) (first neg) vars type mode)))
        vars (or vars (make-list-unique-vars arity))
        type (or type (make-list arity :initial-element :anything)))
  `(progn 
     (let* ((name ',name)
            (pos ',pos)
            (neg ',neg)
            (arity ',arity)
            (vars ',vars)
            (type ',type)
            (bucket (assoc name *extensional-preds*))
            (table-size (+ 1 (* arity 10)))
            (r-struct 
             (make-r 
              :name name
              :arity arity
              :vars vars
              :type type
              :mode ',mode
              :determinacy ',determinacy
              :kind :extensional
              :questions ',questions
              :infix ',infix
              :pos pos
              :neg (if (eq neg ':computed)
                     (remove-if #'(lambda(X) (member X pos :test #'equal)) (all-typed-tuples type))
                     neg)
              :pos-hash (make-pos-hash pos)
              :neg-hash (make-hash-table :test #'equal :size table-size)   
              :neg-hash-for-negated (make-hash-table :test #'equal :size table-size)   
              :slot-value-hash (make-slot-value-hash arity pos)
              :constraint ',constraint
              :commutative ',commutative
              :induction ',induction
              :try-constants ',try-constants
              :sort-fn ,sort-fn
              :sort-fn-def ',sort-fn
              :compute-neg (eq neg ':computed)
              :prolog-function ,(focl-create-fact-defun name arity)
              )))
       (set-r-struct name r-struct)
       (add-r-struct r-struct)
       (if bucket
         (setf (cdr bucket) r-struct)
         (push (cons name r-struct) *extensional-preds*))
       name)))


;;;_______________________________________________________________
;;;  def-reduction-pred         defines a new predicate

(defmacro def-reduction-pred (name &key
                                   pos
                                   (neg nil)
                                   (arity nil)
                                   (vars nil)
                                   (type nil)
                                   (mode nil)
                                   (determinacy nil)
                                   (questions nil)
                                   (constraint nil)
                                   (commutative nil)
                                   (induction t)
                                   (reduction-type nil) 
                                   (reduction-variabilization nil)
                                   (infix nil)
                                   )
  (setf arity (or arity (length (or (first pos) (first neg) vars type mode)))
        vars (or vars (make-list-unique-vars arity))
        type (or type (make-list arity :initial-element :anything)))
  `(progn
     (let* ((name ',name)
            (pos ',pos)
            (neg ',neg)
            (arity ',arity)
            (vars ',vars)
            (type ',type)
            (bucket (assoc name *extensional-preds*))
            (table-size (+ 1 (* arity 10)))
            (r-struct
             (make-reduction-pred
              :name name
              :arity arity
              :type type
              :mode ',mode
              :determinacy ',determinacy
              :kind :extensional
              :questions ',questions
              :infix ',infix
              :pos pos
              :neg (if (eq neg ':computed)
                     (remove-if #'(lambda(X) (member X pos :test #'equal)) (all-typed-tuples type))
                     neg)
              :pos-hash (make-pos-hash pos)
              :neg-hash (make-hash-table :test #'equal :size table-size)   
              :neg-hash-for-negated (make-hash-table :test #'equal :size table-size)   
              :slot-value-hash (make-slot-value-hash arity pos)
              :constraint ',constraint
              :commutative ',commutative
              :induction ',induction
              :reduction-type ',reduction-type
              :reduction-variabilization ',reduction-variabilization
              :prolog-function ,(focl-create-fact-defun name arity)
              :compute-neg (eq neg ':computed)
              )))
       (set-r-struct name r-struct)
       (add-r-struct r-struct)
       (if bucket
         (setf (cdr bucket) r-struct)
         (push (cons name r-struct) *extensional-preds*))
       name)))

;;;_______________________________________________________________
;;;  def-builtin         defines a new builtin

(defmacro def-builtin (name function 
                            &key
                            (arity nil)
                            (vars nil)
                            (type nil)
                            (mode nil)
                            (determinacy nil)
                            (questions nil)
                            (constraint nil)
                            (commutative nil)
                            (treat-as-commutative t)  ;; these defaults are a poor choice
                            (induction t)             ;;
                            (try-constants t)         ;;
                            (sort-fn '#'<)            ;;
                            (equality? nil)
                            (infix nil)
			    (create-prolog-function nil)
                            )
  
  (let* ((fn (second function))
         (lisp-vars (if (and (listp fn) (eql (first fn) 'lambda)) (second fn))))
    (when (and lisp-vars (not (= (length vars) (length lisp-vars))))
      (setf vars (mapcar #'(lambda (var) (if (variable-p var) var (intern (format nil "?~a" var)))) lisp-vars)))
    (setf arity (or arity 
                    (length (or vars type mode
                                (progn
                                  (format t "~%Warning - the number of argments for ~A was not specified, assuming two." name)
                                  '(1 2)))))
          vars (or vars (make-list-unique-vars arity))
          type (or type (make-list arity :initial-element :number))
          mode (or mode (make-list arity :initial-element :+))))
  `(progn 
     
     (let ((r-struct (make-r :name ',name
                             :arity ',arity
                             :vars ',vars
                             :type ',type
                             :mode ',mode
                             :determinacy ',determinacy
                             :infix ',infix
                             :kind :builtin
                             :questions ',questions
                             :equality? ',equality?
                             :function ,function
                             :sort-fn ,sort-fn
                             :constraint ',constraint
                             :commutative ',commutative
                             :treat-as-commutative ',treat-as-commutative
                             :induction ',induction
                             :try-constants ',try-constants
                             :function-def ',function
                             :prolog-function ,(make-prolog-function-for-builtin  name function arity create-prolog-function)
                             :sort-fn-def ',sort-fn
                             ))
           (bucket (assoc ',name *builtin-preds*)))
       (set-r-struct ',name r-struct)
       (add-r-struct r-struct)
       (if bucket (setf (cdr bucket) r-struct)
           (push (cons ',name r-struct) *builtin-preds*))
       ',name)))


;;;_______________________________________________________________
;;;  def-builtin-fn         defines a new builtin function

(defmacro def-builtin-fn (name &key
                               (arity nil)
                               (vars nil)
                               (type nil)
                               (mode nil)
                               (determinacy nil)
                               (questions nil)
                               (constraint :unique-vars)
                               (commutative nil)
                               (treat-as-commutative t)
                               (induction nil)
                               (infix nil)
                               )
  `(let* ((name ',name)
          (arity (or ',arity (length (or ',vars ',type ',mode '(1 2)))))
          (vars (or ',vars (make-list-unique-vars arity)))
          (type (or ',type (cons :number (make-list (- arity 1) :initial-element :anything))))
          (mode (or ',mode (make-list arity :initial-element :?)))
          (bucket (assoc name *arithmetic-ops*))
          (r-struct (make-r :name name
                            :arity arity
                            :vars vars
                            :type type
                            :mode mode
                            :determinacy ',determinacy
                            :kind :builtin-fn
                            :questions ',questions
                            :infix ',infix
                            :function nil
                            :constraint ',constraint
                            :commutative ',commutative
                            :treat-as-commutative ',treat-as-commutative
                            :induction ',induction
                            )))
     (set-r-struct name r-struct)
     (add-r-struct r-struct)
     (if bucket (setf (rest bucket) r-struct)
         (push (cons name r-struct) *builtin-fns*))
     name))

;;;_______________________________________________________________
;;;  def-arithmetic-op         defines a new arithmetic operator

(defmacro def-arithmetic-op (name function
                                  &key
                                  (arity nil)
                                  (vars nil)
                                  (type nil)
                                  (mode nil)
                                  (determinacy nil)
                                  (questions nil)
                                  (constraint :unique-vars)
                                  (commutative nil)
                                  (treat-as-commutative t)
                                  (induction nil)
                                  (infix nil)
                                  (try-constants nil)
                                  (sort-fn '#'<)
                                  )
  `(let* ((name ',name)
          (is-op-name ',(intern (format nil "is-~A" name)))
          (arity (or ',arity (length (or ',vars ',type ',mode '(1 2)))))
          (vars (or ',vars (make-list-unique-vars arity)))
          (type (or ',type (make-list arity :initial-element :number)))
          (mode (or ',mode (make-list arity :initial-element :+)))
          (determinacy ',determinacy)
          (questions ',questions)
          (function ,function)
          (constraint ',constraint)
          (commutative ',commutative)
          (treat-as-commutative ',treat-as-commutative)
          (induction ',induction)   
          (bucket (assoc name *arithmetic-ops*))
          (is-bucket (assoc is-op-name *is-ops*))
          (r-struct
           (make-r :name name
                   :arity arity
                   :vars vars
                   :type type
                   :mode mode
                   :determinacy determinacy
                   :kind :arithmetic-op
                   :questions questions
                   :infix ',infix
                   :function function
                   :constraint constraint
                   :commutative commutative
                   :treat-as-commutative treat-as-commutative
                   :induction induction
                   :try-constants ',try-constants
                   :function-def ',function
                   :sort-fn-def ',sort-fn

                   ))
          (is-op-r-struct
           (make-r :name is-op-name
                   :arity (1+ arity)
                   :type ',(cons (car type) type)
                   :mode (cons ':- mode)
                   :kind :is-op
                   :infix ',infix
                   :function name
                   :constraint constraint
                   :commutative commutative
                   :induction induction
                   )))
     (set-r-struct name r-struct)
     (add-r-struct r-struct)
     (set-r-struct is-op-name is-op-r-struct)
     (if bucket (setf (cdr bucket) r-struct)
         (push (cons name r-struct) *arithmetic-ops*))
     (if is-bucket (setf (cdr is-bucket) is-op-r-struct)
         (push (cons is-op-name is-op-r-struct) *is-ops*))
     name))

;;;_______________________________________________________________
;;;  def-rule         defines a new rule

(defmacro def-rule (name &key
                         clauses
                         (vars (rest (first (first clauses))))
                         (arity (length vars))
                         (type (make-list arity :initial-element :anything))
                         (mode (make-list arity :initial-element :?))
                         (determinacy nil)
                         (questions nil)
                         (from-r-struct nil)
                         (from-cliche nil)
                         (constraint nil)
                         (commutative nil)
                         (induction t)
                         (infix nil)
                         (try-constants nil)
                         (sort-fn nil))
  `(progn 
     (let*  ((function ,(focl-create-prolog-function name arity clauses))
             (name ',name)
             (arity ',arity)
             (vars ',vars)
             (type ',type) 
             (mode ',mode)
             (determinacy ',determinacy)
             (questions ',questions)
             (infix ',infix)
             (clauses ',clauses)
             (from-r-struct ',from-r-struct)
             (from-cliche ',from-cliche)
             (constraint ',constraint)
             (commutative ',commutative) 
             (induction ',induction)
             (bucket (assoc name *intensional-preds*))
             (table-size (+ (* arity 10) 1))
             (clauseno -1)
             (r-struct (make-r
                        :name name
                        :arity arity
                        :vars vars
                        :type type
                        :mode mode
                        :determinacy determinacy
                        :kind :intensional
                        :questions questions
                        :infix infix
                        :constraint constraint
                        :commutative commutative
                        :induction induction
                        :try-constants ',try-constants
                        :sort-fn ,sort-fn
                        :sort-fn-def ',sort-fn
                        :from-r-struct from-r-struct
                        :from-cliche from-cliche
                        :prolog-function function
                        :clauses
                        (mapcar
                         #'(lambda (clause)
                             (let* ((new (uniquify-variables clause))
                                    (head (first new))
                                    (args (rest head))
                                    (body (rest new)))
                               (make-clause
                                :head (first head)
                                :parameters args
                                :new-vars (compute-new-vars body args)
                                :body body
                                :number (incf clauseno)
                                :neg-tuples-hash (make-hash-table :test #'equal :size 2)
                                )))
                         clauses)
                        :neg-hash (make-hash-table :test #'equal :size table-size)   
                        :neg-hash-for-negated (make-hash-table :test #'equal :size table-size)
                        )))
       #+:allegro (declare (ignore-if-unused bucket))
       #+:ccl-2 (declare (ignore-if-unused bucket))
       ,(unless from-r-struct
          `(progn
             (add-r-struct r-struct)
             (if bucket (setf (cdr bucket) r-struct)
                 (push (cons name r-struct) *intensional-preds*))))
       (set-r-struct name r-struct)
       (set-clauses name clauses)
       name)))


(defmacro def-type (name &rest instances)
  `(let ((name ',(if (consp name) (car name) name))
         (properties ',(if (consp name) (cdr name) nil)))
     (setf *all-types* (sort (pushnew name *all-types*) #'universal<))
     (when (fboundp 'update-types) (update-types))
     (setf (get name 'focl-instances) ',instances)
     (mapc #'(lambda(p)(setf (get name p) (if (member p properties) t nil)))
           *type-properties*)
     name))

(defmacro get-type-instances (type)
  `(or (get ,type 'focl-instances)
       (rest (assoc ,type *domain*))))


(defmacro set-type-instances (type instances)
  `(setf (get ,type 'focl-instances) ,instances))

(defmacro pushnew-type-instance (type instance)
  `(pushnew ,instance (get ,type 'focl-instances)))



;; -- focl

;; -- find-literal

(defmacro check-pruning-III-and-prune 
          (covered-all-pos-tuples return-point winner)
  `(when (and ,covered-all-pos-tuples *stop-when-all-pos-covered*)
     (setf *covered-all-pos-tuples* t)
     (return-from ,return-point ,winner
                )))

(defmacro check-pruning-III (covered-all-pos-tuples return-point values)
  `(when (and *stop-when-all-pos-covered* ,covered-all-pos-tuples)
     (return-from ,return-point ,values)))

(defmacro pruning-III-test (covered-all-pos-tuples)
  `(and *stop-when-all-pos-covered* ,covered-all-pos-tuples))

;; -- builtin


;; note these are also used by cliches.lisp

;;  rv  who    date       reason
;;  00  glenn  05/13/91   added to process variabilizations with thresholds for relational
;;    builtins (i.e., non-equality builtins) body is passed in to facilitate use with cliches 
;;    which don't compute info-gain for thresholds in the middle of the cliche

;;;  might want to use these in builtins
(defmacro process-threshold-variabilizations (builtins 
                                              var-restrictions
                                              variables
                                              variable-types
                                              pos-tuples
                                              neg-tuples
                                              variabilization
                                              instantiated-cliche
                                              exit-cond ; for do
                                              &rest body)
  `(when ,builtins
             ; for now assume all builtins require the same # of bits
     (cond 
      ((and *stopping-criteria-enabled* (predicate-requires-too-many-bits (rest (car ,builtins)))) nil)
      (t
       (let (pos
             neg
             boundary-pts
             (overall-builtin-type (overall-builtin-type ,builtins))
             sort-fn
             last-sort-fn) ; to filter variables
         (do* ((old-variables 
                ; (2)
                (apply-thresh-var-restrictions ,var-restrictions 
                                               ,variables
                                               ,instantiated-cliche)
                (cdr old-variables))
               (var (car old-variables) (car old-variables))
               (old-types (retrieve-corresponding-elements old-variables ,variables ,variable-types)
                          (cdr old-types))
               (type (car old-types) (car old-types)))
              
              ((null old-variables) ,exit-cond)
           (setq last-sort-fn nil)
           ;;compute info gain of predicate and negation, updating max if necessary
           (when (and (satisfies-type type overall-builtin-type) ; first filter out vars
                      (bound-var? var))
             (do* ((comps (if (eql overall-builtin-type 'numeric-type) ; use only applicable builtins
                           ,builtins 
                           (remove-if-not 
                            #'(lambda (c) 
                                (satisfies-type type (get-builtin-type (cdr c))))
                            ,builtins))
                         (cdr comps))
                   (comp (cdr (car comps)) (cdr (car comps))))
                  ((null comps))
               ;; sort points and set up sort-fn (hopefully won't intersperce non-numeric and 
               ;; numeric builtins - we may want to separate them or order the builtins by their
               ;; sort functions)
               (when (not (eql last-sort-fn (setq sort-fn (r-sort-fn comp))))
                 (setq pos (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) ,pos-tuples) 
                                 sort-fn))
                 (setq neg (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) ,neg-tuples)
                                 sort-fn))
                 (setq boundary-pts (compute-boundary-points pos neg sort-fn))
                 (setq last-sort-fn sort-fn))
                 
               (do* ((boundary-points boundary-pts (cdr boundary-points))
                     (boundary-point (car boundary-points) (car boundary-points)))
                    ((null boundary-points))
                 (setq ,variabilization (list var boundary-point))
                 ,@body)))))))))


;;  rv  who    date       reason
;;  00  glenn  05/13/91   added to process variabilizations which introduce constants for 
;;    equality builtins body is passed in to facilitate use with cliches which don't 
;;    compute info-gain for thresholds in the middle of the cliche

;;; note eql is a pre-defined builtin for this and will only be available internally
;;; i.e., constants can only be added in via cliches.
;;; or at least figure out later where you want them

(defmacro process-equality-constant-varzns (builtins
                                            var-restrictions
                                            variables 
                                            variable-types
                                            pos-tuples
                                            neg-tuples
                                            variabilization
                                            negated?
                                            instantiated-cliche
                                            exit-cond
                                            &rest body)
  `(when ,builtins
     (let (pos
           neg
           (overall-builtin-type (overall-builtin-type ,builtins))) ; to filter variables
       (do* ((old-variables 
              ; (2)
              (apply-thresh-var-restrictions ,var-restrictions 
                                             ,variables
                                             ,instantiated-cliche)
              (cdr old-variables))
             (var (car old-variables) (car old-variables))
             (old-types (retrieve-corresponding-elements old-variables 
                                                         ,variables ,variable-types)
                        (cdr old-types))
             (type (car old-types) (car old-types)))
            
            ((null old-variables) ,exit-cond)
         ;;run over pos and neg vals only if var is non-numeric and bound
         (when (and (satisfies-type type overall-builtin-type) ; first filter out vars
                    (bound-var? var))
           ; only need to check positive tuple vals when dealing with positive literals - 
           ; otherwise (eql var constant) won't match any positive tuples
           (setq pos (remove-duplicates
                      (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) ,pos-tuples)))
           (setq neg (remove-duplicates
                      (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) ,neg-tuples)))
           (do* ((comps (if (eql overall-builtin-type 'numeric-type) ; use only applicable builtins
                          ,builtins 
                          (remove-if-not 
                           #'(lambda (c) 
                               (satisfies-type type (get-builtin-type (cdr c))))
                           ,builtins))
                        (cdr comps))
                 (comp (cdr (car comps)) 
                        (cdr (car comps))))
                ((null comps))
             (do* ((constants pos (cdr constants))
                   (constant (car constants) (car constants)))
                  ((null constants))
               (setq ,variabilization (list var constant))
               (setq ,negated? nil)
               ,@body)
             ; only need to check negative tuple values for negative eql literals - otherwise
             ; won't exclude any negative examples

             (when *cliches-can-have-negated-components?*
             (do* ((constants neg (cdr constants))
                   (constant (car constants) (car constants)))
                  ((null constants))
               (setq ,variabilization (list var constant))
               (setq ,negated? t)
               ,@body))))))))

(defmacro reset-and-return-values (pred varzn neg? tuples-thru return-list?)
  `(if ,return-list?
     (values (list ,pred) (list ,varzn) (list ,neg?) ,tuples-thru)
     (values ,pred ,varzn ,neg? ,tuples-thru)))

(defmacro augmented-random(n)
  `(if (null (plusp ,n)) 0 (random ,n)))

(defmacro def-example (predicate-name example &rest related-facts)
  `(let ((fact (convert-example-to-fact ',predicate-name ',example)))
     (save-example-template fact ',related-facts)
     (assert-fact fact)
     (dolist (related-fact ',related-facts)
       (assert-fact related-fact))
     fact))

(defmacro def-example-template (predicate-name example &rest related-facts)
  `(let ((template (find-example-template ',predicate-name)))
     (unless template 
       (setq template (make-example-template))
       *example-templates* (push template *example-templates*))
     (setf (example-template-name template) ',predicate-name
           (example-template-vars template) ',example
           (example-template-facts template) ',related-facts)
     (when (fboundp 'update-templates) (update-templates))
     (list 'EXAMPLE-TEMPLATE ',predicate-name)))

(defmacro catch-error-quietly (&body body)
  `(multiple-value-bind (valval ee) 
                        (ignore-errors (let ((catch-error-quietly-value (progn .,body)))
                                         catch-error-quietly-value))
     (cond (ee (values nil ee))
           (t valval))))

(defmacro DEF-FRONTIER-OPERATOR (name &rest body)
  (let ((defun-list (cons 'defun (cons name body))))
    `(progn
       (setf *ALL-FRONTIER-OPERATORS* (pushnew ',name *ALL-FRONTIER-OPERATORS*))
       ,defun-list)))


(defmacro DEF-FAST-FRONTIER-OPERATOR (name &rest body)
  (let ((defun-list (cons 'defun (cons name body))))
    `(progn
       (setf *ALL-FAST-FRONTIER-OPERATORS* (pushnew ',name *ALL-FAST-FRONTIER-OPERATORS*))
       ,defun-list)))


#|
(defmacro clause-deleted (name number)
  `(gethash (list ,name ,number) *deleted-clauses-and-literals*))

(defmacro literal-deleted (name number literals)
  `(gethash (list ,name ,number ,literals) *deleted-clauses-and-literals*))
|#

(defun clear-clause-and-literal-deletions ()
  (clrhash *deleted-clauses-and-literals*))

(defun display-clause-and-literal-deletions ()
  (format t "~%~%Deletion Hash Table ____________________________")
  (maphash #'(lambda (key value) (format t "~%~40A   ~A" key value)) *deleted-clauses-and-literals*)
  (format t "~%________________________________________________"))

(defmacro undelete-clause (rule-name clause-index)
  `(setf (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) nil))

(defmacro delete-clause (rule-name clause-index)
  `(setf (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) t))

(defmacro clause-deleted (rule-name clause-index)
  `(let ((value (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*)))
     (or (eql value t)
         (eql value :deleted))))

(defmacro permanently-undelete-clause (rule-name clause-index)
  `(setf (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) :un-deleted))

(defmacro permanently-delete-clause (rule-name clause-index)
  `(setf (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) :deleted))

(defmacro clause-permanently-deleted (rule-name clause-index)
  `(eql (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) :deleted))

(defmacro clause-deleted-but-not-permanently (rule-name clause-index)
  `(eql (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) t))

(defmacro clause-permanently-added (rule-name clause-index)
  `(eql (gethash (list ,rule-name ,clause-index) *deleted-clauses-and-literals*) :un-deleted))


(defmacro undelete-literal (rule-name clause-index literal-index)
  `(setf (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) nil))

(defmacro delete-literal (rule-name clause-index literal-index)
  `(setf (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) t))

(defmacro permanently-undelete-literal (rule-name clause-index literal-index)
  `(setf (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) :un-deleted))

(defmacro permanently-delete-literal (rule-name clause-index literal-index)
  `(setf (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) :deleted))

(defmacro literal-deleted (rule-name clause-index literal-index)
  `(let ((value (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*)))
     (or (eql value t)
         (eql value :deleted))))

(defmacro literal-permanently-deleted (rule-name clause-index literal-index)
  `(eql (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) :deleted))

(defmacro literal-permanently-added (rule-name clause-index literal-index)
  `(eql (gethash (list ,rule-name ,clause-index ,literal-index) *deleted-clauses-and-literals*) :un-deleted))