;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

(defvar *compile-allowing-deletions* t)  ;;  CAB 8/9/93 these were changed
(defvar *compile-with-rule-trace* nil)

(defvar *last-defun* nil)

(defvar *uncompiled* nil "Prolog symbols that have not been compiled.")
(defvar *predicate* nil "The Prolog predicate currently being compiled")
(defvar *db-predicates* nil "A list of all predicates stored in the database.")

(defvar *rule-trace-string*  (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character))

(defvar *spy-preds* nil "This holds the list of predicates that will start a trace deeply.") 
(defvar *spy-depth* 0 "Holds the starting depth of the trace deeply.  This is the point where deep trace will stop and return to the previous tracing method.")
(defvar *saved-traced-predicates* nil "Holds the old value of *traced-predicates*.  It also is used to tell if we are in a trace-deeply.")

(defparameter *maintain-prolog-rule-trace* nil)

(defparameter *fail-quietly* nil)





(defun get-r-function(name)
  (and (setq name (get-r-struct name))
       (r-prolog-function name)))

(defun call-predicate(name &rest args)
  (let ((f-fun (get-r-function name)))
     (cond (f-fun (cond ((= (length args)
                            (length (arglist f-fun)))
                         (apply f-fun args))
                        (*fail-quietly* nil)
                        (t (format t "~%Warning: The predicate ~A is passed the wrong number of arguments.  Failing" name)
                           nil)))
           (*fail-quietly* nil)
           (t (format t "~%Warning: The predicate ~A is not defined.  Failing" name) nil))))

#|
(defun call-predicate-unless-deleted (name pred clause literal rcont rts &rest args)
  (if (and clause *compile-allowing-deletions*)
    (if (literal-deleted pred clause literal)
      (funcall rcont rts) 
      (apply #'call-predicate name rcont rts args))
    (apply #'call-predicate name rcont rts args)))

(defun call-predicate-unless-deleted (name pred clause literal rcont rts &rest args)
  (if (literal-deleted pred clause literal)
    (funcall rcont rts) 
    (apply #'call-predicate name rcont rts args)))
|#

;;;=====================================================================================================
;;; Modification to call-predicate-unless-deleted that allows regressing problems 

(defconstant succeed "succeed")
(defvar *analyze-calls-to-relation* nil)
(defvar *calls-forced-to-succeed* nil)
(defvar *calls-forced-to-fail* nil)
(defvar *call-context-negated?* nil)
(defvar *most-restrictive-analysis* nil)
(defvar *least-restrictive-analysis* nil)

(defun call-predicate-unless-deleted (name pred clause literal rcont rts &rest args)
  (cond ((literal-deleted pred clause literal)
         (funcall rcont rts))
        ((null *analyze-calls-to-relation*)
         (apply #'call-predicate name rcont rts args))
        ((not (eql name *analyze-calls-to-relation*))
         (if (some-var-bound-to-succeed args)
           (force-succeed name rcont rts args)
           (apply #'call-predicate name rcont rts args)))
        ((member args *calls-forced-to-fail* :test #'arg-equal)
         (force-fail name rcont rts args))
        ((member args *calls-forced-to-succeed* :test #'arg-equal)
         (force-succeed name rcont rts args))
        ((some-var-bound-to-succeed args)
         (force-call-to-succeed name rcont rts args))
        (*most-restrictive-analysis*
         (if *call-context-negated?*
           (force-call-to-succeed name rcont rts args)
           (force-call-to-fail name rcont rts args)))
        (*least-restrictive-analysis*
         (if *call-context-negated?*
           (force-call-to-fail name rcont rts args)
           (force-call-to-succeed name rcont rts args)))
        (t (apply #'call-predicate name rcont rts args))))

(defun add-call-to-call-list (call call-list)
  (if (member call call-list :test #'arg-equal)
    call-list
    (push (return-arg-values call) call-list)))
  
(defun force-call-to-succeed (name rcont rts args)
  (setq *calls-forced-to-succeed* (add-call-to-call-list args *calls-forced-to-succeed*))
  (force-succeed name rcont rts args))

(defun force-succeed (name rcont rts args)
  (declare (ignore name))
  (bind-unbound-args-to-succeed args)
  (funcall rcont rts))

(defun force-call-to-fail (name rcont rts args)
  (setq *calls-forced-to-fail* (add-call-to-call-list args *calls-forced-to-fail*))
  (force-fail name rcont rts args))

(defun force-fail (name rcont rts args)
  (declare (ignore name rcont rts args))
  nil)

(defun bind-unbound-args-to-succeed (args)
  (dolist (arg args)
    (when (unbound-var-p arg)
      (setf (var-binding arg) succeed))))

(defun some-var-bound-to-succeed (args)
  (some #'(lambda (arg) (and (var-p arg) (eq (var-binding arg) succeed))) args))

(defun return-arg-values (args)
  (mapcar #'(lambda (arg) (if (var-p arg) (if (var-binding arg) (var-binding arg) (make-pcvar :id (var-id arg))) arg)) args))

(defun reset-arg-values (args values)
  (mapc #'(lambda (arg value) (when (var-p arg) (setf (var-binding arg) value))) args values))

(defun arg-equal (args1 args2)
  (if (and (consp args1) (consp args2))
    (and (arg-equal (first args1) (first args2))
         (arg-equal (rest args1) (rest args2)))
    (arg-eq args1 args2)))

(defun arg-eq (arg1 arg2)
  (equal (if (var-p arg1) (var-binding arg1) arg1) (if (var-p arg2) (var-binding arg2) arg2)))

;;;=====================================================================================================


(defun compile-unless-macl(x)
 #+:ccl x
 #-:ccl (compile x)
)

;; Clauses are stored on the predicate's plist
(defun predicate (relation) (first relation))

(defmacro <- (&rest clause)
  "Add a clause to the data base."
  `(add-clause ',clause))

(defun clear-db ()
  "Remove all clauses (for all predicates) from the data base."
  (mapc #'clear-predicate *db-predicates*))

(defun clear-predicate (rule-name) 
  "Remove the clauses for a single predicate."
  (set-clauses rule-name nil))

(defconstant unbound "Unbound")
(defvar *var-counter* 0)

(defstruct (var (:constructor ? ())
                (:print-function print-var))
  (id (incf *var-counter*))
  (binding unbound))

(defun bound-p (var) (not (eq (var-binding var) unbound)))

(defmacro deref (exp)
  "Follow pointers for bound variables."
  `(progn (lp while (and (var-p ,exp) (bound-p ,exp))
             do (setf ,exp (var-binding ,exp)))
          ,exp))

(defun unify! (x y)
  "Destructively unify two expressions"
  (cond ((eql (deref x) (deref y)) t)
        ((var-p x) (set-binding! x y))
        ((var-p y) (set-binding! y x))
        ((and (consp x) (consp y))
         (and (unify! (first x) (first y))
              (unify! (rest x) (rest y))))
        (t nil)))


(defparameter *trail* (make-array 100 :fill-pointer 0 :adjustable t))

(defun set-binding! (var value)
  "Set var's binding to value, after saving the variable
  in the trail.  Always returns t."
  (unless (eq var value)
    (vector-push-extend var *trail*)
    (setf (var-binding var) value))
  t)

(defun undo-bindings! (old-trail)
  "Undo all bindings back to a given point in the trail."
  (lp until (= (fill-pointer *trail*) old-trail)
     do (setf (var-binding (vector-pop *trail*)) unbound)))

(defun add-clause (clause)
  "Add a clause to the data base, indexed by head's predicate."
  ;; The predicate must be a non-variable symbol.
  (let ((rule-name (predicate (clause-head clause))))
    (pushnew rule-name *db-predicates*)
    (pushnew rule-name *uncompiled*)
    (set-clauses rule-name (nconc (get-clauses rule-name) (list clause)))
    rule-name))

  "Are the two arguments EQUAL with no unification,
  but with dereferencing?  If so, succeed."

(defun deref-exp (exp)
  "Build something equivalent to EXP with variables dereferenced."
  (if (atom (deref exp))
      exp
      (reuse-cons
        (deref-exp (first exp))
        (deref-exp (rest exp))
        exp)))






(defun variables-in (exp)
  "Return a list of all the variables in EXP."
  (unique-find-if-anywhere #'non-anon-variable-p exp))
;;; replaced ? with ? for anon vars
(defun non-anon-variable-p (x)
  (and (variable-p x) (not (eq x '?))))


(defconstant no-bindings '((T . T)))
(defun binding-val (binding) (rest binding))
(defun get-vbinding (var bindings) (assoc var bindings))
(defun extend-bindings (var val bindings)
  (cons (cons var val)
        (if (and (eq bindings no-bindings) (not (null no-bindings)))
            nil
            bindings)))


;;;  added ? for anon-vars
(defun variable-p (x)
  "Is x a variable (a symbol beginning with `?' or '?')?"
  (and (symbolp x) (eql (elt (symbol-name x) 0) #\?)))



(defun relation-arity (relation)
  "The number of arguments to a relation.
  Example: (relation-arity '(p a b c)) => 3"
  (length (args relation)))

(defun args (x) "The arguments of a relation" (rest x))


(defun deref-equal (x y)
  "Are the two arguments EQUAL with no unification,
  but with dereferencing?"
  (or (eql (deref x) (deref y))
      (and (consp x)
           (consp y)
           (deref-equal (first x) (first y))
           (deref-equal (rest x) (rest y)))))

;;;
(defmacro with-undo-bindings (&body body)
  "Undo bindings after each expression in body except the last."
  (if (length=1 body)
      (first body)
      `(let ((old-trail (fill-pointer *trail*)))
          #+:ccl-2 (declare (ignore-if-unused old-trail))
					
         ,(first body)
         ,@(lp for exp in (rest body)
                 collect '(undo-bindings! old-trail)
                 collect exp))))

;;; added ? as an anon var
(defun deref-copy (exp)
  "Copy the expression, replacing variables with new ones.
  The part without variables can be returned as is."
  (sublis (mapcar (flambda (var) (cons var (?)))
                  (unique-find-if-anywhere #'var-p exp))
          exp))

(defun unbound-var-p (exp)
  "Is EXP an unbound var?"
  (and (var-p exp) (not (bound-p exp))))

;;;



(defun prolog-compiler-macro (name)
  "Fetch the compiler macro for a Prolog predicate."
  ;; Note NAME is the raw name, not the name/arity
  (get name 'prolog-compiler-macro))

(defmacro def-prolog-compiler-macro (name arglist &body body)
  "Define a compiler macro for Prolog."
  `(setf (get ',name 'prolog-compiler-macro)
          #'(lambda ,arglist .,body)))


(defun compile-unify (x y bindings)
  "Return 2 values: code to test if x and y unify,
  and a new binding list."
  (cond
    ;; Unify constants and conses:                       ; Case
    ((not (or (has-variable-p x) (has-variable-p y)))    ; 1,2
     (values (equal x y) bindings))
    ((and (consp x) (consp y))                           ; 3
     (multiple-value-bind (code1 bindings1)
         (compile-unify (first x) (first y) bindings)
       (multiple-value-bind (code2 bindings2)
           (compile-unify (rest x) (rest y) bindings1)
         (values (compile-if code1 code2) bindings2))))
    ;; Here x or y is a variable.  Pick the right one:
    ((variable-p x) (compile-unify-variable x y bindings))
    (t              (compile-unify-variable y x bindings))))

#|
(def-prolog-compiler-macro not (goal body cont bindings rt cn ln)
  (let* ((args (args goal))
         (and-code (compile-body (args goal)
                                 '(flambda(rt)(declare (ignore rt))
                                   (throw 'not-fails t))
                                 bindings rt nil nil)))
    `(if (or (literal-deleted ',*predicate* ',cn ',ln)
             (not (catch 'not-fails ,and-code)))
       ,(if body (compile-body  body cont bindings rt cn (next-ln ln))
            `(funcall ,cont  ;;this comma is very important
                      ,(if *compile-with-rule-trace*
                         `(augment-rule-trace rt (cons 'not (deref-exp ',args))")")
                         'rt))))))
|#

(def-prolog-compiler-macro not (goal body cont bindings rt cn ln)
  (let* ((args (args goal))
         (and-code (compile-body (args goal)
                                 '(flambda(rt)(declare (ignore rt))
                                   (throw 'not-fails t))
                                 bindings rt nil nil)))
    `(if (or (literal-deleted ',*predicate* ',cn ',ln)
             (let (and-code-succeeded?)
               (setq *call-context-negated?* (not *call-context-negated?*)
                     and-code-succeeded? (catch 'not-fails ,and-code)
                     *call-context-negated?* (not *call-context-negated?*))
               (not and-code-succeeded?)))
       ,(if body 
          (compile-body body cont bindings rt cn (next-ln ln))
          `(funcall ,cont
                    ,(if *compile-with-rule-trace*
                       `(augment-rule-trace rt (cons 'not (deref-exp ',args))")")
                       'rt))))))


(defun compile-if (pred then-part)
  "Compile a Lisp IF form. No else-part allowed."
  (case pred
    ((t) then-part)
    ((nil) nil)
    (otherwise `(if ,pred ,then-part))))

;;; anon vars changed to ?

(defun compile-unify-variable (x y bindings)
  "X is a variable, and Y may be."
  (let* ((xb (follow-binding x bindings))
         (x1 (if xb (cdr xb) x))
         (yb (if (variable-p y) (follow-binding y bindings)))
         (y1 (if yb (cdr yb) y)))
    (cond                                                 ; Case:
      ((or (eq x '?) (eq y '?)) (values t bindings))      ; 12
      ((not (and (equal x x1) (equal y y1)))              ; deref
       (compile-unify x1 y1 bindings))
      ((find-anywhere x1 y1) (values nil bindings))       ; 11
      ((consp y1)                                         ; 7,10
       (values `(unify! ,x1 ,(compile-arg y1 bindings))
               (bind-variables-in y1 bindings)))
      ((not (null xb))
       ;; i.e. x is an ?arg variable
       (if (and (variable-p y1) (null yb))
           (values 't (extend-bindings y1 x1 bindings))   ; 4
           (values `(unify! ,x1 ,(compile-arg y1 bindings))
                   (extend-bindings x1 y1 bindings))))    ; 5,6
      ((not (null yb))
       (compile-unify-variable y1 x1 bindings))
      (t (values 't (extend-bindings x1 y1 bindings)))))) ; 8,9

;;; anon vars changed to ?
(defun compile-arg (arg bindings)
  "Generate code for an argument to a goal in the body."
  (cond ((eq arg '?) '(?))
        ((variable-p arg)
         (let ((binding (get-vbinding arg bindings)))
           (if (and (not (null binding))
                    (not (eq arg (binding-val binding))))
             (compile-arg (binding-val binding) bindings)
             arg)))
        ((not (find-if-anywhere #'variable-p arg)) `',arg)
        ((proper-listp arg)
         `(list .,(mapcar (flambda (a) (compile-arg a bindings))
                          arg)))
        (t `(cons ,(compile-arg (first arg) bindings)
                  ,(compile-arg (rest arg) bindings)))))


(defun has-variable-p (x)
  "Is there a variable anywhere in the expression x?"
  (find-if-anywhere #'variable-p x))

(defun proper-listp (x)
  "Is x a proper (non-dotted) list?"
  (or (null x)
      (and (consp x) (proper-listp (rest x)))))

(defun bind-variables-in (exp bindings)
  "Bind all variables in exp to themselves, and add that to
  bindings (except for variables already bound)."
  (lp for var in (variables-in exp) do
        (unless (get-vbinding var bindings)
          (setf bindings (extend-bindings var var bindings))))
  bindings)

(defun follow-binding (var bindings)
  "Get the ultimate binding of var according to bindings."
  (let ((b (get-vbinding var bindings)))
    (if (eq (car b) (cdr b))
        b
        (or (follow-binding (cdr b) bindings)
            b))))

;;; changed anon vars to ?

(defun bind-unbound-vars (parameters exp)
  "If there are any variables in exp (besides the parameters)
  then bind them to new vars."
  (let ((exp-vars (set-difference (variables-in exp)
                                  parameters)))
    (if exp-vars
        `(let ,(mapcar (flambda (var) `(,var (?)))
                       exp-vars)
           ,exp)
        exp)))

(defun self-cons (x) (cons x x))
(defun make-parameters (arity)
  "Return the list (?arg1 ?arg2 ... ?arg-arity)"
  (lp for i from 1 to arity
        collect (intern-symbol '?arg i)))

(defun make-predicate (symbol arity)
  "Return the symbol: symbol/arity"
  (or (get symbol 'prolog-predicate-name)
      (setf (get symbol 'prolog-predicate-name)
	    (intern-symbol symbol '/ arity))))

(defun bind-new-variables (bindings goal)
  "Extend bindings to include any unbound variables in goal."
  (let ((variables (remove-if (flambda (v) (assoc v bindings))
                              (variables-in goal))))
    (nconc (mapcar #'self-cons variables) bindings)))

(defvar *prolog-trace-indent* 1)

(defun print-var (var stream depth)
  (deref var)
  (if (or (and *print-level*
               (>= depth *print-level*))
          (var-p var))
      (format stream "?~a" (var-id var))
      (write var :stream stream)))


(defvar *traced-predicates* nil "A list of the traced predicates")

;;;  accepts one or more predicates to trace (accumulated into predicates) and
;;;  recompiles each of the predicates to add in the necessary code for tracing
;;;  *traced-predicates* is updated to record any new predicates that are traced

(defun p-trace (&rest predicates)
  (dolist (pred predicates)
    (pushnew pred *traced-predicates*)))

;;;  accepts one or more predicates to untrace (accumulated into predicates) and
;;;  recompiles each of them normally to remove the excess trace code.  Note if
;;;  no predicates are supplied then all predicates are untraced (1)

(defun p-untrace (&rest predicates)
  (if (null predicates) (setq  *traced-predicates* nil) ; (1)
      (setq *traced-predicates* (set-difference *traced-predicates* predicates))))

;;;  accepts the name of a predicate (symbol) and optionally the remaining clauses
;;;  associated with the predicate (clauses are processed by arity) and compiles
;;;  the clauses with the same arity of the first clause adding in the necessary 
;;;  code to perform a trace and then recursively compiles the rest of the clauses.
;;;  compile-predicate adds in the code to do the actual tracing







(defun call-and/1 (cont rt goals &aux (goal (car goals))) ; (1)
  "Try to prove a conjunction (i.e., list of goals) by calling them."
  (deref goal)
  (apply (get-r-function (first goal))
         (if (cdr goals)
	     #'(lambda (rt)
		 (call-and/1 cont rt (cdr goals)))
	   cont)
         rt
         (args goal)
         ))

;;;  add in rt to all pre-defined predicates




(defun call/1 ( cont rt goal) ; (1)
  "Try to prove goal by calling it."
  (deref goal)
  (apply (get-r-function (first goal))
         cont
         rt
         (args goal)
         ))





(defun not/1 (cont rt relation) ; (1)
  "Negation by failure: If you can't prove G, then (not G) true."
  ;; Either way, undo the bindings.
  (with-undo-bindings
    (call/1  #'(lambda (rt) (declare (ignore rt)) (return-from not/1 nil)) rt relation)
    (funcall cont (augment-rule-trace rt 
                                     (list 'not relation)))))

(defun bagof/3 (cont rule-trace exp goal result)
  "Find all  solutions to GOAL, and for each solution,
  collect the value of EXP into the list RESULT."
  (let ((answers nil)
        (saved-traces (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character)))
    (call/1 (flambda (rt) 
              (let ((answer (deref-copy (deref-exp exp))))
                (when *maintain-prolog-rule-trace* 
                    (augment-rule-trace rt ")")
                    (with-output-to-string (s saved-traces)
                      (prin1 (fix-rule-trace (read-from-string (subseq *rule-trace-string* rule-trace)))
                             s)
                      ))
                  (push answer answers)
                  ))
            rule-trace  goal)
    (if (and (not (null answers))
             (unify! result answers))
      (funcall cont (augment-rule-trace rule-trace "(" (list 'bagof exp goal result)
                                        saved-traces ")")))))


(defun setof/3 (cont rule-trace exp goal result)
  "Find all unique solutions to GOAL, and for each solution,
  collect the value of EXP into the list RESULT."
  ;; Ex: Assume (p 1) (p 2) (p 3).  Then:
  ;;     (setof ?x (p ?x) ?l) ==> ?l = (1 2 3)
  (let ((answers nil)
        (saved-traces (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character)))
    (call/1 (flambda (rt) 
              (let ((answer (deref-copy (deref-exp exp))))
                (unless (member answer answers :test #'deref-equal)
                  (when *maintain-prolog-rule-trace* 
                    (augment-rule-trace rt ")")
                    (with-output-to-string (s saved-traces)
                      (prin1 (fix-rule-trace (read-from-string (subseq *rule-trace-string* rule-trace)))
                             s)
                      ))
                  (push answer answers)
                  )))
            rule-trace  goal)
    (if (and (not (null answers))
             (unify! result answers))
      (funcall cont (augment-rule-trace rule-trace "(" (list 'setof exp goal result)
                                        saved-traces ")")))))

(defun multi-value-query/1 (cont rt exp)
  (let ((*multi-value-query* t))
    (call/1 (flambda(rt) (declare (ignore rt)) nil) rt exp)
    (funcall cont rt)))

(defun is/2 (cont rt var exp)
  ;; Example: (is ?x (+ 3 (* ?y (+ ?z 4))))
  ;; Or even: (is (?x ?y ?x) (cons (first ?z) ?l))
  (let ((exit-cont #'(lambda (rt) (prolog-trace 'exit 'is var exp) (funcall cont rt))))
    (prolog-trace 'call 'is var exp)
    (if (and (not (find-if-anywhere #'unbound-var-p exp))
             (unify! var (eval (deref-exp exp))))
      (funcall exit-cont
               (augment-rule-trace rt
                                   " ("
                                   (list 'is (deref-exp var) (deref-exp exp))
                                   ")")))
    (prolog-trace 'fail 'is var exp)))

;;;
(defun find-proofs/2 (cont rule-trace goals result)
  "Find all goals that are true"
  (let ((answers nil)
        (saved-traces (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character)))
    (mapc #'(lambda(exp)
              (catch 'found-a-proof
                (call/1 (flambda (rt) 
                                 (let ((answer (deref-copy (deref-exp exp))))
                                   (when *maintain-prolog-rule-trace* 
                                     (augment-rule-trace rt ")")
                                     (with-output-to-string (s saved-traces)
                                       (prin1 (fix-rule-trace (read-from-string (subseq *rule-trace-string* rule-trace)))
                                              s)
                                       ))
                                   (pushnew answer answers)
                                   (throw 'found-a-proof 'found-a-proof)
                                   ))
                rule-trace  exp)))
          goals)
    (if (and (not (null answers))
             (unify! result answers))
      (funcall cont (augment-rule-trace rule-trace "(" (list 'find-proofs  goals result)
                                        saved-traces ")")))))




;;;  check prolog compiler macros and see if they need additions later for now just pass
;;;  the rule trace through

(def-prolog-compiler-macro internal-= (goal body cont bindings rt cn ln) ; (1)
  "Compile a goal which is a call to =."
  (let ((args (args goal)))
    (if (/= (length args) 2)
        :pass ;; decline to handle this goal
        (multiple-value-bind (code1 bindings1)
            (compile-unify (first args) (second args) bindings)
          (compile-if
           code1
           (compile-body body cont bindings1 
                         rt cn ln)))))) ; (2)

(defun make-= (x y) `(internal-= ,x ,y))




;
;;;  accepts a set of compiled expressions (compiled-exps) the name of the predicate being
;;;  compiled (symbol) and the parameters of the predicate (parameters) and creates the
;;;  body of the defun that will be the compilation of the predicate with calls to 
;;;  undo-bindings! and prolog-trace where appropriate.  The additions necessary to 
;;;  implement the trace are:
;;;  (1) created a lambda for the exit-cont which performs a call to prolog-trace before
;;;     funcalling the continuation (cont)
;;;  (2) added a "call" prolog-trace before the first set of unifies
;;;  (3) added a "redo" prolog-trace just after each undo-bindings! (i.e., before each 
;;;     set of unifications starting with the second unification set
;;;  (4) added a "fail" prolog-trace at the end of the body of the let

;;; mp 04/27/91  Used *compile-with-rule-trace to not include trace information
(defun maybe-add-undo-bindings (compiled-exps symbol parameters)
  "Undo any bindings that need undoing.
  If there are any, bind the trail before we start."
  (if (length=1 compiled-exps)
    (if *compile-with-rule-trace*
      `((let ((cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
                       (funcall cont rt)))
              (prolog-trace-indent *prolog-trace-indent*)
              (traced-preds nil)
              )
          (prolog-trace 'call ',symbol ,@parameters)
          (setq traced-preds *traced-predicates*)
          ,@compiled-exps
          (setq *prolog-trace-indent* prolog-trace-indent)
          (setq *traced-predicates* traced-preds)
          (prolog-trace 'redo ',symbol ,@parameters)))
      compiled-exps)
    (if *compile-with-rule-trace*
      `((let ((old-trail (fill-pointer *trail*))
              (cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
                       (funcall cont rt)))
              (prolog-trace-indent *prolog-trace-indent*)
              (traced-preds nil))
          #+:ccl-2 (declare (ignore-if-unused cont old-trail prolog-trace-indent))
          (prolog-trace 'call ',symbol ,@parameters)
          (setq traced-preds *traced-predicates*)
          ,(first compiled-exps)
          ,@(lp for exp in (rest compiled-exps)
                  collect '(undo-bindings! old-trail)
                  collect '(setq *prolog-trace-indent* prolog-trace-indent)
                  collect '(setq *traced-predicates* traced-preds)
                  collect `(prolog-trace 'redo ',symbol ,@parameters) ; (3)
                  collect exp)))
      `((let ((old-trail (fill-pointer *trail*)))
           #+:ccl-2 (declare (ignore-if-unused  old-trail))
          ,(first compiled-exps)
          ,@(lp for exp in (rest compiled-exps)
                  collect '(undo-bindings! old-trail)
                  collect exp))))))


(defun update-rule-trace (clause &optional rt parms)
  (let ((clause-head (if (listp (car clause)) (car clause) clause)))
    (add-to-rule-trace rt 
                          " (" 
                          `(list ',(car clause-head) 
                                 ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                           parms)))))

(defun update-rule-trace-exit (clause rt parms)
  (let ((clause-head (if (listp (car clause)) (car clause) clause)))
    (add-to-rule-trace rt 
                          " (" 
                          `(list ',(car clause-head) 
                                 ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                           parms))
                          ")")))



(defun add-to-rule-trace (rt &rest args)
  (if *compile-with-rule-trace*
    `(augment-rule-trace ,@(if (and (listp rt) 
                                    (eql (car rt) 'augment-rule-trace))
                        (cdr rt)
                        (list rt))
                    ,@args)
    rt))

(defun augment-rule-trace (rt &rest args)
  (when *maintain-prolog-rule-trace*
    (setf (fill-pointer *rule-trace-string*) rt)
    (mapc #'(lambda(string)
              (if (stringp string)
                (dotimes (i (length string))
                  (vector-push-extend (char string i) *rule-trace-string*))
                (with-output-to-string (s *rule-trace-string*)
                  (prin1 string s))))
          args)
    (fill-pointer *rule-trace-string*)))

(defun next-ln (literal-number)
  (if (numberp literal-number)
    (1+ literal-number)))

(defun compile-body (body cont bindings rt &optional clause-number literal-number)
  "Compile the body of a clause."
  (cond
   ((null body)
    `(funcall ,cont ,(add-to-rule-trace 'rt ")")))	; (1) 
   ((eq (first body) '!)		;*** 
    `(progn ,(compile-body (rest body) cont bindings rt clause-number 
                           (next-ln literal-number))
	    (return-from ,*predicate* nil)))
   (t (let* ((goal (first body))
	     (macro (prolog-compiler-macro (predicate goal)))
	     (macro-val 
	      (if macro 
                (funcall macro goal (rest body) cont bindings rt clause-number literal-number
                         ))))
	(if (and macro (not (eq macro-val :pass)))
          macro-val
	  `(call-predicate-unless-deleted ',(predicate goal)
                                  ',*predicate*
                                  ',clause-number
                                  ',literal-number
            
	    ,(if (null (rest body))
               (if *compile-with-rule-trace* 
                 `(flambda (rt)
			   (funcall ,cont ,(add-to-rule-trace 'rt ")")))
                 cont)
	       `(flambda (rt)
		         ,(compile-body 
		           (rest body) cont
		           (bind-new-variables bindings goal)
		           'rt
                           clause-number (next-ln literal-number))))
	    ,rt
	    . ,(mapcar #'(lambda (arg)
                           (compile-arg arg bindings))
                       (args goal))))))))

(defun continue-p (trace)
  "Ask user if we should continue looking for solutions."
  (princ " ")
  (case (read-char)
    (#\? (print-trace trace) (continue-p trace))
    (#\; t)
    (#\. nil)
    (#\newline (continue-p trace))
    (otherwise
      (format t " Type ; to see more or . to stop or ? to trace")
      (continue-p trace))))

(defun print-trace (trace &optional (stream t))
  (augment-rule-trace trace ")")
  (mapc #'(lambda(tr)
            (print-rule-trace (fix-rule-trace tr) 1 1 stream))
        (read-from-string *rule-trace-string*)))

(defun print-rule-trace (trace &optional (indent 1) (level 1) (stream t) &aux tr)
  (cond ((listp (car trace)) (setq tr (instantiate-english (car trace)))
         (when tr (format stream "~&~VT~a. ~{~a ~}" indent level tr))
         (mapc #'(lambda (tr) (print-rule-trace tr (+ indent 3) (+ level 1) stream)) (cdr trace)))
        ((eq (car trace) 'not) 
         (setq tr (instantiate-english (second trace)))
         (when tr (format stream "~&~VT~a. NOT ~a. ~{~a ~}" indent level (+ level 1) tr))
         (mapc #'(lambda (g) (print-rule-trace (list g) (+ indent (if (> level 9) 8 7)) (+ level 1) stream))
               (cddr trace)))
        (t (format stream "~&~VT~a. ~a" indent level trace))))

(defun instantiate-english(literal)
  (let* ((head (car literal))
         (args (cdr literal))
         (s (get-r-struct head))
         (vars (cond (s (r-vars s))
                     ))
         (english (cond 
                   ((pred-p s)
                    (cadr (assoc :fact (r-questions s))))
                   (t
                    (r-questions s))
                   )))
    (mapc #'(lambda(old new)
              (setq english (subst new old english :test #'equalp)))
          vars args)
    english))
        

(defun prolog-trace (kind predicate &rest args)
  
  ; If we are not already in a deep trace, and the current predicate is a member of the deep list, and we are
  ; doing either a "call", or a "redo", then
  ; Save the old list of traced predicates (or save t if the old value was nil)
  ; Set the system to trace all predicates
  ; Save the current indentation level so we know when to stop tracing deeply.
  
  (when (and (not *saved-traced-predicates*)
             (member predicate *spy-preds*) 
             (member kind '(call redo)))
    (if (null *traced-predicates*)
      (setq *saved-traced-predicates* t)
      (setq *saved-traced-predicates* *traced-predicates*))
    (setq *traced-predicates* :all)
    (setq *spy-depth* *prolog-trace-indent*))
  
  
  (when (or (eq *traced-predicates* :all) 
            (member predicate *traced-predicates*))
    (when (member kind '(call redo))
      (incf *prolog-trace-indent* 3))
    
    (format t "~&~VT~a. ~a ~a:~{ ~a~}" *prolog-trace-indent* 
            (/ (1- *prolog-trace-indent*) 3) kind predicate args)
    
    (when (member kind '(fail exit))
      (progn
        (decf *prolog-trace-indent* 3)
        
        ; Now that we have backed out of a predicate, if we are at the same indentation level
        ; we where at when we entered the trace deeply then we are done tracing deeply, and
        ; can reset, and return to the previous tracing method.
        
        (if (eq *spy-depth* *prolog-trace-indent*)
          (reset-trace-deeply)))))
  nil)


          ; This predicate turns off a RUNNING trace deeply.  It resets the
          ; global variable *traced-predicates* to its original value.
          ; This should be called after a top-level-prove, just in case
          ; the user aborted the "prove" with a command-.
 
(defun reset-trace-deeply ()
  (when *saved-traced-predicates*
    (if (eq *saved-traced-predicates* t) 
      (setq *traced-predicates* nil)
      (setq *traced-predicates* *saved-traced-predicates*))
    (setq *saved-traced-predicates* nil)))
 

;;; search backwards for bindings

(defun get-vbinding-from-trail (var &optional (fill-pointer (fill-pointer *trail*)) 
                                   &aux cv)
  (cond ((zerop fill-pointer) nil)
        ((and (var-p (setq cv (aref *trail* (1- fill-pointer)))) (eql (var-id cv) var))
         (var-binding cv))
        (t (get-vbinding-from-trail var (1- fill-pointer)))))

(defun fix-rule-trace (trace)
  
  (cond ((and (symbolp trace) (eql (position #\? (symbol-name trace)) 0))
         (or (get-vbinding-from-trail (read-from-string (subseq (symbol-name trace) 1)))
             trace))
        ((not (consp trace))
         trace)
        ((proper-listp trace)
         (mapcar #'fix-rule-trace trace))
        (t (reuse-cons (fix-rule-trace (car trace)) (fix-rule-trace (cdr trace)) trace))))


;;; supplied code
(DEFUN =/2 ( CONT rt ?ARG1 ?ARG2)
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '= ?ARG1 ?ARG2)
    (when (UNIFY! ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '= (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '= ?ARG1 ?ARG2)))

;;; supplied code

(DEFUN ==/2 ( CONT rt ?ARG1 ?ARG2)
  "Are the two arguments EQUAL with no unification,
  but with dereferencing?  If so, succeed."
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '== ?ARG1 ?ARG2)
    (when (DEREF-EQUAL ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '== (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '== ?ARG1 ?ARG2)))



(DEFUN >/2 ( CONT rt ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '> ?ARG1 ?ARG2) (FUNCALL CONT RT))))
        (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '> ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (> ?arg1 ?arg2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '> ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '> ?ARG1 ?ARG2)))


(DEFUN math-=/2 ( CONT rt ?ARG1 ?ARG2)
  "Is arg1 = arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'math-= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '> ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (=  ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST 'math-= ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'math-= ?ARG1 ?ARG2)))


(DEFUN fail/0 (CONT rt)
  cont rt
  (PROLOG-TRACE 'CALL 'FAIL)
  (PROLOG-TRACE 'FAIL 'FAIL))


(DEFUN </2 (CONT rt ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '< ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '< ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (< ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '< ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '< ?ARG1 ?ARG2)))

(DEFUN <=/2 ( CONT rt ?ARG1 ?ARG2)
  (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '<= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '<= ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (<= ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '<=  ?ARG1 ?ARG2)
                              ")"))
)
    (PROLOG-TRACE 'FAIL '<= ?ARG1 ?ARG2)))

(DEFUN >=/2 ( CONT rt ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '>= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '>= ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (>=  ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST '>= ?ARG1 ?ARG2)
                              ")"))
)
    (PROLOG-TRACE 'FAIL '>= ?ARG1 ?ARG2)))


;; 5/13/91 -ges  added these builtin functions to support non-numeric builtin functionality

;;;  move this to the cruel-compiler.lisp

;;; supplied code
(DEFUN eql/2 ( CONT rt ?ARG1 ?ARG2)
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'eql ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL 'eql ?ARG1 ?ARG2)
    (when (UNIFY! ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST 'EQL (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'eql ?ARG1 ?ARG2)))


;;; a non-numeric relation builtin

(DEFUN string-lessp/2 (CONT rt ?ARG1 ?ARG2)
  "Is (string-lessp arg1 arg2)"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'string-lessp ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL 'string-lessp ?ARG1 ?ARG2)
    (when (and (stringp ?arg1)(stringp ?arg2)
               (string-lessp ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE rt
                              " ("
                              (LIST 'string-lessp ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'string-lessp ?ARG1 ?ARG2)))



;;; got to figure out a way to get rid of this eval if possible

(defun compile-and-prove (goals)
  (let* ((parameter-bindings (collect-parameter-bindings goals nil))
         (i (sublis parameter-bindings goals :test #'var-eq))
         (*predicate* 'USER-GOAL) ;; needs a name for cutting
         
         (*prolog-trace-indent* 1)
         (cont
          #'(lambda (rt)
              (throw 'reg-prove 
                  (values (deref-exp i)
                                 (augment-rule-trace rt ")")))))
         
         (c (focl-compile-clause-function (cons nil i) 0)))  ;needs a fake head- who doesn't
	  (setf (fill-pointer *trail*) 0)
           (setf *var-counter* 0)
	  (catch 'reg-prove
            (funcall c cont (init-trace) ))))

;;;  accepts a list of goals with vars either in focl format or in norvig format (i.e., 
;;;  ?a and ?a are both acceptable

(defun prove (goals &optional (pred nil))
  (cond((cdr goals)
        (let* ((parameter-bindings (collect-parameter-bindings goals nil))
               (i (sublis parameter-bindings goals :test #'var-eq))
               (*prolog-trace-indent* 1)
               (cont
                #'(lambda (rt)
                    (throw 'reg-prove 
                           (if pred t
                               (values (deref-exp i)
                                       (augment-rule-trace rt ")")))))))
	  (setf (fill-pointer *trail*) 0)
	  (setf *var-counter* 0)
          (catch 'reg-prove
            (eval (build-args-and-cont cont goals (init-trace) parameter-bindings)))))
       (t (prove-goal (car goals) pred))))

(defun build-args-and-cont (cont goals rt parameter-bindings)
  (let* ((goal (car goals))
         )
    (if (null (cdr goals))
      `(call-predicate ',(car goal)
        ,cont
        ,rt
        . ,(sublis parameter-bindings (mapcar #'build-arg (cdr goal)) :test #'var-eq))
      `(call-predicate ',(car goal)
        (function(lambda (rt) 
	           ,(build-args-and-cont cont (cdr goals) 'rt parameter-bindings)))
        ,rt
        . ,(sublis parameter-bindings (mapcar #'build-arg (cdr goal)) :test #'var-eq)))))

(defun prove-goal(goal &optional (pred nil))
  (let* ((parameter-bindings (collect-parameter-bindings goal nil))
         (*prolog-trace-indent* 1)
         (goal-predicate (get-r-function (car goal)))
         (i (sublis parameter-bindings goal :test #'var-eq))
         (cont
          #'(lambda (rt) 
               (throw 'reg-prove 
                      (if pred t
                      (values (deref-exp i)
                              (augment-rule-trace rt ")")))))))
      (setf (fill-pointer *trail*) 0)
      (setf *var-counter* 0)
    (catch 'reg-prove
      (apply goal-predicate cont (init-trace)  (cdr i) ))))



(defun prove-function? (function arg-list negation &optional (vlist (create-parameter-bindings arg-list nil)))
    (setf (fill-pointer *trail*) 0)
    (setf *var-counter* 0)
  (if negation (not (catch 'reg-prove
		       (apply function #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))
			      nil  vlist)))
    (catch 'reg-prove
      (apply function #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))
	     nil  vlist))))

(defun setof-function (function vlist outlist)
  (let ((the-answer nil))
    (apply function #'(lambda (rt) (declare (ignore rt)) (pushnew (deref-exp outlist) the-answer :test #'equal))
	   nil  vlist)
    the-answer))



(defun init-trace()
  (when *maintain-prolog-rule-trace* 
  (setf (fill-pointer *rule-trace-string*) 0)
  (vector-push #\( *rule-trace-string*)
  (fill-pointer *rule-trace-string*)))


(defun insert-derefs (exp)
  (cond ((variable-p exp) (list 'deref exp))
        ((proper-listp exp) (cons 'list (mapcar #'insert-derefs exp)))
        ((consp exp) (reuse-cons (insert-derefs (car exp)) (insert-derefs (cdr exp)) exp))
        (t `',exp)))


(defun build-arg (exp)
  (cond ((variable-p exp) exp)
        ((proper-listp exp) (cons 'list (mapcar #'build-arg exp)))
        ((consp exp) (reuse-cons (build-arg (car exp)) (build-arg (cdr exp)) exp))
        (t `',exp)))


;;; collects bindings of variables in an arg list.  Note variables may appear nested
;;; in the arg list, i.e., it works sort of like unify only on a single list and it 
;;; augments the bindings by creating a new variable and associating it with each 
;;; variable it sees for the first time

(defun collect-parameter-bindings (args bindings)
  (cond ((variable-p args) (if (assoc args bindings) bindings (cons (cons args (?)) bindings)))
        ((consp args)
         (collect-parameter-bindings (cdr args)
                               (collect-parameter-bindings (car args) bindings)))
        (t bindings)))



;;; like collect except arglist is guarenteed to be a tuple
;;;returns new-binding list

(defun create-parameter-bindings (args bindings &aux temp)
  (mapcar #'(lambda(arg)
	      (if (variable-p arg)
		  (if (setq temp (assoc arg bindings))
		      (cdr temp)
		    (prog1  (setq temp (?))(push (cons arg temp) bindings)))
		arg))
	  args))
  
(defun create-parameter-bindings-from-tuple (args tuple &optional (bindings nil) (new-vars nil) &aux temp out)
  (setq out (mapcar #'(lambda (arg)
			(if (variable-p arg)
			    (if (new-var? arg new-vars)
				(if (setq temp (assoc arg bindings))
				    (cdr temp)
				  (prog1 (setq temp (?))(push (cons arg temp) bindings)))
			      (nth (pcvar-id arg) tuple))
			  arg))
		    args))
  (values out bindings))


(defun get-focl-clauses (rule-name) (get-clauses rule-name))

(defun focl-clause-head (clause) (first clause))
(defun focl-clause-body (clause) (rest clause))
(defun focl-clause-parameters (clause) (rest (focl-clause-head clause)))
(defun focl-clause-arity (clause) (length (focl-clause-parameters clause)))

;;; modified to deal with focl clause structures



(defvar *named-prolog-functions* nil)

(defun focl-create-clause-functions (name arity clauses &aux (ctr -1))
  (mapcar #'(lambda(c &aux (clause-name (intern-symbol name "-CLAUSE-" (incf ctr))))
	      (focl-create-prolog-function clause-name arity (list c)))
	  clauses))

(defun focl-create-prolog-function (symbol arity clauses &optional (*predicate* symbol ))
  "Compile all the clauses for a given symbol/arity
  into a single LISP function."
  (let* ((parameters (make-parameters arity))
         (clause-number -1)
         (l `(lambda (cont rt . ,parameters)
               #+:allegro (declare (ignore-if-unused cont rt .,parameters))
               #+:ccl-2 (declare (ignore-if-unused cont rt .,parameters))
               (block ,*predicate*
                 ,@(maybe-add-undo-bindings
                    (mapcar (flambda (clause)
                                     (bind-unbound-vars 
                                      parameters
                                      (focl-compile-clause parameters clause
                                                           'cont (incf clause-number)))) 
                            clauses)
                    symbol parameters)
                 ,@(if *compile-with-rule-trace*
                     `((prolog-trace 'fail ',symbol ,@parameters))) ; (2)
                 ))))
    #+:ccl `(function ,l)
    #-:ccl `(compile nil ',l)
    ))

(defun focl-compile-clause-function (clause arity)
  (let* ((parameters (make-parameters arity))
         (*predicate* 'dummy)
         #+:ccl (*warn-if-redefine-kernel* nil)
         #+:ccl (*warn-if-redefine* nil)
         #+:ccl (*compile-definitions* t)
         #+:ccl (*fasl-save-local-symbols* nil)
         #+:ccl (*save-local-symbols* nil) )
    (compile nil (setq *last-defun*
                       `(lambda (cont rt . ,parameters)
                          #+:allegro (declare (ignore-if-unused .,parameters))
                          #+:ccl-2 (declare (ignore-if-unused .,parameters))
                          (block dummy
                            ,@(maybe-add-undo-bindings
                               (list (bind-unbound-vars parameters
                                                        (focl-compile-clause parameters clause
                                                                             'cont))) 
                               *predicate* parameters))
                          )))))

(defun focl-compile-clauses-function (clauses arity)
  (let* ((parameters (make-parameters arity))
         (*predicate* 'dummy)
               #+:ccl (*warn-if-redefine-kernel* nil)
         #+:ccl (*warn-if-redefine* nil)
         #+:ccl (*compile-definitions* t)
         #+:ccl (*fasl-save-local-symbols* nil)
         #+:ccl (*save-local-symbols* nil) )
    (compile nil (setq *last-defun*
                       `(lambda (cont rt . ,parameters)
                          #+:allegro (declare (ignore-if-unused .,parameters))
                          #+:ccl-2 (declare (ignore-if-unused .,parameters))
                          (block dummy
                            ,@(maybe-add-undo-bindings
                               (mapcar (flambda (clause)
                                                (bind-unbound-vars 
                                                 parameters
                                                 (focl-compile-clause parameters clause 'cont)))
                                       clauses)
                               *predicate* parameters))
                          )))))




(defun retrieve-superset-of-matching-tuples-compiled  (pred vs 
                                                            &aux (position 0) (var nil) key 
                                                            (first-const nil))
  (mapc  #'(lambda(i)
             (if (var-p i)
               (setf var t)
               (unless first-const (setf first-const position)))
             (incf position)
             i)
         vs)
  (setq key 
        (if (null var)
          :all-bound
          (if first-const 
            first-const
            :all-vars)))
  (case key
    (:all-vars (r-pos pred))
    (:all-bound (gethash vs (r-pos-hash pred)))
    (otherwise (gethash (nth key vs) (return-argument-hash-table key pred)))))
 
(defun focl-create-fact-defun (symbol arity)
  (let* ((parameters (make-parameters arity))
         (l `(lambda (cont rt . ,parameters) 
	  #+:allegro (declare (ignore-if-unused .,parameters))
          #+:ccl-2 (declare (ignore-if-unused .,parameters))
	  (let* ((old-trail (fill-pointer *trail*))
		 ,@(if *compile-with-rule-trace* 
		       `((cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
				   (funcall cont rt)))) nil)
		 (parameters (list ,@parameters))
		 (iparm (mapcar #'(lambda(i)(deref i)) parameters))
		 ,@(if *compile-with-rule-trace* '((matched nil)(trace-level *prolog-trace-indent*)
                                                   (traced-preds *traced-predicates*)))
		 (matching-tuples 
		  (retrieve-superset-of-matching-tuples-compiled  
		   (get-r-struct ',symbol) iparm)))
	    ,@(if *compile-with-rule-trace* `((prolog-trace 'call ',symbol ,@parameters)
                                              ))
             
            (mapc 
	     #'(lambda (tuple)
		 (cond ((unify! parameters tuple)
		        (funcall cont 
				 ,(add-to-rule-trace 'rt "(" 
						     `(cons ',symbol tuple)
						     ")"))
                        (undo-bindings! old-trail)
		        ,@(if *compile-with-rule-trace* 
                            `((setq *prolog-trace-indent* trace-level)
                              (setq *traced-predicates* traced-preds)
                              (prolog-trace 'redo ',symbol ,@parameters)
                              (setq matched t))))
		       (t (undo-bindings! old-trail))))
	     matching-tuples)
                                          
	    ,@(if *compile-with-rule-trace* 
		  `(
		    (lp (setq matching-tuples (ask-user-for-values  (cons ',symbol iparm)  matched))
			(unless matching-tuples (return))
                        (cond ((unify! parameters (cdr matching-tuples))
		        (funcall cont 
				       ,(add-to-rule-trace 'rt "(" 
							   `(cons ',symbol (cdr matching-tuples))
							   ")"))
                        (undo-bindings! old-trail)
		        ,@(if *compile-with-rule-trace* 
                            `((setq *prolog-trace-indent* trace-level)
                              (setq *traced-predicates* traced-preds)
                              (prolog-trace 'redo ',symbol ,@parameters)
                              (setq matched t))))
		       (t (undo-bindings! old-trail)))
			)                 
		    (prolog-trace 'fail ',symbol ,@parameters))
		'(nil))
	    ))))
     #+:ccl `(function ,l)
    #-:ccl `(compile nil ',l)
    ))





(defun focl-compile-clause (parms clause cont &optional clause-number)
  "Transform away the head, and compile the resulting body."
  (add-deleted clause-number
               (compile-body
                (nconc
                 (mapcar #'make-= parms (focl-clause-parameters clause))
                 (focl-clause-body clause))
                cont
                (mapcar #'self-cons parms)
                (focl-update-rule-trace clause 'rt parms)
                clause-number
                0)))

(defun add-deleted (clause-number code)
  (if (and clause-number *compile-allowing-deletions*)
    `(unless (clause-deleted ',*predicate* ',clause-number)
       ,code)
    code))
     


(defun focl-update-rule-trace (clause &optional rt parms)
  (add-to-rule-trace rt 
                        " (" 
                        `(list ',(car (focl-clause-head clause))
                               ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                         parms))))

(defun focl-update-rule-trace-exit (clause rt parms)
  (add-to-rule-trace rt 
                        " (" 
                        `(list ',(car (focl-clause-head clause)) 
                               ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                         parms))
                        ")"))


(defvar *multi-value-query* nil "makes selection disjoint")

(defun ask-user-for-values (form from-database)
  #-:ccl (declare (ignore form from-database))
  #-:ccl nil
  #+:ccl (unless (or *batch* (never-ask-again? form))
           (let* ((p (car form))
                  (s (get-r-struct p))
                  (qs (r-questions s))
                  (vars (r-vars s))
                  (types (r-type s))
                  vars-in-form
                  vars-not-in-form 
                  restc
                  q)
             (when qs
               (do ((args (cdr form)(cdr args))
                    (vars vars (cdr vars)))
                   ((null vars))
                 (if (var-p (car args))
                   (push (car vars) vars-in-form)
                   (push (car vars) vars-not-in-form)))
               (setq vars-not-in-form  vars-not-in-form)
               (setq vars-in-form  (nreverse vars-in-form))
               (cond ((null vars-in-form) 
                      (if from-database nil
                          (when (okay-to-ask-y-n-question vars s form qs)
                            (ask-y-n-question s vars form (cadr (assoc :question qs))))))
                     (t (let ((q-alist nil) (rest vars-in-form)(changed t))
                          (do ()
                              ((or (null rest)
                                   (null changed))
                               (setq q-alist (nreverse q-alist)))
                            (setq changed nil)
                            (setq restc rest)
                            (dolist (v restc)
                              (setq q (some #'(lambda(vs.q.m?)
                                                (let ((vs (first vs.q.m?)))
                                                  (if (subsetp vs vars-not-in-form)
                                                    vs.q.m?)))  
                                            (cdr(assoc v qs :test #'equalp))))
                              (cond (q (push (cons v q) q-alist)
                                       (push v vars-not-in-form)
                                       (setq rest (remove v rest))
                                       (setq changed t)))))
                          (when (and q-alist (null rest))
                            (ask-value-question s vars types form q-alist from-database))))))))
           )

#+:ccl(defun ask-y-n-question (s vars form q)
        (declare (ignore s))
        (when q
          (let* ((iq (sublis (mapcar #'cons vars (cdr form)) q :test #'equalp))
                 (message (format nil "~{ ~a~}?" iq))
                 (height (+ (compute-message-height message 360 '("Chicago" 12)) 80)))
            (cond ((y-or-n-dialog message :cancel-text nil :position :centered :size (make-point 400 height))
                   (assert-fact form) form)
                  (t
                   (assert-fact form t)
                   nil)))))

#+:ccl(defun ask-value-question(s vars type form q-alist db)
        (unless (and db (some #'(lambda(v.vs.q.m?)
                                  (eq :single-valued (third (cdr v.vs.q.m?))))
                              q-alist))
          (let* ((v-alist (mapcar #'cons vars (cdr form)))
                 (t-alist (mapcar #'cons vars type))
                 (q t)
                 v m (forms nil))
            (mapc #'(lambda(v.vs.q.m?) 
                      (multiple-value-setq  (v m) (get-value-from-user (car v.vs.q.m?)
                                                                       (cdr (assoc (car v.vs.q.m?) t-alist
                                                                                   ))
                                                                       (sublis v-alist (second (cdr v.vs.q.m?))
                                                                               )
                                                                       q
                                                                       s
                                                                       form
                                                                       (and *multi-value-query*
                                                                            (eq (fourth v.vs.q.m?) :multi-valued)       ;;multivalued question
                                                                            (null (cdr (member v.vs.q.m? q-alist)))     ;;last unbound variable
                                                                            )))  ;;don't want to do this, yet
                      (cond ((null v)
                             (never-ask-again form)
                             (return-from ask-value-question nil)))
                      (when m ;;multi-valued
                        (setq forms (mapcar #'(lambda(binding)
                                                (subst binding (cdr (assoc (car v.vs.q.m?) v-alist)) form))
                                            (cdr v)))
                        (never-ask-again form)
                        (setq v (car v)))
                      (setq q nil)
                      (setq form (subst v (cdr (assoc (car v.vs.q.m?) v-alist)) form))
                      (setq v-alist (subst v (cdr (assoc (car v.vs.q.m?) v-alist)) v-alist)))
                  q-alist)
            (assert-fact form)
            (mapc #'assert-fact forms)
            form)))

#+:ccl(defun get-value-from-user (v type q ask? s form &optional (multiple nil))
        (let ((message (format nil "~@(~a~)~{ ~(~a~)~}?" (first q) (rest q)))
              (title "Get Value From User")
              (position :centered))
          (if (or (eq :number type)
                  (eq :anything type))
            (let ((c (catch-cancel (get-string-from-user message :initial-string ""
                                                         :cancel-text "Nothing" :position position
                                                         :size (make-point 400 100)))))
              (unless (eq c :cancel)
                (multiple-value-bind (value error) (catch-error-quietly (read-from-string c nil nil))
                  (cond (error
                         (format t "~%~a ill-formed value" value)
                         (get-value-from-user v type q ask? s form))
                        ((and (eq :number type) (not (numberp value)))
                         (notify-error "Value was suppose to be a :number reenter.")
                         (get-value-from-user v type q ask? s form))
                        ((or (eq :number type)
                             (eq :anything type))
                         value)
                        (t (pushnew-type-instance type value)
                           (setq *facts-changed* t)
                           value)))))
            (let ((items (sort (copy-list (get-type-instances type)) #'universal<)))
              (if multiple
                (let ((c (catch-cancel (select-item items
                                                    :selection-type :disjoint
                                                    :message message
                                                    :window-title title
                                                    :view-size (make-point 400 150)
                                                    :view-position position))))
                  (unless (eq c :cancel)
                    (values c :multiple)))
                (let ((c (catch-cancel
                           (first (select-item items
                                               :selection-type :single
                                               :message message 
                                               :window-title title
                                               :view-size (make-point 400 150)
                                               :view-position position
                                               :type-in-text "Type In"
                                               :cancel-text "Nothing")))))
                  (unless (eq c :cancel)
                    (pushnew-type-instance type c)
                    (setq *facts-changed* t)
                    c)))))))

(defun convert-to-focl-vars (exp)
  (cond ((and (var-p exp)
              (not (var-p (deref exp)))) ; i.e., bound but not to another var
         (deref exp))
        ((var-p exp)
        (make-pcvar :id (var-id exp)))
        ((not (consp exp)) exp)
        ((proper-listp exp)
         (mapcar #'convert-to-focl-vars exp))
        (t (reuse-cons (convert-to-focl-vars (car exp))
                       (convert-to-focl-vars (cdr exp))
                       exp))))

(defun never-ask-again (form) (assert-fact (convert-to-focl-vars form) t))

(defun never-ask-again? (form &aux (tuple (cdr form)) (negs (r-neg (get-r-struct (car form)))))
  (some #'(lambda(neg) (unify-list tuple neg)) negs))

(defun unify (a b &optional (bindings (list t)))
  ;;; Return a most general binding list which unifies a & b
  (cond ((eql a b) bindings)
	((pcvar-p a) (var-unify a b bindings))
	((pcvar-p b) (var-unify b a bindings))
	((or (atom a)(atom b)) nil)
	((setf bindings (unify (first a)(first b) bindings))
         (unify (rest a) (rest b) bindings))))

(defun substitute-vars (form bindings)
  ;;; Substitute the variables in form for their ultimate values specified
  ;;; in the bindings
  (if (rest bindings)
      (substitute1 form bindings)
      form))

(defun substitute1 (form bindings)
  (cond ((null form) nil)
	((pcvar-p form)
	 (let ((binding (get-binding form bindings)))
	   (if binding
	       (substitute1 (second binding) bindings)
	       form)))
	((atom form) form)
	(t (reuse-cons (substitute1 (first form) bindings)
		 (substitute1 (rest form) bindings)
                 form))))

(defun bound-to-p (var1 var2 bindings)
  ;;; Check if var1 is eventually bound to var2 in the bindings
  (cond ((equalp var1 var2) t)
	((let ((val (second (get-binding var1 bindings))))
	   (and val (pcvar-p val) (bound-to-p val var2 bindings))))))
        
(defun unify-list (la lb &optional (bindings (list t)))
  ;;; Return a most general binding list which unifies a & b
  (when (every #'(lambda(a b)
                   (cond ((eql a b) t)
                         ((pcvar-p a) (var-unify a b bindings))
                         ((pcvar-p b) (var-unify b a bindings))))
               la lb)
    bindings))

(defun var-unify (var b bindings)
  ;;; Unify a variable with a wff, if must bind variable and *occur-check*
  ;;; flag is set then check for occur-check violation
  (if (and (pcvar-p b) (var-eq var b))
      bindings
      (let ((binding (get-binding var bindings)))
	(cond (binding (unify (second binding) b bindings))
	      ((and (pcvar-p b)(bound-to-p b var bindings)) bindings)
	      (t (add-binding var b bindings))))))

(defun var-eq (var1 var2)
  ;;; Return T if the two variables are equal
  (eql var1 var2))

(defun get-binding (var bindings)
  ;;; Get the variable binding for var
  (assoc var (rest bindings) ))

(defun add-binding (var val bindings)
  ;;; Add the binding of var to val to the existing set of bindings
  (setf (rest bindings) (cons (list var val) (rest bindings)))
  bindings)

(defun uniquify-variables (form)
  ;;; Make all the variables in form "unique" variables
  (let ((new-names (rename-list form nil)))
    (if (null new-names)
	form
	(rename-variables form new-names))))

(defun rename-list (form &optional new-names)
  (cond ((pcvar-p form)
	 (let ((id (pcvar-id form)))
	   (if (assoc id new-names)
             new-names
             (cons (list id (if *use-gensyms*
                              (make-unique-pcvar :id id)
                              (make-pcvar :id (copy-symbol id))))
                   new-names))))
	((consp form)
         (rename-list (rest form) (rename-list (first form) new-names)))
	(t new-names)))

(defun make-list-unique-vars (length)
  (let ((result nil))
    (dotimes (i length result)
      (setf result (nconc result (list (make-unique-pcvar :id 'v)))))))

(defun rename-variables (form new-names)
  (cond ((pcvar-p form)
	 (let ((entry (assoc (pcvar-id form) new-names)))
	   (if entry (second entry) form)))
	((atom form) form)
	(t (cons (rename-variables (first form) new-names)
		 (rename-variables (rest form) new-names)))))

#+:ccl (defun okay-to-ask-y-n-question (vars s form qs)
         (cond ((cdr vars)
                (or (every #'(lambda(v) 
                               (null (cdr(assoc v qs :test #'equalp))))
                           vars)
                    
                    (null (every #'(lambda(v) 
                                     (or (null (cdr(assoc v qs :test #'equalp))) ;no question
                                         (and (every #'(lambda(v.vs.q.m?)
                                                         (eq :single-valued (third v.vs.q.m?)))
                                                     (cdr(assoc v qs :test #'equalp)))
                                              (has-some-value s v vars (cdr form)))))
                                 vars))))
               (t (not (has-a-value s (cdr form))
                       ))))

#+:ccl (defun has-some-value (s v vars form)
	 (or (some #'(lambda(neg)(unify-list form neg)) (r-neg s))
	     (let* ((n (position v vars))
		    (vform (append (butlast form (- (length vars) n))
				   (list v)
				   (nthcdr (+ n 2) form))))
	       (some #'(lambda(pos)(unify-list vform pos)) (r-pos s)))))

#+:ccl (defun has-a-value (s form)
	 (or (some #'(lambda(neg)(unify-list form neg)) (r-neg s))
             (some #'(lambda(pos)(unify-list form pos)) (r-pos s))))

(defun quiet-=()
  (when (get-r-struct '=)
    (setf (r-questions (get-r-struct '=)) nil)))


(defun make-prolog-function-for-builtin (name fn nargs &optional (overwrite nil))
  (if  overwrite
    (let ((args (make-parameters nargs)))
      `(function (lambda (CONT rt . ,args)
          (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT ',name . ,args) (FUNCALL CONT RT))))
            ,@(mapcar #'(lambda(arg)
                          `(setq ,arg (deref-exp ,arg)))
                      args)
            (PROLOG-TRACE 'CALL ',name . ,args)
            (when (and ,@(mapcar #'(lambda(arg)
				     `(not(unbound-var-p ,arg)))
			         args)
		       (funcall ,fn . ,args))
              (FUNCALL CONT
                       (AUGMENT-RULE-TRACE rt
                                           " ("
                                           (LIST ',name . ,args)
                                           ")"))
              )
            (PROLOG-TRACE 'FAIL ',name . ,args)))
        ))
    (symbol-function (make-predicate name nargs))))



;;; Added to aid in frontier operationalization (CAB 9/92)

(def-prolog-compiler-macro and (goal body cont bindings rt cn ln)
  (compile-body (append (args goal) body) cont bindings rt cn ln))

(def-prolog-compiler-macro or (goal body cont bindings rt cn ln)
  (let ((disjuncts (args goal)))
    (case (length disjuncts)
      (0 'fail)
      (1 (compile-body (cons (first disjuncts) body) cont bindings rt))
      (t (let ((fn (gensym "F")))
           `(flet ((,fn (rt) ,(compile-body body cont bindings rt cn ln)))
              .,(maybe-add-undo-bindings
                 (lp for g in disjuncts collect
                     (compile-body (list g) `#',fn bindings rt cn ln))
                 nil   ;; These two feilds need someone with a brain to fill them in.
                 nil   ;; ACTUALLY A SPECIAL UNDO IS NEEDED FOR OR THAT DOESN'T DO the RETRY CRAP
                       ;; AND A SPECIAL COMPILE BODY THAT DOESN'T add the CALL crap
                       ;; maybe just set *predicate to nil in the let and let them worry about it
                 )))))))

    


