;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: XGINSENG; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: xginseng.lisp,v 1.8 1993/07/17 01:46:18 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :xginseng)


;;;;  Contains definitions for most top level functions as well as the window
;;;;  and interactor definitions.


(defconstant *totaldisplaywidth* 2500 "scrolling width of entire work window")
(defconstant *totaldisplayheight* 2500 "scrolling height of entire work window")
(defconstant *totalwidth* 700 "Total width of XGinseng window")
(defconstant *totalheight* 700 "Total height of XGinseng window")
(defconstant *dialogue-height* 150 "Height of Dialogue Box")
(defconstant *dialogue-width* 550 "Width of Dialogue Box")
(defconstant *button-width* 225 "Width of Dialogue Box")
(defconstant *button-height* 700 "Height of Button Box")





; ==================================================================================
;
; MAKE_NODE_FROM_INPUT_PANEL
; -------------------------- 
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from INPUT-BUTTON-PANEL when
;                                "Make a Blank Node" is called.  This starts 
;                                the process of starting the 
;                                CREATE_BLANK_NODE_INTER_PANEL interactor. 
;                                This is also the 'intermediary function that
;                                places the informative message into the dialogue
;                                box before starting everthing.
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun make_node_from_input_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'node_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the point where you want the new node to be created 
and press the RIGHT mouse button down.

(LEFT down & CTRL down. )
      "))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate node_help_text)
  (opal:update dialogue-window)
  (s-value create_blank_node_inter_panel :start-event :rightdown))




; ==================================================================================
;
; MAKE_ARC_FROM_INPUT_PANEL
; -------------------------- 
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from INPUT-BUTTON-PANEL when
;                                "Make a Blank Node" is called.  This starts 
;                                the process of starting the 
;                                CREATE_BLANK_ARC_INTER_PANEL interactor. 
;                                This is also the 'intermediary function that
;                                places the informative message into the dialogue
;                                box before starting everthing.
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun make_arc_from_input_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'arc_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the node from which you want the new arc to 
originate, press the RIGHT button down, drag to the destination node, 
and then release the button.  A beep will sound if you \"miss\".

(MIDDLE & CTRL down, drag, release.)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate arc_help_text)
  (opal:update dialogue-window)
  (s-value create_blank_arc_inter_panel :start-event :rightdown))




; ==================================================================================
;
; MAKE_DOUBLE_ARC_FROM_INPUT_PANEL
; --------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun make_double_arc_from_input_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'double_arc_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the node from which you want the new arc to 
originate, press the RIGHT button down, drag to the destination node, and then 
release the button.  A beep will sound if you \"miss\".

(MIDDLE, CTRL, SHIFT down; drag; release.)
"))
(clean_up_dialogue_window)
(disable_other_interactors)
(opal:add-component dialogue-aggregate double_arc_help_text)
(opal:update dialogue-window)
(s-value create_blank_double_arc_inter_panel :start-event :rightdown))




; ==================================================================================
;
; EDIT_FROM_INPUT_PANEL
; ---------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun edit_from_input_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'edit_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the label which you want to edit and press RIGHT down. 
A vertical line will appear indicating where new characters will appear.  
When you hit <RETURN> editing is disabled.

(CTRL & RIGHT down.)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate edit_help_text)
  (opal:update dialogue-window)
  (s-value text-inter_panel :start-event :rightdown))




; ==================================================================================
;
; DELETE_PANEL
; ------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun delete_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'delete_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the object you want deleted and press RIGHT down.

(LEFT, CTRL, & SHIFT).
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate delete_help_text)
  (opal:update dialogue-window)
  (s-value node-or-arc-deletor_panel :start-event :rightdown))




; ==================================================================================
;
; ASSERT_PANEL
; ------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun assert_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'assert_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the blank node you want asserted and press RIGHT down.
A single \"!\" will in the displayed node indicating that \"assert\"
instead of \"build\" will be used when the SNePS command is generated.
Please note that this function affects only those nodes that are not
yet part of a SNePS network.

(F3)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate assert_help_text)
  (opal:update dialogue-window)
  (s-value assert-build-toggle-inter_panel :start-event :rightdown))




; ==================================================================================
;
; COPY-LABEL_PANEL
; ----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun copy-label_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'copy-from_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the arc label FROM which you want to extract text 
and press RIGHT down.  Then click MIDDLE down over the label INTO which 
you want the text copied.

(Copy from label: SHIFT & RIGHT
 Copy into label: SHIFT, RIGHT & CTRL)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate copy-from_help_text)
  (opal:update dialogue-window)
  (s-value get-text-of-relation-inter_panel :start-event :rightdown)
  (s-value put-text-of-relation-inter_panel :start-event :middledown))




; ==================================================================================
;
; BASE_PANEL
; ----------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun base_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'base_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the blank node you want to become a base node and 
press RIGHT down.

(F2)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate base_help_text)
  (opal:update dialogue-window)
  (s-value make-a-base-node-inter_panel :start-event :rightdown))




; ==================================================================================
;
; VARIABLE_PANEL
; --------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun variable_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'variable_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the blank node you want to become a variable node 
and press RIGHT down.

(F4)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate variable_help_text)
  (opal:update dialogue-window)
  (s-value make-a-base-pattern-inter_panel :start-event :rightdown))




; ==================================================================================
;
; UNIVERSAL-QUANTIFIER_PANEL
; --------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

(defun universal-quantifier_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'universal_quantifier_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the crusor over the point where you'd like the case frame to appear
and press RIGHT down.

(F5 TWICE)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate universal_quantifier_help_text)
  (opal:update dialogue-window)
  (s-value universal-quantifier-inter_panel :start-event :rightdown))


; ==================================================================================
;
; NUMERICAL-ENTAILMENT_PANEL
; -------------------------
;
;          Arguments         :   None.
;
;          Returns           :   None
;
; ==================================================================================

(defun numerical-entailment_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'numerical_entailment_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the crusor over the point where you'd like the case frame to appear
and press RIGHT down.

(F6 TWICE)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate numerical_entailment_help_text)
  (opal:update dialogue-window)
  (s-value numerical-entailment-inter_panel :start-event :rightdown))




; ==================================================================================
;
; ANDOR_PANEL
; -----------
;
;          Arguments         :   None.
;
;          Returns           :   None
;
; ==================================================================================

(defun andor_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'andor_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the cursor over the point where you'd like the case frame to appear
and press RIGHT down.

(F7 TWICE)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate andor_help_text)
  (opal:update dialogue-window)
  (s-value andor-inter_panel :start-event :rightdown))




; ==================================================================================
;
; THRESH_PANEL
; ------------
;
;          Arguments         :   None.
;
;          Returns           :   None
;
; ==================================================================================

(defun thresh_panel (inter obj)
  (declare (ignore inter obj))
  (create-instance 'thresh_help_text opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Place the crusor over the point where you'd like the case frame to appear
and press RIGHT down.

(F8 TWICE)
"))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (opal:add-component dialogue-aggregate thresh_help_text)
  (opal:update dialogue-window)
  (s-value thresh-inter_panel :start-event :rightdown))




; ==================================================================================
;
; PAUSE
; -----
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       : Function to temporarily break from main-event-loop and 
;                              return i/o control to calling process. (Note: All 
;                              functions called via buttons take 2 args, but they are 
;                              not used here)
;
;          Author            : Martin J.  Zaidel  5/91.  
;
; ==================================================================================

(defun pause (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (s-value input-button-window :value-obj nil)
  (s-value input-button-window :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  (opal:update button-window)
  (inter:exit-main-event-loop))




; ==================================================================================
;
; MAKE-NODE
; ---------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       : Starts the display node process
;
;          Author            : Martin J.  Zaidel  5/91.  
;
; ==================================================================================

(defun make-node (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (declare (special dialogue-aggregate *node-reply* error-text))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (s-value input-button-window :value-obj nil)
  (s-value input-button-window :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (if (and (boundp 'error-text) error-text)
      (progn 
	(opal:remove-component dialogue-aggregate error-text)
	(opal:remove-component dialogue-aggregate prompt)
	(opal:remove-component dialogue-aggregate reply)
	(opal:destroy-me error-text)))
  (opal:add-components dialogue-aggregate prompt reply)
  (opal:update dialogue-window)
  (inter:beep)
  (setf *node-reply* t)
  (inter:start-interactor reply-inter))





; ==================================================================================
;
; OLD-WINDOW-DUMP
; ---------------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       : unused
;
;          Author            : Martin J.  Zaidel  5/91.  
;
; ==================================================================================

(defun old-window-dump (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (declare (special dialogue-aggregate *node-reply* error-text))
  (if (and (boundp 'error-text) error-text)
      (progn (opal:remove-component dialogue-aggregate error-text)
	     (opal:destroy-me error-text)))
  (clean_up_dialogue_window)
  (opal:add-components dialogue-aggregate file-prompt reply)
  (s-value display-window :h-scroll-bar-p nil)
  (s-value display-window :v-scroll-bar-p nil)
  (s-value input-button-window :value-obj nil)
  (s-value input-button-window :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (s-value main-button-panel :value nil)
  (opal:update display-window)
  (setf *node-reply* nil)
  (inter:start-interactor reply-inter))





; ==================================================================================
;
; WINDOW-DUMP
; -----------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Creates a window dump
;
;          Author            :   John S. Lewocz  5/92.
;
; ==================================================================================

(defun window-dump (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (clean_up_dialogue_window)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (create-instance 'new-file-prompt opal:text
		   (:left 5) (:top 5)
		   (:string "Type the file's name: "))
  (create-instance 'new-reply opal:Cursor-text
		   (:left 152) (:top 5) (:width 500))
  (opal:add-components dialogue-aggregate new-file-prompt new-reply)
  (opal:update dialogue-window)
  (create-instance 'new-reply-inter inter:text-interactor
		   (:obj-to-change new-reply)
		   (:start-where t)
		   (:running-where t)
		   (:window dialogue-window)
		   (:start-event t)
		   (:stop-event #\Return)
		   (:abort-event :rightdown)
		   (:abort-action
		    #'(lambda (interactor obj-over event)
			(break)
			(clean_up_dialogue_window)
			(s-value display-window :h-scroll-bar-p t)
			(s-value display-window :v-scroll-bar-p t)))
		   (:final-function #'dumping-function-2)))


(defun dumping-function-2 (interactor obj-over event x y z)
  (declare (ignore obj-over event x y z))
  (s-value display-window :h-scroll-bar-p nil)
  (s-value display-window :v-scroll-bar-p nil)
  (opal:update display-window)
  #+allegro
  (excl:run-shell-command
   (format nil "xwd -nobdrs | xpr -portrait -device ps -output ~a"
	   (g-value new-reply :string)))
  #+lucid
  (let ((temp-file "/tmp/xginsengtmpdump"))
    (lcl:run-unix-program "xwd" :arguments "-nobdrs" :output temp-file)
    (lcl:run-unix-program "xpr" :arguments 
			  `("-portrait" "-device" "ps" 
			    "-output" ,(g-value new-reply :string)
			    ,temp-file))
    (delete-file temp-file))
  #-(or allegro lucid)
  (error-message "This Lisp cannot do window dumps.")
  (clean_up_dialogue_window)
  (s-value display-window :h-scroll-bar-p t)
  (s-value display-window :v-scroll-bar-p t)
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  (opal:update dialogue-window)
  (opal:destroy interactor))




; ==================================================================================
;
; DO-QUIT
; -------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Quits
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun Do-Quit (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (declare (special error-text))
  (opal:update button-window)
  (if (and (boundp 'help-window) help-window) ;; Destroy the HELP-WINDOW if
      (do-help-quit nil nil))
  (if (and (boundp 'error-text) error-text) ;; Destroy the ERROR-TEXT if
      (opal:destroy-me error-text))         ;; it exists
  (opal:destroy display-window)
  (opal:destroy dialogue-window)
  (opal:destroy button-window)
  (opal:destroy input-button-window)
  ;;@i{if not CMU CommonLisp, then exit the main event loop}
  #-cmu (inter:exit-main-event-loop))




; ==================================================================================
;
; CLEAR-DISPLAY
; -------------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Clears the display.
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun clear-display (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (dolist (node (g-value display-window :nodes-on-display))
	  (destroy-node node))
  (dolist (node-or-arc (g-value display-window :temporary-nodes-and-arcs-on-display))
	  (opal:destroy-me node-or-arc))
  (s-value display-window :nodes-on-display nil)  
  (s-value display-window :temporary-nodes-on-display nil)  
  ;; Reset button
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  ;; Move window back to center of scrolling area
  
  
;  (garnet-gadgets:scroll-win-inc display-window 
;				 (floor (- *totalwidth* *totaldisplaywidth*) 2)
;				 (floor (- *totalheight* *totaldisplayheight*) 2))


  (opal:update display-window))




; ==================================================================================
;
; RESTART
; -------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Restarts
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun restart ()
  ;; Reset button
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  (inter:main-event-loop))




; ==================================================================================
;
; DO-HELP
; -------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Starts the help window up
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun Do-Help (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (s-value input-button-window :value-obj nil)
  (s-value input-button-window :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (if (and (boundp 'help-window) help-window)
      (opal:deiconify-window help-window) 
    (make-help))
  ;; Reset button
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil))




; ==================================================================================
;
; CHECK-DESCRIBE-DOWN
; -------------------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun check-describe-down (a-node)
  (if (null (sneps::down.fcs (g-value a-node :sneps-node)))
      (error-message "The node ~a has no children." (g-value a-node :name))
      (describe-down a-node)))




; ==================================================================================
;
; CHECK-DESCRIBE-UP
; -----------------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun check-describe-up (a-node)
  (if (null (sneps::up.fcs (g-value a-node :sneps-node)))
      (error-message "The node ~a has no ancestors." (g-value a-node :name))
      (describe-up a-node)))




; ==================================================================================
;
; CHECK-ALL-DESCENDANTS
; ---------------------
;
;          Arguments         :   <a Garnet node>
;
;          Returns           :   <its descendants, if any>
;
;          Description       :   
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun check-all-descendants (a-node)
  (if (null (sneps::down.fcs (g-value a-node :sneps-node)))
      (error-message "The node ~a has no descendants." (g-value a-node :name))
      (all-descendants a-node)))




; ==================================================================================
;
; CHECK-ALL-ANCESTORS
; -------------------
;
;          Arguments         :   <a Garnet node>
;
;          Returns           :   <its ancestors, if any>
;
;          Description       :   
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun check-all-ancestors (a-node)
  (if (null (sneps::up.fcs (g-value a-node :sneps-node)))
      (error-message "The node ~a has no ancestors." (g-value a-node :name))
      (all-ancestors a-node)))




; ==================================================================================
;
; DELETE-NODE
; -----------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Unused
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

;(defun delete-node (toolkit-obj menu-item)
;  (declare (ignore menu-item))
;  (let* ((selected-node (g-value display-aggregate :selected))
;	 (parent-arcs (g-value selected-node :up-arcs)))
;    (s-value (g-value toolkit-obj :selection-obj) :value NIL)
;    (if parent-arcs
;	(error-message "That node is dominated."))
;    (s-value display-aggregate :selected NIL))
;  (s-value (g-value obj :selection-obj) :value NIL))




; ==================================================================================
;
; CHECK-HIDE-NODE
; ---------------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Calls HIDE-NODE on then node in question.
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun check-hide-node (a-node)
  (let ((dominated-p (do* ((cables (g-value a-node :up-cables) (rest cables))
			   (arc (get-arc (first cables)) (get-arc (first cables))))
			  ((null cables))
			  (when (g-value arc :visible)
				(return t)))))
    (if dominated-p
	(error-message "Node ~a is dominated: It cannot be hidden." (g-value a-node :name))
      (hide-node a-node))))




; ==================================================================================
;
; HIDE-NODE
; ---------
;
;          Arguments         :   <a Garnet node>
;
;          Returns           :   None
;
;          Description       :   Hides the node.
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun hide-node (node)
"This function erases nodes from the display if the following conditions hold:
1. <node> is not dominated by any other node;
2. the immediate descendants of <node> have descendants."
  (clean_up_dialogue_window)
  (disable_other_interactors)
  (s-value input-button-window :value-obj nil)
  (s-value input-button-window :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil)
  (dolist (down-cable (g-value node :down-cables))
	  (let ((down-node (get-node down-cable))
		(arc (get-arc down-cable)))
	    (s-value arc :visible nil)
	    (if (and (null (g-value down-node :down-cables))
		     (do* ((up-cables (g-value down-node :up-cables) (rest up-cables))
			   (up-arc (get-arc (first up-cables)) (get-arc (first up-cables))))
			  ((null up-cables) t)
		       (when (g-value up-arc :visible)
			 (return nil))))
		(s-value down-node :visible nil))))
  (s-value node :visible NIL)
  (opal:update display-window))

  


; ==================================================================================
;
; DESTROY-NODE
; ------------
;
;          Arguments         :   
;
;          Returns           :   
;
;          Description       :  
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun destroy-node (node)
"This function unbinds the symbol, removes all the down-arcs from the display,
 and destroys the Garnet object <node>."
  ;; Destroy all arcs emanating from the node
  (dolist (cable (g-value node :down-cables))
    (opal:destroy-me (get-arc cable)))
  (opal:destroy-me node))




; ==================================================================================
;
; STRINGIFY.GN
; ------------
;
;          Arguments         :   <a Garnet node>
;
;          Description       :   Macro.  Returns the string (minus assertion symbols) 
;                                of <node>
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defmacro stringify.gn (node)
"Returns the string (minus assertion symbols) of <node>"
`(string-upcase (string-right-trim `(#\!) (g-value ,node :name))))




; ==================================================================================
;
; CHANGE-TO-DISPLAY-WINDOW
; ------------------------
;
;          Arguments         :   
;
;          Returns           :   
;
;          Description       :   
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

(defun change-to-display-window (obj)
  (opal:remove-component (g-value obj :parent) obj)
  (opal:add-component (g-value display-window :inner-aggregate) obj))




; ==================================================================================
;
; DO-THE-DUMP
; -----------
;
;          Arguments         :   None.
;
;          Returns           :   
;
;          Description       :   Unused.
;
;          Author            :   Martin J. Zaidel  5/91.
;
; ==================================================================================

;(defun do-the-dump ()
;  ;; Shell programs for ALLEGRO
;  #+allegro(excl:run-shell-command "xwd"
;				   :arguments "-nobdrs"
;				   :output "/tmp/dump")
;  #+allegro(excl:run-shell-command "xpr"
;				  :arguments 
;				  `("-portrait" "-device" "ps" 
;				    "-output" ,(g-value reply :string)
;				    "/tmp/dump"))
;  #+allegro(delete-file "/tmp/dump")
;  ;; Shell programs for LUCID
;  #+lucid(lcl:run-unix-program "xwd"
;			       :arguments "-nobdrs"
;			       :output "/tmp/dump")
;  #+lucid(lcl:run-unix-program "xpr"
;			       :arguments 
;			       `("-portrait" "-device" "ps" 
;				 "-output" ,(g-value reply :string)
;				 "/tmp/dump"))
;  #+lucid(delete-file "/tmp/dump")
;  #-(or allegro lucid)(error-message
;		       "This Lisp cannot do window dumps.")
;  (s-value display-window :h-scroll-bar-p t)
;  (s-value display-window :v-scroll-bar-p t)
;  (opal:remove-components dialogue-aggregate file-prompt reply))




; ==================================================================================
;
; XGINSENG1
; ---------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Main function that sets up the XGinseng display
;                                window, command windows, and dialogue box.
;                                Activates the top window, transferring 
;                                I/O from the process where application was called.
;                                
;           
;          Dependencies      :   None
;
;          Author            :   Martin J Zaidel, 5/91.  Modified by John S. Lewocz  
;                                5/92.
;
; ==================================================================================

(defun xginseng1 ()
  (declare (special display-aggregate))
  (format t "~%~%XGinseng builds four X windows in the following order:

         1. the main, scrollable XGinseng display window;
         2. a window of buttons for building a network;
         2. a window of command buttons; and
         3. a dialogue window for messages and receiving input.

Each of these windows can be placed where desired by clicking when the
`ghost' appears, iconified, resized, and moved like any X window.  A
`Help' window, which contains information on how to use XGinseng, can
be created by clicking left on the `Help' button in the command button
window.~%~%")
  
  ;>>>> Don't want warnings to appear in Lisp window
  (setf kr::*warning-on-create-schema* nil)
  (setf kr::*print-new-instances* nil)




; ==================================================================================
;
; DISPLAY-WINDOW
; --------------
;
;          Arguments         :   N/A
;
;          Returns           :   N/A
;
;          Description       :   Garnet object.  Work window in which nodes and
;                                arcs are created and displayed.
;           
;          Dependencies      :   None
;
;          Author            :   Martin J Zaidel, 5/91.  Modified by John S. Lewocz  
;                                5/92.
;
; ==================================================================================

  (create-instance 'display-window garnet-gadgets:scrolling-window-with-bars
		   (:nodes-on-display NIL) ; List of garnet nodes in current window
		   (:temporary-nodes-and-arcs-on-display NIL) ;my-change - holds nodes
					;not yet known to sneps
		   (:relations-to-define nil) ;;my-change
		   (:parent-window nil)
		   (:position-by-hand T)
		   (:height *totalheight*) 
		   (:width *totalwidth*)
		   (:title "XGinseng Display")
		   (:X-offset 0)
		   (:Y-offset 0)
		   (:total-width *totaldisplaywidth*)
		   (:total-height *totaldisplayheight*)
		   (:visible T)
		   (:h-scroll-bar-p t)
		   (:v-scroll-bar-p t)
		   (:h-scroll-on-top-p NIL)
		   (:v-scroll-on-left-p T)
		   (:v-page-inc (o-formula (- (gvl :height) 10)))
		   (:h-page-inc (o-formula (- (gvl :width) 10)))
		   (:h-scr-incr 100)
		   (:v-scr-incr 100))
  (opal:update display-window);;REQUIRED for scrolling window gadget
  (setq display-aggregate (g-value display-window :inner-aggregate))





; ==================================================================================
;
; DIALOGUE-WINDOW
; ---------------
;
;          Arguments         :   N/A
;
;          Returns           :   N/A
;
;          Description       :   Garnet object.  Work window in which messages are
;                                printed to the user and input is received.
;           
;          Dependencies      :   None
;
;          Author            :   Martin J Zaidel, 5/91. 
;
; ==================================================================================  

  (create-instance 'dialogue-window inter:interactor-window
		   (:parent nil)
		   (:position-by-hand t)
		   (:title "XGinseng Dialogue Box")
		   (:height *dialogue-height*)
		   (:width *dialogue-width*)
		   (:aggregate
		    (create-instance 'dialogue-aggregate opal:aggregate)))
  




; ==================================================================================
;
; BUTTON-WINDOW
; ---------------
;
;          Arguments         :   N/A
;
;          Returns           :   N/A
;
;          Description       :   Garnet object.  Window used for main button
;                                panel.
;           
;          Dependencies      :   None
;
;          Author            :   Martin J Zaidel, 5/91. 
;
; ==================================================================================

  (create-instance 'button-window inter:interactor-window
		   (:parent nil)
		   (:position-by-hand t)
		   (:title "XGinseng Commands")
		   (:width *button-width*)
		   (:height *button-height*)
		   (:aggregate 
		    (create-instance 'button-aggregate opal:aggregate)))
  



; ==================================================================================
;
; CLEAN_UP_DIALOGUE_WINDOW
; ------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.  Removes any text objects from Dialogue
;                                Box left by previous functions.  DESTROYS the object
;                                unless it is one of the following Garnet objects:
;                                REPLY, CLICK-PROMPT, FILE-PROMPT, PROMPT, or ERROR
;                                TEXT.  These are only created once.
;
;          Dependencies      :   Used by functions called from the INPUT-BUTTON-PANEL
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (defun clean_up_dialogue_window ()
    (let ((objects (get-values dialogue-aggregate :components)))
      (dolist (object objects)
	      (opal:remove-component dialogue-aggregate object)
	      (if (or (and (boundp 'reply)
			   reply
			   (equal object reply))
		      (and (boundp 'click-prompt)
			   click-prompt
			   (equal object click-prompt))
		      (and (boundp 'file-prompt)
			   click-prompt
			   (equal object file-prompt))
		      (and (boundp 'prompt)
			   prompt
			   (equal object prompt))
		      (and (boundp 'error-text)
			   error-text
			   (equal object error-text)))
		  (values)
		(opal:destroy object))))
    (opal:update dialogue-window))
  



; ==================================================================================
;
; DISABLE_OTHER_INTERACTORS
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Ensures that only one selection from 
;                                INPUT-BUTTON-WINDOW may be active at one time.
;                                This is accomplished by setting the :start-event
;                                slot of the other functions' interactors to nil.
;                                
;
;          Dependencies      :   Used by functions called from the INPUT-BUTTON-PANEL
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (defun disable_other_interactors ()
    (clean_up_dialogue_window)
    (s-value put-text-of-relation-inter_panel :start-event nil)
    (s-value create_blank_node_inter_panel :start-event nil)
    (s-value create_blank_arc_inter_panel :start-event nil)
    (s-value create_blank_double_arc_inter_panel :start-event nil)
    (s-value text-inter_panel :start-event nil)
    (s-value node-or-arc-deletor_panel :start-event nil)
    (s-value assert-build-toggle-inter_panel :start-event nil)
    (s-value assert-build-toggle-inter_panel :start-event nil)
    (s-value make-a-base-node-inter_panel :start-event nil)
    (s-value make-a-base-pattern-inter_panel :start-event nil)
    (s-value universal-quantifier-inter_panel :start-event nil)
    (s-value numerical-entailment-inter_panel :start-event nil)
    (s-value thresh-inter_panel :start-event nil)
    (s-value andor-inter_panel :start-event nil)
    (s-value get-text-of-relation-inter_panel :start-event nil))
  



; ==================================================================================
;
; ASSERT_A_NODE_START
; -------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "Assert a Node" is called.  This starts the process
;                                of starting the NODE-SELECTOR_ASSERT-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (defun assert_a_node_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-asserting-a-node opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node you want asserted and press
MIDDLE down over it.  This function acts on nodes
that are already existing SNePS nodes."))  
    (opal:add-component dialogue-aggregate user-info-on-asserting-a-node)
    (opal:update dialogue-window)
    (s-value node-selector_assert-node :start-event 
	     (list :middledown :rightdown)))





; ==================================================================================
;
; ERASE-NODE_START
; ----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "Erase a Node" is called.  This starts the process
;                                of starting the NODE-SELECTOR_ERASE-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (defun erase-node_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-erasing-a-node opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node you want erased and press
MIDDLE down over it.  This function acts on nodes
that are already existing SNePS nodes.  If the node
is not eligible for erasing the function will return
with no effect."))
    (opal:add-component dialogue-aggregate user-info-on-erasing-a-node)
  (opal:update dialogue-window)
  (s-value node-selector_erase-node :start-event 
	   (list :middledown :rightdown)))




; ==================================================================================
;
; CHECK-DESCRIBE-DOWN_START
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Descendants" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-DOWN-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (defun check-describe-down_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-describing-down-a-node opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node whose first generation
descendants you want displayed and press
MIDDLE down over it."))
    (s-value node-selector_describe-down-node :start-event 
	     (list :middledown :rightdown)))




; ==================================================================================
;
; CHECK-DESCRIBE-UP_START
; -----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (defun check-describe-up_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-describing-up-a-node opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node whose first generation
ancestors you want displayed and press
MIDDLE down over it."))
    (s-value node-selector_describe-up-node :start-event 
	     (list :middledown :rightdown)))



; ==================================================================================
;
; CHECK-ALL-DESCENDANTS_START
; -----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (defun check-all-descendants_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-describing-all-descendants opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node whose descendants
you want displayed and press MIDDLE down over it."))
    (s-value node-selector_all-descendants-node :start-event 
	     (list :middledown :rightdown)))
  



; ==================================================================================
;
; CHECK-ALL-ANCESTORS_START
; -----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (defun check-all-ancestors_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (s-value main-button-panel :value-obj nil)
    (s-value main-button-panel :value nil)
    (create-instance 'user-info-on-describing-all-ancestors opal:multi-text
		     (:left 10)
		     (:top 10)
		     (:string "Select the node whose ancestors
you want displayed and press MIDDLE down over it."))
    (s-value node-selector_all-ancestors-node :start-event 
	     (list :middledown :rightdown)))
  



; ==================================================================================
;
; CHECK-HIDE-NODE_START
; ---------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (defun check-hide-node_start (inter obj)
    (declare (ignore inter obj))
    (clean_up_dialogue_window)
    (disable_other_interactors)
    (s-value input-button-window :value-obj nil)
    (s-value input-button-window :value nil)
    (s-value node-selector_hide-node :start-event 
	     (list :middledown :rightdown)))
  
  


; ==================================================================================
;
; AUX-BUTTON-PANEL
; ----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Garnet object.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'aux-button-panel Garnet-gadgets:Text-Button-Panel
		   (:items '(""))
		   (:direction :vertical)
		   (:left 10)
		   (:top 500)
		   (:font Opal:Default-font)
		   (:shadow-offset 5)
		   (:final-feedback-p t))




; ==================================================================================
;
; MAIN-BUTTON-PANEL
; -----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Garnet object.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'main-button-panel Garnet-gadgets:Text-Button-Panel
		   (:items '(("XGinseng Pause" pause)
			     ("XGinseng Quit" Do-Quit)
			     ("XGinseng Help" Do-help)
			     ("Clear Display" clear-display)
			     ("Save As Postscript File" window-dump)
			     ("Display A Node" make-node)
			     ("Hide A Node" check-hide-node_start)))
		   (:direction :vertical)
		   (:left 10) (:top 5)
		   (:font Opal:Default-font)
		   (:shadow-offset 5)
		   (:fixed-width-size 200)
		   (:final-feedback-p t))




; ==================================================================================
;
; MAIN-BUTTON-PANEL-2
; ------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Garnet object.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'main-button-panel-2 Garnet-gadgets:Text-Button-Panel
		   (:items '(("1st Generation Descendants" check-describe-down_start)
			     ("All Descendants" check-all-descendants_start)
			     ("1st Generation Ancestors" check-describe-up_start)
			     ("All Ancestors" check-all-ancestors_start)
			     ("Enter the Network" enter_commands)
			     ("Assert A Node" assert_a_node_start)
			     ("Erase A Node" erase-node_start)))
		   (:direction :vertical)
		   (:left (g-value main-button-panel :left))
		   (:top (+ (g-value main-button-panel :top)
			    (g-value main-button-panel :height)
			    50))
		   (:font Opal:Default-font)
		   (:shadow-offset 5)
		   (:final-feedback-p t))

  ;>>>> Put panels into the button window
  (opal:add-components button-aggregate main-button-panel main-button-panel-2)
  



; ==================================================================================
;
; INPUT-BUTTON-WINDOW
; -------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Garnet object.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'input-button-window inter:interactor-window
		   (:parent nil)
		   (:position-by-hand t)
		   (:title "XGinseng Input Commands")
		   (:width *button-width*)
		   (:height *button-height*)
		   (:aggregate 
		    (create-instance 'input-button-aggregate opal:aggregate)))

  


; ==================================================================================
;
; INPUT-BUTTON-PANEL
; ------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'input-button-panel garnet-gadgets:text-button-panel
		   (:items '(("Make a Blank Node" make_node_from_input_panel)
			     ("Draw an Arc" make_arc_from_input_panel)
			     ("Draw an Double Arc" make_double_arc_from_input_panel)
			     ("Edit or Enter a Label" edit_from_input_panel)
			     ("Delete a Node or Arc" delete_panel)
			     ("Mark Node as Asserted" assert_panel)
			     ("Mark Node as a Base Node" base_panel)
			     ("Mark Node as Variable" variable_panel)
			     ("Copy a Label" copy-label_panel)
			     ("Universal Quantifier" universal-quantifier_panel)
			     ("Numerical Entailment" numerical-entailment_panel)
			     ("AndOr" andor_panel)
			     ("Thresh" thresh_panel)))
		   (:direction :vertical)
		   (:fixed-width-size (g-value main-button-panel :fixed-width-size))
		   (:left 10)
		   (:top 5)
		   (:shadow-offset 5)
		   (:final-feedback-p t))
  ;>>>> Put the panel into the window
  (opal:add-components input-button-aggregate input-button-panel)
  
  


; ==================================================================================
;
; MOVER
; -----
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'mover inter:move-grow-interactor
		   (:window (g-value display-window :inner-window))
		   (:running-where t)
		   (:start-where 
		    (LIST :list-element-of display-window 
			  :nodes-on-display :type node)))




; ==================================================================================
;
; NODE-PLACER
; -----------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
;; Create an interactor to PLACE new nodes in the Display window

  (create-instance 'node-placer inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:continuous NIL)
		   (:start-where t)
		   (:start-event :shift-leftdown)
		   (:abort-if-too-small nil)
		   (:final-function
		    #'(lambda (inter points-list)
			(declare (ignore points-list))
			(unless (boundp (g-value inter :node-symbol))
				(new-node (g-value inter :node-symbol)
					  (g-value inter :sneps-node)
					  points-list))
			(opal:remove-component dialogue-aggregate click-prompt)
			(opal:update dialogue-window)
			(opal:update display-window))))
  
; ==================================================================================
;
; MAKE_BASE
; ---------
;
;          Arguments         :   <button-interactor>, <garnet-node>
;
;          Returns           :   None
;
;          Description       :   This function set/unsets the :base-node-p flag
;                                in <garnet-node> each time the :F2 key is pressed
;                                on the keyboard and the cursor is over the node.
;                                The function then sets the :sneps-node slot of 
;                                <garnet-node> to the sneps-node returned by the
;                                execution of the sneps |*| macro.
;                                The name of the <garnet-node> is then set to 
;                                the name of the sneps node
;           
;          Dependencies      :   Called by the button-interactor
;                                make-a-base-node-inter (xginseng.lisp)
;
; ==================================================================================

  (defun make_base (inter a-node)
    (declare (ignore inter))
    (if (null (g-value a-node :base-node-p))
	(progn
	  (s-value a-node :base-node-p t)
	  (s-value a-node :sneps-node (sneps:node (first (sneps:|#| 'x))))
	  (s-value a-node :name (format nil "~A" (g-value a-node :sneps-node)))
	  (s-value a-node :name-symbol (intern (g-value a-node :name) 'xginseng)))))
  
  


; ==================================================================================
;
; MAKE_PATTERN
; ---------
;
;          Arguments         :   <button-interactor>, <garnet-node>
;
;          Returns           :   None
;
;          Description       :   This function set/unsets the :pattern-node-p flag
;                                in <garnet-node> each time the :F2 key is pressed
;                                on the keyboard and the cursor is over the node.
;                                The function then sets the :sneps-node slot of 
;                                <garnet-node> to the sneps-node returned by the
;                                execution of the sneps |*| macro.
;                                The name of the <garnet-node> is then set to 
;                                the name of the sneps node
;           
;          Dependencies      :   Called by the button-interactor
;                                make-a-base-node-inter (xginseng.lisp)
;
; ==================================================================================

  (defun make_pattern (inter a-node)
    (declare (ignore inter a-node))
    (if (null (g-value a-node :base-node-p))
	(progn
	  (s-value a-node :pattern-node-p t)
	  (s-value a-node :base-node-p t)
	  (s-value a-node :sneps-node (sneps:node (first (sneps:|$| 'x))))
	  (s-value a-node :name (format nil "~A" (g-value a-node :sneps-node)))
	  (s-value a-node :name-symbol (intern (g-value a-node :name) 'xginseng)))))
  



; ==================================================================================
;
; NODE-SELECTOR_ASSERT-NODE
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'node-selector_assert-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (assert_a_node final-obj))
			(s-value inter :start-event nil)
			(clean_up_dialogue_window)
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))




; ==================================================================================
;
; NODE-SELECTOR_ERASE-NODE
; ------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_erase-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (erase-node final-obj))
			(s-value inter :start-event nil)
			(clean_up_dialogue_window)		      
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))
  



; ==================================================================================
;
; NODE-SELECTOR_DESCRIBE-DOWN-NODE
; --------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_describe-down-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (check-describe-down final-obj))
			(s-value inter :start-event nil)
			(clean_up_dialogue_window)
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))
  



; ==================================================================================
;
; NODE-SELECTOR_DESCRIBE-UP-NODE
; ------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_describe-up-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (check-describe-up final-obj))
			(s-value inter :start-event nil)
			(clean_up_dialogue_window)
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))




; ==================================================================================
;
; NODE-SELECTOR_ALL-DESCENDANTS-NODE
; ----------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_all-descendants-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (check-all-descendants final-obj))
			(s-value inter :start-event nil)
			(clean_up_dialogue_window)
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))
  
  


; ==================================================================================
;
; NODE-SELECTOR_ALL-ANCESTORS-NODE
; --------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_all-ancestors-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (check-all-ancestors final-obj))
			(s-value inter :start-event nil)
		      (clean_up_dialogue_window)
		      (s-value main-button-panel-2 :value-obj nil)
		      (s-value main-button-panel-2 :value nil)
		      (s-value main-button-panel :value-obj nil)
		      (s-value main-button-panel :value nil))))
  



; ==================================================================================
;
; NODE-SELECTOR_HIDE-NODE
; -----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector_hide-node inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(if (equal (g-value inter :start-char)
				   :middledown)
			    (hide-node final-obj))
			(s-value inter :start-event nil)
			(s-value main-button-panel-2 :value-obj nil)
			(s-value main-button-panel-2 :value nil)
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil))))




; ==================================================================================
;
; NODE-SELECTOR
; -------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'node-selector inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :element-of display-aggregate))
		   (:start-event '(:any-middledown :any-rightdown))
		   (:abort-event :rightdown)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'(lambda (inter final-obj)
			(declare (ignore final-obj))
					;retrieve the function selected in the aux-button-panel
					;and call it on FINAL-OBJ (as long as it isn't :NONE)
			(let ((function (g-value aux-button-panel :value))
			      (a-node (g-value display-aggregate :selected)))
			  (when a-node
				(cond ((equal :rightdown ; Abort on right
					      (g-value inter :start-char))) ; keypress
				      ((string= function "1st Generation Descendants")
				       (check-describe-down a-node))
				      ((string= function "1st Generation Ancestors")
				       (check-describe-up a-node))
				      ((string= function "All Descendants")
				       (check-all-descendants a-node))
				      ((string= function "All Ancestors")
				       (check-all-ancestors a-node))
				      ((string= function "Hide A Node")
				       (check-hide-node a-node))
				      ((string= function "Erase A Node")
				       (erase-node a-node))
				      ((string= function "Assert A Node")
				       (assert_a_node a-node))
				      ((string= function "Mark as a Base Node")
				       (make_base a-node))
				      (t t))))
			;; Reset button
			(s-value aux-button-panel :value-obj nil)
			(s-value aux-button-panel :value nil)
			;; Clean up selection slots
			(s-value display-aggregate :selected NIL)
			(when node (s-value node :selected NIL))
			(opal:update display-window))))




; ==================================================================================
;
; PROMPT
; ------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'prompt opal:text
		   (:left 5) (:top 5)
		   (:string "Type the node's name: "))




; ==================================================================================
;
; FILE-PROMPT
; -----------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'file-prompt opal:text
		   (:left 5) (:top 5)
		   (:string "Type the file's name: "))




; ==================================================================================
;
; CLICK-PROMPT
; ------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'click-prompt opal:multi-text
		   (:left 5) (:top 5)
		   (:string "Press the <SHIFT> key and click left on the Display window
over the point where you want the new node."))




; ==================================================================================
;
; REPLY
; -----
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'reply opal:Cursor-text
		   (:left 152) (:top 5) (:width 300))




; ==================================================================================
;
; REPLY-INTER
; -----------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'reply-inter inter:text-interactor
		   (:obj-to-change reply)
		   (:start-where nil) 
		   (:running-where t)
		   (:window dialogue-window)
		   (:start-event nil)	;Start explicitly right after creation
		   (:stop-event #\Return)
		   (:abort-event :rightdown)
		   (:abort-action
		    #'(lambda (interactor obj-over event)
			(declare (ignore interactor obj-over event))
			(if (and (boundp 'prompt) prompt)
			    (opal:remove-components dialogue-aggregate prompt reply))
			(if (and (boundp 'file-prompt) file-prompt)
			    (progn (opal:remove-components dialogue-aggregate 
							   file-prompt reply)
				   (s-value display-window :h-scroll-bar-p t)
				   (s-value display-window :v-scroll-bar-p t)))
			(s-value reply :string "")
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil)
			(opal:update button-window)
			(opal:update dialogue-window)))
		   (:final-function
		    #'(lambda (inter string evt end-string x y)
			(declare (special *node-reply*))
			(declare (ignore inter string evt end-string x y))
			(cond (*node-reply*
			       ;; Send <reply> to <display-node>
			       ;; without a final !
			       (let ((reply-string (string-trim 
						    '(#\newline #\space #\tab)
						    (g-value reply :string))))
				 (if (equal (char reply-string 0) #\")
				     (let ((final-string 
					    (string-right-trim 
					     '(#\!)
					     (string-trim
					      '(#\")
					      reply-string))))
				       (display-node final-string))
				   (let ((final-string 
					  (string-upcase
					   (string-right-trim '(#\!)
							      reply-string))))
				     (display-node final-string)))))
			      (t nil))
			(s-value reply :string "")
			(opal:update dialogue-window)
			;; Reset button
			(s-value main-button-panel :value-obj nil)
			(s-value main-button-panel :value nil)
			(opal:update button-window))))
  
;; Start window in center of scrolling area
;; (NB: increment values must be NEGATIVE)


; ==================================================================================
;
; CREATE_BLANK_NODE_INTER
; -----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'create_blank_node_inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :control-leftdown)
		   (:start-where t)
		   (:final-function #'create_blank_node)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))
  



; ==================================================================================
;
; CREATE_BLANK_NODE_INTER_PANEL
; -----------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'create_blank_node_inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_blank_node)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))
  
  
  
  
; ==================================================================================
;
; CREATE_BLANK_ARC_INTER
; ----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================  
  
  (create-instance 'create_blank_arc_inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :control-middledown)
		   (:start-where t)
		   (:final-function #'create_blank_arc)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))
  



; ==================================================================================
;
; CREATE_BLANK_ARC_INTER_PANEL
; ----------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'create_blank_arc_inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_blank_arc)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))




; ==================================================================================
;
; CREATE_BLANK_DOUBLE_ARC_INTER
; -----------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'create_blank_double_arc_inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :shift-control-middledown)
		   (:start-where t)
		   (:final-function #'create_blank_double_arc)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))




; ==================================================================================
;
; CREATE_BLANK_DOUBLE_ARC_INTER_PANEL
; -----------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'create_blank_double_arc_inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_blank_double_arc)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))

  


; ==================================================================================
;
; UNIVERSAL-QUANTIFIER-INTER
; --------------------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'universal-quantifier-inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :F5)
		   (:start-where t)
		   (:final-function #'create_universal_quantifier)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))




; ==================================================================================
;
; UNIVERSAL-QUANTIFIER-INTER_PANEL
; -------------------------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'universal-quantifier-inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_universal_quantifier)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))
  



; ==================================================================================
;
; NUMERICAL-ENTAILMENT-INTER
; --------------------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'numerical-entailment-inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :F6)
		   (:start-where t)
		   (:final-function #'create_numerical_entailment_frame)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))




; ==================================================================================
;
; NUMERICAL-ENTAILMENT-INTER_PANEL
; -------------------------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'numerical-entailment-inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_numerical_entailment_frame)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))
  



; ==================================================================================
;
; ANDOR-INTER
; -----------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'andor-inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :F7)
		   (:start-where t)
		   (:final-function #'create_andor_frame)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))




; ==================================================================================
;
; ANDOR-INTER_PANEL
; -----------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'andor-inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_andor_frame)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))
  



; ==================================================================================
;
; THRESH-INTER
; -----------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'thresh-inter inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event :F8)
		   (:start-where t)
		   (:final-function #'create_thresh_frame)
		   (:feedback-obj moving-oval)
		   (:abort-if-too-small nil))



  
; ==================================================================================
;
; THRESH-INTER_PANEL
; -----------------
;
;          Arguments         :   N/A
;
;          Returns           :   
;
;          Description       :   
;                                
;                                
;
;          Dependencies      :   
;                                
;
; ==================================================================================

  (create-instance 'thresh-inter_panel inter:two-point-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-event nil)
		   (:start-where t)
		   (:final-function #'create_thresh_frame)
		   (:feedback-obj moving-arrow-line)
		   (:line-p t)
		   (:min-length 20))



; ==================================================================================
;
; TEXT-INTER
; ----------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance `text-inter inter:text-interactor
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :list-leaf-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event :control-rightdown)
		   (:stop-event #\RETURN))




; ==================================================================================
;
; TEXT-INTER_PANEL
; ----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance `text-inter_panel inter:text-interactor
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :list-leaf-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event nil)
		   (:stop-event #\RETURN))




; ==================================================================================
;
; GET-TEXT-OF-RELATION-INTER
; --------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance `get-text-of-relation-inter inter:text-interactor
		   (:continuous nil)
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :leaf-element-of 
				       (g-value display-window :inner-aggregate)
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event :shift-rightdown)
		   (:final-function 
		    #'(lambda (inter obj f-event f-string x y)
			(declare (ignore inter f-event f-string x y))
			(s-value (eval put-text-of-relation-inter) :output-text
				 (g-value obj :string))))
		   (:stop-event #\RETURN))



 
; ==================================================================================
;
; GET-TEXT-OF-RELATION-INTER_PANEL
; --------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================  

  (create-instance `get-text-of-relation-inter_panel inter:text-interactor
		   (:continuous nil)
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :leaf-element-of 
				       (g-value display-window :inner-aggregate)
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event nil)
		   (:final-function 
		    #'(lambda (inter obj f-event f-string x y)
			(declare (ignore inter f-event f-string x y))
			(s-value (eval put-text-of-relation-inter_panel) :output-text
				 (g-value obj :string))))
		   (:stop-event #\RETURN))




; ==================================================================================
;
; PUT-TEXT-OF-RELATION-INTER
; --------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================  

  (create-instance `put-text-of-relation-inter inter:text-interactor
		   (:continuous nil)
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :list-leaf-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event :shift-control-rightdown)
		   (:output-text "")
		   (:final-function #'(lambda (inter obj f-event f-string x y)
					(declare (ignore f-event f-string x y))
					(s-value obj :string (g-value inter :output-text)))) 
		   (:stop-event #\RETURN))




; ==================================================================================
;
; PUT-TEXT-OF-RELATION-INTER_PANEL
; --------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'put-text-of-relation-inter_panel inter:text-interactor
		   (:continuous nil)
		   (:window (g-value display-window :inner-window))
		   (:feedback-obj nil)
		   (:start-where (list :list-leaf-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type opal:text))
		   (:abort-event #-cmu #\control-g
				 #+cmu #\bell)
		   (:start-event nil)
		   (:output-text "")
		   (:final-function #'(lambda (inter obj f-event f-string x y)
					(declare (ignore inter f-event f-string x y))
					(s-value 
					 obj 
					 :string 
					 (g-value inter :output-text)))) 
		   (:stop-event #\RETURN))




; ==================================================================================
;
; NODE-OR-ARC-DELETOR
; -------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'node-or-arc-deletor inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display))
		   (:start-event :shift-control-leftdown)
		   (:abort-event :rightdown)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'destroy_garnet_object))
  



; ==================================================================================
;
; NODE-OR-ARC-DELETOR_PANEL
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================  

  (create-instance 'node-or-arc-deletor_panel inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display))
		   (:start-event nil)
		   (:abort-event :rightdown)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function 
		    #'destroy_garnet_object))




; ==================================================================================
;
; ASSERT-BUILD-TOGGLE-INTER
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'assert-build-toggle-inter inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event :F3)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'assert_build))
  



; ==================================================================================
;
; ASSERT-BUILD-TOGGLE-INTER_PANEL
; -------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'assert-build-toggle-inter_panel inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'assert_build))




; ==================================================================================
;
; MAKE-A-BASE-NODE-INTER
; ----------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'make-a-base-node-inter inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event :F2)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'make_base))




; ==================================================================================
;
; MAKE-A-BASE-NODE-INTER_PANEL
; ----------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'make-a-base-node-inter_panel inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'make_base))




; ==================================================================================
;
; MAKE-A-BASE-PATTERN-INTER
; -------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================

  (create-instance 'make-a-base-pattern-inter inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event :F4)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'make_pattern))




; ==================================================================================
;
; MAKE-A-BASE-PATTERN-INTER_PANEL
; -------------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Function.   Called from MAIN-BUTTON-PANEL2 when
;                                "1st Generation Ancestors" is called.  This starts 
;                                the process of starting the 
;                                NODE-SELECTOR_DESCRIBE-UP-NODE interactor 
;                                that will act on the node.  
;
;          Author            :   John S. Lewocz, 5/92.
;
; ==================================================================================
  
  (create-instance 'make-a-base-pattern-inter_panel inter:button-interactor
		   (:window (g-value display-window :inner-window))
		   (:start-where (list :list-element-of 
				       display-window 
				       :temporary-nodes-and-arcs-on-display
				       :type node))
		   (:start-event nil)
		   (:continuous NIL)
		   (:how-set :set)
		   (:final-function #'make_pattern))
  
  ;>>>> Update and start the loop

  (garnet-gadgets:scroll-win-inc display-window 
				 (floor (- *totalwidth* *totaldisplaywidth*) 2)
				 (floor (- *totalheight* *totaldisplayheight*) 2))

  (opal:update display-window)  
  (opal:update input-button-window)  
  (opal:update button-window)  
  (opal:update dialogue-window)  
  (inter:main-event-loop))










;;  (garnet-gadgets:scroll-win-to display-window 
;;				 (floor (/ *totaldisplaywidth* 2))
;;				 (floor (/ *totaldisplayheight* 2)))
  

