;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSLOG; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: classifiers.lisp,v 1.6 1993/09/08 23:58:28 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snepslog)


; Modifications:
;   Choi 2/13/92: Included handling of numerical quantifiers
;   Choi 4/28/92: Included handling of set arguments

(defun parentised-snepslog-expression? (exp)
  (and (listp exp)
       (listp (first exp))
       (null (rest exp))
       (snepslog-expression? (first exp))))

(defun snepslog-expression? (exp)
  (or (snepslog-variable? exp)
      (snepslog-atomic-node? exp)
      ; this line is added to test set arguments
      (snepslog-set-arguments? exp)
      (snepslog-relation? exp)
      (snepslog-connected-expression? exp)
      (snepslog-quantified-expression? exp)
      ; this line is added to test numerical quantifiers
      (snepslog-num-quantified-expression? exp)))

(defun wff? (exp &aux name)
"Receives an expression and returns something non-nil if it is a legal wff or nil, otherwise."
  (setq exp (first (flistify exp)))
  (if (symbolp exp)
      (setq name (symbol-name exp)))
  (and (stringp  name)
       (> (length name) 3) ;name should have at least 4 characters
       (string= name
		"WFF"
		:end1 2
		:end2 2); the first 3 chars should be WFF
       (node (intern (concatenate 'string "M" (subseq name 3)) 'snepslog));there should exists the sneps node
       ))

(defun snepslog-variable? (exp)
"Receives an expression and returns t if it is a snepslog variable or nil, otherwise."
  (setq exp (first (flistify exp)))
  (and (symbolp exp)
       (eql #\? (char (symbol-name exp) 0))))

(defun snepslog-atomic-node? (exp)
"Receives as arguments an expression. Returns t if it's an atomic node"
  (let ((list (flistify exp)))
    (and (atom (car list))
	 (car list)
	 (null (rest list)))))

(defun snepslog-relation? (exp)
"Receives as arguments an expression. Returns t if it's a snepslog relation."
  (and (listp exp)
       (atom (first exp))
       (not (non.predicate? (first exp)))
       (second exp)
       (list.of.snepslog.expressions? (second exp))
       (null (cddr exp))))

;; Modified to handle function term in set argument:
(defun snepslog-set-arguments? (exp)
  "Return t when an expression = (arg |,| arg |,| ... |,| arg), where
   each arg is either an atom or a function term."
  (let ((arglist (separate exp 'snepsul:|,|)))
    (dolist (arg arglist t)
      (unless (or (snepslog-atomic-node? arg)
		  (snepslog-relation? arg))
	(return nil)))))

(defun snepslog-connected-expression? (exp)
"Receives as arguments an expression. Returns t if it's a connected expression (with connectives)"
  (or (snepslog-entailment? exp)
      (snepslog-thresh? exp)
      (snepslog-andor? exp)))

(defun snepslog-entailment? (exp)
  (or (snepslog-or-entailment? exp)
      (snepslog-and-entailment? exp)))

(defun snepslog-and-entailment? (exp)
  (and (listp exp)
       (eq 'snepsul:&=> (second exp))
       (third exp) ;the consequent list must exists
       (first exp) ;the antecedent list must exists
       (list.of.snepslog.expressions? (first exp))
       (list.of.snepslog.expressions? (third exp))
       (null (cdddr exp))))

(defun snepslog-or-entailment? (exp)
  (and (listp exp)
       (or (let ((=>-arguments (separate exp 'snepsul:=>)))
	     (and (= (length =>-arguments) 2)
		  (highest.priority.operation? 'snepsul:=> exp)
		  (or (snepslog-expression? (first =>-arguments))
		      (parentised-snepslog-expression? (second =>-arguments)))
		  (or (snepslog-expression? (second =>-arguments))
		      (parentised-snepslog-expression? (first =>-arguments)))))
	   (and (eq 'snepsul:v=> (second exp));Is it an or-entailemnt?
		(list.of.snepslog.expressions? (first exp))
		(first exp) ;the antecedent list must exists
		(third exp) ;the consequent list must exists
		(list.of.snepslog.expressions? (third exp))
		(null (cdddr exp))))))

(defun snepslog-thresh? (exp)
  (or (snepslog-equivalence? exp)
      (snepslog-general-thresh? exp)))

(defun snepslog-equivalence? (exp)
  (and (listp exp)
       (let ((<=>-arguments (separate exp 'snepsul:<=>)))
	 (and (> (length <=>-arguments) 1)
	      (highest.priority.operation? 'snepsul:<=> exp)
	      (or (snepslog-expression? (first <=>-arguments))
		  (parentised-snepslog-expression? (second <=>-arguments)))
	      (or (snepslog-expression? (second <=>-arguments))
		  (parentised-snepslog-expression? (first <=>-arguments)))))))

(defun snepslog-general-thresh? (exp)
  (and (listp exp)
       (eq (first exp) 'sneps:thresh)
       (or (ntuple-of-...? (second exp) #'numberp 'snepsul:\, 1)
	   (ntuple-of-...? (second exp) #'numberp 'snepsul:\, 2))
       (list.of.snepslog.expressions? (third exp))
       (null (cdddr exp))))

(defun snepslog-andor? (exp)
  "Returns t if exp (the argument) is the snepslog-representation of an andor
   connective. An andor-connective may be either a general-andor or a not."
  (or (snepslog-general-andor? exp)
      (snepslog-not? exp)
      (snepslog-or? exp)
      (snepslog-and? exp)))

(defun snepslog-general-andor? (exp)
  "Returns t if exp is the snepslog-representation of a general andor, i.e.,
   if exp have the form (andor (<integer>,<integer>) <set of expressions>)."
  (and (listp exp) 
       (eq 'snepsul:andor (car exp))
       (ntuple-of-...? (second exp) #'numberp 'snepsul:\, 2)
       (list.of.snepslog.expressions? (third exp))
       (null (cdddr exp))))

(defun snepslog-or? (exp)
  (let ((or-arguments (separate exp 'or)))
    (and (> (length or-arguments) 1)
         (highest.priority.operation? 'or exp)
	 (or (snepslog-expression? (first or-arguments))
	     (parentised-snepslog-expression? (first or-arguments))
	 (or (snepslog-expression? (second or-arguments))
	     (parentised-snepslog-expression? (second or-arguments)))))))


(defun snepslog-and? (exp)
  (let ((and-arguments (separate exp 'and)))
    (and (> (length and-arguments) 1)
	 (highest.priority.operation? 'and exp)
	 (or (snepslog-expression? (first and-arguments))
	     (parentised-snepslog-expression? (first and-arguments)))
	 (or (snepslog-expression? (second and-arguments))
		  (parentised-snepslog-expression? (second and-arguments))))))

(defun snepslog-not? (exp)
  (and (listp exp)
       (eq 'snepsul:~ (first exp))
       (highest.priority.operation? 'snepsul:~ exp)
       (snepslog-expression? (make.top.level.list (rest exp)))))

(defun snepslog-quantified-expression? (exp)
  (and (listp exp)
       (or (eq 'snepsul:all (first exp))
	   (eq 'snepsul:exists (first exp)))
       (ntuple-of-...? (second exp) #'symbolp 'snepsul:\, (1+ (floor (length (second exp)) 2)))
       (null (cdddr exp))
       (snepslog-expression? (caddr exp))))






; auxiliar classification functions:


(defparameter priority.operations.list '((snepsul:~) (or) (and) (snepsul:=> snepsul:<=>)))

(defun same.priority? (operation list.of.operations)
  (member operation list.of.operations :test #'eq))

(defun list.of.higher.priority.operations (operation)
  (rest (member operation
		priority.operations.list
		:test #'same.priority?)))


(defun highest.priority.operation? (operation expression)
  (dolist (higher.priority.operation.list (list.of.higher.priority.operations operation) t)
    (dolist (higher.priority.operation higher.priority.operation.list)
      (when (member higher.priority.operation expression)
	(return-from highest.priority.operation? nil)))))


; Check whether an expression is a numerical quantifier expression.
; a numerical quantifier expression consists of 4 components,
;     1) the word 'nexists'
;     2) parameter: (emin, emax, etot)
;        - there are 3 kinds of permissible combinations of parameter types,
;           type 1:  emin, emax, etot are numbers
;           type 2:  emin, etot are numbers, and emax is '_'
;           type 3:  emax is number, and emin, etot are '_'
;     3) quantifier variable list   ex) (x,y)
;     4) quantified expression 
;
;   E.g.,
;
;     nexists(2,4,5)(x,y) ({person(x),dog(y),owns(x,y)}:{spoils(x,y)})
;     nexists(3,_,5)(x) ({professor(x)}:{in(x,meeting)})
;     nexists(_,1,_)(y) ({person(y)}:{mother(y,x)}).

(defun snepslog-num-quantified-expression? (exp)
  (and (listp exp)
       (eq 'snepsul:nexists (first exp))
       (ntuple-of-...? (second exp) 
		       #'(lambda(x) (or (numberp x) 
					(eq x 'snepsul:\_))) 
		       'snepsul:\, 3)
       (ntuple-of-...? (third exp) #'symbolp 'snepsul:\, 
		       (1+ (floor (length (third exp)) 2)))
       (num-quant-exp? (fourth exp))
       (null (cddddr exp))))

; checks whether an expression is a numerically quantified expression.
; a numerically quantified expression has 3 parts,
;     1) antecedent list  ex) {person(x),dog(y)}
;     2) the symbol ':'
;     3) consequent list  ex) {mother(y,x)}

(defun num-quant-exp? (exp)
  (and (listp exp)
       (list.of.snepslog.expressions? (first exp))
       (eq (second exp) 'snepsul:\:)
       (list.of.snepslog.expressions? (third exp))))
