;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: infixfns.lisp,v 1.4 1993/07/20 06:25:00 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; ==========================================================================
;
; = 
; -
;
;      arguments     : ns-sv - (<ns-exp> x <svar>)
;
;      returns       : <node set>
;
;      description   : The <svar> is assigned the <node set> resulting of the
;                      "snevaluation" of the <ns-exp>.
;
;                                         written:  CCC 08/02/83
;                                         modified: ejm 10/23/83
;                                                   njm 04/28/89
;                                                   hc  07/18/93

(defsnepscom = ((nsvar &optional sv)
		(top rearrange ns bns tbns fns))
  (=-fun nsvar sv))

(defun =-fun (nsvar sv)
  (let ((ns (nseval nsvar)))
    (cond ((issys.sv sv)
	   (sneps-error (format nil
				"Attempting to change value of system variable ~a"
				sv)
			"assigment operator"
			"="))
	  (t (set.sv sv ns)))
    (values ns (value.sv 'defaultct))))
 
; ==========================================================================
;
; + 
; -
;
;      arguments     : ns-ns (<ns-exp> x <ns-exp>)
;
;      returns       : <node set>
;
;      description   : The union of the two <node set>s resulting from
;                      "snevaluating" the two <ns-exp>s.
;                      The result is the set of nodes which are either in 
;                      the first or in the second <node set>.
;
;                                         written:  ejm 06/01/84
;                                         modified: njm 04/28/89
;                                                   hc  07/18/93
;
(defsnepscom + ((&rest ns-ns) =)
  (values (union.ns (nseval (first ns-ns)) (nseval (second ns-ns)))
	  (value.sv 'defaultct)))
 
; ==========================================================================
;
; & 
; -
;
;      arguments     : ns-ns (<ns-exp> x <ns-exp>)
;
;      returns       : <node set>
;
;      description   : The intersection of the two <node set>s resulting 
;                      from "snevaluating" the two <ns-exp>s.
;                      The result is the set of nodes which are both in 
;                      the first and in the second <node set>.
;
;                                         written:  ejm 06/01/84
;                                         modified: njm 04/28/89
;                                                   hc  07/18/93
;
(defsnepscom & ((&rest ns-ns) =)
  (values (intersect.ns (nseval (first ns-ns)) (nseval (second ns-ns)))
	  (value.sv 'defaultct)))
 
; ==========================================================================
;
; - 
; -
;
;      arguments     : ns-ns (<ns-exp> x <ns-exp>)
;
;      returns       : <node set>
;
;      description   : The set difference of the two <node set>s resulting 
;                      from "snevaluating" the two <ns-exp>s.
;                      The result is the set of nodes which are in the 
;                      first but not in the second <node set>.
;
;                                         written:  CCC 08/02/83
;                                         modified: ejm 10/25/83
;                                                   njm 04/28/89
;                                                   hc  07/18/93
;
(defsnepscom - ((&rest ns-ns) =)
  (values (compl.ns (nseval (first ns-ns)) (nseval (second ns-ns)))
	  (value.sv 'defaultct)))
 
; ==========================================================================
;
; exceptrels.ns 
; -------------
;
;      arguments     : ns   - <node set> 
;                      rels - <relation set>
;
;      returns       : <node set>
;
;      description   : Returns the <node set> obtained by restricting
;                      "ns" to those <node>s which do not have any arcs in 
;                      the <relation set> "rels".
;
;                                         written:  CCC 08/02/83
;                                         modified:
;
(defun exceptrels.ns (ns rels)
  (declare (special ns rels))
  (cond ((and ns rels)
	 (mapcan #'(lambda (n)
		     (declare (special n))
		     (if (not (anyrels.n n rels)) (list n)))
		 ns))
	(t ns)))
 
; ==========================================================================
;
; anyrels.n 
; ---------
;
;      arguments     : n    - <node>  
;                      rels - <relation set>
;
;      returns       : <boolean>
;
;      description   : If the <node> "n" has any arcs in the <relation set> 
;                      "rels", then it returns "true", otherwise returns 
;                      "false".
;
;                                         written:  CCC 08/02/83
;                                         modified: SCS 06/06/87
;
(defun anyrels.n (n rels)
  "Returns T if the node N has any outgoing arcs in the relation set RELS."
  (do ((rs rels (others.rs rs)))
      ((isnew.rs rs) nil)
    (if (nodeset.n n (choose.rs rs)) (return t))))
 
; ==========================================================================
;
; >
; -
;
;      arguments     : rs-sv  - (<rs-exp> x <svar>)
;
;      returns       : <relation set>
;
;      description   : The <s-var> is assigned the <relation set> obtained from 
;                      "rsevaluating" the <rs-exp>.
;
;                                         written:  CCC 08/15/83
;                                         modified: ejm 10/25/83
;                                                   njm 04/28/89
;                                                   hc  07/18/93
;
(defsnepscom > ((&rest rs-sv) (top rearrange rs))
  (values (set.sv (second rs-sv) (rseval (first rs-sv)))
	  (value.sv 'defaultct)))
 
; ==========================================================================
;
; _
; -
;
;      arguments     : ns-rs  - (<ns-exp> x <rs-exp>)
;
;      returns       : <node set>
;
;      description   : It "snevaluates" the <ns-exp>, and removes 
;                      from the resulting <node set> any <node>s which have
;                      an arc in the <relation set> obtained from "rsevaluating"
;                      the <rs-exp>.
;
;                                         written:  CCC 08/15/83
;                                         modified: ejm 10/25/83
;                                                   njm 04/28/89
;                                                   hc  07/18/93
;
(defsnepscom _ ((&rest ns-rs) =)
  (values (exceptrels.ns (nseval (first ns-rs)) (rseval (second ns-rs)))
	  (value.sv 'defaultct)))
 
; ==========================================================================
;
; !
; -
;
;       arguments     :  snepsul-exp - <ns-exp> (+ context)
;
;       returns       : <node set> x <context>
;
;       description   : It asserts the <molecular nodes> in the <node set>
;                       resulting from "snevaluating" the <ns-exp> in the
;                       context specified in the context description of
;                       <snepsul-exp>. It prints a warning for each
;                       non-<molecular node> in that <node set>.
;
;       side-effects  : It side-effects the system <svar> "assertions". 
;
;                                        written : CCC 
;                                        modified: ejm 10/25/83
;                                                  njm 10/12/88
;                                                  njm 04/28/89
;                                                  njm/hc 05/10/89
;                                                  hc  07/18/93
;
(defsnepscom ! ((&rest snepsul-exp) (top rearrange ns bns tbns fns))
  (let ((crntct (processcontextdescr snepsul-exp)))
    (declare (special crntct))
    (values (assert-nodes (nseval (getsndescr snepsul-exp)) crntct)
	    crntct)))
