;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: path.lisp,v 1.3 1993/06/04 06:24:05 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; =============================================================================
; Data Type:   <followed paths> ::= ( <followed path> .... <followed path> )
;                    
;              <followed path> ::= ((<node> <flag>) ... (<node> <flag>))
;
;              <flag> ::= t | nil
; =============================================================================
; =============================================================================
; A <followed path> represents the nodes gone through by a path followed to 
; reach a certain node. The <flag> indicates whether the associated node was
; required to be asserted (t) or not (nil). The nodes associated to a flag that
; is t will contribute to the support of the inferred node. For example, the
; <followed path> ((m4 nil) (m3 t) (m2 nil) (m1 t)), means that node m4 was
; reached by following a path that started in m1, that the path went through
; nodes m2 and m3, and that the nodes that should be taken into account when
; computing the support of an eventually inferred node are m1 and m3. We will
; say that a followed path ends in the node of its first element, and starts 
; in the node of its last element. 
;             
;
; =======================================================================================================
;
;                
; CONSTRUCTORS   add-nodes         : ((<node> status) (<node> status) (<node> status) ...) 
;                                     x <followed paths> --> <followed paths>
;
;                intersect-paths   : <followed paths> x <followed paths>
;                                      --> <followed paths>
;
;                put-in-front      : <followed paths> x <followed path>
;                                      --> <followed paths>
;
;                assert-and-remove : ((<node> status) (<node> status) (<node> status) ...)
;                                     x <followed paths> x
;                                     ((<node> status) (<node> status) (<node> status) ...)
;                                     --> <followed paths>
;
; SELECTORS      select-paths : <followed paths> x <node> --> <followed paths>
;
; UTILITY        retrieve-supporting-nodes : <followed paths> x
;                                            ((<node> status) (<node> status) (<node> status) ...)
;                                            -->
;                                            ((<node> (<node> ... <node>)) ... 
;                                             (<node> (<node> ... <node>)))
; ======================================================================================================



; ==============================================================================
;
; select-paths 
; ------------
;
;       arguments     : fpaths - <followed paths>
;                       n - <node>
;
;       returns       : <followed paths>
;
;       description   : Finds all the followed paths of fpaths that end in node
;
;                                          written:  mrc 10/24/89


(defun select-paths (fpaths node)
  (apply 'append (mapcar #'(lambda (l)
			     (if (iseq.n (caar l)  node) (list l)))
			 fpaths)))          


; ==============================================================================
;
; add-nodes 
; ---------
;
;       arguments     : nodes - ((<node> status) (<node> status) (<node> status) ...)
;                       fpaths - <followed paths>
;
;       returns       : <followed paths>
;
;       description   : Adds to the end of each path of fpaths a <followed path>, (<node> nil),
;                       for each <node> in nodes.
;
;       example       : (add-nodes '((m5 t) (m6 nil)) 
;                                  '(((m3 nil) (m2 t) (m1 t)) ((m3 nil) (m4 t) (m1 t))))
;                       will return
;                       (((M6 NIL) (M3 NIL) (M4 T) (M1 T)) 
;                        ((M6 NIL) (M3 NIL) (M2 T) (M1 T)) 
;                        ((M5 NIL) (M3 NIL) (M4 T) (M1 T))
;                        ((M5 NIL) (M3 NIL) (M2 T) (M1 T)))
;
;                                          written:  mrc 10/24/89


(defun add-nodes (nodes fpaths)
  (let (result)
    (dolist (node nodes result)
      (dolist (path fpaths)
	(setq result (cons (cons (list (car node) nil) path) result))))))

; ==========================================================================
;
; intersect-paths
; ---------------
;
;       arguments     : fpaths1 - <followed paths>
;                       fpaths2 - <followed paths>
;
;       returns       : <followed paths>
;
;       description   : Computes the followed paths in a path P = (and P1 P2)
;                       fpaths1 are the followed paths for path P1
;                       fpaths2 are the followed paths for path P2
;
;       example       : (path-intersect '(((m3 nil) (m2 t) (m1 t)) ((m3 nil) (m4 t) (m1 t)))
;                                       '(((m3 nil) (m8 t) (m1 t)) ((m3 nil) (m5 t) (m1 t))))
;                       will return
;                       (((M3 NIL) (M5 T) (M4 T) (M1 T)) 
;                        ((M3 NIL) (M8 T) (M4 T) (M1 T)) 
;                        ((M3 NIL) (M5 T) (M2 T) (M1 T))
;                        ((M3 NIL) (M8 T) (M2 T) (M1 T)))
;
;                                          written:  mrc 10/24/89


(defun path-intersect (fpaths1 fpaths2)
  (let (result)
    (dolist (path1 fpaths1 result)
      (dolist (path2 fpaths2)
	(setq result (cons (path-intersect-1 path1 path2) result))))))

(defun path-intersect-1 (p1 p2)
  (let ((result (list (car p1))))
    (dolist (el2 p2)
      (unless (mymember el2 p1)
	(setq result (append result (list el2)))))
    (dolist (el1 (cdr p1) result)
      (setq result (append result (list el1))))))

(defun mymember (el lst)
  (do ((l lst (cdr l)) (res nil (equal el (car l))))
      ((or res (null l))  (if res res))))

; ==========================================================================
;
; assert-and-remove
; -----------------
;
;       arguments     : asserted-nodes - ((<node> status) (<node> status) (<node> status) ...)
;                       fpaths         - <followed paths>
;                       node           - ((<node> status) (<node> status) (<node> status) ...)
;
;       returns       : <followed paths>
;
;       description   : Computes the followed paths when a ! was found in a path.
;                       asserted-nodes is the value returned by the function check-!
;                       (the nodes that were found to be asserted), and node
;                       is the first argument of function path-infer-1 (the nodes
;                       that were required to be asserted).
;       example       : (assert-and-remove '((m3 t)) 
;                                          '(((m3 nil) (m2 nil) (m1 t)) 
;                                            ((m3 nil) (m4 nil) (m1 t)) 
;                                            ((m5 nil) (m4 t) (m1 t)))
;                                          '((m3 t) (m5 t)))
;                       will return
;                        (((M3 T) (M2 NIL) (M1 T)) 
;                         ((M3 T) (M4 NIL) (M1 T)))
;
;
;                                          written:  mrc 10/24/89


(defun assert-and-remove (asserted-nodes fpaths node)
  (apply 'append (mapcar #'(lambda (l) (cond ((is-member (first (first l)) asserted-nodes)
					      (list (cons (list (first (first l)) t) (rest l))))
					     ((is-member (first (first l)) node) nil)
					     (t (list l))))
			 fpaths)))

(defun is-member (node lst)
  (cond ((null lst) nil)
	((eq node (first (first lst))))
	(t (is-member node (rest lst)))))

; ==========================================================================
;
; retrieve-supporting-nodes
; -------------------------
;
;       arguments     : fpaths - <followed paths>
;                       nodes  - ((<node> status) (<node> status) (<node> status) ...)
;
;       returns       : ((<node> (<node> ... <node>)) ... (<node> (<node> ... <node>)))
;
;       description   : For each node in nodes, for each path in fpaths that ends
;                       in that node this function retrieves the nodes that are
;                       part of a followed path whose second element is t.
;
;       example       : (retrieve-supporting-nodes 
;                           '(((a nil) (c t) (d t)) ((a nil) (s nil) (d t)) ((b nil) (x t) (d t)))
;                           '((a t) (b t)))
;                       will return 
;                         ((B (X D)) (A (D)) (A (C D)))
;
;
;                                          written:  mrc 10/24/89



(defun retrieve-supporting-nodes (fpaths nodes)
  (let (result)
    (dolist (node nodes result)
      (do ((path fpaths (cdr path)))
	  ((null path))
	(let* ((el (lisp:find (car node) path :key #'caar))
	       (l (list (car node) (apply 'append (mapcar #'(lambda (x)
							      (if (eq (cadr x) t)
								  (list (car x))))
							  el)))))
	  (unless (or (null el) (mymember l result))
	    (setq result (cons l result)))))))) 

; ==========================================================================
;
; put-in-front
; ------------
;
;       arguments     : fpaths - <followed paths>
;                       fpath - <followed path>
;                       
;       returns       : <followed paths>
;
;       description   : Puts fpath at the end of each followed path of fpaths.
;
;
;                                          written:  mrc 10/24/89


(defun put-in-front (fpaths fpath)
  (mapcar #'(lambda (l) (cons fpath l)) fpaths))


