;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: node2.lisp,v 1.6 1993/07/02 00:30:35 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; =============================================================================
;
; Test Functions
;
; =============================================================================

(defmacro isnodeid (id)
  "Returns T if `id' is a legal node identifier prefix,
      viz. one of: v, p, b, tm, tv, tp.
      Returns NIL otherwise."
  ;; written:  ejm 7/26/83
  `(get 'gennewnode ,id))

(defmacro istempnodeid (id)
  "Returns T if `id' is a legal identifier prefix for a temporary node,
   viz: tm, tv, or tp.
   Returns NIL otherwise."
  ;; written:  ejm 7/26/83
   `(member ,id '(tm tv tp)))
;
; ==============================================================================
;
; Auxiliary Functions
;
; ==============================================================================

(defmacro incridcounter (id)
  "Increments the integer suffix associated with the identifier prefix `id'.
   Returns `id'."
  ;; written:  ejm 7/26/83
   `(setf (get 'gennewnode ,id)
          (lisp::+ (get 'gennewnode ,id) 1)))

(defmacro idcounter (id)
  "Returns the integer last used for a node identifier with prefix `id'."
  ;; written:  ejm 7/26/83
  `(get 'gennewnode ,id))

; ==============================================================================
;
; Constructor Functions
;
; ==============================================================================
;

; ==============================================================================
;
; node
; ----
;
;        arguments     : ident - <identifier> 
;
;        returns       : <node> or NIL
;
;        description   : Given an <identifier> "ident", it returns the
;                        the <node> whose <identifier> is "ident".
;
;                                        written:  ejm 7/26/83
;                                        modified: ssc 11/04/87
;                                                  scs    ? 
;                                                  (hc,njm 3/29/89)
;                                                  hc 10/2/90
;
;(defun node (ident)
;  (or (get ident '=snode)
;      ;; If node is all lowercase letters created by a grammar.
;      (get (intern (nstring-downcase (princ-to-string ident))) '=snode)
;      ;; If node is capitalized as created by a grammar.
;      (get (intern (string-capitalize (princ-to-string ident))) '=snode)))
;
(defun node (ident)
  ;; This is a kludge: On the TI-Explorers GET also takes a list instead
  ;; of a symbol. SNePSLOG uses that in various places, and until it
  ;; gets cleaned up we catch that case here. (hc, 10/1/90)
  (when (consp ident) (setq ident (first ident)))
  ((lambda (symb-ident)
     (or (get symb-ident '=snode)
	 ;; If node is all lowercase letters created by a grammar.
	 ;; Intern into SNEPSUL not SNEPS (hc, 10/1/90)
	 (get (intern (format nil "~(~a~)" symb-ident) 'snepsul) '=snode)
	 ;; If node is capitalized as created by a grammar.
	 ;; Intern into SNEPSUL not SNEPS (hc, 10/1/90)
	 (get (intern (format nil "~:(~a~)" symb-ident) 'snepsul) '=snode)
	 ))
   ;; Guarantees that symb-ident is a symbol.
   (cond ((node-p ident)
	  ;; Already a node, just return it (hc 6/14/90)
	  (return-from node ident))
         ((numberp ident) (un-ize ident))
	 ((symbolp ident) ident)
	 ;; Also allow strings, nodes and maybe other junk (hc 6/7/90)
	 ((stringp ident) (intern ident 'snepsul))
	 ;; This case should always bomb!
	 (t ident))))

;
(defmacro newnode (ident type)
  "Creates and returns a new permanent node with identifier `ident' and type `type'.
      `Type must be one of: :base :mol :var :pat.
   Returns NIL if a node with that identifier already exists."
  ;; written:  ejm 7/26/83
  ;; modified: scs 2/13/87
  `(unless (node ,ident)
     (let ((n (make-node :na ,ident :type ,type :perm t)))
       (setf (get ,ident '=snode) n)
       (set.sv 'nodes (insert.ns n (value.sv 'nodes)))
       n)))

(defun newtempnode (ident type)
  "Creates and returns a new temporary node with identifier 'ident' and type 'type'.
   'type' must be one of: :mol :var :pat."
;;written : ejm 07/26/83
;;modified : scs 02/11/87
  (make-node :na ident :type type))

;
(defmacro newpbase.n (ident)
  "Creates and returns a new permanent base node whose identifier is `ident'.
   Returns NIL if one such already exists."
  ;; written:  ejm 7/29/83
  `(newnode ,ident ':base))

;
(defmacro gennewnode (id type)
  "Generates and returns a new node of type `type' and identifier `id'#
      `id' must be one of: b, m, v, p, tm, tv, tp.
      `type must be one of: :base :mol :var :pat."
  ;; written:  ejm 7/26/83
  ;; modified: scs 2/11/87
  ;;           ssc 5/10/89
  `(cond ((isnodeid ,id)
	  (cond ((istempnodeid ,id)
		 (incridcounter ,id) 
		 (newtempnode
		   (intern
		     (concatenate 'string 
				  (symbol-name ,id)
				  (prin1-to-string (idcounter ,id))))
		   ,type))
		(t (do ((result nil
				(newnode (intern (concatenate 'string
							      (symbol-name ,id)
							      (prin1-to-string (idcounter ,id))))
					 ,type)))
		       (result result)
		     (incridcounter ,id)))))
	 (t (error "function gennewnode -- wrong id -- ~s " ,id))))

;;
;; Keeps track of the last integer used to form a node identifier with
;; given prefix.
;;
;;
;;                      modified:  njm  09/27/88
;;
(mapc #'(lambda (pname value) (setf (get 'gennewnode pname) value))
      '(b m v p tm tv tp) '(0 0 0 0 0 0 0))

(defmacro genpbase.n ()
  "Creates and returns a new permanent base node with a system generated identifier."
  ;; written:  ejm 7/29/83
  ;; modified: scs 2/20/87
  `(gennewnode 'b ':base))

(defmacro genpmol.n ()
  "Creates and returns a new permanent molecular node with system generated identifier."
  ;; written:  ejm 7/29/83
  ;; modified: scs 2/20/87
   `(gennewnode 'm ':mol))

(defmacro genpvar.n ()
  "Creates and returns a new permanent variable node with system generated identifier."
  ;; written:  ejm 07/29/83
  ;; modified: ejm 10/26/83
  ;; modified: scs 02/20/87
  `(let ((varn (gennewnode 'v ':var)))
     (set.sv 'varnodes (insert.ns varn (value.sv 'varnodes)))
     varn))

(defmacro genppat.n ()
  "Creates and returns a new permanent pattern node with system generated identifier."
  ;; written:  ejm 7/29/83
  ;; modified: scs 02/20/87
  ;; modified: scs 04/15/89
  `(let ((patn (gennewnode 'p ':pat)))
     (set.sv 'patterns (insert.ns patn (value.sv 'patterns)))
     patn))

(defmacro gentmol.n ()
  "Creates and returns a new temporary molecular node."
  ;; written  ejm 7/29/83
  ;; modified: scs 02/20/87
  `(gennewnode 'tm ':mol))

(defmacro gentvar.n ()
  "Creates and returns a new temporary variable node."
  ;; written:  ejm 7/29/83
  ;; modified: scs 2/20/87
  `(gennewnode 'tv ':var))
;
;
; ==============================================================================
;
; gentpat.n
; ---------
;
;       arguments     : none
;
;       returns       : <temporary pattern node>
;
;       description   : It creates a new <temporary pattern node>.
;
;       side-effects  : see function gennewnode
;
;                                        written:  ejm 7/29/83
;                                        modified:
;
;

(defun gentpat.n ()
   (gennewnode 'tp ':pat))

;
;
; ==============================================================================
;
; is.n 
; ----
;
;        arguments      : u - <universal>
;
;        returns        : <boolean>
;
;        description    : It returns "true" if "u" is a <node>, or "false"
;                         otherwise.
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/11/87
;
;
(defun is.n (u)
  (node-p u))
;
;
; ==============================================================================
;
; isperm.n 
; --------
;
;        arguments      : n - <node> 
;
;        returns        : <boolean>
;
;        description    : It returns "true" if "n" is a <permanent node>, or
;                         "false" otherwise.
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/11/87
;
;
(defmacro isperm.n (n)
   `(node-perm ,n))
;
;
; ==============================================================================
;
; isbase.n
; --------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : It returns "true" if "n" is a <base node>,
;                       or "false" otherwise.
;
;                                        written:  ejm 7/29/83
;                                        modified: scs 2/11/87
;
;
(defmacro isbase.n (n)
   `(eq (node-type ,n) ':base))
;
;
; ==============================================================================
;
; ismol.n
; -------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : It returns "true" if "n" is a <molecular node>,
;                       or "false" otherwise.
;
;                                        written:  ejm 7/29/83
;                                        modified: scs 2/11/87
;
;
(defmacro ismol.n (n)
     `(eq (node-type ,n) ':mol))
;
;
; ==============================================================================
;
; ispat.n
; -------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : It returns "true" if "n" is a <pattern node>, or
;                       "false" otherwise.
;
;                                        written:  ejm 7/29/83
;                                        modified: scs 2/11/87
;
;
(defmacro ispat.n (n)
   `(eq (node-type ,n) ':pat))



; ==============================================================================
;
(defmacro node-to-number.n (n)
  "If the name of the node n looks like a number, returns the number."
  ;; written: scs 6/20/88
  `(read-from-string (symbol-name (node-na ,n))))
;
; ==============================================================================
;
; isnumber.n
; ----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean.
;
;       description   : It returns "true" if "n" is a <number node>, or
;                       "false" otherwise.
;
;                                        written :  ejm 10/03/83
;                                        modified:  scs 02/11/87
;                                        modified:  scs 09/29/88
;
(defun isnumber.n (n)
  ;; n is a <number node> iff the entire symbol-name of its node access forms a
  ;; number.
  (multiple-value-bind (name namelen)
      (read-from-string (symbol-name (node-na n)))
    (and (numberp name)
	 (>= namelen (length (symbol-name (node-na n)))))))
	 

;(defmacro isnumber.n (n)
;   `(numberp (node-to-number.n ,n)))
;
; ==============================================================================
;
; nodeset.n
; ---------
;
;        arguments     : n - <node>
;                        r - <relation> 
;
;        returns       : <node set>
;
;        description   : It returns the <node set> related to the <node> "n"
;                        by the <relation> "r".
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/13/87
;
;
(defmacro nodeset.n (n r)
   `(getnodeset.fcs ,r (node-fcableset ,n)))
;
;
; =============================================================================
;
(defun quantifier-of.n (v)
  "Returns the quantifier arc that quantifies the variable node v, or NIL, if none."
  ;; written by: scs 4/22/88
  (or (and (nodeset.n v 'forall-) 'forall)
      (and (nodeset.n v 'exists-) 'exists)
      (and (nodeset.n v 'pevb-) 'pevb)))
; ==============================================================================
;
; setnodeset.n
; ------------
;
;        arguments     : n  - <node>
;                        r  - <relation>
;                        ns - <node set> 
;
;        returns       : <node>
;
;        description   : It adds the <cable> <"r", "ns">
;                        to the cableset of <node> "n". 
;
;        side-effects  : It side effects "n". If the <relation> "r" was
;                        already in "n" the corresponding <node set>
;                        is replaced by the new one. If "ns" is a <new nodeset>
;                        the pair "r-ns" is removed from "n".
;
;                                        written:  ejm 07/26/83
;                                        modified: scs 02/13/87
;                                                  ssc 04/24/87
;                                                  hc   6/07/90
;
(defmacro setnodeset.n (n r ns)
  `(setf (node-fcableset ,n)
         (cond ((isnew.ns ,ns)
	        (delete.fcs ,r (node-fcableset ,n)))
	       (t (replace.fcs ,r ,ns (node-fcableset ,n))))))
;
;
; ==============================================================================
; setfreevars.n
; -------------
;
;        arguments     : n - <node> 
;                        freevars - <node set>
;
;        returns       : <node>
;
;        description   : It inserts the <node set> containing the free
;                        variables dominated by <node> "n" in the 
;                        <node>'s information.
;                           
;
;                                        written:  ejm 09/30/83
;                                        modified: scs 02/13/87
;
;
(defmacro setfreevars.n (n freevars)
  `(prog2 (setf (node-freevars ,n) ,freevars)
	  ,n))
;
;
; ==============================================================================
;
; updatenodeset.n
; ---------------
;
;        arguments     : n  - <node>
;                        r  - <relation> 
;                        newn - <node> 
; 
;        returns       : <node>
;
;        description   : Inserts "newn" in the <node set> related to "n" by "r"
;                        if the <node> "newn" is not already there.
;
;        side-effects  : It side effects the <node> "n".
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/13/87
;;;                                                ssc 2/21/87
;
;
(defmacro updatenodeset.n (n r newn)
  `(setf (node-fcableset ,n)
	 (insert.fcs ,r 
		     (insert.ns ,newn (new.ns))
		     (node-fcableset ,n))))
;
;
; ==============================================================================
;
; freevars.n
; ----------
;
;        arguments     : n - <node> 
;
;        returns       : <node set>
;
;        description   : It returns the <node set> containing the free
;                        variables dominated by <node> "n".
;                           
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/13/87
;
;
(defmacro freevars.n (n)
   `(or (node-freevars ,n)
        (and (isvar.n ,n)
	     (list ,n))))
;
;
; ==============================================================================
;
; iseq.n
; ------
;
;       arguments     : n1 - <node>
;                       n2 - <node>
;
;       returns       : <boolean>
;
;       description   : It returns "true" if n1 and n2 are the same node,
;                       "false" otherwise.
;
;                                        written:  ejm 8/10/83
;                                        modified:
;
;
(defun iseq.n (n1 n2)
  (eq n1 n2))
;
;
; ==============================================================================
; 
; ==============================================================================
;
; isless.n
; --------
;
;       arguments     : n1 - <node>
;                       n2 - <node>
;
;       returns       : <boolean>
;
;       description   : It returns "true" if n1 compares as less than n2;
;                       "false" otherwise.
;
;                                        written:  scs 06/06/87
;                                        modified:
;
;
(defun isless.n (n1 n2)
  (string< (node-na n1) (node-na n2)))
;
;
; ==============================================================================
; ==============================================================================
;
;   The following functions are particular to this implementation:
;
; ==============================================================================
;
; nodeaccess
; ----------
;
;        arguments     : n - <node> 
;
;        returns       : <atom>
;
;        description   : It returns the interned <atom> used to access the 
;                        uninterned atom representing the <node> "n".
;
;                                        written:  ejm 7/26/83
;                                        modified: scs 2/13/87
;
;
(defmacro nodeaccess (n)
  `(node-na ,n))

; ==============================================================================
;
; Print-description.n
; -------------------
;
;       arguments     : n - <node>
;                       outunit - <unit>
;
;       returns       : nil
;
;       description   : It prints a description of "n" into "outunit" --
;                       a <dotted pair> composed of a description of
;                       the  <node> "n" itself and of a description  of
;                       its <flat cable set>.
;                       Not to be confused with the SNePSUL function
;                       "describe" which describes also the dominated
;                       <node>s.
;
;       side-effects  : It prints a description of the <node> into "outunit".
;
;                                        written :  ejm 10/04/83
;                                        modified:  scs 02/20/87
;                                                   ssc 02/21/87
;
(defmacro Print-description.n (n outunit)
  `(format ,outunit "~A~%" (cons ,n (node-fcableset ,n))))

;
; ==============================================================================
;
; describe.n
; ----------
;
;       arguments     : n - <node>
;
;       returns       : <atom>
;
;       description   : It returns an  <atom> which is a description of
;                       the  <node> "n" to be printed.
;                       The description includes a "!" if n is an
;                       <assertion node>.
;
;                                        written :  ejm 06/05/84
;                                        modified:  scs 02/20/87
;
(defun describe.n (n) n)
;
;
; =============================================================================
;
; activation.n
; ------------
;
;       arguments     : n - <node>
;
;       returns       : <process>
;
;       description   : returns the activation process of "n"
;                       (assumes that "n" does have one)
;
;                                        written :  rgh 11/11/85
;                                        modified:
;
;
(defmacro activation.n (n)
  `(node-activation ,n))


;
; =============================================================================
;
; activated.n
; -----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "n" has an activation process
;                        already attached, "false" otherwise
;
;                                        written :  rgh 11/11/85
;                                        modified:
;
;
(defmacro activated.n (n)
  `(not (null (activation.n ,n))))

; =============================================================================
;
; activate.n
; ----------
;
;       arguments     : n - <node>
;
;       description   : attaches an activation process to the <node> "n" if
;                        it does not already have one
;
;       side-effects  : affects the function cell of "n"
;
;       implementation: the process-id of the activation process is stored
;                        in the function cell of the uninterned atom which
;                        represents the <node> "n"
;
;                                        written :  rgh 11/11/85
;                                        modified:
;
;
(defmacro activate.n (n)
  `(cond ((not (activated.n ,n))
	  (setf (node-activation ,n) (snip::node-to-process ,n)))))

(defmacro deactivate.n (node)
  "Sets the activation of NODE to nil and frees process registers."
  `(when (activated.n ,node)
    (let ((activation (activation.n ,node)))
      (when (multi:is-process-name activation)
	;; Unbind process name to make register values garbage-collectible
	(makunbound activation))
      (setf (activation.n ,node) nil))))
;
;
; =============================================================================
;
; quantified-vars.n
; -----------------
;
;       arguments     : n - <node>
;
;       returns       : <node set>
;
;       description   : returns the <node set> containing the variable
;                       nodes bound by the quantifier (if any) emanating
;                       from the <node> "n"
;
;                                        written :  rgh 10/06/85
;                                        modified:
;
;
(defmacro quantified-vars.n (n)
  `(let ((av (nodeset.n ,n 'forall))
         (ev (nodeset.n ,n 'exists))
         (pev (nodeset.n ,n 'pevb)))
      (cond ((not (isnew.ns av)) av)
            ((not (isnew.ns ev)) ev)
            ((not (isnew.ns pev)) pev)
            (t (new.ns)))))
;
;
; =============================================================================
;
; dominates.n
; -----------
;
;       arguments     : m1 - <node>
;                       m2 - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "m1" dominates "m2",
;                               "false" otherwise
;
;                                        written :  rgh 07/31/85
;                                        modified:  rgh 08/02/85
;
;
(defun dominates.n (m1 m2)
  (do* ((rem-cs (n-to-downcs m1) (others.cs rem-cs))
	(c (choose.cs rem-cs) (choose.cs rem-cs)))
       ((or (null rem-cs)
	    (do* ((rem-ns (nodeset.c c) (others.ns rem-ns))
		  (n (choose.ns rem-ns) (choose.ns rem-ns)))
		 ((or (null rem-ns)
		      (iseq.n n m2)
		      (dominates.n n m2))
		  rem-ns)))
	rem-cs)))
;
;
; =============================================================================
;
; is-v-ent.n
; ----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "n" is an or-entailment node,
;                               "false" otherwise
;
;                                        written :  rgh 11/11/85
;                                        modified:
;
;
(defmacro is-v-ent.n (n)
  `(not (isnew.ns (nodeset.n ,n 'ant))))
;
;
; =============================================================================
;
; is-&-ent.n
; ----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "n" is an and-entailment,
;                       "false" otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-&-ent.n (n)
  `(and (not (isnew.ns (nodeset.n ,n '&ant)))
        (isnew.ns (nodeset.n ,n 'thresh))
        (isnew.ns (nodeset.n ,n 'pevb))))
;
;
; =============================================================================
;
; is-thresh.n
; -----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "n" is a thresh rule, "false"
;                       otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-thresh.n (n)
  `(and (not (isnew.ns (nodeset.n ,n 'thresh)))
        (not (isnew.ns (nodeset.n ,n 'arg)))))
;
;
; =============================================================================
;
; is-and-or.n
; -----------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "n" is an and-or, "false" otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-and-or.n (n)
  `(not (isnew.ns (nodeset.n ,n 'min))))

(defmacro is-nor.n (n)
  "Returns T if the node represents a NOR, NIL otherwise."
  `(let ((maxargs (nodeset.n ,n 'max)))
     (and maxargs (zerop (node-to-number.n (choose.ns maxargs))))))

(defmacro is-and.n (n)
  "Returns T if the node represents an AND, NIL otherwise."
  ;; written: scs 6/21/88
  `(let ((minargs (nodeset.n ,n 'min))
	 (tot (cardinality.ns (nodeset.n ,n 'arg))))
     (and minargs tot (lisp:= (node-to-number.n (choose.ns minargs)) tot))))
;
;
; =============================================================================
;
; is-num-ent.n
; ------------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" is "n" is a numerical entailment,
;                       "false" otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-num-ent.n (n)
  `(and (not (isnew.ns (nodeset.n ,n 'thresh)))
        (not (isnew.ns (nodeset.n ,n '&ant)))))
;
;
; =============================================================================
;
; is-num-quant.n
; --------------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" is "n" is numerically quantified,
;                       "false" otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-num-quant.n (n)
  `(not (isnew.ns (nodeset.n ,n 'pevb))))
;
;
; =============================================================================
;
; is-non-deriv.n
; --------------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true" is "n" is a rule stating the
;                       non-derivability of its argument, "false" otherwise
;
;                                        written :  rgh  3/31/86
;                                        modified:
;
;
(defmacro is-non-deriv.n (n)
  `(not (isnew.ns (nodeset.n ,n 'i-//-))))
;
;
; =============================================================================
;
(defun symbol-type.n (node-name)
  "Determines the type of a node with NODE-NAME."
  (let* ((node-name (symbol-name node-name)))
    (if (and (member (aref node-name 0) '(#\M #\P #\V))
	     (lisp:>= (length node-name) 2)
	     (every #'digit-char-p (subseq node-name 1)))
	(case (aref node-name 0)
	  (#\M :MOL)
	  (#\P :PAT)
	  (#\V :VAR))
      :BASE)))

(defun n^ (node-name)
  ;; Yet another node referencer (used for forward references during printing).
  ;; If a node with NODE-NAME does not yet exist then a new node gets built
  ;; with its type inferred from NODE-NAME.
  (or (node node-name)
      (newnode node-name (symbol-type.n node-name))))

;
; =============================================================================
;
(defun copy-assertion-state.n (oldnode newnode)
  "If oldnode is asserted, makes newnode asserted.
   Returns newnode."
  ;; written by:  scs 5/12/88
  ;; modified by: njm 11/23/88
  (declare (special crntct))
  (when (isassert.n oldnode)
    (snip:addsupport.n (filter.ctcs (node-asupport oldnode)
				    (if (is.ct crntct) crntct (value.sv crntct)))
		       newnode))
  newnode)

;
; =============================================================================
;
;
; newjust
; --------
; 
;
;       arguments     : n - <node>
;                       ot    - <otag>
;                       ct    - <context>
;
;       returns       : <node>
;
;       description   : Takes as arguments a <node> `n', an <otag> `ot' and a 
;                       <context> "ct". 
;
;                       It adds a new derivation to the node `n', with 
;                       origin tag `ot' and context `ct'.
;
;                       Returns `n'.
;
;       
;
;
;                                  written :  jpm 12/02/82 
;                                  modified:  njm 10/06/88
;
;
;
(defun newjust (n ot ct) 
  (setf (node-asupport n)
	 (insert.ctcs ot ct (node-asupport n)))
  n)

