;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: context3.lisp,v 1.3 1993/06/04 06:23:38 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; =============================================================================
;
; assert.n
; --------
;
;       arguments     : n - <node>
;                       crntname - <svar>  (optional)
;
;       returns       : <node> | ---
;
;       description   : it sets <node> "n" as an hypothesis, performing the 
;                       following operations:
;                          1 - creates a new context with "n" as hyps;
;                          2 - creates a new context which hyps are the union 
;                              of the current context's hyps with "n", and 
;                              computes its restriction;
;                          3 - If the second argument is supplied then:
;                                a) the context "crntct is no more called
;                                   crntname;
;                                b) names the context created in 
;                                   step 2 as "crntname";
;                                c) the value of the special variable "crntct"
;                                   is the context created in step 2
;
;       side-effects  : it side-effects the node "n", and the system 
;                       svar "contexts" and the special variable "crntct"
;
;
;
;                                        written :  ejm 10/03/83
;                                        modified:  scs 02/13/87 
;                                        modified:  scs 10/07/87
;                                        modified:  njm 09/20/88 
;                                        modified:  njm 10/03/88 
;                                        modified:  njm 10/13/88
;                                        modified:  mrc 10/26/88
;
;
;
(defmacro assert.n (n &optional (crntname nil))
  `(let* ((ct (if (is.ct crntct) crntct (value.sv crntct)))
	  (newassrt (buildcontext (makeone.ns ,n)))
	  (newct    (fullbuildcontext (makeone.ns ,n) (makeone.cts ct))))
     (declare (special crntct))
     (setf (node-asupport ,n) (insert.ctcs 'hyp
					   newassrt
					   (node-asupport ,n)))
     (setf (context-restriction newassrt)
	   (remove-hyp-itself ,n (context-restriction (getcontext (new.ns)))))
     (setf (context-restriction newct) (context-restriction ct))
     (if ,crntname (name.ct newct ,crntname))
     (snebr:ck-contradiction ,n newct 'assertion)))

;
;
; ==============================================================================
;
; isassert.n
; ----------
;
;       arguments     : n - <node>
;
;       returns       : <context> | NIL
;
;       description   : It returns the context justifying "n" if "n" is an 
;                       Hypothesis and is believed in the current context.
;                       Returns "false" otherwise.
;
;
;                                        written :  ejm 10/03/83
;                                        modified:  scs 02/11/87
;                                        modified:  njm 09/20/88
;                                        modified:  njm 10/03/88
;                                        modified:  njm  4/27/89
;
;
(defun isassert.n (n &optional cntx)
  (declare (special crntct))
  (let ((ctcs (node-asupport n))
	(ct (if cntx
		(if (is.ct cntx) cntx (value.sv cntx))
		(if (is.ct crntct) crntct (value.sv crntct)))))
    (declare (special ct))
    (or (isassert1.n (getcontextset.ctcs 'hyp ctcs))
	(isassert1.n (getcontextset.ctcs 'der ctcs)) 
	(isassert1.n (getcontextset.ctcs 'ext ctcs)))))

;
;
; ==============================================================================
;
; isassert1.n
; -----------
;
;       arguments     : cts - <context set>
;
;       returns       : <boolean>
;
;       description   : If any of the contexts of the context set "cts" is a 
;                       subset of the current context "crntct" then returns
;                       "true". Returns "false" otherwise.
;
;
;                                        written :  njm 10/03/88
;                                        modified:  njm 02/13/89
;                                        modified:  njm  4/27/89
;
;
(defun isassert1.n (cts)
  (declare (special ct))
  (cond ((isnew.cts cts) nil)
	((issubset.ct (choose.cts cts) ct) t)
	(t (isassert1.n (others.cts cts)))))


;
;
; ==============================================================================
;
; isassert-print.n
; ----------------
;
;       arguments     : n - <node>
;
;       returns       : <boolean>
;
;       description   : returns "true if the node "n" has a non empty 
;                       node-asupport. Returns "false" otherwise.
;
;
;       ATTENTION     : This function is only used in the print of a
;                       node structure, elsewhere the "isassert.n"
;                       function must be used.
;
;
;                                        written :  njm 10/03/88
;                                        modified:  
;
;
(defun isassert-print.n (n)
  (if (node-asupport n) t))

  
;
;
; ==============================================================================
;
;remove-hyp-itself
; ----------------
;
;       arguments     : n   - <node>
;                       cts - <context set>
;
;       returns       : <context set>
;
;       description   : removes 'n' from the contexts in 'cts'.
;
;
;
;
;                                        written :  mrc 11/28/88
;                                        modified:  
;
;  
(defun remove-hyp-itself (n cts)
  (let ((newcts (new.cts)))
    (do.cts (ct cts)
      (setq newcts (insert.cts (fullbuildcontext (compl.ns (context-hyps ct)
							   (makeone.ns n))
						 (new.cts))
			       newcts)))
    newcts))


