;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEBR; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: sniphandler.lisp,v 1.5 1993/06/04 06:23:13 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snebr)


; =============================================================================
;
; snip-contr-handler 
; -------------------
; 
;
;       arguments     : newnode - <node> 
;                       contrnd - <node>
;                       context - <context>
;
;       returns       : ---
;
;       description   : This function takes as arguments a node `contrnd' which 
;                       contradicts some newly derived node `newnode' (refer 
;                       to the function ck-contradiction), and a context
;                       `context' under which the newly asserted node was derived.
;                       It warns the user of the detected contradiction and
;                       calls the function contr-h to handle the contradiction.
;
;
;                                   
;                                  written :  mrc  11/15/88
;
;
;                                                
;
(defun snip-contr-handler (newnode contrnd context)
  (declare (special sneps:outunit snip:crntctname))
  (progn (format sneps:outunit
		 "~%~%~T A contradiction was detected within context ~A.~
                   ~%~T The contradiction involves the newly derived node:~
                   ~%~T~T~T ~A ~
                   ~%~T and the previously existing node:~
                   ~%~T~T~T ~A"
		 snip:crntctname
		 (snip:describe-or-surface newnode nil)
		 (snip:describe-or-surface contrnd nil))
	 (contr-h (ctcs-to-cts (sneps:node-asupport newnode))
		  (ctcs-to-cts (sneps:node-asupport contrnd))
		  (sneps:ctcs-to-ots (sneps:node-asupport newnode))
		  (sneps:ctcs-to-ots (sneps:node-asupport contrnd))
		  context)))		
 
;
; =============================================================================
;
;
; contr-h 
; -------
;
;       arguments     : newnd-supps    - <context set>
;                       contrnd-supps  - <context set>
;                       newnd-otlst    - <ot list>
;                       contrnd-otlst  - <ot list>
;                       context        - <context>
;
;       returns       : <node set>
;
;       description   : This function handles contradictions detected during
;                       inference.
;                       In the current implementation the contradiction is
;                       resolved by the user which may choose one of three
;                       options:
;                        1. [C]ontinue anyway, knowing that a contradiction 
;                           is derivable;
;	                 2. [R]e-start the exact same run in a different 
;                           context which is not inconsistent;
;                        3. [D]rop the run altogether.
;
;                       This function asks the user what his choice is and
;                       takes the necessary actions to execute that choice.
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                  modified:  mrc  11/15/88
;                                             hc   04/26/89
;
;                                         
;

(defun contr-h (newnd-supps contrnd-supps newnd-otlst contrnd-otlst context)
  (declare (special sneps:outunit))
  (options-in-contr-h)
  (let* ((ans (read-contr-h-option))
	 (lastcommand  #-explorer (value.sv 'sneps:command)
		       #+explorer (if (eql (first (value.sv 'sneps:command)) 'sys:displaced)
				      (second (value.sv 'sneps:command))
				      (value.sv 'sneps:command))
		       ))
    (cond ((or (eq ans 'snepsul:C) (eql ans 1)))
	  (t (multi:clear-all-queues)
	     (sneps:clear-infer)
	     (cond ((or (eq ans 'snepsul:R)(eql ans 2))
		    (change-context newnd-supps contrnd-supps newnd-otlst contrnd-otlst context)
		    (sneps:topsneval lastcommand))
		   ((is-add lastcommand) (remove-new-hyp lastcommand)))
	     (throw 'snip:Stop-Handled-by-Contradiction-Handler nil)))))

;
; =============================================================================
;
;
; options-in-contr-h 
; ------------------
;
;
;       arguments     : 
;
;       returns       : nil
;
;       description   : This function tells the user the possibilities he
;                       has to handle a contradiction detected during 
;                       inference.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;
;
;
(defun options-in-contr-h ()
  (declare (special sneps:outunit))
  (format sneps:outunit
	  "~%~T You have the following options:~
	   ~%~T~T 1. [C]ontinue anyway, knowing that a contradiction is derivable;~
	   ~%~T~T 2. [R]e-start the exact same run in a different context which is~
	   ~%~T~T    not inconsistent;~
	   ~%~T~T 3. [D]rop the run altogether.
           ~%~T (please type c, r or d)"))

;
; =============================================================================
;
;
; read-contr-h-option 
; -------------------
;
;
;       arguments     : 
;
;       returns       : 'C | 'R | 'D
;
;       description   : This function reads an answer of the user.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                             hc  10/19/88 (get rid of repeat)
;
(defun read-contr-h-option ()
  (declare (special sneps:outunit sneps:inunit))
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (if (or (eq ans 'snepsul:c)
		  (eq ans 'snepsul:R)
		  (eq ans 'snepsul:D)
		  (eql ans 1)
		  (eql ans 2)
		  (eql ans 3))
	      (RETURN ans))
	  (format sneps:outunit "Please type c, r or d"))))

;
; =============================================================================
;
; change-context 
; --------------
;
;
;       arguments     : newnd-supps   - <context set>
;                       contrnd-supps - <context set>
;                       newnd-otlst   - <origin tag list>
;                       contrnd-otlst - <origin tag list>
;                       context       - <context>
;
;       returns       : <node set>
;
;       description   : This function takes as arguments:
;                        'newnd-supps'   - set of contexts that support the
;                                          newly derived node
;                        'contrnd-supps' - set of contexts that support the
;                                          node which contradicts the newly 
;                                          derived node.
;                        'newnd-otlst'   - list of origin tags corresponding
;                                          to 'newnd-supps'.
;                        'contrnd-otlst' - list of origin tags corresponding
;                                          to 'contrnd-supps'.
;                        'context'       - context in which the new node was 
;                                          derived.
; 
;                      This function changes the current context, which is
;                      inconsistent, to a consistent context. The user may
;                      change the current context by
;                        - removing one or more hypotheses from inconsistent
;                          sets of hypotheses that are part of the cuurent
;                          context (see make-consistent) 
;                        - removing hypotheses from a set of hypotheses that
;                          is not known to be inconsistent and that is
;                          also part of the current context (see browse-
;                          through)
;                        - adding new hypotheses to the current context
;                          (see add-new-hyps)
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  mrc 12/12/88
;

(defun change-context (newnd-supps contrnd-supps newnd-otlst contrnd-otlst context) 
  (inform-user newnd-supps contrnd-supps)
  (let ((crntct (context-hyps context))
	(unseen-hyps (context-hyps context))
	(new-hyps (new.ns))
	(new-ot nil)
	(contr-ot nil)
	(contr-otlst contrnd-otlst))
    (sneps:do.cts (new-supp newnd-supps)
      (setq new-ot (first newnd-otlst))
      (setq newnd-otlst (rest newnd-otlst))
      (setq contr-otlst contrnd-otlst)
      (sneps:do.cts (contr-supp contrnd-supps)
	(let ((inc-hyps (union.ns (context-hyps new-supp)
				  (context-hyps contr-supp))))
	  (setq contr-ot (first contr-otlst))
	  (setq contr-otlst (rest contr-otlst))
	  (when (sneps:issubset.ns inc-hyps  crntct)
	    (setq crntct (union.ns (compl.ns crntct inc-hyps)
				   (make-consistent inc-hyps
						    (fullbuildcontext crntct (new.cts))
						    new-supp
						    contr-supp
						    new-ot
						    contr-ot))))
	  (setq unseen-hyps (compl.ns unseen-hyps inc-hyps)))))
    (when (not (isnew.ns unseen-hyps))
      (setq crntct (union.ns (compl.ns crntct unseen-hyps)
			     (browse-through unseen-hyps crntct))))
    (change-context-name crntct)
    (if (not (isnew.ns (setq new-hyps (add-new-hyps))))
	(change-context-name (union.ns crntct new-hyps))
	crntct)))


(defun change-context-name (lhyps)
  (declare (special snip:crntctname)) 
  (name.ct (fullbuildcontext lhyps (new.cts))  snip:crntctname)
  lhyps)

;
; =============================================================================
;
;
; inform-user 
; -----------
;
;
;       arguments     : newnd-supps   - <context set>
;                       contrnd-supps - <context set>
;
;       returns       : <>
;
;       description   : This function tells the user from which sets he
;                       must remove one or more hypotheses in order to 
;                       make the current consistent.
;
;       
;
;
;                                  written :  mrc 11/16/88 


(defun inform-user (newnd-supps contrnd-supps)
  (declare (special sneps:outunit))
  (format sneps:outunit "~%~%~T In order to make the context consistent you must delete at least~
                         ~%~T one hypothesis from each of the following sets of hypotheses:")
 (sneps:do.cts (newsupp newnd-supps)
    (sneps:do.cts (contrsupp contrnd-supps)
      (format sneps:outunit "~%~T~T~T ~A" 
	      (snip:slight-describe-or-surface.ns
	       (union.ns (context-hyps newsupp)
		         (context-hyps contrsupp))
	       nil))))
 (format sneps:outunit "~%~%~T"))


;
; =============================================================================
;
; browse-through 
; --------------
;
;
;       arguments     : hyplst     - <node set>
;                       fullcthyps - <node set>
;
;       returns       : <node set>
;
;       description   : This function allows the user to inspect and
;                       eventually discard the hypotheses in 'hyplst'
;                       (see browse-through-hyps).
;
;
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:
;
;
(defun browse-through (hyplst fullcthyps)
  (declare (special sneps:outunit))
  (format sneps:outunit
	  "~%~T The following (not known to be inconsistent) set of ~
	   ~%~T hypotheses was also part of the context where the ~
	   ~%~T contradiction was derived: ~
	   ~%~T~T~T ~A
	   ~%~T Do you want to inspect or discard some of them?"
	  hyplst)
  (cond ((user-says-yes) (browse-through-hyps hyplst fullcthyps))
	(t hyplst)))
;
; =============================================================================
;
; add-new-hyps
; ------------
;
;
;       arguments     : -----
;
;       returns       : <node set>
;
;       description   : This function asks the user if he wants to
;                       add new hypotheses to the current context
;                       and returns the hypotheses added by the user
;                       (eventually none).
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:
;
;
(defun add-new-hyps ()
  (declare (special sneps:outunit))
  (format sneps:outunit
	  "~%~T Do you want to add some new hypotheses?")
  (if (user-says-yes) (request-new-hyps)
      (new.ns)))

;
; =============================================================================
;
;
; request-new-hyps 
; ----------------
;
;
;       arguments     : ----
;
;       returns       : <node set>
;
;       description   : This function reads the hypotheses entered by the
;                       user and returns them.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                             hc  10/19/88 (get rid of repeat)
;
;
(defun request-new-hyps ()
  (declare (special sneps:outunit sneps:inunit))
  (format sneps:outunit
	  "~T Enter the hypotheses using the SNePS command `assert'~
                   ~%~T (type 'end' to finish)")
  (let (nh ans newhyps)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (cond ((eq ans 'snepsul:end)
		 (RETURN newhyps))
		(t (setq nh (sneps:topsneval (insert-context ans)))
		   (if nh
		       (setq newhyps (insert.ns nh newhyps))
		       (format sneps:outunit
			       "~T Oops... something went wrong~
                                 ~%~T Please continue")))))))


(defun insert-context (comm)
  (declare (special snip:crntctname))
  `(,@comm :context ,(intern (string snip:crntctname) 'snepsul)))

;
; =============================================================================
;
;
; remove-new-hyp 
; --------------
;
;
;       arguments     : <command>
;
;       returns       : <node set>
;
;       description   : This function removes the node built by
;                       the last command (which was an add command)
;                       from the current context.
;       
;
;
;                                  written :  mrc  20/12/88
;                                  modified:  
;

(defun remove-new-hyp (lastcommand)
  ;; Assume (is-add lastcommand) is True
  (let ((addcommand (if (eql (first lastcommand) 'snip:add)
			(cons 'sneps:assert (rest lastcommand))
			(cons 'sneps:assert (rest (second lastcommand))))))
    (remove-new-hyp-1 (sneps:topsneval addcommand))))

(defun remove-new-hyp-1 (newhyp)
  (declare (special snip:crntctname))
  (change-context-name (compl.ns (context-hyps (value.sv snip:crntctname))
				 (makeone.ns newhyp))))
;
; =============================================================================
;
;
; is-add 
; ------
;
;
;       arguments     : <command>
;
;       returns       : T | NIL
;
;       description   : This function returns T if 'command'
;                       is an add command.
;       
;
;
;                                  written :  mrc  20/12/88 
;                                  modified:  
;
;
(defun is-add (command) 
  (or (eq (first command) 'snip:add)
      (and (consp (second command))
	   (eql (first (second command)) 'snip:add))))
