;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEBR; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: snepshandler.lisp,v 1.3 1993/06/04 06:23:11 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snebr)


;
; =============================================================================
;
; sneps-contr-handler 
; -------------------
; 
;
;       arguments     : newnode - <node> 
;                       contrnd - <node>
;
;       returns       : 
;
;       description   : This function is called whenever a contradiction is detected,
;                       as the result of the assertion of a new node.
;                       'newnode' is the newly asserted node and 'contrnd' is the
;                       node that contradicts 'newnode' (it is assumed that 'contrnd'
;                       is asserted in the BS defined by the context in which 'newnode'
;                       was asserted - see the function ck-contradiction).
;                       The function sneps-contr-handler i) warns the user of the
;                       detected contradiction, ii) gives the user several choices to
;                       handle the contradiction (see the function options-in-sneps-h),
;                       and iii) implements the user's choice (see the function
;                       implement-sneps-option).
;                       
;                                  written :  njm  10/11/88
;                                  modified:  mrc  10/28/88
;                                  modified:  mrc  12/27/88
;
;                                                
;

(defun sneps-contr-handler (newnode contrnd)
  (declare (special sneps:outunit sneps:crntct))
  (format sneps:outunit
	  "~%~%~T A contradiction was detected within context ~A.~
                   ~%~T The contradiction involves the node you want to assert:~
                   ~%~T~T~T ~A ~
                   ~%~T and the previously existing node:~
                   ~%~T~T~T ~A"
	  sneps:crntct
	  (snip:describe-or-surface newnode nil)
	  (snip:describe-or-surface contrnd nil))
  (options-in-sneps-h)
  (implement-sneps-option (read-sneps-option) newnode contrnd))		
 
;
; =============================================================================
;
;
; options-in-sneps-h 
; ------------------
;
;
;       arguments     : ----
;
;       returns       : ----
;
;       description   : This function informs the user of the possibilities he
;                       has to handle a contradiction detected upon the assertion
;                       of a new node.
;
;       
;
;
;                                  written :  njm 10/06/88 
;                                  modified:  
;
;
;
(defun options-in-sneps-h ()
  (declare (special sneps:outunit))
  (format sneps:outunit
	  "~%~%~%~T You have the following options:~
	   ~%~T~T 1. [c]ontinue anyway, knowing that a contradiction is ~
                     derivable;~
	   ~%~T~T 2. remove [a]nother assertion from this context~
	   ~%~T~T 3. [r]emove the new assertion from this context~
           ~%~T (please type c, a or r)")
  (values))
;
; =============================================================================
;
;
; implement-sneps-option
; ----------------------
;
;       arguments     : option  - <char>
;                       newnode - <node>
;                       contrnd - <node>
;
;       returns       : <node> |  NIL
;
;       description   : This function implements the option chosen by the user
;                       (see the function  options-in-sneps-h).
;                       'newnode' is the newly asserted node and 'contrnd' is
;                       the node that contradicts 'newnode'.
;                       'option' may have the following values:
;                         'c' - meaning that the user chose to continue in spite
;                               of the contradiction. In this case nothing is done.
;                         'r' - meaning that the user chose to remove the newly 
;                               asserted node from the current context. In this
;                               case 'newnode' is removed from the current context. 
;                         'a' - meaning that the user wants to remove one or more
;                               previously existing nodes in order to resolve the
;                               contradiction. In this case, the function 
;                               make-consistent is called for each context that
;                               supports 'contrnd'. 
;
;                                  written :  njm 10/06/88 
;                                  modified:  mrc 12/18/88
;
;
;
;
(defun implement-sneps-option (option newnode contrnd)
  (declare (special sneps:crntct sneps:outunit)) 
  (cond ((equal option 'snepsul:r) (name.ct (buildcontext
					    (remove.ns newnode
						       (context-hyps
							 (value.sv sneps:crntct))))
					  sneps::crntct))
	((equal option 'snepsul:a)
	 (let ((contrsup (ctcs-to-cts (sneps:node-asupport contrnd)))
	       (otlst (sneps:ctcs-to-ots (sneps:node-asupport contrnd))))
;for each context 'ctext' that supports 'contrnd'
	   (dolist (ctext contrsup)
;if 'ctext' is a subset of the current context
	     (when (sneps:issubset.ct ctext
				      (value.sv sneps:crntct)) 
	       (name.ct (fullbuildcontext (union.ns (make-consistent (context-hyps ctext)
                                                                     (value.sv sneps:crntct) 
								     (sneps:getcontext
								       (makeone.ns newnode))
								     ctext
								     'sneps:hyp
								     (first otlst)
								     newnode)
						    (compl.ns (context-hyps
								(value.sv sneps:crntct))
							      (context-hyps ctext)))
					  (new.cts))
			sneps:crntct))
	     (setq otlst (rest otlst)))))))




;
; =============================================================================
;
; read-sneps-option 
; -----------------
;
;
;       arguments     : -------
;
;       returns       : 'C | 'A | 'R
;
;       description   : This function reads the option typed by the user.
;
;       
;
;
;                                  written :  njm 10/11/88 
;                                  modified:  hc 10/19/88 (get rid of repeat)
;
;
;
(defun read-sneps-option ()
  (declare (special sneps:outunit sneps:inunit))
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (if (or (eq ans 'snepsul:C)
		  (eq ans 'snepsul:R)
		  (eq ans 'snepsul:A))
	      (RETURN ans))
	  (format sneps:outunit "Please type c, a or r"))))

;
; =============================================================================
;
;
; make-consistent
; ---------------
;
;
;       arguments     : hyps       - <node set>
;                       context    - <context>
;                       new-ct     - <context>
;                       contr-ct   - <context>
;                       new-ot     - <otag>
;                       contr-ot   - <otag>
;                       newnode    - <node>
;
;       returns       : <node set>
;
;       description   : This function is called in two distinct situations:
;                        1 - When a contradiction is detected as the result
;                            of the assertion of a new node (see implement-
;                            sneps-option), and the user wants to
;                            resolve the contradiction by removing
;                            one or more previously existing nodes. In this
;                            situation, 'hyps' is the set of nodes from which
;                            the user must remove at least one node; the 
;                            optional parameter 'newnode' is present and 
;                            represents the newly asserted node.
;                        2 - When a contradiction is detected during inference,
;                            and the user wants to re-start the same inference
;                            in a new context (see change-context). 
;                            In this situation, 'hyps' is the set of
;                            incompatible hypotheses, from which the user
;                            must remove at least one, in order to make the context
;                            consistent, and the optional parameter 'newnode'
;                            is not present.
;
;                       In both situations we have:
;
;                        'new-ct'   - context that supports the new node (this node 
;                                     may have been either derived or asserted).
;
;                        'contr-ct' - context that supports the node that contradicts
;                                     the new node.
;
;                        'new-ot'   - origin tag associated with 'new-ct'.
;
;                        'contr-ot' - origin tag associated with 'contr-ct'.  
;
;                       Returns the set of hypotheses that remain in 'hyps'
;                       after the interaction with the user.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                  modified:  mrc 11/03/88
;                                             hc  10/19/88 (get rid of repeat)
;
;
;
(defun make-consistent (hyps context new-ct contr-ct new-ot contr-ot &optional newnode)
  (declare (special sneps:outunit))
  (let (ct)
    (loop (setq ct (make-consistent-1 hyps context))
	  (when  (not (sneps:iseq.ns ct hyps)) 
	    (negate-hyps (compl.ns hyps ct)
			 (fullbuildcontext (if newnode (insert.ns newnode ct)
					       ct)
					   (new.cts))
			 new-ct contr-ct new-ot contr-ot)
	    (RETURN ct)))))

;
; =============================================================================
;
;
; make-consistent-1
; -----------------
;
;
;       arguments     : inc-h   - <node set>
;                       context - <context>
;
;       returns       : <node set>
;
;       description   : This function takes as argument an inconsistent 
;                       set of hypotheses and interacts with the user 
;                       until he has removed at least one hypothesis
;                       from this set.
;
;                       Returns the remaining hypotheses in 'inc-h'.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;
;
;
(defun make-consistent-1 (inc-ct context)
  (options-in-make-consistent-1 inc-ct)
  (browse-through-hyps inc-ct (context-hyps context)))

;
; =============================================================================
;
;
; options-in-make-consistent-1
; ----------------------------
;
;
;       arguments     : inc-hyps - <node set>
;
;       returns       : ---
;
;       description   : 
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;
;
;
(defun options-in-make-consistent-1 (inc-hyps)
  (declare (special sneps:outunit))
  (format sneps:outunit
	  "~T   In order to make the context consistent you must delete some hypotheses~
	   ~%~T from the set ~A ~
	   ~%~T You are now entering a package that will enable you to delete some~
	   ~%~T hypotheses from this set.~
	   ~%~T Do you need guidance about how to use the package?"
	  (snip:slight-describe-or-surface.ns inc-hyps nil))
  (if (user-says-yes)
      (format sneps:outunit
	      "~T   For each hypothesis defining the context you have to decide whether~
	       ~%~T you want to keep it or delete it from the context. This process will~
	       ~%~T terminate as soon as you have decided the fate of every hypothesis.~
	       ~%~T At any point you may postpone your decision about what to do to the~
	       ~%~T hypothesis.~%")))


;
; =============================================================================
;
; browse-through-hyps 
; -------------------
;
;
;       arguments     : hyplst     - <node set>
;                       fullcthyps - <node set>
;
;       returns       : <node set>
;
;       description   : For each node in 'hyplst' this function
;                        - inspects the node (see inspect-hyp)
;                        - asks the user about the node's fate
;                          (see enquire-hyp-fate)
;                        
;                       Returns the set of nodes that the user
;                       did not choose to discard from the context. 
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  hc  10/19/88 (get rid of repeat)
;
;
(defun browse-through-hyps (hyplst fullcthyps)
  (let ((keep (new.ns))
	curnt-hyp ans)
    (loop (setq curnt-hyp (car hyplst))
	  (setq hyplst (cdr hyplst))
	  (inspect-hyp curnt-hyp fullcthyps)
	  (setq ans (enquire-hyp-fate)) 
	  (cond ((eq ans 'snepsul:k) (setq keep (insert.ns curnt-hyp keep)))
		((eq ans 'snepsul:u) (setq hyplst (append hyplst (list curnt-hyp))))
		((eq ans 'snepsul:q)
		 (RETURN (insert.ns curnt-hyp (union.ns hyplst keep)))))
	  (if (null hyplst)
	      (RETURN keep)))))

; =============================================================================
;
;
; inspect-hyp
; -----------
;
;
;       arguments     : descr   - <node>
;                       fullct  - <node set>
;
;       returns       : ---
;
;       description   : Tells the user how many and which nodes depend on
;                       'descr' (see  consequences-of-this-hyp),
;                       and asks the user what he wants to
;                       do with hypothesis 'descr'. 
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  
;
;
;
(defun inspect-hyp (curnt-hyp fullct)
  (declare (special sneps:outunit))
  (consequences-of-this-hyp curnt-hyp
			    (nodes-supported-by curnt-hyp  (fullbuildcontext fullct (new.cts))))
  (format sneps:outunit
	  "~%~T What do you want to do with hypothesis ~A?~
	   ~%~T [d]iscard from the context, [k]eep in the context,~
	   ~%~T [u]ndecided, [q]uit this package ~
	   ~%~T (please type d, k, u or q)"
	  (snip:slight-describe-or-surface curnt-hyp nil)))

;
; =============================================================================
;
; consequences-of-this-hyp
; ------------------------
;
;
;       arguments     : h       - <node>
;                       depend  - <node set>
;
;       returns       : <node set>
;
;       description   : This function takes as arguments an hypothesis 'h' and
;                       a list of nodes 'depend' which depend on the hypothesis
;                       `h'.
;                       It enables the user to look at 'h' and at the nodes in 
;                       'depend'.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;
;
;
(defun consequences-of-this-hyp (h depend)
  (declare (special sneps:outunit))
  (format sneps:outunit "Do you want to take a look at hypothesis ~A?"
	  (snip:slight-describe-or-surface h nil))
  (if (user-says-yes)
      (snip:describe-or-surface h sneps:outunit)
      (format sneps:outunit "~%"))
  (consequences-of-this-hyp-1 h depend))



(defun consequences-of-this-hyp-1 (h depend)
  (declare (special sneps:outunit sneps:inunit))
  (cond ((zerop (length depend))
	 (format sneps:outunit
		 "~%~T No nodes depend on hypothesis ~A.~%"
		 (snip:slight-describe-or-surface h nil)))
	(t (format sneps:outunit
		   "~%~T There are ~A nodes depending on hypothesis ~A:~
                    ~%~T~T ~A. ~
                    ~%~T Do you want to look at [a]ll of them, ~
                         [s]ome of them, or [n]one? ~
                    ~%~T (please type a, s or n)"
		   (cardinality.ns depend)
		   (snip:slight-describe-or-surface h nil)
		   (snip:slight-describe-or-surface.ns depend nil))
	   (let ((ans (read-a-s-n)))
	     (cond ((eq ans 'snepsul:a) (show-nodes depend))
		   ((eq ans 'snepsul:s) (show-some-nodes depend))
		   ((eq ans 'snepsul:n) t))))))

;
; =============================================================================
;
;
; read-a-s-n 
; ----------
;
;
;       arguments     : ---
;
;       returns       : 'A | 'S | 'N
;
;       description   : This function reads an answer of the user, and returns it.
;
;       
;
;
;                                  written :  mrc 11/10/88 
;                                  modified:  hc  10/19/88 (get rid of repeat)
;
;
(defun read-a-s-n ()
  (declare (special sneps:outunit sneps:inunit))
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (if (or (eq ans 'snepsul:A)
		  (eq ans 'snepsul:S)
		  (eq ans 'snepsul:N))
	      (RETURN ans))		
	  (format sneps:outunit "Please type a, s or n"))))

;
; =============================================================================
;
; enquire-hyp-fate
; ----------------
;
;
;       arguments     : ---
;
;       returns       : 'D | 'K | 'U | 'Q
;
;       description   : 
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  hc  10/19/88 (get rid of repeat)
;
;
;
(defun enquire-hyp-fate ()
  (declare (special sneps:outunit sneps:inunit))
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (if (member ans '(snepsul:d snepsul:k snepsul:u snepsul:q))
	      (RETURN ans))
	  (format sneps:outunit "Please type d, k, u or q"))))



; =============================================================================
;
;
; nodes-supported-by 
; ------------------
;
;
;       arguments     : hyp     - <node>
;                       ct  - <context>
;
;       returns       : <node set>
;
;       description   : Returns all the nodes which have a support that:
;                        - contains 'hyp'
;                        - is a subset of 'ct'
;
;       
;
;
;                                  written :  mrc 11/03/88 
;                                  modified:  
;
;
;
(defun  nodes-supported-by (hyp ct)
  (nodes-supported-by-1 (value.sv 'sneps:nodes) hyp ct))

(defun nodes-supported-by-1 (allnodes hyp ct)
  (let ((result (new.ns)))
    (sneps:do.ns (nd allnodes result)
      (if (is-supported-by-and-in-context nd hyp ct)
	  (setq result (insert.ns nd result))))))



; =============================================================================
;
;
; is-supported-by-and-in-context 
; ------------------------------
;
;
;       arguments     : nd  - <node>
;                       hyp - <node>
;                       ct  - <context>
;
;       returns       : T | NIL
;
;       description   : Returns T if 'nd' has a support which:
;                         - contains 'hyp'
;                         - is a subset of 'ct'
;       
;
;
;                                  written :  mrc 11/03/88 
;                                  modified:  
;
;
;
(defun is-supported-by-and-in-context (nd hyp ct)
  (is-supported-by-and-in-context-1 (ctcs-to-cts (node-asupport nd)) 
				    hyp
				    ct))

(defun is-supported-by-and-in-context-1 (supps hyp ct)
  (cond ((isnew.cts supps) nil)
	((and (ismemb.ns hyp (context-hyps (choose.cts supps)))
	      (sneps:issubset.ct (choose.cts supps) ct))
	 t)
	(t (is-supported-by-and-in-context-1 (others.cts supps) hyp ct))))






; =============================================================================
;
;
; show-nodes 
; ----------
;
;
;       arguments     : ndlst - <node set>
;
;       returns       : ---
;
;       description   : This function takes as argument a set of nodes
;                       'ndlst' and describes each node in 'ndlst'.
;
;       
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                  modified:  mrc 11/03/88
;
;
;
(defun show-nodes (ndlst)
  (declare (special sneps:outunit))
  (cond ((isnew.ns ndlst) t)
	(t (format sneps:outunit 
		   "~%~T About to describe node ~A "
		   (choose.ns ndlst))
	   (ct-prompt)
	   (snip:describe-or-surface (choose.ns ndlst) sneps:outunit)
	   (show-nodes (others.ns ndlst)))))



;
; =============================================================================
;
;
; show-some-nodes 
; ---------------
;
;
;       arguments     : lhyps   - <node set>
;
;       returns       : <>
;
;       description   : This function takes as argument a set of nodes,
;                       asks the user which nodes from this set he wants
;                       to have described and describes those nodes.
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                             hc  10/19/88 (get rid of repeat)
;
;
(defun show-some-nodes (lhyps)
  (declare (special sneps:outunit sneps:inunit))
  (let (nd)
    (loop (format sneps:outunit
		  "~T   Which node do you want to have described?~
                   ~%~T Note: it should belong to the list ~A"
		  (snip:slight-describe-or-surface.ns lhyps nil))
	  (ct-prompt)
	  (setq nd (read sneps:inunit))
	  (cond ((member (node nd) lhyps)
		 (snip:describe-or-surface (node nd) sneps:outunit)
		 (format sneps:outunit "~%"))
		(t (setq nd nil)))
	  (when (cond (nd (format sneps:outunit
				  "~T Do you want to examine more nodes from ~
                                  the list ~A?"
				  (snip:slight-describe-or-surface.ns lhyps nil))
			  (cond ((user-says-yes) nil)
				(t lhyps)))
		      (t (format sneps:outunit
				 "~T Oops... the node you typed doesn't belong to~
			         ~%~T ~A ~%"
				 (snip:slight-describe-or-surface.ns lhyps nil))
			 nil))
	    (RETURN nil)))))

;
; =============================================================================
;
;
; negate-hyps 
; -----------
;
;
;       arguments     : disb-hyps - <node set>
;                       supp      - <context>
;                       new-supp    - <context>
;                       contr-supp  - <context>
;                       new-ot    - <otag>
;                       contr-ot  - <otag>
;
;       returns       : <>
;
;       description   : Negates each hypothesis in 'disb-hyps'. 
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;                                  modified:  mrc 12/07/88
;
;
;

(defun negate-hyps (disb-hyps supp new-supp contr-supp new-ot contr-ot) 
  (sneps:newjust (choose.ns (apply #'mybuild (list 'max 0 'min 0 'arg disb-hyps)))
		 (make-ot new-ot contr-ot new-supp contr-supp) supp))

(defun mybuild (&rest snd)
  (sneps:find-or-build (sneps:evalsnd snd t)))


(defun make-ot (new-ot contr-ot new-supp contr-supp)
  (cond ((sneps:iseq.ct new-supp contr-supp) (snip:combine-ots* (list new-ot contr-ot)))
	(t 'sneps:ext)))


;
; =============================================================================
;
;
; user-says-yes 
; -------------
;
;
;       arguments     : ---
;
;       returns       : T | NIL
;
;       description   : This function returns T if the user types "yes", "y",
;                       "ok" or "sure", and returns NIL if the user types "n"
;                       or "no".
;
;       
;
;
;                                  written :  jpm 11/30/82 
;                                  modified:  njm 10/06/88
;

(defun user-says-yes ()
  (member (user-says-yes-1) '(snepsul:yes snepsul:y snepsul:ok snepsul:sure)))

(defun user-says-yes-1 () 
  (declare (special sneps:outunit sneps:inunit))
  (let (ans)
    (loop (ct-prompt)
	  (setq ans (read sneps:inunit))
	  (if (or (eq ans 'snepsul:Y)
		  (eq ans 'snepsul:yes)
		  (eq ans 'snepsul:ok)
		  (eq ans 'snepsul:sure)
		  (eq ans 'snepsul:n)
		  (eq ans 'snepsul:no))
	      (RETURN ans))		
	  (format sneps:outunit "Please type n or y"))))

;
; =============================================================================
;
;
; ct-prompt 
; ---------
;
;
;       arguments     : ---
;
;       returns       :  ---
;
;       description   : This function prints a prompt
;
;       
;
;
;                                  written :  jpm 11/30/82 
(defun ct-prompt ()
  (declare (special sneps:outunit))
  (format sneps:outunit "~%=><= "))
