;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)

;This file implements the show macro and related show tactics.

(defmac auto-show-goal
  (show-internal *goal*)
  t
  (lisp-let ((?phi *goal*)) 
    (show-internal ?phi)))

(defmac lemma
  (lemma . ?proof)
  t
  (lisp-bind *goal* nil (progn . ?proof)))

(defmac query-macro
  (? ?phi)
  t
  (lisp-bind *showing* '?phi
    (query-sequent ?phi)))

(emacs-indent ensure 1)

(defmac ensure-macro
  (ensure ?phi . ?body)
  (not (eq ?phi '*goal*))
  (show-sequent ?phi (apply tintern (?phi)
		       (progn . ?body)
		       (note ?phi))))

(defmac ensure-macro-1
  (ensure ?phi . ?body)
  (eq ?phi '*goal*)
  (show-sequent ?phi (progn
		       (progn . ?body)
		       (note ?phi))))

(emacs-indent ensure-wca 1)
(defmac ensure-macro-wca
  (ensure-wca ?phi . ?body)
  t
  (ensure ?phi . ?body))

;(defmac ensure-macro-wca
;  (ensure-wca ?phi . ?body)
;  t
;  (first
;   (ensure ?phi . ?body)
;   (ensure ?phi
;     (refutation-sequent
;      (sequent ((assume (not ?phi)))
;	(ensure (false)
;	  (ensure ?phi
;	    (case-analysis))))))))

(defmac maybe-ensure
  (maybe-ensure ?phi . ?body)
  t
  (protect (ensure ?phi . ?body)))

(defmac protect
  (protect ?proof)
  t
  (first ?proof
	 (do-nothing)))

;;;
;;;(defmac show-using
;;;  (show ?phi using (is ?x ?y))
;;;  t
;;;  (both-sequent
;;;   (ensure (is ?x ?y))
;;;   (show-sequent ?phi
;;;		 (first
;;;		  (ensure ?phi)
;;;		  (second-try-using (is ?x ?y) (show ?phi))))))

;;; Taken out for Case Analysis
;;;(defmac auto-show
;;;  (show ?phi . ?body)
;;;  (not (eq ?phi '*goal*))
;;;  (show-sequent ?phi
;;;		(apply tintern (?phi)
;;;		  (progn . ?body)
;;;		  (first
;;;		   (ensure ?phi)
;;;		   (possible-second-try ?phi)))))

(emacs-indent show-contradiction 1)
(defmac show-contradiction
  (show-contradiction ?phi . ?body)
  t
  (lisp-let ((?not-phi (negation ?phi)))
    (show-internal (and ?phi ?not-phi) . ?body)))

(defvar *current-second-try* nil)

(defmac possible-second-try
  (possible-second-try ?phi)
  t
  (lisp-let ((?t-phi (translate ?phi)))
    (lisp-when (not (equal '?t-phi *current-second-try*))
      (lisp-bind *current-second-try* '?t-phi
	(ensure ?phi
	  (second-try ?t-phi))))))

(defmac second-try-is
  (second-try (is ?c ?w))
  (not (selectmatch ?w
	 ((db-lambda-fun . :anything) t)
	 ((lambda-rel0 . :anything) t)
	 ((apply (apply (rel-from-to) . :anything) . :anything) t)
	 ((apply (apply (total-rel-from-to) . :anything) . :anything) t)
	 ((apply (apply (partial-fun-from-to) . :anything) . :anything) t)
	 ((intersection . :anything) t)
	 ((apply (subset-operator) . :anything) t)
	 ((db-some-such-that . :anything) t)
	 (:anything nil)))
  (lisp-let ((?x (new-proof-variable 'X))
	     (?y (new-proof-variable 'Y)))
    (suppose-there-is ((?x ?c))
      (first
       (ensure (is ?x ?w))
       (suppose-there-might-be ((?y ?w))
	 (ensure-wca (is ?x ?w)))))))

(defmac second-try-is-both
  (second-try (is ?c (intersection ?a ?b)))
  t
  (progn
    (show-internal (is ?c ?a))
    (show-internal (is ?c ?b))))

(defmac second-try-is-subset
  (second-try (is ?c (apply (subset-operator) ?w)))
  t
  (show-internal (is (a-member-of ?c) (a-member-of ?w))))

(defmac second-try-is-lambda-fun-one-arg
  (second-try (is ?c (db-lambda-fun ?type (db-class-combinator ?comb))))
  t
  (lisp-let ((?new-var (new-proof-variable 'X))
	     (?new-body (expression-db-subst ?comb ?new-var 1)))
    (suppose-there-is ((?new-var ?type))
      (show-internal (at-most-one (?c ?new-var)))
      (show-internal (there-exists (?c ?new-var)))
      (show-internal (is (?c ?new-var) ?new-body)))))

(defun create-lambda-bindings (types)
  (let ((first (first types)))
    (when (not (eq first 'to))
      (cons (list (new-proof-variable 'X)
		  first)
	    (create-lambda-bindings (cdr types))))))

(defmac second-try-is-lambda-rel-nil
  (second-try (is ?c (lambda-rel0 ?range-type)))
  t
  (show-internal (is (?c) ?range-type)))

(defmac second-try-is-rel-from-to
  (second-try (is ?c (apply (apply (rel-from-to) ?type) ?range-type)))
  t
  (show-internal (is (?c (a-domain-member-of ?c)) (a-member-of ?range-type))))

(defmac second-try-is-partial-fun-from-to
  (second-try (is ?c (apply (apply (partial-fun-from-to) ?type) ?range-type)))
  t
  (lisp-let ((?new-var (new-proof-variable 'X)))
    (suppose-there-is ((?new-var (a-member-of ?type)))
      (show-internal (at-most-one (?c ?new-var)))
      (show-internal (is (?c (a-domain-member-of ?c)) (a-member-of ?range-type))))))

(defmac second-try-is-total-rel-from-to
  (second-try (is ?c (apply (apply (total-rel-from-to) ?type) ?range-type)))
  t
  (lisp-let ((?new-var (new-proof-variable 'X)))
    (suppose-there-is ((?new-var (a-member-of ?type)))
      (show-internal (there-exists (?c ?new-var)))
      (show-internal (is (?c (a-domain-member-of ?c)) (a-member-of ?range-type))))))

(defmac second-try-is-some-such-that
  (second-try (is ?c (db-some-such-that ?type (db-formula-combinator ?comb))))
  t
  (lisp-let ((?new-var (new-proof-variable 'X)))
    (lisp-let ((?subst (expression-db-subst ?comb ?new-var 1)))
      (progn
	(show-internal (is ?c ?type))
	(suppose-there-is ((?new-var ?c))
	  (show-internal ?subst))))))

(defmac second-try-and
  (second-try (not (implies ?phi (not ?psi))))    ;;; (and ?phi ?psi)
  t
  (progn (show-internal ?phi)
	 (show-internal ?psi)))


;; This subsumes the old second-try-or (since
;; (or ?phi ?psi) translates to (implies (not ?phi) ?psi)).
(defmac second-try-implies
  (second-try (implies ?phi ?psi))
  t
  (first (show-internal (not ?phi))
	 (suppose ?phi (show-internal ?psi))))

(defmac second-try-iff
  (second-try (= ?phi ?psi))
  (eq (expression-category ?phi) 'formula)
  (lemma
    (suppose ?phi
      (show-internal ?psi))
    (suppose ?psi
      (show-internal ?phi))))

(defmac second-try-forall
  (second-try (db-forall ?type (db-formula-combinator ?comb)))
  t
  (lisp-let ((?new-var (new-proof-variable 'Y))
	     (?new-phi (expression-db-subst ?comb ?new-var 1)))
    (suppose-there-is ((?new-var ?type))
      (show-internal ?new-phi))))

(defmac second-try-at-most-one
  (second-try (at-most-one ?c))
  t
  (lisp-let ((?x1 (new-proof-variable 'X))
	     (?x2 (new-proof-variable 'X))
	     (?c1 (new-proof-variable 'C)))
    (suppose-there-might-be ((?c1 ?c))
      (first (ensure (at-most-one ?c))
	     (suppose-there-is ((?x1 ?c)
				(?x2 ?c))
	       (show-internal (= ?x1 ?x2)))))))

(defmac second-try-not-there-exists
  (second-try (not (there-exists ?c)))
  t
  (lisp-let ((?c1 (new-proof-variable 'C1)))
    (classify! ?c
      (suppose-there-might-be ((?c1 ?c))
	(ensure-wca (not (there-exists ?c)))))))

(defmac second-try-there-exists
  (second-try (there-exists ?c))
  (not (selectmatch ?c
	 ((db-some-such-that :anything (db-formula-combinator
					(is ?closed-exp :anything)))
	  (expression-db-closed? (translate ?closed-exp)))
	 (:anything nil)))
  (lisp-let ((?c1 (new-proof-variable 'C1)))
    (suppose-there-might-be ((?c1 ?c))
      (ensure-wca (there-exists ?c)))))


(defmac second-try-there-exists-some-such-that
  (second-try (there-exists (db-some-such-that ?type
					       (db-formula-combinator
						(is ?closed-exp ?open-exp)))))
  (expression-db-closed? (translate ?closed-exp))
  (lisp-let ((?var (new-proof-variable 'VAR))
	     (?open-exp-of-var (expression-db-subst ?open-exp ?var 1)))
    (consider ((?var ?type)) such-that (is ?closed-exp ?open-exp-of-var))))


(defmac second-try-=
  (second-try (= ?x1 ?x2))
  (not (eq 'formula (expression-category ?x1)))
  (lisp-let ((?c1 (new-proof-variable 'C1)))
    (lisp-let ((?c2 (new-proof-variable 'C2)))
      (suppose-there-might-be ((?c1 ?x1) (?c2 ?x2))
	(proof-cond ((and (or (is ?x1 (a-set))
			      (is ?x2 (a-set)))
			  (singleton ?x1)
			  (singleton ?x2))
		     (show-internal (is ?x1 (a-set)))
		     (show-internal (is ?x2 (a-set)))
		     (show-internal (is (a-member-of ?x1)
					(a-member-of ?x2)))
		     (show-internal (is (a-member-of ?x2)
					(a-member-of ?x1)))
		     (ensure-wca (= ?x1 ?x2)))
		    ((and (or (is ?x1 (a-thunk))
			      (is ?x2 (a-thunk)))
			  (singleton ?x1)
			  (singleton ?x2))
		     (show-internal (is ?x1 (a-thunk)))
		     (show-internal (is ?x2 (a-thunk)))
		     (show-internal (is (?x1) (?x2)))
		     (show-internal (is (?x2) (?x1)))
		     (ensure-wca (= ?x1 ?x2)))
		    ((and (at-most-one ?x1)
			  (at-most-one ?x2))
		     (ensure-wca (= ?x1 ?x2)))
		    ((true)
		     (first (ensure (= ?x1 ?x2))
			    (ensure (= ?x1 ?x2)
			      (ensure (is ?x1 ?x2)
				(suppose (there-exists ?x1)))
			      (ensure (is ?x2 ?x1)
				(suppose (there-exists ?x2)))))))))))

(defmac second-try-not-is
  (second-try (not (is ?s ?t)))
  t
  (proof-cond ((singleton ?s)
	       (classify! ?s
		 (classify! ?t
		   (ensure-wca (not (is ?s ?t))))))))


(defmac second-try-other
  (second-try ?phi)
  (selectmatch ?phi
    ((not (implies :anything (not :anything))) nil)
    ((implies (not :anything) :anything) nil)
    ((not (there-exists :anything)) nil) 
    ((not (is . :anything)) nil) 
    ((?constructor . :anything)
     (not (member ?constructor '(is at-most-one = there-exists db-forall))))
    (:anything t))
  (ensure-wca ?phi))



;********************************************************************
;hl-write-as
;********************************************************************

(defun write-as-constant (type)
  `(not-general-const ,(new-gensym) ,type))

;; The following function, given texpr, returns a value such that
;; (is value texpr) and (at-most-one value) are both true.
;; This value is an arbitrary value of texpr, or (fail), if texpr
;; has no values.

;write-as-eval should only be called from context extenders.  In particular, axiom-fun should only
;be called from context extenders.

(defun write-as-eval (texpr)

  (selectmatch texpr

    ((apply (db-lambda ?type (db-class-combinator ?body))
	    ?arg)
     (let ((?val (write-as-eval ?arg)))
       (let ((?const (write-as-constant `(apply ,(second texpr) ,?val)))
	     (written-body (write-as-eval
			    `(when (is ,?val ,?type)
			      ,(expression-db-subst ?body ?val 1)))))
	 (axiom-fun `(= ,?const ,written-body))
	 written-body)))
    
    ((apply (db-lambda-fun ?type (db-class-combinator ?body))
	    ?arg)
     (let ((written-f (write-as-constant (second texpr)))
	   (written-arg (write-as-eval ?arg)))
       (let ((result (write-as-constant `(apply ,written-f ,written-arg))))
	 (axiom-fun `(= ,result
		      ,(write-as-eval
			`(when (is ,written-arg ,?type)
			  ,(expression-db-subst ?body written-arg 1)))))
	 result)))

    ((apply (db-lambda-rel ?type (db-class-combinator ?body))
	    ?arg)
     (let ((written-f (write-as-constant (second texpr)))
	   (written-arg (write-as-eval ?arg)))
       (let ((result (write-as-constant `(apply ,written-f ,written-arg))))
	 (axiom-fun `(= ,result
		      ,(write-as-eval
			`(when (is ,written-arg ,?type)
			  ,(expression-db-subst ?body written-arg 1)))))
	 result)))

    ((apply (member-operator) (the-set-of-all-internal ?class))
     (let ((result (write-as-eval ?class)))
       (axiom-fun `(is ,result ,texpr))
       result))

    ((apply (funcall0-operator) (lambda0 ?class))
     (let ((result (write-as-eval ?class)))
       (axiom-fun `(is ,result ,texpr))
       result))

    ;; This case is for curried functions.
    ((apply (apply ?f ?arg1) ?arg2)
     (let ((written-fun (write-as-eval `(apply ,?f ,?arg1))))
       (write-as-eval `(apply ,written-fun ,?arg2))))

    ;; A helper case for curried functions.
    ((apply (when ?cond (db-lambda ?type (db-class-combinator ?body)))
	    ?arg)
     (let ((?val (write-as-eval ?arg)))
       (let ((?const (write-as-constant
		      `(when ,?cond
			(apply (db-lambda ,?type
				(db-class-combinator ,?body)) ,?val))))
	     (written-body (write-as-eval
			    `(when (and (is ,?val ,?type) ,?cond)
			      ,(expression-db-subst ?body ?val 1)))))
	 (axiom-fun `(= ,?const ,written-body))
	 written-body)))

    ((apply ?f ?arg)
     (let ((written-f (write-as-eval ?f))
	   (written-arg (write-as-eval ?arg)))
       (write-as-constant `(apply ,written-f ,written-arg))))

    ((amb ?a ?b)
     (let ((written-a (write-as-eval ?a))
	   (written-b (write-as-eval ?b)))
       (write-as-constant `(amb ,written-a ,written-b))))

    ((intersection ?a ?b)
     (let ((written-a (write-as-eval ?a))
	   (written-b (write-as-eval ?b)))
       (write-as-constant `(intersection ,written-a ,written-b))))

    ((db-some-such-that ?type (db-formula-combinator ?body))
     (let* ((written-type (write-as-eval ?type))
	    (reduct (expression-db-subst ?body written-type 1)))
       (let ((result
	      (write-as-constant `(when ,reduct
				   ,written-type))))
	 (axiom-fun `(implies (there-exists ,texpr) (there-exists ,result)))
	 result)))

    ((const :anything :anything)
;     (axiom-fun `(= ,texpr ,(write-as-eval ?type)))
     texpr)

    ((db-lambda :anything :anything)
     texpr)

    ((when ?phi ?class)
     (let ((written-class (write-as-eval ?class)))
       `(when ,?phi ,written-class)))

    (:anything
     (write-as-constant texpr))))
    

(defextender hl-write-as (arg)
  (let* ((const (first arg))
	 (texpr (second arg))
	 (written-expression (write-as-eval (translate texpr))))
    (execute-extension `(match-mark! ,written-expression))
    (execute-extension `(axiom (= ,const ,written-expression)))))

(control-extender hl-write-as)

(defpiece (notice-definition mark-for-expansion) (symbol translated-def keylist)
  (when (not (null-context?))
    (setf-undo (expansion-mark symbol) t)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Case Analysis Code:                                                     ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *cases-ulist* (ulist-make))

(defpiece (ontic-init-phase0 :reset-cases-queue) ()
  (setf *cases-ulist* (ulist-make)))

(defun unknown-case? ()
  (ulist-next *cases-ulist*
	      #'(lambda (x)
		  (null (is-true? (uf-find (tintern x)))))))

(declare-variables (class ?c1 ?c ?t1 ?t2 ?c2) (formula ?phi))

(rule when-case-analysis ((= ?c1 (when ?phi ?c))
			  (simplify! ?c1))
  (queue *delay-q* 
    (:lisp
      (when (db-closed? ?c1)
	(ulist-add *cases-ulist* (object-expression ?phi)
		   :justification ?justification)))))

(rule either-case-analysis ((simplify! ?c)
			    (at-most-one ?c)
			    (there-exists ?c)
			    (constant ?a-thing)
			    (= ?c2 (amb ?t1 ?t2))
			    (is ?c ?c2))
  (queue *delay-q*
    (:lisp (when (and (not (is? ?c ?t1))
		      (not (is? ?c ?t2))
		      (not (uf-equal ?a-thing ?c2))
		      (db-closed? ?c)
		      (db-closed? ?c2))
	     (ulist-add *cases-ulist*
			`(is ,(object-expression ?c)
			     ,(object-expression ?t1))
			:justification ?justification)))))

(emacs-indent show-sequent 1)

(defmac auto-show
  (show-internal ?phi ?form1 . ?body)
  (not (eq ?phi '*goal*))
  (match-mark! ?phi
    (show-sequent ?phi
      (progn (progn ?form1 . ?body)
	     (first
	      (ensure ?phi)
	      (setup-goals ?phi))))))

(defmac auto-show-light
  (show-internal ?phi)
  (not (eq ?phi '*goal*))
  (show-sequent ?phi
    (first (ensure ?phi)
	   (match-mark! ?phi
	     (setup-goals ?phi)))))

(defmac show-external
  (show ?phi ?form . ?body)
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (lisp-when (setq *inference-limit* (+ *inferences* *inference-increment*))
    (lisp-bind *showing* '?phi
      (show-internal ?phi ?form . ?body))))

(defmac show-external-no-body
  (show ?phi)
  t
  (show ?phi (show (true))))

(defmac case-analysis
  (case-analysis)
  t
  (evaluate-proof
   (progn
     (goto-context *real-context*)
     (expand-case-analysis))))

(defun expand-case-analysis ()
  (if (and (unknown-case?)
	   (<= *inferences* *inference-limit*))
      (let ((?psi (unknown-case?)))
	`(ensure *goal*
	   (case-sequent
	     (sequent ((assume ,?psi))
	       (first (ensure *goal*)
		      (case-analysis)))
	     (sequent ((assume (not ,?psi)))
	       (first (ensure *goal*)
		      (case-analysis))))))
      '(ensure *goal*)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Failure expansion code:                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun nth-proof-variable (n)
  (if (< n 26)
      (read-from-string (format nil "~a" (code-char (+ n 65))))
      (read-from-string (format nil "N~a" (- n 25)))))

(defun new-proof-variable-not (&rest list)
  (let ((next 0)
	(var (nth-proof-variable 0)))
    (while (or (member var list) (definition-of var))
      (setq next (1+ next))
      (setq var (nth-proof-variable next)))
    var))

(defmacro mi (exp)
  `(macro-invert ,exp))

(defun true-p (exp)
  (eq :true (is-true? (ti exp))))

(defun expand-failure (goal)
  (selectmatch goal
    ((is ?c (intersection ?a ?b))
     `(progn ~%
	(show (is ,(mi ?c) ~% ,(mi ?a))) ~%
	(show (is ,(mi ?c) ~% ,(mi ?b)))))
    ((is ?c (apply (subset-operator) ?w))
     `(show (is (a-member-of ,(mi ?c)) ~% (a-member-of ,(mi ?w)))))
    ((is ?c (db-lambda-fun ?type (db-class-combinator ?comb)))
     (let* ((?new-var (new-proof-variable-not))
	    (?new-body (expression-db-subst ?comb ?new-var 1)))
       `(suppose-there-is ((,?new-var ,(mi ?type))) ~%
	  (show (at-most-one (,(mi ?c) ,?new-var))) ~%
	  (show (there-exists (,(mi ?c) ,?new-var))) ~%
	  (show (is (,(mi ?c) ,?new-var) ~% ,(mi ?new-body))))))
    ((is ?c (lambda-rel0 ?range-type))
     `(show (is (,(mi ?c)) ~% ,(mi ?range-type))))
    ((is ?c (apply (apply (rel-from-to) ?type) ?range-type))
     (let ((?new-var (new-proof-variable-not)))
       `(suppose-there-is ((,?new-var (a-member-of ,(mi ?type)))) ~%
	  (show (is (,(mi ?c) (a-domain-member-of ,(mi ?c))) ~% (a-member-of ,(mi ?range-type)))))))
    ((is ?c (apply (apply (partial-fun-from-to) ?type) ?range-type))
     (let ((?new-var (new-proof-variable-not)))
       `(suppose-there-is ((,?new-var (a-member-of ,(mi ?type)))) ~%
	  (show (at-most-one (,(mi ?c) ,?new-var))) ~%
	  (show (is (,(mi ?c) (a-domain-member-of ,(mi ?c))) ~% (a-member-of ,(mi ?range-type)))))))
    ((is ?c (apply (apply (total-rel-from-to) ?type) ?range-type))
     (let ((?new-var (new-proof-variable-not)))
       `(suppose-there-is ((,?new-var (a-member-of ,(mi ?type)))) ~%
	 (show (there-exists (,(mi ?c) ,?new-var))) ~%
	 (show (is (,(mi ?c) (a-domain-member-of ,(mi ?c))) ~% (a-member-of ,(mi ?range-type)))))))
    ((is ?c (db-some-such-that ?type (db-formula-combinator ?comb)))
     (let* ((?new-var (new-proof-variable-not))
	    (?subst (expression-db-subst ?comb ?new-var 1)))
       `(progn ~%
	  (show (is ,(mi ?c) ~% ,(mi ?type))) ~%
	  (suppose-there-is ((,(mi ?new-var) ,(mi ?c))) ~%
	    (show ,(mi ?subst))))))
    ((is ?c ?w)
     (let ((?x (new-proof-variable-not)))
       `(suppose-there-is ((,?x ,(mi ?c))) ~%
	  (show (is ,?x ,(mi ?w))))))
    ((not (implies ?phi (not ?psi)))
     `(progn ~%
	(show ,(mi ?phi)) ~%
	(show ,(mi ?psi))))
    ((implies ?phi ?psi)
     `(first ~%
	(show (not ,(mi ?phi))) ~%
	(suppose ,(mi ?phi) ~%
	  (show ,(mi ?psi)))))
    ((= ?phi ?psi)
     (if (eq (expression-category ?phi) 'formula)
	 `(lemma ~%
	   (suppose ,(mi ?phi) ~%
	     (show ,(mi ?psi))) ~%
	   (suppose ,(mi ?psi) ~%
	     (show ,(mi ?phi))))
	 (let ((?x1 ?phi)
	       (?x2 ?psi))
	   (cond ((and (or (true-p `(is ,?x1 (a-set)))
			   (true-p `(is ,?x2 (a-set))))
		       (true-p `(singleton ,?x1))
		       (true-p `(singleton ,?x2)))
		  `(progn ~%
		     (show (is ,(mi ?x1) (a-set))) ~%
		     (show (is ,(mi ?x2) (a-set))) ~%
		     (show (is (a-member-of ,(mi ?x1)) ~%
			       (a-member-of ,(mi ?x2)))) ~%
		     (show (is (a-member-of ,(mi ?x2)) ~%
			       (a-member-of ,(mi ?x1))))))
		 ((and (or (true-p `(is ,?x1 (a-thunk)))
			   (true-p `(is ,?x2 (a-thunk))))
		       (true-p `(singleton ,?x1))
		       (true-p `(singleton ,?x2)))
		  `(progn ~%
		     (show (is ,(mi ?x1) (a-thunk))) ~%
		     (show (is ,(mi ?x2) (a-thunk))) ~%
		     (show (is (,(mi ?x1)) (,(mi ?x2)))) ~%
		     (show (is (,(mi ?x2)) (,(mi ?x1))))))
		 (t
		   `(progn ~%
		      (show (is ,(mi ?x1) ~%
				,(mi ?x2))) ~%
		      (show (is ,(mi ?x2) ~%
				,(mi ?x1)))))))))
    ((db-forall ?type (db-formula-combinator ?comb))
     (let* ((?new-var (new-proof-variable-not))
	    (?new-phi (expression-db-subst ?comb ?new-var 1)))
       `(suppose-there-is ((,?new-var ,(mi ?type)))
	  (show ,(mi ?new-phi)))))
    ((at-most-one ?c)
     (let* ((?x1 (new-proof-variable-not))
	    (?x2 (new-proof-variable-not ?x1)))
       `(suppose-there-is ((,?x1 ,(mi ?c)) ~%
			   (,?x2 ,(mi ?c))) ~%
	  (show (= ,?x1 ,?x2)))))))




;========================================================================
;evaluation
;========================================================================


;show-using (show by meta-computation)

(defmac show-using-macro
  (show ?phi using (is ?term ?type) . ?body)
  (or (not (consp ?type))
      (not (symbolp (car ?type)))
      (null (fixpoint-definition-of (car ?phi))))
  (lisp-let ((?level (or (when (eq 'evaluation-level (first ?body))
			   (second ?body))
			 1)))
    (lisp-let ((?real-body (if (eq 'evaluation-level (first ?body))
			       (cddr ?body)
			       ?body)))
      (progn
	(ensure (is ?term ?type))
	(ensure (singleton ?term))
	(progn . ?real-body)
	(internal-evaluation ?level ?phi using (is ?term ?type))))))

(emacs-indent show-by-evaluation 1)


(defmac show-by-evaluation-is
  (show-by-evaluation (is ?term ?val) . ?body)
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (lisp-let ((?arg (new-proof-variable 'eval-arg)))
    (lisp-let ((?location *location*))
      (lisp-bind *evaluation-stack* (append *evaluation-stack* `((suppose-there-is ?arg ?term)))
	(lisp-let ((?ignore (progn (when *trace-evaluation*
				     (rprint `(suppose-there-is ?arg ?term))))))
	  (show (is ?term ?val)
	    (suppose-there-is ((?arg ?term))
	      ((hlps-tag show ?location) (is ?arg ?val) using (is ?arg ?term) . ?body))))))))

(defmac show-by-evaluation-=
  (show-by-evaluation (= ?term ?val) . ?body)
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (progn (show (singleton ?term))
	 (show (= ?term ?val) using (is ?term ?term) . ?body)))

;internal-evaluation proofs

(defvar *evaluation-stack* nil)

(defun eval-context ()
  (rprint (mapcar (lambda (exp) (cons (car exp)
				      (mapcar (lambda (subexp) (macro-invert (translate subexp)))
					      (cdr exp))))
		  *evaluation-stack*)))

(defvar *trace-evaluation* nil)

(def-proof-function (internal-evaluation ?level ?phi using (is ?term ?exp)) context
  (setq ?exp (translate ?exp))
  (when *trace-evaluation* (rprint `(eval ,(macro-invert ?exp))))
  (let ((*evaluation-stack* (append *evaluation-stack* `((eval ,?exp)))))
    (selectmatch ?exp
      ((if-builder ?psi (pair-classes ?x ?y))
       (cond ((obvious-sequent? context ?psi)
	      (evaluate-proof `(internal-evaluation ,?level ,?phi using (is ,?term ,?x))
			      context))
	     ((obvious-sequent? context `(not ,?psi))
	      (break)
	      (evaluate-proof `(internal-evaluation ,?level ,?phi using (is ,?term ,?y))
			      context))
	     (t
	      (let ((new-context (add-extension context `(match-mark! ,?psi))))
		(cond ((obvious-sequent? new-context ?psi)
		       (evaluate-proof `(internal-evaluation ,?level ,?phi using (is ,?term ,?x))
				       new-context))
		      ((obvious-sequent? new-context `(not ,?psi))
		       (evaluate-proof `(internal-evaluation ,?level ,?phi using (is ,?term ,?y))
				       new-context))
		      (t
		       (evaluate-proof `(first
					 (ensure ,?phi)
					 (show ,?phi
					   (eval-suppose ,?psi
					     (internal-evaluation ,?level ,?phi
								  using (is ,?term ,?x)))
					   (eval-suppose (not ,?psi)
					     (internal-evaluation ,?level ,?phi
								  using (is ,?term ,?y)))))
				       new-context)))))))
      ((apply :anything :anything)
       (eval-reduction ?level ?phi ?term ?exp context))
      ((amb ?x ?y)
       (evaluate-proof `(first
			 (ensure ,?phi)
			 (show ,?phi
			   (eval-suppose (is ,?term ,?x)
			     (internal-evaluation ,?level ,?phi
						  using (is ,?term ,?x)))
			   (eval-suppose (not (is ,?term ,?x))
			     (eval-suppose (is ,?term ,?y)
			       (internal-evaluation ,?level ,?phi
						    using (is ,?term ,?y))))))
		       context))
      ((when ?psi ?x)
       (unless (obvious-sequent? context ?psi)
	 (declare-ontic-bug "failure to maintain the existence invariant in evaluation --- show an ontic hacker"))
       (evaluate-proof `(internal-evaluation ,?level ,?phi using (is ,?term ,?x))
		       context))
      (:anything
       (ieval-base-case ?phi ?term ?exp context)))))

(defun eval-reduction (?level ?phi ?term ?exp context)
  (if (< ?level 1)
      (ieval-base-case ?phi ?term ?exp context)
      (mvlet (((?exp2 new-context) (ontic-eval ?term ?exp context)))
	(if (equal ?exp2 ?exp)
	    (ieval-base-case ?phi ?term ?exp context)
	    (let ((*evaluation-stack* *evaluation-stack*))
	      (evaluate-proof `(first
				(ensure ,?phi)
				(show ,?phi
				  (axiom (is ,?term ,?exp2))
				  (internal-evaluation ,(1- ?level) ,?phi using (is ,?term ,?exp2))))
			      new-context))))))

(defun ieval-base-case (?phi ?term ?exp context)
  (evaluate-proof `(first
		    (ensure ,?phi)
		    (hl-write-as (,?term ,?exp)
		      (show ,?phi)))
		  context))

(defmac eval-suppose-macro
  (eval-suppose ?psi . ?body)
  t
  (lisp-let ((?ignore (progn (when *trace-evaluation*
			       (rprint `(suppose ,(macro-invert (translate ?psi))))))))
    (lisp-bind *evaluation-stack* (append *evaluation-stack* '((suppose ?psi)))
      (suppose ?psi . ?body))))

(defun ontic-eval (eval-arg exp context)
  (let ((scontext context))
    (labels ((ontic-eval-2 (exp continuation)
	       (selectmatch exp
		 ((apply (apply (cons-function) ?x) ?y)
		  (ontic-eval-2 ?x (lambda (?x-val)
				     (ontic-eval-2 ?y (lambda (?y-val)
							`(apply (apply (cons-function) ,?x-val) ,?y-val))))))
		 ((apply (car-function) ?x)
		    (ontic-eval-2 ?x (lambda (?x-val)
				       (selectmatch ?x-val
					 ((apply (apply (cons-function) ?car) :anything)
					  (raise-conditionals ?car continuation))
					 (:anything
					  (funcall continuation `(apply (car-function) ,?x-val)))))))
		 ((apply (cdr-function) ?x)
		    (ontic-eval-2 ?x (lambda (?x-val)
				       (selectmatch ?x-val
					 ((apply (apply (cons-function) :anything) ?cdr)
					  (raise-conditionals ?cdr continuation))
					 (:anything
					  (funcall continuation `(apply (car-function) ,?x-val)))))))
		 ((apply (funcall0-operator) ?t)
		  (ontic-eval-2 ?t (lambda (?t-val)
				     (selectmatch ?t-val
				       ((lambda0 ?body)
					(raise-conditionals ?body continuation))
				       (:anything (funcall continuation `(funcall0 ?t-val)))))))
		 ((apply ?f ?arg)
		  (ontic-eval-2 ?f
		     (lambda (?f-val)
		       (selectmatch ?f-val
			 ((db-lambda ?t (db-class-combinator ?db-body))
			  (let* ((need-new-variable (and (not (symbolp ?arg))
							 (not (obvious-sequent? scontext `(at-most-one ,?arg)))))
				 (arg (if (not need-new-variable) ?arg (new-proof-variable 'ARG))))
			    (when need-new-variable
			      (setf scontext  (add-extension (add-extension scontext `(axiom (classify! ,?arg)))
							     `(let-be ,arg ,(translate `(not-general-constant ,?arg)))))
			      (setf *evaluation-stack* (append *evaluation-stack* `((consider ,arg ,?arg
											      such-that
											      (is ,eval-arg (,?f ,arg))))))
			      (when *trace-evaluation* (rprint `(consider ,arg ,(macro-invert (translate ?arg))
								 such-that ,(macro-invert
									     (translate `(is ,eval-arg (,?f ,arg))))))))
			    (if (not (obvious-sequent? scontext `(is ,arg ,?t)))
				(ontic-error (format nil "type check failure during evaluation: failure to show ~s"
						     (macro-invert `(is ,?arg ,?t))))
				(raise-conditionals (expression-db-subst ?db-body arg 1)
						    continuation))))
			 (:anything (funcall continuation `(apply ,?f-val ,?arg)))))))
		 (:anything
		  (if (symbolp exp)
		      (raise-conditionals (or (fixpoint-definition-of exp)
					      (definition-of exp)
					      exp)
					  continuation)
		      (raise-conditionals exp continuation))))))
      (values (ontic-eval-2 exp (lambda (x) x))
	      scontext))))

(defun raise-conditionals (exp continuation)
  (selectmatch exp
    ((if-builder ?phi (pair-classes ?x ?y))
     `(if-builder ,?phi (pair-classes ,(raise-conditionals ?x continuation) ,(raise-conditionals ?y continuation))))
    ((amb ?x ?y)
     `(amb ,(raise-conditionals ?x continuation) ,(raise-conditionals ?y continuation)))
    ((when ?phi ?x)
     `(when ,?phi ,(raise-conditionals ?x continuation)))
    (:anything
     (funcall continuation exp))))

(defun ontic-is? (?x ?y context)
  (let ((x-is-y `(is ,?x ,?y)))
    (or (obvious-sequent? context x-is-y)
	(obvious-sequent? (add-extension context `(match-mark! ,?x))
			  x-is-y)
	(obvious-sequent? (add-extension (add-extension context `(match-mark! ,?x))
						`(match-mark! ,?y))
			  x-is-y))))

(defun ontic-eval-formula (?phi context)
  (cond ((obvious-sequent? context ?phi)
	 :true)
	((obvious-sequent? context `(not ,?phi))
	 :false)
	(t
	 (let ((next-context (add-extension context `(match-mark! ,?phi))))
	   (cond ((obvious-sequent? next-context ?phi)
		  :true)
		 ((obvious-sequent? next-context `(not ,?phi))
		  :false)
		 (t :unknown))))))



;========================================================================
;induction
;========================================================================

(defmac recursive-show-using
  (show ?phi using (is ?term (?f . ?args)) . ?body)
  (and (symbolp ?f)
       (fixpoint-definition-of ?f))
  (induction-proof let-be nil show ?phi ?term ?f ?args ?body))

(defmac let-be-is-show-using
  (let-be ?bindings ((hlps-tag show ?loc) ?phi using (is ?term (?f . ?args)) . ?body))
  (and (symbolp ?f)
       (fixpoint-definition-of ?f))
  (induction-proof let-be ?bindings (hlps-tag show ?loc) ?phi ?term ?f ?args ?body))

(defmac suppose-there-is-show-using
  (suppose-there-is ?bindings ((hlps-tag show ?loc) ?phi using (is ?term (?f . ?args)) . ?body))
  (and (symbolp ?f)
       (fixpoint-definition-of ?f))
  (induction-proof suppose-there-is ?bindings (hlps-tag show ?loc) ?phi ?term ?f ?args ?body))

(defextender induction-hypothesis (tag phi)
  (assume-fun phi))

(defmac induction-proof-macro
  (induction-proof ?binder ?bindings ?show-form ?phi ?term ?f ?args ?body)
  t
  (lisp-let ((?wf (create-name 'wishful ?f)))
    (lisp-let ((?wf-type (compute-wf-type ?f)))
      (lisp-let ((?new-f (sublis (acons ?f ?wf nil) (fixpoint-definition-of ?f))))
	(lisp-let ((?f-bindings (second (macro-invert ?new-f))))
	  (lisp-let ((?f-args (mapcar 'car ?f-bindings)))
	    (lisp-let ((?ind-hyp (compute-induction-hypothesis ?bindings ?phi ?term ?f ?args ?wf)))
	      (?binder ?bindings
		       (invisible-show (is ?term (?f . ?args)))
		       (invisible-show (singleton ?term))
		       (show ?phi
			 (let-be ((?wf (not-general-constant ?wf-type)))
			   (context-extension (induction-hypothesis ?wf ?ind-hyp))
			   (context-extension (:lisp (setf-undo (fixpoint-definition-of '?wf) '?new-f)))
			   (context-extension (assume (forall ?f-bindings (is (?wf . ?f-args) (?f . ?f-args)))))
			   (context-extension (assume (forall ?f-bindings (is (?wf . ?f-args) (?new-f . ?f-args)))))
			   (context-extension (assume (is ?term (?new-f . ?args))))
			   (context-extension (hl-write-as (?term (?new-f . ?args))))
			   (?show-form ?phi using (is ?term (?new-f . ?args)) . ?body)))))))))))

(defmacro show-induction-hypothesis (&optional tag)
  `(rprint
    (third (find-if (lambda (ext) (selectmatch ext ((induction-hypothesis ?tag :anything)
						    (or (null ',tag)
							(eq ?tag ',tag)))))
	    (reverse *context*)))))

(defmacro sih (&optional tag)
  `(show-induction-hypothesis ,tag))

(defun compute-wf-type (fun)
  (let ((fp (fixpoint-definition-of fun)))
    (if (eq (car fp) 'lambda0)
	`(lambda-rel0 ,(second fp))
	`(an-operator-from (the-set-of-all (a-domain-member-of ,fun))
			   to (the-set-of-all (a-range-element ,fun))))))

(defun compute-induction-hypothesis (?bindings ?phi ?term ?f ?args ?wf)
  (let ((default-hyp `(forall ,?bindings (implies (is ,?term (,?wf ,@?args)) ,?phi))))
    (if (not (symbolp ?term))
	default-hyp
	(let ((bindings (remove-if-not (lambda (binding) (eq (car binding) ?term))
				       ?bindings)))
	  (if (not (and (= (length bindings) 1)
			(equal (second (first bindings))
			       `(,?f . ,?args))))
	      default-hyp
	      (let ((new-bindings (mapcar (lambda (binding)
					    (if (eq (car binding) ?term)
						`(,?term (,?wf . ,?args))
						binding))
					  ?bindings)))
		`(forall ,new-bindings ,?phi)))))))
		   

;The following is a real crock.  It allows arbitrary context extensions to be inserted directly
;into proofs.
  
(defsequent extension-sequent
    ((sequent (?extension)
       (theorem (true))))
  ?extension)

(defmac extension-macro
  (context-extension ?ext)
  t
  (lisp-let ((?ext2 (untag ?ext)))
    (extension-sequent (sequent (?ext2) (show (true))))))

(defextender :lisp (form)
  (eval form))

;extending induction to work with show-by-evaluation

(defmac show-by-evaluation-is-2
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) (is ?term ?val) . ?body))
  (and (symbolp ?term) (not (and (consp ?body) (eq (car ?body) 'using))))
  (let-be ?bindings ((hlps-tag show ?loc) (is ?term ?val) using (is ?term ?term) . ?body)))

(defmac show-by-evaluation-is-2.5
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) (is ?term ?val) . ?body))
  (and (not (symbolp ?term))
       (not (and (consp ?body) (eq (car ?body) 'using))))
  (lisp-let ((?arg (new-proof-variable 'eval-arg)))
    (lisp-let ((?new-bindings (append ?bindings `((,?arg ,?term)))))
      (lisp-bind *evaluation-stack* (append *evaluation-stack* `((suppose-there-is ?arg ?term)))
	(lisp-let ((?ignore (progn (when *trace-evaluation*
				     (rprint `(suppose-there-is ?arg ?term))))))
	  (suppose-there-is ?new-bindings
	    ((hlps-tag show ?loc) (is ?arg ?val) using (is ?arg ?term) . ?body)))))))

(defmac show-by-evaluation-is-3
  (suppose-there-is ?bindings ((hlps-tag show-by-evaluation ?loc) (is ?term ?val) . ?body))
  (and (symbolp ?term)
       (not (and (consp ?body) (eq (car ?body) 'using))))
  (suppose-there-is ?bindings ((hlps-tag show ?loc) (is ?term ?val) using (is ?term ?term) . ?body)))

(defmac show-by-evaluation-is-2.5
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) (is ?term ?val) . ?body))
  (and (not (symbolp ?term))
       (not (and (consp ?body) (eq (car ?body) 'using))))
  (lisp-let ((?arg (new-proof-variable 'eval-arg)))
    (lisp-let ((?new-bindings (append ?bindings `((,?arg ,?term)))))
      (lisp-bind *evaluation-stack* (append *evaluation-stack* `((suppose-there-is ?arg ?term)))
	(lisp-let ((?ignore (progn (when *trace-evaluation*
				     (rprint `(suppose-there-is ?arg ?term))))))
	  (suppose-there-is ?new-bindings
	    ((hlps-tag show ?loc) (is ?arg ?val) using (is ?arg ?term) . ?body)))))))

(defmac show-by-evaluation-is-3.5
  (suppose-there-is ?bindings ((hlps-tag show-by-evaluation ?loc) (is ?term ?val) . ?body))
  (and (not (symbolp ?term))
       (not (and (consp ?body) (eq (car ?body) 'using))))
  (lisp-let ((?arg (new-proof-variable 'eval-arg)))
    (lisp-let ((?new-bindings (append ?bindings `((,?arg ,?term)))))
      (lisp-bind *evaluation-stack* (append *evaluation-stack* `((suppose-there-is ?arg ?term)))
	(lisp-let ((?ignore (progn (when *trace-evaluation*
				     (rprint `(suppose-there-is ?arg ?term))))))
	  (suppose-there-is ?new-bindings
	    ((hlps-tag show ?loc) (is ?arg ?val) using (is ?arg ?term) . ?body)))))))

(defmac show-by-evaluation-=-2
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) (= ?term ?val) . ?body))
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (let-be ?bindings
    ((hlps-tag show ?loc) (= ?term ?val) using (is ?term ?term) . ?body)))

(defmac show-by-evaluation-=-3
  (suppose-there-is ?bindings ((hlps-tag show-by-evaluation ?loc) (= ?term ?val) . ?body))
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (suppose-there-is ?bindings
	 ((hlps-tag show ?loc) (= ?term ?val) using (is ?term ?term) . ?body)))

(defmac show-by-evaluation-using
  (show-by-evaluation ?phi using ?psi . ?body)
  t
  (show ?phi using ?psi
    (show-by-evaluation ?phi . ?body)))

(defmac show-by-evaluation-using-with-let-be
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) ?phi using ?psi . ?body))
  t
  (let-be ?bindings
    ((hlps-tag show ?loc) ?phi using ?psi
	  (show-by-evaluation ?phi . ?body))))

(defmac show-by-evaluation-using-with-suppose-there-is
  (suppose-there-is ?bindings ((hlps-tag show-by-evaluation ?loc) ?phi using ?psi . ?body))
  t
  (suppose-there-is ?bindings
    ((hlps-tag show ?loc) ?phi using ?psi
	  (show-by-evaluation ?phi . ?body))))

;========================================================================
;existence evaluation
;========================================================================
(defmac show-by-evaluation-there-exists
  (show-by-evaluation (there-exists ?term) . ?body)
  (not (and (consp ?body) (eq (car ?body) 'using)))
  (lisp-let ((?level (or (when (eq 'evaluation-level (first ?body))
			   (second ?body))
			 1)))
    (lisp-let ((?real-body (if (eq 'evaluation-level (first ?body))
			       (cddr ?body)
			       ?body)))
      (progn
	(show (there-exists ?term)
	  (progn . ?real-body)
	  (eval-for-existence ?level ?term))))))

(def-proof-function (eval-for-existence ?level ?exp) context
  (setq ?exp (translate ?exp))
  (when *trace-evaluation* (rprint `(existence-eval ,(macro-invert ?exp))))
  (let ((*evaluation-stack* (append *evaluation-stack* `((existence-eval ,?exp)))))
    (selectmatch ?exp
      ((if-builder ?psi (pair-classes ?x ?y))
       (cond ((obvious-sequent? context ?psi)
	      (evaluate-proof `(eval-for-existence ,?level ,?x)
			      context))
	     ((obvious-sequent? context `(not ,?psi))
	      (evaluate-proof `(eval-for-existence ,?level ,?y)
			      context))
	     (t
	      (let ((new-context (add-extension context `(match-mark! ,?psi))))
		(cond ((obvious-sequent? new-context ?psi)
		       (evaluate-proof `(eval-for-existence ,?level ,?x)
				       new-context))
		      ((obvious-sequent? new-context `(not ,?psi))
		       (evaluate-proof `(eval-for-existence ,?level ,?y)
				       new-context))
		      (t
		       (evaluate-proof `(first
					 (ensure (there-exists ,?exp))
					 (show (there-exists ,?exp)
					   (eval-suppose ,?psi
					     (eval-for-existence ,?level ,?x))
					   (eval-suppose (not ,?psi)
					     (eval-for-existence ,?level ,?y))))
				       new-context)))))))
      ((apply ?f ?arg)
       (evaluate-proof `(first
			 (show (there-exists ,?f))
			 (eval-for-existence ,?level ,?f))
		       context)
       (evaluate-proof `(first
			 (show (there-exists ,?arg))
			 (eval-for-existence ,?level ,?arg))
		       context)
       (if (< ?level 1)
	   (existence-base-case ?exp context)
	   (mvlet (((?exp2 new-context)
		    (ontic-eval nil ?exp (add-extension (add-extension context
								       `(theorem (there-exists ,?f)))
							`(theorem (there-exists ,?arg))))))
	     (if (equal ?exp2 ?exp)
		 (existence-base-case ?exp context)
		 (let ((*evaluation-stack* *evaluation-stack*))
		   (evaluate-proof `(first
				     (ensure (there-exists ,?exp2))
				     (show (there-exists ,?exp2)
				       (eval-for-existence ,(1- ?level) ,?exp2)))
				   new-context))))))
      ((amb ?x ?y)
       (evaluate-proof `(first
			 (ensure (there-exists ,?exp))
			 (show (there-exists ,?exp)
			   (first (eval-for-existence ,?level ,?x)
				  (eval-suppose (not (there-exists ,?x))
				    (eval-for-existence ,?level ,?y)))))
		       context))
      ((when ?psi ?x)
       (evaluate-proof
	`(progn
	  (show ,?psi)
	  (eval-for-existence ,?level ,?x))
	context))
      (:anything
       (existence-base-case ?exp context)))))

(defun existence-base-case (?exp context)
  (evaluate-proof `(show (there-exists ,?exp))
		  context))



;========================================================================
;at-most-one inference
;========================================================================

(defmac show-by-evaluation-at-most-one
  (show-by-evaluation (at-most-one ?term) . ?body)
  (and (not (and (consp ?term) (symbolp (car ?term)) (fixpoint-definition-of (car ?term))))
       (not (and (consp ?body) (eq (car ?body) 'using))))
  (show-by-evaluation-at-most-one ?term . ?body))

(defmac show-by-evaluation-at-most-one-internal
  (show-by-evaluation-at-most-one ?term . ?body)
  t
  (lisp-let ((?level (or (when (eq 'evaluation-level (first ?body))
			   (second ?body))
			 1)))
    (lisp-let ((?real-body (if (eq 'evaluation-level (first ?body))
			       (cddr ?body)
			       ?body)))
      (progn
	(show (at-most-one ?term)
	  (progn . ?real-body)
	  (eval-for-at-most-one ?level ?term))))))

(def-proof-function (eval-for-at-most-one ?level ?exp) context
  (setq ?exp (translate ?exp))
  (when *trace-evaluation* (rprint `(at-most-one-eval ,(macro-invert ?exp))))
  (let ((*evaluation-stack* (append *evaluation-stack* `((at-most-one-eval ,?exp)))))
    (selectmatch ?exp
      ((if-builder ?psi (pair-classes ?x ?y))
       (cond ((obvious-sequent? context ?psi)
	      (evaluate-proof `(eval-for-at-most-one ,?level ,?x)
			      context))
	     ((obvious-sequent? context `(not ,?psi))
	      (evaluate-proof `(eval-for-at-most-one ,?level ,?y)
			      context))
	     (t
	      (let ((new-context (add-extension context `(match-mark! ,?psi))))
		(cond ((obvious-sequent? new-context ?psi)
		       (evaluate-proof `(eval-for-at-most-one ,?level ,?x)
				       new-context))
		      ((obvious-sequent? new-context `(not ,?psi))
		       (evaluate-proof `(eval-for-at-most-one ,?level ,?y)
				       new-context))
		      (t
		       (evaluate-proof `(first
					 (ensure (at-most-one ,?exp))
					 (show (at-most-one ,?exp)
					   (eval-suppose ,?psi
					     (eval-for-at-most-one ,?level ,?x))
					   (eval-suppose (not ,?psi)
					     (eval-for-at-most-one ,?level ,?y))))
				       new-context)))))))
      ((apply :anything :anything)
       (if (< ?level 1)
	   (at-most-one-base-case ?exp context)
	   (mvlet (((?exp2 new-context)
		    (ontic-eval nil ?exp context)))
	     (if (equal ?exp2 ?exp)
		 (existence-base-case ?exp context)
		 (let ((*evaluation-stack* *evaluation-stack*))
		   (evaluate-proof `(first
				     (ensure (at-most-one ,?exp2))
				     (show (at-most-one ,?exp2)
				       (eval-for-at-most-one ,(1- ?level) ,?exp2)))
				   new-context))))))
      ((amb ?x ?y)
       (evaluate-proof `(show (at-most-one ,?exp)
			 (proof-cond
			  ((= ?x ?y)
			   (first
			    (ensure (a-most-one ?x))
			    (show (at-most-one ,?exp)
			      (eval-for-at-most-one ,?level ,?x))))
			  ((not (there-exists ,?x))
			   (first
			    (ensure (a-most-one ?y))
			    (show (at-most-one ,?exp)
			      (eval-for-at-most-one ,?level ,?y))))
			  ((not (there-exists ,?y))
			   (first
			    (ensure (a-most-one ?x))
			    (show (at-most-one ,?exp)
			      (eval-for-at-most-one ,?level ,?x))))
			 ((true)
			  (suppose (there-exists ,?x)
			    (show (not (there-exists ,?y)))
			    (show (at-most-one ,?exp)
			      (eval-for-at-most-one ,?x)))			    
			  (suppose (not (there-exists ,?x))
			    (show (at-most-one ,?exp)
			      (eval-for-at-most-one ,?y))))))))
      ((when ?psi ?x)
       (evaluate-proof
	`(show (at-most-one ,?exp)
	  (suppose ,?psi
	    (first (ensure (at-most-one ,?x))
		   (eval-for-at-most-one ,?level ,?x))))
	context))
      (:anything
       (existence-base-case ?exp context)))))

(defun at-most-one-base-case (?exp context)
  (evaluate-proof `(show (at-most-one ,?exp))
		  context))




;========================================================================
;induction proofs for show-at-most-one
;========================================================================

(defmac show-by-evaluation-at-most-one-recursive
  (show-by-evaluation (at-most-one ?term) . ?body)
  (and (not (and (consp ?body) (eq (car ?body) 'using)))
       (consp ?term)
       (symbolp (car ?term))
       (fixpoint-definition-of (car ?term)))
  (lisp-let ((?loc *location*))
    (at-most-one-induction-proof let-be nil ?loc ?term ?body)))

(defmac show-by-evaluation-at-most-one-2
  (let-be ?bindings ((hlps-tag show-by-evaluation ?loc) (at-most-one ?term) . ?body))
  (and (not (and (consp ?body) (eq (car ?body) 'using)))
       (consp ?term)
       (symbolp (car ?term))
       (fixpoint-definition-of (car ?term)))
  (at-most-one-induction-proof let-be ?bindings ?loc ?term ?body))

(defmac show-by-evaluation-at-most-one-3
  (suppose-there-is ?bindings ((hlps-tag show-by-evaluation ?loc) (at-most-one ?term) . ?body))
  (and (not (and (consp ?body) (eq (car ?body) 'using)))
       (consp ?term)
       (symbolp (car ?term))
       (fixpoint-definition-of (car ?term)))
  (at-most-one-induction-proof suppose-there-is ?bindings ?loc ?term ?body))

(defmac show-at-most-one-induction-macro
  (at-most-one-induction-proof ?binder ?bindings ?loc ?term ?body)
  t
  (lisp-let ((?f (car ?term))
	     (?f-args (cdr ?term))
	     (?wf (create-name 'wishful ?f))
	     (?wf-internal (create-name 'wishful 'internal ?f))
	     (?wf-type (compute-wf-type ?f))
	     (?new-f (sublis (acons ?f ?wf nil) (fixpoint-definition-of ?f)))
	     (?new-term (cons ?new-f ?f-args))
	     (?mi-new-f (macro-invert ?new-f))
	     (?f-bindings (second ?mi-new-f)))
    (?binder ?bindings
	     ((hlps-tag show ?loc) (at-most-one ?term)
	      (context-extension (define ?wf-internal (not-general-constant ?wf-type)))
	      (context-extension (define ?wf (lambda ?f-bindings (?wf-internal . ?f-args))))
	      (context-extension (assume (forall ?bindings (at-most-one (?wf . ?f-args)))))
	      (context-extension (:lisp (setf-undo (fixpoint-definition-of '?wf) '?new-f)))
	      (context-extension (assume (forall ?f-bindings (is (?wf . ?f-args) (?f . ?f-args)))))
	      (context-extension (assume (forall ?f-bindings (is (?wf . ?f-args) (?new-f . ?f-args)))))
	      (show-by-evaluation (at-most-one ?new-term) . ?body)
	      (axiom (at-most-one ?term))))))

(defun pretty-print-expression (exp)
  (let ((*print-pretty* t)
	(*print-length* nil)
	(*print-level* nil))
    (format nil "~a" exp)))

(defvar some-vars '(x y z s t j k l m n))
(defun nth-proof-variable (n)
  (or (nth n some-vars)
      (progn
	(setf n (- n (length some-vars)))
	(if (< n 26)
	    (read-from-string (format nil "~a" (code-char (+ n 65))))
	    (read-from-string (format nil "N~a" (- n 25)))))))

(defun make-proof (evaluation-extenders &optional avoid-vars)
  (when evaluation-extenders
    (selectmatch (car evaluation-extenders)
      ((eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((existence-eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((at-most-one-eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((suppose ?phi) (format nil "(suppose ~a ~% ~a)"
			      (pretty-print-expression (macro-invert (translate ?phi)))
			      (make-proof (cdr evaluation-extenders) avoid-vars)))
      ((consider ?arg ?exp such-that ?phi)
       (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
	 (format nil "(consider ((~a ~a)) ~% such-that ~a ~% ~a)"
		 new-arg
		 (pretty-print-expression (macro-invert (translate ?exp)))
		 (pretty-print-expression (macro-invert (translate (sublis `((,?arg . ,new-arg)) ?phi))))
		 (make-proof (sublis `((,?arg . ,new-arg))
				     (cdr evaluation-extenders))
			     (cons new-arg avoid-vars)))))
      ((let-be ?arg ?exp) (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
			    (format nil "(let-be ((~a ~a)) ~% ~a)"
				    new-arg (pretty-print-expression (macro-invert (translate ?exp)))
				    (make-proof (sublis `((,?arg . ,new-arg))
							(cdr evaluation-extenders))
						(cons new-arg avoid-vars) ))))
      ((suppose-there-is ?arg ?exp) (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
				      (format nil "(suppose-there-is ((~a ~a)) ~% ~a)"
					      new-arg (pretty-print-expression (macro-invert (translate ?exp)))
					      (make-proof (sublis `((,?arg . ,new-arg))
								  (cdr evaluation-extenders))
							  (cons new-arg avoid-vars)))))
      ((show ?phi)
       (format nil "(show ~a)" (pretty-print-expression ?phi)))
      (:anything (format nil "(unknown-evaluation-extender ~a ~% ~a)"
			 (car evaluation-extenders)
			 (make-proof (cdr evaluation-extenders) avoid-vars))))))

