;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)


;********************************************************************
;Specification
;********************************************************************

;A language is defined by a set of categories and phrase constructors.
;Each phrase constructor has argument categories and an output category.

;The following defines the phrase constructors a, apply, P, and R
;
;  (declare-categories formula class)
;
;  (bnf (class (a) (apply class class))
;       (formula (P class) (R class class)))


;``formula'' is a distinguished category.  ``='' and ``not'' and ``is-true''
;are distinguished constructors and should not be defined by the user.

;This file defines the macros declare-categories and bnf.

(export '(declare-categories bnf))

;It also defines the functions

(export '(assert-true assert-false))

;(assert-true <formula-expression>)

;(assert-false <formula-expression>)

;(truth <formula-expression>) which returns :true, :false, or nil.


;For each constructor FOO, we construct the following

;(make-foo <non-terminal1> ... <non-terminal-n>)  --- this is equivalent to
;(cintern `(foo ,<non-terminal-1> ... ,<non-terminal-n>))

;For each monadic (one argument) predicate (formula constructor) pred we have
;the following functions.

;(assert-pred <non-terminal> <truth>) --- declares that pred has value <truth>
;on the given weak node.

;(pred? <non-terminal>)  --- returns :true, :false, or nil.

;(pred-internal <non-terminal>) --- this is the actual slot accessor and gives
;:true :false or nil

;the user-modifiable functions (notice-true-pred <obj>) and (notice-false-pred <obj>)
;These are used for inferene rule propagation.

;The user-modifiable function (notice-pred-phase1 <obj>)
;This is called after the assertion is done but before the above functions
;are called (hence, before propagation is done).

					;(add-true-demon-pred <non-terminal> <demon>)

;(add-false-demon-pred <non-terminal> <demon>)


;For each binary predicate rel we have the following.

;(assert-rel <non-terminal1> <non-terminal2>) --- declares that rel
;is true on the given weak node.

;(rel? <non-terminal1> <non-terminal2>)  --- returns :true or nil.

;(rel-forward <non-terminal-1>) --- returns a list of all weak nodes <non-terminal-2> such
;that (rel? <non-terminal-1> <non-terminal-2>) returns :true.

;(rel-forward <non-terminal-1>) --- returns a list of all weak nodes <non-terminal-2> such
;that (rel? <non-terminal-1> <non-terminal-2>) returns :true.

;(rel-forward-internal <nt>) --- this is the actual slot accessor

;(rel-backward <non-terminal-2>) --- returns a list of all weak nodes <non-terminal-1> such
;that (rel? <non-terminal-1> <non-terminal-2>) returns :true.

;(rel-backward-internal <nt>) --- this is the actual slot accessor.

;The user-modifiable function (notice-rel <obj1> <obj2>)
;This is used for inference rule propagation.

;The user-modifiable function (notice-rel-phase1 <obj1> <obj2>)
;This is called after the assertion is made (the data structures are updated)
;but before the above function is called (and hence before propagation is done).

;(add-forward-demon-rel <non-terminal> <demon>)

;(add-backward-demon-rel <non-terminal> <demon>)



;********************************************************************
;Implementation
;********************************************************************

(defvar *categories* '(basic-object))

(property-macro parent-category)
(defun declare-category-fun (cat-name parent-category)
  (when (eq cat-name 'basic-object)
    (error "Cannot re-declare built-in category BASIC-OBJECT"))
  (check-category parent-category)
  (pushnew cat-name *categories*)
  (setf (parent-category cat-name) parent-category))

(defun check-category (category)
  (unless (member category *categories*)
    (error "undeclared category ~s" category)))

(defun sub-category? (cat1 cat2)
  (or (eq cat1 cat2)
      (let ((parent (parent-category cat1)))
	(and parent
	     (sub-category? parent cat2)))))

(defvar *constructors* nil)

(defsetf output-category (symbol) (category)
  `(setf (get ,symbol 'output-category) ,category))

(defmacro output-category (constructor)
  (if (numberp constructor)
      'ontic-number
      `(if (numberp ,constructor)
	'ontic-number
	(get ,constructor 'output-category))))

(defsetf argument-categories (symbol) (categories)
  `(setf (get ,symbol 'argument-categories) ,categories))

(defmacro argument-categories (constructor)
  (if (numberp constructor)
      nil
      `(if (numberp ,constructor)
	nil
	(get ,constructor 'argument-categories))))

(defmacro declare-constructor (const-name arg-categories output-category)
  `(eval-when (compile load eval)
     (check-category ,output-category)
     (mapc #'check-category ,arg-categories)
     (when (> (length ,arg-categories) 2)
       (error "phrase constructors can not accept more than two arguments"))
     (pushnew ,const-name *constructors*)
     (setf (argument-categories ,const-name) ,arg-categories)
     (setf (output-category ,const-name) ,output-category)))


;
(defmacro declare-categories (&rest cat-descriptors)
  `(eval-when (compile load eval)
     ,@(mapcan 'category-definitions  cat-descriptors)))

(defun category-definitions (cat-descriptor)
  (let ((cat-name (if (listp cat-descriptor) (first cat-descriptor) cat-descriptor))
	(parent-cat (if (listp cat-descriptor) (second cat-descriptor) 'anything)))
    `((declare-category-fun ',cat-name ',parent-cat)
      (defpiecestruct ,cat-name :include ,parent-cat)
      (defun ,(category-print-function cat-name) (stream object)
	(print-basic-object stream object)))))


(defmacro bnf (&body forms)
  `(eval-when (compile eval load)
     ,@(mapcan (lambda (cat-description)
		 (let ((cat (car cat-description))
		       (const-forms (cdr cat-description)))
		   (mapcan (lambda (const-form)
			     (constructor-definitions (car const-form) (cdr const-form) cat))
			   const-forms)))
	       forms)))

(defun constructor-definitions (const-name full-arg-categories output-category)
  (let ((arg-categories (mapcar (lambda (cat)
				  (if (consp cat)
				      (car cat)
				      cat))
				full-arg-categories)))
    (adjoin `(declare-constructor ',const-name ',arg-categories ',output-category)
	    (append (maker-checker-definitions const-name arg-categories output-category)
		    (when (eq output-category 'formula)
		      (cond ((= 1 (length arg-categories))
			     (predicate-definitions const-name arg-categories))
			    ((= 2 (length arg-categories))
			     (relation-definitions const-name full-arg-categories))))
		    (unless (eq const-name 'is-true)
		      (constructor-rules const-name arg-categories output-category))))))

(defun constructor-rules (const-name arg-cats output-cat)
  (when (and arg-cats (eq 'formula output-cat))
    (let* ((args (mapcar (lambda (cat)
			   (let ((arg (gentemp "?X-")))
			     (declare-variable-2 arg cat)
			     arg))
			 arg-cats))
	   (form `(,const-name ,@args))
	   (formvar (gentemp "?P-")))
      (declare-variable-2 formvar 'formula)
      `((declare-variables ,@(mapcar (lambda (arg cat) `(,cat ,arg))
				     args
				     arg-cats)
			   (formula ,formvar))
	(rule ,(combine-symbols const-name 'intern-rule-1)
	      ((= ,formvar ,form) ,form)
	  ,formvar)
	(rule ,(combine-symbols const-name 'intern-rule-2)
	      ((= ,formvar ,form) ,formvar)
	  ,form)
	,@(when (null (cdr args))
	    `((rule ,(combine-symbols const-name 'intern-rule-3)
		    ((= ,formvar ,form) ,(negation form))
		(not ,formvar))
	      (rule ,(combine-symbols const-name 'intern-rule-4)
		    ((= ,formvar ,form) (not ,formvar))
		,(negation form))))))))

(defvar *maker-table* (make-hash-table))

(defmacro constructor-maker (constructor)
  `(gethash ,constructor *maker-table*))

(defun maker-function (constructor)
  (if (numberp constructor)
      'make-ontic-number
      (constructor-maker constructor)))

(defvar *checker-table* (make-hash-table))

(defmacro constructor-checker (constructor)
  `(gethash ,constructor *checker-table*))

(defun checker-function (constructor)
  (if (numberp constructor)
      (lambda (args)
	(when args
	  (ontic-error "attempt to provide an argument to a number constructor")))
      (constructor-checker constructor)))

(defun create-name-internal (&rest symbols)
  (combine-symbol-list (append symbols '(internal)) (find-package 'ontic)))

(defun maker-checker-definitions  (const-name arg-categories output-category)
  (let ((args (mapcar (lambda (ignore) (gentemp "ARG-")) arg-categories)))
    `((defun ,(create-name 'make const-name) (,@args &key justification)
	(cintern (list ',const-name ,@args) :justification justification))
      (setf (constructor-maker ',const-name)
       ',(category-constructor-function output-category))
      (defun ,(create-name 'check const-name) (args)
	,(cond ((null arg-categories)
		'(null args))
	       ((eq const-name '=)
		'(unless (and (= (length args) 2)
			  (eq (type-of (first args))
			   (type-of (second args))))
		  (ontic-error (format nil "illegal arguments in ~s" (cons '= args)))))
	       (t
		`(unless
		  (and args
		   (,(category-predicate (first arg-categories)) (first args))
		   ,(if (null (second arg-categories))
			'(null (cdr args))
			`(and (cdr args)
			  (,(category-predicate (second arg-categories))
			   (second args))
			  (null (cddr args)))))
		  (ontic-error (format nil "illegal arguments in ~s" (cons ',const-name args)))))))
      (setf (constructor-checker ',const-name)
       ',(create-name 'check const-name)))))


(defun make-number (n)
  (hashlist n nil))

(defun arglist (categories)
  (when categories
    (if (null (cdr categories))
	`(arg1)
	`(arg1 arg2))))

(defun add-? (symbol)
  (intern (concatenate 'string (format nil "~s" symbol) "?")
	  *package*))

(defun predicate-definitions (pred-name arg-categories)
  (unless (eq pred-name 'not)
    `((defslot-bit ,(first arg-categories) ,(create-name-internal pred-name 'true-bit))
      (defslot-bit ,(first arg-categories) ,(create-name-internal pred-name 'false-bit))

      (defmacro ,(create-name-internal pred-name) (arg)
	`(let ((x ,arg))
	  (cond ((= (,',(create-name-internal pred-name 'true-bit) x) 1)
		 :true)
		((= (,',(create-name-internal pred-name 'false-bit) x) 1)
		 :false)
		(t
		 nil))))

      (defmacro ,(create-name-internal pred-name 'true) (arg)
	`(let ((x ,arg))
	   (eql (the bit (,',(create-name-internal pred-name 'true-bit) x)) 1)))
      (defmacro ,(create-name-internal pred-name 'false) (arg)
	`(let ((x ,arg))
	   (eql (the bit (,',(create-name-internal pred-name 'false-bit) x)) 1)))

      ;;HOG      (defslot ,(first arg-categories) ,(create-name-internal pred-name 'false-demons)) 
      ;;HOG      (defslot ,(first arg-categories) ,(create-name-internal pred-name 'true-demons))
      (usually-null-property ,(create-name-internal pred-name 'false-demons));;new-hog
      (usually-null-property ,(create-name-internal pred-name 'true-demons));;new-hog

      (defextendable ,(add-? pred-name) (arg)
	(run-queue)
	(,(create-name-internal pred-name) (uf-find arg)))

      ;;HOG and new-hog
      (defextendable ,(create-name 'add-true-demon pred-name) (obj demon)
	(if (eq :true (,(create-name-internal pred-name) obj))
	    (funcall demon)
	    (push-undo demon (,(create-name-internal pred-name 'true-demons) obj))))

      ;;HOG and new-hog
      (defextendable ,(create-name 'add-false-demon pred-name) (obj demon)
	(if (eq :false (,(create-name-internal pred-name) obj))
	    (funcall demon)
	    (push-undo demon (,(create-name-internal pred-name 'false-demons) obj))))

      (defextendable ,(create-name 'assert pred-name) (arg truth &key justification)
	(,(create-name-internal 'assert pred-name) (uf-find arg) truth :justification justification)
	(run-queue))

      (defextendable ,(create-name-internal 'assert pred-name) (arg truth &key justification)
	(unless (or (contradiction?)
		    (bogus-non-terminal? arg))
	  (let ((old-truth (,(create-name-internal pred-name) arg)))
	    (cond ((eq old-truth truth) nil)
		  ((null old-truth)
		   (when justification
		     (setf (,(create-name pred-name 'frame) arg)
			   (make-invocation-frame (,(create-name 'assert pred-name) arg truth)
			     justification)))
		   (let ((*quiescent?* nil))
		     (if (eq truth :true)
			 (setf-undo (,(create-name-internal pred-name 'true-bit) arg) 1)
			 (setf-undo (,(create-name-internal pred-name 'false-bit) arg) 1))
		     (,(create-name-internal 'propagate pred-name) arg truth)))
		  (t
		   (assert-contradiction
		    :justification
		    (make-justification
		     ":"
		     (list (make-invocation-frame (,(create-name 'assert pred-name) arg truth)
			     justification)
			   (gethash arg ,(create-name pred-name 'frame 'hash 'table))))))))))

      (defpiecefun ,(create-name 'notice-true pred-name) (obj))
      (defmergefun ,(create-name 'merged-notice-true pred-name) (?obj))
      (definterpfun ,(create-name 'interpreted-notice-true pred-name) (?obj))
      (defpiece (,(create-name 'notice-true pred-name) :merged-propagator) (obj)
	(,(create-name 'merged-notice-true pred-name) obj))
      (defpiece (,(create-name 'notice-true pred-name) :interpreted-propagator) (obj)
	(,(create-name 'interpreted-notice-true pred-name) obj))

      (defpiecefun ,(create-name 'notice-false pred-name) (obj))
      (defmergefun ,(create-name 'merged-notice-false pred-name) (?obj))
      (definterpfun ,(create-name 'interpreted-notice-false pred-name) (?obj))
      (defpiece (,(create-name 'notice-false pred-name) :merged-propagator) (obj)
	(,(create-name 'merged-notice-false pred-name) obj))
      (defpiece (,(create-name 'notice-false pred-name) :interpreted-propagator) (obj)
	(,(create-name 'interpreted-notice-false pred-name) obj))

      (defpiecefun ,(create-name 'notice pred-name 'phase1) (obj))

      (defextendable ,(create-name-internal 'propagate pred-name) (arg truth)
	(,(create-name 'notice pred-name 'phase1) arg)
	(notice-monadic-inference ',pred-name arg)
	(cond ((eq truth :true)
	       (,(create-name 'notice-true pred-name) arg)
	       ;;HOG and new-hog
;;	       (mapc #'funcall (,(create-name-internal pred-name 'true-demons) arg))
	       ;; Clear out the demons so that they don't get rerun if this node is equated.
;;	       (setf-undo (,(create-name-internal pred-name 'true-demons) arg) nil)
	       )

	      (t
	       (,(create-name 'notice-false pred-name) arg)
	       ;;HOG and new-hog	       
;;	       (mapc #'funcall (,(create-name-internal pred-name 'false-demons) arg))
	       ;; Clear out the demons so that they don't get rerun if this node is equated.
;;	       (setf-undo (,(create-name-internal pred-name 'false-demons) arg) nil)
	       )))

      (hash-table-macro ,(create-name pred-name 'frame))

      (defpiece (ontic-init-phase0 ,(create-name 'clear pred-name 'frame)) ()
	  (,(create-name 'clear-all pred-name 'frame))))))

(defun relation-definitions (pred-name full-arg-categories)
  (unless (eq pred-name '=)
  (let ((arg-categories (mapcar (lambda (cat)
				  (if (consp cat)
				      (car cat)
				      cat))
				full-arg-categories)))
    
    `(,(if (consp (first full-arg-categories))
	   `(usually-null-property ,(create-name-internal pred-name 'forward))
	   `(defslot ,(first arg-categories) ,(create-name-internal pred-name 'forward)))

 ;;HOG      (defslot ,(first arg-categories) ,(create-name-internal pred-name 'forward-demons))
      (usually-null-property ,(create-name-internal pred-name 'forward-demons))  ;;new-hog

      ,(if (consp (second full-arg-categories))
	   `(usually-null-property ,(create-name-internal pred-name 'backward))
	   `(defslot ,(second arg-categories) ,(create-name-internal pred-name 'backward)))

;HOG      (defslot ,(second arg-categories) ,(create-name-internal pred-name 'backward-demons))
      (usually-null-property ,(create-name-internal pred-name 'backward-demons))  ;;new-hog
      
      (defextendable ,(add-? pred-name) (arg1 arg2)
	(run-queue)
	(member (uf-find arg2)
		(,(create-name-internal pred-name 'forward) (uf-find arg1))))

      (defextendable ,(create-name pred-name 'forward) (arg1)
	(run-queue)
	(,(create-name-internal pred-name 'forward) (uf-find arg1)))

      (defextendable ,(create-name pred-name 'backward) (arg2)
	(run-queue)
	(,(create-name-internal pred-name 'backward) (uf-find arg2)))

      (defextendable ,(create-name 'assert pred-name) (arg1 arg2 &key justification)
	(,(create-name-internal 'assert pred-name) (uf-find arg1) (uf-find arg2)
	 :justification justification)
	(run-queue))

      (defextendable ,(create-name-internal 'assert pred-name) (arg1 arg2 &key justification)
	(unless (or (contradiction?)
		    (bogus-non-terminal? arg1)
		    (bogus-non-terminal? arg2)
		    (member arg2 (,(create-name-internal pred-name 'forward) arg1)))
	  (let ((cell1 (cons arg1 (,(create-name-internal pred-name 'backward) arg2)))
		(cell2 (cons arg2 (,(create-name-internal pred-name 'forward) arg1))))
	    (setf-undo (,(create-name-internal pred-name 'forward) arg1) cell2)
	    (setf-undo (,(create-name-internal pred-name 'backward) arg2) cell1))
	  (when justification
	    (setf-undo (,(create-name pred-name 'frame) arg1 arg2)
		       (make-invocation-frame (,(create-name 'assert pred-name) arg1 arg2)
			 justification)))
	  (let ((*quiescent?* nil))
	    (,(create-name-internal 'propagate pred-name) arg1 arg2))))

      ;;HOG and new-hog
      (defextendable ,(create-name 'add-forward-demon pred-name) (arg1 demon)
	(push-undo demon (,(create-name-internal pred-name 'forward-demons) arg1))
	(dolist (arg2 (,(create-name-internal pred-name 'forward)
		       arg1))
	  (funcall demon arg2)))

      ;;HOG and new-hog
      (defextendable ,(create-name 'add-backward-demon pred-name) (arg2 demon)
	(push-undo demon (,(create-name-internal pred-name 'backward-demons) arg2))
	(dolist (arg1 (,(create-name-internal pred-name 'backward) arg2))
	  (funcall demon arg1)))

      (defpiecefun ,(create-name 'notice pred-name) (obj1 obj2))

      (defmergefun ,(create-name 'merged-notice pred-name) (?obj1 ?obj2))
      (definterpfun ,(create-name 'interpreted-notice pred-name) (?obj1 ?obj2))
      (defpiece (,(create-name 'notice pred-name) :merged-propagator) (obj1 obj2)
	(,(create-name 'merged-notice pred-name) obj1 obj2))
      (defpiece (,(create-name 'notice pred-name) :interpreted-propagator) (obj1 obj2)
	(,(create-name 'interpreted-notice pred-name) obj1 obj2))

      (defpiecefun ,(create-name 'notice pred-name 'phase1) (obj1 obj2))

      (defextendable ,(create-name-internal 'propagate pred-name) (arg1 arg2)
	(notice-binary-inference ',pred-name arg1 arg2)
	(,(create-name 'notice pred-name 'phase1) arg1 arg2)
	(,(create-name 'notice pred-name) arg1 arg2)

	;;HOG and new-hog
	(dolist (demon (,(create-name-internal pred-name 'forward-demons)
			arg1))
	  (funcall demon arg2))
	;;HOG and new-hog
	(dolist (demon (,(create-name-internal pred-name 'backward-demons)
			arg2))
	  (funcall demon arg1))
	)

      
      (defvar ,(create-name pred-name 'frame 'table) (make-hash-table :test #'equal))
      
      (defmacro ,(create-name pred-name 'frame) (arg1 arg2)
	`(gethash (cons ,arg1 ,arg2) ,',(create-name pred-name 'frame 'table)))

      (defpiece (ontic-init-phase0 ,(create-name 'clear pred-name 'frame 'table)) ()
	(clrhash ,(create-name pred-name 'frame 'table)))))))

(defun declare-bnf-piece (piece-name)
  (setf (get piece-name 'bnf-piece) t))

(defun bnf-piece? (piece-name)
  (get piece-name 'bnf-piece))

(defun assert-true (formula &key justification)
  (assert-is-true (cintern formula) :true :justification justification))

(defun assert-false (formula &key justification)
  (assert-is-true< (cintern formula) :false :justification justification))

(defpiece (notice-equate-phase1 :check-types) (dyer survivor)
  (unless (eq (type-of dyer) (type-of survivor))
    (error "Equate of expressions of inconsistent syntactic category, or of two numbers: ~
              ~s ~s" dyer survivor)))



;********************************************************************
;Translation
;********************************************************************


(defun tintern (expression &key justification)
  (cintern (translate expression) :justification justification))

(defun ti (exp &key justification) (tintern exp :justification justification))

;Assumptions are different from axioms --- assumptions interact
;with the controls on universal generalization.

(property-macro translator)

(defvar *uncurriable-operators* '(cons))  ;; binary translator operators

;; The following two variables are used for controlling translate.
;; They are global variables instead of keyword arguments because
;; the recursive control flow of translate passes through
;; deftranslator forms, each of which would have to be changed.

(defvar *translate-expand-definitions* nil)

(defvar *currently-translating* nil)

(property-macro expansion-mark)

(defun translate-expand (expression)
  (let ((*translate-expand-definitions* t))
    (translate expression)))

(property-macro fixpoint-definition-of)

(defun translate (expression)
  (cond ((numberp expression)
	 (translate `(quote ,expression)))
	((and *translate-expand-definitions*
	      (symbolp expression)
	      (expansion-mark expression))
	 (let ((fd (fixpoint-definition-of expression))
	       (*translate-expand-definitions* nil))
	   (if (and fd (not (member expression *currently-translating*)))
	       (let ((*currently-translating* (cons expression
						    *currently-translating*)))
		 (translate fd))
	       (translate (definition-of expression)))))
	((atom expression) expression)
	((and (consp (car expression))
	      (member (caar expression) *uncurriable-operators*)
	      (= (length (car expression)) 2))  ;;car of expression needs an arg
	 (translate (append (car expression) (cdr expression))))
	((and (symbolp (car expression))
	      (translator (car expression)))
	 (apply (translator (car expression))
		(cdr expression)))
	((member (car expression) *constructors*)
	 (cons (car expression)
	       (mapcar #'translate (cdr expression))))
	((numberp (car expression)) expression)
	(t (application-translation expression))))

(defun application-translation (expression)
  (cond ((null (cdr expression))
	 (translate `(funcall0 ,(car expression))))
	((null (cddr expression))
	 `(apply ,(translate (first expression))
		 ,(translate (second expression))))
	(t
	 `(apply ,(application-translation (butlast expression))
	   ,(translate (car (last expression)))))))

(defvar *translated-symbols* nil)

(emacs-indent deftranslator 2)
(defmacro deftranslator (symbol args &body body)
  (let ((name (combine-symbols symbol 'translator)))
    `(progn
       (pushnew ',symbol *translated-symbols*)
       (defun ,name ,args ,@body)
       (setf (translator ',symbol)
	     #',name))))

(defpiecefun clear-translators ())

(defpiece (clear-translators clear-translator-functions) ()
  (dolist (symbol *translated-symbols*)
    (setf (translator symbol) nil))
  (setf *translated-symbols* nil))

;
(defun macro-invert (exp)
  (cond ((hlps-tag-p exp)
	 (macro-invert (untag exp)))
	((not (consp exp))
	 exp)
	((and (eq (first exp) 'apply)
	      (consp (second exp))
	      (eq (first (second exp)) 'apply))
	 (macro-invert `(apply ,(second (second exp))
			 ,@(cons (third (second exp))
			    (rest (rest exp))))))
	(t
	 (simplify (cons (first exp) (mapcar #'macro-invert (rest exp)))))))

(defun simplify (exp)
  (let ((result
	  (let ((s-exp (rewrite exp)))
	      (if (eq s-exp exp)
		  exp
		  (simplify s-exp)))))
    result))

(defvar *available-variables* '(x y z v w a b c i j k d e f g h n1 n2 n3 n4 n5 n6 n7))


(defun rewrite (exp)
  (selectmatch exp
    ((apply ?fun . ?args)
     `(,?fun ,@?args))
    (((domain-operator) . :anything)
     (rewrite-special-fun 'a-domain-member-of 1 exp))
    (((cons-function) . :anything)
     (rewrite-special-fun 'cons 2 exp))
    (((car-function) . :anything)
     (rewrite-special-fun 'car 1 exp))
    (((cdr-function) . :anything)
     (rewrite-special-fun 'cdr 1 exp))
    (((funcall0-operator) ?fun)
     `(,?fun))
    (((rel-from-to) ?t1 ?t2)
     `(an-operator-from ,?t1 to ,?t2))
    (((total-rel-from-to) ?t1 ?t2)
     `(a-total-operator-from ,?t1 to ,?t2))
    (((partial-fun-from-to) ?t1 ?t2)
     `(a-partial-function-from ,?t1 to ,?t2))
    ((both
      (a-total-operator-from ?s1 to ?s2)
      (a-partial-function-from ?s1 to ?s2))
     `(a-function-from ,?s1 to ,?s2))
    ((a-function-from ?s1 to (a-total-operator-from . ?rest))
     `(a-total-operator-from ,?s1 ,@?rest))
    ((a-function-from ?s1 to (a-partial-function-from . ?rest))
     `(a-partial-function-from ,?s1 ,@?rest))
    ((a-function-from ?s1 to (an-operator-from . ?rest))
     `(an-operator-from ,?s1 ,@?rest))
    ((a-function-from ?s1 to (a-function-from . ?rest))
     `(a-function-from ,?s1 ,@?rest))

    ((lambda0 ?body)
     `(lambda () ,?body))
    ((lambda-rel0 ?body)
     `(lambda-rel () ,?body))

    ((db-lambda ?type (db-class-combinator ?body))
     (let ((var (first-var-unused (cons ?type ?body) *available-variables*)))
       `(lambda ((,var ,?type))
	  ,(replace-dbnum1 var ?body))))
    (((lambda ((?var ?type)) . ?body) ?type)
     `(let ((,?var ,?type)) ,@?body))
    ((lambda ?bindings1 (lambda ?bindings2 ?body))
     `(lambda (,@?bindings1 ,@?bindings2) ,?body))
    ((let ((?var ?type))
       (let ((?var1 ?type1)) . ?body))
     `(let ((,?var ,?type)
	    (,?var1 ,?type1)) ,@?body))
    ((let ((?x ?form)) ?x)
     ?form)
    ((lambda ((:anything :anything)) . :anything)
     exp)
    ((lambda () . :anything)
     exp)
    ((not (implies ?phi1 (not ?phi2)))
     `(and ,?phi1 ,?phi2))
    ((not (not ?phi))
     ?phi)
    ((db-some-such-that ?type (db-formula-combinator ?phi))
     (let ((var (first-var-unused ?phi *available-variables*)))
       `(some-such-that ,var ,?type ,(replace-dbnum1 var ?phi))))
    ((db-forall ?type (db-formula-combinator ?phi))
     (let ((var (first-var-unused ?phi *available-variables*)))
       `(forall ((,var ,?type)) ,(replace-dbnum1 var ?phi))))
    ((forall ?bindings1 (forall ?bindings2 ?phi))
     `(forall (,@?bindings1 ,@?bindings2) ,?phi))
    ((there-exists (some-such-that ?var ?type ?phi))
     `(exists ((,?var ,?type)) ,?phi))
    ((exists ?var-list1 (exists ?var-list2 ?phi))
     `(exists ,(append ?var-list1 ?var-list2) ,?phi))
    ((not (exists ?var-list (not ?phi)))
     `(forall ,?var-list ,?phi))
;;    ((a-set) `a-set)
;;    ((a-thing) `a-thing)
    (((member-operator) ?class)
     `(a-member-of ,?class))
    ((amb ?x ?y)
     `(either ,?x ,?y))
    ((either ?x (either . ?args))
     `(either ,?x ,@?args))
    ((intersection ?x ?y)
     `(both ,?x ,?y))
    ((both ?x (both . ?args))
     `(both ,?x ,@?args))
    ((lisp-object ?gensym)
     (let* ((*quiescent?* nil)
	    (object (gensym-expression>object ?gensym)))
       (if (numberp object)
	   object
	   `',object)))
    ((either (when ?phi ?x) (when (not ?phi) ?y))
     `(if ,?phi ,?x ,?y))
    ((if-builder ?phi (pair-classes ?x ?y))
     `(if ,?phi ,?x ,?y))
    ((?x . ?rest)
     (if (hlps-tag-p ?x)
	 `(,(untag ?x) ,@?rest)
	 exp))
    (?x ?x)))

(defun rewrite-special-fun (fun arity exp)
  (cond ((< (length (cdr exp)) arity)
	 exp)
	((= (length (cdr exp)) arity)
	 (cons fun (cdr exp)))
	(t
	 (ecase arity
	   (1 (cons (list fun (cadr exp)) (cddr exp)))
	   (2 (cons (list fun (cadr exp) (caddr exp)) (cdddr exp)))))))

(defun replace-dbnum1 (var formula)
  (cond ((or (atom formula)
	     (null formula))
	 formula)
	(t (selectmatch formula
	     ((de-bruijn (1))
	      var)
	     ((de-bruijn (?n))
	      `(de-bruijn (,(1- ?n))))
	     ((?constructor . ?args)
	      `(,(replace-dbnum1 var ?constructor)
		,@(mapcar (lambda (exp) (replace-dbnum1 var exp))
			  ?args)))))))

(defun first-var-unused (formula vars)
  (or (first (member-if (lambda (var)
			  (not (internal-member var formula)))
			vars))
      (gentemp "VAR-")))



;========================================================================
;type checking
;========================================================================
;
;These functions are used to type check rules.

(defun check-formula (expression)
  (unless (eq 'formula (check-expression expression))
    (error "~s is not a formula" expression)))




;========================================================================
;variable declarations
;========================================================================

;;;(declare-variables (<category> <var1> ... <varn>
;;;                                (<constructor-var>)
;;;                                (<constructor-var> <arg-category> <arg-category>))
;;;                   (<category> ...))

;The simplest variable declaration declares a variable to be of a certain caterogy.
;As described in the file syntax, categories can be organized into a supercategory-subcategory
;tree. We also allow for variables that range over constructors.  Currently
;constructor variables are given "arrow categories" (zero to one input categories and
;an output category).  However, it seems possible to do without any category assignments
;for constructor variables.  The main role of categories is to ensure that
;slot accesses created by formula antecedents are well typed.  This can be ensured
;even when categories are only assigned to node variables.

(defvar *variables* nil)
(defvar *constants* nil)
(property-macro variable-category)
(property-macro constant-expression)
(property-macro constant-value)

;(declare-variables
;  (class ?x ?y ?z)
;  (formula ?p ?q ?r)
;  (relation ?r ?f))

(defmacro declare-variables (&body forms)
  `(eval-when (compile load eval) (declare-variables-fun ',forms)))

(defun declare-variables-fun (forms)
  (dolist (form forms)
    (dolist (var (cdr form))
      (declare-variable-2 var (car form)))))
  
(defun declare-variable-2 (variable category)
  (let ((var (selectmatch variable
	       ((?var . :anything ) ?var)
	       (:anything variable))))
    (unless (variable? var)
      (error "attempt to declare the variable-type of the non-variable ~s" variable))
    (pushnew var *variables*)
    (setf *constants* (delete var *constants*))
    (selectmatch variable
      ((?var . ?arg-cats)
       (setf (argument-categories ?var) ?arg-cats)
       (setf (output-category ?var) category))
      (?var
       (setf (variable-category ?var) category)))))

(defun node-variable? (var)
  (not (null (variable-category var))))

(defun clear-variables ()
  (dolist (var *variables*)
    (setf (variable-category var) nil))
  (setf *variables* nil))

(defun check-expression (expression)
  (cond ((symbolp expression)
	 (cond ((not (variable? expression))
		(error "the constant ~s is given without parentheses" expression))
	       ((null (variable-category expression))
		(error "undeclared variable ~s" expression))
	       (t
		(variable-category expression))))
	((numberp expression)
	 'class)
	((basic-object-p expression)
	 (type-of expression))
	((eq (car expression) '=)
	 (unless (= (length expression) 3)
	   (error "wrong number of arguments in ~s" expression))
	 (let ((first-cat (check-expression (second expression)))
	       (second-cat (check-expression (third expression))))
	   (unless (or (sub-category? first-cat second-cat)
		       (sub-category? second-cat first-cat))
	     (error "mismatched categories in arguments of ~s" expression)))
	 'formula)
	((numberp (car expression))
	 (when (cdr expression)
	   (error "the expression ~s has illegal arguments" expression))
	 'ontic-number)
	(t
	 (let ((given-categories (mapcar 'check-expression (rest expression)))
	       (arg-categories (argument-categories (first expression))))
	   (unless (and (= (length arg-categories) (length given-categories))
			(every #'sub-category? given-categories arg-categories))
	     (error "the expression ~s has illegal arguments" expression))
	   (output-category (first expression))))))

