;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Sequent Based High Level Proof System                                   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; A CONTEXT-EXTENSION is either an assumption, an axiom, or a definition. ;;;
;;; A context is defined by a sequence of context extensions.               ;;;
;;; The order of the extensions is important --- for example, only the      ;;;
;;; if a symbl has two different definitions, only the last is in force.    ;;;
;;; For technical reasons, the order of axioms and assumptions is also      ;;;
;;; significant.                                                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *faith-mode* nil)
(defvar *quiet-mode* nil)
(defvar *region-mode* nil)
(defvar *batch-mode* nil)
(defvar *showing* nil)
(defvar *equivalents* nil)
(defvar *failure-cache* nil)
(defvar *warnings* nil)

(defvar *rep-seen* nil)

(defvar *location* nil)
(defvar *backtrackable* nil)
(defvar *goal* nil)
(defvar *goal-stack* nil)
(defvar *user-goal* nil)
(defvar *ontic-libraries* nil)
(defvar *time* 0)

(defvar *sequent-cache* nil)

(eval-when (compile load eval)
  (setq *sequent-cache* (make-hash-table :test #'equal)))

(defvar *proof-cache* nil)

(eval-when (compile load eval)
  (setq *proof-cache* (make-hash-table :test #'equal)))

(defpiece (ontic-init-phase0 init-epcache) ()
  (clrhash *proof-cache*)
  (clrhash *sequent-cache*))

(defvar *srules* nil)
(setq *srules* nil)
(defvar *special-forms* nil)

(setq *special-forms* '(show-sequent invisible-show note let-sequent either-sequent
			both-sequent trap read-eval-print lisp-when lisp-bind
			proof-cond query-sequent do-nothing evaluate-proof
			suppose))

(emacs-indent lisp-when 1)
(emacs-indent lisp-bind 2)

(defvar *executep* nil)

(defvar *error-break* nil)



;This is the ``sequent module''.  It should only be accessed via
;the operations obvious-sequent? and extend-base-context.

(defvar *null-context* nil)

(defvar *context*)

(defun null-context? ()
  (equal *null-context* *context*))

(defpiece (ontic-init-phase0 init-context) ()
  (setq *context* nil))

(defun add-extension (ctxt ext)
  (append ctxt (list ext)))

(defun concat-contexts (ctxt1 ctxt2)
  (remove-duplicates (append ctxt1 ctxt2) :from-end t))

(defun obvious-sequent? (ctxt formula &key (use-ref? t))
  (let ((key (append (list formula) (reverse ctxt) (list use-ref?))))
    (mvlet (((ans cached?) (gethash key *sequent-cache*)))
       (if cached?
	   ans
	   (let ((ans (obvious-sequent-nocache? ctxt formula :use-ref? use-ref?)))
	     (setf (gethash key *sequent-cache*)
		   ans)
	     ans)))))

(defvar *calls* 0)

(defun obvious-sequent-nocache? (ctxt formula &key (use-ref? t))
  (incf *calls*)
  (goto-context ctxt)
  (or (contradiction?)
      (value-from-undo-frame (eq :true (is-true? (tintern formula))))
      (and use-ref?
	   (value-from-undo-frame
	    (assume-fun `(not ,formula))
	    (contradiction?)))))

;; *lemma-library* is the list of context extenders which
;; have been run in the base context.
(defvar *lemma-library* nil)

(defpiece (ontic-init-phase0 init-lemma-library) ()
  (setf *lemma-library* nil))

;The following runs for real even in faith mode.

(defun extend-base-context (derived-context)
  (goto-context *null-context*)
  (clrhash *sequent-cache*)
  (clrhash *proof-cache*)
;;  (when-visible-eval '(hlps-set-runbar))
  (mapc 'execute-extension derived-context)
  (setf *lemma-library* (append *lemma-library* derived-context))
;;  (when-visible-eval '(hlps-clear-runbar))
  ;; Dont't return the whole lemma library as the value of this
  ;; function; it takes too long to print.
  t
  )

;The following four procedures should never be called directly.

(defun goto-context (new-context)
  (let ((runbar-on nil))
    (unless (equal new-context (current-context))
;;      (when-visible-eval '(hlps-set-runbar))
      (setf runbar-on t))
    (if (null new-context)
	(dotimes (n (length (current-context)))
	  (pop-extension))
	(let ((tail (current-context)))
	  (dolist (extension new-context)
	    (cond ((and tail
			(equal extension (car tail)))
		   (pop tail))
		  (tail
		    (dotimes (n (length tail))
		      (pop-extension))
		    (setq tail nil)
		    (extend-frame extension))
		  (t
		    (extend-frame extension))))
	  (when tail
	    (dotimes (n (length tail))
	      (pop-extension)))))
;;    (when runbar-on (when-visible-eval '(hlps-clear-runbar)))
    ))

(defvar *trace-context* nil)

(defun pop-extension ()
  (when *trace-context*
    (format t "~%pop from extension ~s" (length *context*)))
  (util::pop-undo-frame)
  (pop *context*))

(defvar *monitor-goto-context* nil)

(defvar *extend-frame-info* nil)

(defun extend-frame (extension)
  (when *trace-context*
    (format t "~% pushing ~s ~s" (1+ (length *context*)) extension))
  (util::push-undo-frame)
  (push `(attempt ,extension) *context*)
  (if *monitor-goto-context*
      (let ((old-inferences *inferences*)
	    (old-run-time (get-internal-run-time))
	    (retval (execute-extension extension)))
	(push (list extension (- *inferences* old-inferences)
		    (- (get-internal-run-time) old-run-time))
	      *extend-frame-info*)
	retval)
      (execute-extension extension))
  (pop *context*)
  (push extension *context*)
  (when *trace-context*
    (format t "~% push completed")))

(defun execute-extension (extension)
  (unless (contradiction?)
    (let ((*contradiction-hook* (lambda ()
				  (clear-queues)
				  (return-from execute-extension))))
      (apply (get (car extension) 'execution-function) (cdr extension)))))

;; the following two may

(defun context-member? (phi context)
  (member-if (lambda (psi) (or (equal psi '(theorem (false)))
			       (equal psi phi)))
	     context))

(defun current-context ()
  (reverse *context*))
  
;end of sequent module.


;********************************************************************
;The following procedure allows for setting and restoring reset points.
;********************************************************************

(defvar *resets-availible* 0 "this is the number of reset points on the stack")

(defpiece (ontic-init-phase0 :reset-available-states) ()
    (setf *resets-availible* 0))

(defun push-reset-state ()
  (goto-context *null-context*)
  (util::push-undo-frame)
  (incf *resets-availible*))

(defun pop-reset-state ()
  (if (= *resets-availible* 0)
      (ontic-error "no saved states")
      (progn (goto-context *null-context*)
	     (util::pop-undo-frame)
	     (decf *resets-availible*))))

(defun reset ()
  (clrhash *proof-cache*)
  (clrhash *sequent-cache*)
  (if (= *resets-availible* 0)
      (ontic-init)
      (progn (goto-context *null-context*)
	     (util::pop-undo-frame)
	     (util::push-undo-frame))))

;Here are some functions for violating the abstraction barier
;of the sequent module.

(defun base-frame ()
    (goto-context nil))

(defmacro ext-context (form)
  `(goto-context (add-extension (current-context) ',form)))

(defmacro ext-assume (form)
  `(goto-context (add-extension (current-context) '(assume ,form))))

(defmacro ext-axiom (form)
  `(ext-axiom-fun ',form))

(defun ext-axiom-fun (form)
 (goto-context (add-extension (current-context) `(axiom ,form))))

(defmacro ext-defontic (name body &rest keywords)
  `(goto-context (add-extension (current-context) '(define ,name ,body ,@keywords))))

(defmacro ext-define (name body &rest keywords)
  `(goto-context (add-extension (current-context) '(define ,name ,body ,@keywords))))

(defun ext-pop ()
  (when *context*
    (goto-context (butlast (current-context)))))


;The context extensions used in meta-ontic.

(setf (get 'assume 'execution-function) 'assume-fun)

(setf (get 'theorem 'execution-function) 'axiom-fun)
(setf (get 'invisible-theorem 'execution-function) 'axiom-fun)

(setf (get 'axiom 'execution-function) 'axiom-fun)

(setf (get 'define 'execution-function) 'defontic-fun)

(defmacro defextender (extender-name arguments &rest body)
  `(eval-when (eval load compile)
    (setf (get ',extender-name 'execution-function) ',(create-name extender-name 'xyzpdq))
    (defun ,(create-name extender-name 'xyzpdq) ,arguments
      ,@body)))

;the following hook is left empty in this file but is called
;on all assumptions and axioms when a frame is extended.

(defpiecefun install-formula (form))

(defmacro axiom (expression)
  `(extend-base-context '((axiom ,expression))))

(defmacro theorem (expression)
  `(extend-base-context '((axiom ,expression))))

(defun axiom-fun (expression)
  (let* ((tform (translate expression)))
    (mvlet (((form iframe) (cintern tform)))
      (unless (formula-p form)
	(ontic-error (format nil "Attempt to assert the non-formula ~s" expression)))
      (install-formula tform)
      (assert-is-true form :true
		      :justification
		      (when *record-justifications*
			(let ((axiom-frame (make-invocation-frame (axiom expression)
					     (make-justification 'axiom nil))))
			  (make-justification 'axiom
			    (if iframe
				(list axiom-frame iframe)
				axiom-frame))))))))

(defmacro defproof (&body proofs)
  `(progn
     (setq *rep-seen* nil)
     (run-proof '(progn ,@proofs))))

(defmacro proof (&body proofs)
  `(progn
     (setq *rep-seen* nil)
     (run-proof ',@proofs)))

(defmacro defmodule (&body proofs)
  `(progn
     (setq *rep-seen* nil)
     (run-proof
      '(progn ,@(mapcar #'rename-defontic proofs)))))

;The following is used in proofs that are run in def-o-structs.
;The proofs are generated by macro expansion and hence are not visible.

(defmacro invisible-module (&body proofs)
  `(with-batch
     (defmodule ,@proofs)))

(defun rename-defontic (proof)
  (unless (consp proof) (ontic-error (format nil "What sort of proof is ~s?" proof)))
  (if (member (car proof) '(define defontic))
      `(module-definition ,@(cdr proof))
      (if (eq (car proof) 'def-o-struct)
	  `(module-def-o-struct ,@(cdr proof))
	  proof)))

(emacs-indent defproof 0)
(emacs-indent defmodule 0)

(defvar *variables-used* 0 "for proof cache")

(defun new-proof-variable (name)
  (intern (format nil "!~s-~s" name (incf *variables-used*))))

(defvar *inferences* 0)
(defvar *inference-limit* nil)
;; The following limits the number of inferences that can be
;; done without intervening cursor movement, when *visible-evaluation?*
;; is set.  On the machine hammer.ai.mit.edu, which seems to do
;; about 100 inferences/second, the following is 2 minutes.
(defvar *inference-increment* 1200)

(defpiece (notice-binary-inference increment-inference) (pred obj1 obj2)
  (increment-inferences))

(defpiece (notice-monadic-inference increment-inference) (pred object)
  (increment-inferences))

(defun increment-inferences ()
  (incf *inferences*))

;; set the following to t to get counts of inferences for successful proofs
(defvar *report-inferences* nil)

(defun run-proof (proof)
  (catch 'run-proof
    (setf *equivalents* nil)
    (setf *failure-cache* nil)
    (setf *variables-used* 0)
    (setf *inferences* 0)
    (setf *inference-limit* *inference-increment*)
    (let ((derived-context (evaluate-proof (tag proof)
					   *null-context*)))
      (when-visible-eval '(ensure-source-window
			    (beginning-of-defun)
			    (forward-sexp 1)))
      (extend-base-context derived-context)
      derived-context)))

(defun gc-noticer (state)
  (case  state
    (:before (when-visible-eval '(hlps-lisp-gc-start)))
    (:after (when-visible-eval '(hlps-lisp-gc-end)))))

(defun compile-noticer (state)
  (case  state
    (:before (emacs-eval '(hlps-lisp-compile-start)))
    (:after (emacs-eval '(hlps-lisp-compile-end)))))

(setf *compile-noticer* #'compile-noticer)

#+lucid
(setf user::*gc-silence* #'gc-noticer)

;Assumptions are different from axioms --- assumptions interact
;with the controls on universal generalization.

(defmacro assume (expression)
  `(extend-base-context '((assume ,expression))))

(defun assume-fun (expression)
  (let* ((tform (translate expression)))
    (mvlet (((form iframe) (cintern tform)))
      (unless (formula-p form)
	(ontic-error (format nil "Attempt to assert the non-formula ~s" expression)))
      (install-formula tform)
      (unless (eq (is-true? form) :true)
	(setf-undo *last-assumption-max-const*
		   (max *last-assumption-max-const* (max-constant form))))
      (assert-is-true form :true
		      :justification
		      (when *record-justifications*
			(let ((axiom-frame (make-invocation-frame (axiom expression)
					     (make-justification 'axiom nil))))
			  (make-justification 'axiom
			    (if iframe
				(list axiom-frame iframe)
				axiom-frame))))))))

(defun assume-node (node &optional (truth :true))
  (unless (eq (is-true? node) truth)
    (setf-undo *last-assumption-max-const*
	       (max *last-assumption-max-const* (max-constant node))))
  (assert-is-true node truth))

(defvar *last-assumption-max-const* 0)

(defpiece (ontic-init-phase0 init-assumed-state) ()
  (setf *last-assumption-max-const* 0))

(defmacro ? (exp)
  `(ask ',exp))

(defun ask (exp)
  (value-from-undo-frame
   (let ((form (tintern exp)))
     (unless (formula-p form)
       (ontic-error (format nil "attempt to ask the non-question ~s" exp)))
     (let ((truth (is-true? (tintern exp))))
       (case truth
	 (:true :yes)
	 (:false :no)
	 (t :unknown))))))

(property-macro untranslated-definition)

;The following macro applies when a defontic expression
;is evaluated as a lisp expression.  when a defontic appears
;in proof position inside an ontic proof the definition is made
;in the local context rather than the null (global) context.

(defmacro defontic (symbol &rest body)
  `(define ,symbol ,@body))

(defmacro define (symbol &rest body)
  `(run-proof '(module-definition ,symbol ,@body)))

(emacs-indent defontic 1)
(emacs-indent define 1)

(defmacro undefine (symbol)
  `(define ',symbol nil))

(defun defontic-fun (symbol expression &rest keylist &key &allow-other-keys)
  (if (listp symbol)
      (apply #'defontic-fun2 (car symbol) `(lambda ,(cdr symbol) ,expression) keylist)
      (apply #'defontic-fun2 symbol expression keylist)))

(defpiecefun check-definition (symbol translated-def keylist))

(defpiecefun notice-definition (symbol translated-def keylist))

(defun defontic-fun2 (symbol expression &rest other-keys &key (resize t) allow-redefinition &allow-other-keys)
  (when (or (member symbol *constructors*)
	    (translator symbol))
    (ontic-error (format nil "Cannot re-define built-in symbol ~s" symbol)))
  (when (and (definition-of symbol)
	     (not allow-redefinition))
    (ontic-error (format nil "Cannot locally redefine the pre-existing symbol ~s" symbol)))
  (let ((translated-def (translate expression)))
    (setf-undo (untranslated-definition symbol) expression)
    (if (internal-member symbol translated-def)
	(recursively-define symbol expression other-keys :resize resize)
	(basic-define symbol translated-def other-keys :resize resize))))

(defun definition (symbol)
  (unless (symbolp symbol)
    (format t "~% Error:  Definition expects a symbol, not '~s'" symbol))
  (untranslated-definition symbol))

;recursive-define is defined in the file induction.lisp.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generic Support Functions:                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun not-in (symbol exp2)
  (cond ((null exp2) t)
	((not (consp exp2)) (not (eq symbol exp2)))
	(t
	  (and (not-in symbol (car exp2))
	       (not-in symbol (cdr exp2))))))





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Hlps code:                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(forward-declare-nondeterministic-functions
  evaluate-let-sequent
  run-sequents
  run-sequents2
  run-individual-sequent
  expand-macro-top-level)

(defun declare-ontic-bug (msg)
  (format t msg)
  (error "this is a bug in the ontic system --- tell an ontic hacker"))

(defun ontic-warning (msg &optional (test t))
  (setq *warnings* (cons (cons test msg) *warnings*)))

(defun execute-warnings ()
  (dolist (warning *warnings*)
    (if (eval (car warning))
	(format t "~a" (cdr warning)))))

(defun ontic-error (msg)
  (when *error-break*
    (error "*error-break* is set to t: ~s" msg))
  (cond (*visible-evaluation?*
	 (when *location*
	   (emacs-eval `(hlps-goto-location ',(hlps-tag-location *location*))))
	 (when (not *quiet-mode*)
	   (emacs-eval '(beep)))
	 (throw-to-emacs msg))
	(t
	 (if *location*
	     (progn (format t  "~% ~a at ~s" msg (hlps-tag-location *location*))
		    (throw 'run-proof nil))
	     (progn (format t "~% ~a" msg)
		    (throw 'run-proof nil))))))

(defun backtrackable-failure (msg)
  (if *backtrackable*
      (fail)
      (ontic-error msg)))

(defun backtrackable-proof-failure (ctxt)
  (when (not *backtrackable*)
    (goto-context ctxt)
    (if *evaluation-stack*
	(setq *failure-cache*
	      (list (string-downcase (make-proof (append *evaluation-stack*
							 `((show ,*showing*)))))))
	(setq *failure-cache*
	      (remove nil
		      (mapcar #'(lambda (goal)
				  (let ((failure (expand-failure goal)))
				    (if failure
					(format
					  nil (string-downcase
						(format nil "~a" failure))))))
			      (cdr (assoc *showing* *equivalents*
					  :test #'equal))))))
    (emacs-eval `(progn (setq *ontic-failures*
			      ',(mapcar #'quote-string *failure-cache*))
			(setq *ontic-next-failure* *ontic-failures*))))
  (if *goal*
      (backtrackable-failure
	(if (equal *showing* '(false))
	    (format nil "Failure to derive contradiction")
	    (format nil "Failure to show ~s" (macro-invert (translate *showing*)))))
      (backtrackable-failure "Pointless Proof")))

(defun backtrackable-proof-failure (ctxt)
  (when (not *backtrackable*)
    (goto-context ctxt)
    (if *evaluation-stack*
	(setq *failure-cache*
	      (list (string-downcase (make-proof (append *evaluation-stack*
							 `((show ,*showing*)))))))
	(setq *failure-cache*
	      (remove nil
		      (mapcar #'(lambda (goal)
				  (let ((failure (expand-failure goal)))
				    (if failure
					(format
					  nil (string-downcase
						(format nil "~a" failure))))))
			      (cdr (assoc *showing* *equivalents*
					  :test #'equal))))))
    (emacs-eval `(progn (setq *ontic-failures*
			      ',(mapcar #'quote-string *failure-cache*))
			(setq *ontic-next-failure* *ontic-failures*))))
  (if *goal*
      (backtrackable-failure
	(if (equal *showing* '(false))
	    (format nil "Failure to derive contradiction")
	    (format nil "Failure to show ~s" (macro-invert (translate *showing*)))))
      (backtrackable-failure "Pointless Proof")))

(defun pretty-print-expression (exp)
  (let ((*print-pretty* t)
	(*print-length* nil)
	(*print-level* nil))
    (format nil "~a" exp)))

(defvar some-vars '(x y z s t j k l m n))
(defun nth-proof-variable (n)
  (or (nth n some-vars)
      (progn
	(setf n (- n (length some-vars)))
	(if (< n 26)
	    (read-from-string (format nil "~a" (code-char (+ n 65))))
	    (read-from-string (format nil "N~a" (- n 25)))))))

(defun make-proof (evaluation-extenders &optional avoid-vars)
  (when evaluation-extenders
    (selectmatch (car evaluation-extenders)
      ((eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((existence-eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((at-most-one-eval :anything) (make-proof (cdr evaluation-extenders) avoid-vars))
      ((suppose ?phi) (format nil "(suppose ~a ~% ~a)"
			      (pretty-print-expression (macro-invert (translate ?phi)))
			      (make-proof (cdr evaluation-extenders) avoid-vars)))
      ((consider ?arg ?exp such-that ?phi)
       (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
	 (format nil "(consider ((~a ~a)) ~% such-that ~a ~% ~a)"
		 new-arg
		 (pretty-print-expression (macro-invert (translate ?exp)))
		 (pretty-print-expression (macro-invert (translate (sublis `((,?arg . ,new-arg)) ?phi))))
		 (make-proof (sublis `((,?arg . ,new-arg))
				     (cdr evaluation-extenders))
			     (cons new-arg avoid-vars)))))
      ((let-be ?arg ?exp) (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
			    (format nil "(let-be ((~a ~a)) ~% ~a)"
				    new-arg (pretty-print-expression (macro-invert (translate ?exp)))
				    (make-proof (sublis `((,?arg . ,new-arg))
							(cdr evaluation-extenders))
						(cons new-arg avoid-vars) ))))
      ((suppose-there-is ?arg ?exp) (let ((new-arg (apply #'new-proof-variable-not avoid-vars)))
				      (format nil "(suppose-there-is ((~a ~a)) ~% ~a)"
					      new-arg (pretty-print-expression (macro-invert (translate ?exp)))
					      (make-proof (sublis `((,?arg . ,new-arg))
								  (cdr evaluation-extenders))
							  (cons new-arg avoid-vars)))))
      ((show ?phi)
       (format nil "(show ~a)" (pretty-print-expression ?phi)))
      (:anything (format nil "(unknown-evaluation-extender ~a ~% ~a)"
			 (car evaluation-extenders)
			 (make-proof (cdr evaluation-extenders) avoid-vars))))))

(defpiece (ontic-init-phase0 clear-libraries) ()
  (setq *ontic-libraries* nil))

(defmacro ontic-require
    (symbol &key (file (format nil "~a/lib/~a.ont"
			       *ontic-directory*
			       (string-downcase (format nil "~a" symbol))))
	    verify)
  `(progn
     (emacs-eval '(hlps-goto-location '(0)))
     (unless (member ',symbol *ontic-libraries*)
       (if (not (probe-file ,file))
	   (ontic-error (format nil "Require file does not exists: ~a" ,file)))
       (emacs-eval
	 '(eval-require ,(quote-string file) ,verify ',symbol)))))

(defmacro ontic-provide (symbol)
  `(progn
     (emacs-eval '(hlps-goto-location '(0)))
     (goto-context nil)
     (setf-undo *ontic-libraries*
		(cons ',symbol (remove ',symbol *ontic-libraries*)))))

(defun check-provided (symbol)
  (if (member symbol *ontic-libraries*)
      (emacs-eval '(eval-region-continue))
      (ontic-error (format nil "Failure to provide: ~a" symbol))))

(defmacro with-error-notification (&rest body)
  `(let ((normal nil))
     (unwind-protect
	 (progn
	   (setq *warnings* nil)
	   ,@body
	   (setq normal t))
       (if (not normal)
	   (emacs-eval '(eval-region-error)))
       (execute-warnings))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Rule maintenance code:                                                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun get-rule (name)
  (find name *srules* :key #'car))

(defun sequent-p (name)
  (get-rule name))

(defun rule-antecedents (rule)
  (second rule))

(defun rule-consequent (rule)
  (third rule))

(defmacro defsequent (name antecedents conclusion)
  (clrhash *sequent-cache*)
  `(progn (setq *srules* (cons (list ',name ',antecedents ',conclusion)
			       (remove ',name *srules* :key #'car)))
	  ',name))

(emacs-indent defsequent 2)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Evaluate-proof support code.                                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (hlps-tag (:type list) :named)
  symbol
  location)

(defun tag (exp &optional loc (numcdr 0))
  (cond ((not (consp exp)) exp)
	((and (zerop numcdr)
	      (or (macro-p (car exp)) (get-rule (car exp)) (member (car exp) *special-forms*)))
	 `(,(make-hlps-tag :symbol (car exp) :location (reverse (cons numcdr loc)))
	   ,@(tag (rest exp) loc (1+ numcdr))))
	(t
	  (cons (tag (car exp) (cons numcdr loc) 0)
		(tag (cdr exp) loc (1+ numcdr))))))

(defun untag (exp)
  (cond ((null exp) nil)
	((hlps-tag-p exp)
	 (hlps-tag-symbol exp))
	((not (consp exp)) exp)
	(t
	  (cons (untag (car exp)) (untag (cdr exp))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Evaluate-proof code:                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Called from Top level.

(defvar *traced-proof-macros* nil)

(defun evaluate-proof (proof ctxt)
  (if (not (consp proof))
      (ontic-error "Illegal proof."))
  (when (member (untag (car proof)) *traced-proof-macros*)
    (rprint (untag proof)))
  (when (hlps-tag-p (car proof))
    (when-visible-eval `(hlps-goto-location ',(hlps-tag-location (car proof)))))
  (let ((key (list (untag proof) (reverse ctxt) *goal-stack*)))
    (mvlet (((value found?) (gethash key *proof-cache*)))
      (if found?
	  (progn (incf *variables-used* (car value)) (cdr value))
	  (let ((old-vars-used *variables-used*)
		(ans (evaluate-proof-nocache proof ctxt)))
	    (setf (gethash key *proof-cache*) (cons (- *variables-used* old-vars-used) ans))
	    (when (and (hlps-tag-p (car proof)) *location*)
	      (when-visible-eval `(hlps-goto-location ',(hlps-tag-location *location*))))
	    ans)))))

(defmacro proof-trace (&rest forms)
  `(dolist (form ',forms)
    (push form *traced-proof-macros*)))

(defun proof-untrace ()
  (setq *traced-proof-macros* nil))

(property-macro proof-function-of)

(defun evaluate-proof-nocache (proof ctxt)
  (let ((*location* (if (hlps-tag-p (car proof)) (car proof) *location*))
	(proof2  (cons (untag (car proof)) (cdr proof))))
    (cond ((macro-p (car proof2))
	   (evaluate-proof (macro-expand-one-level proof2) ctxt))
	  ((member  (car proof2) *special-forms*)
	   (dispatch-special-form proof2 ctxt))
	  ((proof-function? (car proof2))
	   (funcall (proof-function-of (car proof2)) proof ctxt))			   
	  ((sequent-p (car proof2))
	   (evaluate-proof-sequent proof2 ctxt))
	  (t
	   (ontic-error (format nil "Unrecognized Proof Construct ~s" (car proof)))))))

(defun proof-function? (symbol)
  (proof-function-of symbol))

(defmacro def-proof-function (pattern cvar &rest body)
  (let ((fun-name (create-name (car pattern) 'proof 'fun)))
    `(eval-when (eval load compile)
      (defun ,fun-name (arg ,cvar)
	(selectmatch arg
	  (,pattern
	   ,@body)
	  (:anything (error "illegal syntax for ~s" ',(car pattern)))))
      (setf (proof-function-of ',(car pattern)) ',fun-name))))

(emacs-indent def-proof-function 2)

(defun evaluate-proof-sequent (proof ctxt)
  (let ((extension (evaluate-proof-sequent-notest proof ctxt)))
    (if (null extension)
	(backtrackable-proof-failure ctxt)
	extension)))

;advanced comment ---
;the following function evaluates a trivial proof before evaluating the
;body of pf-seq.  This is done so that an ill formed antecedent
;in the sequent causes the cursor to stop at sequent rule
;rather than some show in the body where the "real"
;call to obvious-sequent occurs.  For example, this causes the cursor to
;stop at ill formed definitions in modules.  Without this trivial
;proof evaluation ill formed definitions are accepted and do not cause
;errors until the first show in a proof following the definition.

(defun evaluate-proof-sequent-notest (proof ctxt)
  (let ((rule (create-expression (get-rule (car proof)))))
    (when (not (= (length (rule-antecedents rule))
		  (length (cdr proof))))
      (sequent-bug))
    (all-values
      (let* ((r-seqs (rule-antecedents rule))
	     (pf-vals (mapcar (lambda (pf-seq r-seq)
				(let ((new-ctxt (concat-contexts ctxt (second pf-seq))))
				  (assert-equal! (second r-seq) (second pf-seq))
				  (when (not *faith-mode*) (evaluate-proof '(note (true)) new-ctxt))
				  (let ((result (evaluate-proof (third pf-seq) new-ctxt)))
				    (when (null result)
				      (fail))
				    result)))
			      (cdr proof)
			      r-seqs)))
	(run-sequents pf-vals r-seqs (rule-consequent rule))))))

(defun sequent-bug ()
  (declare-ontic-bug "There appears to be an ill-formed fundamental macro"))

(defun-nondeterministic run-sequents (pf-vals r-ants r-conc)
   (cond ((null pf-vals)
	  (when (unbound-variable-p r-conc)
	    (declare-ontic-bug "There appears to be an ill-formed sequent rule"))
	  (check-conc (apply-substitution r-conc)))
	 (t (unless (member? (third (car r-ants)) (car pf-vals))
	      (fail))
	    (run-sequents (cdr pf-vals) (cdr r-ants) r-conc))))

(defun check-conc (conc)
  (cond ((eq (car conc) 'when)
	 (if (eval (second conc))
	     (check-conc (third conc))
	     (fail)))
	((eq (car conc) 'if)
	 (if (eval (second conc))
	     (check-conc (third conc))
	     (check-conc (fourth conc))))
	((eq (car conc) 'lisp)
	 (eval (second conc)))
	(t conc)))

(defun dispatch-special-form (proof ctxt)
  (cond ((eq (car proof) 'read-eval-print)
	 (goto-context ctxt)
	 (when (not *quiet-mode*)
	   (emacs-eval '(beep)))
	 (emacs-eval
	   `(message ,(quote-string "Proof temporarily stopped.")))
	 (enter-rep-sbhlps)
	 nil)
	((eq (car proof) 'proof-fail)
	 (backtrackable-proof-failure ctxt))
	((eq (car proof) 'note)
	 (selectmatch proof
	   ((note ?phi :goal-stack ?bool)
	    (eval-note ?phi ctxt :goal-stack ?bool))
	   ((note ?phi)
	    (eval-note ?phi ctxt :goal-stack t))
	   (:anything (sequent-bug))))
	((eq (car proof) 'show-sequent)
	 (evaluate-show-sequent proof ctxt))
	((eq (car proof) 'invisible-show)
	 (evaluate-invisible-show proof ctxt))
	((eq (car proof) 'either-sequent)
	 (evaluate-either-sequent proof ctxt))
	((eq (car proof) 'let-sequent)
	 (all-values (evaluate-let-sequent proof ctxt)))
	((eq (car proof) 'both-sequent)
	 (let* ((f1 (evaluate-proof (second proof) ctxt))
		(f2 (evaluate-proof (third proof) (concat-contexts ctxt f1))))
	   (append f1 f2)))
	((eq (car proof) 'proof-cond)
	 (evaluate-proof-cond (cdr proof) ctxt))
	((eq (car proof) 'lisp-bind)
	 (evaluate-lisp-bind proof ctxt))
	((eq (car proof) 'lisp-when)
	 (evaluate-lisp-when proof ctxt))
	((eq (car proof) 'query-sequent)
	 (evaluate-query-sequent proof ctxt))
	((eq (car proof) 'do-nothing)
	 nil)
	((eq (car proof) 'suppose)
	 (evaluate-suppose proof ctxt))
	((eq (car proof) 'evaluate-proof)
	 (evaluate-evaluate-proof (cdr proof) ctxt))))

;; The special forms which call eval (currently lisp-bind, lisp-when,
;; and evaluate-proof) may need to be in the correct context.  On
;; the other hand, they may operate only on text and not need
;; to have anything to do with the current context.  Rather than
;; having lisp-bind, lisp-when, and evaluate-proof do (goto-context ctxt),
;; they bind the special variable *real-context* to the context.
;; The body of a lisp-when could then do (goto-context *real-context*).
(defvar *real-context* nil)

(defun evaluate-lisp-bind (proof ctxt)
  (let ((*real-context* ctxt))
    (if (not (= (length proof) 4))
	(sequent-bug)
	(progv
	    (list (second proof))
	    (list (eval (third proof)))
	  (evaluate-proof (fourth proof) ctxt)))))

(defun evaluate-lisp-when (proof ctxt)
  (let ((*real-context* ctxt))
    (if (not (= (length proof) 3))
	(sequent-bug)
	(when (eval (second proof))
	  (evaluate-proof (third proof) ctxt)))))

(defun eval-note (note ctxt &key (goal-stack t))
  (if (eq note '*goal*)
      (when *goal* (maybe-note *goal* ctxt :goal-stack goal-stack))
      (maybe-note note ctxt :goal-stack goal-stack)))

(defun maybe-note (note ctxt &key (goal-stack t))
  (append
   (when goal-stack
     (mapcan (lambda (goal)
	       (when (obvious-sequent? ctxt goal :use-ref? nil)
		 `((theorem ,goal))))
	     (cdr *goal-stack*)))	;; don't include *goal*
   (when (obvious-sequent? ctxt note)
     `((theorem ,note)))
   (when (obvious-sequent? ctxt '(false))
     '((theorem (false))))))

(defun evaluate-invisible-show (proof ctxt)
  (selectmatch proof
    ((invisible-show ?phi . ?body)
     (let ((*showing* ?phi))
       (let ((extension (evaluate-proof `(show-internal ,?phi ,@?body) ctxt)))
	 (cons `(invisible-theorem ,?phi)
	       (remove `(theorem ,?phi) extension :test #'equal)))))
    (:anything (sequent-bug))))

(defun evaluate-show-sequent (proof ctxt)
  (if (not (or (= (length proof) 3)
	       (= (length proof) 4)))
      (sequent-bug)
      (if (or (not (eq (second proof) '*goal*))
	      *goal*)
	  (let* ((proof-goal (or (second proof) '(true)))
		 (new-goal (or (and (eq proof-goal '*goal*) *goal*)
			       proof-goal))
		 (goal-changed (not (equal *goal* new-goal)))
		 (*goal* new-goal)
		 (*goal-stack* (if (member *goal* *goal-stack* :test #'equal)
				   *goal-stack*
				   (cons *goal* *goal-stack*)))
		 (*user-goal* (or (and (not goal-changed) *user-goal*)
				  (fourth proof)
				  *goal*)))
	    (let ((result `(theorem ,*goal*)))
	      (cond ((and *faith-mode* (not *backtrackable*))
		     (list result))
		    ((obvious-sequent? ctxt *goal* :use-ref? nil)
		     (maybe-note *goal* ctxt))
		    (t
		     (let ((extension (evaluate-proof (third proof) ctxt)))
		       (unless (context-member? result extension)
			 (backtrackable-proof-failure ctxt))
		       (intersection
			(mapcar (lambda (goal)
				  `(theorem ,goal))
				*goal-stack*)
			extension :test #'equal))))))
	  (backtrackable-proof-failure ctxt))))

(defun evaluate-query-sequent (proof ctxt)
  (when (not (= (length proof) 2))
    (sequent-bug))
  (let* ((*goal* (second proof))
	 (*user-goal* *goal*))
    (unless (obvious-sequent? ctxt *goal* :use-ref? nil)
      (backtrackable-proof-failure ctxt))
    nil))

(defun evaluate-either-sequent (proof ctxt)
  (if (not (= (length proof) 3))
      (sequent-bug))
  (one-value (either
	      (let ((*backtrackable* t))
		(evaluate-proof (second proof) ctxt))
	      (evaluate-proof (third proof) ctxt))
	     (fail)))

(defun-nondeterministic evaluate-let-sequent (proof ctxt)
  (if (or (not (= (length proof) 3))
	  (not (= (length (second proof)) 2)))
      (sequent-bug))
  (let* ((formula (member-of (evaluate-proof
			       (second (second proof)) ctxt)))
	 (vars (variables-of (first (second proof))))
	 (mp (create-expression-rv
	       (list (first (second proof)) (third proof))
	       vars))
	 (pattern (first mp))
	 (body (second mp)))
    (when (not (structure-compatable-p pattern formula))
      (backtrackable-failure
       "Unable to unify variable in let-sequent."))
    (when-equal pattern formula
      (if (unbound-variable-p body)
	  (ontic-error
	    "Unbound variable in body of let-sequent."))
      (member-of (evaluate-proof
		   (apply-substitution body) (add-extension ctxt formula))))))

(defun evaluate-proof-cond (clauses ctxt)
  (cond ((null clauses) nil)
	((not (consp clauses))
	 (ontic-error "Illegal syntax for proof-cond"))
	(t (selectmatch (car clauses)
	     ((?test . ?body)
	      (if (obvious-sequent? ctxt ?test)
		  (evaluate-proof `(progn . ,?body) ctxt)
		  (evaluate-proof-cond (cdr clauses) ctxt)))
	     (:anything
	      (ontic-error "Illegal syntax for proof-cond"))))))

(defun evaluate-suppose (proof ctxt)
  (when (<= (length proof) 1)
    (sequent-bug))
  (let ((formula (second proof))
	(body (cddr proof))
	(ret nil))
    (dolist (extender (evaluate-proof `(proof-body . ,body)
				      (add-extension (add-extension ctxt `(match-mark! ,formula))
						     `(assume ,formula))))
      (selectmatch extender
	((theorem ?thm)
	 (push `(theorem (implies ,formula ,?thm)) ret))))
    ret))


;(defun evaluate-suppose (proof ctxt)
;  (when (<= (length proof) 1)
;    (sequent-bug))
;  (let ((formula (second proof))
;	(body (cddr proof))
;	(ret nil))
;    (dolist (extender (evaluate-proof `(proof-body . ,body)
;				      (add-extension (add-extension ctxt `(match-mark! ,formula))
;						     `(assume ,formula))))
;      (selectmatch extender
;	((theorem ?thm)
;	 (if (member extender
;		     (evaluate-proof `(note ,?thm :goal-stack nil)
;				     (add-extension (add-extension ctxt `(match-mark! ,formula))
;						    `(assume (not ,formula))))
;		     :test #'equal)
;	     (push extender ret)
;	     (push `(theorem (implies ,formula ,?thm)) ret)))))
;    ret))

(defun evaluate-evaluate-proof (proof ctxt)
  (when (not (= (length proof) 1))
    (ontic-error "Illegal syntax for evaluate-evaluate-proof"))
  (let ((new-proof (let ((*real-context* ctxt))
		     (eval (car proof)))))
    (evaluate-proof new-proof ctxt)))
  
  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Macro Code:                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *macros* nil)
(setq *macros* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Macro rule maintence code.                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro defmac (name &rest body)
  `(defmac-fun ',name ',body))

(defun defmac-fun (name body)
  (let ((form (first body))
	(condition (second body))
	(replacement (third body)))
    (clrhash *proof-cache*)

    ;first we remove previous versions of the macro
    (let ((old-macro (assoc name *macros*)))
      (when old-macro
	(let ((old-form (second old-macro)))
	  (util::defselectpiece-fun
	      (create-name-internal 'macro-expand (car old-form))
	      name
	    '(form) nil nil nil))))
    (setq *macros* (remove name *macros* :key #'car))

    ;now we add the new version
    (when form
      (setq *macros* (append *macros* (list (list name form condition replacement))))
      (create-macro-expander (car form))
      (util::defselectpiece-fun
	  (create-name-internal 'macro-expand (car form))
	  name
	'(form) form condition (list (create-expansion-code replacement (variables-of form)))))))

(defun create-macro-expander (name)
  (let ((expander-name (create-name-internal 'macro-expand name)))
    (util::defselectfun-fun expander-name '(form) nil)
    (setf (get name 'macro-expander) expander-name)))

(eval-when (compile load eval)
  (defun create-expansion-code (replacement bound-vars)
    (selectmatch replacement
      ((lisp-let () ?body)
       (create-expansion-code ?body bound-vars))
      ((lisp-let (?binding . ?rest-bindings) ?body)
       `(let (,?binding)
	 ,(create-expansion-code `(lisp-let ,?rest-bindings ,?body)
	   (adjoin (car ?binding) bound-vars))))
      ((?a . ?b)
       (optimize-cons (create-expansion-code ?a bound-vars)
		      (create-expansion-code ?b bound-vars)))
      (:anything
       (if (member replacement bound-vars)
	   replacement
	   `(quote ,replacement)))))

  (defun optimize-cons (first second)
    (if (and (listp first) (eq (car first) 'quote)
	     (listp second) (eq (car second) 'quote))
	`(quote ,(cons (second first) (second second)))
	`(cons ,first ,second)))
  );; eval-when

(emacs-indent defmac 1)

(defun macro-p (symbol)
  (find symbol *macros* :key #'caadr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Macro expansion code.                                                   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun macro-expand-one-level (proof)
  (if (and (listp proof) (or (symbolp (car proof))
			     (hlps-tag-p (car proof))))
      (let ((expander (get (untag (car proof)) 'macro-expander)))
	(if expander
	    (let ((expanded (funcall expander (cons (untag (car proof)) (cdr proof)))))
	      (when (not expanded)
		(ontic-error (format nil "Ill-formed ~s construct: ~s"
				     (car proof) proof)))
	      expanded)
	    (ontic-error (format nil "Unrecognized proof construct ~s"
				 (car proof)))))
      (ontic-error (format nil "Ill-formed proof ~s" proof))))

;simple utilities


(defun show-full-context ()
  (let ((*print-level* 100)
	(*print-length* 100))
    (pprint (mapcar #'macro-invert (current-context)))))

(defun show-context ()
  (let ((*print-level* 100)
	(*print-length* 100)
	(result nil))
    (dolist (extension (current-context))
      (unless (or (matches? extension (tintern :anything))
		  (matches? extension (apply tintern :anything))
		  (matches? extension (match-mark! :anything))
		  (matches? extension (recursively-axiom . :anything))
		  (matches? extension (invisible-theorem . :anything))
		  (member extension result :test 'equal))
	(push extension result)))
    (pprint (reverse (mapcar (lambda (ext) (selectmatch ext
					     ((:lisp :anything) ext)
					     (:anything
					      (cons (car ext)
						    (mapcar 'cleanup (cdr ext))))))
			     result)))))

(defun cleanup (exp)
  (macro-invert (translate exp)))

(defun show-inferential-context ()
  (let ((*print-level* 100)
	(*print-length* 100))
    (pprint (remove-if (lambda (c-item) (selectmatch c-item
					  ((axiom (?pred . :anything))
					   (macro-p ?pred))
					  ((recursively-axiom . :anything) t)
					  (:anything nil)))
		       (current-context)))))


(defvar *proof*)

(defun expand (&optional proof)
  (when proof (setq *proof* (tag proof)))
  (setq *proof* (macro-expand-one-level *proof*))
  (rprint *proof*))

(defun expand1-proof (proof)
  (expand-proof proof))

(defun expand-plength ()
  (setq *print-level* 15)
  (setq *print-length* 15))


;********************************************************************
;the macros progn and first
;********************************************************************

(defmac proof-body
  (proof-body . ?forms)
  t
  (both-sequent
    (progn . ?forms)
    (show-internal *goal*)))

(defmac progn-no-body
   (progn)
   t
   (do-nothing))

(defmac progn-multiple-bodies
  (progn ?proof . ?rest-proofs)
  (not (and (consp ?proof)
	    (member (car ?proof) '(defontic define))))
  (both-sequent ?proof
    (progn . ?rest-proofs)))

(defsequent definition-sequent
    ((sequent ((define ?var ?form . ?keywords))
       (theorem ?phi)))
  (if (not (internal-member '?var (translate '?phi)))
      (theorem ?phi)
      (lisp (sublis (acons '?var '?form nil) '(theorem ?phi)))))

(defsequent module-definition-sequent
    ((sequent ((define ?var ?form . ?keywords))
       (theorem (true))))
  (define ?var ?form . ?keywords))

(defmac progn-multiple-bodies2
  (progn (?defontic ?symbol ?body) . ?rest-proofs)
  (and (symbolp ?symbol)
       (member ?defontic '(defontic define)))
  (definition-sequent
    (sequent ((define ?symbol ?body :resize nil))
      (progn . ?rest-proofs))))

(defmac progn-multiple-bodies3
  (progn (?defontic (?op . ?args) ?body) . ?rest-proofs)
  (and (symbolp ?op)
       (member ?defontic '(defontic define)))
  (definition-sequent
    (sequent ((define ?op (lambda ?args ?body) :resize nil))
      (progn . ?rest-proofs))))

(defmac progn-multiple-bodies4
  (module-definition ?symbol ?body)
  (symbolp ?symbol)
  (module-definition-sequent
    (sequent ((define ?symbol ?body :allow-redefinition t))
      (note (true)))))

(defmac progn-multiple-bodies5
  (module-definition (?op . ?args) ?body)
  (symbolp ?op)
  (module-definition-sequent
    (sequent ((define ?op (lambda ?args ?body) :allow-redefinition t))
      (note (true)))))



;========================================================================
;declarations in definitions
;========================================================================

;this is modified from sbhlps

(defmac progn-multiple-bodies6
  (module-definition ?symbol ?body . ?declarations)
  (and ?declarations (symbolp ?symbol))
  (lisp-let ((?new-decls (process-declarations nil ?declarations)))
    (progn (module-definition ?symbol ?body)
	   . ?new-decls)))

(defmac progn-multiple-bodies7
  (module-definition (?op . ?args) ?body . ?declarations)
  (and ?declarations (symbolp ?op))
  (lisp-let ((?new-decls (process-declarations ?args ?declarations)))
    (progn (module-definition (?op . ?args) ?body)
	   . ?new-decls)))

(defun process-declarations (args declarations)
  (mapcar #'(lambda (decl) (process-declaration args decl))
	  declarations))

(defun process-declaration (args proof)
  `(suppose-there-is ,args ,proof))

(defun starts-with (expression operator)
  (and (consp expression) (eq (car expression) operator)))

;; powerful sequent rules!!  don't abuse them!!
(defsequent assume-sequent
    ((sequent ((assume ?psi))
       (theorem (true))))
  (theorem ?psi))

(defmac progn-multiple-bodies8
  (assume ?phi)
  t
  (assume-sequent
    (sequent ((assume ?phi))
      (note (true)))))

(defmac progn-multiple-bodies9
  (axiom ?phi)
  t
  (assume-sequent
    (sequent ((assume ?phi))
      (note (true)))))


(defmac first-no-body
  (first)
  t
  (do-nothing))

(defmac first-one-form
  (first ?proof)
  t
  ?proof)

(defmac first-multiple-bodies
  (first ?proof1 ?proof2 . ?rest-proofs)
  t
  (either-sequent
    ?proof1
    (first ?proof2 . ?rest-proofs)))

(defmac invisible-module
  (invisible-module . ?proofs)
  t
  (progn . ?proofs))

;********************************************************************
;The macros let and insulate
;********************************************************************

(defmac let-no-args
   (let-proof () . ?body)
   t
   (progn . ?body))

(defmac let-multiple-args
   (let-proof ((?pattern ?bind) . ?rest-args) . ?body)
   t
   (let-sequent (?pattern ?bind)
     (let-proof ?rest-args . ?body)))

(defmac insulate
  (insulate . ?body)
  t
  (show-internal (true) . ?body))

(defextender apply (fun args)
  (apply fun args))

(defsequent apply-sequent
    ((sequent ((apply ?fun ?args))
       (theorem ?psi)))
  (theorem ?psi))

(defmac apply
  (apply ?fun ?args . ?body)
  t
  (apply-sequent
   (sequent ((apply ?fun ?args))
     (progn . ?body))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Read-eval-print code:                                                   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Protected variables should go here.
(defun enter-rep-sbhlps (&optional (stream t))
  (let ((*visible-evaluation?* nil)
	(*faith-mode* *faith-mode*)
	(*location* *location*)
	(*backtrackable* *backtrackable*)
	(*goal* *goal*)
	(*sequent-cache* (let ((ret (make-hash-table :test #'equal)))
			   (maphash #'(lambda (key val)
					(setf (gethash key ret) val))
				    *sequent-cache*)
			   ret))
	(*proof-cache* (let ((ret (make-hash-table :test #'equal)))
			 (maphash #'(lambda (key val)
				      (setf (gethash key ret) val))
				  *proof-cache*)
			 ret))
	(*executep* *executep*)
	(*error-break* *error-break*)
	(*context* *context*)
	(*last-assumption-max-const* *last-assumption-max-const*))
    (when (and (not *rep-seen*) *visible-evaluation?*)
      ;; Figure out how to make ilisp print the first prompt.
      (setq *rep-seen* t))
    (format stream "~%")
    (ontic-rep stream)))

(defun ontic-rep (&optional (stream t))
  (let ((form nil)
	(return nil)
	(exit nil)
	(throw nil))
    (tagbody continue
      (unwind-protect
	  (loop
	    ;; Read
	    (if (contradiction?)
		(format t "**> ")
		(format t "--> "))
	    (setq form (read stream))
	    ;; Eval
	    (cond ((or (eq form :cont)
		       (eq form :continue))
		   (setq exit t)
		   (return))
		  ((eq form :quit)
		   (setq throw t)
		   (return))
		  (t
		    (setq return (eval form))))
	    ;; Print
	    (format t "~&~a~%" return)
	    (setq +++ ++)
	    (setq ++ +)
	    (setq + form)
	    (setq *** **)
	    (setq ** *)
	    (setq * return)
	    (setq /// //)
	    (setq // /)
	    (setq / (list return)))
	(cond (throw (throw 'emacs nil))
	      (exit nil)
	      (t (go continue)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Emacs Faith-mode control code:                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro with-faith (&rest body)
  `(let ((*faith-mode* t))
     ,@body))

(defmac without-fath
  (without-faith . ?body)
  t
  (lisp-bind *faith-mode* nil
    (progn . ?body)))

(defmac with-faith
  (with-faith . ?body)
  t
  (lisp-bind *faith-mode* t
    (progn . ?body)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Quieting Code:                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro without-beeping (&rest body)
  `(let ((ret nil))
     (unwind-protect
	 (progn
	   (setq *quiet-mode* t)
	   (setq ret (progn ,@body)))
       (setq *quiet-mode* nil))
     ret))

(defmacro with-region (&rest body)
  `(let ((ret nil))
     (unwind-protect
	 (progn
	   (setq *region-mode* t)
	   (setq ret (progn ,@body)))
       (setq *region-mode* nil))
     ret))

(defmacro with-batch (&rest body)
  `(let ((ret nil)
	 (ve *visible-evaluation?*))
     (unwind-protect
	 (progn
	   (setq *batch-mode* t)
	   (setq *visible-evaluation?* nil)
	   (setq ret (progn ,@body)))
       (setq *batch-mode* nil)
       (setq *visible-evaluation?* ve))
     ret))



;********************************************************************
;Control assertions
;********************************************************************

;A control assertion, such as (classify! ?x), has no import other than
;to control the inference process.

;Each control assertion should be associated with a sequent rule that
;eliminates that control assertion.

;(control-predicate classify!)
;
;abbreviates
;
;(defsequent eliminate-classify!
;    ((sequent ((axiom (classify! ?x)))
;       (theorem ?phi)))
;  (theorem ?phi))
;
;(defmac classify!-macro-0
;  (classify! ?x . ?body)
;  t
;  (eliminate-classify!
;   (sequent ((axiom (classify! ?x)))
;       (first (ensure *goal*)
;	       (progn . ?body)))))

(defmacro control-predicate (pred)
  `(progn
    (defsequent ,(create-name 'eliminate pred)
	((sequent ((axiom (,pred ?x)))
	   (theorem ?phi)))
      (theorem ?phi))

    (defmac ,(create-name pred 'macro-0)
      (,pred ?x . ?body)
      t
      (,(create-name 'eliminate pred)
       (sequent ((axiom (,pred ?x)))
	 (first (ensure *goal*)
		(progn . ?body)))))

    (emacs-indent ,pred 1)

    (defsequent ,(create-name 'recursive-eliminate pred)
	((sequent ((recursively-axiom ,pred ?x))
	   (theorem ?phi)))
      (theorem ?phi))

    (defmac ,(create-name pred 'macro-1)
      (,(create-name 'recursively pred) ?x . ?body)
      t
      (,(create-name 'recursive-eliminate pred)
	(sequent ((recursively-axiom ,pred ?x))
	  (first (ensure *goal*)
		 (progn . ?body)))))

    (emacs-indent ,(create-name 'recursively pred) 1)))

(defmacro control-extender (extender)
  `(progn
    (defsequent ,(create-name 'eliminate extender)
	((sequent ((,extender ?x))
	   (theorem ?phi)))
      (theorem ?phi))

    (defmac ,(create-name extender 'macro-0)
      (,extender ?x . ?body)
      t
      (,(create-name 'eliminate extender)
       (sequent ((,extender ?x))
	 (progn . ?body))))

    (emacs-indent ,extender 1)))


(defextender recursively-apply (function expression &rest args)
  (recursively-apply-2 function (translate expression) args))

(defun recursively-apply-2 (function texp args)
  (apply function texp args)
  (when (consp texp)
    (mapc (lambda (exp)
	    (recursively-apply-2 function exp args))
	  (rest texp))))