;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic :use '(util lisp))
;This is the rule compiler.  The code is organized into sections.  Appropriate comments
;are given for each individual section.


;========================================================================
;rule constants
;========================================================================

;(def-rule-const ?foo (the-foo-expression))
;
;rule constants are varaibles that appear in rules but which are bound
;globally, i.e., outside the rule.  All rule constants must be declared
;using def-rule-const.  If a rule constant is used in a rule the rule
;must contain an antecedent of the form (constant ?foo) where ?foo is the
;rule constant.  The need for these antecedents is historical and they should
;eventually be eliminated.

(defpiecefun make-immortal! (node))

(defmacro def-rule-const (variable expression)
  (unless (variable? variable)
    (error "attempt to declare constant without initial '?': ~s" variable))
  (let ((cat (check-expression expression)))
    `(eval-when (compile load eval)
      (pushnew ',variable *constants*)
      (setf *variables* (delete ',variable *variables*))
      (setf (constant-expression ',variable) ',expression)
      (setf (variable-category ',variable) ',cat)
      (defslot-bit ,cat ,(create-name cat 'constant?))
      (defpiece (equate-constant?-2 ,(create-name 'check cat)) (node)
	(when (and (eq (type-of node) ',cat)
		   (= (,(create-name cat 'constant?) node) 1))
	  (throw 'constant t)))
      (defpiece (make-immortal! ,cat) (node)
	(when (eq (type-of node) ',cat)
	  (setf-undo (,(create-name cat 'constant?) node) 1)))
      (defpiece (ontic-init-phase0 ,(create-name 'clear variable)) ()
	(setf (constant-value ',(create-name 'constant variable)) nil))
      (defun ,(create-name 'constant variable) ()
	(let ((val (tintern ',expression)))
	  (setf-undo (constant-value ',(create-name 'constant variable)) val)
	  (make-immortal! val)
	  val)))))



;========================================================================
;rules
;========================================================================
;(rule <rule-name> <antecedents> <conclusion>)
 
(defmacro rule (name &optional antecedents &body conclusions)
  (setq antecedents (expand-rule-antecedent-macros antecedents))
  (cond ((and (null antecedents) (null conclusions))
	 `(eval-when (eval load compile)
	   (clear-compiled-rule ',name)
	   (setf *rules* (remove ',name *rules*))
	   (defun ,name (&rest ignore) nil)))
	((cdr conclusions)
	 `(eval-when (load eval compile)
	   ,@(let ((n 0))
	       (mapcar #'(lambda (conc)
			   (incf n)
			   `(rule ,(create-name name 'conclusion (format nil "~s" n))
			     ,antecedents
			     ,conc))
		       conclusions))))		       
	(t
	 (install-rule name antecedents (car conclusions))
	 `(eval-when (eval load compile)
	   (clear-compiled-rule ',name)
	   (install-rule ',name ',antecedents ',(car conclusions))
	   ,@(rule-code name)
	   (setf (rule-count ',name) 0)
	   ))))

(defmacro interpreted-rule (name &optional antecedents &body conclusions)
  (setq antecedents (expand-rule-antecedent-macros antecedents))
  (cond ((and (null antecedents) (null conclusions))
	 `(eval-when (eval load compile)
	   (clear-interpreted-rule ',name)
	   (setf *rules* (remove ',name *rules*))
	   (defun ,name (&rest ignore) nil)))
	((cdr conclusions)
	 `(eval-when (load eval compile)
	   ,@(let ((n 0))
	       (mapcar #'(lambda (conc)
			   (incf n)
			   `(interpreted-rule ,(create-name name 'conclusion (format nil "~s" n))
			     ,antecedents
			     ,conc))
		       conclusions))))		       
	(t
	 (install-rule name antecedents (car conclusions))
	 `(eval-when (eval load compile)
	   (clear-interpreted-rule ',name)
	   (install-rule ',name ',antecedents ',(car conclusions))
	   ,@(rule-code name :compiled? nil)
	   (setf (rule-count ',name) 0)
	   ))))




;========================================================================
;installing rules
;========================================================================

(defstruct (rule (:print-function
		   (lambda (self stream &rest ignore)
		     ignore ;avoids compiler warnings
		     (format stream "[rule ~s]" (rule-name self))))
		 (:conc-name nil))
  rule-name
  antecedents
  conclusion)

(defvar *rule-table* (make-hash-table))

(defvar *rules* nil)

(defun find-rule (rule-name)
  (or (gethash rule-name *rule-table*)
      (let ((new-rule (make-rule :rule-name rule-name)))
	(setf (gethash rule-name *rule-table*) new-rule)
	(pushnew rule-name *rules*)
	new-rule)))

(property-macro congruence-rule)
(property-macro rule-variables)

(defun install-rule (rule-name antecedents conclusion)
  (check-rule antecedents conclusion)
  (let ((rule (find-rule rule-name)))
    (setf (antecedents rule) antecedents)
    (setf (conclusion rule) conclusion))
  (mvlet (((antecedents conclusion) (congruence-conversion antecedents conclusion)))
    (setf (congruence-rule rule-name) (cons conclusion antecedents))
    (setf (rule-variables rule-name) (variables antecedents))))


(defun check-rule (antecedents conclusion)
  (dolist (ant antecedents)
    (cond ((symbolp ant))
	  ((atom ant) (error "Bad antecedent: ~s" ant))
	  ((eq (car ant) 'when) (check-when-ant ant))
	  ((variable? (car ant)) (error "Bad antecedent: ~s" ant))
	  ((not (eq (car ant) 'constant)) (check-formula ant))))
  (check-conclusion conclusion))

(defun check-when-ant (when-ant)
  (unless (every #'(lambda (notice-form)
		     (and (consp notice-form)
			  (mergefun-p (create-name 'merged (first notice-form)))
			  (every #'(lambda (arg)
				     (and (variable? arg)
					  (variable-category arg)))
				 (rest notice-form))))
		 (cddr when-ant))
    (error "Bad when antecedent ~s" when-ant)))

(defun check-conclusion (conclusion)
  (selectmatch conclusion
    ((:lisp . :anything))
    ((force ?phi)
     (check-formula ?phi))
    ((=intern ?e1 ?e2)
     (check-formula `(= ,?e1 ,?e2)))
    ((intern ?expression)
     (check-expression ?expression))
    ((queue :anything ?conc2)
     (check-conclusion ?conc2))
    (?phi (check-formula ?phi))))

(defun declare-rule-piece (piece-name)
  (setf (get piece-name 'rule-piece) t))

(eval-when (compile load eval)
  (defun rule-piece? (piece-name)
    (get piece-name 'rule-piece)))

(defun show-rules ()
  (mapcar 'rule-expression *rules*))

(defun rule-expression (rule-name)
  (list rule-name (antecedents  (find-rule rule-name)) (conclusion (find-rule rule-name))))



;========================================================================
;congruence-conversion
;========================================================================
;congruence conversion

;To describe the output of congruence conversion some new terminology is needed.

;After congruence conversion, every antecedent is one of the following.
;
;1. An equation (= ?x (op ?y1 ... ?yn)) where ?x and each ?yi are variables
;2. A formula (pred ?x ?y) where ?x and ?y are variables
;3. a formula (pred ?x) where ?x is a variable.
;4. A formula (not (pred ?x)) where ?x is a variable.
;5. An expression of the form (when <test>) or (constant <rule-var>)
;   where <test> is lisp code and <rule-var> is a rule constant.

;
;Formula variables are allowed, but antecedents of the form ?P  and (not ?P)
;are represented as (is-true ?P) and (not (is-true ?P)) where is-true is a
;a distinguished phrase constructor.

;The first antecedent (the head) of the congruence converted rule is guaranteed not
;to have repeated variables.

;After congruence conversion, a conclusion is either a an expresion of the
;form (:lisp <lisp-form>), an equation between variables, or one of 2, 3, or 4 above.
;


(defun congruence-conversion (antecedents conclusion)
  (gensym 1)
  (let* ((antecedents1 (mapcan (lambda (form)
				 (if (and (consp form) (or (eq (car form) 'when)
							   (eq (car form) 'constant)))
				     (list form)
				     (congruence-convert-formula form)))
			       antecedents))
	 (conclusion1 (congruence-convert-conclusion conclusion))
	 (antecedents2 (append antecedents1 (cdr conclusion1)))
	 (conclusion2 (car conclusion1))
	 (equality-antecedents (remove-if-not (lambda (form)
						(and (consp form)
						     (eq (car form) '=)))
					      antecedents2))
	 (find-map (congruence-closure equality-antecedents)))
    (values (type-check-equations
	     (remove-if (lambda (ant) (selectmatch ant ((= ?x ?x) t)))
			(remove-duplicates
			 (sublis find-map antecedents2)
			 :test #'equal)))
	    (sublis find-map conclusion2))))

(defun congruence-convert-conclusion (conclusion)
  (selectmatch conclusion
    ((queue ?qname ?c2)
     (let ((result (congruence-convert-conclusion ?c2)))
       `((queue ,?qname ,(car result)) ,@(cdr result))))
    ((:lisp :anything) (list conclusion))
    ((=intern . :anything) (list conclusion))
    ((force . :anything) (list conclusion))
    ((intern :anything) (list conclusion))
    (:anything (congruence-convert-formula conclusion))))

;The following function reduces the given expression to a single variable.

(defun congruence-convert (expression)
  (cond ((basic-object-p expression)
	 (let ((new-var (intern (string (gensym "?GENSYMED-VAR-")))))
	   (setf (variable-category new-var) (type-of expression))
	   `(,new-var
	     (= ,new-var ,expression))))
	((variable? expression)
	 (list expression))
	(t
	 (let ((congruence-args (mapcar #'congruence-convert
					(cdr expression)))
	       (new-var (intern (string (gensym "?GENSYMED-VAR-")))))
	   (setf (variable-category new-var) (check-expression expression))
	   `(,new-var
	     (= ,new-var (,(car expression) ,@(mapcar #'car congruence-args)))
	     ,@(mapcan #'cdr congruence-args))))))

;The following function replaces top level subexpressions by variables.

(defun congruence-convert2 (exp)
  (selectmatch (congruence-convert exp)
    ((?var (= ?var (:anything)))
     exp)	   
    ((?var (= ?var ?exp2) . ?rest)
     (if (basic-object-p ?exp2)
	 `((= ,?var ,?exp2) ,@?rest)
	 `(,?exp2 ,@?rest)))
    (?x ?x)))

;The following function normalizes an antecedent.

(defun congruence-convert-formula (formula)
  (selectmatch (congruence-convert-formula2 formula)
    (((not ?form) . ?rest)
     (if (symbolp ?form)
	 `((not (is-true ,?form)) ,@?rest)
	 `((not ,?form) ,@?rest)))
    ((?form . ?rest)
     (if (symbolp ?form)
	 `((is-true ,?form) ,@?rest)
	 `(,?form ,@?rest)))
    (?x ?x)))

(defun congruence-convert-formula2 (formula)
  (selectmatch formula
    ((:lisp :anything) (list formula))
    ((when . :anything) (list formula))
    ((constant . :anything) (list formula))
    ((not ?p)
     (cond ((truth-slot? ?p)
	    (selectmatch (congruence-convert2 ?p)
	      ((?p2 . ?rest)
	       `((not ,?p2) ,@?rest))))
	   (t (congruence-convert2 formula))))
    (:anything
      (congruence-convert2 formula))))

(defun truth-slot? (exp)
  (or (symbolp exp)
      (and (= (length exp) 2)
	   (not (eq (car exp) 'not)))))

(defun relation? (exp)
  (and (= (length exp) 3)
       (not (eq (car exp) '=))))

;Whenever the antecedents require to variables ?x and ?y to have the same value
;the rule can be rewritten to use a single variable ?z rather than ?x and ?y.
;A congruence closure procedure is used to determine all forced identities
;between variables in the antecedent set.

;Congruence-convert-formula might return an equation between variables.
;The congruence-closure elimination of redundant variables will eliminate
;all such equations.

;The following procedure assumes that the first argument in each given equation
;is a variable.

(defun congruence-closure (equations)
  (let ((find-alist nil)
	(congruence-alist nil))
    (dolist (equation equations)
      (let ((e1 (second equation))
	    (e2 (third equation)))
	(if (variable? e2)
	    (setq find-alist (alist-union e1 e2 find-alist))
	    (push (cons e2 e1) congruence-alist))))
    (congruence-closure2 find-alist congruence-alist)))

(defun alist-find (var alist)
  (or (cdr (assoc var alist))
      var))

(defun alist-union (var1 var2 alist)
  (let ((v1 (alist-find var1 alist))
	(v2 (alist-find var2 alist)))
    (if (eq v1 v2)
	alist
	(mvlet (((v2 v1)
		 (cond ((sub-category? (variable-category v1) (variable-category v2))
			(values v1 v2))
		       ((sub-category? (variable-category v2) (variable-category v1))
			(values v2 v1))
		       (t (error "Type clash in rule")))))
	  (acons v1 v2 (mapcar (lambda (cell)
				 (if (eq (cdr cell) v1)
				     (cons (car cell) v2)
				     cell))
			       alist))))))

(defun congruence-closure2 (find-alist congruence-alist)
  (let ((new-congruence-alist (sublis find-alist congruence-alist)))
    (mvlet (((v1 v2 smaller-congruence-alist) (find-equation new-congruence-alist)))
      (if v1
	  (congruence-closure2 (alist-union v1 v2 find-alist)
			      smaller-congruence-alist)
	  find-alist))))

(defun find-equation (congruence-alist)
  (if (null congruence-alist)
      (values nil nil nil)
      (let* ((first-cell (car congruence-alist))
	     (redundant-cell (assoc (car first-cell)
				    (cdr congruence-alist)
				    :test #'equal)))
	(if redundant-cell
	    (values (cdr first-cell) (cdr redundant-cell) (cdr congruence-alist))
	    (mvlet (((v1 v2 rest-alist) (find-equation (cdr congruence-alist))))
	      (values v1 v2 (cons first-cell rest-alist)))))))

(defun type-check-equations (antecedents)
  (mapcan (lambda (ant)
	    (selectmatch ant
	      ((= ?node (= ?arg1 ?arg2))
	       (let ((type (check-expression ?arg1)))
		 (unless (eq type (check-expression ?arg2))
		   (error "an equality with mismatched types"))
		 `((= ,?node (= ,?arg1 ,?arg2))
		   (when (,(create-name type 'p) ,?arg1)))))
	      (?x `(,?x))))
	  antecedents))


;========================================================================
;rule-code
;========================================================================
;This is the function that generates the code for a rule.
;rule expansion is done in two phases so that macroepxanding a rule
;give a human readable intermediate form with useful information such as
;the estimated cost of each piece of the rule.  Cost estimates are only
;provided in empirical sorting mode.  To get into empirical sorting
;mode one must first go to a complex context to use as a benchmark in
;measuring cost. One then sets the variable *sort-empirically* to t.

(defvar *sort-empirically* nil)

(defvar *constant-vars* nil)

(defvar *compiling-rules* nil)

(defun rule-code (rule-name &key (compiled? t))
  (when *sort-empirically*
    (format t "~%compiling rule ~s" rule-name))
  (let ((rule (congruence-rule rule-name)))
    (let ((pieces (make-pieces rule-name rule)))
      (append (mapcar (lambda (piece)
			(if compiled?
			    `(rulepiece ,rule-name
			      ,(first piece) ,(third piece) ,(cdddr piece) ,(second piece))
			    `(interpreted-rulepiece ,rule-name
			      ,(first piece) ,(third piece) ,(cdddr piece) ,(second piece))))
		      pieces)
	      (list (justification-definition rule-name))))))

(defun make-pieces (rule-name congruence-rule)
  (read-sort-cache)
  (let ((antecedents (cdr congruence-rule))
	(consequent (car congruence-rule)))
    (unless (consequent-bound? consequent (variables antecedents))
      (error "Unbound consequent in rule:  ~s" rule-name))
    (let* ((constant-antecedents (mapcar (lambda (constant) `(constant ,constant))
					 (intersection (union (variables antecedents)
							      (variables consequent))
						       *constants*)))
	   (non-constant-antecedents (set-difference antecedents constant-antecedents :test #'equal)))
      (when (some (lambda (ant) (matches? ant (constant :anything)))
		  non-constant-antecedents)
	(error "undeclared constant"))
      (when *sort-empirically*
	(dolist (const constant-antecedents)
	  (proclaim `(special ,(second const)))
	  (let ((expression (constant-expression (second const))))
	    (when (null expression)
	      (error "there is no definition of the constant ~s" (second const)))
	    (set (second const) (ti expression)))
	  (push (second const) *constant-vars*)))
      (mvlet (((heads real-antecedents) (simplify-antecedents non-constant-antecedents)))
	(let ((pieces (mapcar #'(lambda (head)
				  (let* ((res
					  (sort-antecedents
					   rule-name
					   head
					   (remove head real-antecedents)
					   (variables (cons head constant-antecedents))
					   (remove '?justification (variables consequent))))
					 (sorted-antecedents (first res))
					 (comment (second res)))
				    (list* comment
					   consequent
					   head
					   (append constant-antecedents sorted-antecedents))))
			      heads)))
	  (unless *compiling-rules* (save-sort-cache))
	  pieces)))))

(defun consequent-bound? (conseq vars)
  (totally-bound? conseq
		  (selectmatch conseq
		    ((:lisp . :anything) (cons '?justification vars))
		    ((queue :anything (:lisp . :anything)) (cons '?justification vars))
		    (:anything vars))))

(defun simplify-antecedents (antecedents)
  (let ((output-ants nil)
	(heads nil))
    (dolist (ant antecedents)
      (selectmatch ant
	((when ?test . ?heads)
	 (dolist (head2 ?heads)
	   (push `(:lisp ,head2) heads))
	 (push `(when ,?test) output-ants))
	(:anything
	 (push ant output-ants)
	 (push ant heads))))		   
    (values heads output-ants)))

(defmacro rulepiece (name comment head antecedents conclusion)
  (declare (ignore comment))
  `(eval-when (load eval compile)
    ,@(build-defmergepiece name head antecedents conclusion)))

(emacs-indent rulepiece 3)

(defmacro interpreted-rulepiece (name comment head antecedents conclusion)
  (declare (ignore comment))
  `(eval-when (load eval compile)
    ,@(build-interppiece name head antecedents conclusion)))

(emacs-indent interpreted-rulepiece 3)

(property-macro piece-owner)

(defun build-defmergepiece (rule-name head antecedents conclusion)
  (let* ((piece-name (gentemp (symbol-name rule-name)))
	 (low-level-head (low-level-head head))
	 (real-head (car low-level-head))
	 (body (reduce-piece (append (cdr low-level-head)
				     (low-level-antecedents antecedents (variables head))
				     (make-conclusion conclusion
						      rule-name
						      (variables (cons head antecedents)))))))
    `((declare-rule-piece ',piece-name)
      (setf (piece-owner ',piece-name) ',rule-name)
      (defmergepiece (,(create-name 'merged (car real-head)) ,piece-name) ,(cdr real-head)
       ,@body))))

(defun build-interppiece (rule-name head antecedents conclusion)
  (let* ((piece-name (gentemp (symbol-name rule-name)))
	 (low-level-head (low-level-head head))
	 (real-head (car low-level-head))
	 (body (reduce-piece (append (cdr low-level-head)
				     (low-level-antecedents antecedents (variables head))
				     (make-conclusion conclusion
						      rule-name
						      (variables (cons head antecedents)))))))
    `((declare-rule-piece ',piece-name)
      (setf (piece-owner ',piece-name) ',rule-name)
      (definterppiece (,(create-name 'merged (car real-head)) ,piece-name) ,(cdr real-head)
       ,@body))))

(defun reduce-piece (piece)
  (when piece
    (let ((tail (reduce-piece (cdr piece)))
	  (head (car piece)))
      (selectmatch head
	((let ?var :anything)
	 (if (internal-member ?var tail)
	     (cons head tail)
	     tail))
	((dolist ?var ?exp)
	 (if (internal-member ?var tail)
	     (cons head tail)
	     (cons `(when ,?exp) tail)))
	(:anything
	  (cons head tail))))))

(defun clear-compiled-rule (rule-name)
  (clear-mergepieces
   (lambda (piece-name) (eq (piece-owner piece-name) rule-name))))

(defun clear-interpreted-rule (rule-name)
  (clear-interppieces
   (lambda (piece-name) (eq (piece-owner piece-name) rule-name))))

(defun show-rule-pieces (rule-name)
  (show-mergepieces
   (lambda (piece-name) (eq (piece-owner piece-name) rule-name))))



;========================================================================
;low-level-head
;========================================================================

(defun low-level-head (head)
  (let* ((messaged-head (message-head head))
	 (safe-head (car messaged-head))
	 (head-equality-antecedents (cdr messaged-head))
	 (low-level-head-internal (low-level-head-internal safe-head)))
    (append low-level-head-internal head-equality-antecedents)))

;(message-head (foo ?x ?x)) is ((foo ?x ?y) (when (eq ?x ?y)))
;(message-head (= ?x (f ?x ?z))) is ((= ?x (f ?y ?z)) (when (eq ?x ?y)))
;message-head ensures that all the variables inthe head are distinct.
;This allows the head to be converted to a piece with distinct argument
;names for the lambda-list of the propagator function.

(defun message-head (head)
  (mvlet (((new-head when-conditions) (message-expression head nil)))
    (cons new-head when-conditions)))

(defun message-expression (expression bound-vars)
  (cond ((variable? expression)
	 (if (member expression bound-vars)
	     (let ((new-var (gentemp "?VAR-")))
	       (setf (variable-category new-var) (check-expression expression))
	       (values new-var `((when (eq ,expression ,new-var))) bound-vars))
	     (values expression nil (adjoin expression bound-vars))))
	((not (consp expression))
	 (values expression nil bound-vars))
	(t
	 (mvlet (((car-expression car-whens car-bound-vars)
		  (message-expression (car expression) bound-vars)))
	   (mvlet (((cdr-expression cdr-whens cdr-bound-vars)
		    (message-expression (cdr expression) car-bound-vars)))
	     (values (cons car-expression cdr-expression)
		     (nconc car-whens cdr-whens)
		     cdr-bound-vars))))))

(defun low-level-head-internal (head)
  (let ((?constructor-var (gentemp "?CONSTRUCTOR-")))
    (selectmatch head

      ((:lisp (?fun . ?args))
       `((,?fun ,@?args) ,@(mapcar (lambda (arg) `(when (,(create-name (variable-category arg) 'p) ,arg)))
			    ?args)))

      ((= ?x (?constructor . ?args))
       (case (length ?args)
	 (0 `((notice-nullary-production ,?x ,?constructor-var)
	      ,@(if (variable? ?constructor)
		    `((let ,?constructor ,?constructor-var)
		      (when (,(create-name (output-category ?constructor) 'p) ,?x)))
		    `((when (eq ',?constructor ,?constructor-var))))
	      ,@(possibly-check-category ?x (output-category ?constructor))))
	 (1 `((notice-unary-production ,?x ,?constructor-var ,@?args)
	      ,@(if (variable? ?constructor)
		    `((let ,?constructor ,?constructor-var)
		      (when (,(create-name (output-category ?constructor) 'p) ,?x))
		      (when (,(create-name (first (argument-categories ?constructor)) 'p) ,(first ?args))))
		    `((when (eq ',?constructor ,?constructor-var))))
	      ,@(possibly-check-category ?x (output-category ?constructor))
	      ,@(possibly-check-category (first ?args) (first (argument-categories ?constructor)))))
	 (2 `((notice-binary-production ,?x ,?constructor-var ,@?args)
	      ,@(if (variable? ?constructor)
		    `((let ,?constructor ,?constructor-var)
		      (when (,(create-name (output-category ?constructor) 'p) ,?x))
		      (when (,(create-name (first (argument-categories ?constructor)) 'p)
			      ,(first ?args)))
		      (when (,(create-name (second (argument-categories ?constructor)) 'p)
			      ,(second ?args))))
		    `((when (eq ',?constructor ,?constructor-var))))
	      ,@(possibly-check-category ?x (output-category ?constructor))
	      ,@(possibly-check-category (first ?args) (first (argument-categories ?constructor)))
	      ,@(possibly-check-category (second ?args) (second (argument-categories ?constructor)))))))

      ((when ?test ?notice-form)
       `(,?notice-form
	 ,@(mapcar #'(lambda (arg) `(when (,(create-name (check-expression arg) 'p) ,arg)))
	    (rest ?notice-form))
	 (when ,?test)))

      ((?pred ?arg1 ?arg2)
       `((,(create-name 'notice ?pred) ,?arg1 ,?arg2)
	 ,@(possibly-check-category ?arg1 (first (argument-categories ?pred)))
	 ,@(possibly-check-category ?arg2 (second (argument-categories ?pred)))))

      ((not (?pred ?arg))
       `((,(create-name 'notice-false ?pred) ,?arg)
	 ,@(possibly-check-category ?arg (first (argument-categories ?pred)))))

      ((?pred ?arg)
       `((,(create-name 'notice-true ?pred) ,?arg)
	 ,@(possibly-check-category ?arg (first (argument-categories ?pred)))))

      (:anything
       (error "unrecognized head in rule piece")))))

(defun possibly-check-category (var category)
  (unless (eq (variable-category var) category)
    `((when (,(create-name (variable-category var) 'p) ,var)))))



;========================================================================
;low-level-antecedent
;========================================================================

(defun low-level-antecedent (ant bound-vars)
  (selectmatch ant

    ((when ?form) `((when ,?form)))

    ((constant ?const)
     (let ((const-name (create-name 'constant ?const)))
       (if (member ?const bound-vars)
	   `((when (eq ,?const (or (constant-value ',const-name)
				   (,const-name)))))
	   `((let ,?const (or (constant-value ',const-name)
			      (,const-name)))))))

    ((= ?w (?constructor . ?args))
     (strong-piece ?w ?constructor ?args bound-vars))

    ((?pred ?arg1 ?arg2)
     (if (member ?arg1 bound-vars)
	 (if (member ?arg2 bound-vars)
	     `((when (member ,?arg2 (,(create-name ?pred 'forward 'internal)
				      ,?arg1))))
	     `((dolist ,?arg2 (,(create-name ?pred 'forward 'internal) ,?arg1))
	       ,@(possibly-check-category ?arg2 (second (argument-categories ?pred)))))
	 `((dolist ,?arg1 (,(create-name ?pred 'backward 'internal) ,?arg2))
	   ,@(possibly-check-category ?arg1 (first (argument-categories ?pred))))))

    ((not (?pred ?arg))
     `((when (,(create-name-internal ?pred 'false) ,?arg))))

    ((?pred ?arg)
     `((when (,(create-name-internal ?pred 'true) ,?arg))))))

(defun strong-piece (?w ?constructor ?args bound-vars)
  (let ((?s (gentemp "?STRONG-NODE-")))
    (if (member ?w bound-vars)
	`((dolist ,?s (productions-from ,?w))
	  ,@(destructuring-piece ?s ?constructor ?args (cons ?s bound-vars)))
	(let ((subvar (find-if (lambda (var) (member var bound-vars))
			       (reverse ?args))))	;avoids function position.
	  (if subvar
	      `((dolist ,?s (productions-to ,subvar))
		(let ,?w (lhs ,?s))
		,@(possibly-check-category ?w (output-category ?constructor))
		,@(destructuring-piece ?s ?constructor ?args (list* ?s ?w bound-vars)))
	      `((let ,?w (,(create-name 'make ?constructor)))
		,@(possibly-check-category ?w (output-category ?constructor))))))))

(defun destructuring-piece (?production ?constructor ?rhs bound-vars)
  `(,@(cond ((not (variable? ?constructor))
	     `((when (has-constructor? ,?production ',?constructor))))
	   ((member ?constructor bound-vars)
	    `((when (has-constructor? ,?production ,?constructor))))
	   (t `((let ,?constructor (phrase-constructor ,?production))
		(when (= ,(length ?rhs) (length (rhs ,?production)))))))
    ,@(bind-list ?rhs (argument-categories ?constructor) `(rhs ,?production) bound-vars
       (variable? ?constructor))
    ,@(when (variable? ?constructor)
	`((when (,(create-name (output-category ?constructor) 'p) (lhs ,?production)))))))

(definline has-constructor? (strong-node constructor)
  (eq (phrase-constructor strong-node) constructor))

(defun bind-list (variables constructor-arg-types list-form bound-vars variable-constructor?)
  (when variables
    (let ((listvar (gentemp "?LIST-")))
      `((let ,listvar ,list-form)
	,@(if (member (car variables) bound-vars)
	     `((when (eq ,(car variables) (car ,listvar))))
	     `((let ,(car variables) (car ,listvar))
	       ,@(possibly-check-category (car variables) (car constructor-arg-types))))
	,@(when variable-constructor?
	    `((when (,(create-name (first constructor-arg-types) 'p) ,(car variables))))) 
	,@(bind-list
	   (cdr variables)
	   (cdr constructor-arg-types)
	   `(cdr ,listvar)
	   (cons (car variables) bound-vars)
	   variable-constructor?)))))

      

;========================================================================
;low-level-conclusion
;========================================================================
(defun make-conclusion (conclusion name bound-vars)
  `((let ?justification (when *record-justifications*
			  (,(create-name 'make name 'justification) ,@(rule-variables name))))
    (progn ',name
	   (when (not-bogus? ,@(remove-if-not #'node-variable? bound-vars))
	     (incf (rule-count ',name))
	     ,(make-conclusion-body conclusion bound-vars '?justification)))))

;the rule-count property counts the number of times a given rule fires.
;the macro must be defined here because it is used 

(property-macro rule-count)

(defun clear-rule-counts ()
  (mapc (lambda (rule)
	  (setf (rule-count rule) 0))
	*rules*)
  (values))

(defpiece (ontic-init-phase0 clear-rule-counts) ()
  (clear-rule-counts))

(defun make-conclusion-body (conclusion bound-vars just-var)
  (selectmatch conclusion
    ((queue ?qname ?conclusion)
     `(ontic-queue-with ,(intersection (variables ?conclusion) bound-vars)
       ,?qname
       ,(make-conclusion-body ?conclusion nil just-var)))
    ((:lisp ?form) ?form)
    ((intern ?expression)
     `(if *record-justifications*
       (cintern ,(simple-compile-time-substitution ?expression) :justification ,just-var)
       ,(build-intern-code ?expression)))
    ((force (not (?pred ?arg)))
      `(ontic-queue-with ,(intersection (variables conclusion) bound-vars)
	*default-q*
	(if (null *record-justifications*)
	    (,(create-name 'assert ?pred) ,(build-intern-code ?arg) :false)
	    (let ((exp ,(build-intern-expression ?arg)))
	      (mvlet (((node frame) (cintern exp :justification ,just-var)))
		(,(create-name 'assert ?pred) node :false
		  :justification (add-frames ,just-var frame)))))))
    ((force (not ?phi))
      `(ontic-queue-with ,(intersection (variables conclusion) bound-vars)
	*default-q*
	(if (null *record-justifications*)
	    (assert-is-true ,(build-intern-code ?phi) :false)
	    (let ((exp ,(build-intern-expression ?phi)))
	      (mvlet (((node frame) (cintern exp :justification ,just-var)))
		(assert-is-true node :false :justification (add-frames ,just-var frame)))))))
    ((force (= ?e1 ?e2))
     (make-conclusion-body `(=intern ,?e1 ,?e2) bound-vars just-var))
    ((=intern ?e1 ?e2)
     `(ontic-queue-with ,(intersection (variables conclusion) bound-vars)
       *default-q*
       (if (null *record-justifications*)
	   ,(build-=intern-code ?e1 ?e2)
	   (let ((exp1 ,(build-intern-expression ?e1))
		 (exp2 ,(build-intern-expression ?e2)))
	     (mvlet (((node1 frame1) (cintern exp1 :justification ,just-var))
		     ((node2 frame2) (cintern exp2 :justification ,just-var)))
	       (equate! node1 node2
			:justification
			(if (or frame1 frame2)
			    (make-justification '=intern-after-intern
			      (cons (make-invocation-frame (=intern exp1 exp2)
				      ,just-var)
				    (cons-when frame1 (cons-when frame2 nil))))
			    ,just-var)))))))
    ((force (?pred ?arg))
      `(ontic-queue-with ,(intersection (variables conclusion) bound-vars)
	*default-q*
	(if (null *record-justifications*)
	    (,(create-name 'assert ?pred) ,(build-intern-code ?arg) :true)
	    (let ((exp ,(build-intern-expression ?arg)))
	      (mvlet (((node frame) (cintern exp :justification ,just-var)))
		(,(create-name 'assert ?pred) node :true
		  :justification (add-frames ,just-var frame)))))))
    ((force (?pred ?arg1 ?arg2))
      `(ontic-queue-with ,(intersection (variables conclusion) bound-vars)
	*default-q*
	(if (null *record-justifications*)
	    (,(create-name 'assert ?pred) ,(build-intern-code ?arg1) ,(build-intern-code ?arg2))
	    (let ((exp1 ,(build-intern-expression ?arg1))
		  (exp2 ,(build-intern-expression ?arg2)))
	      (mvlet (((node1 frame1) (cintern exp1 :justification ,just-var))
		      ((node2 frame2) (cintern exp2 :justification ,just-var)))
		(,(create-name 'assert ?pred) node1 node2
		  :justification (add-frames ,just-var frame1 frame2)))))))
    ((force . :anything)
     (error "unrecognized conclusion form ~s" conclusion))
    ((= ?var1 ?var2) `(equate! ,?var1 ,?var2 :justification ,just-var))
    ((?pred ?arg1 ?arg2)
     `(,(create-name-internal 'assert ?pred)
       ,?arg1 ,?arg2 :justification ,just-var))
    ((not (?pred ?x))
     `(,(create-name-internal 'assert ?pred) ,?x :false :justification ,just-var))
    ((?pred ?x)
     `(,(create-name-internal 'assert ?pred) ,?x :true :justification ,just-var))))

 
(defmacro ontic-queue-with ((&rest vars) priority &rest body)
  `(queue-with ,vars ,priority
      (when (not-bogus? ,@vars)
	,@body)))

(emacs-indent queue 1)
 
(defun build-=intern-code (e1 e2)
  (if (and (variable? e1) (variable? e2))
      `(equate! ,e1 ,e2)
      (if (variable? e2)
	  (build-=intern-code e2 e1)
	  (if (variable? e1)
	      `(add-production ,e1 ',(car e2) (list ,@(mapcar #'build-intern-code (cdr e2))))
	      `(equate-productions
		',(car e1) (list ,@(mapcar #'build-intern-code (cdr e1)))
		',(car e2) (list ,@(mapcar #'build-intern-code (cdr e2))))))))

(defun build-intern-code (expression)
  (cond ((variable? expression)
	 expression)
	(t
	 `(hashlist ',(car expression)
	   (list ,@(mapcar #'build-intern-code (cdr expression)))))))

(defun build-intern-expression (expression)
  (cond ((variable? expression)
	 expression)
	(t
	 `(cons ',(car expression)
	   (list ,@(mapcar #'build-intern-expression (cdr expression)))))))



;========================================================================
;Justifications
;========================================================================
;The basic structure of justifications is described in the file congruence.

(defun justification-definition (rule-name)
  `(defun ,(create-name 'make rule-name 'justification) ,(rule-variables rule-name)
    (rule-justification ',rule-name
     (mapcar #'cons
      ',(rule-variables rule-name)
      (list ,@(rule-variables rule-name))))))

(defun rule-justification (rule-name bindings)
  (let ((bindings (mapcar #'cons
			  (rule-variables rule-name)
			  (mapcar #'cdr bindings))))
    (make-justification rule-name
      (mapcan (lambda (ant)
		(let ((new-subst (mapcar (lambda (binding) (cons (car binding) (object-definition (cdr binding))))
					 bindings)))
		  (let ((ant-expression (sublis new-subst ant)))
		    (unless (matches? ant-expression (constant :anything))
		      (selectmatch (translate ant-expression)
			((= ?x ?x)
			 (list (cons (antecedent-just ant bindings)
				     `(definition-of ,(macro-invert ?x)))))
			(?trans
			 (list (cons (antecedent-just ant bindings)
				     (macro-invert ?trans)))))))))
	      (cdr (congruence-rule rule-name))))))

(defun antecedent-just (ant bindings)
  (cond ((not (listp ant))
	 (gethash (cdr (assoc ant bindings)) is-true-frame-hash-table))
	((eq (car ant) '=)
	 (prod-frame
	   (list* (cdr (assoc (second ant) bindings))
		  (first (third ant))
		  (sublis bindings (cdr (third ant))))))
	((eq (car ant) 'not)
	 (antecedent-just (second ant) bindings))
	((and (member (car ant) *constructors*)
	      (eq (output-category (car ant)) 'formula))
	 (if (cddr ant)
	     (gethash (cons (assoc-value (second ant) bindings)
			    (assoc-value (third ant) bindings))
		      (symbol-value (create-name (car ant) 'frame 'table)))
	     (gethash (cdr (assoc (second ant) bindings))
		      (symbol-value (create-name (car ant) 'frame 'hash 'table)))))))

(defun show-rule ()
  (let* ((current-just (frame-justification *current-frame*))
	 (rule-name (justification-comment current-just))
	 (cong-rule (congruence-rule rule-name))
	 (conclusion (car cong-rule))
	 (antecedents (remove-if (lambda (ant) (matches? ant (constant :anything))) (cdr cong-rule))))
    (rprint (list rule-name 'derives conclusion 'from (add-numbers-list antecedents)))))

(defun add-numbers (objects)
  (let ((n 0))
    (mapcar (lambda (obj)
	      (incf n)
	      (cons n obj))
	    objects)))

(defun add-numbers-list (objects)
  (let ((n 0))
    (mapcar (lambda (obj)
	      (incf n)
	      (list n obj))
	    objects)))

(defun compile-time-substitution (expression)
  (cond ((null (variables expression))
	 `(quote ,expression))
	((variable? expression) `(object-definition ,expression))
	(t `(cons
	     ,(compile-time-substitution (car expression))
	     ,(compile-time-substitution (cdr expression))))))

(defun simple-compile-time-substitution (expression)
  (cond ((null (variables expression))
	 `(quote ,expression))
	((variable? expression) expression)
	(t `(cons
	     ,(simple-compile-time-substitution (car expression))
	     ,(simple-compile-time-substitution (cdr expression))))))

(defun antecedent-just-code (ant)
  (cond ((not (listp ant))
	 `(is-true-frame ,ant))
	((eq (car ant) '=)
	 `(prod-frame
	   (list
	    ,(second ant)
	    ',(first (third ant))
	    ,@(cdr (third ant)))))
	((eq (car ant) 'not)
	 (antecedent-just-code (second ant)))
	((and (member (car ant) *constructors*)
	      (eq (output-category (car ant)) 'formula))
	 `(,(create-name (car ant) 'frame) ,@(cdr ant)))))


;========================================================================
;sorting antecedents
;;========================================================================
;A given single rule can have many "pieces".  Each piece corresponds
;to selecting a single antecedent as the "head" and the propagation code
;for that piece is included in the propagation function for the selected
;head.  In each piece the antecedents other than the head must be sorted.
;The particular sort selected can be critical to the efficiency of the rule.
;The sorting can be done either emirically or nonempirically.
;The default is nonempirical sorting.  To sort empirically one must first
;construct a context to use as a benchmark.  One can then compile files
;containing rule using the function compile-rules just as one would use
;compile-files.  Once a piece has been sorted empirically the sort is stored
;in a chache.  This cache is used to sort the piece in future compilations
;even if those future compilation are not using empirical sorting.
 
(defvar *sort-cache* (make-hash-table :test #'equal))

(defvar *sort-cache-installed* nil)

(defun read-sort-cache ()
  (unless *sort-cache-installed*
    (dolist (form (file-forms *sort-cache-file*))
      (unless (gethash (first form) *sort-cache*)
	(setf (gethash (first form) *sort-cache*)
	      (second form))))
    (setf *sort-cache-installed* t)))

(defun clear-sort-cache ()
  (clrhash *sort-cache*)
  (setq *sort-cache-installed* nil))

(defun compile-rules (filenames &key nocache)
  (let ((*compiling-rules* t))
    (unless (> *node-count* 1000)
      (error "the current context is not large enough for empirical antecedent sorting"))
    (let ((*sort-empirically* t))
      (when nocache
	(clrhash *sort-cache*)
	(setf *sort-cache-installed* t))
      (mapc #'compile-file filenames)
      (when nocache
	(setf *sort-cache-installed* nil)
	(read-sort-cache))
      (save-sort-cache))))

(defvar *sort-file-obsolete* nil)

(defun save-sort-cache ()
  (when *sort-file-obsolete*
    (let ((sort-forms nil))
      (maphash (lambda (key val)
		 (push (list key val) sort-forms))
	       *sort-cache*)
      (write-forms sort-forms *sort-cache-file*))
    (setq *sort-file-obsolete* nil)))

(defun sort-antecedents (rule-name head antecedents bound-vars conclusion-vars)
  (let ((key (list rule-name head antecedents bound-vars conclusion-vars)))
    (or (gethash key *sort-cache*)
	(let ((res (sort-antecedents-nocache
		    rule-name head antecedents bound-vars conclusion-vars)))
	  (when *sort-empirically*
	    (setf *sort-file-obsolete* t)
	    (setf (gethash key *sort-cache*) res))
	  res))))

(defun sort-antecedents-nocache (rule-name head antecedents initial-vars conclusion-vars)
  (if *sort-empirically*
      (empirically-sort-antecedents rule-name head antecedents initial-vars conclusion-vars)
      (list (basic-sort-antecedents rule-name antecedents initial-vars)
	    "basic-sort")))


;this is the non-empirical antecedent sorter

(defun basic-sort-antecedents (rule-name antecedents bound-vars)
  (when antecedents
    (let ((next (get-next antecedents bound-vars)))
      (unless next (error "Disconnected rule:  ~s" rule-name))
      (cons next (basic-sort-antecedents rule-name
					 (remove next antecedents)
					 (union bound-vars (variables next)))))))

(defun get-next (antecedents bound-vars)
  (or (find-if (lambda (ant)
		 (essentially-bound? ant bound-vars))
	       antecedents)
      (find-if (lambda (ant)
		 (and (possibly-next? ant bound-vars)
		      (equality-ant? ant)))
	       antecedents)
      (find-if (lambda (ant)
		 (possibly-next? ant bound-vars))
	       antecedents)))

(defun essentially-bound? (ant bound-vars)
  (or (totally-bound? ant bound-vars)
      (and (equality-ant? ant)
	   (totally-bound? (third ant) bound-vars))))

(defun equality-ant? (ant)
  (and (consp ant) (eq (car ant) '=)))

(defun possibly-next? (exp bound-vars)
  (or (totally-bound? exp bound-vars)
      (selectmatch exp
	((when . :anything) nil)
	((= ?lhs (?constructor . ?rhs))
	 (or (totally-bound? (cons ?constructor ?rhs) bound-vars) ;handles constant case
	     (intersection (cons ?lhs ?rhs) bound-vars)))
	(:anything
	  (intersection (variables exp) bound-vars)))))

(defun totally-bound? (exp bound-vars)
  (every (lambda (var) (or (member var bound-vars)
			   (member var *constants*)))
	 (variables exp)))


(defvar *current-best-cost* nil)

(defun empirically-sort-antecedents (rule-name head antecedents bound-vars conclusion-vars)
  (let ((sorts (all-sorts antecedents bound-vars)))
    (when (null sorts)
      (error "unthreaded rule ~s" rule-name))
    (setq *current-best-cost* nil)
    (let ((running-best nil)
	  (sort-number 0)
	  (sort-costs nil)
	  (basic-sort-cost nil))
      (dolist (sort sorts)
	(let ((sort-cost (antecedent-cost head sort bound-vars conclusion-vars)))
	  ;;the basic sort comes first
	  (when (null running-best)
	    (setf basic-sort-cost sort-cost))
	  (when (or (null running-best)
		    (< sort-cost *current-best-cost*))
	    (setf running-best sort)
	    (setf *current-best-cost* sort-cost))
	  (incf sort-number)
	  (push sort-cost sort-costs)))
      (list
       running-best
       (format nil "the best sort costs ~s.  This is ~s% of the cost of the old sort"
	       *current-best-cost*
	       (floor (* 100.0 (/ (float *current-best-cost*) basic-sort-cost))))))))

(defvar *cost-var* 0)

(defvar *total-firings* 0)

(defun antecedent-cost (head antecedents bound-vars conclusion-vars)
  (let* ((*cost-var* 0)
	 (*total-firings* 0))
    (let* ((cost-fun (antecedent-cost-function head antecedents bound-vars conclusion-vars))
	   (arguments (second cost-fun))
	   (compiled-cost-fun (compile nil cost-fun)))
      (catch 'antecedent-cost
	(map-head head compiled-cost-fun arguments)))
    (values *cost-var* *total-firings*)))

(defvar foo 1)

(defun antecedent-cost-function (head antecedents bound-vars conclusion-vars)
  (let* ((body (if (eq 'formula (output-category (car head)))
		   (antecedent-code (cons head antecedents)
				    (cons (first (variables head))
					  *constant-vars*)
				    conclusion-vars)
		   (antecedent-code antecedents bound-vars conclusion-vars)))
	 (arguments (if (eq 'formula (output-category (car head)))
			(list (first (variables head)))
			(intersection (variables head) (variables body)))))
    `(lambda ,arguments
      ,@body)))

(defun antecedent-code (antecedents bound-vars conclusion-vars)
  (nest-bindings (insert-cost-increments (reduce-piece (nconc
							(low-level-antecedents antecedents bound-vars)
							(list `(progn ,@conclusion-vars)))))))
(defun low-level-antecedents (antecedents bound-vars)
  (when antecedents
    (append (low-level-antecedent (car antecedents) bound-vars)
	    (low-level-antecedents (cdr antecedents) (union (variables (car antecedents))
							    bound-vars)))))

(defun insert-cost-increments (antecedents)
  (if antecedents
      (list* '(setq *cost-var* (1+ *cost-var*))
	     (car antecedents)
	     (insert-cost-increments (cdr antecedents)))
      '((setq *total-firings* (1+ *total-firings*)))))
      
(defun nest-bindings (piece)
  (when piece
    (selectmatch (car piece)
      ((let ?var ?exp)
       `((let ((,?var ,?exp))
	   ,@(nest-bindings (cdr piece)))))
      ((when ?exp)
       `((when ,?exp
	   ,@(nest-bindings (cdr piece)))))
      ((dolist ?var ?form)
       `((dolist (,?var ,?form)
	   ,@(nest-bindings (cdr piece)))))
      (:anything
       (cons (car piece)
	     (nest-bindings (cdr piece)))))))

(defun map-head (head cost-fun bound-vars)
  (dolist (subst (head-substs head))
    (apply cost-fun (mapcar (lambda (var) (cdr (assoc var subst)))
			    bound-vars))
    (when (and *current-best-cost*
	       (> *cost-var* *current-best-cost*))
      (throw 'antecedent-cost nil))))


(defun head-substs (head)
  (if (eq 'formula (output-category (car head)))
      (variable-substs (list (first (variables head))))
      (variable-substs (variables head))))

(defun variable-substs (vars)
  (when (> (length vars) 1)
    (format t "%Warning --- doing an (expensive) unthreaded data base search"))
  (let ((result nil))
    (labels ((continue-on (subst-so-far rest-vars)
	       (if (null rest-vars)
		   (push subst-so-far result)
		   (let ((type-tester (symbol-function (create-name (variable-category (car vars)) 'p))))
		     (map-on-nodes
		      (lambda (node)
			(when (funcall type-tester node)
			  (continue-on (acons (car vars) node subst-so-far)
				       (cdr vars)))))))))
      (continue-on nil vars))
    result))

;the following makes an attempt to generate the best sorts first
;by starting with the standard next piece.  This helps in early pruning
;of bad sorts.

(defun all-sorts (ants bound-vars)
  (let ((sorts nil))
    (labels ((install-sorts (sorted-ants unsorted-ants bound-vars)
	       (if (null unsorted-ants)
		   (push (reverse sorted-ants) sorts)
		 (let ((bound-ant (or (find-if (lambda (ant)
						 (and (consp ant)
						      (= (length ant) 2)
						      (totally-bound? ant bound-vars)))
					       unsorted-ants)
				      (find-if (lambda (ant)
						 (essentially-bound? ant bound-vars))
					       unsorted-ants))))
		   (if bound-ant
		       (use-next bound-ant sorted-ants unsorted-ants bound-vars)
		     (let* ((possible-nexts (remove-if-not (lambda (ant)
							     (possibly-next? ant bound-vars))
							   unsorted-ants))
			    (standard-next (get-next unsorted-ants bound-vars)))
		       (dolist (possible-next (cons standard-next
						    (remove standard-next possible-nexts)))
			 (use-next possible-next sorted-ants unsorted-ants bound-vars)))))))
	     (use-next (next-ant sorted-ants unsorted-ants bound-vars)
	       (install-sorts (cons next-ant sorted-ants)
			      (remove next-ant unsorted-ants)
			      (union (variables next-ant) bound-vars))))
	     
      (install-sorts nil ants bound-vars))
    (reverse sorts)))



;========================================================================
;clear-language
;========================================================================

(defmacro clear-language ()
  `(progn
    (clear-mergepieces 'rule-piece?)
    (clear-pieces 'bnf-piece?)
    (clear-variables)
    (setf *rules* nil)
    (setf *categories* '(basic-object))
    (setf *constructors* nil)
    (clrhash *maker-table*)
    (clrhash *checker-table*)
    (clear-structure-definitions)
    (declare-categories (anything basic-object) ontic-gensym ontic-number formula)
    (bnf (formula (is-true formula)
	  (not formula)
	  (= basic-object basic-object)))))

(defun print-rule-counts (&key all)
  (dolist (i (sort (remove-if-not (lambda (rule)
				    (or all (> (rule-count rule) 0)))
				  *rules*)
		   #'> :key #'(lambda (x) (rule-count x))))
    (format t "~%~s: ~s" i (rule-count i))))





;========================================================================
;antecedent macros
;========================================================================

;the second macro is used to define antecedent macros.
;an antecedent macro is used to abbreviate a set of
;antecedents by a single expression.
;
;(define-antecedent-macro <macro-name> <arglist> <code-to-generate-expansion>)

(property-macro antecedent-macro)

(emacs-indent define-antecedent-macro 2)

(defmacro define-antecedent-macro (name arguments &rest body)
  `(antecedent-macro-fun ',name #'(lambda ,arguments ,@body)))

(defun antecedent-macro-fun (name procedure)
  (setf (antecedent-macro name) procedure))

(defun expand-rule-antecedent-macros (antecedents)
  (when antecedents
    (let((ant (car antecedents))
	 (cdr-expansion (expand-rule-antecedent-macros (cdr antecedents))))
      (if (not (consp ant))
	  (cons ant cdr-expansion)
	  (let ((macro-fun (when (symbolp (car ant))
			     (antecedent-macro (car ant)))))
	    (if macro-fun
		(append (expand-rule-antecedent-macros (apply macro-fun (cdr ant)))
			cdr-expansion)
		(cons ant cdr-expansion)))))))

;A useful rule antecedent macro.

(define-antecedent-macro add-head (expression)
  `((when t ,expression)))

(define-antecedent-macro min= (var expression)
  `((= ,var ,expression)
    (when (and (print-size ,var)
	       ,@(mapcar #'(lambda (arg) `(print-size ,arg)) (cdr expression))
	       (print-size-= (print-size ,var) ,(expression-print-size expression))))
    (add-head (notice-print-size ,var))
    ,@(mapcar #'(lambda (arg) `(add-head (notice-print-size ,arg)))
       (cdr expression))))


(defun expression-print-size (expression)
  (selectmatch expression
    ((?const)
     `(constructor-weight ,?const))
    ((?const . ?args)
     `(print-size-+ (constructor-weight ,(if (variable? ?const) ?const `',?const))
       ,@(mapcar #'(lambda (arg) `(print-size ,arg)) ?args)))))

(define-antecedent-macro print-size-<= (var1 var2)
  (unless (and (variable? var1)
	       (variable? var2))
    (error "the antecedent macro print-size-<= requires that both arguments be variables"))
  `((when (print-size-<= (print-size ,var1) (print-size ,var2)))
    (add-head (notice-print-size ,var1))
    (add-head (notice-print-size ,var2))))

(define-antecedent-macro print-size-= (var1 var2)
  (unless (and (variable? var1)
	       (variable? var2))
    (error "the antecedent macro print-size-<= requires that both arguments be variables"))
  `((when (print-size-= (print-size ,var1) (print-size ,var2)))
    (add-head (notice-print-size ,var1))
    (add-head (notice-print-size ,var2))))

(define-antecedent-macro print-size-< (var1 var2)
  (unless (and (variable? var1)
	       (variable? var2))
    (error "the antecedent macro print-size-<= requires that both arguments be variables"))
  `((when (print-size-< (print-size ,var1) (print-size ,var2)))
    (add-head (notice-print-size ,var1))
    (add-head (notice-print-size ,var2))))

(define-antecedent-macro free-for-generalization (const-var other-vars)
  `((when (> (max-constant ,const-var)
	     (max *last-assumption-max-const*
		  ,@(mapcar (lambda (var) `(max-constant ,var))
			    other-vars)))
      (notice-max-constant ,const-var)
      ,@(mapcar (lambda (var)
		  `(notice-max-constant ,var))
		other-vars))))