;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'util :use '(lisp))

(eval-when (compile load eval)
  (in-package 'util :use '(lisp)))

(export '(queue-make queue-clear queue-empty-p queue-length queue-head
	  enqueue dequeue queue-with run-queues check-for-empty-queue
	  clear-queues queue-in-p ulist-make ulist-clear ulist-empty-p
	  ulist-length ulist-head ulist-in-p ulist-add ulist-next))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Normal Queue functions:                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (queue (:constructor queue-make))
  elements
  lptr
  used)

(defun queue-clear (q)
  (setf-undo (queue-elements q) nil)
  (setf-undo (queue-lptr q) nil))

(defun queue-empty-p (q)
  (not (queue-elements q)))

(defun queue-length (q)
  (length (queue-elements q)))

(defun queue-head (q)
  (car (queue-elements q)))

(defun queue-in-p (q element)
  (if (member element (queue-elements q) :test #'eq)
      t))

(defun enqueue (q element)
  ;; If the queue is empty:
  (if (queue-empty-p q)
      ;; Set up the one element queue.
      (progn
	(setf-undo (queue-elements q) (list element))
	(setf-undo (queue-lptr q) (queue-elements q)))
      ;; Otherwise, place the element in the queue destructively. 
      (progn
	(setf-undo (cdr (queue-lptr q)) (list element))
	(setf-undo (queue-lptr q) (cdr (queue-lptr q)))))
  ;; Return the queue.
  q)

(defun dequeue (q)
  (let ((element (car (queue-elements q))))
    ;; Remember the element if necessary
    (if element
	(setf-undo (queue-used q) (cons element (queue-used q))))
    ;; If this is the last element on the queue:
    (if (null (cdr (queue-elements q)))
	;; Clear out the queue.
	(queue-clear q)
	;; Othewise, get rid of the element.
	(setf-undo (queue-elements q) (cdr (queue-elements q))))
    ;; Return the element.
    element))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Priority Queue functions:                                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *num-priorities* 50)
(defvar *queues* (make-array *num-priorities*))
(defvar *min-priority-occupied* *num-priorities*)

(defmacro queue-with ((&rest vars) priority  &rest body)
  `(re-bind ,vars
     (when (< ,priority *min-priority-occupied*)
       (setf-undo *min-priority-occupied* ,priority))
     (push-undo #'(lambda () ,@body)
		(aref *queues* ,priority))))

(emacs-indent queue-with 2)

(defun run-queues (&optional (priority-bound *num-priorities*))
  (while (< *min-priority-occupied* priority-bound)
    (let ((next-thunk (pop-undo (aref *queues* *min-priority-occupied*))))
      (setf-undo *min-priority-occupied*
		 (do ((i *min-priority-occupied* (1+ i)))
		     ((or (= i *num-priorities*)
			  (aref *queues* i))
		      i)))
      (funcall next-thunk))))

(defun check-for-empty-queue (&optional (priority-bound *num-priorities*))
  (when (< *min-priority-occupied* priority-bound)
    (error "The queue is unexpectedly non-empty")))

(defun clear-queues ()
  (dotimes (i *num-priorities*) (setf (aref *queues* i) nil))
  (setf *min-priority-occupied* *num-priorities*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Undo List functions:                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (ulist (:constructor ulist-make))
  elements
  lptr)

(defun ulist-clear (u)
  (setf-undo (ulist-elements u) nil)
  (setf-undo (ulist-lptr u) nil))

(defun ulist-empty-p (u)
  (not (ulist-elements u)))

(defun ulist-length (u)
  (length (ulist-elements u)))

(defun ulist-head (u)
  (caar (ulist-elements u)))

(defun ulist-in-p (u element)
  (if (member element (ulist-elements u) :test #'eq :key #'car)
      t))

(defun ulist-next (u test)
  (caar (remove-if-not test (ulist-elements u) :key #'car)))

(defun ulist-add (u element &key justification)
  ;; If the ulist is empty:
  (if (ulist-empty-p u)
      ;; Set up the one element ulist.
      (progn
	(setf-undo (ulist-elements u) (list (cons element justification)))
	(setf-undo (ulist-lptr u) (ulist-elements u)))
      ;; Otherwise, place the element in the ulist destructively. 
      (progn
	(setf-undo (cdr (ulist-lptr u)) (list (cons element justification)))
	(setf-undo (ulist-lptr u) (cdr (ulist-lptr u)))))
  ;; Return the ulist.
  u)
