;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'util :use '(lisp))

;;
;;(defmacro defpiecefun (name args)
;;  (let ((piece (gensym)))
;;    `(defun ,name ,args				;
;;       (mapc #'(lambda (,piece) (funcall (cdr ,piece) ,@args))
;;	     (get ',name 'pieces)))))
;;
;;(defmacro defpiece ((fun piece-name) args &body body)
;;  `(pushnew (cons ',piece-name #'(lambda ,args ,@body))
;;	    (get ',fun 'pieces)
;;	    :key #'car))
;;


;;piece functions:
;;   (defpiecefun 'fun-name args) must be executed before any pieces are defined
;;   (defpiece (fun-name piece-name) args body) adds or replaces the piece named
;;      by piece-name.
;;   any piece of the form (when test body) will be consolidated with other
;;      pieces of that form with the same test.
;; piece-functions are compiled the first time they are used after being
;;    redefined.  at that time, the new compiled definition is saved and reused until
;;    another redefinition occurs.
;;


;; a "piece" is a cons of a test and a list of forms to run when that test is true

(defvar *the-piecefuns* nil)

(eval-when (compile eval load)
  (defun make-piece (test body-forms) (cons test body-forms)))
(defun body-part (piece) (rest piece))
(defun test-part (piece) (first piece))

(defun thing-type-test-p (test of-what)		;Kind of gross
  (and (consp test)
       (eq (car test) 'thing-typep)
       (consp (cdr test))
       (eq (cadr test) of-what)
       (consp (cddr test))
       (consp (caddr test))
       (eq (car (caddr test)) 'quote)))

(defun thing-type-test-type (test)
  (cadr (caddr test)))

(defun collate (pred list)
  (values (remove-if-not pred list)
	  (remove-if pred list)))

; Test this with e.g.
;  (let ((*print-level* 4)) (print (build-piece-fun 'do-is-rules)) (values))

(defun compute-piecefun-definitions ()
  (mapcar #'(lambda (fun)
	      (selectmatch (build-piece-fun fun)
		((lambda ?args . ?body)
		 `(defun ,fun ,?args . ,?body))))
	  (reverse *the-piecefuns*)))

(defexport build-piece-fun (fun)
  (insert-ignores (build-piece-fun-internal fun)))

(defun insert-ignores (procedure)
  (selectmatch procedure
    ((lambda ?args . ?body)
     `(lambda ,?args
       ,@(append (mapcan #'(lambda (arg) (when (not (internal-member arg ?body))
					   `((declare (ignore ,arg)))))
		  ?args)
	  ?body)))
    ((defun ?name ?args . ?body)
     `(defun ,?name ,?args
       ,@(append (mapcan #'(lambda (arg) (when (not (internal-member arg ?body))
					   `((declare (ignore ,arg)))))
		  ?args)
	  ?body)))    
    (:anything
     (error "insert-ignores applied to non-procedure"))))

(defun build-piece-fun-internal (fun)
  (let ((args (get fun 'piece-args))
	(pieces (mapcar #'(lambda (piece-name)
			    (get fun piece-name))
			(get fun 'pieces))))
    (mvlet (((dispatch-pieces other-pieces)
	     (collate #'(lambda (piece)
			  (thing-type-test-p (test-part piece) (car args)))
		      pieces)))
      `(lambda ,args
	 ,@(mapcan #'(lambda (piece)
		       (if (eq (test-part piece) t)
			   (copy-list (body-part piece))
			   (list `(when ,(test-part piece) ,@(body-part piece)))))
		   (collapse-same-tests other-pieces))
	 ,@(if (null dispatch-pieces)
	       `()
	       `((case (o-thing-type-tag ,(car args))
		   ,@(mapcar #'(lambda (piece)
				 `((,(thing-type-test-type (test-part piece)))
				   ,@(body-part piece)))
			     (collapse-same-tests dispatch-pieces)))))))))

(defun collapse-same-tests (pieces)
  (iterate collapse-same-tests
	   ((pieces (sort pieces #'compare-sexpr :key #'test-part))
	    (when-forms '()))
    (cond ((null pieces) when-forms)
	  ((and (rest pieces)
		(equal (test-part (first pieces))
		       (test-part (second pieces))))
	   (collapse-same-tests
	     (cons (make-piece (test-part (first pieces))
			       (append (body-part (second pieces))
				       (body-part (first pieces))))
		   (rest (rest pieces)))
	     when-forms))
	  (t (collapse-same-tests
	       (rest pieces)
	       (cons (first pieces) when-forms))))))

(defun compare-sexpr (s1 s2)
  (let ((s1-hash (sxhash s1))
	(s2-hash (sxhash s2)))
    (or (< s1-hash s2-hash)
	(and (= s1-hash s2-hash)
	     (not (equal s1 s2))
	     (string-lessp (format nil "~s" s1) (format nil "~s" s2))))))

(defun invalidate-piecefun-compilation (fun)
  (setf (symbol-function fun)
	(lambda (&rest args)
	  (redo-structures)
	  (notify-and-compile fun (build-piece-fun fun))
	  (apply fun args)))
  fun)

(defun when-test (when-form) (second when-form))
(defun when-forms (when-form) (rest (rest when-form)))

(eval-when (compile load eval)
  (defun one-when-p (body)
    (and body
	 (null (rest body))
	 (consp (first body))
	 (eq (first (first body)) 'when))))

;; The pieces property is a list of the piece names.

(emacs-indent defpiece 2)
(defexportmacro defpiece ((fun piece-name) args &body body)
  `(defpiece-fun ',fun ',piece-name ',args ',body))

(defun defpiece-fun (fun piece-name args body)
  (unless (get fun 'piece-wise)
    (error "~s not defined as piecewise" fun))
  (unless (equal args (get fun 'piece-args))
    (error "illegal arguments for ~s" fun))
  (setf (get fun piece-name)
	(if (one-when-p body)
	    (make-piece (when-test (first body))
			(when-forms (first body)))
	    (make-piece t body)))
  (setf (get fun 'pieces)
	(adjoin piece-name (get fun 'pieces)))
  (invalidate-piecefun-compilation fun)
  ;the following is for error detection at piece evaluation time.
  (when *visible-evaluation?*
    (notify-and-compile (intern (concatenate 'string
				  (string fun)
				  "=="
				  (string piece-name)))
	     (insert-ignores
	      `(lambda ,args . ,body)))))

(defexportmacro defpiecefun (fun args)
  `(defpiecefun-fun ',fun ',args))

(defun defpiecefun-fun (fun args)
  (pushnew fun *the-piecefuns*)
  (setf (get fun 'piece-wise) t)
  (let ((old-args (get fun 'piece-args)))
    (setf (get fun 'piece-args) args)
    (when (not (equal args old-args))
      (setf (get fun 'pieces) '())
      (invalidate-piecefun-compilation fun))))

(defmacro clear-local-pieces (&body pieces)
  `(progn
     ,@(mapcar (lambda (piece) (list (first piece) (second piece) (third piece)))
	       pieces)))

(defexport clear-pieces (piece-selector)
  (dolist (fun *the-piecefuns*)
    (setf (get fun 'pieces) (remove-if piece-selector (get fun 'pieces)))
    (invalidate-piecefun-compilation fun)))



;This is a block compilation system that integrates pieces of function
;definitions.
;
;A piece is a series of forms that represent parts of lisp
;code.  For example,
;
;
;(defmergefun function (args))
;
;(defmergepiece (function piece-name) (?x ?y)
; (let ?z (bar ?x ?y))
; (when (pred ?x ?y ?z))
; (dolist ?w (grith ?y ?z))
; (bar ?w ?y))
;
;;should be viewed as an abbreviation for
;
;(defpiece (function piece-name) (?x ?y)
;  (let ((?z (bar ?x ?y)))
;    (when (pred ?x ?y ?z)
;      (dolist (?w (gritch ?y ?z))
;	 (bar ?w ?y)))))
;
;The body of a mergepiece is a series of forms where each form
;other than the last form must be on of the following.

;(let var value)
;(dolist var list-value)
;(when test)

;Multiple pieces can be merged.  For example.

;(defmergepiece (foo part1) (?x ?y)
; (dolist ?w (gritch ?x ?y))
; (let ?z (bar ?x ?w))
; (when (pred ?x ?y ?z))
; (baz1?w ?y))

;and

;(defmergepiece (foo part2) (?x ?y)
; (dolist ?w (gritch ?x ?y))
; (let ?z (bar ?x ?w))
; (baz2 ?w ?y))

;we get

;(defun foo (?x ?y)
;  (dolist (?w (gritch ?x ?y))
;    (let ((?z (bar ?x ?w)))
;      (baz2 ?w ?y)
;      (when (pred ?x ?y ?z)
;	 (baz1 ?w ?y)))))


;;;;;;;  March 1992   -rlg
;;;
;;;  Added (definterpfun ....) same as (defmergfun ...) except that the resulting
;;;  code is not compiled but run by a low-level interpreter, which is included below.
;;;  This is for rules added at run-time.
;;;

;

(defvar *do-merging* t)

(emacs-indent defmergepiece 2)
(defexportmacro defmergepiece ((fun piece-name) args &body body)
  `(defmergepiece-fun ',fun ',piece-name ',args ',body))

(export '*do-merging*)

(property-macro external-pieces)

(property-macro internal-pieces)

(property-macro mergefun-demons)

(defun piece-control-flag (mergefun piecename)
  (create-name mergefun piecename 'flag))

(defun external-fun-name (mergefun piecename)
  (create-name mergefun piecename 'external 'fun))

(defun defmergepiece-fun (fun piece-name args body)
  (let ((funargs (get fun 'piece-args)))
    (unless (get fun 'merge-wise)
      (error "~s not defined as merge-wise" fun))
    (unless (= (length args) (length funargs))
      (error "~s does not match ~s, the arglist of ~s"
	     args
	     funargs
	     fun))
    (let ((rbody (rename-piece funargs args (tag-last-part body (piece-control-flag fun piece-name)))))
      (setf (get fun piece-name) rbody)
      (if *do-merging*
	  (progn (pushnew piece-name (internal-pieces fun))
		 (invalidate-mergefun-compilation fun)
		 (set (piece-control-flag fun piece-name) t)
		 (setf (mergefun-demons fun)
		       (remove (external-fun-name fun piece-name) (mergefun-demons fun)))
		 (setf (external-pieces fun)
		       (remove piece-name (external-pieces fun))))
	  (progn (pushnew piece-name (external-pieces fun))
		 (set (piece-control-flag fun piece-name) nil)
		 (setf (internal-pieces fun)
		       (remove piece-name (internal-pieces fun)))
		 (compile-defuns (external-defuns rbody (external-fun-name fun piece-name) funargs))
		 (pushnew (external-fun-name fun piece-name) (mergefun-demons fun))))
					;The following is for error detection at piece creation time.
      (when *visible-evaluation?*
	(compile-defuns
	 (external-defuns rbody (create-name fun '== piece-name) funargs))))))

(emacs-indent definterppiece 2)
(defexportmacro definterppiece ((fun piece-name) args &body body)
  `(definterppiece-fun ',fun ',piece-name ',args ',body))

(defun definterppiece-fun (fun piece-name args body)
  (let ((funargs (get fun 'piece-args)))
    (unless (get fun 'interp-wise)
      (error "~s not defined as interp-wise" fun))
    (unless (= (length args) (length funargs))
      (error "~s does not match ~s, the arglist of ~s"
	     args
	     funargs
	     fun))
    (let ((rbody (rename-piece funargs args (tag-last-part-simply body))))
      (setf (get fun piece-name) rbody)
      (pushnew piece-name (get fun 'pieces))
      (invalidate-interpfun-compilation fun))))

(eval-when (compile eval load) (defvar *the-mergefuns* '()))
(eval-when (compile eval load) (defvar *the-interpfuns* '()))

(defexport mergefun-p (fun-name) (member fun-name *the-mergefuns*))
(defexport interpfun-p (fun-name) (member fun-name *the-interpfuns*))

(defexportmacro defmergefun (fun args)
  `(eval-when (compile load eval) (defmergefun-fun ',fun ',args)))

(defun defmergefun-fun (fun args)
  (unless (= (length args) (length (remove-duplicates args)))
    (error "Duplicate formal parameter in arglist ~s to function ~s"
	   args fun))
  (unless (every 'variable? args)
    (error "Arguments to DEFMERGEFUN functions must begin with the character '?': ~s"
	    fun))
  (pushnew fun *the-mergefuns*)
  (when (or (not (get fun 'merge-wise))
	    (not (= (length (get fun 'piece-args)) (length args))))
    (setf (internal-pieces fun) nil)
    (setf (external-pieces fun) nil)
    (setf (mergefun-demons fun) nil)
    (setf (get fun 'piece-args) args)
    (setf (get fun 'merge-wise) t)
    (invalidate-mergefun-compilation fun)))

(defexportmacro definterpfun (fun args)
  `(eval-when (compile load eval) (definterpfun-fun ',fun ',args)))

(defun definterpfun-fun (fun args)
  (unless (= (length args) (length (remove-duplicates args)))
    (error "Duplicate formal parameter in arglist ~s to function ~s"
	   args fun))
  (unless (every 'variable? args)
    (error "Arguments to DEFINTERPFUN functions must begin with the character '?': ~s"
	    fun))
  (pushnew fun *the-interpfuns*)
  (when (or (not (get fun 'interp-wise))
	    (not (= (length (get fun 'piece-args)) (length args))))
    (setf (get fun 'pieces) nil)
    (setf (get fun 'piece-args) args)
    (setf (get fun 'interp-wise) t)
    (invalidate-interpfun-compilation fun)
    (setf (symbol-function fun) (lambda (&rest args)
				  (progv (get fun 'piece-args) args
				    (when (get fun 'needs-remerging)
				      (setf (get fun 'merged-body) (interpfun-definition fun))
				      (setf (get fun 'needs-remerging) nil))
				    (interpret (get fun 'merged-body)))))))

(defun invalidate-mergefun-compilation (fun)
  (setf (symbol-function fun)
	(lambda (&rest args)
	  (redo-structures)
	  (compile-defuns (mergefun-definitions fun))
	  (apply fun args)))
  fun)

(defun invalidate-interpfun-compilation (fun)
  (setf (get fun 'needs-remerging) t)
  fun)

(defexport clear-mergepieces (piece-selector)
  (dolist (fun *the-mergefuns*)
    (setf (internal-pieces fun)
	  (remove-if (lambda (piece)
		       (when (funcall piece-selector piece)
			 (if *do-merging*
			     (invalidate-mergefun-compilation fun)
			     (set (piece-control-flag fun piece) nil))
			 t))
		     (internal-pieces fun)))
    (setf (external-pieces fun)
	  (remove-if (lambda (piece)
		       (when (funcall piece-selector piece)
			 (setf (mergefun-demons fun)
			       (remove (external-fun-name fun piece)
				       (mergefun-demons fun)))
			 t))
		     (external-pieces fun)))))

(defexport clear-interppieces (piece-selector)
  (dolist (fun *the-interpfuns*)
    (setf (get fun 'pieces)
	  (remove-if (lambda (piece)
		       (when (funcall piece-selector piece)
			 (invalidate-interpfun-compilation fun)
			 t))
		     (get fun 'pieces)))))

(defexport show-mergepieces (piece-selector)
  (mapcan (lambda (fun)
	    (mapcan (lambda (piecename)
		      (if (funcall piece-selector piecename)
			  `((defmergepiece (,fun ,piecename) ,(get fun 'piece-args)
			      ,(get fun piecename)))
			  nil))
		    (append (internal-pieces fun) (external-pieces fun))))
	  *the-mergefuns*))

(defexport show-interppieces (piece-selector)
  (mapcan (lambda (fun)
	    (mapcan (lambda (piecename)
		      (if (funcall piece-selector piecename)
			  `((definterppiece (,fun ,piecename) ,(get fun 'piece-args)
			      ,(get fun piecename)))
			  nil))
		    (get fun 'pieces)))
	  *the-interpfuns*))


(defun tag-last-part (piece flag)
  (when piece
    (if (null (rest piece))
	`((:lisp ,(if *do-merging*
		      `(when ,flag ,(first piece))
		      (first piece))))
	(cons (first piece)
	      (tag-last-part (rest piece) flag)))))

(defun tag-last-part-simply (piece)
  (when piece
    (if (null (rest piece))
	`((:lisp ,(first piece)))
	(cons (first piece)
	      (tag-last-part-simply (rest piece))))))


(defun rename-piece (fun-args piece-args piece)
  (append (whens-for-equal-args piece-args fun-args)
	  (sublis (mapcar 'cons
			  piece-args
			  fun-args)
		  piece)))

(defun whens-for-equal-args (piece-args fun-args)
  (iterate do-rest ((a-list (mapcar #'cons piece-args fun-args)) (result nil))
    (if a-list
	(let* ((next-arg (car (first a-list)))
	       (equal-car-pairs (remove-if-not (lambda (pair) (eq (car pair) next-arg))
					       (rest a-list))))
	  (do-rest (remove-if (lambda (pair) (eq (car pair) next-arg))
			      (rest a-list))
		   (append (mapcar (lambda (equal-pair)
				     `(when (eq ,(cdr (first a-list))
						,(cdr equal-pair))))
				   equal-car-pairs)
			   result)))
	result)))

(defvar *residual-definitions* nil)

(defun all-mergepieces (name)
  (append (internal-pieces name) (external-pieces name)))

(defun mergefun-definitions (fun-name)
  (let ((body-pieces (mapcar (lambda (piece-name) (get fun-name piece-name))
			     (internal-pieces fun-name)))
	(args (get fun-name 'piece-args)))
    (append (mapcar (lambda (piece-name)
		      `(defvar ,(piece-control-flag fun-name piece-name)))
		    (internal-pieces fun-name))
	    (combine-internal-pieces-to-defuns body-pieces fun-name args))))

(defun interpfun-definition (fun-name)
  (let ((body-pieces (mapcar (lambda (piece-name) (get fun-name piece-name))
			     (get fun-name 'pieces)))
	(args (get fun-name 'piece-args)))
    (combine-pieces body-pieces args :make-code? nil)))

(defun compute-mergefun-definitions ()
  (mapcan 'mergefun-definitions
	  *the-mergefuns*))
  

;Piece combination.
;In the recursive case we have a set of pieces and a set of ``bound variables'',
;i.e., variables that will have values when the code in the pieces is run.

(defun combine-internal-pieces-to-defuns (pieces fun-name args)
  (let* ((*residual-definitions* nil)
	 (main-defun (insert-ignores
		      `(defun ,fun-name ,args
			(dolist (demon (mergefun-demons ',fun-name))
			  (funcall demon ,@args))
			,@(combine-pieces pieces args)))))
    (mapcan #'remember-body
	    (append (reverse *residual-definitions*)
		    (list main-defun)))))

(defun external-defuns (piece fun-name args)
  (let* ((*residual-definitions* nil)
	 (main-defun (insert-ignores
		      `(defun ,fun-name ,args
			,@(combine-pieces (list piece) args)))))
    (mapcan #'remember-body
	    (append (reverse *residual-definitions*)
		    (list main-defun)))))

;; kcz -- change remember-body to return a list of the defun and a setf form
;;
(defun remember-body (def)
  `((setf (get ',(second def) 'defun) ',def)
    ,def))

(defun combine-pieces (pieces bound-vars &key (make-code? t))
  (when pieces
    (let ((next-form (first (first pieces))))
      (unless next-form
	(error "no available next form"))
      (make-body next-form pieces bound-vars :make-code? make-code?))))

(defvar *dolist-count* 0)
(defvar *dolist-included-count* 0)

(defun make-body (head-form pieces bound-vars &key (make-code? t))
  (let ((new-head (alpha-rename-head head-form)))
    (let ((included-pieces nil)
	  (excluded-pieces nil))
      (dolist (piece pieces)
	(let ((first-form (first piece)))
	  (if (form-matches? first-form new-head)
	      (let ((rest-piece (sublis (form-match first-form new-head)
					(cdr piece))))
		(when rest-piece
		  (push rest-piece included-pieces)))
	      (push piece excluded-pieces))))
      (if make-code?
	  (selectmatch new-head
	    ((let ?var ?exp)
	     `((let ((,?var ,?exp))
		 ,@(combine-pieces (nconc included-pieces excluded-pieces)
				   (cons ?var bound-vars)))))
	    ((when ?exp)
	     (let* ((when-body (combine-pieces included-pieces bound-vars)))
	       `((when ,?exp
		   ,@(if (and (some #'cdddr included-pieces)
			      excluded-pieces) ;heuristic for breaking up definitions
			 (call-to-residual when-body bound-vars)
			 when-body))
		 ,@(combine-pieces excluded-pieces bound-vars))))
	    ((dolist ?var ?form)
	     (incf *dolist-count*)
	     (incf *dolist-included-count* (length included-pieces))
	     `((dolist (,?var ,?form)
		 ,@(let ((dolist-body (combine-pieces included-pieces (cons ?var bound-vars))))
		     (if (and (some #'cdddr included-pieces)
			      excluded-pieces)
			 (call-to-residual dolist-body (cons ?var bound-vars))
			 dolist-body)))
	       ,@(combine-pieces excluded-pieces bound-vars)))
	    ((:lisp ?form)
	     (when included-pieces
	       (error "A Lisp form must be the final form in a piece"))
	     `(,?form
	       ,@(combine-pieces excluded-pieces bound-vars)))
	    (:anything (error "unrecognized head in make-body")))
	  (progn
	    (when (and (consp new-head)
		       (eq (car new-head) :lisp)
		       included-pieces)
	      (error "A Lisp form must be the final form in a piece"))
	    (let ((new-bound-vars (append (vars-bound-in new-head) bound-vars)))
	      `(,new-head
		,(combine-pieces included-pieces new-bound-vars :make-code? nil)
		,(combine-pieces excluded-pieces new-bound-vars :make-code? nil))))))))

(defun vars-bound-in (head)
  (and (consp head)
       (when (member (car head) '(let dolist))
	 (let ((var (second head)))
	   (and var
		(list var))))))

(defun measure-sharing ()
  (setq *dolist-count* 0)
  (setq *dolist-included-count* 0)
  (mapc 'mergefun-definitions *the-mergefuns*)
  (format t "~% ~s dolists ~s average included pieces"
	  *dolist-count*
	  (/ (Float *dolist-included-count*) *dolist-count*)))

(defun call-to-residual (body bound-vars)
  (let* ((new-fun (gentemp "FUN-"))
	 (new-args (intersection bound-vars
				 (variables body)))
	 (defun-form `(defun ,new-fun ,new-args
		       ,@body)))
    (push defun-form
	  *residual-definitions*)
    (setf (get new-fun 'defun) defun-form)
    `((,new-fun ,@new-args))))

(defun alpha-rename-head (head)
  (if (member (car head) '(let dolist))
      `(,(car head) ,(copy-var (second head)) ,(third head))
      head))

(defun form-matches? (head form)
    (if (member (car head) '(let dolist))
	(and (eq (car head) (car form))
	     (equal (third head) (third form)))
	(equal head form)))

(defun form-match (new-head form)	;
  (when (member (car new-head) '(let dolist))
    (list (cons (second new-head) (second form)))))

(defun interpret (merged-body)
  (when (consp merged-body)
    (let* ((head (car merged-body))
	   (cdr-body (cdr merged-body))
	   (cddr-body (cdr cdr-body))
	   (included-body (car cdr-body))
	   (excluded-body (car cddr-body)))
      (when (consp head)
	(case (car head)
	  (:lisp (eval (second head)))
	  (let (let* ((cdr-head (cdr head))
		      (var-name (car cdr-head))
		      (cddr-head (cdr cdr-head))
		      (var-val (car cddr-head)))
		 (progv (list var-name) (list (eval var-val))
		   (interpret included-body))))
	  (dolist (let* ((cdr-head (cdr head))
			 (var-name (car cdr-head))
			 (cddr-head (cdr cdr-head))
			 (list-exp (car cddr-head)))
		    (dolist (i (eval list-exp))
		      (progv (list var-name) (list i)
			(interpret included-body)))))
	  (when (when (eval (cadr head))
		  (interpret included-body))))
	(interpret excluded-body)))))



;
;piecewise definition of structures.
;
;(defpiecestruct foo)
;
;(defslot foo slot1)
;
;(defpiecestruct bar :include foo)
;
;(defslot bar slot2)
;
;(piece-structure-definitions)
;
;piece structures are strictly monotonic --- removal of structure names
;or slots is not allowed.  This avoids a bunch of implementation problems
;and is not really much of an inconvenience (the non-removal of slots
;will make structures temporarilly space inefficient.)
;
;defslot defines the slot macro independent of the creation of the
;structure definition.  This allows the slot macro to be used
;as soon as the slot is defined.
;
;default values are not allowed in slots.
;
;no structor specifications are allowed in piece structures.
;there is no conc-name.
;the print function for structure foo is print-foo and must be user defined.
;
;No structure functions are defined until one evaluates or compiles
;(piece-structure-definitions).
;

(defvar *the-piecestructs* nil)

(defexport category-constructor-function (name)
  (combine-symbols 'make name (symbol-package name)))

(defexport category-predicate (name)
  (combine-symbols name 'p (symbol-package name)))

(defexport category-print-function (name)
  (combine-symbols 'print name (symbol-package name)))

(property-macro slot-names)
		
(property-macro byte-slot-names)

(property-macro bit-slot-names)

(property-macro parent-structure)

(defvar *redo-structures* nil)

(defun structure-functions (struct)
  (list* (category-constructor-function struct)
	 (category-predicate struct)
	 (mapcar #'car (slot-names struct))))

(defun invalidate-compilations ()
  (setq *redo-structures* t)
  (mapc (lambda (struct)
	  (setf (symbol-function (category-constructor-function struct))
		(lambda ()
		  (redo-structures)
		  (funcall (category-constructor-function struct)))))
	 *the-piecestructs*)
  (mapc 'invalidate-piecefun-compilation *the-piecefuns*)
  (mapc 'invalidate-mergefun-compilation *the-mergefuns*))

(defexportmacro defpiecestruct (name &key include)
  `(defpiecestruct-fun ',name ',include))

(defun defpiecestruct-fun (name parent)
  (when parent
    (when (and (parent-structure name)
	       (not (eq (parent-structure name) parent)))
      (error "attempt to change the parent structure of a piecestruct"))
    (setf (parent-structure name) (or parent (list 'no-parent))))
  (when (not (member name *the-piecestructs*))
    (push name *the-piecestructs*)
    (if *do-merging*
	(invalidate-compilations)
	(evaluate-properly
	 `(defstruct (,name
		      (:conc-name nil)
		      (:constructor ,(category-constructor-function name))
		      (:predicate ,(category-predicate name))
		      (:print-function (lambda (self stream &rest ignore)
					 (declare (ignore ignore))
					 (,(category-print-function name)
					   self stream)))
		      ,@(when (parent-structure name)
			  `((:include ,(parent-structure name))))))))))

(defun evaluate-properly (exp)
  (if (and (consp exp)
	   (eq (car exp) 'progn))
      (mapc 'evaluate-properly (cdr exp))
      (funcall
       (notify-and-compile nil
			   `(lambda () ,exp)))))


(defexportmacro defslot (struct-name slot-name &optional init-val)
  `(defslot-fun ',struct-name ',slot-name ',init-val))

(property-macro init-demons)

(property-macro calls-init-demons?)

(property-macro slot-owner)

(property-macro slot-type)

(defun defslot-fun (name slot init-val)
  (cond ((declare-slot slot name 'normal)
	 (setf (slot-names name)
	       (append (slot-names name) (list (list slot init-val))))
	 (if *do-merging*
	     (invalidate-compilations)
	     (define-external-slot name slot)))
	((not (equal (assoc-value slot (slot-names name))
		     (list init-val)))
	 (setf (assoc-value slot (slot-names name))
	       (list init-val))
	 (if *do-merging*
	     (invalidate-compilations)
	     (install-init-demon name slot)))))

(defun declare-slot (slot owner type)
  (unless (member owner *the-piecestructs*)
    (error "~s has not been properly declared" owner))
  (if (slot-owner slot)
      (progn (unless (and (equal (slot-owner slot) owner)
			  (equal (slot-type slot) type))
	       (error "attempt to change the owner or type of slot ~s" slot))
	     nil)
      (progn (setf (slot-owner slot) owner)
	     (setf (slot-type slot) type)
	     t)))

(defun define-external-slot (name slot)
  (evaluate-properly
   `(progn
     (defvar ,(create-name slot 'macro 'table) (make-hash-table))
     (defmacro ,slot (x)
       `(gethash ,x ,',(create-name slot 'macro 'table)))))
  (install-init-demon name slot))

(defun install-init-demon (name slot)
  (install-init-demons name)
  (evaluate-properly
   `(defun ,(create-name slot 'initializer) (object)
     (setf (,slot object) ,(car (assoc-value slot (slot-names name))))))
  (pushnew (create-name slot 'initializer)
	   (init-demons name)))

(defun install-init-demons (name)
  (unless (calls-init-demons? name)
    (setf (calls-init-demons? name) t)
    (let ((old-maker (symbol-function (create-name 'make name))))
      (setf (symbol-function (create-name 'make name))
	    (lambda ()
	      (let ((object (funcall old-maker)))
		(call-init-demons name object)
		object))))
    (dolist (name2 *the-piecestructs*)
      (when (eq (parent-structure name2) name)
	(install-init-demons name2)))))

(defun call-init-demons (name object)
  (when (parent-structure name)
    (call-init-demons (parent-structure name) object))
  (dolist (demon (init-demons name))
    (funcall demon object)))
	     

(defexportmacro defslot-byte (name slot len)
  `(defslot-byte-fun ',name ',slot ',len))

(defun defslot-byte-fun (name slot len)
  (cond ((declare-slot slot name 'byte)
	 (defslot-fun name (combine-symbols name 'byte-slots) 0)
	 (setf (byte-slot-names name)
	       (append (byte-slot-names name) (list (list slot len))))
	 (if *do-merging*
	     (invalidate-compilations)
	     (evaluate-properly
	      `(definline ,slot (struct)
		(ldb (byte ,len ,(byte-slot-position slot (byte-slot-names name) 0))
		 (,(combine-symbols name 'byte-slots) struct))))))
	((not (equal (assoc-value slot (byte-slot-names name))
		     (list len)))
	 (error "attempt to change the length of a byte slot"))))

(defun byte-slot-position (slot-name name-alist sum-so-far)
  (cond ((null name-alist)
	 (error "attempt to get the byst-slot-position of a non-byte-slot"))
	((eq (caar name-alist) slot-name)
	 sum-so-far)
	(t (byte-slot-position slot-name (cdr name-alist) (+ sum-so-far (second (car name-alist)))))))

(defexportmacro defslot-bit (name slot)
  `(defslot-bit-fun ',name ',slot))

(defun defslot-bit-fun (name slot)
  (when (declare-slot slot name 'bit)
    (setf (bit-slot-names name)
	  (append (bit-slot-names name) (list slot)))
    (defslot-fun name (combine-symbols name 'bit-slots)
      `(make-array '(,(length (bit-slot-names name)))
	:element-type 'bit :initial-element 0))
    (if *do-merging*
	(invalidate-compilations)
	(evaluate-properly
	 `(definline ,slot (struct)
	   (aref (the (simple-bit-vector ,(length (bit-slot-names name)))
		  (,(combine-symbols name 'bit-slots) struct))
	    ,(bit-slot-position slot (bit-slot-names name) 0)))))))

(defun bit-slot-position (slot names pos-sofar)
  (cond ((null names)
	 (error "attempt to find the bit-slot-position of a non-bit-slot"))
	((eq slot (car names))
	 pos-sofar)
	(t (bit-slot-position slot (cdr names) (1+ pos-sofar)))))


(defexport redo-structures ()
  
  ;;In lucid, when a defsruct is evaluated all the structure functions are compiled
  ;;except for the constructor function.  If we try to compile an already compiled
  ;;function then we get an error.  I don't know how to write a program to tell
  ;;if a function is compiled.  So I just compile the maker.  This is almost certainly
  ;;lucid-specific. -- dam

  (when *redo-structures*
    (mapc 'evaluate-properly (compute-structure-definitions))
    (setf *redo-structures* nil)))
  

(defexportmacro piece-structure-definitions ()
  `(progn
    ,@(compute-structure-definitions)
    (setf *redo-structures* nil)))

;The following does atopological sort of the structures so that the most
;general structure comes first.

(defun compute-structure-definitions ()
  (let ((done-so-far nil))
    (labels ((next-definition ()
	       (let ((next-invalid (find-if (lambda (name) (not (member name done-so-far)))
					    *the-piecestructs*)))
		 (when next-invalid
		   (let ((parent-invalid (find-invalid-parent next-invalid)))
		     (push parent-invalid done-so-far)
		     (structure-definition parent-invalid)))))
	     (find-invalid-parent (name)
	       (let ((parent (parent-structure name)))
		 (if (and parent
			  (member parent *the-piecestructs*)
			  (not (member parent done-so-far)))
		     (find-invalid-parent parent)
		     name)))
	     (all-definitions ()
	       (let ((next (next-definition)))
		 (when next
		   (cons next (all-definitions))))))
      #-cmu(all-definitions)
      ;; the following is necessary in CMU CL 15d,
      ;; to handle included structures with conc-name nil
      #+cmu(let ((defs (all-definitions)))
	     (append defs (reverse defs))))))

(defun structure-definition (name)
  `(progn
    (defstruct (,name
		 (:conc-name nil)
		 (:constructor ,(category-constructor-function name))
		 (:predicate ,(category-predicate name))
		 (:print-function (lambda (self stream &rest ignore)
				    (declare (ignore ignore))
				    (,(category-print-function name) self stream)))
		 ,@(when (parent-structure name)
		     `((:include ,(parent-structure name)))))
      ,@(slot-names name))
    ,@(do ((byte-slots (byte-slot-names name) (cdr byte-slots))
	   (pos 0 (+ pos (second (first byte-slots))))
	   (macro-defs nil (cons `(definline ,(first (first byte-slots)) (struct)
				   (ldb (byte ,(second (first byte-slots)) ,pos)
				    (,(combine-symbols name 'byte-slots) struct)))
				 macro-defs)))
	  ((null byte-slots) macro-defs))
    ,@(do ((bit-slots (bit-slot-names name) (cdr bit-slots))
	   (pos 0 (1+ pos))
	   (macro-defs nil (cons `(definline ,(first bit-slots) (struct)
				   (aref (the (simple-bit-vector ,(length (bit-slot-names name)))
					  (,(combine-symbols name 'bit-slots) struct))
				    ,pos))
				 macro-defs)))
	  ((null bit-slots) macro-defs))))


(defexport clear-structure-definitions ()
  (invalidate-compilations)
  (dolist (struct *the-piecestructs*)
    (setf (parent-structure struct) nil)
    (setf (calls-init-demons? struct) nil)
    (setf (init-demons struct) nil)
    (dolist (slot (append (mapcar 'car (slot-names struct))
			  (bit-slot-names struct)
			  (mapcar 'car (byte-slot-names struct))))
      (setf (slot-owner slot) nil))
    (setf (slot-names struct) nil)
    (setf (byte-slot-names struct) nil)
    (setf (bit-slot-names struct) nil))
  (setf *the-piecestructs* nil))




(defun avoid-redefun-error (defun-form)
  (selectmatch defun-form
    ((defun ?function-name ?args . ?body)
     `(;;Due to a Lucid bug, this is less efficient than the defun.
       ;;(setf (symbol-function ',?function-name) #'(lambda ,?args ,@?body))
       ,defun-form))
    (:anything `(,defun-form))))

(defexportmacro all-piecefuns ()
  `(progn
    (eval-when (compile load eval)
      (piece-structure-definitions))
    ,@(mapcan #'avoid-redefun-error (compute-piecefun-definitions))
    ,@(mapcan #'avoid-redefun-error (compute-mergefun-definitions))
    ,@(mapcan #'avoid-redefun-error (compute-selectfun-definitions))))


;The following is useful for making a function a piecefunction by default.
;It is also useful for functions which use slots defined by the incremental
;structure definition facility.

(defexportmacro defextendable (name args &rest body)
  `(eval-when (compile eval load)
    (defpiecefun ,name ,args)
    (defpiece (,name :primary-piece) ,args . ,body)))

;;; Select pieces
;; Here's yet another system for defining functions one piece
;; at a time.  Selectfuns are different from piecefuns and mergefuns,
;; though, in that selectfuns run only one of the pieces, instead
;; of all of them.  The piece to run is selected by pattern-matching
;; on the first argument of the function.

;; (defselectfun foo (what-to-do to-what)
;;   (error "I don't know how to do ~s to ~s." what-to-do to-what))
;;
;; (defselectpiece (foo add) (what-to-do to-what)
;;     (add ?n) (numberp ?n)
;;   (+ ?n to-what))
;;
;; (defselectpiece (foo acons) (what-to-do to-what)
;;     (cons ?foo ?bar) t
;;   (acons ?foo ?bar to-what))

(emacs-indent defselectpiece 4)

(defexportmacro defselectpiece ((fun piece-name) &optional args form condition
			  &rest body)
  `(defselectpiece-fun ',fun ',piece-name ',args ',form ',condition ',body))

(defun defselectpiece-fun (fun piece-name args form condition body)
  (let ((funargs (get fun 'piece-args)))
    (unless (get fun 'select-wise)
      (error "~s not defined as select-wise" fun))
    (unless (equal args funargs)
      (error "~s does not match ~s, the arglist of ~s" args funargs fun))
    (setf (get fun piece-name) (list* form condition body))
    (pushnew piece-name (get fun 'pieces))
    (invalidate-selectfun-compilation fun)))

(defvar *the-selectfuns* nil)

(defexportmacro defselectfun (fun args &rest default-body)
  `(defselectfun-fun ',fun ',args ',default-body))

(defun defselectfun-fun (fun args default-body)
  (pushnew fun *the-selectfuns*)
  (when (or (not (get fun 'select-wise))
	    (not (equal args (get fun 'piece-args)))
	    (not (equal default-body (get fun 'default-body))))
    (setf (get fun 'pieces) nil)
    (setf (get fun 'piece-args) args)
    (setf (get fun 'default-body) default-body)
    (setf (get fun 'select-wise) t)
    (invalidate-selectfun-compilation fun)))

(defun invalidate-selectfun-compilation (fun)
  (setf (symbol-function fun)
	(lambda (&rest args)
	  (redo-structures)
	  (compile-defuns (list (selectfun-definition fun)))
	  (apply fun args)))
  fun)

(defun selectfun-definition (fun)
  `(defun ,fun ,(get fun 'piece-args)
    (guarded-selectmatch ,(car (get fun 'piece-args))
     ,.(mapcar (lambda (piece)
		 (get fun piece))
	(get fun 'pieces))
     (:anything t ,@(get fun 'default-body)))))


(defun compute-selectfun-definitions ()
  (mapcar #'selectfun-definition *the-selectfuns*))
