;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)

(bnf (class (db-fix class-combinator)
	    (wishful-version-of-internal class class)))

(declare-variables (class ?db-fix)
		   (class-combinator ?comb ?fix-comb))


;We allow fix to be applied to operators WITH NO DOMAIN TYPE.
;Such an operator can be represented internally by a db-open expression.

(deftranslator fix (f)
  (selectmatch f
    ((lambda (?var) ?body)
     `(db-fix ,(translate `(class-combinator ,?var ,?body))))
    (:anything
      (ontic-error (format nil "illegal syntax for fix")))))

(deftranslator wishful-version-of (f tag)
  `(wishful-version-of-internal
    ,(translate `(quote ,f))
    ,(translate `(quote ,tag))))

(rule db-fix-at-most-one-1 ((= ?db-fix (db-fix ?comb)))
  (at-most-one ?db-fix))


;(untranslated-definition f) is a recursive definition of f, i.e., an expression body[f].

(defun recursively-define (symbol expression keylist &key (resize t))
  (selectmatch expression
    ((lambda ?args ?body)
     (let ((expression (untranslated-definition symbol)))
       (unless expression
	 (ontic-error
	   (format nil "attempt to construct a fixed point of the undefined function ~s" symbol)))
       (unless (internal-member symbol (translate expression))
	 (ontic-error
	   (format nil "attempt to construct a fixed point of the non-recursive function ~s" symbol)))
       (basic-define
	symbol
	(translate
	 `(lambda ,?args
	   ((fix (lambda (,symbol) ,expression))
	    ,@(mapcar #'car ?args))))
	keylist
	:resize resize))
     (if (acceptable-recursion? symbol expression)
	 (let ((lambda-exp (translate `(lambda ,?args ,?body))))
	   (execute-extension `(axiom (= ,symbol ,lambda-exp)))
	   (setf-undo (fixpoint-definition-of symbol) lambda-exp))
	 (progn
	   (ontic-warning (format nil "Warning:  unable to establish fixed point property for ~s"
				  symbol)
			  `(not (eq :yes (? (= ,symbol ,(translate `(lambda ,?args ,?body)))))))
	   nil)))
    (:anything
     (ontic-error
       (format nil "attempted recursive definition of something other than a thunk or operator")))))

(defun acceptable-recursion? (symbol definition)
  (or (well-founded-recursion? symbol definition)
      (and (monotone-recursion? symbol definition)
	   (or (continuous-recursion? symbol definition)
	       (bounded-recursion? symbol definition)))))

;a recursion is well founded if every recursive call reduces some well founded order.
;this test is not yet implemented.

(defun well-founded-recursion? (symbol definition)
  (declare (ignore symbol definition))
  nil)

(defun monotone-recursion? (symbol definition)
  (selectmatch definition
    ((lambda ?args ?body)
     (eq 'plus (tonicity-as-operator symbol (translate ?body) (length ?args))))
    (:anything
     (error "ill-formed definition in monotone-recursion?"))))


;the following tests whether the set of possible values of expressions
;increases monotonicially (or decreases monotonically) as symbol
;increases as an operator of n arguments.
;note that the symbol itself does not have this property --- if the operator
;meaning of the symbol of the symbol changes this does not produce a tonic
;influence on the set of values of the symbol.

(defun tonicity-as-operator (symbol expression arity)
  (cond ((eq expression symbol)
	 'both)
	((not (internal-member symbol expression))
	 'constant)
	(t
	 (selectmatch expression
	   ((apply (funcall0-operator) ?thunk)
	    (application-tonicity symbol (list ?thunk) arity))
	   ((apply :anything :anything)
	    (application-tonicity symbol (uncurry expression) arity))
	   ((if-builder ?phi (pair-classes ?x ?y))
	    (let ((test-tone (tonicity-as-operator symbol ?phi arity)))
	      (case test-tone
		(both 'both)
		(constant (combine-tonicities (tonicity-as-operator symbol ?x arity)
					      (tonicity-as-operator symbol ?y arity)))
		(t 'both))))
	   ((when ?phi ?x)
	    (combine-tonicities (tonicity-as-operator symbol ?phi arity)
				(tonicity-as-operator symbol ?x arity)))
	   ((amb ?x ?y)
	    (combine-tonicities (tonicity-as-operator symbol ?x arity)
				(tonicity-as-operator symbol ?y arity)))
	   ((intersection ?x ?y)
	    (combine-tonicities (tonicity-as-operator symbol ?x arity)
				(tonicity-as-operator symbol ?y arity)))
	   ((is ?x ?y)
	    (combine-tonicities (invert-tonicity (tonicity-as-operator symbol ?x arity))
				(tonicity-as-operator symbol ?y arity)))
	   ((db-some-such-that ?x ?phi)
	    (combine-tonicities (invert-tonicity (tonicity-as-operator symbol ?x arity))
				(tonicity-as-operator symbol ?phi arity)))
	   ((there-exists ?x)
	    (tonicity-as-operator symbol ?x arity))
	   ((at-most-one ?x)
	    (invert-tonicity (tonicity-as-operator symbol ?x arity)))
	   ((singleton :anything) 'both)
	   ((forall ?t (formula-combinator ?phi))
	    (combine-tonicities (invert-tonicity (tonicity-as-operator symbol ?t arity))
				(tonicity-as-operator symbol ?phi arity)))
	   ((not ?phi)
	    (invert-tonicity (tonicity-as-operator symbol ?phi arity)))
	   ((implies ?psi ?phi)
	    (combine-tonicities (invert-tonicity (tonicity-as-operator symbol ?psi arity))
				(tonicity-as-operator symbol ?phi arity)))
	   (:anything 'both)))))

(defun application-tonicity (symbol expression arity)
  (let ((op (uncurry-operator (car expression)))
	(args (cdr expression)))
    (cond ((eq op symbol)
	   (if (= arity (length args))
	       (reduce #'combine-tonicities
		       (mapcar (lambda (subexp) (tonicity-as-operator symbol subexp arity))
			       args)
		       :initial-value 'plus)
	       'both))
	  ((not (internal-member symbol op))
	   (reduce #'combine-tonicities
		   (mapcar (lambda (subexp) (tonicity-as-operator symbol subexp arity))
			   args)
		   :initial-value 'constant))
	  (t (selectmatch op
	       ((lambda ?args2 ?body)
		(if (not (= (length ?args2) (length args)))
		    'both
		    (reduce #'combine-tonicities
			    (mapcar (lambda (subexp) (tonicity-as-operator symbol subexp arity))
				    (cons ?body (append ?args2 args)))
			    :initial-value 'constant)))
	       (:anything 'both))))))

(defun uncurry (exp)
  (selectmatch exp
    ((apply ?f ?arg)
     (if (matches? ?f (apply . :anything))
	 (nconc (uncurry ?f) (list ?arg))
	 (list ?f ?arg)))
    (?exp ?exp)))

(defun uncurry-operator (op)
  (selectmatch op
    ((lambda0 ?body)
     `(lambda () ,?body))
    ((db-lambda ?t (db-class-combinator ?body))
     (let ((?new-body (uncurry-operator ?body)))
       (selectmatch ?new-body
	 ((lambda ?args ?body3)
	  `(lambda (,?t ,@?args) ,?body3))
	 (:anything `(lambda (,?t) ,?body)))))
    (:anything op)))

(defun invert-tonicity (sign)
  (cond ((eq sign 'plus) 'minus)
	((eq sign 'minus) 'plus)
	((eq sign 'constant) 'constant)
	((eq sign 'both) 'both)))

(defun combine-tonicities (tone1 tone2)
  (cond ((eq tone1 tone2) tone1)
	((eq tone1 'constant) tone2)
	((eq tone2 'constant) tone1)
	((eq tone1 'both) 'both)
	((eq tone2 'both) 'both)
	(t 'both)))

(defun continuous-recursion? (symbol definition)
  (selectmatch definition
    ((lambda ?args ?body)
     (continuous-as-operator? symbol (translate ?body) (length ?args)))
    (:anything
     (error "ill-formed definition in monotone-recursion?"))))

(defun continuous-as-operator? (symbol expression arity)
  (and (not (eq expression symbol))
       (or (not (internal-member symbol expression))
	   (selectmatch expression
	     ((apply (funcall0-operator) ?thunk)
	      (continuous-application? symbol (list ?thunk) arity))
	     ((apply :anything :anything)
	      (continuous-application? symbol (uncurry expression) arity))
	     ((if-builder ?phi (pair-classes ?x ?y))
	      (and (not (internal-member symbol ?phi))
		   (continuous-as-operator? symbol ?x arity)
		   (continuous-as-operator? symbol ?y arity)))
	     ((when ?phi ?x)
	      (and (not (internal-member symbol ?phi))
		   (continuous-as-operator? symbol ?x arity)))
	     ((amb ?x ?y)
	      (and (continuous-as-operator? symbol ?x arity)
		   (continuous-as-operator? symbol ?y arity)))
	     ((intersection ?x ?y)
	      (and (continuous-as-operator? symbol ?x arity)
		   (continuous-as-operator? symbol ?y arity)))
	     ((db-some-such-that ?x ?phi)
	      (and (not (internal-member symbol ?phi))
		   (continuous-as-operator? symbol ?x arity)))
	     (:anything
	      (not (internal-member symbol expression)))))))

(defun continuous-application? (symbol expression arity)
  (let ((op (car expression))
	(args (cdr expression)))
    (and (every (lambda (arg) (continuous-as-operator? symbol arg arity))
		args)
	 (or (and (eq op symbol)
		  (= arity (length args)))
	     (not (internal-member symbol op))
	     (selectmatch (uncurry-operator op)
	       ((lambda ?args2 ?body)
		(and (= (length ?args2) (length args))
		     (every (lambda (subexp) (continuous-as-operator? symbol subexp arity))
			    (cons ?body ?args2))))
	       (:anything nil))))))

;a recursion is bounded if one can prove that the fixpoint is under some expression that
;does not contain the fixpoint itself.

(defun bounded-recursion? (symbol definition)
  (declare (ignore symbol definition))
  nil)

(defun recursion-induction-bindings? (b)
  (and (consp b)
       (consp (car b))
       (let ((last-b (last b)))
	 (and (null (cdr last-b))
	      (selectmatch (car last-b)
		((:anything (?function-name . :anything))
		 (or (unless (untranslated-definition ?function-name)
		       (ontic-error (format nil "~s not defined" ?function-name)))
		     (unless (internal-member ?function-name (translate (untranslated-definition ?function-name)))
		       (ontic-error (format nil "~s not a recursive definition" ?function-name)))
		     t)))))))

(defmac induction-macro-add-tag
  (show-by-induction-on ?bindings
    ?phi
    . ?body)
  (recursion-induction-bindings? ?bindings)
  (lisp-let ((?tag (new-proof-variable 'INDUCTION))
	     (?function-name (caadar (last ?bindings)))
	     (?fixed-body (sublis (acons `(wishful-version-of ,?function-name)
					 `(wishful-version-of ,?function-name ,?tag)
					 nil)
				  ?body
				  :test #'equal)))
    (tagged-induction-on ?tag ?bindings ?phi . ?fixed-body)))

(defmac induction-macro-with-tag
  (show-by-induction-on ?tag ?bindings
    ?phi
    . ?body)
  (and (symbolp ?tag)
       (recursion-induction-bindings? ?bindings))
  (tagged-induction-on ?tag ?bindings ?phi . ?body))


(defsequent induction-sequent
    ((sequent ((axiom (forall ?induction-bindings
			(forall ((?n ((wishful-version-of ?function ?tag) . ?fun-args)))
			  ?phi)))
	       (axiom (forall ?fun-arg-bindings
			(is ((wishful-version-of ?function ?tag) . ?def-args)
			    (both (?function . ?def-args)
				  ?wishful-body-original-args)))))
       (theorem (forall ?induction-bindings
		  (forall ((?n ?wishful-body-induction-args))
		    ?phi)))))
  (theorem (forall ?induction-bindings (forall ((?n (?function . ?fun-args))) ?phi))))


(emacs-indent tagged-induction-on 3)


(defmac recursion-induction-macro
  (tagged-induction-on ?tag
      ?bindings
      ?phi
    . ?body)
  t
  (lisp-let ((?butlast-bindings (butlast ?bindings))  ;;; argument list bindings for args to rec. defined function
	     (?last-bind (car (last ?bindings)))      ;;; temp variable
	     (?n (car ?last-bind))                    ;;; the induction variable
	     (?function (caadr ?last-bind))      ;;; the name of the rec. defined function to induct on
	     (?args (cdadr ?last-bind))               ;;; the argument list for that function
	     (?fun-arg-bindings (second (untranslated-definition ?function)))
	     (?def-args (mapcar #'first ?fun-arg-bindings))
	     (?wishful-body (sublis (acons ?function
					   `(wishful-version-of ,?function ,?tag)
					   nil) 
				    (untranslated-definition ?function)))
	     (?wishful-body-args
	      (beta-reduce-curried-app
	       (translate `(,?wishful-body ,@?args))))
	     (?wishful-body-def-args
	      (beta-reduce-curried-app
	       (translate `(,?wishful-body ,@?def-args))))	     
	     )
    (show (forall ?bindings ?phi)
      (induction-sequent
       (sequent ((axiom (forall ?butlast-bindings
			  (forall ((?n ((wishful-version-of ?function ?tag) . ?args)))
			    ?phi)))
		 (axiom (forall ?fun-arg-bindings
			  (is ((wishful-version-of ?function ?tag) . ?def-args)
			      (both (?function . ?def-args)
				    ?wishful-body-def-args)))))
	 (show-internal (forall ?butlast-bindings (forall ((?n ?wishful-body-args)) ?phi))
	   (suppose-there-is ?butlast-bindings
	     (suppose-there-is ((?n ?wishful-body-args))
	       (show ?phi . ?body)))))))))


(emacs-indent show-by-induction-on 2)

(defun beta-reduce-curried-app (t-app)
  (selectmatch t-app
    ((apply ?app ?arg)
     (let ((fun (beta-reduce-curried-app ?app)))
       (selectmatch fun
	 ((db-lambda :anything (db-class-combinator ?body))
	  (expression-db-subst ?body ?arg 1))
	 (:anything `(apply ,fun ,?arg)))))
    (:anything t-app)))



;;;
;;; Code to determine syntactically whether fixed point property holds
;;; of a recursive definition

(property-macro well-behaved-positions)
(progn
  (setf (well-behaved-positions 'db-some-such-that) '(1))
;;  (setf (well-behaved-positions 'apply) '()) ;; special-cased below
  (setf (well-behaved-positions 'intersection) '(1 2))
  (setf (well-behaved-positions 'amb) '(1 2))
  (setf (well-behaved-positions 'when) '(2)))

(defun well-behaved-for (expr target num-args &key (application-depth 0) funcall0-argument allowable-position)
  (cond ((equal expr target)
	 (cond ((not allowable-position) nil)
	       ((null num-args) t)
	       ((zerop num-args) funcall0-argument)
	       (t (>= application-depth num-args))))
	((atom expr) t)
	(t (let* ((const (first expr))
		  (args (rest expr))
		  (positions (when (symbolp const)
			       (well-behaved-positions const))))
	     (cond ((eq const 'apply)
		    (and (well-behaved-for (first args) target num-args
					   :application-depth (1+ application-depth)
					   :allowable-position allowable-position)
			 (well-behaved-for (second args) target num-args
					   :funcall0-argument (equal (first args) '(funcall0-operator))
					   :allowable-position allowable-position)))
		   ((and (eq const 'db-lambda) (> application-depth 0))
		    (and (well-behaved-for (first args) target num-args
					   :allowable-position allowable-position)
			 (well-behaved-for (second (second args)) target num-args
					   :application-depth (1- application-depth)
					   :allowable-position allowable-position)))
		   (positions
		    (if (= (length args) 1)
			(well-behaved-for (first args) target num-args
					  :allowable-position (and (member 1 positions :test #'=) allowable-position))
			(and (well-behaved-for (first args) target num-args
					       :allowable-position (and (member 1 positions :test #'=)
									allowable-position))
			     (well-behaved-for (second args) target num-args
					       :allowable-position (and (member 2 positions :test #'=)
									allowable-position)))))
		   (t
		    (every (lambda (arg) (well-behaved-for arg target num-args)) args)))))))
