;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)

(defmacro forward-declare-nondeterministic-functions (&rest functions)
  `(progn
     ,@(mapcar #'(lambda (x) `(forward-declare-nondeterministic ,x))
	       functions)))

(defun export-nondeterministic (symbols)
  (mapcar #'(lambda (sym)
	      (export
		(intern (format nil "~A-NONDETERMINISTIC"  sym)
			(symbol-package sym)))
	      (export sym))
	  (if (listp symbols) symbols (list symbols))))

(export-nondeterministic '(member? determine-matches))

;;; KCZ Introduced variable-bound-p so that we can bind variables
;;; to NIL.
(defstruct (expression-variable (:conc-name nil)
				(:print-function print-expression-var)
				(:predicate expression-variable?))
  variable-pname
  variable-type
  variable-binding
  variable-bound-p
  binding-noticers)

;;; KCZ modified to work with variable-bound-p
(defun print-expression-var (variable stream x)
  (declare (ignore x))
  (if (variable-bound-p variable)
      (format stream "[~s]" (variable-binding variable))
      (format stream "[~s]" (variable-pname variable))))

(defun create-expression-variable (name-symbol &optional (type :untyped))
  (make-expression-variable :variable-pname name-symbol :variable-type type))

;;; 5/10 KCZ (modified to keep internal structure eq)
(defun apply-substitution (exp)
  (let ((imm (immediate-binding exp)))
    (if (consp imm)
	(let ((subst (cons (apply-substitution (car imm))
			   (apply-substitution (cdr imm)))))
	  (if (and (eq (car subst) (car imm))
		   (eq (cdr subst) (cdr imm)))
	      imm
	      subst))
	imm)))

;;; 5/10 KCZ (introduced)
(defun unbound-variable-p (exp)
  (cond ((null exp) nil)
	((expression-variable? exp)
	 (not (variable-bound-p exp)))
	((not (consp exp)) nil)
	(t
	  (or (unbound-variable-p (car exp))
	      (unbound-variable-p (cdr exp))))))

;;; KCZ modified to work with variable-bound-p
(defun immediate-binding (exp)
  (if (not (expression-variable? exp))
      exp
      (let ((binding (variable-binding exp)))
	(if (variable-bound-p exp)
	    (immediate-binding binding)
	    exp))))

;;; KCZ modified to work with variable-bound-p
(defun ecbind! (variable expression)
  (when (occurs-in? variable expression)
    (fail))
  (when (not (equal (expression-type variable) (expression-type expression)))
    (fail))
  (when (variable-bound-p variable)
    (error "attempt to bind a variable that is already bound"))
  (locally-setf (variable-binding variable) expression)
  (locally-setf (variable-bound-p variable) t)
  (mapc 'funcall (binding-noticers variable)))

(defun occurs-in? (var expression)
  (let ((imm (immediate-binding expression)))
    (cond ((eq var imm) t)
	  ((consp imm)
	   (or (occurs-in? var (car imm))
	       (occurs-in? var (cdr imm))))
	  (t nil))))

(defun assert-equal! (exp1 exp2)
  (let ((imm1 (immediate-binding exp1))
	(imm2 (immediate-binding exp2)))
    (cond ((equal imm1 imm2) t)
	  ((expression-variable? imm1)
	   (ecbind! imm1 imm2))
	  ((expression-variable? imm2)
	   (ecbind! imm2 imm1))
	  ((and (consp imm1) (consp imm2))
	   (assert-equal! (car imm1) (car imm2))
	   (assert-equal! (cdr imm1) (cdr imm2)))
	  ((not (and (atom imm1) (atom imm2) (eq imm1 imm2)))
	   (fail)))))

;;; KCZ 5/10
(defun structure-compatable-p (exp1 exp2)
  (cond ((or (expression-variable? exp1) (expression-variable? exp2)) t)
	((eq exp1 exp2) t)
	((and (consp exp1) (consp exp2))
	 (and (structure-compatable-p (car exp1) (car exp2))
	      (structure-compatable-p (cdr exp1) (cdr exp2))))))

(defun known-equal? (exp1 exp2)
  (let ((imm1 (immediate-binding exp1))
	(imm2 (immediate-binding exp2)))
    (or (eq imm1 imm2)
	(and (consp imm1)
	     (consp imm2)
	     (known-equal? (car imm1) (car imm2))
	     (known-equal? (cdr imm1) (cdr imm2))))))

(defun possibly-equal? (exp1 exp2)
  (one-value (progn (assert-equal! exp1 exp2) t)
	     nil))

(defun assert-not-equal! (exp1 exp2)
  (when (possibly-equal? exp1 exp2)
    (let ((pair (cons exp1 exp2))
	  (self nil))
      (let ((noticer #'(lambda ()
			 (when (known-equal? exp1 exp2)
			   (fail))
			 (dolist (var (expression-variables-of pair))
			   (unless (member self (binding-noticers var))
			     (locally-setf (binding-noticers var)
					   (cons self (binding-noticers
							var))))))))
	(setf self noticer)
	(funcall noticer)))))

(defun expression-variables-of (exp)
  (let ((variables nil))
    (labels ((variables-of-2 (exp2)
	       (let ((imm (immediate-binding exp2)))
		 (cond ((expression-variable? imm)
			(pushnew imm variables))
		       ((consp exp2)
			(variables-of-2 (car exp2))
			(variables-of-2 (cdr exp2)))))))
	    (variables-of-2 exp))
    variables))

(defmacro if-equal (exp1 exp2 &body cases)
  (let ((exp1var (gensym "EXP1-"))
	(exp2var (gensym "EXP2-")))
    `(let ((,exp1var ,exp1)
	   (,exp2var ,exp2))
       (either (progn (assert-equal! ,exp1var ,exp2var)
		      ,(first cases))
	       (progn (assert-not-equal! ,exp1var ,exp2var)
		      ,(second cases))))))

;; Open-coded if-equal, above, in the body of when-equal,
;; so that when-equal is deterministic.
(defmacro when-equal (x y &body body)
  `(progn
     (assert-equal! ,x ,y)
     ,@body))

(defun create-expression (template)
  (let ((subst nil))
    (labels ((create-copy2 (template)
	       (cond ((ecvariable? template)
		      (or (cdr (assoc template subst))
			  (let ((new-var
				  (create-expression-variable
				    (eccopy-var template)
				    (expression-type template))))
			    (push (cons template new-var) subst)
			    new-var)))
		     ((consp template)
		      (cons (create-copy2 (car template))
			    (create-copy2 (cdr template))))
		     (t template))))
      (create-copy2 template))))

;;; KCZ 5/13
(defun create-expression-rv (template vars)
  (let ((subst nil))
    (labels ((create-copy2 (template)
	       (cond ((member template vars)
		      (or (cdr (assoc template subst))
			  (let ((new-var
				  (create-expression-variable
				    (eccopy-var template)
				    (expression-type template))))
			    (push (cons template new-var) subst)
			    new-var)))
		     ((consp template)
		      (cons (create-copy2 (car template))
			    (create-copy2 (cdr template))))
		     (t template))))
      (create-copy2 template))))

(defun ecvariable? (x)
  (and (symbolp x) (string= "?" (subseq (string x) 0 1))))

;;; KCZ 5/13
(defun variables-of (exp)
  (cond ((null exp) nil)
	((ecvariable? exp) (list exp))
	((not (consp exp)) nil)
	(t
	  (append (variables-of (car exp))
		  (variables-of (cdr exp))))))

(defun eccopy-var (var)
  (let ((string (string var)))
    (let ((pos (position #\- string)))
      (if pos
	  (gensym (subseq string 0 (1+ pos)))
	  (gensym (concatenate 'string string "-"))))))



(defun test-unify (e1 e2)
  (one-value
    (progn (assert-equal! e1 e2)
	   (apply-substitution e1))
    'fail))


(defmacro type-declarations (&body declarations)
  (let ((decl (gensym "DACEL-")))
    `(dolist (,decl ',declarations)
       (make-declaration (first ,decl) (second ,decl)))))

(defun make-declaration (symbol type)
  (setf (get symbol 'symbol-type) type))

(defun well-formed-list? (x)
  (and (consp x)
       (null (cdr (last x)))))

(defun expression-type (expression)
  (cond ((expression-variable? expression) (variable-type expression))
	((symbolp expression)
	 (let ((type (get expression 'symbol-type)))
	   (if type
	       type
	       :untyped)))
	((not (well-formed-list? expression)) :untyped) ;; 5/6 KCZ
	(t
	  (let ((exp-type (expression-type (first expression)))
		(?input-types nil)
		(?output-type nil))
	    (if (eq exp-type :untyped)
		(progn
		  (dotimes (x (length (cdr expression)))
		    (setq ?input-types (cons :untyped ?input-types)))
		  (setq ?output-type :untyped))
		(progn
		  (setq ?input-types (first exp-type))
		  (setq ?output-type (second exp-type))))
	    (unless (= (length ?input-types) (length (cdr expression)))
	      (error "wrong number of arguments in ~s" expression))
	    (mapc #'(lambda (input input-type)
			(unless (eq (expression-type input) input-type)
			  (error "~s is an illegal argument in ~s"
				 input
				 expression)))
		    (cdr expression)
		    ?input-types)
	      ?output-type))))

(defun combination-type (types)
  (let* ((type (first types))
	 (?input-types nil)
	 (?output-type nil))
    (if (eq type :untyped)
	(progn
	  (dotimes (x (length (cdr types)))
	    (setq ?input-types (cons :untyped ?input-types)))
	  (setq ?output-type :untyped))
	(progn
	 (setq ?input-types (first type))
	 (setq ?output-type (second type))))
    (when (and (= (length ?input-types) (length (cdr types)))
	       (every #'equal
		      (cdr types)
		      ?input-types))
      ?output-type)))


;;;  utilities built on exp-con and screamer


(defun-nondeterministic member? (x set)
  (and (not (null set))
       (if-equal x (car set)
	 t
	 (member? x (cdr set)))))


(defun-nondeterministic determine-matches (x set)
  (and (not (null set))
       (let ((cdr-result (determine-matches x (cdr set))))
	 (if-equal x (car set)
	   t
	   cdr-result))))

(defun-nondeterministic subset? (set1 set2)
  (or (null set1)
      (and (member? (car set1) set2)
	   (subset? (cdr set1) set2))))

(defun-nondeterministic intersects? (set1 set2)
  (and (not (null set1))
       (or (member? (car set1) set2)
	   (intersects? (cdr set1) set2))))

(defun-nondeterministic set-diff (set1 set2)
  (cond ((null set1) nil)
	((member? (car set1) set2)
	 (set-diff (cdr set1) set2))
	(t
	 (cons (car set1)
	       (set-diff (cdr set1) set2)))))

(defun-nondeterministic rem-dups (set)
  (when set
    (if (member? (car set) (cdr set))
	(rem-dups (cdr set))
	(cons (car set) (rem-dups (cdr set))))))

(defun-nondeterministic set-union (set1 set2)
  (cond ((null set1)
	 set2)
	((member? (car set1) set2)
	 (set-union (cdr set1) set2))
	(t
	 (cons (car set1) (set-union (cdr set1) set2)))))

(defmacro equal? (x y)
  `(if-equal ,x ,y
     t
     nil))

(defmacro ensuring (phi &body body)
  `(if ,phi
       (progn ,@body)
       (fail)))

(defmacro sst (type var phi)
  `(let ((,var ,type))
     (if ,phi
	 ,var
	 (fail))))

(defun gensym-var-of-type (type &optional (string "VAR-") (package *package*))
  (let ((var
	  (loop
	    (let ((var-string (format nil "?~s" (gentemp string))))
	      (when (not (find-symbol var-string package))
		(return (intern var-string package)))))))
    (make-declaration var type)
    var))

(defun-nondeterministic nd-some (pred list)
  (when list
    (if (funcall-nondeterministic pred (first list))
	t
	(nd-some pred (rest list)))))

(defun-nondeterministic nd-every (pred list)
  (or (null list)
      (and (funcall-nondeterministic pred (first list))
	   (nd-every pred (rest list)))))

(defun-nondeterministic nd-every2 (pred list1 list2)
  (or (and (null list1)
	   (null list2))
      (and (not (null list1))
	   (not (null list2))
	   (funcall-nondeterministic pred (first list1) (first list2))
	   (nd-every2 pred (rest list1) (rest list2)))))


(defun convert-exp-vars (exp)
  (let ((exp (immediate-binding exp)))
    (cond ((consp exp) (cons (convert-exp-vars (car exp))
			     (convert-exp-vars (cdr exp))))
	  ((expression-variable? exp)
	   (let ((var-sym (intern (symbol-name (variable-pname exp)))))
	     (make-declaration var-sym (variable-type exp))
	     var-sym))
	  (t exp))))

(defun alpha-rename (expression)
  (create-expression (convert-exp-vars expression)))
