;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic)

(defun view-nodes (i j)
  (rprint (mapcar
	   (lambda (k)
	     (let ((p  (smallest-lhs-production (n k))))
	       (if p
		   (cons k (phrase-constructor p))
		   nil)))
	   (numbers-between i j))))

(defun numbers-between (i j)
  (if (= i j) (list j) (cons i (numbers-between (1+ i) j))))

(defun view-last-nodes (n)
  (view-nodes (- *node-count* n) (- *node-count* 1)))

(defun view-nodes-of-type (i j const)
  (rprint (remove-if 'null
		     (mapcar
		      (lambda (k)
			(let ((p (smallest-lhs-production (n k))))
			  (when (and p (eq const (phrase-constructor p)))
			    (list k (macro-invert (object-expression (n k)))))))
		      (numbers-between i j)))))


(defvar *the-list* nil)

(defun nodes-satisfying (k &optional pred view-fun)
  (setq *the-list* nil)
  (dotimes (j k)
    (let ((nn  (+ (- *node-count* k) j)))
      (when (or (null pred) (funcall pred (n nn)))
	(push (n nn) *the-list*))))
  (the-items view-fun))
  
(defun prods-satisfying (k pred &optional view-fun)
  (setq *the-list* nil)
  (dotimes (j k)
    (let ((nn  (+ (- *node-count* k) j)))
      (dolist (prod (productions-from (n nn)))
	(when (funcall pred prod)
	  (push prod *the-list*)))))
  (the-items view-fun))

(defun the-items (&optional view-function)
  (let ((n 1))
    (dolist (item *the-list*)
      (format t "~%~s  ~s" n (if view-function (funcall view-function item) item))
      (incf n))))

(defun the-item (n)
  (nth (1- n) *the-list*))

(defun view-satisfying (k &optional pred)
  (nodes-satisfying k pred (lambda (node)
			     (cons (node-number node)
				   (phrase-constructor (first (productions-from node)))))))

(defun view-prods-satisfying (k &optional pred)
  (prods-satisfying k pred (lambda (prod) (phrase-constructor prod))))

(defmacro equivalents (exp)
  `(equivalents-fun ',exp))

(defun equivalents-fun (exp)
  (value-from-undo-frame
   (rprint
    (remove-if 'null
	       (mapcar (lambda (prod)
			 (when (not (member (phrase-constructor prod) '(apply-class-combinator apply-formula-combinator)))
			   (macro-invert (cons (phrase-constructor prod) (mapcar 'object-expression (rhs prod))))))
		       (productions-from (ti exp)))))))

(defmacro types (exp)
  `(types-fun ',exp))

(defun types-fun (exp)
  (value-from-undo-frame
   (rprint
    (mapcar #'print-expression (is-forward (ti exp))))))


; just like describe, but returns the object so that * works
(defun show (obj)
  (describe obj)
  obj)

(defun interned? (exp)
  (when (= (value-from-undo-frame (ti exp) *node-count*) *node-count*)
    (ti exp)))

(defvar *discovery-breaks* nil)
(defextender break-on-discovery (phi)
  (push-undo (ti phi) *discovery-breaks*))

(defpiece (notice-undo-popping :discovery-break) ()
  (dolist (phi *discovery-breaks*)
    (when (or *contradiction*
	      (eq :true (is-true? phi)))
      (break "~%I have discovered ~s" phi))))

(emacs-indent break-on-discovering 1)
(defsequent discovery-sequent
    ((sequent ((break-on-discovery ?phi))
       (theorem ?psi)))
  (theorem ?psi))

(defmac discovery
  (break-on-discovering ?phi . ?body)
  t
  (discovery-sequent
   (sequent ((break-on-discovery ?phi))
     (progn . ?body))))
			
(defun singletons-of-type (type)
  (remove-duplicates (remove-if #'(lambda (s) (or (not (not-bogus? s))
						  (not (value-from-undo-frame (is? s (tintern type))))))
				*singletons*)))

;to get the following information set the variable *monitor-goto-context*
;to t.

(defun extend-frame-info-sorted ()
  (rprint
   (setf * (sort (mapcar #'(lambda (info) (list (first info) (/ (third info)
								1000000.0)))
			 *EXTEND-FRAME-INFO*) #'< :key #'second))))

(defun total-frame-info ()
  (/ (reduce #'+ (mapcar #'third *EXTEND-FRAME-INFO*)) 1000000.0))



(defun time-file (filename)
  (clrhash *proof-cache*)
  (clrhash *sequent-cache*)
  (let ((forms (file-forms filename)))
    (let ((ontic-forms (mapcar 'coerce-to-ontic-form forms)))
      (time (mapc 'eval ontic-forms)))))

(defun coerce-to-ontic-form (form)
  (selectmatch form
    ((defstruct . ?rest)
     `(def-o-struct ,@?rest))
    ((provide . :anything) t)
    ((require . :anything) t)
    ((defmodule . :anything)
     form)
    ((define . ?rest)
     `(defontic ,@?rest))
    (:anything
     `(defproof ,form))))

(defun map-on-nodes (fun)
  (dotimes (j *node-count*)
    (let ((node (n (1+ j))))
      (when (not-bogus? node)
	(funcall fun node)))))

(defun map-on-productions (fun)
  (map-on-nodes (lambda (node)
		  (mapc fun (productions-from node)))))

(defun count-productions ()
  (let ((count 0))
    (map-on-productions (lambda (ignore)
			 (declare (ignore ignore))
			 (incf count)))
    count))

(defun map-on-closed-nodes (fun)
  (map-on-nodes (lambda (node)
		  (when (db-closed? node)
		    (funcall fun node)))))

(defun all-lemmas ()
  (let ((lemmas nil))
    (map-on-nodes (lambda (node)
		    (when (lemma? node)
		      (push node lemmas))))
    (nreverse (mapcar 'print-expression lemmas))))

(defun current-library ()
  *lemma-library*)

(defun install-library (library)
  (dolist (form library)
    (unless (and (eq (car form) 'define)
		 (or (definition-of (second form))
		     (and (listp (second form))
			  (definition-of (first (second form))))))
      (execute-extension form)))
  t)

(defun load-library (filename)
  (ontic-init)
  (let ((library nil))
    (with-open-file (istream filename :direction :input)
      (setq library (read istream))
      (setq *ontic-libraries* (read istream)))
    (install-library library)
    (setq *lemma-library* library))
  t)

(defun save-library (filename)
  (let ((*print-level* nil)
	(*print-length* nil)
	(*print-circle* t))
    (with-open-file (istream filename :direction :output :if-exists :supersede
			     :if-does-not-exist :create)
      (format istream "~s" *lemma-library*)
      (format istream "~s" *ontic-libraries*)))
  t)

(defun lemma? (node)
  (and (db-closed? node)
       (typep node 'formula)
       (eq (is-true? node) :true)
       (some (lambda (prod)
	       (eq (phrase-constructor prod) 'db-forall))
	     (productions-from node))))

(defun print-expression (object)
  (macro-invert (object-expression object)))

(defun faith-load (filename)
  (let ((forms (file-forms filename))
	(*faith-mode* t))
    (let ((ontic-forms (mapcar 'coerce-to-ontic-form forms)))
      (mapc 'eval ontic-forms))))

(defun ontic-load (filename)
  (let ((forms (file-forms filename))
	(*faith-mode* nil))
    (let ((ontic-forms (mapcar 'coerce-to-ontic-form forms)))
      (mapc 'eval ontic-forms))))


(defun lemmas-with-handle (handle)
  (let ((lemmas nil))
    (dolist (prod (productions-to handle))
          (when (and (eq (phrase-constructor prod) 'pair-classes)
		 (eq handle (first (rhs prod))))
	    (dolist (formula (forall-handle-backward (lhs prod)))
	      (push (print-expression formula) lemmas))))
    lemmas))




;========================================================================
;the query macro
;========================================================================

(defmacro query (antecedent &rest antecedents)
  `(query-top-fun ',(cons antecedent antecedents)))

(defun query-top-fun (antecedents)
  (let ((variables (variables antecedents)))
    (rprint 
     (cons 'relation
	   (mapcar (lambda (bindings)
		     (mapcar (lambda (cell)
			       (list (car cell)
				     (print-expression (cdr cell))))
			     bindings))
		   (project-fun (query-fun antecedents) variables))))))

(defmacro query-count (antecedent &rest antecedents)
  `(length (query-fun ',(cons antecedent antecedents))))

(defmacro query-substs (antecedent &rest antecedents)
  `(query-fun ',(cons antecedent antecedents)))

(defun query-fun (antecedents)
  (mvlet (((fixed-antecedents initial-bindings) (fix-antecedents antecedents)))
    (let ((var (first (variables (first fixed-antecedents))))
	  (result nil))
      (labels
	  ((query-internal (antecedents bindings)
	     (unless (eq bindings 'fail)
	       (if (null antecedents)
		   (pushnew bindings result :test #'equal)
		   (selectmatch (first antecedents)
		     ((= ?var2 ?template)
		      (let ((var2-node (node-value ?var2 bindings)))
			(if var2-node
			    (dolist (prod (productions-from var2-node))
			      (query-internal (cdr antecedents)
					      (bind-production (car antecedents)
							       prod
							       bindings)))
			    (let ((internal-var (some (lambda (var)
							(node-value var bindings))
						      ?template)))
			      (unless internal-var
				(error "unthreaded equality antecedent pattern"))
			      (dolist (prod (productions-to internal-var))
				(query-internal (cdr antecedents)
						(bind-production (car antecedents)
								 prod
								 bindings)))))))
		     ((?pred ?arg)
		      (let ((arg-node (node-value ?arg bindings)))
			(unless arg-node
			  (error "unthreaded pattern"))
			(when (eq :true (funcall (add-? ?pred) arg-node))
			  (query-internal (cdr antecedents)
					  bindings))))
		     ((?pred ?arg1 ?arg2)
		      (let ((arg1-val (node-value ?arg1 bindings)))
			(if arg1-val
			    (dolist (arg2-val (funcall (create-name ?pred 'forward) arg1-val))
			      (query-internal (cdr antecedents)
					      (query-bind! ?arg2 arg2-val bindings)))
			    (let ((arg2-val (node-value ?arg2 bindings)))
			      (unless arg2-val
				(error "unthreaded pattern"))
			      (dolist (arg1-val (funcall (create-name ?pred 'backward) arg2-val))
				(query-internal (cdr antecedents)
						(query-bind! ?arg1 arg1-val bindings))))))))))))
	(map-on-nodes (lambda (node)
			(when (and (subtypep (type-of node) (variable-category var))
				   (not-bogus? node))
			  (query-internal fixed-antecedents (query-bind! var node initial-bindings)))))
	result))))

(defmacro project (query &rest variables)
  `(project-fun ,query ',variables))

(defun project-fun (relation variables)
  (mapcar (lambda (bindings)
	    (remove-if (lambda (pair) (not (member (car pair) variables)))
		       bindings))
	  relation))

(defmacro sproject (query var)
  `(sproject-fun ,query ',var))

(defun sproject-fun (relation var)
  (mapcar (lambda (subst) (assoc-value var subst))
	  relation))

(defun sproject-sort (substs var &optional bound)
  (let ((remaining-substs substs)
	(result nil))
    (while remaining-substs
      (let ((subst (car remaining-substs)))
	(let ((val (assoc-value var subst)))
	  (let ((these-substs (remove-if-not (lambda (subst2) (eq val (assoc-value var subst2)))
					     remaining-substs)))
	    (setq remaining-substs (set-difference remaining-substs these-substs))
	    (push (list (length these-substs) (macro-invert (object-expression val)))
		  result)))))
    (let ((sorted-r (sort result '>= :key 'car)))
      (If (null bound)
	  (rprint sorted-r)
	  (rprint (first-n bound sorted-r))))))

(defun first-n (n list)
  (if (or (< n 1) (null list))
      nil
      (cons (car list) (first-n (1- n) (cdr list)))))
	  
			  
			 


(defun bind-production (antecedent prod bindings)
  (query-bind-several! (cdr (third antecedent))
		       (rhs prod)
		       (query-bind! (second antecedent)
				    (lhs prod)
				    (constructor-bind (car (third antecedent))
						      (phrase-constructor prod)
						      bindings))))

(defun constructor-bind (constructor-var constructor bindings)
  (if (variable? constructor-var)
      (let ((const-val (assoc-value constructor-var bindings)))
	(if const-val
	    (if (eq constructor const-val)
		bindings
		'fail)
	    (acons constructor-var constructor bindings)))
      (if (eq constructor-var constructor)
	  bindings
	  'fail)))


(defun query-bind-several! (variables nodes bindings)
  (if (and (null variables) (null nodes))
      bindings
      (if (or (null variables) (null nodes) (eq bindings 'fail))
	  'fail
	  (query-bind-several! (cdr variables)
			     (cdr nodes)
			     (query-bind! (car variables) (car nodes) bindings)))))

(defun query-bind! (variable node bindings)
  (if (or (not (not-bogus? node))
	  (eq bindings 'fail))
      'fail
      (let ((node-val (node-value variable bindings)))
	(if node-val
	    (if (eq node-val node)
		bindings
		'fail)
	    (if (subtypep (type-of node) (variable-category variable))
		(acons variable node bindings)
		'fail)))))

(defun node-value (variable bindings)
  (if (non-terminal-p variable)
      variable
      (assoc-value variable bindings)))

(defun fix-antecedents (antecedents)
  (let ((result-bindings nil)
	(result-antecedents nil))
    (labels ((fix-antecedent (antecedent)
	       (selectmatch (translate antecedent)
		 ((= ?var (?constructor . ?args))
		  (push `(=
			  ,(flatten ?var)
			  (,?constructor ,@(mapcar #'flatten ?args)))
			result-antecedents))
		 ((= . :anything)
		  (error "the second argument to an equality must be an application of a phrase constructor"))
		 ((?pred . ?args)
		  (push `(,?pred ,@(mapcar #'flatten ?args))
			result-antecedents))))
	     (flatten (exp)
	       (cond ((variable? exp) exp)
		     ((null (variables exp))
		      (let ((newvar (gentemp "?VAR-")))
			(setf (variable-category newvar)
			      (expression-category exp))
			(push (cons newvar (ti exp))
			      result-bindings)
			newvar))
		     (t
		      (let ((newvar (gentemp "?VAR-")))
			(setf (variable-category newvar)
			      (expression-category exp))
			(push `(= ,newvar ,(cons (car exp) (mapcar #'flatten (cdr exp))))
			      result-antecedents)
			newvar)))))			      
      (mapc #'fix-antecedent antecedents)
      (values result-antecedents result-bindings))))



