;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'ontic :use '(util lisp))

;This file implements expression interning with equational reasoning.
;Equational reasoning is a form of inference.  For debugging reasons
;it is very important that it is possible to record the justifications for
;inference so that erroneous conclusions can be traced to either erroneous
;premises or erroneous inference rules.  This file starts with some code
;for walking justification trees.



;========================================================================
;ontic-init
;========================================================================


(defun ontic-init ()
  (while (not (base-frame?))
    (util::pop-undo-frame))
  (clear-expressions)
  (redo-structures)
  (ontic-init-phase0)
  (ontic-init-phase1)
  (ontic-init-phase2)
  (ontic-init-phase3)
  t)

(defpiecefun ontic-init-phase0 ())

(defpiecefun ontic-init-phase1 ())

(defpiecefun ontic-init-phase2 ())

(defpiecefun ontic-init-phase3 ())




;********************************************************************
;Justifications
;********************************************************************

;The following is a flag which, if set to t, causes justifications to start being
;recorded.  Justification recording can be turned on at any time.  Most justification
;trees will terminate in facts with unrecorded justifications.  This is true even
;if justification recording has been turned on at all times.  Facts generated by
;top level proofs are not given top level justifications.


(defvar *record-justifications* nil)

;Justifications are organized around the concept of a nonlinear frame.  A nonlinear frame
;is analogous to an ordinary lisp stack frame except that the nonlinear frames
;are organized in a nonlinear dag (directed acyclic graph) rather than a linear stack.
;A nonlinear frame represent a "computational event", either
;a certain procedure being invoked on certain arguments, or a procedure invocation
;returning a certain value.
;The former is called an "invocation frame" and the latter a "return frame".
;Each frame has a justification --- a set of computational events which imply
;the event of the frame.  For example, the invocation frame for
;(equate! a b) might be justified by the invocation frames (assert-production a f c)
;and (assert-production b f c).  A justification is a data structure that contains
;both a comment expression and a list of justifying frames.  Each element of the
;list of justifying frames may be a cons cell rather than a frame in which
;case the car of the cell must be a frame and the cdr of the cell is a way
;of printing the frame when it is printed as part of the justification.

(defmacro make-invocation-frame ((fun . args) justification)
  `(lambda ()
     (cons (list ',fun ,@args) ,justification)))

(emacs-indent make-invocation-frame 1)

(defmacro make-return-frame ((fun . args) returned-value justification)
  `(lambda ()
     (cons (list (list ',fun ,@args) '==> ,returned-value)
      ,justification)))

(emacs-indent make-return-frame 2)

(defun frame-descriptor (frame)
  (car (funcall frame)))

(defun frame-justification (frame)
  (cdr (funcall frame)))

(defmacro make-justification (comment subframes)
  `(lambda ()
    (cons ,comment ,subframes)))

(emacs-indent make-justification 1)

(defun justification-comment (justification)
  (car (funcall justification)))

(defun justification-subframes (justification)
  (cdr (funcall justification)))

(defun add-frames (justification &rest frames)
    (when *record-justifications*
      (unless justification
	(error "failure to provide expected justification"))
      (make-justification (justification-comment justification)
	(append (justification-subframes justification)
		(remove-if 'null frames)))))

;we now have code for walking the frame call dag.

(defvar *frame-stack* nil)

(defvar *current-frame* nil)

(defun push-frame (frame)
  (push *current-frame* *frame-stack*)
  (setf *current-frame* frame))

(defun pop-frame ()
  (setf *current-frame* (pop *frame-stack*))) 

(defmacro why (expression)
  `(why-fun ',expression))


(defun why-fun (expression)
  (cond ((numberp expression)
	 (let ((parent-frame (parent-frame expression)))
	   (cond (parent-frame
		  (push-frame parent-frame)
		  (display-frame))
		 (t
		  (format t "~%there is no justification for ~s" expression)
		  nil))))	 
	((eq expression 'previous)
	 (pop-frame)
	 (display-frame))
	((equal expression '(false))
	 (if *contradiction*
	     (goto-frame (gethash (ti '(false)) is-true-frame-hash-table))
	     "there is no contradiction present"))
	(t (let ((*record-justifications* t))
	     (mvlet (((node iframe) (cintern (translate expression)
					     :justification
					     (make-justification 'call-from-why
					       (make-invocation-frame (why expression)
						 (make-justification 'user nil))))))
	       (if (not (and (formula-p node)
			     (eq :true (is-true? node))))
		   (format nil "~s is not a true formula" expression)
		   (goto-frame (if iframe
				   (make-return-frame (? expression) (is-true? node)
				     (make-justification 'intern&get-truth
				       (list iframe
					     (gethash node is-true-frame-hash-table))))
				   (gethash node is-true-frame-hash-table)))))))))

(defun goto-frame (frame)
  (cond (frame
	 (push-frame frame)
	 (display-frame))
	(t
	 (format t "no recorded justification")
	 nil)))

(defun display-frame ()
  (rprint (expand-nodes (frame-expression *current-frame*))))

(defun display-justification (just)
  (goto-frame (make-invocation-frame (dummy) just)))

(defun frame-expression (frame)
  (selectmatch (funcall frame)
    ((?frame-descriptor . ?justification)
     (if (null ?justification)
	 `(,?frame-descriptor without-recorded-justification)
	 (selectmatch (funcall ?justification)
	   ((?call-descriptor . ?pframes)
	    `(,?frame-descriptor
	      ,?call-descriptor
	      ,(let ((n 0))
		 (cons 'from
		       (mapcar #'(lambda (subjust)
				   (incf n)
				   (list n (cond ((consp subjust)
						  (cdr subjust))
						 ((null subjust)
						  'unknown-antecedent)
						 (t (frame-descriptor subjust)))))
			       ?pframes))))))))))

(defun expand-nodes (exp)
  (cond ((non-terminal-p exp)
	 (object-definition exp))
	((consp exp)
	 (cons (expand-nodes (car exp))
	       (expand-nodes (cdr exp))))
	(t exp)))

(defun parent-frame (n)
  (let ((justification (cdr (funcall *current-frame*))))
    (unless justification
      (error "there is no subjustification ~s" n))
    (let ((parent-frames (cdr (funcall justification))))
      (when (or (< n 1) (> n (length parent-frames)))
	(error "there is no subjustification ~s" n))
      (let ((parent-frame (nth (1- n) parent-frames)))
	(if (consp parent-frame)
	    (car parent-frame)
	    parent-frame)))))

(equal-hash-table-macro prod-frame)

(defpiece (ontic-init-phase0 clear-prod-justs) ()
  (clear-all-prod-frame))

(hash-table-macro nonterm-definition)

(defpiece (ontic-init-phase0 clear-next-find-justs) ()
  (clear-all-nonterm-definition))

(defmacro why-interned (exp)
  `(why-interned-fun ',exp))

(defun why-interned-fun (exp)
  (when (interned? exp)
    (let ((*record-justifications* t))
      (mvlet (((node iframe) (cintern (translate exp))))
	(declare (ignore node))
	(goto-frame iframe)))))

(defun why-interned-node (node)
  (why-interned-prod (first (productions-from node))))

(defun why-interned-prod (prod)
  (goto-frame (prod-frame (prod-expression prod))))







;********************************************************************
;Here is the interning and equality system
;********************************************************************

;Equivalence classes of terms are represented by grammars --- for
;each equivalence class there is a non-terminal symbol in the grammar
;that generates the (possibly infinite) set of terms in that equivalence
;class.

;Each production in the grammar is of one of the following three forms
;
;   X -> (f)
;   X -> (f Y)
;   X -> (f Y Z)

;In the above productions f is called a phrase constructor and the list
;of arguments of f is calles the right hand side (rhs) of the production.
;A production is formalized as a triple of an lhs, a phrase constructor,
;and an rhs.  The lhs is a non-terminal and the rhs is a (possibly empty)
;list of non-terminals.  The phrase constructor can be anything, but is usually
;a symbol.  This efficiency decisions made in this implementation assume that
;only a small set of objects are used as phrase constructors.  A common
;phrase constructor is the symbol APPLY or the symbol CONS.

;Both productions and non-terminals are implemented as structures.
;The following symbols define the interface to this system.

;any procedure starting with ``notice-'' is a user-modifiable piecewise defined
;procedure.

;

(defvar *node-count* 0)

(defstruct (non-terminal  (:print-function
			   (lambda (self stream &rest ignore)
			     (declare (ignore ignore))
			     (format stream "[non-terminal ~s]" (node-number self))))
			  (:conc-name nil))
  node-number
  productions-from
  productions-to
  next-find)

(defstruct (production (:print-function
			  (lambda (self stream &rest ignore)
			    (declare (ignore ignore))
			    (format stream "[prod ~s ~s]"
				    (phrase-constructor self)
				    (rhs self))))
			(:conc-name nil))
  phrase-constructor
  lhs
  rhs
  htable-entry)


(definline bogus-non-terminal? (x)
  (next-find x))

(defmacro not-bogus? (&rest nodes)
  `(and ,@(mapcar (lambda (node) `(not (bogus-non-terminal? ,node))) nodes)))


;This file implements aggressive replacement of bogus non-terminals.
;uf-find is rarely applied to bogus nodes.  This makes path compression
;irrelevant.

(hash-table-macro next-find-frame)

(defpiece (ontic-init-phase0 clear-next-find-frames) ()
  (clear-all-next-find-frame))

(defun uf-find (node)
  (let ((n (next-find node)))
    (if n (uf-find n) node)))

(defun uf-find-ret-frame (node find-node)
  (when (not (eq node find-node))
    (make-return-frame (uf-find node) find-node
      (make-justification 'uf-find-computation
	(uf-just-frames node find-node)))))

(defun uf-just-frames (node find-node)
  (unless (eq node find-node)
    (let ((n (next-find node)))
      (when (null n)
	(error "attempt to find required, but non-existent, next-find pointer"))
      (cons (next-find-frame node)
	    (uf-map-ret-frames n find-node)))))

(defun uf-map-ret-frames (nodes find-nodes)
  (when nodes
    (cons-when (uf-find-ret-frame (car nodes) (car find-nodes))
	       (uf-map-ret-frames (cdr nodes) (cdr find-nodes)))))

(defun cons-when (item list)
  (if item (cons item list) list))

(definline uf-equal (n1 n2)
  (eq (uf-find n1) (uf-find n2)))
  


;========================================================================
;creating productions
;========================================================================

;A new expression is constructed by constructing a new production
;with a new left hand side symbol.  This is done with the function
;hashlist.  Hashlist takes a constructor and a right hand side.  It
;either finds an existing production with the given data or creates
;a new production if no such production already exists.  If a
;new production is created, a new left hand non-terminal is
;also created.  Hashlist returns the left hand side non-terminal
;of the production.

;Non-terminals are often instances of structures ``built on'' 
;(that include) the structure non-terminal.  An intern function
;that maps expressions to non-terminal symbols must be provided
;with a mapping between phrase consrtuctors and maker functions
;where the maker function constructs a data object of the correct
;type.  In addition to a maker function, each constructor is associated
;with a function for type checking its arguments.  Numbers are allowed
;as trivial phrase constructors.

;A production represents a fact --- the fact that the right hand side,
;viewed as a constant symbol, equals the left hand side, viewed as
;an expression.  Since it is a fact we must be able to record justifications
;for it.  We can also record a "definition" for each nonterminal symbol.
;The definition is an expression such that the equation between the nonterminal
;and its definition is "true by definition" and need not be justified.


;this function should give non-bogus nodes even when the production has
;been eliminated.

(defun prod-expression (prod)
  (list* (lhs prod)
	 (phrase-constructor prod)
	 (copy-list (rhs prod))))

(defun hashlist (constructor non-terminals &key justification)
  (check-production constructor non-terminals)
  (let ((*quiescent?* nil))
    (multiple-value-prog1
	(if *record-justifications*
	    (recording-hashlist constructor non-terminals justification)
	    (progn
	      (setq non-terminals (mapcar #'uf-find non-terminals))
	      (let ((prod (intern-production constructor non-terminals)))
		(when (null (lhs prod)) (initialize-production prod))
		(uf-find (lhs prod)))))
      (run-queue))))

(defun recording-hashlist (constructor non-terminals justification)
  (let ((new-nonterminals (mapcar #'uf-find non-terminals)))
    (let ((prod (intern-production constructor new-nonterminals)))
      (when (null (lhs prod))
	(initialize-production prod
			       :justification
			       (if (not (equal new-nonterminals non-terminals))
				   (make-justification 'uf-find-replacement
				     (cons (make-invocation-frame
					       (hashlist (cons constructor non-terminals))
					     justification)
					   (uf-map-ret-frames non-terminals new-nonterminals)))
				   justification)))
      (values (uf-find (lhs prod))
	      (let ((exp (prod-expression prod)))
		(append (uf-map-ret-frames non-terminals (cddr exp))
			(list (prod-frame exp))))))))

(defun check-production (constructor non-terminals)
  (let ((checker (checker-function constructor)))
    (unless checker
      (ontic-error (format nil "unknown constructor ~s" constructor)))
    (funcall checker non-terminals)))

;;
;; note:  init-prod is not allowed to do inference until
;;   it has set the LHS of prod, and of any prod coming
;;   in via the keyword &other-prod  --rlg (bug fixed 2/15/92)
;
(defun initialize-production (prod  &key justification other-prod)
  (let ((constructor (phrase-constructor prod)))
    (let ((maker (maker-function constructor)))
      (unless maker
	(ontic-error (format nil "unknown constructor ~s" constructor)))
      (let ((object (funcall maker)))
	(when *record-justifications*
	  (setf-undo (nonterm-definition object)
		     (cons constructor (copy-list (rhs prod)))))
	(setf-undo (lhs prod) object)
	(when other-prod (setf-undo (lhs other-prod) object))
	(initialize-nonterminal object)
	(install-production object prod :justification justification)))))

(defun install-production (nonterm prod &key justification)
  (when *record-justifications* (save-definition nonterm))
  (setf-undo (lhs prod) nonterm)
  (push-undo prod (productions-from nonterm))
  (assert-production prod justification)
  (uf-find (lhs prod)))

(defun assert-production (prod just)
  (record-prod-just prod just)
  (notice-production prod))

(defpiecefun notice-production (production))

(defpiece (notice-production primary-piece) (production)
  (let ((lhs (lhs production))
	(constructor (phrase-constructor production))
	(rhs (rhs production)))
    (if (null rhs)
	(notice-nullary-production lhs constructor)
	(let ((arg1 (first rhs))
	      (rest (rest rhs)))
	  (if (null rest)
	      (notice-unary-production lhs constructor arg1)
	      (notice-binary-production lhs constructor arg1 (first rest)))))))

(defpiecefun notice-nullary-production (lhs constructor))
(defpiecefun notice-unary-production (lhs constructor arg))
(defpiecefun notice-binary-production (lhs constructor arg1 arg2))

(defmergefun merged-notice-binary-production (?lhs ?constructor ?arg1 ?arg2))
(definterpfun interpreted-notice-binary-production (?lhs ?constructor ?arg1 ?arg2))
(defpiece (notice-binary-production :merged-propagator) (lhs constructor arg1 arg2)
  (merged-notice-binary-production lhs constructor arg1 arg2))
(defpiece (notice-binary-production :interpreted-propagator) (lhs constructor arg1 arg2)
  (interpreted-notice-binary-production lhs constructor arg1 arg2))

(defmergefun merged-notice-unary-production (?lhs ?constructor ?arg))
(definterpfun interpreted-notice-unary-production (?lhs ?constructor ?arg))
(defpiece (notice-unary-production :merged-propagator) (lhs constructor arg)
  (merged-notice-unary-production lhs constructor arg))
(defpiece (notice-unary-production :interpreted-propagator) (lhs constructor arg)
  (interpreted-notice-unary-production lhs constructor arg))

(defmergefun merged-notice-nullary-production (?lhs ?constructor))
(definterpfun interpreted-notice-nullary-production (?lhs ?constructor))
(defpiece (notice-nullary-production :merged-propagator) (lhs constructor)
  (merged-notice-nullary-production lhs constructor))
(defpiece (notice-nullary-production :interpreted-propagator) (lhs constructor)
  (interpreted-notice-nullary-production lhs constructor))

(defun save-definition (node)
  (when (null (nonterm-definition node))
    (let ((prod (smallest-lhs-production node)))
      (when prod
	(setf-undo (nonterm-definition node) (cons (phrase-constructor prod)
						   (copy-list (rhs prod))))))))

;the following procedures are only called when *record-justifications* is nil.

(defun add-production (node constructor non-terminals)
  (check-production constructor non-terminals)
  (setq non-terminals (mapcar #'uf-find non-terminals))
  (let ((prod (intern-production constructor non-terminals)))
    (if (lhs prod)
	(progn (equate! node (lhs prod))
	       (uf-find node))
	(install-production node prod))))

(defun equate-productions (c1 n1 c2 n2)
  (check-production c1 n1)
  (setq n1 (mapcar #'uf-find n1))
  (check-production c2 n2)
  (setq n2 (mapcar #'uf-find n2))
  (let ((p1 (intern-production c1 n1))
        (p2 (intern-production c2 n2)))
    (if (lhs p1)
	(if (lhs p2)
	    (progn (equate! (lhs p1) (lhs p2))
		   (uf-find (lhs p1)))
	    (install-production (lhs p1) p2))
	(if (lhs p2)
	    (install-production (lhs p2) p1)
	    (progn
	      (initialize-production p1 :other-prod p2)
	      (install-production (lhs p1) p2))))))

(defvar *creation-breaks* nil)

(defun initialize-nonterminal (object)
  (setf (node-number object) (incf-undo *node-count*))
  (when (member (node-number object) *creation-breaks*)
    (format t "~% Creation break on node number ~s" (node-number object))
    (break))
  (notice-non-terminal object))

(defun set-creation-break (&rest breaks)
  (setf *creation-breaks* (union *creation-breaks* (copy-list breaks))))

(defun clear-creation-breaks (&rest breaks)
  (setf *creation-breaks*
	(when breaks
	  (set-difference *creation-breaks* (copy-list breaks)))))


(defpiecefun notice-non-terminal (non-terminal))

;the following are defaults

;(defun maker-function (ignore)
;  (lambda () (make-non-terminal)))
;
;(defun checker-function (ignore)
;  (lambda (args)
;    (not (> (length args) 2))))
;

;
;Two productions are called CONGRUENT if they have the same phrase constructor
;and the same right hand side (list of argument non-terminals).
;
;The CONGRUENCE INVARIANT states that any two congruent productions
;should have the same left hand side. This is enforced by equating the
;left hand sides of congruence productions.

(defvar *initial-size* (* 1024. 64.))
(defvar *key-array1* (make-array *initial-size*))
(defvar *key-array2* (make-array *initial-size*))
(defvar *key-array3* (make-array *initial-size*))
(defvar *value-array* (make-array *initial-size*))
(defvar *rehash-offset* (* 3 3 3 3))
(defvar *current-size* *initial-size*)
(defvar *entry-count* 0)
(defvar *entry-limit* (/ *initial-size* 2))
(defvar *exceeded-entry-limit* nil)

(defun plug-htable (production)
  (let ((index (htable-entry production)))
    (when (and index (not (eq index 'dead)))
      (setf-undo (aref *value-array* index) t)
      (setf-undo (aref *key-array1* index) -1)
      (setf-undo (htable-entry production) nil))))

;grow-tables can only be called in the base context.

(defpiece (notice-undo-popped grow-tables) ()
  (grow-production-table))

(defun grow-production-table ()
  (when (and (base-frame?)
	     *exceeded-entry-limit*)
    (format t "~%There are now ~s productions"
	    *entry-count*)
    (format t "~%Growing production table")
    (let* ((old-size *current-size*)
	   (old-vals *value-array*)
	   (new-size (* 2 *current-size*)))
      (setf *current-size* new-size)
      (setf *entry-limit* (* 2 *entry-limit*))
      (setf *value-array* (make-array new-size))
      (setf *key-array1* (make-array new-size))
      (setf *key-array2* (make-array new-size))
      (setf *key-array3* (make-array new-size))
      (setf *entry-count* 0)
      (setf *exceeded-entry-limit* nil)
      (dotimes (n old-size)
	(let ((production (aref old-vals n)))
	  (when (and production
		     (not (eq t production)))
	    (congruence-check production)))))))

;intern-production returns either an existing production or a
;new production with the given constructor and rhs but no lhs.

(defun intern-production (operator rhs)
  (let ((number1 (if (numberp operator) operator 0))
	(number2 (if (null rhs) 0 (node-number (first rhs))))
	(number3 (if (null (cdr rhs)) 0 (node-number (second rhs))))
	(first-plug-found nil))
    (iterate loop ((index (mod (+ number1 number2 number3) *current-size*)))
      (let ((val (aref *value-array* index)))
	(cond ((null val)
	       (when first-plug-found
		 (setf index first-plug-found)
		 (decf-undo *entry-count*))
	       (let ((new-node (create-production operator rhs)))
		 (setf-undo (aref *key-array1* index) operator)
		 (setf-undo (aref *key-array2* index) number2)
		 (setf-undo (aref *key-array3* index) number3)
		 (setf-undo (aref *value-array* index) new-node)
		 (setf-undo (htable-entry new-node) index)
		 (let ((new-count (incf-undo *entry-count*)))
		   (when (> new-count *entry-limit*)
		     (setf *exceeded-entry-limit* t)))
		 new-node))
	      ((and (eq (aref *key-array1* index) operator)
		    (= (aref *key-array2* index) number2)
		    (= (aref *key-array3* index) number3))
	       val)
	      (t
	       (when (and (eq t val) (not first-plug-found))
		 (setf first-plug-found index))
	       (unless (< *entry-count* (1- *current-size*))
		 (error "the production table is full"))
	       (loop (mod (+ index *rehash-offset*) *current-size*))))))))

(defpiece (notice-undo-pushing check-table-size) ()
  (when *exceeded-entry-limit*
    (format t "~%Warning the production table is ~s % full"
	    (* 100 (/ (float *entry-count*)
		      (float *current-size*))))))

(defun create-production (operator rhs)
  (let ((prod (make-production :phrase-constructor operator :rhs rhs)))
    (when rhs
      (push-undo prod (productions-to (car rhs)))
      (let ((cell2 (cdr rhs)))
	(when (and cell2 (not (eq (car cell2) (car rhs))))
	  (push-undo prod (productions-to (car cell2))))))
    prod))

  ;Chapter:  Equating Weak Nodes.

(defun equate-constant? (node)
  (catch 'constant
    (equate-constant?-2 node)
    nil))

(defpiecefun equate-constant?-2 (node))

(defun equate! (nterm1 nterm2 &key justification)
  (let ((find1 (uf-find nterm1))
	(find2 (uf-find nterm2)))
    (unless (eq find1 find2)
      (when (and *record-justifications*
		 (or (not (eq find1 nterm1))
		     (not (eq find2 nterm2))))
	(setq justification (make-justification 'uf-find-replacement
			      (append (list (make-invocation-frame (equate! nterm1 nterm2)
					      justification))
				      (cons-when (uf-find-ret-frame nterm1 find1)
						 (cons-when (uf-find-ret-frame nterm2 find2) nil))))))
      (let ((c1 (equate-constant? find1))
	    (c2 (equate-constant? find2)))
	(cond ((and c1 c2)
	       (assert-contradiction :justification
				     (make-justification 'equating-destinct-constants
				       (list (make-invocation-frame (equate! nterm1 nterm2)
					       justification)))))
	      (c1
	       (equate!2 find2 find1 justification))
	      (c2
	       (equate!2 find1 find2 justification))
					;this test is essential for the termination of some algorithms.
	      ((< (node-number find1) (node-number find2))
	       (equate!2 find2 find1 justification))
	      (t
	       (equate!2 find1 find2 justification)))))))
(defun in-htable? (production)
  (numberp (htable-entry production)))

(defvar *delay-q* 20)
(defvar *default-q* 10)

(defun equate!2 (dyer survivor justification)

  (when *record-justifications* (save-definition survivor))

  (let ((equate-frame (when justification
			(make-return-frame (next-find dyer) survivor justification))))

    (setf-undo (next-find dyer) survivor)
    (setf-undo (next-find-frame dyer) equate-frame)

    (dolist (prod (productions-from dyer))
      (let ((just (when *record-justifications*
		    (equality-prod-just (prod-expression prod) equate-frame))))
	(setf-undo (lhs prod) survivor)
	(push-undo prod (productions-from survivor))
	(when *record-justifications*
	  (record-prod-just prod just))))
    (dolist (prod (productions-to dyer))
      (let ((just (when *record-justifications*
		    (equality-prod-just (prod-expression prod) equate-frame))))
	(do-tails (tail (rhs prod))
	  (when (eq (car tail) dyer)
	    (setf-undo (car tail) survivor)))
	(plug-htable prod)
	(push-undo prod (productions-to survivor))
	(when *record-justifications*
	  (record-prod-just prod just))))

    (mapc 'congruence-check (productions-to dyer))

    (queue-with (dyer survivor) *default-q*
      (mapc #'(lambda (prod) (when (in-htable? prod) (notice-production prod)))
	    (productions-from dyer))
     
      (when (formula-p dyer)
	(transfer-truth dyer survivor equate-frame))

      (notice-equate-phase1 dyer survivor)
      (notice-equate-phase2 dyer survivor))))


(defun equality-prod-just (prod-exp equate-frame)
  (make-justification 'production-whacking
		      (list (cons (prod-frame prod-exp) `(assert-production ,prod-exp))
			    equate-frame)))

(defun record-prod-just (prod just)
  (when *record-justifications*
    (let ((exp (prod-expression prod)))
      (when (null (prod-frame exp))
	(setf-undo (prod-frame exp)
		   (make-invocation-frame (assert-production exp)
		     just))))))

(defextendable transfer-truth (dyer survivor equate-frame)
  (let ((truth (is-true-internal dyer)))
    (when truth
      (assert-is-true survivor truth
		      :justification
		      (make-justification 'truth-transfer
			(list (cons (is-true-frame dyer)
				    `(is-true ,dyer))
			      equate-frame))))))

(defpiecefun notice-equate-phase1 (dyer survivor))
(defpiecefun notice-equate-phase2 (dyer survivor))

(defun congruence-check (production)
  (unless (bogus-production? production)
    (let* ((rhs (rhs production))
	   (operator (phrase-constructor production))
	   (number1 (if (numberp operator) operator 0))
	   (number2 (if (null rhs) 0 (node-number (first rhs))))
	   (number3 (if (null (cdr rhs)) 0 (node-number (second rhs))))
	   (first-plug-found nil)	       )
      (unless (< *entry-count* (1- *current-size*))
	(error "the production table is full"))
      (iterate loop ((index (mod (+ number1 number2 number3) *current-size*)))
	(let ((val (aref *value-array* index)))
	  (cond ((null val)
		 (when first-plug-found
		   (setf index first-plug-found)
		   (decf-undo *entry-count*))
		 (setf-undo (aref *key-array1* index) operator)
		 (setf-undo (aref *key-array2* index) number2)
		 (setf-undo (aref *key-array3* index) number3)
		 (setf-undo (aref *value-array* index) production)
		 (setf-undo (htable-entry production) index)
		 (queue-with (production) *default-q*
		   (notice-production production))
		 (let ((new-count (incf-undo *entry-count*)))
		   (when (> new-count *entry-limit*)
		     (format t "~%Warning: hash table getting full")
		     (setf *exceeded-entry-limit* t))))
		((and (eq (aref *key-array1* index) operator)
		      (= (aref *key-array2* index) number2)
		      (= (aref *key-array3* index) number3))
		 (unless (eq production val)
		   (eliminate production)
		   (equate! (lhs production)
			    (lhs val)
			    :justification
			    (when *record-justifications*
			      (congruence-just (prod-expression production)
					       (prod-expression val))))))
		(t
		 (when (and (eq t val) (not first-plug-found))
		   (setf first-plug-found index))
		 (loop (mod (+ index *rehash-offset*) *current-size*)))))))))


(defun congruence-just (pexp1 pexp2)
  (make-justification 'congruence-check
		      (list (prod-frame pexp1) (prod-frame pexp2))))

(defun eliminate (production)
  (setf (htable-entry production) 'dead)
  (let ((rhs (rhs production)))
    (when rhs
      (let ((s1 (first rhs)))
	(delete-undo production (productions-to s1))
	(let ((cell2 (cdr rhs)))
	  (when cell2
	    (let ((s2 (car cell2)))
	      (delete-undo production (productions-to s2))))))))
  (delete-undo production (productions-from (lhs production))))

(defun bogus-production? (prod)
  (eq (htable-entry prod) 'dead))


;
;********************************************************************
;cintern and define
;********************************************************************
;
;The implementation of cintern and define is equivalent to the following.
;
(defvar *definitions* (make-hash-table))

(defmacro definition-of (symbol)
  `(gethash ,symbol *definitions*))

;
;(defun cdefine (symbol expression)
;  (setf-undo (definition-of symbol) expression))
;
;(defun cintern (expression)
;  (cond ((symbolp expression)
;         (let ((def (definition-of symbol)))
;	    (unless def (error "undefined symbol"))
;           (cintern def)))
;        ((listp expression)
;         (hashlist (car expression)
;                   (mapcar #'cintern (cdr expression))))
;	(t
;       expression)))

;The above implementation behaves well with respect to
;re-definition --- if foo is defined in terms of bar then
;redefining bar effectively redefines foo.  Unfortunately,
;uder the above implementation a call to cintern may require time that is
;exponential in the number of previous calls to define.
;The following implementation is functionally equivalent to the
;above --- preserving the desirable way of handling redefinition ---
;but avoids exponential running time.

(defvar *intern-cache* (make-hash-table))

(defmacro intern-cache (symbol)
  `(gethash ,symbol *intern-cache*))

(property-macro intern-frame)

(defvar *dependent-symbols* (make-hash-table))

(defmacro dependent-symbols (symbol)
  `(gethash ,symbol *dependent-symbols*))

(defun cdefine (symbol expression)
  (when (implicit-member? symbol expression)
    (error "recursion in internal expression"))
  (clear-intern-cache symbol)
  (setf-undo (definition-of symbol) expression))



;********************************************************************
;Interning expressions
;********************************************************************

(defvar *quiescent?* t)

(defun cintern (expression &key justification)
  (let ((nonterm nil)
	(ret-frame nil))
    (unwind-protect (let ((*quiescent?* nil))
		      (mvlet (((node ignore iframe) (cintern-internal expression justification)))
			(declare (ignore ignore))
			(setq nonterm node)
			(setq ret-frame iframe)))
      (run-queue))
    (if (not-bogus? nonterm)
	(values nonterm ret-frame)
	(values (uf-find nonterm)
		(when *record-justifications*
		  (make-return-frame (cintern expression) (uf-find nonterm)
		    (make-justification 'queue-changed-uf-find
		      (list ret-frame (uf-find-ret-frame nonterm (uf-find nonterm))))))))))

;cintern internal returns three values --- the node which is the result of the interning,
;a list of symbols whose definition was used in the interning, and a return frame.

(defun cintern-internal (expression justification)
  (cond ((symbolp expression)
	 (intern-symbol expression))
        ((consp expression)
	 (mvlet (((args symbols-used ret-frames) (intern-list (cdr expression) justification)))
	   (let ((hashlist-just
		  (if (null ret-frames)
		      justification
		      (make-justification 'interning
			(cons (make-invocation-frame (intern expression) justification)
			      ret-frames)))))
	     (mvlet (((node hashlist-frames) (hashlist (car expression)
						       args
						       :justification hashlist-just)))
	       (let ((all-frames (append ret-frames hashlist-frames)))
		 (values node
			 symbols-used
			 (when all-frames
			   (make-return-frame (intern expression) node
			     (make-justification 'recursive-interning
			       all-frames)))))))))
	((non-terminal-p expression) ; Modified by KCZ
	 (values expression nil nil))
	((numberp expression)
	 (ontic-error "attempt to cintern a number without parenthesis"))
	((tag-p expression)
	 (cintern-internal (untag expression) justification))
	(t
	 (ontic-error (format nil "~s is not a valid expression for cintern" expression)))))

(defun intern-symbol (symbol)
  (let ((cache-val (intern-cache symbol))
	(cache-ret-frame (intern-frame symbol)))
    (if cache-val
	(mvlet (((val ret-frame) (uf-find cache-val)))
	  (values val
		  (list symbol)
		  (when (and *record-justifications*
			     (or cache-ret-frame ret-frame))
		    (make-return-frame (intern symbol) val
		      (make-justification 'cached-symbol
			(remove-if #'null (list cache-ret-frame ret-frame)))))))
	(let ((def (definition-of symbol)))
	  (unless def (ontic-error (format nil "undefined symbol ~s" symbol)))
	  (mvlet (((computed-val symbols-used ret-frame)
		   (cintern-internal def nil)))
	    (let ((ret-frame2 (when (and *record-justifications* ret-frame)
				(make-return-frame (intern symbol) computed-val
				  (make-justification 'intern-definition
				    ret-frame)))))
	      (setf-undo (intern-cache symbol) computed-val)
	      (setf-undo (intern-frame symbol) ret-frame2)
	      (setf-undo (nonterm-definition computed-val) symbol)
	      (dolist (symbol2 symbols-used)
		(push-undo symbol (dependent-symbols symbol2)))
	      (values computed-val (list symbol) ret-frame2)))))))

(defun intern-list (arg-expressions justification)
  (if (null arg-expressions)
      (values nil nil nil)
      (mvlet (((first-arg symbols-used frame)
	       (cintern-internal (first arg-expressions) justification)))
	(mvlet (((rest-args symbols-used2 rest-frames) (intern-list (cdr arg-expressions)
								    justification)))
	  (values (cons first-arg rest-args)
		  (union symbols-used symbols-used2)
		  (cons-when frame rest-frames))))))


(defun clear-intern-cache (symbol)
  (when (intern-cache symbol)
    (setf-undo (intern-cache symbol) nil)
    (mapc #'clear-intern-cache (dependent-symbols symbol))
    (setf-undo (dependent-symbols symbol) nil)))

(defvar *examined-symbols* nil)

(defun implicit-member? (symbol expression)
  (let ((*examined-symbols* nil))
    (implicit-member2? symbol expression)))

(defun implicit-member2? (symbol expression)
  (unless (and (symbolp expression)
	       (member expression *examined-symbols*))
    (when (symbolp expression)
      (push expression *examined-symbols*))
    (or (eq symbol expression)
	(and (symbolp expression)
	     (definition-of expression)
	     (implicit-member2? symbol (definition-of expression)))
	(and (listp expression)
	     (some (lambda (arg) (implicit-member2? symbol arg))
		   (cdr expression))))))


;
;pieces of notice functions may queue demons.

(defvar *queue* nil)

(defun clear-queue ()
  (clear-queues))

;(defmacro delay-with ((&rest vars) &body body)
;  `(let ,(mapcar (lambda (var) (list var var)) vars)
;     (push (lambda () ,@body)
;	   *queue*)))


(defmacro delay-with ((&rest vars) &body body)
  `(queue-with ,vars *delay-q* (when (not-bogus? ,@vars) ,@body)))

(emacs-indent delay-with 1)

(defvar *contradiction* nil)

(defvar *contradiction-hook* nil)

(defextendable assert-contradiction (&key justification)
  (let ((false-node (ti '(false))))
    (setf-undo (is-true-true-bit-internal false-node) 1)
    (setf-undo (is-true-frame false-node)
	       (lambda () (cons "contradiction"
				justification))))
  (when (null-context?)
    (ontic-error "Contradiction found in base-frame, notify Bertrand Russell"))
  (setf-undo *contradiction* t)
  (when *contradiction-hook*
    (funcall *contradiction-hook*)))

(defun contradiction? ()
  *contradiction*)

(defvar *running-queue* nil)

(defun run-queue ()
  (unless (or (not *quiescent?*)
	      *running-queue*)
    (let ((*running-queue* t))
      (run-queues))
    (when (base-frame?)
      (grow-production-table))
    t))


;
(defun clear-expressions ()
  (setf *node-count* 0)
  (setf *entry-count* 0)
  (setf *contradiction* nil)
  (setf *quiescent?* t)
  (fill-array *key-array1* nil)
  (fill-array *key-array2* nil)
  (fill-array *key-array3* nil)
  (fill-array *value-array* nil)
  (setf *exceeded-entry-limit* nil)
  (clrhash *definitions*)
  (clrhash *intern-cache*)
  (clrhash *dependent-symbols*)
  (clear-queue)
  t)

(defvar *initialized* nil)


;What appears below used to be the file basic-object.lisp


(defvar *infinity* (expt 2 25))

(defstruct (basic-object
	     (:conc-name nil)
	     (:print-function
	       (lambda (self stream &rest ignore)
		 (declare (ignore ignore))
		 (print-basic-object self stream)))
	     (:include non-terminal))
  print-size
  print-names
  (max-constant *infinity*)
  ;;HOG: lhs-demons
  ;;HOG: rhs-demons
  usually-null-properties  ;new-hog
  db-index)

(defmacro usually-null-property (name)
  `(eval-when (compile load eval)
    (defmacro ,name (node)
      `(assoc-value ',',name (usually-null-properties ,node)))))


(usually-null-property lhs-demons)  ;;new-hog
(usually-null-property rhs-demons)  ;;new-hog

;;HOG and new-hog
(defpiece (notice-production fire-demons) (production)
  (dolist (lhs-demon (lhs-demons (lhs production)))
    (funcall lhs-demon production))
  (dolist (nonterm (rhs production))
    (dolist (rhs-demon (rhs-demons nonterm))
      (funcall rhs-demon production))))

;;HOG and new-hog
(defun add-rhs-demon (obj demon)
  (push-undo demon (rhs-demons obj))
  (mapc demon (productions-to obj)))

;;HOG and new-hog
(defun add-lhs-demon (obj demon)
  (push-undo demon (lhs-demons obj))
  (mapc demon (productions-from obj)))

;;; For hog-rule demons, any demon on the dyer will just get recreated
;;; on the survivor, so copying them will result in duplication.

;;;(defpiece (notice-equate-phase2 lhs-and-rhs-demons) (dyer survivor)
;;;  (dolist (rhs-demon (rhs-demons dyer))
;;;    (add-rhs-demon survivor rhs-demon))
;;;  (dolist (lhs-demon (lhs-demons dyer))
;;;    (add-lhs-demon survivor lhs-demon)))

;;new-hog
(usually-null-property identity-demons)

(defun add-identity-demon (obj demon)
  (push-undo demon (identity-demons obj))
  (funcall demon obj))

(defpiece (notice-equate-phase2 identity-demons) (dyer survivor)
  (dolist (identity-demon (identity-demons dyer))
    (add-identity-demon survivor identity-demon)
    (funcall identity-demon survivor)))

(defun print-basic-object (obj stream)
  (let ((*print-level* 15)
	(*print-length* 15))
    (format stream "[~s ~s]" (type-of obj) (macro-invert (object-expression obj)))))

;
;The following is the default for basic objects

;(defun maker-function (ignore)
;  (lambda () (make-basic-object)))

;; Print sizes can be either a single number, or a cons of two
;; numbers, which are compared lexicographically.  A single number
;; x is equivalent to (0 . x).  The sum of two print sizes is
;; the max of the car and the sum of the cdr.

;; These routines also work with a print size of nil, which is
;; larger than any other print size.  The print size of an object
;; is initialized to nil.  Routines which look at the print size of
;; an object should probably treat nil specially, since nothing is
;; known about the "real" (quiescent) print size of an object whose
;; print-size slot is nil.


;; Almost all print-size manipulation goes through the following:
;; *smallest-print-size* print-size-<, print-size-<=, print-size-=, and print-size-+.
;; Thus, it's easy to use a new representation for print sizes.
;; The only exception is the code that actually sets certain objects
;; to have a larger print size than a single integer; currently,
;; this includes basic-define and (constructor-weight 'db-lambda).

(defvar *smallest-print-size* (cons 0 0))

(defun print-size-< (a b)
  (cond ((null a)
	 nil)
	((null b)
	 t)
	(t (mvlet (((car-a cdr-a) (if (numberp a)
				      (values 0 a)
				      (values (car a) (cdr a))))
		   ((car-b cdr-b) (if (numberp b)
				      (values 0 b)
				      (values (car b) (cdr b)))))
	     (or (< car-a car-b)
		 (and (= car-a car-b)
		      (< cdr-a cdr-b)))))))

(defun print-size-<= (a b)
  (cond ((null b)
	 t)
	((null a)
	 nil)
	(t (mvlet (((car-a cdr-a) (if (numberp a)
				      (values 0 a)
				      (values (car a) (cdr a))))
		   ((car-b cdr-b) (if (numberp b)
				      (values 0 b)
				      (values (car b) (cdr b)))))
	     (or (< car-a car-b)
		 (and (= car-a car-b)
		      (<= cdr-a cdr-b)))))))

(defun print-size-= (a b)
  (cond ((null a)
	 (null b))
	((null b)
	 nil)
	(t (mvlet (((car-a cdr-a) (if (numberp a)
				      (values 0 a)
				      (values (car a) (cdr a))))
		   ((car-b cdr-b) (if (numberp b)
				      (values 0 b)
				      (values (car b) (cdr b)))))
	     (and (= car-a car-b) (= cdr-a cdr-b))))))

(defun print-size-+ (a b &rest r)
  (if r
      (apply #'print-size-+ (print-size-+ a b) r)
      (cond ((or (null a) (null b))
	     nil)
	    (t (mvlet (((car-a cdr-a) (if (numberp a)
					  (values 0 a)
					  (values (car a) (cdr a))))
		       ((car-b cdr-b) (if (numberp b)
					  (values 0 b)
					  (values (car b) (cdr b)))))
		 (cons (+ car-a car-b) (+ cdr-a cdr-b)))))))

(defpiece (notice-production compute-print-size) (production)
  (propagate-print-size production))

(defpiecefun notice-production-print-size (production))

(defun propagate-print-size (production)
  (let ((new-size (prod-size production)))
    (decrement-print-size (lhs production) new-size)
    (notice-production-print-size production)))

(defpiecefun notice-print-size (nt))


(defmergefun merged-notice-print-size (?node))

(defpiece (notice-print-size :merged-noticer) (nt)
  (merged-notice-print-size nt))


(defun decrement-print-size (object new-size)
  (let ((old-size (print-size object)))
    (when (and new-size
	       (or (null old-size)
		   (print-size-< new-size old-size)))
      (setf-undo (print-size object) new-size)
      (notice-print-size object)
      (mapc #'propagate-print-size (productions-to object)))))

(defun null-print-sizes (object)
  (when (print-size object)
    (let ((affected-prods (remove-if (lambda (prod)
				       (let ((prods (prod-size prod))
					     (prints (print-size (lhs prod))))
					 (and prods prints (print-size-< prints prods))))
				     (productions-to object))))
      (setf-undo (print-size object) nil)
      (cons object (mapcan #'null-print-sizes (mapcar #'lhs affected-prods))))))

(defun increment-print-size (object new-size)
  (let ((affected-nodes (remove object (null-print-sizes object) :test #'eq)))
    (decrement-print-size object new-size)
    (mapc #'reset-print-size affected-nodes)))

(defun set-print-size (object new-size)
  (let ((old-size (print-size object)))
    (cond ((or (null old-size)
	       (print-size-< new-size old-size))
	   (decrement-print-size object new-size))
	  ((print-size-< old-size new-size)
	   (increment-print-size object new-size)))))

(defun reset-print-size (object)
  (let ((size nil))
    (dolist (prod (productions-from object))
      (let ((psize (prod-size prod)))
	(when (print-size-< psize size)
	  (setf size psize))))
    (set-print-size object size)))

(defpiece (notice-equate-phase1 :transfer-print-size) (dyer survivor)
  (decrement-print-size survivor (print-size dyer)))
	  
(defvar *constructor-weight* (make-hash-table))

(defun constructor-weight (constructor)
  (or (gethash constructor *constructor-weight*)
      1))

(defun prod-size (production)
  (let ((rhs (rhs production))
	(weight (constructor-weight (phrase-constructor production))))
    (if (null rhs)
	weight
	(let ((s1 (print-size (first rhs))))
	  (when s1
	    (if (null (cdr rhs))
		(print-size-+ weight s1)
		(let ((s2 (print-size (second rhs))))
		  (when s2
		    (print-size-+ weight s1 s2)))))))))


(defun small-lhs-productions (weak-node)
  (let ((print-size (print-size weak-node)))
    (when print-size
      (remove-if-not (lambda (prod)
		       (let ((prod-size (prod-size prod)))
			 (and prod-size (print-size-= prod-size print-size))))
		     (productions-from weak-node)))))

(defun smallest-lhs-production (object)
  (find-best (lambda (prod1 prod2)
	       (print-size-< (prod-size prod1) (prod-size prod2)))
	     (productions-from object)))

;
;object-expression

(defun clear-definition (symbol)
  (cdefine symbol nil))

(defun basic-define (symbol expression keylist &key (resize t) module)
  (cdefine symbol expression)
  (let ((node (cintern symbol)))
      (add-print-name symbol node)
      (cond ((and module (not (null-context?)))
	     (set-print-size node (cons 1 1)))
	    (resize (decrement-print-size node 1))))      
  (notice-definition symbol expression keylist)
  (check-definition symbol expression keylist)
  t)

(defun add-print-name (symbol object)
  (when (not (member symbol (print-names object)))
    (push-undo symbol (print-names object))
    (notice-print-name object symbol)))

(defpiecefun notice-print-name (node new-name))

(defpiece (notice-equate-phase1 propagate-print-name) (dyer survivor)
  (dolist (symbol (print-names dyer))
    (add-print-name symbol survivor)))

(defun object-definition (object)
  (macro-invert (object-def2 object)))

(defun object-def2 (object)
  (let ((def (nonterm-definition object)))
    (cond ((and def (symbolp def)) def)
	  ((not (null def))
	   (cons (first def) (mapcar #'object-def2 (cdr def))))
	  (t
	   (object-expression object)))))

(defun object-expression (object)
  (or (let ((prod (find-if (lambda (prod) (eq 'lisp-object (phrase-constructor prod)))
			   (productions-from object))))
	(when prod
	  (cons 'lisp-object
		(mapcar 'object-expression (rhs prod)))))
      (find-if (lambda (symbol)
		 (let ((*quiescent?* nil))
		   (eq object (cintern symbol))))
	       (reverse (remove-if #'ontic-proof-variable?
				   (print-names object))))
      (find-if (lambda (symbol)
		 (let ((*quiescent?* nil))
		   (eq object (cintern symbol))))
	       (reverse (print-names object)))
      (when (print-names object)
	`(obsolete ,(first (print-names object))))
      (let ((prod (smallest-lhs-production object)))
	(if prod
	    (cons (phrase-constructor prod)
		  (mapcar #'object-expression (rhs prod)))
	    `(,(type-of object) ,(node-number object))))))

(defun ontic-proof-variable? (x)
  (and (symbolp x) (string= "!" (subseq (string x) 0 1))))


;
;node numbers

(defvar *node-array-block-size* 10000)

(defvar *current-node-array-size* 10000)

(defvar *nodes* (make-array *node-array-block-size* :initial-element nil :adjustable t))

(defun n (num) (aref *nodes* num))

(defpiece (notice-non-terminal install-in-node-array) (non-terminal)
  (let ((n (node-number non-terminal)))
    (when (>= n *current-node-array-size*)
    (adjust-array *nodes*
		  (incf *current-node-array-size* *node-array-block-size*)))
    (setf (aref *nodes* n) non-terminal)))

(defpiece (ontic-init-phase0 clear-nodes) ()
  (fill-array *nodes* nil))


;; The max-constant of an
;; EXPRESSION is the numerical maximum of all the numbers in that expression.
;; Each constant has a numerical index and the max-constant is used
;; to determine an upper bound on the numerical index of all the constants
;; in an expression.

;; The max-constant of a NON-TERMINAL is the minimum of the max-constant of
;; all the expressions in the equivalence class represented by the non-terminal.

;; The max-constant of a non-terminal is initialized to *infinity*.
;; Max-constant propagation monotonically reduces the max-constant of non-terminals.

;Max-const is used for universal generalization.  Typed constants have numerical
;indeces.  Consider a constant (const n t) of index n and type t.  Further,
;suppose that all assumptions only involve constants with indeces smaller than n.
;Universal generalization inference rules are based on the invariant that
;in any model M of the current assumptions, and any element x of the set
;denoted by t, there exists a model M' in which (const n t) denotes x
;and such that M' agrees with M on all nodes of max-const less than n.

;Without special treatment, the invariant underlying universal generalization
;would be violated by the variables used in holders.
;When a variable v is bound to an node z (when we equate (value v) and z
;and assert that v is bound) then the value of v must "track"
;the value of z.
;Consider generalizing Phi[(const n t)] to forall (x t) Phi[x].
;Consider a model M of the current assumptions and consider
;an element x of the type t.  There should exist a model M'
;in which (const n t) has value x and in which every node of
;index less than x has the same value as in M.
;If z has index greater than x then M and M' may differ
;on z.  But if v is bound to z then M and M' must differ
;on the value of v even though the value of v has an index
;earlier than x.  This is an invariant violation and can
;lead to unsound universal generalization.  For example,
;if an early variable is bound to (const n t) we might generalize
;"(is (const n t) (as-large-as (const n t)))" to
;to "forall (x t) (is x (as-large-as (const n t)))".

;The fix is to assign variable values infinite index.  The index
;of a variable value should be inhereted from its binding.
;If variable values have infinite index then the universal generalization
;invariant does not apply to them.

(defpiece (notice-production propagate-max-constant) (production)
  (propagate-max-const-up-production production))

(defun assert-max-constant (node new-max-const)
  (let ((old-max-const (max-constant node)))
    (when (< new-max-const old-max-const)
      (setf-undo (max-constant node) new-max-const)
      (notice-max-const node new-max-const))))

(defpiecefun notice-max-const (node new-max-const))
(defmergefun merged-notice-max-constant (?node))

(defpiece (notice-max-const call-merged-noticer) (node new-max-const)
  (merged-notice-max-constant node))

(defpiece (notice-max-const propagate-to-productions-to) (node new-max-const)
  (mapc 'propagate-max-const-up-production
	(productions-to node)))

(defun propagate-max-const-up-production (production)
  (assert-max-constant (lhs production)
		       (production-max-const production)))

(defun production-max-const (production)
  (let ((constructor (phrase-constructor production)))
    (cond ((numberp constructor)
	   constructor)
	  ((eq constructor 'value)
	   *infinity*)
	  (t
	   (let ((args (rhs production)))
	     (if (null args)
		 0
		 (let ((cdr-args (cdr args)))
		   (if (null cdr-args)
		       (max-constant (car args))
		       (max (max-constant (car args))
			    (max-constant (car cdr-args)))))))))))




;monitoring

(defvar *monitored-objects* nil)

(defun monitored? (obj)
  (when *monitored-objects*
    (find-if (lambda (obj2) (eq (uf-find obj2)
				(uf-find obj)))
	     *monitored-objects*)))

(defun monitor! (obj)
  (push obj *monitored-objects*))

(defun clear-monitors! ()
  (setf *monitored-objects* nil))

(defmacro monitor-break (obj)
  `(progn
    (format t "~% Breaking for inference on monitored object ~s" ,obj)
    (break)))

(defpiece (notice-equate-phase1 monitor-piece) (dyer survivor)
  (when *monitored-objects*
    (when (monitored? dyer)
      (monitor-break dyer))
    (when (monitored? survivor)
      (monitor-break survivor))))

(defpiecefun notice-monadic-inference (pred object))
(defpiecefun notice-binary-inference (pred obj1 obj2))

(defpiece (notice-monadic-inference monitor) (pred object)
  (when *monitored-objects*
    (when (monitored? object)
      (monitor-break object))))

(defpiece (notice-binary-inference monitor) (pred obj1 obj2)
  (when *monitored-objects*
    (when (monitored? obj1)
      (monitor-break obj1))
    (when (monitored? obj2)
      (monitor-break obj2))))

(defmergefun merged-notice-monadic-inference (?pred ?object))
(defmergefun merged-notice-binary-inference (?pred ?obj1 ?obj2))
  
(defpiece (notice-binary-inference mergefun-call) (pred obj1 obj2)
  (merged-notice-binary-inference pred obj1 obj2))

(defpiece (notice-monadic-inference mergefun-call) (pred object)
  (merged-notice-monadic-inference pred object))

