;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File sharing utilities for NFS                                          ;;;
;;;                                                                         ;;;
;;; M-x share                                                               ;;;
;;; M-x unshare                                                             ;;;
;;; M-x lock                                                                ;;;
;;; M-x grab-lock (not safe, but could be useful)                           ;;;
;;; M-x unlock                                                              ;;;
;;; M-x unlock-all                                                          ;;;
;;; M-x unlock-some                                                         ;;;
;;;                                                                         ;;;
;;; C-x C-q (toggles read only - does locks if share file.)                 ;;;
;;; C-x C-s (save buffer - unlocks if share file)                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'share)

(global-set-key "\C-x\C-q" 'share:lock)
(global-set-key "\C-x\C-s" 'share:save-buffer)

(defvar share:*share-marker* "- NFS Share File -")
(defvar share:*lisp-marker* (cons ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;; "
				  " ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;"))
(defvar share:*c-marker* (cons "/*                             "
			       "                            */"))
(defvar share:*mode-markers*
	'((lisp-mode . share:*lisp-marker*)
	  (emacs-lisp-mode . share:*lisp-marker*)
	  (scheme-mode . share:*lisp-marker*)
	  (ontic-mode . share:*lisp-marker*)
	  (text-mode . share:*lisp-marker*)
	  (c-mode . share:*c-marker*)))
(defvar share:*locked-extension* ".lck")
(defvar share:*orig-kill-buffer* (symbol-function 'kill-buffer))
(setq kill-emacs-hook '(lambda () (unlock-all t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utilities:                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun share:file-uid (file)
  (nth 2 (file-attributes file)))

(defun share:lock-name ()
  (concat (buffer-file-name) share:*locked-extension*))

(defun share:create-file (file)
  (write-region (point-min) (point-min) file nil nil)
  (set-file-modes file 438))

(defun share:rm (file)
  (delete-file file))

(defun share:share-file-p ()
  (save-excursion
    (goto-char 0)
    (let ((eol (save-excursion (end-of-line) (point))))
      (and (buffer-file-name)
	   (search-forward share:*share-marker* eol t)))))

(defun share:locked-p ()
  (and (share:share-file-p)
       (file-exists-p (share:lock-name))))

(defun share:locked-by-user-p ()
  (and (share:share-file-p)
       (not buffer-read-only)))

(defun share:apparently-locked-by-user-p ()
  (and (share:locked-p)
       (= (share:file-uid (share:lock-name))
	  (user-uid))))

(defun delete-line ()
  (let ((point (point)))
    (save-excursion
      (end-of-line 1)
      (delete-region point (point)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Commands:                                                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun share ()
  (interactive)
  (if (share:share-file-p)
      (error "File is already a share file."))
  (if buffer-read-only
      (error "Can't share a read-only file."))
  (if (not (buffer-file-name))
      (error "Can't share a buffer not associated with a file."))
  (if (file-exists-p (share:lock-name))
      (error "Locked file name clash."))
  (save-excursion
    (goto-char 0)
    (let ((type (cdr (assoc major-mode share:*mode-markers*))))
      (if type
	  (insert (car (symbol-value type))
		  share:*share-marker*
		  (cdr (symbol-value type))
		  (format "\n"))
	  (error "Major mode has no NFS Share type.")))
    (save-buffer))
  (share:create-file (share:lock-name))
  (message "File Locked."))

(defun unshare ()
  (interactive)
  (if (not (share:share-file-p))
      (error "File is not a share file."))
  (if (not (share:apparently-locked-by-user-p))
      (error "File is apparently not locked by you."))
  (if (not (share:locked-by-user-p))
      (error "File is not locked by you."))
  (save-excursion
    (goto-char 0)
    (delete-line))
  (save-buffer)
  (share:rm (share:lock-name))
  (message "File no longer a share file."))

(defun lock ()
  (interactive)
  (if (not (share:share-file-p))
      (error "File is not a share file."))
  (if (share:locked-by-user-p)
      (error "File is already locked by you."))
  (if (share:locked-p)
      (error "File is locked by someone else."))
  (if (not (verify-visited-file-modtime (current-buffer)))
      (progn
	(revert-buffer t t)
	(if (not (share:share-file-p))
	    (error "File is no longer a share file."))))
  (share:create-file (share:lock-name))
  (setq buffer-read-only nil)
  (set-buffer-modified-p (buffer-modified-p))
  (message "File Locked."))

(defun grab-lock ()
  (interactive)
  (if (not (share:share-file-p))
      (error "File is not a share file."))
  (if (share:locked-by-user-p)
      (error "File is already locked by you."))
  (if (not (share:locked-p))
      (error "File is not locked."))
  (if (not (eq 'y (read-minibuffer "Forceably obtain lock? (y/n): ")))
      (error "File remains locked by someone else."))
  (if (not (verify-visited-file-modtime (current-buffer)))
      (progn
	(revert-buffer t t)
	(if (not (share:share-file-p))
	    (error "File is no longer a share file."))))
  (share:rm (share:lock-name))
  (share:create-file (share:lock-name))
  (setq buffer-read-only nil)
  (set-buffer-modified-p (buffer-modified-p))
  (message "File Locked."))

(defun unlock (&optional quiet ask)
  (interactive)
  (if (not (share:share-file-p))
      (error "File is not a share file."))
  (if (not (share:locked-by-user-p))
      (error "File is not locked by you."))
  (if (not (share:apparently-locked-by-user-p))
      (error "File is apparently not locked by you."))
  (if (or (not ask)
	  (eq 'y (read-minibuffer
		   (format "Unlock Buffer: %s? (y/n): "
			   (buffer-name (current-buffer))))))
      (progn
	(if (buffer-modified-p)
	    (save-buffer))
	(share:rm (share:lock-name))
	(setq buffer-read-only t)
	(if (not quiet)
	    (message "File Unlocked.")))
      (if (not quiet)
	  (message "File Remains Locked.")))
    (set-buffer-modified-p (buffer-modified-p)))

(defun unlock-all (&optional ask)
  (interactive)
  (let ((buffers (buffer-list))
	(cb (current-buffer)))
    (while buffers
      (set-buffer (car buffers))
      (if (share:locked-by-user-p)
	  (unlock t ask))
      (setq buffers (cdr buffers)))))

(defun unlock-some ()
  (interactive)
  (unlock-all t))

(defun share:lock ()
  (interactive)
  (if (share:share-file-p)
      (if buffer-read-only
	  (lock)
	  (progn
	    (beep)
	    (message "Use C-x C-s to unlock share file.")))
      (if buffer-read-only
	  (toggle-read-only)
	  (if (buffer-modified-p)
	      (progn
		(beep)
		(toggle-read-only)
		(message "Warning - making modified buffer read-only."))
	      (toggle-read-only)))))

(defun share:save-buffer ()
  (interactive)
  (if (share:share-file-p)
      (if (share:locked-by-user-p)
	  (unlock)
	  (if (buffer-modified-p)
	      (error "Can't save: buffer was modified while unlocked.")
	      (message "(No changes need to be saved.)")))
      (save-buffer)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Add a find file hook:                                                   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq find-file-hooks
      (cons 'share:find-file-hook find-file-hooks))

(defun share:find-file-hook ()
  (if (share:share-file-p)
      (setq buffer-read-only t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with kill-buffer.                                                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun kill-buffer (buffer-or-name)
  (interactive "bKill Buffer: ")
  (let ((cb (current-buffer)))
    (set-buffer buffer-or-name)
    (if (and (share:locked-by-user-p)
	     (share:apparently-locked-by-user-p))
	(unlock t t))
    (set-buffer cb))
  (funcall share:*orig-kill-buffer* buffer-or-name))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with revert-buffer                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq revert-buffer-function 'share-revert-buffer)

(defun share-revert-buffer (&optional arg1 arg2)
  (let ((revert-buffer-function nil))
    (if (share:share-file-p)
	(if buffer-read-only
	    (revert-buffer arg1 arg2)
	    (progn
	      (revert-buffer arg1 arg2)
	      (setq buffer-read-only nil)))
	(revert-buffer arg1 arg2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with write-file                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar share:*write-file-fun* (symbol-function 'write-file))

(defun write-file (filename)
  (interactive "FWrite file: ")
  (if (file-exists-p (concat filename share:*locked-extension*))
      (error "Destination file is locked."))
  (let ((locked-p (and (share:locked-by-user-p)
		       (share:apparently-locked-by-user-p))))
    (if locked-p
	(progn
	  (setq buffer-read-only t)
	  (share:rm (share:lock-name))))
    (funcall share:*write-file-fun* filename)
    (if locked-p
	(lock))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with save-some-buffers                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun save-some-buffers (&optional arg exiting)
  "Save some modified file-visiting buffers.  Asks user about each one.
Optional argument (the prefix) non-nil means save all with no questions.
Optional second argument EXITING means ask about certain non-file buffers
 as well as about file buffers."
  (interactive "P")
  (let ((cb (current-buffer)))
    (let (considered (list (buffer-list)))
      (while list
	(let ((buffer (car list)))
	  (set-buffer buffer)
	  (and (or (buffer-modified-p)
		   (share:locked-by-user-p))
	       (save-excursion
		 (and
		   (or buffer-file-name
		       (and exiting buffer-offer-save (> (buffer-size) 0)))
		   (setq considered t)
		   (or arg
		       (y-or-n-p (if buffer-file-name
				     (format "Save file %s? "
					     buffer-file-name)
				     (format "Save buffer %s? "
					     (buffer-name)))))
		   (condition-case ()
		       (if (share:share-file-p)
			   (unlock t)
			   (save-buffer))
		     (error nil))))))
	(setq list (cdr list)))
      (set-buffer cb)
      (and save-abbrevs abbrevs-changed
	   (progn
	     (setq considered t)
	     (if (or arg
		     (y-or-n-p (format "Save abbrevs in %s? "
				       abbrev-file-name)))
		 (write-abbrev-file nil))
	     ;; Don't keep bothering user if he says no.
	     (setq abbrevs-changed nil)))
      (if considered
	  (message "")
	  (message "(No files need saving)")))))
