(provide 'RCS)

;Here's a package I've been working on for some time, and which I have
;come to find tremendously helpful. If you use RCS to help manage your
;source code (especially on multi-person projects), or could possibly
;benefit from doing so, this library makes it much more convenient.
;Read the documentation for more details. I've used this with two
;different versions of RCS, but this will be the first release of the
;package "to the world", so I would not be surprised if there were some
;problems at some sites. The more informative feedback I get, the more
;likely I'll be able to fix incompatibilities.

;I posted this earlier today with the comment that, although it worked
;quite well for me, posting it to the net would no doubt reveal all
;sorts of unexpected surprises as people with different system
;configurations tried it. Well, it was pointed out to me that there was
;an even more fundamental problem that that: My source file contained
;embedded control characters, which were gobbled up by some sites on
;the net. I have corrected that and cancelled my original posting. Here
;is a revised, more networthy (sort of like seaworthy, I guess) version
;of my rcs.el:

;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; rcs.el --- GNU-EMACS interface to RCS revision control system
;; Author          : James Elliott, elliott@cs.wisc.edu
;; Created On      : February, 1990
;; Last Modified By: James Elliott
;; Last Modified On: Wed Oct 24 18:23:06 1990
;; Update Count    : 41
;; Status          : Author uses daily; first release to the world.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;  Written by James J. Elliott
;;  5395 Computer Sciences & Statistics
;;  University of Wisconsin--Madison
;;  1210 W. Dayton St.
;;  Madison, WI 53715

;; Copyright (C) 1990 James J. Elliott
;; See general documentation which follows the licensing information below

;; This file is not officially part of GNU Emacs, but may be
;; donated to the Free Software Foundation.  As such, it is
;; subject to the standard GNU-Emacs General Public License,
;; referred to below.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;; HOW TO USE THIS PACKAGE
;; -----------------------
;; rcs.el provides a convenient, automated, often nearly transparent
;; way to use the Revision Control System (RCS) source-code management
;; utilities. When the library is loaded and the variable rcs-active is
;; non-nil, emacs will notice and respond appropriately to RCS files
;; associated with the files you visit. These RCS files will be found
;; either in the same directory as the files you're working with, or
;; (preferentially, for organization's sake) in an RCS subdirectory of
;; that directory. See the manual pages of rcs(1) for information
;; about how RCS itself works. rcs.el assumes that you are using
;; strict locking in RCS, which is necessary to derive the full
;; benefits of its protection.

;; rcs.el installs three hooks that customize emacs' file-manipulation
;; behavior. One of them is called when a find-file request fails
;; because the file is not found. It checks to see if there is an RCS
;; file corresponding to the file requested, and offers to check that
;; file out for you. It makes sense for this to be the first hook that
;; emacs tries, since it will only prompt you if the RCS file actually
;; exists, and you will usually want to check it out in those
;; instances. If you reply negatively, the remaining file-not-found
;; hooks you may have installed will be tried until one succeeds
;; (these may, for example, offer to create a default header for the
;; file based on a boilerplate, etc). rcs.el will install its own hook
;; at the beginning of this list; you should make sure that any other
;; libraries you load after rcs.el append their own hooks to the END
;; of the list.

;; The second hook is a very simple routine which is called once a
;; file has been loaded. It checks to see whether that file is under
;; RCS' supervision, and whether you currently have a lock on the file
;; (that is, you are the person who checked it out for editing). If
;; so, it puts the buffer into the minor mode rcs-mode, which displays
;; an "RCS" in the mode line. For similar reasons to those outlined
;; above, this should be the first hook in the find-file-hooks list.

;; The third hook rcs.el installs is called when you are about to save
;; a file. It checks whether you are the current locker of the file;
;; if so, it asks what kind of a save you want to perform--the options
;; are J)ust save, which performs a normal emacs style file save, A)dd
;; version, which saves the file and then checks it in as a new
;; version to RCS, but retains your RCS editing lock on the file, and
;; C)heck in, which saves the file, checks it in as a new version, and
;; releases your editing lock. Whenever you add a new version to the
;; RCS file, you will be prompted for a log message describing the
;; changes you have made. Again, this hook should be the first hook in
;; write-file-hooks.

;; When rcs.el is active, and you are editing RCS-controlled files,
;; the read-only status of the file's buffer defaults to being
;; synonymous with the file's locked status within RCS. If you have a
;; file locked for edit, the buffer will be writeable. Otherwise, the
;; buffer will be read-only. (Note that buffers not associated with
;; RCS files continue to behave as they used to.) The key sequence C-x
;; C-q (toggle-read-only) is remapped to call rcs-toggle-read-only,
;; which will ask if you want to check out an RCS-controlled file if
;; its buffer was read-only and you don't presently have a lock on
;; that file. Notice that once you have checked a file out for edit,
;; you can toggle the read-only status of the buffer repeatedly
;; without changing its RCS status, but there's probably no pressing
;; reason why you would want to do this.

;; The automatic behavior outlined above--noticing files are available
;; from RCS and offering to check them out, locking them for edit when
;; you hit the toggle-read-only key, and offering to unlock them when
;; you save them, will address most of the needs of RCS users in
;; day-to-day development. There are additional features available,
;; some of which give you information about the status of files, and
;; some of which let you perform more infrequent operations. These are
;; bound to mnemonic keys, and are described below. (I chose to bind
;; them to C-c sequences, although I know that C-c is really supposed
;; to be a mode-specific prefix, since there just wasn't room in the
;; C-x keymap, and I couldn't think of a better solution. Suggestions
;; are welcome.)

;; C-c i  (rcs-check-in) lets you unlock a file. It prompts you for
;;        the name of the file, which defaults to the current buffer's
;;        file. If given a prefix argument, it will check in a new
;;        version of the file, but retain your lock. Handy when you
;;        want to check in a file which isn't your current buffer's
;;        file, or when its buffer currently has no changes to save
;;        (so you can't use the save-file hook conveniently). If you
;;        have just created a file, and it has not yet been entered
;;        into RCS' control, this is the ONLY way you can do so.
;;        (Apart, of course, from doing it from a shell prompt... :^)

;; C-c o  (rcs-check-out) lets you get the current version of a file
;;        from RCS. Given a prefix argument, it will also lock it for
;;        edit. Necessary if you have an older version of the file
;;        present in your directory, since find-file will just load
;;        that.

;; C-c d  (rcs-diff) creates a buffer showing the differences between
;;        the version of a file you're editing and the last version
;;        that was checked in to RCS. Useful if you don't remember
;;        what you've changed, and want to provide a descriptive log
;;        entry when you check it back in.

;; C-c u  (rcs-unlock) releases the RCS write lock on a given file;
;;        this is different from checking the file in because no new
;;        version of the file is created, and any changes that you
;;        have made will be DISCARDED. If you haven't actually made
;;        any changes, this operation is completed silently. However,
;;        if you >have< made changes, rcs.el will present you with a
;;        buffer showing you the changes that you have made (much as
;;        does rcs-diff) and asking you to confirm that you want to
;;        discard the changes.

;; C-c l  (rcs-log) presents you with a buffer showing the log
;;        information for all versions of a given RCS file.

;; C-c s  (rcs-status) displays a brief message in the mode-line
;;        showing the lock status of a given file. If the file is not
;;        under RCS supervision (i.e., no RCS control file for it
;;        exists), this is stated. Otherwise, If the file is not
;;        locked, "No current locker" is displayed. If the file has
;;        been locked by someone, their username and the version they
;;        have locked are displayed.

;; C-c w  (rcs-who) presents a buffer showing the status of each
;;        locked file in the current directory, showing who the locker
;;        is, and what version is locked. If no files are locked, this
;;        is stated.

;; If any RCS operation initiated by rcs-mode fails (or generates
;; unexpected output), an error is signalled, and the RCS output is
;; displayed in a buffer.


(defvar rcs-active t
  "*Should rcs-related behavior take place when manipulating files that have an
associated RCS subdirectory?")

(defvar rcs-mode nil
  "Is the current buffer associated with a locked RCS file?")

; Tell Emacs about this new kind of minor mode
(if (not (assoc 'rcs-mode minor-mode-alist))
    (progn
      (setq minor-mode-alist (cons '(rcs-mode " RCS") minor-mode-alist))
      (make-variable-buffer-local 'rcs-mode)))


(defun find-rcs-file-not-found-hook ()
  "Called when a find-file command has not been able to find the specified
file in the current directory. Sees if it makes sense to offer to check
the file out from RCS."

  (let (result)
    (if rcs-active
	(if (rcs-status buffer-file-name)
	    (let (edit-it)
	      (setq edit-it (y-or-n-p "Check out for editing? "))
	      (if edit-it
		  (rcs-check-out buffer-file-name t)
		(rcs-check-out buffer-file-name nil))
	      (setq result t) ; Indicate file was found?
	      )
            )
    )
    result
  )
)


; Install the above routine, making it the first hook
(or (memq 'find-rcs-file-not-found-hook find-file-not-found-hooks)
      (setq find-file-not-found-hooks
            (cons 'find-rcs-file-not-found-hook find-file-not-found-hooks))
)


(defun smart-revert ()
  "Reverts the current buffer to version on disk, trying to keep point where
the user expects it to be, if the file size has changed (because the RCS ID
string has changed)."

  (let* ((oldstr (buffer-substring (point)
				   (min (point-max) (+ 100 (point)))))
	 (oldlen (- (point-max) (point-min))))

    (revert-buffer t t)
    (if (not (string-equal oldstr (buffer-substring (point)
						    (min (point-max)
							 (+ 100 (point))))))
	(forward-char (- (- (point-max) (point-min)) oldlen)))
  )
)


(defun rcs-check-out (filename locked)
  "Attempt to check the specified file out using rcs. If a prefix argument
is supplied, will try to lock it for editing."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Check out file: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Check out file: " nil nil nil))
		     current-prefix-arg))

  (message "Working...")
  (setq filename (expand-file-name filename))
  (let ((my-buffer (get-buffer-create "*RCS-output*"))
	(old-buffer (current-buffer))
        )
    (delete-windows-on my-buffer)
    (set-buffer my-buffer)
    (erase-buffer)
    (setq default-directory (file-name-directory filename))
    (if locked
	(call-process "co" nil t nil "-M" "-l" "-q" filename)
      (call-process "co" nil t nil "-M" "-q" filename)
    )

    (if (or (and locked (not (file-writable-p filename)))
	    (not (file-readable-p filename))
	    (> (point-max) 1))
	(progn (pop-to-buffer "*RCS-output*")
	       (error "RCS command failed")))

    ;;Now, update the status of the buffer associated with this file if one
    ;;exists.
    (setq my-buffer (get-file-buffer filename))
    (if my-buffer
	(progn
	  (set-buffer my-buffer)
	  (smart-revert)
	)
    )

    ;;Go back where we were
    (set-buffer old-buffer)
  )
  (message "")
)


(defun rcs-edit-buffer (buffer-name)
  "Attempt to edit the specified buffer which is under rcs control. NOTE: If
the buffer's file exists and is writable, this function will assume that it
has been validly checked out locked. Yet another reason to let only RCS set
the file permissions."

  (interactive "bRCS edit buffer:")

  (let* ((buffer (get-buffer buffer-name))
	 (file (buffer-file-name buffer))
	)

    (set-buffer buffer)
    (if (not (cdr (assoc 'buffer-read-only (buffer-local-variables))))
	(error "Buffer is already writeable"))

    (if (and (file-exists-p file) (file-writable-p file))
	(if (we-locked buffer-file-name)
	    (error "You already have this buffer locked for edit")
	  (error "You have an illegaly writable copy of this file")))

    (rcs-check-out file t)  ; Try to get a locked version. May error.
    
    (if (interactive-p) (switch-to-buffer buffer))
  )
)


(defun we-locked (fname)
  "Returns t if the locker of the specified file matches the current user."

  (let ((ustr (user-login-name))
	(lstr (rcs-status fname)))
    (if (and lstr (> (length lstr) (length ustr)))
	(progn
	  (setq lstr (substring lstr 0 (1+ (length ustr))))
	  (and (string-equal (substring lstr 0 (length ustr)) ustr)
	       (string-equal (substring lstr -1) ":"))
        )
      nil
    )
  )
)

(defun rcs-load-postprocessor ()
  "Peeks at files after they are loaded to see if they merit special
treatment by virtue of being active RCS files."

  (setq rcs-mode (and rcs-active (we-locked buffer-file-name)))
  (if rcs-mode
      (progn
	(make-local-variable 'make-backup-files)
	(setq make-backup-files nil)))
)


; Install above routine
(or (memq 'rcs-load-postprocessor find-file-hooks)
      (setq find-file-hooks
            (cons 'rcs-load-postprocessor find-file-hooks)))


(defun rcs-write-file-hook ()
  "If the current buffer is RCS-active, see if the user wants to check in a
revision upon saving."

  ;; Much of this section is stolen from files.el
  (let (userchoice logmsg result
	(looping t)
	(tempsetmodes (not (file-writable-p buffer-file-name))))
    
    (if rcs-mode
	(let ((echo-keystrokes 0) (win (selected-window))
	      (buf (current-buffer))
	      (fname (file-name-nondirectory buffer-file-name)))
	  (while looping
	    (message "%s: C)heck in, A)dd revision, or J)ust save? " fname)
	    (setq userchoice (downcase (read-char)))
	    (if (or (eq userchoice ?c) (eq userchoice ?a) (eq userchoice ?j))
		(setq looping nil)
	      (beep)))
	  (message "")
	  (set-buffer buf)  ; May well be distinct from win's buffer.
	  (if (eq userchoice ?j)
	      nil ;Let the standard save routine handle it
	    (progn
	      (if file-precious-flag
		  ;; If file is precious, rename it away before
		  ;; overwriting it.
		  (let ((rename t)
			(file (concat buffer-file-name "#")))
		    (condition-case ()
			(progn (rename-file buffer-file-name file t)
			       (setq setmodes (file-modes file)))
		      (file-error (setq rename nil)))
		    (unwind-protect
			(progn (clear-visited-file-modtime)
			       (write-region (point-min) (point-max)
					     buffer-file-name nil t)
			       (setq rename nil))
		      ;; If rename is still t, writing failed.
		      ;; So rename the old file back to original name,
		      (if rename
			  (progn
			    (rename-file file buffer-file-name t)
			    (clear-visited-file-modtime))
			;; Otherwise we don't need the original file,
			;; so flush it.
			(condition-case ()
			    (delete-file file)
			  (error nil)))))
		;; If file not writable, see if we can make it writable
		;; temporarily while we write it.
		;; But no need to do so if we have just backed it up
		;; (setmodes is set) because that says we're superseding.
		(cond ((and tempsetmodes (not setmodes))
		       ;; Change the mode back, after writing.
		       (setq setmodes (file-modes buffer-file-name))
		       (set-file-modes buffer-file-name 511)))
		(write-region (point-min) (point-max) 
			      buffer-file-name nil t))

	      ;; Now we need to try to perform the desired RCS operation.
	      (setq logmsg (read-string "Log message: "))
	      (rcs-check-in buffer-file-name logmsg (eq userchoice ?a))

	      ;; Want to leave file's modes the way rcs set them
	      (setq setmodes nil)

	      (setq result t))) ;Indicate we've saved it.
	  ))
    result ;Return indication of whether it's saved.
  )
)


; Install above routine
(or (memq 'rcs-write-file-hook write-file-hooks)
      (setq write-file-hooks
            (cons 'rcs-write-file-hook write-file-hooks)))


(defun rcs-check-in (filename log locked)
  "Attempt to check the specified file back in using rcs. If a prefix argument
is supplied, will keep it locked for editing."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Check in file: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Check in file: " nil nil nil))
		     (read-string "Log message: " "")
		     current-prefix-arg))

  (setq filename (expand-file-name filename))
  (let* ((my-buffer (get-buffer-create "*RCS-output*"))
	 (f-buffer (get-file-buffer filename))
	 (dirname (file-name-directory default-directory))
	 (old-buffer (current-buffer))
        )
    (if (and (interactive-p) f-buffer (buffer-modified-p f-buffer)
	     (not (y-or-n-p "Check in despite unsaved changes to file? ")))
	(error "Check-in aborted"))
  
    (message "Working...")
    (delete-windows-on my-buffer)
    (set-buffer my-buffer)
    (erase-buffer)
    (setq default-directory (file-name-directory filename))
    (setq log (concat "-m" log))
    (if locked
	(call-process "ci" nil t nil "-M" "-l" log "-t/dev/null" "-q" filename)
      (call-process "ci" nil t nil "-M" "-u" log "-t/dev/null" "-q" filename)
    )

    (if (> (point-max) 1)
	(progn (pop-to-buffer "*RCS-output*")
	       (error "RCS command failed")))

    ;;Now, update the status of the buffer associated with this file if one
    ;;exists.
    (setq my-buffer (get-file-buffer filename))
    (if my-buffer
	(progn
	  (set-buffer my-buffer)
	  (smart-revert)
	)
    )

    ;;Go back where we were
    (set-buffer old-buffer)
  )
  (message "")
)


(defun rcs-unlock (filename)
  "Attempt to remove the rcs lock on the specified file."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Unlock RCS file: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Unlock RCS file: " nil nil nil))))

  (setq filename (expand-file-name filename))

  (if (not (we-locked filename))
      (error "File was not locked to begin with."))

  (let* ((my-buffer (get-buffer-create "*RCS-output*"))
	 (f-buffer (get-file-buffer filename))
	 (old-buffer (current-buffer))
	 (modified nil)
        )
    
    ;;See if they've made any changes to the file. If so, double-check that
    ;;they want to discard them.
    (setq modified (and f-buffer (buffer-modified-p f-buffer)))
    (delete-windows-on my-buffer)
    (set-buffer my-buffer)
    (erase-buffer)
    (setq default-directory (file-name-directory filename))
    (message "Examining file...")
    (call-process "rcsdiff" nil t nil filename)
    (goto-char (point-min))
    (re-search-forward "^diff -r.*$")
    (forward-char 1) ;Skip past command itself
    (delete-region (point-min) (point)) ;Get rid of all but results
    (if modified ;There are changes to the buffer itself too.
	(progn
	  (goto-char (point-max))
	  (insert "\nNote--There are unsaved changes to the buffer itself!\n"))
      (setq modified (< (point) (point-max))))
    (message "")

    (if modified
	(display-buffer "*RCS-output*"))
    (if (and modified (not (y-or-n-p
			    "Discard changes made since file locked? ")))
	(error "Unlock aborted"))

    ;;Okay, do it.
    (erase-buffer)
    (call-process "rcs" nil t nil "-u" "-q" filename)
    (if (> (point-max) 1)
	(progn (pop-to-buffer "*RCS-output*")
	       (error "RCS command failed")))

    ;;Get rid of existing writable copy of the file so it can safely be checked
    ;;out again.
    (if (and (file-exists-p filename) (file-writable-p filename))
	(delete-file filename))

    ;;Go back where we were, in preparation of dropping into rcs-check-out
    (set-buffer old-buffer)
  )

  ;;Finally, check out a current copy.
  (rcs-check-out filename nil)
)


(defun rcs-toggle-read-only ()
  "If the buffer being changed is under rcs, prompt if the user wants
to perform the corresponding rcs action."

  (interactive)

  (if (and rcs-active (not rcs-mode) buffer-read-only buffer-file-name
	   (rcs-status buffer-file-name))
      (if (y-or-n-p "Check buffer out for edit? ")
	  (rcs-edit-buffer (buffer-name))
	(barf-if-buffer-read-only))
    (toggle-read-only))
  (message "")
)


(defun rcs-status (fname)
  "Returns the locker and version of this file if it is locked, an empty
string if it is not locked, and nil if it is not an rcs file."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Query RCS status of: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Query RCS status of" nil nil nil))))

  (setq fname (expand-file-name fname))
  (setq rname (concat (file-name-directory fname) "RCS/"
		      (file-name-nondirectory fname) ",v"))
  (if (not (file-readable-p rname))
      (setq rname (concat fname ",v")))
  (if (file-readable-p rname)
      (let ((oldbuf (current-buffer))
	    (buf (get-buffer-create "*RCS-temp*"))
	    result)
	(set-buffer buf)
	(erase-buffer)
	(insert-file rname)
	(goto-char (point-min))
	;;(re-search-forward "^locks\\s-+\\([^;]*\\)")
	(re-search-forward "^locks;*\\s-+\\([^;]*\\)")
	(setq result (buffer-substring (match-beginning 1) (match-end 1)))
	(erase-buffer)
	(if (interactive-p)
	    (if (> (length result) 0)
		(message (concat "Locker:version--" result))
	      (message "No current locker"))
	  result)
	(set-buffer oldbuf)
	(kill-buffer buf)
	result
      )
    (if (interactive-p)
	(message "Not an RCS-controlled file")
      nil) ; Not an RCS controlled file
  )
)


(defun rcs-log (fname)
  "Show the log of changes for the specified file."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Show RCS log of: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Show RCS log of" nil nil nil))))

  (setq fname (expand-file-name fname))
  (if (not (rcs-status fname))
      (error "File is not under RCS control."))
  
  (let ((my-buffer (get-buffer-create "*RCS-output*"))
	(oldbuf (current-buffer)))

    (set-buffer my-buffer)
    (erase-buffer)
    (setq default-directory (file-name-directory fname))
    (message "Requesting log...")
    (call-process "rlog" nil t nil fname)
    (goto-char (point-min))
    (message "")
    (display-buffer "*RCS-output*")
    (set-buffer oldbuf)
  )
)


(defun rcs-diff (fname)
  "Show the differences between current file and the last revision checked in."

  (interactive (list (if buffer-file-name
			 (read-file-name
			  (format "Show RCS diffs for: (default %s) "
				  (file-name-nondirectory buffer-file-name))
			  nil (file-name-nondirectory buffer-file-name) nil)
		       (read-file-name "Show RCS diffs for" nil nil nil))))

  (if (not (rcs-status fname))
      (error "File is not under RCS control."))

  (setq fname (expand-file-name fname))
  (let ((my-buffer (get-buffer-create "*RCS-output*"))
	(f-buffer (get-file-buffer fname))
	(oldbuf (current-buffer))
	oldpos)

    (if (and (interactive-p) f-buffer (buffer-modified-p f-buffer)
	     (y-or-n-p "Save file before comparing? "))
	(save-buffer))

    (set-buffer my-buffer)
    (erase-buffer)
    (setq default-directory (file-name-directory fname))
    (insert (concat "rcsdiff for " fname ":\n\n"))
    (setq oldpos (point))
    (message "Comparing files...")
    (call-process "rcsdiff" nil t nil fname)
    (goto-char (point-min))
    (re-search-forward "^diff -r.*$")
    (forward-char 1) ;Skip past command itself
    (delete-region oldpos (point)) ;Get rid of all but results
    (if (= (point) (point-max))
       (insert "\nNo differences between this file and version checked in.\n"))
    (goto-char (point-min))
    (message "")
    (display-buffer "*RCS-output*")
    (set-buffer oldbuf)
  )
)


(defun rcs-who ()
  "Show the locker of each of the locked files under RCS control in the
current directory."

  (interactive "")

  (message "Working...")
  (let* ((my-buffer (get-buffer-create "*RCS-output*"))
	 (dirname (file-name-directory default-directory))
	 (oldbuf (current-buffer))
	 fname user version oldpos)
    (set-buffer my-buffer)
    (erase-buffer)
    (setq fname (concat "Status of RCS files in " dirname ":\n"))
    (insert fname)
    (insert (concat (make-string (- (length fname) 1) ?-)) "\n\n")
    (setq oldpos (point))
    (setq default-directory dirname)
    (if (file-directory-p (concat dirname "RCS"))
	(setq dirname (concat dirname "RCS/*,v " dirname))
      (setq dirname (concat dirname "*,v")))
    (call-process "sh" nil t nil "-c"
		  (concat "egrep '^locks[ \\t]+[a-z]+[^;]+' " dirname))
    (if (= (point) oldpos)
	(insert "No files are locked.\n")
      (goto-char oldpos)
      (if (looking-at "egrep: can't open")
	  (progn
	    (delete-region oldpos (point-max))
	    (insert "No files under RCS control.\n"))
	(while (re-search-forward
		"^.*/\\([^,]*\\),v:locks\\s-+\\([^:]*\\):\\([^;]*\\).*$"
		nil t nil)
	  (setq fname (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq user (buffer-substring (match-beginning 2) (match-end 2)))
	  (setq version (buffer-substring (match-beginning 3) (match-end 3)))
	  (delete-region (match-beginning 0) (match-end 0))
	  (goto-char (match-beginning 0))
	  (insert (concat "File: " fname))
	  (insert (make-string (- 17 (length fname)) ? ))
	  (insert (concat "Locker: " user))
	  (insert (make-string (- 10 (length user)) ? ))
	  (insert (concat "Version locked: " version))
	)
      )
      (goto-char (point-max))
    )
    (display-buffer "*RCS-output*")
    (set-buffer oldbuf)
  )
  (message "")
)


; Local maps can override this if they want, since C-c is usually a local
; prefix.
(global-set-key "\C-crd" 'rcs-diff);
(global-set-key "\C-cri" 'rcs-check-in)
(global-set-key "\C-crl" 'rcs-log)
(global-set-key "\C-cro" 'rcs-check-out)
(global-set-key "\C-crs" 'rcs-status)
(global-set-key "\C-cru" 'rcs-unlock)
(global-set-key "\C-crw" 'rcs-who)
(global-set-key "\C-x\C-q" 'rcs-toggle-read-only)
