;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; - NFS Share File - ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ontic Mode:                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'ontic-mode)

(defvar *ontic-prefix* "\C-z")
(defvar *quiet-p* nil)
(defvar *report-inferences* nil)
(defvar *split-windows* t)
(defvar *visible-source-buffer*	nil)
(defvar *visible-source-window* nil)
(defvar *eval-region-next-defun* nil)
(defvar *eval-region-last-defun* nil)
(defvar *translation-alist*
	'(("defstruct" . "def-o-struct")
	  ("define" . "defontic")))
(defvar *ontic-forms*
	'(defstruct define defmodule axiom
	  def-o-struct defontic defproof assume))

;; Make .ont and .ontic files come up in ontic-mode.
(setq auto-mode-alist (append auto-mode-alist
			      '(("\\.ontic$" . ontic-mode)
				("\\.ont$" . ontic-mode))))

(defun setup-ontic-mode-map (map prefix)
  (define-key map (concat prefix "e") 'eval-defun-ontic)
  (define-key map (concat prefix "r") 'eval-region-ontic)
  (define-key map (concat prefix "b") 'eval-buffer-ontic)
  (define-key map (concat prefix "g") 'abort-commands-lisp)
  (define-key map (concat prefix "I") 'ontic-init)
  (define-key map (concat prefix "i") 'ontic-init-with-resets)
  (define-key map (concat prefix "v") 'ontic-version)
  (define-key map (concat prefix "l") 'ontic-switch-to-lisp)
  (define-key map (concat prefix "k") 'ontic-kill)
  (define-key map (concat prefix "s") 'ontic-start)
  (define-key map (concat prefix "m") 'ontic-toggle-mode-line)
  (define-key map (concat prefix "f") 'ontic-faith-load)
  (define-key map (concat prefix " ") 'ontic-push-pop-reset-state)
  (define-key map (concat prefix "q") 'ontic-toggle-quieting)
  (define-key map (concat prefix "y") 'ontic-insert-failure)
  (define-key map (concat prefix "S") 'ontic-save-library)
  (define-key map (concat prefix "L") 'ontic-load-library)
  map)

(defvar ontic-mode-map
  (progn
    (let ((map (copy-keymap lisp-mode-map)))
      (setup-ontic-mode-map map *ontic-prefix*)
      (setup-ontic-mode-map map (concat *ontic-prefix* "o")))))

(defun ontic-mode ()
  "Major mode for Ontic proofs."
  (interactive)
  ;; Go into lisp mode for a second
  (lisp-mode)
  ;; Get rid of local variables
  (kill-all-local-variables)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'lisp-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start ";")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip ";+ *")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'lisp-comment-indent)
  ;; Install the keyboard map and other standard stuff.
  (use-local-map ontic-mode-map)
  (setq mode-name "Ontic")
  (setq major-mode 'ontic-mode)
  (setq local-abbrev-table lisp-mode-abbrev-table)
  (set-syntax-table lisp-mode-syntax-table)
  (setq mode-line-process 'ilisp-status)
  ;; Run any hooks the user has installed.
  (run-hooks 'ontic-mode-hook))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Interface Support:                                                      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun ontic-running-p ()
  (and (fboundp 'ilisp-process)
       ilisp-buffer
       (get-buffer ilisp-buffer)
       (ilisp-process)
       (string= *ontic-dialect*	(process-name (ilisp-process)))))

(defun ontic-busy-p ()
  (and (ontic-running-p)
       (not (string= ilisp-status " :ready"))))

(defmacro ensure-source-window (&rest body)
  (` (let ((cw (selected-window)))
       (if *visible-source-buffer*
	   (set-buffer *visible-source-buffer*))
       (unwind-protect
	   (progn
	     (if (and *visible-source-window*
		      (window-point *visible-source-window*)
		      (eq (window-buffer *visible-source-window*)
			  *visible-source-buffer*))
		 (select-window *visible-source-window*))
	     (,@ body))
	 (select-window cw)))))

(defun add-cleanup-deamon (exp deamon)
  (format "(lisp:unwind-protect %s (util:emacs-eval '%s))"
	  exp
	  deamon))

(defun add-termination-deamon (exp deamon)
  (format "(lisp:progn %s (util:emacs-eval '%s))"
	  exp
	  deamon))

(defun add-emacs-catch (exp)
  (format
    "(lisp:catch 'util:emacs (lisp:let ((util:*visible-evaluation?* t)) %s))"
    exp))

(defun add-faith (faithp exp)
  (if faithp
      (format "(with-faith %s)" exp)
      exp))

(defun add-region-quieting (exp)
  (format "(with-region %s)" exp))

(defun add-quieting (exp)
  (if *quiet-p*
      (format "(without-beeping %s)" exp)
      exp))

(defun add-proof (exp)
  (format "(proof %s)" exp))

;; Replace s1 with s2 in str.
(defun string-replace (s1 s2 str)
  (let* ((qs1 (regexp-quote s1))
	 (current 0)
	 (ls1 (length s1))
	 (ret "")
	 (match (string-match qs1 str)))
    (while match
      (setq ret (concat ret (substring str current match) s2))
      (setq current (+ match ls1))
      (setq match (string-match qs1 str current)))
    (setq ret (concat ret (substring str current)))
    ret))

(defun next-defun ()
  (forward-line 1)
  (beginning-of-line)
  (if (re-search-forward "^(" nil t nil)
      (goto-char (- (point) 1))
      (goto-char (point-max))))

(defun in-defun-p ()
  (save-excursion
    (let ((point (point)))
      (and (progn (end-of-line) (beginning-of-defun))
	   (progn (forward-sexp 1)
		  (end-of-line) t)
	   (>= (point) point)))))

(defun form-in ()
  (if (not (in-defun-p))
      nil
      (save-excursion
	(end-of-line)
	(beginning-of-defun)
	(forward-char 1)
	(let ((end (progn (forward-sexp 1) (point))))
	  (backward-sexp 1)
	  (car (read-from-string (buffer-substring (point) end)))))))

(defun form-type ()
  (let ((fi (form-in)))
    (if (null fi)
	nil
	(if (not (memq fi *ontic-forms*))
	    'proof
	    fi))))

(defun apply-translations (exp)
  (let ((translation *translation-alist*)
	(expr (format "%s" exp)))
    (while translation
      (setq expr (string-replace (car (car translation))
				 (cdr (car translation)) expr))
      (setq translation (cdr translation)))
    expr))

(defun eval-region-continue (&optional with-faith kill-buffer)
  (if (<= *eval-region-next-defun*
	  *eval-region-last-defun*)
      (ensure-source-window
	(goto-char *eval-region-next-defun*)
	(push-mark)
	(forward-sexp)
	(let ((defun-string (apply-translations
			      (buffer-substring (mark) (point))))
	      (this-defun *eval-region-next-defun*))
	  (if (eq (form-type) 'proof)
	      (setq defun-string (add-proof defun-string)))
	  (next-defun)
	  (setq *eval-region-next-defun* (point))
	  (goto-char this-defun)
	  (ilisp-eval-string
	    (add-emacs-catch
	      (add-termination-deamon
		(add-faith with-faith
		  (add-region-quieting
		    (add-quieting defun-string)))
		(list 'eval-region-continue with-faith kill-buffer)))))
	(pop-mark))
      (progn
	(if kill-buffer
	    (kill-buffer *visible-source-buffer*))
	(if with-faith
	    (message "Region Believed.")
	    (message "Region Correct."))
	(if (not *quiet-p*)
	    (progn
	      (beep)
	      (beep))))))

(defun faith-load-region-error ()
  (set-window-buffer (selected-window) *visible-source-buffer*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Interface Commands:                                                     ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun eval-defun-ontic ()
  (interactive)
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (let ((form (form-type)))
	    (if (not form)
		(error "Not inside a proper form."))
	    (setq *visible-source-buffer* (current-buffer))
	    (setq *visible-source-window* (selected-window))
	    (end-of-defun)
	    (set-mark (point))
	    (beginning-of-defun)
	    (let ((defun-string (apply-translations
				  (buffer-substring (point) (mark)))))
	      (if (eq form 'proof)
		  (setq defun-string (add-proof defun-string)))
	      (ilisp-eval-string
		(add-emacs-catch
		  (add-faith (and current-prefix-arg t)
		    (add-quieting defun-string)))))))
      (ontic)))

(defun eval-region-ontic (start end)
  (interactive "r\n")
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (progn
	    (setq *visible-source-buffer* (current-buffer))
	    (setq *visible-source-window* (selected-window))
	    (goto-char end)
	    (end-of-line)
	    (beginning-of-defun)
	    (setq *eval-region-last-defun* (point))
	    (goto-char start)
	    (if (in-defun-p)
		(progn
		  (end-of-line)
		  (beginning-of-defun))
		(next-defun))
	    (setq *eval-region-next-defun* (point))
	    (eval-region-continue (and current-prefix-arg t))))
      (error "Ontic isn't running.")))

(defun eval-buffer-ontic ()
  (interactive)
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (progn
	    (setq *visible-source-buffer* (current-buffer))
	    (setq *visible-source-window* (selected-window))
	    (goto-char (point-max))
	    (end-of-line)
	    (beginning-of-defun)
	    (setq *eval-region-last-defun* (point))
	    (goto-char (point-min))
	    (if (in-defun-p)
		(progn
		  (end-of-line)
		  (beginning-of-defun))
		(next-defun))
	    (setq *eval-region-next-defun* (point))
	    (eval-region-continue (and current-prefix-arg t))))
      (error "Ontic isn't running.")))

(defun ontic-init ()
  (interactive)
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (let ((ans (read-minibuffer "Initialize Ontic Completely (y/n): ")))
	    (if (eq ans 'y)
		(ilisp-eval-string
		  "(lisp:progn (ontic-init)
                     (util:emacs-eval
                      '(message
                          \"\\\"Ontic reset to initial state\\\"\")))"))))
      (ontic)))

(defun ontic-version ()
  (interactive)
  (if (ontic-running-p)
      (message "Ontic Version %s." *ontic-release*)
      (error "Ontic is not running.")))

(defun ontic-switch-to-lisp ()
  (interactive)
  (if (ontic-running-p)
      (let ((pop-up-windows *split-windows*))
	(switch-to-lisp t))
      (error "Ontic isn't running.")))

(defun ontic-kill ()
  (interactive)
  (if (ontic-running-p)
      (let ((kill (read-minibuffer "Kill Ontic (y/n): ")))
	(if (eq kill 'y)
	    (progn
	      (kill-buffer (ilisp-buffer))
	      (setq ilisp-status " :killed"))))
      (error "Ontic isn't running.")))

(defun ontic-start ()
  (interactive)
  (if (ontic-running-p)
      (error "Ontic is already running.")
      (ontic)))

(defun ontic-toggle-mode-line ()
  (interactive)
  (if (eq mode-line-process 'ilisp-status)
      (setq mode-line-process nil)
      (setq mode-line-process 'ilisp-status))
  (set-buffer-modified-p (buffer-modified-p)))

(defun ontic-faith-load (file)
  (interactive "FFaith Load File: ")
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (let ((buffer (generate-new-buffer "*faith*"))
		(cb (current-buffer)))
	    (setq *visible-source-buffer* buffer)
	    (setq *visible-source-window* nil)
	    (set-buffer buffer)
	    (ontic-mode)
	    (insert-file file)
	    (goto-char (point-max))
	    (beginning-of-defun)
	    (setq *eval-region-last-defun* (point))
	    (goto-char (point-min))
	    (if (in-defun-p)
		(progn
		  (end-of-line)
		  (beginning-of-defun))
		(next-defun))
	    (setq *eval-region-next-defun* (point))
	    (eval-region-continue t t)
	    (setq buffer cb)))
      (error "Ontic isn't running.")))

(defun ontic-push-pop-reset-state ()
  (interactive)
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (if current-prefix-arg
	      (ilisp-eval-string
		"(if (= *resets-availible* 0)
                     (util:emacs-eval
                       '(message \"\\\"No more saved states available.\\\"\"))
                     (util:emacs-eval
                       `(message
                          ,(format nil \"\\\"Popped to state ~a\\\"\" 
                             (pop-reset-state)))))")
	      (ilisp-eval-string 
		"(util:emacs-eval
                   `(message
                      ,(format nil \"\\\"Pushed to state ~a\\\"\" 
                                   (push-reset-state))))")))
      (error "Ontic isn't running.")))

(defun ontic-init-with-resets ()
  (interactive)
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (let ((ans (read-minibuffer "Initialize Ontic (y/n): ")))
	    (if (eq ans 'y)
		(ilisp-eval-string
		  "(lisp:progn (reset)
                     (util:emacs-eval
                       '(message
                          \"\\\"Ontic reset\\\"\")))"))))
      (ontic)))

(defun ontic-toggle-quieting ()
  (interactive)
  (if *quiet-p*
      (progn
	(setq *quiet-p* nil)
	(message "Ontic is no longer in quiet mode."))
      (progn
	(setq *quiet-p* t)
	(message "Ontic is now in quiet mode."))))

(defun ontic-save-library (file)
  (interactive "FSave State in File: ")
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (ilisp-eval-string (format "(save-library \"%s\")" file)))
      (error "Ontic isn't running.")))

(defun ontic-load-library (file)
  (interactive "FLoad State From File: ")
  (if (ontic-running-p)
      (if (ontic-busy-p)
	  (error "Ontic is busy.")
	  (ilisp-eval-string (format "(load-library \"%s\")" file)))
      (error "Ontic isn't running.")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Failure expansion code:                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *ontic-failures* nil)
(defvar *ontic-next-failure* nil)

(defun ontic-get-next-failure ()
  (if (null *ontic-next-failure*)
      (setq *ontic-next-failure* *ontic-failures*))
  (let ((next (car *ontic-next-failure*)))
    (setq *ontic-next-failure* (cdr *ontic-next-failure*))
    next))

(defun ontic-insert-failure ()
  (interactive)
  (let ((this-command 'ontic-insert-failure)
	(failure (ontic-get-next-failure)))
    (if (not failure)
	(error "No failures to expand."))
    (if (eq last-command 'ontic-insert-failure)
	(progn
	  (kill-sexp 1)
	  (save-excursion
	    (insert failure))
	  (indent-sexp))
	(progn
	  (save-excursion
	    (insert failure))
	  (indent-sexp)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Old High-level proof code:                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq lisp-mode-hook
      '(lambda nil (interactive)
	(put 'show-by-induction-on 'lisp-indent-hook 2)
	(put 'interpreted-rule 'lisp-indent-hook 2)
	(put 'rule 'lisp-indent-hook 2)
	(put 'classify 'lisp-indent-hook 1)
	(put 'sequent 'lisp-indent-hook 1)
	(put 'when-equal 'lisp-indent-hook 2)
	(put 'induction-on 'lisp-indent-hook 1)
	(put 'selectmatch 'lisp-indent-hook 1)
	(put 'forall 'lisp-indent-hook 1)
	(put 'let-be 'lisp-indent-hook 1)
	(put 'let-be-focus 'lisp-indent-hook 1)
	(put 'exists 'lisp-indent-hook 1)
	(put 'some-such-that 'lisp-indent-hook 2)
	(put 'the-set-of-all 'lisp-indent-hook 3)
	(put 'show 'lisp-indent-hook 1)
	(put 'show-internal 'lisp-indent-hook 1)
	(put 'do 'lisp-indent-hook 2)
	(put 'do* 'lisp-indent-hook 2)
	(put 'dolist 'lisp-indent-hook 1)
	(put 'dotimes 'lisp-indent-hook 1)
	(put 'multiple-value-setq 'lisp-indent-hook 1)
	(put 'defframe 'lisp-indent-hook 1)
	(put 'defstruct 'lisp-indent-hook 1)
	(put 'defun 'lisp-indent-hook 2)
	(put 'defun-nondeterministic 'lisp-indent-hook 2)
	(put 'defmacro 'lisp-indent-hook 2)
	(put 'defmac 'lisp-indent-hook 1)
	(put 'lambda 'lisp-indent-hook 1)
	(put 'lambda-nondeterministic 'lisp-indent-hook 1)
	(put 'let 'lisp-indent-hook 1)
	(put 'let* 'lisp-indent-hook 1)
	(put 'catch 'lisp-indent-hook 1)
	(put 'throw 'lisp-indent-hook 1)
	(put 'while 'lisp-indent-hook 1)
	(put 'unless 'lisp-indent-hook 1)
	(put 'when 'lisp-indent-hook 1)
	(put 'case 'lisp-indent-hook 1)
	(put 'return-from 'lisp-indent-hook 1)
	(put 'suppose 'lisp-indent-hook 1)
	(put 'lambda-rel 'lisp-indent-hook 1)
	(put 'lambda-fun 'lisp-indent-hook 1)
	(put 'some-such-that 'lisp-indent-hook 2)
	(put 'let-be 'lisp-indent-hook 1)
	(put 'forall 'lisp-indent-hook 1)
	(put 'lisp-let 'lisp-indent-hook 1)
	(put 'consider 'lisp-indent-hook 1)
	(put 'suppose-there-is 'lisp-indent-hook 1)
	(put 'suppose-there-is-focus 'lisp-indent-hook 1)
	(put 'suppose-for-refutation 'lisp-indent-hook 1)
	(auto-fill-mode 0)		;Turn on auto-fill
	))


(defvar hlps-*proof-position*)

(defun hlps-start-proof ()
  (if *visible-source-buffer*
      (set-buffer *visible-source-buffer*))
  (setq hlps-*proof-position* (point)))

(defun hlps-end-proof ()
  nil)

(defun hlps-forward-sexp (number)
  (unwind-protect
      (progn
	(setq parse-sexp-ignore-comments t)
	(forward-sexp number))
    (setq parse-sexp-ignore-comments nil)))

(defun hlps-goto-location (location)
  (ensure-source-window
    (goto-char hlps-*proof-position*)
    (if (not (in-defun-p))
	(error "Failure to update proof location - cursor not in valid form."))
    (let ((next location))
      (end-of-line)
      (beginning-of-defun)
      (while next
	;; Increase depth by one
	(forward-char 1)
	(hlps-forward-sexp 1)
	(backward-sexp 1)
	;; Go to the correct width
	(if (> (car next) 0)
	    (progn
	      (hlps-forward-sexp (1+ (car next)))
	      (backward-sexp 1)))
	(setq next (cdr next))))))

(defun hlps-goto-right-paren (location)
  (hlps-goto-location location)
  (backward-up-list 1)
  (forward-sexp 1))

(defvar *hlps-status-stack* nil)

(defun hlps-push-status ()
  (setq *hlps-status-stack* (cons ilisp-status *hlps-status-stack*)))

(defun hlps-pop-status ()
  (setq ilisp-status (car *hlps-status-stack*))
  (setq *hlps-status-stack* (cdr *hlps-status-stack*))
  (if comint-show-status
      (progn
	(save-excursion (set-buffer (other-buffer)))
	(sit-for 0))))


(defun hlps-lisp-gc-start ()
  (hlps-push-status)
  (ilisp-update-status 'GC))

(defun hlps-lisp-gc-end ()
  (hlps-pop-status))

(defun hlps-set-runbar ()
  (hlps-push-status)
  (ilisp-update-status 'think))

(defun hlps-clear-runbar ()
  (hlps-pop-status))


(defun hlps-lisp-compile-start ()
  (hlps-push-status)
  (ilisp-update-status 'compile))

(defun hlps-lisp-compile-end ()
  (hlps-pop-status))

(defun hlps-lisp-compile-final-start ()
  (hlps-push-status)
  (ilisp-update-status 'compile-final))
