;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File sharing utilities for LISP.                                        ;;;
;;;                                                                         ;;;
;;; M-x share                                                               ;;;
;;; M-x find-file (also bound to C-x C-f) is destructively modified!        ;;;
;;; M-x lock                                                                ;;;
;;; M-x grab-lock (not safe, but could be useful)                           ;;;
;;; M-x unlock                                                              ;;;
;;; M-x unlock-all                                                          ;;;
;;; M-x unlock-some                                                         ;;;
;;;                                                                         ;;;
;;; C-x C-q (toggles read only - does locks if share file.)                 ;;;
;;; C-x C-s (save buffer - unlocks if share file)                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'lshare)

(global-set-key "\C-x\C-q" 'share-lock)
(global-set-key "\C-x\C-s" 'share-save-buffer)

(defvar lshare:*share-marker* (concat ";" "lshare:"))
(defvar lshare:*orig-kill-buffer* (symbol-function 'kill-buffer))
(setq kill-emacs-hook '(lambda () (unlock-all t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utilities:                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun lshare:share-file-p ()
  (save-excursion
    (goto-char 0)
    (search-forward lshare:*share-marker* nil t)))

(defun lshare:locked-p ()
  (save-excursion
    (goto-char 0)
    (search-forward lshare:*share-marker*)
    (string= (libbuf:get-characters 6) "locked")))

(defun lshare:locked-by-user-p ()
  (and (lshare:share-file-p)
       (not buffer-read-only)))

(defun lshare:apparently-locked-by-user-p ()
  (save-excursion
    (goto-char 0)
    (search-forward lshare:*share-marker*)
    (if (string= (libbuf:get-characters 6) "locked")
	(let ((user (user-full-name)))
	  (forward-char 7)
	  (if (string= (libbuf:get-characters (length user)) user)
	      t)))))

(defun delete-line ()
  (let ((point (point)))
    (save-excursion
      (end-of-line 1)
      (delete-region point (point)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Commands:                                                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun share ()
  (interactive)
  (if (lshare:share-file-p)
      (error "File is already a share file."))
  (if buffer-read-only
      (error "Can't share a read-only file."))
  (save-excursion
    (goto-char 0)
    (insert lshare:*share-marker* "locked:" (user-full-name) ":"
	    (current-time-string) (format "\n"))
    (save-buffer))
  (message "File Locked."))

(defun share-find-file (file)
  (interactive "FFind File:")
  (find-file file))

(defun lock ()
  (interactive)
  (if (not (lshare:share-file-p))
      (error "File is not a share file."))
  (if (lshare:locked-by-user-p)
      (error "File is already locked by you."))
  (if (buffer-modified-p)
      (error "Can't lock a modified buffer."))
  (let ((pos (point)))
    (revert-buffer t t)
    (goto-char (min pos (point-max))))
  (setq buffer-read-only t)
  (if (not (lshare:share-file-p))
      (error "File is no longer a share file."))
  (if (lshare:locked-p)
      (error "File is locked by someone else."))
  (setq buffer-read-only nil)
  (save-excursion
    (goto-char 0)
    (search-forward lshare:*share-marker*)
    (delete-line)
    (insert "locked:" (user-full-name) ":" (current-time-string))
    (save-buffer))
  (message "File Locked."))

(defun grab-lock ()
  (interactive)
  (if (not (lshare:share-file-p))
      (error "File is not a share file."))
  (if (lshare:locked-by-user-p)
      (error "File is already locked by you."))
  (if (buffer-modified-p)
      (error "Can't lock a modified buffer."))
  (let ((pos (point)))
    (revert-buffer t t)
    (goto-char (min pos (point-max))))
  (setq buffer-read-only t)
  (if (not (lshare:share-file-p))
      (error "File is no longer a share file."))
  (if (lshare:locked-p)
      (if (not (eq 'y (read-minibuffer "Forceably obtain lock? (y/n): ")))
	  (error "File is locked by someone else.")))
  (setq buffer-read-only nil)
  (save-excursion
    (goto-char 0)
    (search-forward lshare:*share-marker*)
    (delete-line)
    (insert "locked:" (user-full-name) ":" (current-time-string))
    (save-buffer))
  (message "File Locked."))

(defun unlock (&optional quiet ask)
  (interactive)
  (if (not (lshare:share-file-p))
      (error "File is not a share file."))
  (if (not (lshare:locked-by-user-p))
      (error "File is not locked by you."))
  (if (or (not ask)
	  (eq 'y (read-minibuffer
		   (format "Unlock Buffer: %s? (y/n): "
			   (buffer-name (current-buffer))))))
      (save-excursion
	(goto-char 0)
	(search-forward lshare:*share-marker*)
	(delete-line)
	(insert "unlocked:" (current-time-string))
	(save-buffer)
	(setq buffer-read-only t)
	(if (not quiet)
	    (message "File Unlocked.")))
      (if (not quiet)
	  (message "File Remains Locked."))))

(defun unlock-all (&optional ask)
  (interactive)
  (let ((buffers (buffer-list))
	(cb (current-buffer)))
    (while buffers
      (set-buffer (car buffers))
      (if (lshare:locked-by-user-p)
	  (unlock t ask))
      (setq buffers (cdr buffers)))))

(defun unlock-some ()
  (interactive)
  (unlock-all t))

(defun share-lock ()
  (interactive)
  (if (lshare:share-file-p)
      (if buffer-read-only
	  (lock)
	  (progn
	    (beep)
	    (message "Use C-x C-s to unlock lshare file.")))
      (if buffer-read-only
	  (toggle-read-only)
	  (if (buffer-modified-p)
	      (progn
		(beep)
		(toggle-read-only)
		(message "Warning - making modified buffer read-only."))
	      (toggle-read-only)))))

(defun share-save-buffer ()
  (interactive)
  (if (lshare:share-file-p)
      (unlock)
      (save-buffer)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with find-file itself to make everything find files the            ;;;
;;; lshare way.                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun find-file (filename)
  "Edit file FILENAME.
Switch to a buffer visiting file FILENAME,
creating one if none already exists."
  (interactive "FFind file: ")
  (switch-to-buffer (find-file-noselect filename))
  (if (lshare:share-file-p)
      (setq buffer-read-only t)))

(defun find-file-other-window (filename)
  "Edit file FILENAME, in another window.
May create a new window, or reuse an existing one;
see the function display-buffer."
  (interactive "FFind file in other window: ")
  (switch-to-buffer-other-window (find-file-noselect filename))
  (if (lshare:share-file-p)
      (setq buffer-read-only t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with kill-buffer.                                                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun kill-buffer (buffer-or-name)
  (interactive "bKill Buffer: ")
  (let ((cb (current-buffer)))
    (set-buffer buffer-or-name)
    (if (and (lshare:locked-by-user-p)
	     buffer-file-name)
	(unlock t t))
    (set-buffer cb))
  (funcall lshare:*orig-kill-buffer* buffer-or-name))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Muck with revert-buffer                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq revert-buffer-function 'share-revert-buffer)

(defun share-revert-buffer (&optional arg1 arg2)
  (let ((revert-buffer-function nil))
    (if (lshare:share-file-p)
	(if buffer-read-only
	    (progn
	      (revert-buffer arg1 arg2)
	      (if (lshare:share-file-p)
		  (setq buffer-read-only t)))
	    (revert-buffer arg1 arg2))
	(revert-buffer arg1 arg2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Libbuf:                                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Emacs Buffer routines.                                                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:switch-to-buffer                                           ;;;
;;;                                                                         ;;;
;;; Creation Date:     Mon Jul 16 12:27:14 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 19:50:10 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: buffer -- A buffer string.                                        ;;;
;;;                                                                         ;;;
;;; Description: This function switches buffers for the user in a           ;;;
;;;              convienent manner.  If there is only one window up,        ;;;
;;;              it splits the screen and puts the buffer up in the         ;;;
;;;              other window, and switches to it.  If the windows          ;;;
;;;              are split, then it switches to the buffer in the           ;;;
;;;              currently selected window.                                 ;;;
;;;                                                                         ;;;
;;; Returns: Nothing.                                                       ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:switch-to-buffer "*scratch*")                         ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:switch-to-buffer (buffer)
  (cond ((get-buffer-window buffer)
	 (select-window (get-buffer-window buffer)))
	((one-window-p t)
	 (switch-to-buffer-other-window buffer))
	(t
	  (switch-to-buffer buffer))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:get-characters                                             ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 19:53:46 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: x -- an integer.                                                  ;;;
;;;                                                                         ;;;
;;; Description: This function is used to get the next X characters         ;;;
;;;              from the buffer starting at the point.  If X               ;;;
;;;              characters are not available, the greatest integer         ;;;
;;;              less than X are returned.                                  ;;;
;;;                                                                         ;;;
;;; Returns: A string of characters.                                        ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:get-characters 5)                                     ;;;
;;;           "Ab"                                                          ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:get-characters (x)
  (save-excursion
    ;; Find out how many characters we really can get.
    (let ((can-get (min x (+ (- (buffer-size) (point)) 1)))
	  (point (point)))
      ;; Return those characters.
      (forward-char can-get)
      (buffer-substring point (point)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:all-previous-char                                          ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 19:57:12 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: fun -- Any function                                               ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine if all the              ;;;
;;;              characters before the point on the same line pass the      ;;;
;;;              test FUN.                                                  ;;;
;;;                                                                         ;;;
;;; Returns: T or NIL.                                                      ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:all-previous-char '(lambda (x) t))                    ;;;
;;;          T                                                              ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:all-previous-char (fun)
  (save-excursion
    ;; Find the beginning of the line.
    (let ((begin (save-excursion (beginning-of-line) (point))))
      ;; Make sure all the preceeding characters pass the given test.
      (while (and (/= (point) begin)
		  (funcall fun (preceding-char)))
	(backward-char 1))
      (= begin (point)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:last-line-p                                                ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 19:58:52 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine if the point is on      ;;;
;;;              the last line of the buffer.                               ;;;
;;;                                                                         ;;;
;;; Returns: T or NIL.                                                      ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:last-line-p)                                          ;;;
;;;          NIL                                                            ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:last-line-p ()
  (save-excursion
    (end-of-line)
    (= (point) (point-max))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:first-line-p                                               ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 19:59:57 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine if the point is on      ;;;
;;;              the first line of the buffer.                              ;;;
;;;                                                                         ;;;
;;; Returns: T or NIL.                                                      ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:first-line-p)                                         ;;;
;;;          T                                                              ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:first-line-p ()
  (save-excursion
    (beginning-of-line)
    (= (point) (point-min))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:column                                                     ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 20:02:30 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine what column the         ;;;
;;;              point is in.                                               ;;;
;;;                                                                         ;;;
;;; Returns: An integer.                                                    ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:column)                                               ;;;
;;;          12                                                             ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:column ()
  (- (point) (save-excursion (beginning-of-line) (point))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:line-length                                                ;;;
;;;                                                                         ;;;
;;; Creation Date:     Tue Jul  3 12:07:36 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 20:03:48 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine the length of the       ;;;
;;;              line that the point is on.                                 ;;;
;;;                                                                         ;;;
;;; Returns: An integer.                                                    ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:line-length)                                          ;;;
;;;          3                                                              ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:line-length ()
  (save-excursion (- (progn (end-of-line) (point))
		     (progn (beginning-of-line) (point)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:previous-line                                              ;;;
;;;                                                                         ;;;
;;; Creation Date:     Tue Jul  3 12:12:40 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 20:05:19 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function is used to go to the previous line.          ;;;
;;;              It tries to keep the column the same as that on the        ;;;
;;;              starting line.  When it can't, it goes to the end of       ;;;
;;;              the line.                                                  ;;;
;;;                                                                         ;;;
;;; Returns: Nothing.                                                       ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:previous-line)                                        ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:previous-line ()
  (let ((column (libbuf:column)))
    (forward-line -1)
    (if (>= (libbuf:line-length) column) (forward-char column)
      (end-of-line))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:next-line                                                  ;;;
;;;                                                                         ;;;
;;; Creation Date:     Tue Jul  3 12:13:37 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 20:06:52 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: *NONE*                                                            ;;;
;;;                                                                         ;;;
;;; Description: This function moves the point to the next line.  It        ;;;
;;;              tries to keep the point in the same column.  If it         ;;;
;;;              can't, the point is moved to the end of the line.          ;;;
;;;                                                                         ;;;
;;; Returns: Nothing.                                                       ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:next-line)                                            ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:next-line ()
  (let ((column (libbuf:column)))
    (forward-line 1)
    (if (>= (libbuf:line-length) column) (forward-char column)
      (end-of-line))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:indent                                                     ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 20:26:08 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: stablep -- Either T or NIL.                                       ;;;
;;;                                                                         ;;;
;;; Description: This function tabs so that the point lines up with         ;;;
;;;              text on the line above the current one.  If STABLEP        ;;;
;;;              is T, this function will only move the point until there   ;;;
;;;              is no whitespace above it.  If STABLEP is NIL              ;;;
;;;              (the default), then the point will be moved to the         ;;;
;;;              beginning of the next block of text.                       ;;;
;;;                                                                         ;;;
;;; Returns: Nothing.                                                       ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:indent)                                               ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:indent (&optional stablep)
  (interactive)
  ;; Do nothing if this is the first line.
  (cond ((libbuf:first-line-p) 0)
	(t
	 (let ((end nil)
	       (ws 0))
	   ;; Go up to the previous line.
	   (save-excursion
	     (libbuf:previous-line)
	     ;; Find the end of the line.
	     (setq end (save-excursion (end-of-line) (point)))
	     ;; If the next character is a whitespace, find out where 
	     ;; the next non whitespace is.
	     (cond ((libaux:simple-whitespace-p (following-char))
		    (while (and (/= (point) end)
				(libaux:simple-whitespace-p (following-char)))
		      (forward-char 1)
		      (setq ws (1+ ws))))
		   ;; If the next character isn't whitespace, and 
		   ;; we're stable, don't do anything.  If we are 
		   ;; stable:
		   ((not stablep)
		    ;; Find the next whitespace.
		    (while (and (/= (point) end)
				(not (libaux:simple-whitespace-p
				      (following-char))))
		      (forward-char 1)
		      (setq ws (1+ ws)))
		    ;; Find the end of the whitespace.
		    (while (and (/= (point) end)
				(libaux:simple-whitespace-p (following-char)))
		      (forward-char 1)
		      (setq ws (1+ ws))))))
	   ;; Insert the appropriate amount of spaces.
	   (insert-char 32 ws)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libbuf:insert-end-of-buffer                                       ;;;
;;;                                                                         ;;;
;;; Creation Date:     Mon Jul 23 13:53:54 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 21:21:59 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: process -- A process.                                             ;;;
;;;       buffer -- The buffer associated with that process.                ;;;
;;;       string -- The string to insert in the buffer.                     ;;;
;;;                                                                         ;;;
;;; Description: This function inserts text at the end of the               ;;;
;;;              buffer in a nice manner.  Nice scrolling and               ;;;
;;;              typeahead is supported.                                    ;;;
;;;                                                                         ;;;
;;; Returns: Nothing.                                                       ;;;
;;;                                                                         ;;;
;;; Example: >(libbuf:insert-end-of-buffer process "classic" "3")           ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libbuf:insert-end-of-buffer (process buffer string)
  (let ((current (current-buffer))
	(current-window (selected-window))
	(window (get-buffer-window buffer)))
    (set-buffer buffer)
    ;; If our next insertion point is where the point is:
    (if (= (point) (marker-position (process-mark process)))
	;; Insert the string and move the point.
	(progn
	  (insert-string string)
	  (set-marker (process-mark process) (point)))
	;; Otherwise, insert the string but don't move the point.
	(save-excursion
	  (goto-char (marker-position (process-mark process)))
	  (insert-string string)
	  (set-marker (process-mark process) (point))))
    ;; If the buffer is visible, and the point isn't visisble in the 
    ;; window, and the marker is at the end of the buffer, then 
    ;; scroll the window nicely.
    (if (and window
	     (/= scroll-step 0)
	     (not (pos-visible-in-window-p (point) window)) (eobp))
	(progn
	  (select-window window)
	  (recenter -1)
	  (select-window current-window)))
    (set-buffer current)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Libaux:                                                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Auxiliary Library.                                                      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:search-char                                                ;;;
;;;                                                                         ;;;
;;; Creation Date:     Sun Jul 22 21:00:19 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 14:13:18 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: target -- The character we're looking for.                        ;;;
;;;       string -- The string we're looking in.                            ;;;
;;;       offset -- The position in the string to start looking.            ;;;
;;;                 This defaults to the beginning.                         ;;;
;;;                                                                         ;;;
;;; Description: This function is used to locate a character in a           ;;;
;;;              given string.  It returns the position of the first        ;;;
;;;              occurence of the character in the string or NIL if         ;;;
;;;              the character doesn't exist.                               ;;;
;;;                                                                         ;;;
;;; Returns: An integer or NIL.                                             ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:search-char ?i "Heide is god")                        ;;;
;;;          2                                                              ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:search-char (target string &optional offset)
  (let ((next (or offset 0))
	(len (length string))
	(found nil))
    ;; Loop until we find the target character or we exhaust the 
    ;; string. 
    (while (and (not found)
		(< next len))
      ;; If we find the character, save its position.
      (if (char-equal (aref string next) target)
	  (setq found next))
      (setq next (1+ next)))
    ;; Return the position of the character.
    found))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:prefix-equal                                               ;;;
;;;                                                                         ;;;
;;; Creation Date:     Sun Jul 22 21:04:15 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 14:16:13 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: str-a -- A string to compare                                      ;;;
;;;       str-b -- The other string to compare                              ;;;
;;;       offset-a -- Where to start comparing in string A.  Defaults       ;;;
;;;                   to the beginning.                                     ;;;
;;;       offset-b -- Where to start comparing in string B.  Defaults       ;;;
;;;                   to the beginning.                                     ;;;
;;;                                                                         ;;;
;;; Description: This function is used to see if parts of strings are       ;;;
;;;              equal.  If str-a is the same as str-b starting at          ;;;
;;;              the appropriate offsets and extending until one of         ;;;
;;;              the strings terminate, then T is returned.                 ;;;
;;;                                                                         ;;;
;;; Returns: Either T or NIL.                                               ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:prefix-equal "Himan" "ma" 2)                          ;;;
;;;          T                                                              ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:prefix-equal (str-a str-b &optional offset-a offset-b)
  (let ((next-a (or offset-a 0))
	(next-b (or offset-b 0))
	(len-a (length str-a))
	(len-b (length str-b))
	(same t))
    ;; Loop while the strings are the same and we still have some of 
    ;; each string left.
    (while (and same (< next-a len-a) (< next-b len-b))
      ;; If the strings aren't the same, prepare to return nil.
      (if (not (char-equal (aref str-a next-a) (aref str-b next-b)))
	  (setq same nil))
      ;; Do it for the next charactr in each string.
      (setq next-a (1+ next-a))
      (setq next-b (1+ next-b)))
    same))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:search-string                                              ;;;
;;;                                                                         ;;;
;;; Creation Date:     Sun Jul 22 21:12:09 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 14:17:47 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: target -- The string we're searching for.                         ;;;
;;;       string -- The string we're searching in.                          ;;;
;;;       offset -- Where to start looking in the string.  Defaults         ;;;
;;;                 to the beginning.                                       ;;;
;;;                                                                         ;;;
;;; Description: This function is used to find the ocurrence of one         ;;;
;;;              string inside another.  If it finds an match, the          ;;;
;;;              position of the match is returned.  If it finds a          ;;;
;;;              partial match (a full match might exist but we ran         ;;;
;;;              out of STRING so we can't be sure) a list of the           ;;;
;;;              position is returned.  If no match was found, NIL is       ;;;
;;;              returned.                                                  ;;;
;;;                                                                         ;;;
;;; Returns: A list of an integer, an integer, or NIL.                      ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:search-string "Hello" "Will He")                      ;;;
;;;          (5)                                                            ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:search-string (target string &optional offset)
  (let ((start nil)
	(hit (- (or offset 0) 1))
	(found nil))
    ;; While we have a first character match and we haven't found a 
    ;; real match:
    (while (and hit (not found))
      (setq start (1+ hit))
      ;; See if the first character of target is in the string.
      (setq hit (libaux:search-char (aref target 0) string start))
      ;; If so:
      (if hit
	  ;; See if the rest of the string is in there.
	  (setq found (libaux:prefix-equal target string 0 hit))))
    ;; If we found a match but the whole target wasn't found, return 
    ;; a list of the position.
    (if (and found (< (- (length string) hit) (length target)))
	(list hit)
	;; Otherwise, return either the position of the full hit or 
	;; NIL. 
	(and found hit))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:remove-if-reverse                                          ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 16:31:59 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: fun -- Any function                                               ;;;
;;;       lst -- Any list                                                   ;;;
;;;                                                                         ;;;
;;; Description: This function removes elements from LST if FUN             ;;;
;;;              returns T when called with the element.                    ;;;
;;;                                                                         ;;;
;;; Returns: The list in reverse order without the appropriate              ;;;
;;;          elements.                                                      ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:remove-if-reverse '(lambda (x) nil) '(a b c))         ;;;
;;;          (C B A)                                                        ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:remove-if-reverse (fun lst)
  (let ((l lst)
	(ret nil))
    (while l
      (cond ((not (funcall fun (car l)))
	     (setq ret (cons (car l) ret))))
      (setq l (cdr l)))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:simple-whitespace-p                                        ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 16:39:49 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: char -- Any character                                             ;;;
;;;                                                                         ;;;
;;; Description: This function tests to see if CHAR is a tab or a           ;;;
;;;              space.                                                     ;;;
;;;                                                                         ;;;
;;; Returns: T if CHAR is a tab or a space.                                 ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:simple-whitespace-p ?a)                               ;;;
;;;          NIL                                                            ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:simple-whitespace-p (char)
  (or (char-equal char ?\t)
      (char-equal char 32)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Name: libaux:lisp-comment-character-p                                   ;;;
;;;                                                                         ;;;
;;; Creation Date:     Fri Jun 29 13:12:21 1990                             ;;;
;;; Created By:        Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Last Modification: Thu Feb 14 16:41:47 1991                             ;;;
;;; Last Modified By:  Kevin Zalondek                                       ;;;
;;;                                                                         ;;;
;;; Args: char -- Any character                                             ;;;
;;;                                                                         ;;;
;;; Description: This function is used to determine if CHAR is a            ;;;
;;;              LISP comment character.                                    ;;;
;;;                                                                         ;;;
;;; Returns: T or NIL.                                                      ;;;
;;;                                                                         ;;;
;;; Example: >(libaux:lisp-comment-character-p ?A)                          ;;;
;;;          NIL                                                            ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun libaux:lisp-comment-character-p (char)
  (or (libaux:simple-whitespace-p char)
      (char-equal char 59)))

(defun libaux:subst (target object tree)
  (cond ((null tree) nil)
	((consp tree)
	 (cons (libaux:subst target object (car tree))
	       (libaux:subst target object (cdr tree))))
	((eq tree target) object)
	(t tree)))
