;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? various KNOWBEL resolve methods


(eval-when (load compile eval)

  (require 'tools)
  (require 'prove-structs)
  (require 'prove)

)

(defun do-resolve-unprovable (atom sigma state belief clause theory)
  (dlet* (((expr belief-expr theory-expr direction no-meta) (cdr atom)))
    (let ((use-theory (and (not (clause-var-p theory-expr)) (or (theory-find-theory theory theory-expr)
								(theory-find-theory theory 'root)))))
      (if (and use-theory (or (null belief-expr) (not (clause-var-p belief-expr))))
	(let* ((use-belief (or belief-expr belief))
	       (proved (horn-inference (if (listp expr) expr (list expr)) use-theory use-belief direction no-meta)))
	  (if proved
	    (values nil nil nil)
	    (values t nil (list 'unprovable))
	    ))
	(values nil nil nil)
	))
    )
  )


(defun do-resolve-provable (atom sigma state belief clause theory)
  (if state
    (if (eq state :init)
      (dlet* (((expr belief-expr theory-expr direction no-meta) (cdr atom)))
	(let ((use-theory (and (not (clause-var-p theory-expr)) (or (theory-find-theory theory theory-expr)
								(theory-find-theory theory 'root)))))
	  (if (and use-theory (or (null belief-expr) (not (clause-var-p belief-expr))))
	    (let ((use-belief (or belief-expr belief)))
	      (multiple-value-bind (proved new-state)
		  (horn-inference (if (listp expr) expr (list expr)) use-theory use-belief direction no-meta)
		(if proved
		  (values t new-state (list 'provable))
		  (values nil nil nil)
		  )))
	    (values nil nil nil)
	    )))
      (let ((proved (backward-inference-loop state)))
	(if  proved
	  (values t new-state (list 'provable))
	  (values nil nil nil)
	  )))
    )
  )


(defnamed-resolve unprovable resolve-unprovable (atom sigma state belief clause theory)
   (if (eq (car atom) 'not)
     (do-resolve-unprovable (cadr atom) sigma state belief clause theory)
     (do-resolve-provable atom sigma state belief clause theory)
     )
   )



(defnamed-resolve provable resolve-provable (atom sigma state belief clause theory)	  
  (if (eq (car atom) 'not)
    (do-resolve-provable (cadr atom) sigma state belief clause theory)
    (do-resolve-unprovable atom  sigma state belief clause theory)
    )
  )





(defun atom-meta-check (atom belief theory)
					;  this is application dependent, KNOWBEL identifies (not (attr ...)) as
					;  goals possibly inviting reflection
  (when (and (eq (car atom) 'not)
	     (eq (car (cadr atom)) 'attr))
    (dlet* (((pred token label class-label prop val time) (cadr atom))
	    (history (var-type time))
	    (token-attr (if (not (clause-var-p class-label))
			  (cond ((clause-var-p token) (find-token-attr class-label (var-type token) history belief))
				((kb-token-p token) (find-class-attr class-label (tok-inst-of token) history belief))
				((token-description-p token)
				 (doloop (inst-link (tok-desc-instance-of token))
				   :for (attr nil
					      (find-token-attr class-label (car inst-link)
							       (or (var-type (cadr inst-link)) history) belief))
				   :when attr :return attr))))
			))
      (if token-attr
	(get-attr-value (attr-token token-attr) (theory-meta-attr theory) history belief)
	(doloop (class (cond ((clause-var-p prop) (list (var-type prop)))
			     ((kb-token-p prop) (tok-inst-of prop))))
	 :splice (get-attr-value (if (prop-object-p class) (prop-dest class) class) (theory-meta-attr theory) history belief))
	)
      )
    )
  )



(defun find-meta-goal (clause)
  (cond ((active-clause-p clause)
	 (if (member (clause-number (ac-clause clause)) '(:meta-goal :meta-state :meta-resolvent))
	   (clause-number (ac-clause clause))
	   (doloop (just (ac-justifications clause))
	    :find (find-meta-goal just))
	   ))
	)
  )


(defun interpret-meta-sigma (sigma meta-sigma owner)
  (doloop (binding (cdr meta-sigma))
   :when (consp binding)
    (let ((var (cadr binding))
	  (val (caddr binding)))
      (when (and (clause-var-p var) (eq owner (var-owner var)))
	(setf (aref sigma (var-number var) 0) val)
	)
      ))
  )



(defun find-meta-inference (active-clause state gen)
  (let* ((belief (istate-belief state))
	 (theory (istate-root-theory state))
	 (atom (car (clause-expression (ac-clause active-clause)))))
    (if (sg-meta-state gen)
      (let* ((new (eq (sg-meta-state gen) :init))
	     (meta-theories (if new
			      (atom-meta-check atom belief theory)
			      (car (sg-meta-state gen)))
			    ))
	(if meta-theories
	  (let* ((meta-state (if new
			       (and meta-theories
				    (new-state (make-theory :name (gensym "meta-inference") :kb (theory-kb theory))
					       (mapcar #'(lambda (attr) (attr-value attr)) meta-theories)
					       belief
					       t))
			       (cdr (sg-meta-state gen)))))
	    (when new
	      (add-cnf-clause meta-state (car (cnf `(not (resolvent ,state ,active-clause $resolvent)))) belief :meta-resolvent '($resolvent))
	      (add-cnf-clause meta-state (car (cnf `(not (state ,state ,active-clause $state)))) belief :meta-state '($state))
	      (add-cnf-clause meta-state (car (cnf `(not (goal ',(cadr atom) ,state $sigma $props)))) belief :meta-goal '($sigma $props))
	      (add-cnf-clause meta-state (car (cnf `(not (goal ',(cadr atom) ,state $sigma)))) belief :meta-goal '($sigma))
	      )
	    (let ((contradiction (backward-inference-loop meta-state)))
	      (if contradiction
		(let ((sigma (istate-get-sigma state (clause-n-vars (ac-clause active-clause)) 2))
		      (props nil))
		  (reset-sigma sigma (ac-clause active-clause))
		  (case (find-meta-goal contradiction)
		    (:meta-resolvent)
		    (:meta-state)
		    (:meta-goal (interpret-meta-sigma sigma (car (ac-collect contradiction)) (ac-clause active-clause))
				(setf props (cadr (ac-collect contradiction))))
		    (t nil)
		    )
		  (let* ((new-clause (build-resolvent (ac-clause active-clause)
						      (clause-expression (ac-clause active-clause))
						      nil
						      nil
						      sigma
						      belief
						      t))
			 (nc (active-stash-state new-clause state)))
		    (setf (ac-collect nc) (apply-sigma (ac-collect active-clause) sigma
						       new-clause
						       (ac-clause active-clause) nil))
		    (doloop :iter (prop-tail props (cddr prop-tail))
		     :while prop-tail
		      (setf (get-clause-prop new-clause (car prop-tail)) (cadr prop-tail))
		      )
		    (setf (ac-justifications nc) (list (cons active-clause (cons :meta (collect-explanation contradiction t)))))
		    (setf (sg-meta-state gen) (cons meta-theories meta-state))
		    (when (and (istate-best-p state) (null (clause-get-prop new-clause :value)))
		      (setf (clause-get-prop new-clause :value) (clause-get-prop (ac-clause active-clause) :value))
		      )
		    nc
		    ))
		(progn (setf (sg-meta-state gen) nil) nil)
		)
	      )
	    ))
	))
    )
  )




;;; what do we need to test in meta

;;;   (varp <expr>) (true if <expr> is a variable that is not a variable of the current clause)
;;;   (state <state> <prop> <val>) props are :horn (t or nil) :search-style (:depth-first :breadth-first :best-first)
;;;                              :belief-time    :theory-list  :number-of-resolvents 



(defresolve varp (atom sigma state belief clause theory)
  (declare (ignore state belief clause theory))
  (if (eq (car atom) 'not)
    (dlet* (((expr) (cdr (cadr atom))))
      (if (and (clause-var-p expr)
	       (not (eq (var-owner expr) (ac-clause clause))))
	(values t nil (list 'varp))
	)))
  )



(defresolve state (atom sigma state belief clause theory)
  (declare (ignore state belief clause theory))
  (if (eq (car atom) 'not)
    (dlet* (((state prop val) (cdr (cadr atom))))
      (if (inference-state-p state)
	(let ((prop-val (case prop
			  (:horn (istate-horn state))
			  (:search-style (istate-search-style state))
			  (:belief-time (istate-belief state))
			  (:theory-list (istate-theory-list state))
			  (:number-of-resolvents (length (theory-clauses (istate-active-theory state))))
			  (t :fail)
			  )))
	  (when (not (eq :fail prop-val))
	    (if (unify-var-p val (ac-clause clause) nil)
	      (progn
		(setf (aref sigma (var-number val) 0) prop-val)
		(values t nil (list 'prop))
		)
	      (when (eql val prop-val)
		(values t nil (list 'prop))
		)
	      )
	    )
	  )))
    )
  )


(defresolve belief-now (atom sigma state belief clause theory)
  (declare (ignore state belief clause theory))
  (if (eq (car atom) 'not)
    (dlet* (((time) (cdr (cadr atom)))
	    (actual-time  (get-universal-time)))
      (if (unify-var-p time (ac-clause clause) nil)
	(progn
	  (setf (aref sigma (var-number time) 0) actual-time)
	  (values t nil (list (list 'belief-now actual-time)))
	  )
	(when (eql time actual-time)
	  (values t nil (list (list 'belief-now actual-time)))
	  )
	)
      ))
  )


(defresolve now (atom sigma state belief clause theory)
  (declare (ignore state belief clause theory))
  (if (eq (car atom) 'not)
    (dlet* (((time) (cdr (cadr atom)))
	    (actual-time  (get-universal-time)))
      (if (unify-var-p time (ac-clause clause) nil)
	(progn
	  (setf (aref sigma (var-number time) 0) actual-time)
	  (values t nil (list (list 'now actual-time)))
	  )
	(when (eql time actual-time)
	  (values t nil (list (list 'now actual-time)))
	  )
	)
      ))
  )


(defresolve history-now (atom sigma state belief clause theory)
  (declare (ignore state belief clause theory))
  (if (eq (car atom) 'not)
    (dlet* (((time) (cdr (cadr atom)))
	    (actual-time  (parse-history 'now)))
      (if (unify-var-p time (ac-clause clause) nil)
	(progn
	  (setf (aref sigma (var-number time) 1) actual-time)
	  (values t nil (list (list 'history-now actual-time)))
	  )
	)
      ))
  )


(defun class-all-isa-parents (class type history belief state)
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc))
      (nd-loop (isa-prop (tok-parents class))
	  (nd-if (belief-satisfies belief (prop-belief isa-prop))
	    (nd-progn
	     (setq h (time-intersect history (prop-history isa-prop) belief))
	     (nd-if h
	       (tconc to-check (cons h (prop-dest isa-prop))))
	     ))
	)
      (nd-while (car to-check)
		(nd-if (or (null type) (type-instance (cdaar to-check) type (caaar to-check) belief))
		  (success (caar to-check)))
		(nd-loop (isa-prop (tok-parents (cdaar to-check)))
		    (nd-if (belief-satisfies belief (prop-belief isa-prop))
		      (nd-progn
		       (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
		       (nd-if h
			 (tconc to-check (cons h (prop-dest isa-prop))))
		       ))
		  )
		(setf (car to-check) (cdar to-check))
		)
      )
    )
  )


(defun all-isa-parents (class type history belief state)
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc nil (cons history class)))
     loop
      (nd-if (or (null type) (type-instance (cdaar to-check) type (caaar to-check) belief))
	(success (caar to-check)))
      (nd-loop (isa-prop (tok-parents (cdaar to-check)))
	  (nd-if (belief-satisfies belief (prop-belief isa-prop))
	    (nd-progn
	     (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	     (nd-if h
	       (tconc to-check (cons h (prop-dest isa-prop))))
	     ))
	)
      (setf (car to-check) (cdar to-check))
      (nd-if (car to-check) (go loop))
      )
    )
  )


(defun class-all-subclasses (class type history belief state)
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc))
      (nd-loop (isa-prop (tok-children class))
	  (nd-if (belief-satisfies belief (prop-belief isa-prop))
	    (nd-progn
	     (setq h (time-intersect history (prop-history isa-prop) belief))
	     (nd-if h
	       (tconc to-check (cons h (prop-src isa-prop))))
	     ))
	)
      (nd-while (car to-check)
		(nd-if (or (null type) (type-instance (cdaar to-check) type (caaar to-check) belief))
		  (success (caar to-check)))
		(nd-loop (isa-prop (tok-children (cdaar to-check)))
		    (nd-if (belief-satisfies belief (prop-belief isa-prop))
		      (nd-progn
		       (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
		       (nd-if h
			 (tconc to-check (cons h (prop-src isa-prop))))
		       ))
		  )
		(setf (car to-check) (cdar to-check))
		)
      )
    )
  )



(defun instance-path (object class history belief state) ;; return  the time interval for which the path is true
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc))
      (nd-if (special-token-p object)
	(tconc to-check (cons history (special-token-instance-of object)))
	(nd-if (token-description-p object)
	  (nd-loop (inst-desc (tok-desc-instance-of object))
	      (setq h (time-intersect history (var-type (cadr inst-desc)) belief))
	    (if h (tconc to-check (cons h (car inst-desc))))
	    )
	  (nd-loop (inst-prop (tok-inst-of object))
	      (if (belief-satisfies belief (prop-belief inst-prop))
		(progn
		  (setq h (time-intersect history (prop-history inst-prop) belief))
		  (if h (tconc to-check (cons h (prop-dest inst-prop))))
		  )
		))))
     loop
      (nd-if (not (car to-check)) (fail))
      (nd-if (eq class (cdaar to-check))
	(nd-progn (success (caaar to-check)))
	(nd-loop (isa-prop (tok-parents (cdaar to-check)))
	    (nd-if (belief-satisfies belief (prop-belief isa-prop))
	      (nd-progn
	       (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	       (nd-if h
		 (tconc to-check (cons h (prop-dest isa-prop)))
		 )
	       ))
	  ))
      (setf (car to-check) (cdar to-check))
      (go loop)
      )
    )
  )


(defun gen-all-attrs (object history belief state)
  (ndprog state nil
    (nd-for isa-prop parent-cont (isa-all-parents object nil history belief parent-cont)
	    (nd-loop (obj-attr (lookup-index-cdr (tok-attrs (cdr isa-prop)) :all))
		(success obj-attr)
	      ))
    )
  )
	    

;; returns reason if necessary attribute check FAILS

(defun do-necessary-check (object class history belief theory)
  (let ((necessary (lookup-type 'necessary theory))
	(result nil)
	(found nil)
	h)
    (when (or (clause-var-p class) (clause-var-p object)) (break))
    (ndprog :init nil
      (nd-for parent parent-cont (all-isa-parents class nil history belief parent-cont)
	      (nd-loop (attr (lookup-index-cdr (tok-attrs (cdr parent)) :all))
		  (nd-if (belief-satisfies belief (attr-belief attr))
		    (nd-progn
		     (setq h (time-intersect (attr-history attr) (car parent) belief))
		     (nd-if h
		       (nd-for necessary-h necessary-cont (instance-path attr necessary h belief necessary-cont)
			       (setq found nil)
			       (nd-for obj-attr obj-cont (gen-all-attrs object necessary-h belief obj-cont)
				   (nd-for attr-h attr-cont (instance-path obj-attr attr necessary-h belief attr-cont)
					   (setq found t)
					   (go exit)
					   ))
			       exit
			       (nd-if (not found)
				 (nd-progn
				  (setq result attr)
				  (fail)
				  )))
		       )))
		)))
    result)
  )


(defresolve necessary-check (atom sigma state belief clause theory)
  ;; only run if everything bound
  (if (eq (car atom) 'necessary-check)
    (dlet* (((object class time) (cdr atom))
	    r)
      (if (and (unify-var-p time (ac-clause clause) nil)
	       (not (unify-var-p object (ac-clause clause) nil))
	       (not (unify-var-p class (ac-clause clause) nil))
	       (setq r (do-necessary-check object class (var-type time) belief theory)))
	(values t nil (list 'prim-necessary-check r))
	(values :fail nil nil)
	))
    (value :fail nil nil)
    )
  )


(defun class-check-unique (h inst checked belief)
  (not (doloop (pair checked)
	:when (and (eq (cdr pair) inst) (time-test (car pair) :=  h belief))
	:return t)
       )
  )



(defun class-all-instances (class type history belief state) ;; return  the time interval for which the path is true
  (let (h r)
    (ndprog state (to-check checked)
      (setf to-check (tconc nil (cons history class)))
     loop
      (nd-loop (inst-prop (tok-instances (cdaar to-check)))
	  (nd-if (belief-satisfies belief (prop-belief inst-prop))
	    (nd-progn
	     (setq h (time-intersect (caaar to-check) (prop-history inst-prop) belief))
	     (nd-if (and h (or (null type) (type-instance (prop-src inst-prop) type h belief)))
	       (nd-if (class-check-unique h (prop-src inst-prop) checked belief)
		 (nd-progn
		  (setq r (cons h (prop-src inst-prop)))
		  (setf checked (cons r checked))
		  (success r)
		  ))
	       ))
	    ))
      (nd-loop (isa-prop (tok-children (cdaar to-check)))
	  (if (belief-satisfies belief (prop-belief isa-prop))
	    (progn
	      (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	      (if h (tconc to-check (cons h (prop-src isa-prop))))
	      )
	    ))
      (setf (car to-check) (cdar to-check))
      (nd-if (car to-check) (go loop))
      )
    ))

(defun special-token-instance-of (&rest ignore) nil)


(defun attr-special-token (object label labelv class-label class-labelv value valuev prop propv history historyv
			   sigma state belief clause theory state)
  nil)

(defun instance-all-parents (object type history belief state) ;; return  the time interval for which the path is true
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc))
      (nd-if (special-token-p object)
	(tconc to-check (cons history (special-token-instance-of object)))
	(nd-if (token-description-p object)
	  (nd-loop (inst-desc (tok-desc-instance-of object))
	      (setq h (time-intersect history (var-type (cadr inst-desc)) belief))
	    (if h (tconc to-check (cons h (car inst-desc))))
	    )
	  (nd-loop (inst-prop (tok-inst-of object))
	      (if (belief-satisfies belief (prop-belief inst-prop))
		(progn
		  (setq h (time-intersect history (prop-history inst-prop) belief))
		  (if h (tconc to-check (cons h (prop-dest inst-prop))))
		  )
		))))
     loop
      (nd-if (not (car to-check)) (fail))
      (nd-if (or (null type) (type-instance (cdaar to-check) type (caaar to-check) belief))
	(success (cons (caaar to-check) (cdaar to-check))))
      (nd-loop (isa-prop (tok-parents (cdaar to-check)))
	  (nd-if (belief-satisfies belief (prop-belief isa-prop))
	    (nd-progn
	     (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	     (nd-if h
	       (tconc to-check (cons h (prop-dest isa-prop))))
	     ))
	)
      (setf (car to-check) (cdar to-check))
      (go loop)
      )
    ))


(defun all-inst-parents (object-type class-type history belief state)
  (ndprog state nil
    (nd-for instance inst-cont (class-all-instances object-type nil history belief inst-cont)
	    (nd-for class class-cont (instance-all-parents (cdr instance) class-type (car instance) belief class-cont)
		    (success (list (car class) (cdr instance) (cdr class)))
		    )
	    )
    )
  )


(defun do-resolve-instance-of (object class time atom sigma state belief clause theory)
  (cond ((unify-var-p object (ac-clause clause) nil)
	 (if (unify-var-p class (ac-clause clause) nil)
	   (dlet* ((((h o c) . new-state)
		    (all-inst-parents (sigma-lookup object sigma 1) (sigma-lookup class sigma 1) (var-type time) belief state)))
	     (if new-state
	       (progn
		 (setf (sigma-lookup time sigma 1) h)
		 (setf (sigma-lookup object sigma 0) o)
		 (setf (sigma-lookup class sigma 0) c)
		 (values t new-state (list 'inst-all-all))
		 )
	       )
	     )
	   (dlet* ((((h . o) . new-state) (class-all-instances class (sigma-lookup object sigma 1) (var-type time) belief state)))
	       (if new-state
		 (progn
		   (setf (sigma-lookup time sigma 1) h)
		   (setf (sigma-lookup object sigma 0) o)
		   (values t new-state (list 'inst-object-all))
		   )
		 )
	       )))
	((unify-var-p class (ac-clause clause) nil)
	 (dlet* ((((h . c) . new-state) (instance-all-parents object (sigma-lookup class sigma 1) (var-type time) belief state)))
	   (if new-state
	     (progn
	       (setf (sigma-lookup time sigma 1) h)
	       (setf (sigma-lookup class sigma 0) c)
	       (values t new-state (list 'inst-all-class))
	       )
	     )
	   ))
	((prim-type-instance object class) (values t nil (list 'prim-inst)))
	((and (or (kb-token-p object) (token-description-p object) (special-token-p object)) (kb-token-p class))
	 (dlet* (((h . new-state) (instance-path object class (var-type time) belief state)))
	   (if new-state
	     (progn
	       (setf (sigma-lookup time sigma 1) h)
	       (values t new-state (list 'inst-all-paths))
	       )
	     )
	   )))
  )



(defresolve (instance instance-of) (atom sigma state belief clause theory)
  (if (and state (eq (car atom) 'not))
    (dlet* (((object class time) (cdr (cadr  atom))))
      (do-resolve-instance-of object class time atom sigma state belief clause theory)))  
  )




(defun isa-all-parents (object type history belief state) ;; includes the object itself, returns (h . object)
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc nil (cons history object)))
     loop
      (nd-if (not (car to-check)) (fail))
      (nd-if (or (null type) (type-instance (cdaar to-check) type (caaar to-check) belief))
	(success (caar to-check)))
      (nd-loop (isa-prop (tok-parents (cdaar to-check)))
	  (nd-if (belief-satisfies belief (prop-belief isa-prop))
	    (nd-progn
	     (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	     (nd-if h
	       (tconc to-check (cons h (prop-dest isa-prop))))
	     ))
	)
      (setf (car to-check) (cdar to-check))
      (go loop)
      )
    ))


(defun attr-all-labels (attr-prop history belief class-labelv state)
  (ndprog state nil
    (nd-if (not class-labelv)
      (success nil)
      (nd-for parent parent-cont (instance-all-parents attr-prop nil history belief parent-cont)
	(nd-if (kb-attr-p (cdr parent))
	  (success (attr-label (cdr parent)))
	  ))
      ))
  )
      

(defun do-attr-object (object-spec history belief clause sigma state)
  ;;; account for a variable in the object position
  (ndprog state nil
    (nd-if (unify-var-p object-spec (ac-clause clause) nil)
      (nd-for object-prop obj-state (class-all-instances (sigma-lookup object-spec sigma 1) nil history belief obj-state)
	      (success object-prop))
      (success (cons history object-spec))
      )
    )
  )


(defun desc-gen-attr (desc label labelv class-label class-labelv value valuev prop propv history historyv sigma active-clause belief state)
  (declare (ignore valuev))
  (let* (h
	(r (and (or propv (clause-wild-card-p prop))
		history
		(ndprog state nil
		  (nd-loop (attr-group (tok-desc-attrs desc))
		      (nd-if (or class-labelv (clause-wild-card-p class-label) (eq class-label (car attr-group)))
			(nd-loop (attr-prop (cdr attr-group))
			    (reset-sigma sigma (ac-clause active-clause))
			    (nd-if (and (or labelv (clause-wild-card-p label) (eq label (car attr-prop)))
					(unify-work value (cadr attr-prop) sigma (ac-clause active-clause) nil belief))
			      (nd-progn
			       (setq h (time-intersect history (var-type (caddr attr-prop)) belief))
			       (nd-if h
				 (nd-progn
				  (if labelv (setf (aref sigma labelv 0) (car attr-prop)))
				  (if class-labelv (setf (aref sigma class-labelv 0) (car attr-group)))
				  (if propv (setf (aref sigma propv 0) attr-prop))
				  (if historyv (setf (aref sigma historyv 1) h))
				  (success attr-prop))
				 )))
			  ))
		    ))
		)))
    r)
  )


(defun attr-special (object class-label value labelv valuev historyv sigma history belief state)
  (ndprog state (special)
    (nd-for prop prop-cont (instance-all-parents object nil history belief prop-cont)
      (setf special (assoc class-label (tok-get-prop (cdr prop) :lisp-rep)))
      (nd-if special
	(nd-for val val-cont (apply (cadr special) object class-label value valuev
				      history belief val-cont (cdddr special))
	  (if labelv (setf (aref sigma labelv 0) (car val)))
	  (if valuev (setf (aref sigma valuev 0) (cadr val)))
	  (if historyv (setf (aref sigma historyv 1) (caddr val)))
	  (success special)
	  )
	)
      )
    )
  )


(defmacro system-object-p (object clause)
  `(mlet ((object ,object))
	 (and (not (unify-var-p object (ac-clause ,clause) nil))
	      (not (kb-token-p object))
	      (not (token-description-p object))
	      ))
  )


(defun attr-system-object (object label labelv class-label class-labelv value valuev prop propv history
			   sigma state belief clause theory)
  (if (not class-labelv)
      (cond ((inference-state-p object)	   
	     (let ((val (case-string= class-label
			  ("current-clause" (istate-current-clause object))
			  ("search-mode" (istate-search-style object))
			  ("active-theory" (istate-active-theory object))
			  (t :error)
			  )))
	       (if (not (eq val :error))
		   (if valuev
		       (progn (setf (aref sigma valuev 0) val) (cons nil 'inference-state))
		     (if (eql val value)
			 (cons nil 'inference-state)
		       nil))
		 )))
	    ((active-clause-p object)	   
	     (let ((val (case-string= class-label
			  ("clause" (ac-clause object))
			  ("justifications" (ac-justifications object))
			  (t :error)
			  )))
	       (if (not (eq val :error))
		   (if valuev
		       (progn (setf (aref sigma valuev 0) val) (cons nil 'active-clause))
		     (if (eql val value)
			 (cons nil 'inference-state)
		       nil)
		     ))))
	    ((clause-p object)	   
	     (let ((val (case-string= class-label
			  ("expression" (clause-expression object))
			  ("name" (clause-number object))
			  ("belief-time" (clause-belief-time object))
			  (t :error)
			  )))
	       (if (not (eq val :error))
		   (if valuev
		       (progn (setf (aref sigma valuev 0) val) (cons nil 'clause))
		     (if (eql val value)
			 (cons nil 'inference-state)
		       nil)
		     ))))
	    )
    nil
    )
  )

(defun special-token-p (object)
  nil)
  

(defun do-resolve-attr (atom sigma state belief clause theory istate)
  (cond ((or (null state) (not (eq (car atom) 'not))) nil)
	(t
	 (dlet* ((args (cdadr atom))
		 ((object label class-label value prop history-var) args)
		 (objectv (when (unify-var-p object (ac-clause clause) nil) (var-number object)))
		 (object
		  (if (or (symbolp object) (stringp object))
		    (find-or-create (string-downcase (type-name object theory))
				    (theory-root-object theory) theory (theory-all-time theory) belief)
		   object))
		 (labelv (when (unify-var-p label (ac-clause clause) nil) (var-number label)))
		 (class-labelv (when (unify-var-p class-label (ac-clause clause) nil) (var-number class-label)))
		 (valuev (when (unify-var-p value (ac-clause clause) nil) (var-number value)))
		 (propv (when (unify-var-p prop (ac-clause clause) nil) (var-number prop)))
		 (historyv (when (unify-var-p history-var (ac-clause clause) nil) (var-number history-var)))
		 (history (if historyv
			    (sigma-lookup history-var sigma 1)
			    (error "~%Bad history var ~a ~a~%"
				   (normal-clause history-var)
				   (and (clause-var-p history-var) (normal-clause (var-owner history-var))))))
		 h attrs
		 (success
		  (if (special-token-p object)
		    (attr-special-token object label labelv class-label class-labelv value valuev prop propv history historyv
					sigma state belief clause theory istate)
		    (if (system-object-p object clause)
		      (attr-system-object object label labelv class-label class-labelv value valuev prop propv history
					  sigma state belief clause theory)
		      (if (token-description-p object)
			(desc-gen-attr object label labelv class-label class-labelv value
				       valuev prop propv history historyv sigma clause belief state)
			(ndprog state nil
			  (nd-for object-prop obj-cont (do-attr-object object history belief clause sigma obj-cont)
					; for each object satisfying arg 1
			    (nd-for isa-prop isa-cont (object-all-supers (cdr object-prop) (car object-prop) belief isa-cont)
			      (setf attrs (if (or class-labelv (clause-wild-card-p class-label))
					    (lookup-index-cdr (tok-attrs (cdr isa-prop)) :all)
					    (lookup-index-cdr (tok-attrs (cdr isa-prop)) class-label)))
			      (nd-if attrs
				(nd-loop (attr-prop attrs)
				    (nd-if (belief-satisfies belief (attr-belief attr-prop))
				      (nd-progn
				       (setq h (time-intersect (car isa-prop) (attr-history attr-prop) belief))
				       (nd-if (and h
						   (or labelv (clause-wild-card-p label) (eq label (attr-label attr-prop)))
						   (if propv
						     (type-instance attr-prop (aref sigma propv 1)  history belief)
						     (or (clause-wild-card-p prop) (eq attr-prop prop))
						     )
						   (or valuev (clause-wild-card-p value) (eql value (attr-value attr-prop)))
						   )
					 (nd-for label-val label-cont (attr-all-labels attr-prop h belief class-labelv label-cont)
					   (if objectv (setf (aref sigma objectv 0) (cdr object-prop)))
					   (if labelv (setf (aref sigma labelv 0) (attr-label attr-prop)))
					   (if class-labelv (setf (aref sigma class-labelv 0) label-val))
					   (if valuev (setf (aref sigma valuev 0) (attr-value attr-prop)))
					   (if propv (setf (aref sigma propv 0) attr-prop))
					   (if historyv (setf (aref sigma historyv 1) h))
					   (success attr-prop))
					 )
				       )))
				(nd-if (and (kb-token-p object)(not class-labelv))
				  (nd-for special special-cont (attr-special object class-label value labelv valuev historyv sigma
									     history belief special-cont)
				    (success special))
				  )
				)
			      ))
			  )
			))))
		 )
	   (if success (values t (cdr success) (list (car success))))
	   )))
  )
				 
(defresolve-s attr (atom sigma state belief clause theory istate)
  (do-resolve-attr atom sigma state belief clause theory istate)
  )



(defun declare-lisp-rep (class-name attr-label lookup-function tell-function theory &rest args)
  (let ((class (lookup-type class-name theory))
	(attr-label (find-string attr-label theory)))
    (let ((attr (find-token-attr (find-string attr-label theory) class (theory-all-time th) (std-belief)))
	  (info (list* attr-label lookup-function tell-function args)))
      (when attr
	(setf (tok-get-prop attr :special-rep) info)
	(push info (tok-get-prop class :lisp-rep))
	))
    )
  )




(defun isa-path (class1 class2 history belief state) ;; return  the time interval for which the path is true
  (let (h)
    (ndprog state (to-check)
      (setf to-check (tconc))
      (nd-if (token-description-p class1)
	(nd-loop (inst-desc (tok-desc-instance-of class1))
	    (setq h (time-intersect history (var-type (cadr inst-desc)) belief))
	  (if h (tconc to-check (cons h (car inst-desc))))
	  )
	(nd-loop (isa-prop (tok-parents class1))
	    (if (belief-satisfies belief (prop-belief isa-prop))
	      (progn
		(setq h (time-intersect history (prop-history isa-prop) belief))
		(if h (tconc to-check (cons h (prop-dest isa-prop))))
		)
	      )))
     loop
      (nd-if (not (car to-check)) (fail))
      (nd-if (eq class2 (cdaar to-check))
	(nd-progn (success (caaar to-check)))
	(nd-loop (isa-prop (tok-parents (cdaar to-check)))
	    (nd-if (belief-satisfies belief (prop-belief isa-prop))
	      (nd-progn
	       (setq h (time-intersect (caaar to-check) (prop-history isa-prop) belief))
	       (nd-if h
		 (tconc to-check (cons h (prop-dest isa-prop)))
		 )
	       ))
	  ))
      (setf (car to-check) (cdar to-check))
      (go loop)
      )
    )
  )


(defun gen-all-is-a (class1-type class2-type history belief state)
  (ndprog state nil
    (nd-for instance inst-cont (class-all-instances class1-type nil history belief inst-cont)
	    (nd-for class class-cont (class-all-isa-parents (cdr instance) class2-type (car instance) belief class-cont)
		    (success (list (car class) (cdr instance) (cdr class)))
		    )
	    )
    )
  )



(defun do-resolve-is-a (class1 class2 time atom sigma state belief clause theory)
  (cond ((unify-var-p class1 (ac-clause clause) nil)
	 (if (unify-var-p class2 (ac-clause clause) nil)
	   (dlet* ((((h o c) . new-state)
		    (gen-all-is-a (sigma-lookup class1 sigma 1) (sigma-lookup class2 sigma 1) (var-type time) belief state)))
	     (if new-state
	       (progn
		 (setf (sigma-lookup time sigma 1) h)
		 (setf (sigma-lookup class1 sigma 0) o)
		 (setf (sigma-lookup class2 sigma 0) c)
		 (values t new-state (list 'inst-all-all))
		 )
	       )
	     )
	   (dlet* ((((h . o) . new-state) (class-all-subclasses class2 (sigma-lookup class1 sigma 1) (var-type time) belief state)))
	       (if new-state
		 (progn
		   (setf (sigma-lookup time sigma 1) h)
		   (setf (sigma-lookup class1 sigma 0) o)
		   (values t new-state (list 'inst-class1-all))
		   )
		 )
	       )))
	((unify-var-p class2 (ac-clause clause) nil)
	 (dlet* ((((h . c) . new-state) (class-all-isa-parents class1 (sigma-lookup class2 sigma 1) (var-type time) belief state)))
	   (if new-state
	     (progn
	       (setf (sigma-lookup time sigma 1) h)
	       (setf (sigma-lookup class2 sigma 0) c)
	       (values t new-state (list 'inst-all-class2))
	       )
	     )
	   ))
	((and (kb-token-p class1) (kb-token-p class2))
	 (dlet* (((h . new-state) (isa-path class1 class2 (var-type time) belief state)))
	   (if new-state
	     (progn
	       (setf (sigma-lookup time sigma 1) h)
	       (values t new-state (list 'inst-all-paths))
	       )
	     )
	   )))
  )





(defresolve (isa is-a) (atom sigma state belief clause theory)
  (if (and state (eq (car atom) 'not))
    (dlet* (((object class time) (cdr (cadr  atom))))
      (do-resolve-is-a object class time atom sigma state belief clause theory)))  
  )



(defresolve find-prop (atom sigma state belief clause theory)
  (if (and state (eq (car atom) 'not))
    (dlet* (((expression prop-spec) (cdr (cadr  atom)))
	    (prop-state (find-prop expression theory belief state)))
      (if prop-state
	(if (unify-var-p prop-spec (ac-clause clause) nil)
	  (progn
	    (setf (sigma-lookup prop-spec sigma 0) (car prop-state))
	    (values t (cdr prop-state) (list 'find-prop)))
	  (if (eq prop-spec (car prop-state))
	    (values t (cdr prop-state) (list 'find-prop))
	    (values :fail nil nil))
	  )
	(values :fail nil nil))
      )
    )
  )



;; Return the justification of the current active clause

(defresolve justification (atom sigma state belief clause theory)
  ;; only run if arg is a var
  (if (eq (car atom) 'not)
    (dlet* (((justification) (cdr (cadr atom))))
      (if (unify-var-p justification (ac-clause clause) nil)
	(progn
	  (setf (sigma-lookup justification sigma 0) (ac-justifications clause))
	  (values t nil (list 'prim-just)))
	(values :fail nil nil)
	))
    (value :fail nil nil)
    )
  )


(defresolve set-of (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((member set) (cdr (cadr atom))))
      (if (not (unify-var-p member (ac-clause clause) nil))
	(progn
	  (if (unify-var-p set (ac-clause clause) nil)
	    (setf (sigma-lookup set sigma 0) (tconc nil member))
	    (when (not (member member (car set)))
	      (tconc set member)
	      ))
	  (values t nil (list 'set-of)))
	(values :fail nil nil)
	))
    (value :fail nil nil)
    )
  )





(defun object-all-supers (object history belief state) ; include object . history
  (ndprog state (h)
    (success (cons history object))
    (nd-loop (isa-prop (tok-parents object))
      (nd-if (belief-satisfies belief (prop-belief isa-prop))
	(nd-progn
	 (setf h (time-intersect history (prop-history isa-prop) belief))
	 (nd-if h
	   (nd-for super-prop super-state (object-all-supers (prop-dest isa-prop) h belief super-state)
	     (success super-prop)))
	 ))
      )
    )
  )
  



(defresolve compiled (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((theory-name proposition class) (cdr (cadr atom))))
      (if (and (not (unify-var-p proposition (ac-clause clause) nil))
	       (not (unify-var-p class (ac-clause clause) nil)))
	(let ((compile-to-theory (theory-find-theory theory theory-name)))
	  (if compile-to-theory
	    (progn
	      (dbgn-print 'compiled theory-name class proposition)
	      (doloop (clause proposition)
		(stash clause compile-to-theory t belief nil (list 'compiled class) nil))
	      (values :fail nil nil))
	    (values t nil (list 'compile-failed))
	    ))
	(values t nil (list 'compile-failed))
	))
    (values t nil (list 'compile-failed))
    )
  )






(defresolve time-test (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((arg1 op arg2) (cdr (cadr atom))))
      (when (keywordp op)
	(let ((int1 (if (and (unify-var-p arg1 (ac-clause clause) nil)
			     (time-int-p (var-type arg1)))
		      (var-type arg1)
		      (abstract-prop-history arg1)))
	      (int2 (if (and (unify-var-p arg2 (ac-clause clause) nil)
			     (time-int-p (var-type arg2)))
		      (var-type arg2)
		      (abstract-prop-history arg2))))
	  (when (and int1 int2 (time-test int1 op int2 belief))
	    (values t nil (list 'time-test op)))))
      )
    (value :fail nil nil)
    )
  )


(defresolve time-constrain (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((arg1 op arg2) (cdr (cadr atom))))
      (when (keywordp op)
	(let* ((int1 (if (and (unify-var-p arg1 (ac-clause clause) nil) (time-int-p (var-type arg1)))
		      (var-type arg1)
		      (abstract-prop-history arg1)))
	      (int2 (if (and int1 (unify-var-p arg2 (ac-clause clause) nil) (time-int-p (var-type arg2)))
		      (var-type arg2)
		      (abstract-prop-history arg2))))
	  (when (and int1 int2)
	    (let ((ints (time-assert int1 op int2 belief belief belief)))	
	      (if (unify-var-p arg1 (ac-clause clause) nil)      
		(setf (sigma-lookup arg1 sigma 1) (car ints)))
	      (if (unify-var-p arg2 (ac-clause clause) nil)
		(setf (sigma-lookup arg2 sigma 1) (cadr ints)))
	      (values t nil (list 'time-constrain op)))))
	)
      )
    (value :fail nil nil)
    )
  )


(defresolve time-assert (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((arg1 op arg2) (cdr (cadr atom))))
      (when (keywordp op)
	(let ((int1 (if (and (unify-var-p arg1 (ac-clause clause) nil)
			     (time-int-p (var-type arg1)))
		      (var-type arg1)
		     nil))
	      (int2 (if (and (unify-var-p arg2 (ac-clause clause) nil)
			     (time-int-p (var-type arg2)))
		      (var-type arg2)
		      (abstract-prop-history arg2))))
	  (when (and int1 int2)
	    (let ((new-int (get-time-int int1 belief)))
	      (time-assert new-int op int2 belief belief belief)
	      (setf (sigma-lookup arg1 sigma 1) new-int)
	      (values t nil (list 'time-constrain op)))))
	)
      )
    (value :fail nil nil)
    )
  )



(defresolve set-off (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((object arg) (cdr (cadr atom))))
      (values t nil (list 'set-off)))
    (value :fail nil nil)
    )
  )


(defresolve set-on (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((object arg) (cdr (cadr atom))))
      (values t nil (list 'set-on)))
    (value :fail nil nil)
    )
  )



(defresolve add-interpretation (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((object arg) (cdr (cadr atom))))
      (dbgn-print 'add-interp (cdr (cadr atom)))
      (values t nil (list 'add-interp)))
    (value :fail nil nil)
    )
  )





(defresolve time-of (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((object arg) (cdr (cadr atom)))
	    (int (get-history-interval object)))
      (if (and (unify-var-p arg (ac-clause clause) nil)
	       (time-int-p int))
	(progn
	  (setf (sigma-lookup arg sigma 1) int)
	  (values t nil (list 'time-of)))
	(value :fail nil nil))
      )
    (value :fail nil nil)
    )
  )





(defresolve print (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((object arg) (cdr (cadr atom))))
      (dbgn-print 'print (cdr (cadr atom)))
      (values t nil (list 'time-of)))
    (value :fail nil nil)
    )
  )





(defresolve current-clause-value (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((base-state val) (cdr (cadr atom))))
      (if (and (unify-var-p val (ac-clause clause) nil)
	       (inference-state-p base-state))
	(progn
	  (if (istate-current-clause base-state)
	    (setf (sigma-lookup val sigma 0) (or (clause-get-prop (ac-clause (istate-current-clause base-state)) :value) 0))
	    (setf (sigma-lookup val sigma 0) 0))
	  (values t nil (list 'current-clause-value)))
	(value :fail nil nil))
      )
    (value :fail nil nil)
    )
  )


(defun match-var-vals (vars sigma belief clause state)
  (if (null vars)
    '(t)
    (let ((r (dlet* (((label var h) (caar vars)))
	       (ndprog state nil
		 (nd-if (null (cdar vars))
		   (nd-for match match-cont (match-var-vals (cdr vars) sigma belief clause match-cont)
		     (success t)
		     )
		   (nd-loop (val (cdar vars)) 
		       (nd-if (if (kb-attr-p val)
				(and (unify var (attr-value val) (ac-clause clause) nil belief sigma)
				     (unify-time (attr-history val) h sigma (ac-clause clause) belief))
				(and (unify var (cadr val) (ac-clause clause) nil belief sigma)
				     (unify-time (caddr val) h sigma (ac-clause clause) belief))
				)
			 (nd-for match match-cont (match-var-vals (cdr vars) sigma belief clause match-cont)
			   (success t)
			   )
			 )
		     ))
		 ))))
      r)
    )
  )


;; ignore parents for the moment because assume all descriptions are tokens

(defun match-object-description (object description atom sigma state belief clause theory istate)
  (when (or (not (eq state :init))
	    (doloop (inst-pair (tok-desc-instance-of description))
	     :result t
	     :when (not (weak-type-instance object (car inst-pair) belief))
	     :return nil
	      ))
    (let* (vals
	   (result
	    (ndprog state (vars)
	      (nd-loop (attr-group (tok-desc-attrs description))
		  (setf vals (get-attr-value object (car attr-group) (theory-all-time theory) belief istate))
		(nd-loop (val-spec (cdr attr-group)) ; every spec must match
		    (nd-if vals
		      (push (cons val-spec vals) vars)
		      )
		  )
		)
	      (nd-if (null vars)
		(success t)
		(nd-for match match-cont (match-var-vals vars sigma belief clause state)
		  (success t)
		  )
		))
	    ))
      (if result
	(values t (cdr result) (list 'match-obj-desc))
	))
    ) 
  )


(defun match-desc-description (desc1 desc2 belief)
  nil
  )


; the description, if any, is considered the pattern

	 
(defresolve-s matches (atom sigma state belief clause theory istate)
  (if (eq state :init)
    (if (and (eq (car atom) 'not) (eq (caadr atom) 'matches))
    (dlet* (((obj1 obj2) (cdadr atom)))
      (if (unify-var-p obj1 (ac-clause clause) nil)
	(progn (setf (sigma-lookup obj1 sigma 0) obj2)
	       (values t nil (list 'matches)))
	(if (unify-var-p obj2 (ac-clause clause) nil)
	  (progn (setf (sigma-lookup obj2 sigma 0) obj1)
		 (values t nil (list 'matches)))
	  (if (token-description-p obj1)
	    (match-object-description obj2 obj1 atom sigma state belief clause theory istate)
	    (match-object-description obj1 obj2 atom sigma state belief clause theory istate)
	    )
	  ))
      ))
    (match-object-description nil nil atom sigma state belief clause theory istate)
    )
  )



(defun get-desc-attr-value (description class-label history belief istate)
  (doloop (attr-group (tok-desc-attrs description))
   :vars ((result nil))
   :return result
   :when (eq (car attr-group) class-label)
    (doloop (val-spec (cdr attr-group))
      (push val-spec result)
      )
    )
  )