/* Micro Quixote           */
/* Copyright (C) 1993 ICOT */
/* Written by gniibe       */

/* print routines */

#include <stdio.h>
#include <string.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "extern.h"

/* function prototype for debugging */
static void print_atom _P((MQ_Atom));
static void print_name_var _P((MQ_NameVar));
static void print_var _P((MQ_Var));
static void print_vterm _P((MQ_VTerm));
static void print_term _P((MQ_Term));
static void print_dot _P((MQ_Dot));
static void print_vterm_list _P((MQ_VTermList));
static void print_constraint _P((MQ_Constraint));
static void print_constraints_internal _P((MQ_Constraints));
static void print_rule _P((MQ_Rule));
static void print_var_name_list _P((MQ_VarNameList));
static void print_all_subrel_sub _P((MQ_SubRel));
static void print_all_rules_sub _P((MQ_RuleList));

static int var_number = 0;
static UnwindProtect up_print;

static
void print_atom (atom)
     MQ_Atom atom;
{
  int quote = 0;
  unsigned char *name;
  int c;

  if ((name = atom->name))
    {
      if ((c = *name++) == '&'
	  || (c >= 'a' && c <= 'z')
	  || (c >= '0' && c <= '9')
	  || c >= 0x80)
	while ((c = *name++))
	  {
	    if (!((c >= 'A' && c <= 'Z')
		  || (c >= 'a' && c <= 'z')
		  || (c >= '0' && c <= '9')
		  || c == '_'
		  || c >= 0x80))
	      {
		quote = 1;
		break;
	      }
	  }
      else
	quote = 1;
      if (quote)
	printf ("\"%s\"", atom->name);
      else
	printf ("%s", atom->name);
    }
  else
    {
      printf ("&nil");
    }
}

static
void print_name_var (name_var)
     MQ_NameVar name_var;
{
  if (name_var->number)
    printf ("_V%d", name_var->number);
  else
    printf ("%s", name_var->name);
}

void print_obj (obj)
     MQ_Obj obj;
{
  int i;

  print_atom (obj->atom);
  if (obj->arity)
    {
      printf ("[");
      i = 0;
      while (1)
	{
	  print_atom (obj->attr[i].label);
	  printf ("=");
	  print_vterm (obj->attr[i].vterm);
	  if (++i >= obj->arity)
	    break;
	  printf (",");
	}
      printf("]");
    }
}

static
void print_var (var)
     MQ_Var var;
{
  MQ_NameVar name_var;

  if (var->value)
    {
      print_vterm (var->value);
      return;
    }
  name_var = make_name_var (++var_number, NULL);
  bind (var, (MQ_VTerm)name_var);
  print_name_var (name_var);
}

static
void print_vterm (vterm)
     MQ_VTerm vterm;
{
  switch (vterm->type)
    {
    case TT_Obj:
      print_obj ((MQ_Obj) vterm);
      break;
    case TT_Var:
      print_var ((MQ_Var) vterm);
      break;
    case TT_NameVar:
      print_name_var ((MQ_NameVar) vterm);
      break;
    default:
      fatal ("something wrong in print_vterm\n");
      break;
    }
}

static
void print_term (term)
     MQ_Term term;
{
  switch (term->type)
    {
    case TT_NameVar:
      print_name_var ((MQ_NameVar)term);
      break;
    case TT_Obj:
      print_obj ((MQ_Obj)term);
      break;
    case TT_Var:
      print_var ((MQ_Var)term);
      break;
    case TT_Dot:
      print_dot ((MQ_Dot)term);
      break;
    default:
      fatal ("print_term\n");
      break;
    }
}

static
void print_dot (dot)
     MQ_Dot dot;
{
  print_vterm (dot->vterm);
  printf (".");
  print_atom (dot->label);
}

static
void print_vterm_list (vtl)
     MQ_VTermList vtl;
{
  if (vtl == NULL)
    return;
  print_vterm_list (vtl->next);
  if (vtl->next)
    printf (", ");
  print_vterm (vtl->vterm);
}

static
void print_constraint (cnstr)
     MQ_Constraint cnstr;
{
  switch (cnstr->rel)
    {
    case Subsumes:
    case SubsumesVarVar:
    case SubsumesVarObj:
    case SubsumesObjVar:
      print_term (cnstr->term);
      printf (">=");
      print_vterm (cnstr->vterm);
      break;

    case DotCongruent:
    case DotCongruentVar:
    case DotCongruentObj:
    case Congruent:
      print_term (cnstr->term);
      printf ("==");
      print_vterm (cnstr->vterm);
      break;

    case ExternalCnstr:
      print_vterm (((MQ_Obj)cnstr->vterm)->attr[0].vterm);
      printf ("#%s#", ((MQ_Obj)cnstr->vterm)->atom->name);
      print_vterm (((MQ_Obj)cnstr->vterm)->attr[1].vterm);
      break;

    case ExternalExpr:
      print_term (cnstr->term);
      printf ("##");
      print_vterm (cnstr->vterm);
      break;

    default:
      fatal ("print_constraint\n");
      break;
    }
}

static
void print_constraints_internal (cnstrs)
     MQ_Constraints cnstrs;
{
  if (cnstrs == mQ_void_cnstrs)
    return;
  print_constraints_internal (cnstrs->next);
  if (cnstrs->next != mQ_void_cnstrs)
    printf (", ");

  print_constraint (cnstrs->cnstr);
}

void print_constraints (cnstrs, condition)
     MQ_Constraints cnstrs;
     int condition;
{
  unsigned char *exec_obj;

  if (cnstrs == mQ_void_cnstrs)
    return;

  var_number = 0;
  up_print = up;
  exec_obj = (unsigned char *)obstack_alloc (mm_exec, 0);

  if (condition)
    printf (" if ");
  printf ("{ ");
  print_constraints_internal (cnstrs);
  printf (" }");
  unwind_variables (up_print);
  obstack_free (mm_exec, exec_obj);
}

/* Don't follow rule->next, it is chain for another purpose. */
static
void print_rule (rule)
     MQ_Rule rule;
{
  MQ_VarList vl;

  for (vl=rule->var_list; vl; vl = vl->next)
    vl->var->vterm_addr_list = NULL;
  print_vterm (rule->head);
  if (rule->head_cnstrs != mQ_void_cnstrs)
    {
      printf (" | { ");
      print_constraints_internal (rule->head_cnstrs);
      printf (" }");
    }

  if (rule->body)
    {
      printf (" <= ");
      print_vterm_list (rule->body);

      if (rule->body_cnstrs != mQ_void_cnstrs)
	{
	  printf (" || { ");
	  print_constraints_internal (rule->body_cnstrs);
	  printf (" }");
	}
    }

  printf (";;\n");
}

static
void print_all_subrel_sub (sl)
     MQ_SubRel sl;
{
  if (sl == NULL)
    return;
  print_all_subrel_sub (sl->next);
  print_atom (sl->a1);
  printf (" >= ");
  print_atom (sl->a2);
  printf (";;\n");
}

void print_all_subrel ()
{
  print_all_subrel_sub (subrel_list);
}

static
void print_all_rules_sub (rl)
     MQ_RuleList rl;
{
  while (rl)
    {
      print_rule (rl->rule);
      rl = rl->next;
    }
}

void print_all_rules ()
{
  unsigned char *exec_obj;

  up_print = up;
  exec_obj = (unsigned char *)obstack_alloc (mm_exec, 0);
  var_number = 0;
  print_all_rules_sub (rule_list->next);
  unwind_variables (up_print);
  obstack_free (mm_exec, exec_obj);
}

static
void print_var_name_list (vnl)
     MQ_VarNameList vnl;
{
  if (vnl == NULL)
    return;

  print_var_name_list (vnl->next);
  if (vnl->next)
    printf (", ");

  print_name_var (vnl->name_var);
  printf (" = ");
  switch (vnl->vterm->type)
    {
    case TT_Obj:
      print_obj ((MQ_Obj)vnl->vterm);
      break;

    case TT_NameVar:
      if (vnl->name_var == (MQ_NameVar)vnl->vterm)
	printf ("Unbound");
      else
	print_name_var ((MQ_NameVar)vnl->vterm);
      break;

    case TT_Var:
    default:
      fatal ("something wrong in print_name_var_list\n");
      break;
    }
}

void print_answer (query)
     MQ_Query query;
{
  MQ_VarNameList vnl;
  unsigned char *exec_obj;

  up_print = up;
  var_number = 0;
  exec_obj = (unsigned char *)obstack_alloc (mm_exec, 0);

  for (vnl=query->var_name_list; vnl; vnl=vnl->next)
    if (vnl->vterm->type == TT_Var)
      {
	MQ_Var v;

	v = (MQ_Var)vnl->vterm;
	bind (v, (MQ_VTerm)vnl->name_var);
      }

  print_var_name_list (query->var_name_list);
  if (cnstrs_asmpts->dot_cnstrs != mQ_void_cnstrs
      || cnstrs_asmpts->sub_cnstrs != mQ_void_cnstrs
      || cnstrs_asmpts->ext_cnstrs != mQ_void_cnstrs)
    {
      printf (" { ");
      print_constraints_internal (cnstrs_asmpts->dot_cnstrs);
      if (cnstrs_asmpts->dot_cnstrs != mQ_void_cnstrs
	  && (cnstrs_asmpts->sub_cnstrs != mQ_void_cnstrs
	      || cnstrs_asmpts->ext_cnstrs != mQ_void_cnstrs))
	printf (", ");
      print_constraints_internal (cnstrs_asmpts->sub_cnstrs);
      if (cnstrs_asmpts->sub_cnstrs != mQ_void_cnstrs
	  && cnstrs_asmpts->ext_cnstrs != mQ_void_cnstrs)
	printf (", ");
      print_constraints_internal (cnstrs_asmpts->ext_cnstrs);
      printf (" }");
    }
  if (cnstrs_asmpts->dot_asmpts != mQ_void_cnstrs
      || cnstrs_asmpts->sub_asmpts != mQ_void_cnstrs
      || cnstrs_asmpts->ext_asmpts != mQ_void_cnstrs)
    {
      printf (" if { ");
      print_constraints_internal (cnstrs_asmpts->dot_asmpts);
      if (cnstrs_asmpts->dot_asmpts != mQ_void_cnstrs
	  && (cnstrs_asmpts->sub_asmpts != mQ_void_cnstrs
	      || cnstrs_asmpts->ext_asmpts != mQ_void_cnstrs))
	printf (", ");
      print_constraints_internal (cnstrs_asmpts->sub_asmpts);
      if (cnstrs_asmpts->sub_asmpts != mQ_void_cnstrs
	  && cnstrs_asmpts->ext_asmpts != mQ_void_cnstrs)
	printf (", ");
      print_constraints_internal (cnstrs_asmpts->ext_asmpts);
      printf (" }");
    }
  printf ("\n");
  fflush (stdout);
  unwind_variables (up_print);
  obstack_free (mm_exec, exec_obj);
}

void print_goal (g)
     MQ_Goal g;
{
  MQ_Goal sg;
  unsigned char *exec_obj;

  var_number = 0;
  up_print = up;
  exec_obj = (unsigned char *)obstack_alloc (mm_exec, 0);
  if (g->goal_vterm)
    print_vterm (g->goal_vterm);
  else
    printf ("()\n");
  if (g->head_cnstrs != mQ_void_cnstrs)
    {
      printf (" |{");
      print_constraints_internal (g->head_cnstrs);
      printf ("}");
    }
  sg = g->subgoal;
  if (sg)
    {
      printf (" <= ");
      while (1)
	{
	  print_vterm (sg->goal_vterm);
	  sg = sg->next;
	  if (sg == NULL)
	    break;
	  printf (", ");
	}
    }

  if (g->body_cnstrs != mQ_void_cnstrs)
    {
      printf ("||{");
      print_constraints_internal (g->body_cnstrs);
      printf ("}");
    }
  printf (";;\n");
  unwind_variables (up_print);
  obstack_free (mm_exec, exec_obj);
}
