/* Micro Quixote           */
/* Copyright (C) 1993 ICOT */
/* Written by gniibe       */

/* Parser for the language `Micro Quixote' */

%{
#include <stdio.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "tree.h"
#include "extern.h"
#include "emit.h"

static MQT_Constraints head_cnstrs;
static MQ_Query query;
int mq_mode;
int error_recovering;
int reading_object;
MQT_Obj input_object;
%}
%start quixote
%pure_parser
%union { 
  int i;
  AttrRel       ar;
  Rel           sr;
  MQ_Atom       atom;
  MQT_AtomList  atom_list;

  MQT_Obj       obj;
  MQT_Dot       dot;
  MQT_Var       variable;

  MQT_Term      term;
  MQT_VTerm     vterm;

  MQT_VTermList	vterm_list;

  MQT_AttrList    attr_list;
  MQT_Constraints cnstrs;
}

%token ERROR_TOKEN

%token CLEAR LOAD QUERY QUIT SET SHOW
%token LEX_TRUE
%token PROGRAM RULE SUBSUMPTION END

%token <atom>     ATOM
%token <variable> VARIABLE

%token DOT
%token TERMINATER
%token SOLVE COMMITS CONSTRAINED

%token <sr>  SUBREL
%token <ar>  ATTRREL

%type <atom_list> atom_list
%type <vterm_list> body a_term_list
%type <attr_list> attribute_list
%type <vterm> vterm a_term
%type <term> term
%type <obj>  object

%type <dot>  dot_term
%{
extern int yylex _P((YYSTYPE *));
%}
%%

quixote:
	 /* empty */
	| quixote program_query_command
	| object
		{
		  if (!reading_object)
		    YYERROR;
		  input_object = $1;
		}
	;

program_query_command:
	  program
	| query
	| command
	;

program:
	program_def '.'
		{ mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	| error '.'
		{
		  free_parse_tree ();
		  yyerrok;
		  error_recovering = 0;
		  mq_mode = MODE_COMMAND;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

program_def:
	PROGRAM TERMINATER
		{ mq_mode = MODE_PROGRAM; last_char_is_newline = TRUE; }
	definition_seq END
		{
		  mq_mode = PS_EXPECT_PRD;
		  generate_lattice ();
		}
	;

definition_seq:
	/* empty */
	| definition_seq definition
	| error TERMINATER
		{
		  free_parse_tree ();
		  yyerrok;
		  error_recovering = 0;
		  mq_mode = MODE_PROGRAM;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

definition:
	SUBSUMPTION TERMINATER
		{ mq_mode = MODE_PROGRAM_SUBSUMPTION;
		  last_char_is_newline = TRUE;}
	  subsumption_seq
	| RULE TERMINATER
		{ mq_mode = MODE_PROGRAM_RULE;
		  last_char_is_newline = TRUE;}
	  rule_desc_seq
	;

subsumption_seq:
	/* empty */
	| subsumption_seq subsumption
		{ free_parse_tree (); }
	;

rule_desc_seq:
	/* empty */
	| rule_desc_seq rule
		{ free_parse_tree (); }
	;

atom_list:
	ATOM
		{ $$ = makeT_atom_list ($1, NULL); }
	| atom_list ',' ATOM
		{ $$ = makeT_atom_list ($3, $1); }
	;

subsumption:
	ATOM SUBREL ATOM TERMINATER
		{ emit_subrel ($2, $1, $3);
		  mq_mode = MODE_PROGRAM_SUBSUMPTION;
		  last_char_is_newline = TRUE; }
	| ATOM SUBREL '{' atom_list '}' TERMINATER
		{ MQT_AtomList al;

		  for (al = $4; al; al = al->next)
		    emit_subrel ($2, $1, al->atom);
		  mq_mode = MODE_PROGRAM_SUBSUMPTION;
		  last_char_is_newline = TRUE;
		}
	| error TERMINATER
		{
		  yyerrok;
		  error_recovering = 0;
		  mq_mode = MODE_PROGRAM_SUBSUMPTION;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

rule:	{ current_cnstrs = NULL; } clause
		{
		  mq_mode = MODE_PROGRAM_RULE;
		  last_char_is_newline = TRUE;
		}
	| error TERMINATER
		{
		  yyerrok;
		  error_recovering = 0;
		  mq_mode = MODE_PROGRAM_RULE;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

clause: a_term TERMINATER
		{ emit_rule ($1, current_cnstrs, NULL, NULL); }
	| a_term COMMITS
		{ head_cnstrs = current_cnstrs; current_cnstrs = NULL; }
	  body TERMINATER
		{ emit_rule ($1, head_cnstrs, $4, current_cnstrs); }
	;

body:
	a_term_list
	| a_term_list CONSTRAINED '{' constraint_list '}'
	;

a_term_list:
	a_term
		{ $$ = makeT_vterm_list ($1, NULL); }
	| a_term_list ',' a_term
		{ $$ = makeT_vterm_list ($3, $1); }
	;

a_term:
	vterm
		{ $$ = $1; }
	| vterm '/' '[' attribute_list ']'
		{
		  MQT_AttrList atl;

		  for (atl = $4; atl; atl = atl->next)
		    {
		      MQT_Dot dot;

		      dot = makeT_dot ((MQT_Term)$1, atl->label);
		      current_cnstrs
			= makeT_constraints (Congruent, (MQT_Term)dot,
					     (MQT_Term)atl->vterm, NULL,
					     current_cnstrs);
		    }
		  $$ = $1;
		}
	| vterm '|' '{' constraint_list '}'
		{ $$ = $1; }
	| vterm '/' '[' attribute_list ']' '|' '{' constraint_list '}'
		{
		  MQT_AttrList atl;

		  for (atl = $4; atl; atl = atl->next)
		    {
		      MQT_Dot dot;

		      dot = makeT_dot ((MQT_Term)$1, atl->label);
		      current_cnstrs
			= makeT_constraints (Congruent, (MQT_Term)dot,
					     (MQT_Term)atl->vterm, NULL,
					     current_cnstrs);
		    }
		  $$ = $1;
		}
	;

attribute_list:
	ATOM ATTRREL term
		{ $$ = makeT_attr_list ($2, $1, (MQT_VTerm)$3, NULL); }
	| attribute_list ',' ATOM ATTRREL term
		{ $$ = makeT_attr_list ($4, $3, (MQT_VTerm)$5, $1); }

constraint_list:
	term SUBREL term
		{
		  MQ_Atom op;

		  if ($2 == ExternalCnstr)
		    op = external_cnstr_op;
		  else
		    op = NULL;
		  current_cnstrs
		    = makeT_constraints ($2, $1, $3, op, current_cnstrs);
		}
	| constraint_list ',' term SUBREL term
		{
		  MQ_Atom op;

		  if ($4 == ExternalCnstr)
		    op = external_cnstr_op;
		  else
		    op = NULL;
		  current_cnstrs =
		    makeT_constraints ($4, $3, $5, op, current_cnstrs);
		}
	;

vterm:
	object
		{ $$ = (MQT_VTerm) $1; }
	| VARIABLE
		{ $$ = (MQT_VTerm) $1; }
	;

term:
	object
		{ $$ = (MQT_Term) $1; }
	| dot_term
		{ $$ = (MQT_Term) $1; }
	| VARIABLE
		{ $$ = (MQT_Term) $1; }
	;

object:
	ATOM
		{ $$ = makeT_object ($1, NULL); }
	| ATOM '[' attribute_list ']'
		{ $$ = makeT_object ($1, $3); }
	| LEX_TRUE
		{ $$ = mqTO_True; }
	;

dot_term:
	term DOT ATOM
		{ $$ = makeT_dot ($1, $3); }
	;

query:
	SOLVE
		{
		  mq_mode = PS_EXPECT_Q_PRD;
		  current_cnstrs = NULL;
		}
	  query_body_period_q
	| QUERY TERMINATER
		{ mq_mode = MODE_QUERY; last_char_is_newline = TRUE; }
	  query_with_no_solve_seq END '.'
		{ mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	;

query_with_no_solve_seq:
	/* empty */
	| query_with_no_solve_seq
		{ current_cnstrs = NULL; }
	  query_body_period
		{ mq_mode = MODE_QUERY; last_char_is_newline = TRUE; }
	;

query_body_period:
	body '.'
		{
		  begin_exec ();
		  emit_body ($1, current_cnstrs, &query);
		  free_parse_tree ();
		  do_query (query);
		  end_exec ();
		}
	| body TERMINATER
		{
		  begin_exec ();
		  begin_tmp ();
		  emit_body ($1, current_cnstrs, &query);
		  free_parse_tree ();
		  suspend_exec ();
		}
	  program_def '.'
		{  
		  resume_exec ();
		  do_query (query);
		  end_exec ();
		  end_tmp ();
		}
	| error '.'
		{
		  free_parse_tree ();
		  yyerrok;
		  error_recovering = 0;
		  end_exec ();
		  if (temporary)
		    end_tmp ();
		  mq_mode = MODE_QUERY;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

query_body_period_q:
	body '.'
		{
		  begin_exec ();
		  emit_body ($1, current_cnstrs, &query);
		  free_parse_tree ();
		  do_query (query);
		  end_exec ();
		  mq_mode = MODE_COMMAND;
		  last_char_is_newline = TRUE;
		}
	| body TERMINATER
		{
		  begin_exec ();
		  begin_tmp ();
		  emit_body ($1, current_cnstrs, &query);
		  free_parse_tree ();
		  suspend_exec ();
		}
	  program_def '.'
		{  
		  resume_exec ();
		  do_query (query);
		  end_exec ();
		  end_tmp ();
		  mq_mode = MODE_COMMAND;
		  last_char_is_newline = TRUE;
		}
	| error '.'
		{
		  free_parse_tree ();
		  yyerrok;
		  error_recovering = 0;
		  end_exec ();
		  if (temporary)
		    end_tmp ();
		  mq_mode = MODE_COMMAND;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

command:
	set_command
	| load_command
	| clear_command
	| show_command
	| quit_command
	| error TERMINATER
		{
		  yyerrok;
		  error_recovering = 0;
		  mq_mode = MODE_COMMAND;
		  last_char_is_newline = TRUE;
		  output_prompt ();
		}
	;

set_command:
	SET  TERMINATER
		{ cmd_set (NULL, NULL);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	| SET ATOM  TERMINATER
		{ cmd_set ($2, NULL);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	| SET ATOM ATOM  TERMINATER
		{ cmd_set ($2, $3);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	;

load_command:
	LOAD ATOM  TERMINATER
		{ cmd_load ($2);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	;

clear_command:
	CLEAR  TERMINATER
		{ cmd_clear (NULL);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	| CLEAR ATOM  TERMINATER
		{ cmd_clear ($2);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	;

show_command:
	SHOW  TERMINATER
		{ cmd_show (NULL);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	| SHOW ATOM  TERMINATER
		{ cmd_show ($2);
		  mq_mode = MODE_COMMAND; last_char_is_newline = TRUE; }
	;

quit_command:
	QUIT  TERMINATER
		{ cmd_quit (); }
	;
%%
int yyerror (message)
     char *message;
{
  error_recovering = 1;
  if (interactive)
    fprintf (stderr, " %s\n", message);
  else
    fprintf (stderr, "%d: %s\n", lineno, message);
  return 0;
}
