/*
 *	(C)1993 Institute for New Generation Computer Technology
 *	Read COPYRIGHT for detailed information.
 *
 *
 *	y.y	---	Grammar for DP-System.
 *
 */

%{
#include	<stdio.h>
#include	<ctype.h>

#include	"define.h"
#include	"typedef.h"
#include	"global.h"
#include	"proto.h"

#pragma segment	parser
%}

%union {
  int      num;		/* numbers                  */
  char     *name;	/* variables or identifiers */

  varblrec *expr;	/* expressions              */
  pphandle *ltrl;	/* atomic constraints       */
  litrlrec *ftrlist;	/* list of features         */
  arglst   *argmts;	/* list of arguments        */
  double   coeff;	/* relevance coefficient    */
  int      sign;	/* sign                     */
}

%token <num>   INT_NUM
%token <name>  VARIABLE IDENTIFIER
%token <coeff> REAL_NUM

%type <expr> expression elem_list
%type <ltrl> binding feature_spec atomic_formula
             proper_atomic_constraint literal_list
             literal feature_pair
%type <ftrlist> feature_list
%type <argmts>  argument_list
%type <sign>    sign clause

%%

clause :
  /* empty */
{ $$ = NULL;                                                 }
| literal_list ':' REAL_NUM '/' REAL_NUM
{ $$ = make_clause($1, $3,           $5);                    }
| literal_list ':' REAL_NUM
{ $$ = make_clause($1, $3,           EXCL_DEFAULT);          }
| literal_list ':'          '/' REAL_NUM
{ $$ = make_clause($1, DISJ_DEFAULT, $4);                    }
| literal_list ':'
{ $$ = make_clause($1, DISJ_DEFAULT, EXCL_DEFAULT);          }


literal_list :
  literal                  { $$ = check_literal(NULL, $1, FALSE); }
| literal '$'              { $$ = check_literal(NULL, $1, TRUE);  }
| literal_list literal     { $$ = check_literal($1, $2, FALSE);   }
| literal_list literal '$' { $$ = check_literal($1, $2, TRUE);    }


literal :
  sign REAL_NUM proper_atomic_constraint
{ $$ = set_rel_coeff($3, $1, $2);               }
| sign proper_atomic_constraint
{ $$ = set_rel_coeff($2, $1, REL_DEFAULT);      }
| sign REAL_NUM VARIABLE '=' '=' VARIABLE
{ $$ = make_equation($1, $2, $3, $6);                 }
| sign VARIABLE '=' '=' VARIABLE
{ $$ = make_equation($1, EQUAL_SUBS_DEFAULT, $2, $5); }


proper_atomic_constraint :
  atomic_formula { if ($1 != NULL && $1->body->tag == FUNCTION)
		     $1->body->tag = CONSTRAINT;
		   $$ = $1;                                }
| binding        { $$ = $1;                                }
| feature_spec   { $$ = $1;                                }


atomic_formula :
  IDENTIFIER '(' argument_list ')' { $$ = make_formula($1, $3);                  }
| error '(' argument_list ')'      { yyerror(": at function/constraint name\n");
				     return 1;                                   }
| IDENTIFIER '(' error ')'         { yyerror(": in argument list.\n");
				     return 1;                                   }


argument_list :
  /* empty */                  { $$ = NULL;                 }
| expression                   { $$ = chain_args($1, NULL); }
| expression ',' argument_list { $$ = chain_args($1, $3);   }


binding :
  VARIABLE '=' expression { if ($3->paren == NULL) {
                              yyerror(" syntax error : atomic formula is required.\n");
			      return 1;
			    } else
			      $$ = make_binding($1, $3, TRUE);        }
| error '=' expression    { yyerror(": at left-hand of binding.\n");
			    return 1;                                 }
| VARIABLE '=' error      { yyerror(": at right-hand of binding.\n");
			    return 1;                                 }


feature_spec :
  '#' IDENTIFIER '(' VARIABLE ')' '=' expression
{ $$ = add_feature($4, $2, $7);                                        }
| error IDENTIFIER '(' VARIABLE ')' '=' expression
{ yywarning("# is omitted?\n"); $$ = add_feature($4, $2, $7); yyerrok; }
| '#' error '(' VARIABLE ')' '=' expression
{ yyerror(": feature name is required\n"); return 1;                   }
| '#' IDENTIFIER '(' error ')' '=' expression
{ yyerror(": variable name is required\n"); return 1;                  }
| '#' IDENTIFIER '(' VARIABLE ')' '=' error
{ yyerror(": at right-hand of f-spec.\n"); return 1;                   }

/*
 *    Note: Following program is accepted but recognized in the `wrong' way.
 *
 *		foo(X) -#aaa(X)=bbb -X={ccc/ddd, eee/fff} :.
 *		( Variable X doesn't have feature 'ccc' nor 'eee'.)
 */


feature_list :
  feature_pair                  { $$ = chain_feature(NULL, $1->body); }
| feature_list ',' feature_pair { $$ = chain_feature($1, $3->body);   }


feature_pair :
  IDENTIFIER '/' expression { $$ = add_feature("", $1, $3);            }
| error      '/' expression { yyerror(": feature name is required\n");
			      return 1;                                }


elem_list :
  /* empty */              { $$ = sym_node(NIL_NAME);                }
| expression               { $$ = cons_node($1, sym_node(NIL_NAME)); }
| expression ',' elem_list { $$ = cons_node($1, $3);                 }


expression :
  INT_NUM                           { $$ = num_node($1);                  }
| IDENTIFIER                        { $$ = sym_node($1);                  }
| VARIABLE                          { $$ = var_node($1);                  }
| atomic_formula                    { $$ = fnc_node($1);                  }
| '{' feature_list '}'              { $$ = pst_node($2);                  }
| '{' error '}'                     { yyerror(": in PST.\n"); return 1;   }
| '[' elem_list ']'                 { $$ = $2;                            }
| '[' error ']'                     { yyerror(": in list.\n"); return 1;  }
| '[' expression '|' expression ']' { $$ = cons_node($2, $4);             }


sign :
  /* empty */ { $$ = UNSIGNED; }
| '+'         { $$ = POSITIVE; }
| '-'         { $$ = NEGATIVE; }


/* End of grammar */
%%

int yylex()
{
  int c;

  /* skip white space and comments */
  while (1) {
    do {
      c = fgetc(Gparams.infile);
    } while (c == ' ' || c == '\t' || c == '\n');
    if (c == '%') {
      do {
	c = fgetc(Gparams.infile);
      } while (c != '\n' && c != EOF);
      if (c == EOF)
	return 0;
    } else if (c == EOF) {
      return 0;
    } else {
      break;
    }
  }

  /* Char starts a period and a number => parse the real number.	*/
  if (c == '.') {
    int d;

    d = fgetc(Gparams.infile);
    ungetc(d, Gparams.infile);
    if (isdigit(d)) {
      ungetc(c, Gparams.infile);
      fscanf(Gparams.infile, "%lf", &yylval.coeff);
      return REAL_NUM;
    }
  }

  /* Char starts a number => determine if int or real number.	*/
  if (isdigit(c)) {
    double temp;

    for (temp = 0.0; isdigit(c); c = fgetc(Gparams.infile))
      temp = temp * 10.0 + (double)(c - '0');
    ungetc(c, Gparams.infile);
    if (c == '.') {
      fscanf(Gparams.infile, "%lf", &yylval.coeff);
      yylval.coeff += temp;
      return REAL_NUM;
    } else {
      yylval.num = (int)temp;
      return INT_NUM;
    }
  }

  /* Char starts an alphabet => read the name. */
  if (isalpha(c) || c == '@' || c == '!') {
    char *symbuf = 0;
    int  length = 15;
    int  i;

    /* Initially make the buffer long enough for a 15-char. symbol name. */
    symbuf = (char *) malloc(length+1);

    i = 0;
    do {
      /* If buffer is full, make it bigger.	*/
      if (i == length) {
	length *= 2;
	symbuf = (char *) realloc(symbuf, length+1);
      }
      /* Add this character to the buffer.	*/
      symbuf[i++] = c;
      /* Get another character.			*/
      c = fgetc(Gparams.infile);
    } while (c != EOF && (isalnum(c) || c == '_' || c == '-'));

    ungetc(c, Gparams.infile);
    symbuf[i] = '\0';

    yylval.name = symbuf;
    return (isupper(symbuf[0]) ? VARIABLE : IDENTIFIER);
  }

  /* Char starts '_' => place folder (generate new variable name) */
  if (c == '_') {
    yylval.name = new_name('_');
    return VARIABLE;
  }

  /* Any other character is a token by itself except for ';'.	*/
  return (c == '.' ? 0 : c);
}


int yyerror(s)
     char *s;
{
  printf(" %s", s);
  return 1;
}


int yywarning(s)
     char *s;
{
  printf("%s", s);
  return 1;
}
