/*
 *	(C)1993 Institute for New Generation Computer Technology
 *	Read COPYRIGHT for detailed information.
 *
 *
 *	backp.c	---	Backpropagation manager.
 *
 */

#include	<stdio.h>

#define	PROTO_BACKP_C
#include	"config.h"
#include	"define.h"
#include	"typedef.h"
#include	"global.h"
#include	"proto.h"
#include	"debug.h"
#undef	PROTO_BACKP_C

#pragma segment	spread


linkrec *choose_link(maxdir, maxest)
     int *maxdir;
     double *maxest;
{
  litrlrec *sltrl, *ultrl;
  linkrec *link, *maxlnk;
  double assm1, assm2, cmpl1, cmpl2;
  double penalty;

  Gparams.cnvrgdp = FALSE;
  backpropagation(Gparams.itrmax);
  calc_aform_importance();
  *maxdir = -1;
  *maxest = 0.0;
  penalty = 0.0;
  maxlnk = NULL;
  for (link = Gcontrol.links; link != NULL; link = link->ctrl.nxt) {
    if (link->tag != INFERENCE)
      continue;
    link->dsub = 0.0;
    if ((link->ptr[0]->joint->ltrl->tag == CONSTRAINT &&
	 strcmp(link->ptr[0]->joint->ltrl->body.afm.name, "true") == 0) ||
	(link->ptr[1]->joint->ltrl->tag == CONSTRAINT &&
	 strcmp(link->ptr[1]->joint->ltrl->body.afm.name, "true") == 0)) {
      link->sub = 1.0;

#ifndef PROFILE
      if (link->ptr[0]->joint->ltrl->pol == POSITIVE)
	penalty = 1.0 - link->ptr[0]->joint->ltrl->act;
      else
	penalty = 1.0 - link->ptr[1]->joint->ltrl->act;
#endif

      continue;
    }
    aform_assimilation_parameter(&assm1, link);
    link_assimilation_parameter(&assm2, link);
    link->dsub -= Gparams.assmWT * (assm1 + assm2) / Gparams.tmprtr;
    if (link->ptr[0]->joint->ltrl->pol == UNSIGNED ||
	link->ptr[1]->joint->ltrl->pol == COST_POSITIVE ||
	link->ptr[1]->joint->ltrl->pol == COST_NEGATIVE) {
      sltrl = link->ptr[1]->joint->ltrl;
      ultrl = link->ptr[0]->joint->ltrl;
      signed_completion_parameter(&cmpl1, sltrl, ultrl);
      unsigned_completion_parameter(&cmpl2, sltrl, ultrl);
      link->dsub -= Gparams.cmplWT * (cmpl1 + cmpl2) / Gparams.tmprtr;
    } else if (link->ptr[1]->joint->ltrl->pol == UNSIGNED ||
	       link->ptr[0]->joint->ltrl->pol == COST_POSITIVE ||
	       link->ptr[0]->joint->ltrl->pol == COST_NEGATIVE) {
      sltrl = link->ptr[0]->joint->ltrl;
      ultrl = link->ptr[1]->joint->ltrl;
      signed_completion_parameter(&cmpl1, sltrl, ultrl);
      unsigned_completion_parameter(&cmpl2, sltrl, ultrl);
      link->dsub -= Gparams.cmplWT * (cmpl1 + cmpl2) / Gparams.tmprtr;
    }

    /*
     * choose_link() does NOT pick up...
     *
     * (1) strong unification links (They are handled in choose_probe().)
     * (2) links not connected to the subsumed literal.
     * (3) links whose subsumption coefficient is fixed to 1
     * (4) links connecting between binding or feature spec.
     *     (not in case of similation of speech recognition)
     * (5) links which directs directly to the top clause.
     */
    if (maxlnk == NULL || fabs(link->dsub) > *maxest) {
      if (link->ptr[0]->joint->total || link->ptr[1]->joint->total)
	continue;
      else if (! link->ptr[0]->joint->ltrl->sbsm &&
	       ! link->ptr[1]->joint->ltrl->sbsm)
	continue;
      else if (link->sub >= 1.0)
	continue;
      else if (Gparams.option != SPEECH_RECOG &&
	       (link->ptr[0]->joint->ltrl->tag == FUNCTION ||
		link->ptr[0]->joint->ltrl->tag == FEATURE))
	continue;
      else if (link->ptr[1]->joint->ltrl->pphndl->top &&
	       link->ptr[0]->joint->ltrl->pphndl->top)
	continue;
      /*
       * Decide direction
       */
      *maxest = fabs(link->dsub);
      maxlnk = link;
      if (link->ptr[0]->joint->ltrl->sbsm &&
	  ! link->ptr[1]->joint->ltrl->sbsm)
	*maxdir = 1;
      else if (link->ptr[1]->joint->ltrl->sbsm &&
	       ! link->ptr[0]->joint->ltrl->sbsm)
	*maxdir = 0;
      else if (! link->ptr[0]->joint->ltrl->pphndl->top &&
	       link->ptr[1]->joint->ltrl->pphndl->top)
	*maxdir = 0;
      else if (! link->ptr[1]->joint->ltrl->pphndl->top &&
	       link->ptr[0]->joint->ltrl->pphndl->top)
	*maxdir = 1;
      else
	*maxdir = -1;
    }
  }

#ifndef PROFILE
  printf("Penalty function = %14.11lf\n\n", penalty);
#endif

  return maxlnk;
}


/*
 *   `link' and `dir' specify ab.
 *   Result = z_a x_a(1-x_a) d^2 E_assm / dx_a ds_ab
 *          + z_b x_b(1-x_b) d^2 E_assm / dx_b ds_ab
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_a       = - A s_ab (nDXDx_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   ==> d^2 E_assm / dx_a ds_ab = - A (nDXDx_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *
 *   Result = - A { z_a x_a(1-x_a) (nDXDx_a) (X_b - S(dP/ds_b))
 *                + z_b x_b(1-x_b) (nDXDx_b) (X_a - S(dP/ds_a)) } Pi X_e_ab
 */
void aform_assimilation_parameter(res, link)
     double *res;
     linkrec *link;
{
  litrlrec *ltrl0, *ltrl1;
  double delta;
  int i;

  *res = 0.0;
  ltrl0 = link->ptr[0]->joint->ltrl;
  ltrl1 = link->ptr[1]->joint->ltrl;
  for (delta = 1.0, i = 0; i < link->n; i++)
    delta *= nAct(link->coeff[i].body.act);

  *res += ltrl0->z * ltrl0->act*(1.0-ltrl0->act)
    * nDXDx(ltrl0->act) * (nAct(ltrl1->act) - negP(ltrl1->imp));
  *res += ltrl1->z * ltrl1->act*(1.0-ltrl1->act)
    * nDXDx(ltrl1->act) * (nAct(ltrl0->act) - negP(ltrl0->imp));

  *res *= - ltrl0->coeff.assm * link->confuse * delta;
}


/*
 *   `link' and `dir' specify ab.
 *   Result = Sigma_j z_e_j x_e_j(1-x_e_j) d^2 E_assm / dx_e_j ds_ab
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_e_k
 *       = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                                     (nDXDx_e_k) Pi_{d!=k} X_e_ab_d
 *   ==> d^2 E_assm / dx_e_k ds_ab
 *       = - A (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                                (nDXDx_e_k) Pi_{d!=k} X_e_ab_d
 *
 *   Result = - A (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                Sigma_j z_j x_j(1-x_j) (nDXDx_e_j) Pi_{i!=j} X_e_i
 */
void link_assimilation_parameter(res, link)
     double *res;
     linkrec *link;
{
  litrlrec *ltrl0, *ltrl1;
  double delta;
  int i, j;

  *res = 0.0;
  ltrl0 = link->ptr[0]->joint->ltrl;
  ltrl1 = link->ptr[1]->joint->ltrl;
  for (j = 0; j < link->n; j++) {
    for (delta = 1.0, i = 0; i < link->n; i++)
      if (i != j)
	delta *= nAct(link->coeff[i].body.act);
    *res += link->coeff[j].body.z
      * link->coeff[j].body.act*(1.0-link->coeff[j].body.act)
	* nDXDx(link->coeff[j].body.act) * delta;
  }

  *res *= - ltrl0->coeff.assm * link->confuse
    * (nAct(ltrl0->act) - negP(ltrl0->imp))
      * (nAct(ltrl1->act) - negP(ltrl1->imp));
}


/*
 *   `sliteral' and `uliteral' specify i and l, respectively.
 *   Result = z_i x_i(1-x_i) d^2 E_cmpl / dx_i ds_il
 *          + Sigma_k z_k x_k(1-x_k) d^2 E_cmpl / dx_k ds_il
 *   ------------------------------------------------------------
 *   E_cmpl = C X_i Pi_{i@j} {1 - s_ij X_j}
 *   --> (1) d   E_cmpl / dx_i     = C (nDXDx_i) Pi_{i@j} (1 - s_ij X_j)
 *       (2) d   E_cmpl / dx_k     =
 *                   - C X_i s_ik (nDXDx_k) Pi_{i@j,j!=k} (1 - s_ij X_j)
 *   ==> (1) d^2 E_cmpl / dx_i s_il =
 *                     - C (nDXDx_i) X_l Pi_{i@j,j!=l} (1 - s_ij X_j)
 *       (2) d^2 E_cmpl / dx_k s_ik =
 *                     - C X_i (nDXDx_k) Pi_{i@j,j!=k} (1 - s_ij X_j)
 *       (3) d^2 E_cmpl / dx_k s_il =
 *                     C X_i s_ik (nDXDx_k) X_l Pi_{i@j,j!=k,l} (1 - s_ij X_j)
 *
 *   Result of case (1) = z_i x_i(1-x_i) d^2 E_cmpl / dx_i ds_il
 *          = - z_i x_i(1-x_i) C (nDXDx_i) X_l Pi_{i@j,j!=l} {1 - s_ij X_j}
 *
 *   *NOTICE*
 *   x_l and x_j are not activation values of LITERALS.
 *   This routine calculates POSITIVE completion parameter.
 */
void signed_completion_parameter(res, sliteral, uliteral)
     double *res;
     litrlrec *sliteral, *uliteral;
{
  litrlrec *ultrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;

  *res = 1.0;
  for (jnt = sliteral->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ultrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      if (ultrl != uliteral)
	*res *= 1.0 - lnk->sub * lnk->confuse * nAct(ultrl->act);
    }
  *res *= - sliteral->z * sliteral->act*(1.0-sliteral->act)
    * sliteral->coeff.cmpl * nDXDx(sliteral->act) * nAct(uliteral->act);
}


/*
 *   sliteral and uliteral specify `i' and `l', respectively.
 *   Result = z_i x_i(1-x_i) d^2 E_cmpl / dx_i ds_il
 *          + Sigma_k z_k x_k(1-x_k) d^2 E_cmpl / dx_k ds_il
 *   ------------------------------------------------------------
 *   E_cmpl = C X_i Pi_{i@j} {1 - s_ij X_j}
 *   --> (1) d   E_cmpl / dx_i     = C (nDXDx_i) Pi_{i@j} (1 - s_ij X_j)
 *       (2) d   E_cmpl / dx_k     =
 *                     - C X_i s_ik (nDXDx_k) Pi_{i@j,j!=k} (1 - s_ij X_j)
 *   ==> (1) d^2 E_cmpl / dx_i s_il =
 *                     - C (nDXDx_i) X_l Pi_{i@j,j!=l} (1 - s_ij X_j)
 *       (2) d^2 E_cmpl / dx_k s_ik = 
 *                     - C X_i (nDXDx_k) Pi_{i@j,j!=k} (1 - s_ij X_j)
 *       (3) d^2 E_cmpl / dx_k s_il =
 *                     C X_i s_ik (nDXDx_k) X_l Pi_{i@j,j!=k,l} (1 - s_ij X_j)
 *
 *   Result of case (2),(3) = Sigma_k z_k x_k(1-x_k) d^2 E_cmpl / dx_k ds_il
 *          = C X_i Sigma_k z_k x_k(1-x_k) (nDXDx_k) *
 *             (2) - Pi_{i@j,j!=k} (1 - s_ij X_j)           (k == l)
 *             (3) s_ik X_l Pi_{i@j,j!=k,l} (1 - s_ij X_j)  (k != l)
 *
 *   *NOTICE*
 *   x_l and x_j are not activation values of LITERALS.
 *   This routine calculates POSITIVE completion parameter.
 */
void unsigned_completion_parameter(res, sliteral, uliteral)
     double *res;
     litrlrec *sliteral, *uliteral;
{
  litrlrec *ultrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;
  double df, temp;

  *res = 0.0;
  for (jnt = sliteral->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ultrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      if (ultrl == uliteral) {
	/*=== case (2) ===*/
	completion_parameter_aux(&df, sliteral, ultrl, NULL);
	temp = - df;
      } else {
	/*=== case (3) ===*/
	completion_parameter_aux(&df, sliteral, ultrl, uliteral);
	temp = lnk->sub * lnk->confuse * nAct(uliteral->act) * df;
      }
      *res += ultrl->z * ultrl->act * (1.0-ultrl->act)
	* nDXDx(ultrl->act) * temp;
    }
  *res *= sliteral->coeff.cmpl * nAct(sliteral->act);
}


/*
 *   `sliteral', `uliteral1' and `uliteral2' specify k, l and m,
 *   respectively.
 *   Result = Pi_{k@j,j!=l,m} {1 - s_ij X_j}
 *
 *   *NOTICE*
 *   x_j is not activation value of LITERAL.
 *   This routine calculates POSITIVE completion parameter (aux).
 */
void completion_parameter_aux(res, sliteral, uliteral1, uliteral2)
     double *res;
     litrlrec *sliteral, *uliteral1, *uliteral2;
{
  litrlrec *ultrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;

  *res = 1.0;
  for (jnt = sliteral->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ultrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      if (ultrl != uliteral1 && ultrl != uliteral2)
	  *res *= 1.0 - lnk->sub * lnk->confuse * nAct(ultrl->act);
    }
}


void calc_aform_importance()
{
  litrlrec *literal;

  for (literal = Gcontrol.unsigned_preds;
       literal != NULL;
       literal = literal->ctrl.nxt) {
    if (literal->tag == PSTERM ||
	literal->tag == NUMBER || literal->tag == SYMBOL) {
      literal->imp = 0.0;
      continue;
    }
    literal->imp = - literal->z * literal->act * (1.0-literal->act)
      * nDXDx(literal->act) / Gparams.tmprtr;
    if (literal->pol == POSITIVE &&
	literal->tag == CONSTRAINT &&
	strcmp(literal->body.afm.name, "true") == 0)
      literal->imp += literal->act;
  }
  for (literal = Gcontrol.signed_preds;
       literal != NULL;
       literal = literal->ctrl.nxt) {
    if (literal->tag == PSTERM ||
	literal->tag == NUMBER || literal->tag == SYMBOL) {
      literal->imp = 0.0;
      continue;
    }
    literal->imp = - literal->z * literal->act * (1.0-literal->act)
      * nDXDx(literal->act) / Gparams.tmprtr;
    if (literal->pol == POSITIVE &&
	literal->tag == CONSTRAINT &&
	strcmp(literal->body.afm.name, "true") == 0)
      literal->imp += literal->act;
  }
}


void backpropagation(itrmax)
     int itrmax;
{
  int i, j;
  litrlrec *literal;
  linkrec *link;
  double disj, excl, assm, cmpl, eqlz, trns, dpnd;

#ifndef PROFILE
  printf("Backpropagation");
#endif
  for (i = 0; i < itrmax && !Gparams.cnvrgdp; i++) {
    Gparams.cnvrgdp = TRUE;
    for (literal = Gcontrol.unsigned_preds;
	 literal != NULL; literal = literal->ctrl.nxt) {
      if (literal->tag == PSTERM ||
	  literal->tag == NUMBER || literal->tag == SYMBOL) {
	literal->z = 0.0;
	continue;
      } else if (literal->pphndl != NULL)
	calc_clausal_estimations(literal->pphndl->begin, literal->pphndl,
				&disj, &excl);
      calc_literal_estimations(literal, &assm, &cmpl, &dpnd);
      update_literal_zvalue(literal, disj, excl, assm, cmpl, dpnd);
    }
    for (literal = Gcontrol.signed_preds;
	 literal != NULL; literal = literal->ctrl.nxt) {
      if (literal->tag == PSTERM ||
	  literal->tag == NUMBER || literal->tag == SYMBOL) {
	literal->z = 0.0;
	continue;
      } else if (literal->pphndl != NULL)
	calc_clausal_estimations(literal->pphndl->begin, literal->pphndl,
				&disj, &excl);
      calc_literal_estimations(literal, &assm, &cmpl, &dpnd);
      update_literal_zvalue(literal, disj, excl, assm, cmpl, dpnd);
    }
    for (link = Gcontrol.links; link != NULL; link = link->ctrl.nxt)
      for (j = 0; j < link->n; j++) {
	calc_link_estimations(link, j, &assm, &eqlz, &trns, &dpnd);
	update_link_zvalue(link, j, assm, eqlz, trns, dpnd);
      }
#ifndef PROFILE
    putchar('.');
#endif
  }
#ifndef PROFILE
  if (Gparams.cnvrgdp)
    printf("converged (%d)\n", i);
  else
    printf("not converged (%d, %lf)\n", i, Gparams.ampl);
#endif
}


void calc_clausal_estimations(clause, literal, disj, excl)
     pphandle *clause, *literal;
     double *disj, *excl;
{
  pphandle *ltrl;
  double disjaux, exclaux;

  *disj = *excl = 0.0;
  for (ltrl = clause; ltrl != NULL; ltrl = ltrl->next) {
    if (ltrl->body->tag == PSTERM ||
	ltrl->body->tag == NUMBER || ltrl->body->tag == SYMBOL)
      continue;
    if (Gparams.disjSW)
      disjunction_estimation(&disjaux, clause, literal, ltrl);
    else
      disjaux = 0.0;
    if (Gparams.exclSW)
      exclusion_estimation(&exclaux, clause, literal, ltrl);
    else
      exclaux = 0.0;
    *disj += ltrl->body->z
      * ltrl->body->act * (1.0-ltrl->body->act) * disjaux;
    *excl += ltrl->body->z
      * ltrl->body->act * (1.0-ltrl->body->act) * exclaux;
  }
}


void calc_literal_estimations(literal, assm, cmpl, psdo)
     litrlrec *literal;
     double *assm, *cmpl, *psdo;
{
  double assm1aux, assm2aux;

  if (Gparams.cmplSW)
    completion_estimation(cmpl, literal);
  else
    *cmpl = 0.0;
  if (Gparams.assmSW) {
    afmafm_assimilation_estimation(&assm1aux, literal);
    afmlnk_assimilation_estimation(&assm2aux, literal);
  } else {
    assm1aux = 0.0;
    assm2aux = 0.0;
  }
  *assm = assm1aux + assm2aux;
  if (Gparams.psdoSW)
    aform_dependency_estimation(psdo, literal);
  else
    *psdo = 0.0;
}


void update_literal_zvalue(literal, disj, excl, assm, cmpl, psdo)
     litrlrec *literal;
     double disj, excl, assm, cmpl, psdo;
{
  double est, subtotal, temp;

  aform_estimation_diff_backp(&est, literal);
  subtotal = Gparams.disjWT * disj
    + Gparams.exclWT * excl
      + Gparams.cmplWT * cmpl
	+ Gparams.assmWT * assm
	  + Gparams.psdoWT * psdo;

  temp = - subtotal / Gparams.tmprtr + est;
  if (fabs(literal->z-temp) > Gparams.epsilon) {
    Gparams.ampl = fabs(literal->z-temp);
    Gparams.cnvrgdp = FALSE;
  }
  literal->z = temp;
}


void calc_link_estimations(link, n, assm, eqlz, trns, dpnd)
     linkrec *link;
     int n;
     double *assm, *eqlz, *trns, *dpnd;
{
  double assm1aux, assm2aux;

  if (Gparams.assmSW) {
    lnklnk_assimilation_estimation(&assm1aux, link, n);
    lnkafm_assimilation_estimation(&assm2aux, link, n);
  } else {
    assm1aux = 0.0;
    assm2aux = 0.0;
  }
  *assm = assm1aux + assm2aux;
  if (Gparams.eqlzSW)
    equalization_estimation(eqlz, link, n);
  else
    *eqlz = 0.0;
  if (Gparams.trnsSW)
    transitivity_estimation(trns, link, n);
  else
    *trns = 0.0;
  link_dependency_estimation(dpnd, link, n);
}


void update_link_zvalue(link, n, assm, eqlz, trns, dpnd)
     linkrec *link;
     int n;
     double assm, eqlz, trns, dpnd;
{
  double est, subtotal, temp;

  link_estimation_diff_backp(&est, link, n);
  subtotal = Gparams.assmWT * assm
    + Gparams.eqlzWT * eqlz
      + Gparams.trnsWT * trns
	+ Gparams.psdoWT * dpnd;
  temp = - subtotal / Gparams.tmprtr + est;
  if (fabs(link->coeff[n].body.z-temp) > Gparams.epsilon) {
    Gparams.ampl = fabs(link->coeff[n].body.z-temp);
    Gparams.cnvrgdp = FALSE;
  }
  link->coeff[n].body.z = temp;
}


/*
 *   `literal1' and `literal2' specify k and l.
 *   Result = d^2 E_disj / dx_k dx_l
 *   ------------------------------------------------------------
 *   E_disj = D Pi (1 - r_i Y_i)
 *   --> d   E_disj / dx_k
 *           = - SIGN(k) D r_k (nDXDx_k) Pi_{i!=k} (1 - r_i Y_i)
 *   ==> (1) d^2 E_disj / dx_k^2
 *           = - SIGN(k) D r_k (nD2XDx2_k) Pi_{i!=k} (1 - r_i Y_i)
 *       (2) d^2 E_disj / dx_k dx_l
 *           = SIGN(k) SIGN(l) D r_k r_l (nDXDx_k) (nDXDx_l)
 *                                       Pi_{i!=k,l} (1 - r_i Y_i)
 *
 *   where Y_i = X_i     (literal i is positive)
 *               1-X_i   (literal i is negative)
 *
 *     SIGN(i) = 1       (literal i is positive)
 *               -1      (literal i is negative)
 *
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void disjunction_estimation(res, clause, literal1, literal2)
     double *res;
     pphandle *clause, *literal1, *literal2;
{
  pphandle *ltrl;

  *res = 1.0;
  if (literal1 == literal2) {
    /*=== case (1) ===*/
    for (ltrl = clause; ltrl != NULL; ltrl = ltrl->next)
      if (ltrl != literal1 &&
	  ltrl->body->tag != SYMBOL &&
	  ltrl->body->tag != NUMBER &&
	  ltrl->body->tag != PSTERM)
	*res *= 1.0 - ltrl->body->rel * nActV(ltrl->body);
    if (literal1->body->pol == POSITIVE ||
	literal1->body->pol == COST_POSITIVE ||
	literal1->body->pol == UNSIGNED)
      *res *= literal1->body->rel;
    else
      *res *= - literal1->body->rel;
    *res *= - nD2XDx2(literal1->body->act)
      * clause->body->coeff.disj * clause->body->coeff.nOmega;
  } else {
    /*=== case (2) ===*/
    for (ltrl = clause; ltrl != NULL; ltrl = ltrl->next)
      if (ltrl != literal1 && ltrl != literal2 &&
	  ltrl->body->tag != SYMBOL &&
	  ltrl->body->tag != NUMBER &&
	  ltrl->body->tag != PSTERM)
	*res *= 1.0 - ltrl->body->rel * nActV(ltrl->body);
    if (((literal1->body->pol == NEGATIVE ||
	  literal1->body->pol == COST_NEGATIVE) &&
	 (literal2->body->pol == NEGATIVE ||
	  literal2->body->pol == COST_NEGATIVE)) ||
	((literal1->body->pol == POSITIVE ||
	  literal1->body->pol == COST_POSITIVE ||
	  literal1->body->pol == UNSIGNED) &&
	 (literal2->body->pol == POSITIVE ||
	  literal2->body->pol == COST_POSITIVE ||
	  literal2->body->pol == UNSIGNED)))
      *res *= literal1->body->rel * literal2->body->rel;
    else
      *res *= - literal1->body->rel * literal2->body->rel;
    *res *= nDXDx(literal1->body->act) * nDXDx(literal2->body->act)
      * clause->body->coeff.disj * clause->body->coeff.nOmega;
  }
}


/*
 *   `literal1' and `literal2' specify k and l.
 *   Result = d^2 E_excl / dx_k dx_l
 *   ------------------------------------------------------------
 *   E_excl = E Sigma_{i!=j} (r_i Y_i r_j Y_j)
 *   --> d   E_excl / dx_k
 *           = SIGN(k) E r_k (nDXDx_k) Sigma_{i!=k} (r_i Y_i)
 *   ==> (1) d^2 E_excl / dx_k^2
 *           = SIGN(k) E r_k (nD2XDx2_k) Sigma_{i!=k} (r_i Y_i)
 *       (2) d^2 E_excl / dx_k dx_l
 *           = SIGN(k) SIGN(l) E r_k (nDXDx_k) r_l (nDXDx_l)
 *
 *   where Y_i = X_i     (literal i is positive)
 *               1-X_i   (literal i is negative)
 *
 *     SIGN(i) = 1       (literal i is positive)
 *               -1      (literal i is negative)
 *
 *     nDXDx   = d   X / dx   = 1 - T'{log(x) + log(1-x) + 2}
 *     nD2XDx2 = d^2 X / dx^2 = - T'(1-2x) / x(1-x)
 */
void exclusion_estimation(res, clause, literal1, literal2)
     double *res;
     pphandle *clause, *literal1, *literal2;
{
  pphandle *ltrl;

  if (literal1 == literal2) {
    /*=== case (1) ===*/
    for (*res = 0.0, ltrl = clause; ltrl != NULL; ltrl = ltrl->next)
      if (ltrl != literal1 &&
	  ltrl->body->tag != SYMBOL &&
	  ltrl->body->tag != NUMBER &&
	  ltrl->body->tag != PSTERM)
	*res += ltrl->body->rel * nActV(ltrl->body);
    if (literal1->body->pol == POSITIVE ||
	literal1->body->pol == COST_POSITIVE ||
	literal1->body->pol == UNSIGNED)
      *res *= literal1->body->rel;
    else
      *res *= - literal1->body->rel;
    *res *= nD2XDx2(literal1->body->act)
      * clause->body->coeff.excl * clause->body->coeff.omegaN;
  } else {
    /*=== case (2) ===*/
    if (((literal1->body->pol == NEGATIVE ||
	  literal1->body->pol == COST_NEGATIVE) &&
	 (literal2->body->pol == NEGATIVE ||
	  literal2->body->pol == COST_NEGATIVE)) ||
	((literal1->body->pol == POSITIVE ||
	  literal1->body->pol == COST_POSITIVE ||
	  literal1->body->pol == UNSIGNED) &&
	 (literal2->body->pol == POSITIVE ||
	  literal2->body->pol == COST_POSITIVE ||
	  literal2->body->pol == UNSIGNED)))
      *res = literal1->body->rel * literal2->body->rel;
    else
      *res = - literal1->body->rel * literal2->body->rel;
    *res *= nDXDx(literal1->body->act) * nDXDx(literal2->body->act)
      * clause->body->coeff.excl * clause->body->coeff.omegaN;
  }
}


/*
 *   `literal' specifies k.
 *   Result = Sigma_l z_l x_l(1-x_l) d^2 E_cmpl / dx_k dx_l
 *   ------------------------------------------------------------
 *   E_cmpl = C X_i Pi_{i@j} {1 - s_ij X_j}
 *   --> d   E_cmpl / dx_i
 *       = C (nDXDx_i) Pi_{i@j} {1 - s_ij X_j}
 *       d   E_cmpl / dx_k
 *       = - C X_i s_ik (nDXDx_k) Pi_{i@j,j!=k} {1 - s_ij X_j}
 *   ==> (1) d^2 E_cmpl / dx_i dx_k
 *           = - C (nDXDx_i) s_ik (nDXDx_k) Pi_{i@j,j!=k} {1 - s_ij X_j}
 *       (2) d^2 E_cmpl / dx_i^2
 *           = C (nD2XDx2_i) Pi_{i@j} {1 - s_ij X_j}
 *       (3) d^2 E_cmpl / dx_k dx_l
 *           = C X_i s_ik (nDXDx_k) s_il (nDXDx_l)
 *                                  Pi_{i@j,j!=k,j!=l} {1 - s_ij X_j}
 *       (4) d^2 E_cmpl / dx_k^2
 *           = - C X_i s_ik (nD2XDx2_k) Pi_{i@j,j!=k} {1 - s_ij X_j}
 *
 *   *NOTICE*
 *   x_i and x_k are not activation values of LITERALS.
 *   This routine calculates POSITIVE completion estimation.
 */
void completion_estimation(res, literal)
     double *res;
     litrlrec *literal;
{
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;
  double df;

  *res = 0.0;
  if (literal->tag != CONSTRAINT)
    return;
  if (literal->pol != UNSIGNED) {
    /*=== case (1) and (2) ===*/
    /*=== check whether `literal' is a cost or a defined literal ===*/
    if (literal->pol == COST_POSITIVE || literal->pol == COST_NEGATIVE ||
	((jnt=literal->joint) != NULL && (dpnd=jnt->depend) != NULL &&
	 (ltrl=dpnd->link->ptr[dpnd->dir]->joint->ltrl) != NULL &&
	 ltrl->pol == UNSIGNED)) {
      /*=== case (1) ===*/
      for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
	for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	  lnk = dpnd->link;
	  ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
	  completion_force_aux1(&df, literal, ltrl);
	  *res -= ltrl->z * ltrl->act * (1.0-ltrl->act)
	    * nDXDx(literal->act) * nDXDx(ltrl->act) * df;
	}
      /*=== case (2) ===*/
      completion_force_aux1(&df, literal, NULL);
      *res += literal->z * literal->act * (1.0-literal->act)
	*nD2XDx2(literal->act) * df;
    } else if (jnt != NULL && dpnd != NULL && ltrl != NULL &&
	       (ltrl->pol == COST_POSITIVE || ltrl->pol == COST_NEGATIVE)) {
      /*=== `literal' is a cost literal ===*/
      /*=== case (1), (3) and (4) ===*/
      for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
	for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	  lnk = dpnd->link;
	  ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
	  /*=== case (1) ===*/
	  completion_force_aux1(&df, ltrl, literal);
	  *res -= ltrl->z * ltrl->act*(1.0-ltrl->act)
	    * nDXDx(ltrl->act) * nDXDx(literal->act) * df;
	  /*=== case (3) === */
	  completion_force_aux2(&df, ltrl, literal);
	  *res += df;
	  /*=== case (4) ===*/
	  completion_force_aux1(&df, ltrl, literal);
	  *res -= nAct(ltrl->act) * literal->z * literal->act*(1.0-literal->act)
	    * nD2XDx2(literal->act) * df;
	}
    }
  } else {
    /*=== `literal' is an unsigned literal (i.e. a defined literal)===*/
    /*=== case (1), (3) and (4) ===*/
    for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
      for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
	lnk = dpnd->link;
	ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
	/*=== case (1) ===*/
	completion_force_aux1(&df, ltrl, literal);
	*res -= ltrl->z * ltrl->act*(1.0-ltrl->act)
	  * nDXDx(ltrl->act) * nDXDx(literal->act) * df;
	/*=== case (3) === */
	completion_force_aux2(&df, ltrl, literal);
	*res += df;
	/*=== case (4) ===*/
	completion_force_aux1(&df, ltrl, literal);
	*res -= nAct(ltrl->act) * literal->z * literal->act*(1.0-literal->act)
	  * nD2XDx2(literal->act) * df;
      }
  }
  *res *= literal->coeff.cmpl;
}


/*
 *   `sliteral' specifies i and `uliteral' specifies k.
 *   This routine deals with the case (3) of the above computation:
 *
 *   d^2 E_cmpl / dx_k dx_l =
 *       C X_i s_ik (nDXDx_k) s_il (nDXDx_l) Pi_{i@j,j!=k,l} {1 - s_ij X_j}
 *
 *   Result = X_i s_ik (nDXDx_k)
 *                Sigma_{l!=k} [ z_l x_l(1-x_l) s_il (nDXDx_l)
 *                               Pi_{i@j,j!=k,l} {1 - s_ij X_j} ]
 */
void completion_force_aux2(res, sliteral, uliteral)
     double *res;
     litrlrec *sliteral, *uliteral;
{
  litrlrec *ltrl0, *ltrl1;
  jointrec *jnt0, *jnt1;
  dependrec *dpnd0, *dpnd1;
  linkrec *lnk0, *lnk1;
  double df, s;

  *res = 0.0;
  for (jnt0 = sliteral->joint; jnt0 != NULL; jnt0 = jnt0->nxt)
    for (dpnd0 = jnt0->depend; dpnd0 != NULL; dpnd0 = dpnd0->nxt) {
      lnk0 = dpnd0->link;
      ltrl0 = lnk0->ptr[dpnd0->dir]->joint->ltrl;
      if (ltrl0 == uliteral) {
	s = lnk0->sub * lnk0->confuse;
	continue;
      }
      df = 1.0;
      for (jnt1 = sliteral->joint; jnt1 != NULL; jnt1 = jnt1->nxt)
	for (dpnd1 = jnt1->depend; dpnd1 != NULL; dpnd1 = dpnd1->nxt) {
	  lnk1 = dpnd1->link;
	  ltrl1 = lnk1->ptr[dpnd1->dir]->joint->ltrl;
	  if (ltrl1 == uliteral)
	    continue;
	  else if (ltrl1 == ltrl0)
	    df *= lnk1->sub * lnk1->confuse * nDXDx(ltrl1->act);
	  else
	    df *= 1.0 - lnk1->sub * lnk1->confuse * nAct(ltrl1->act);
	}
      *res += ltrl0->z * ltrl0->act*(1.0-ltrl0->act) * df;
    }
  *res *= nAct(sliteral->act) * s * nDXDx(uliteral->act);
}


/*
 *   `literal' specifies x_a.
 *   Result = Sigma_b z_b d^2 E_assm / dx_a dx_b
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_a      = - A s_ab (nDXDx_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   ==> d^2 E_assm / dx_a^2    = - A s_ab (nD2XDx2_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *       d^2 E_assm / dx_a dx_b = - A s_ab (nDXDx_a) (nDXDx_b) Pi X_e_ab
 *
 *   Result = - A Sigma_b s_ab [ z_a x_a(1-x_a) (nD2XDx2_a) (X_b - S(dPds_b))
 *                             + z_b x_b(1-x_b) (nDXDx_a) (nDXDx_b) ] Pi X_e_ab
 */
void afmafm_assimilation_estimation(res, literal)
     double *res;
     litrlrec *literal;
{
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;
  int i;
  double delta;

  *res = 0.0;
  for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      for (delta = 1.0, i = 0; i < lnk->n; i++)
	delta *= nAct(lnk->coeff[i].body.act);
      *res += lnk->sub * lnk->confuse
	* (literal->z * literal->act*(1.0-literal->act)
	   * nD2XDx2(literal->act) * (nAct(ltrl->act)-negP(ltrl->imp))
	   + ltrl->z * ltrl->act*(1.0-ltrl->act)
	   * nDXDx(literal->act) * nDXDx(ltrl->act)) * delta;
    }
  *res *= - literal->coeff.assm;
}


/*
 *   `literal' specifies x_a.
 *   Result = Sigma_b Sigma_m z_e_ab_m x_e_ab_m(1-x_e_ab_m)
 *                            d^2 E_assm / dx_a dx_e_ab_m
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_a
 *          = - A s_ab (nDXDx_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   ==> d^2 E_assm / dx_a^2
 *          = - A s_ab (nD2XDxw_a) (X_b - S(dP/ds_b)) Pi X_e_ab_i
 *       d^2 E_assm / dx_a de_ab_m
 *          = - A s_ab (nDXDx_a) (nDXDx_ab_m) (X_b - S(dP/ds_b))
 *                                            Pi_{i!=m} X_e_ab_i
 *
 *   Result = - A Sigma_b s_ab [
 *                z_a x_a(1-x_a) (nD2XDx2_a) (X_b - S(dP/ds_b)) Pi X_e_ab
 *              + (nDXDx_a) (X_b - S(dP/ds_b))
 *                          Sigma_m z_m x_e_ab_m(1-x_e_ab_m)
 *                                  (nDXDx_ab_m) Pi_{i!=m} X_e_ab_i
 *                ]
 */
void afmlnk_assimilation_estimation(res, literal)
     double *res;
     litrlrec *literal;
{
  litrlrec *ltrl;
  jointrec *jnt;
  dependrec *dpnd;
  linkrec *lnk;
  int i, m;
  double temp, delta;

  *res = 0.0;
  for (jnt = literal->joint; jnt != NULL; jnt = jnt->nxt)
    for (dpnd = jnt->depend; dpnd != NULL; dpnd = dpnd->nxt) {
      lnk = dpnd->link;
      ltrl = lnk->ptr[dpnd->dir]->joint->ltrl;
      for (temp = 0.0, m = 0; m < lnk->n; m++) {
	for (delta = 1.0, i = 0; i < lnk->n; i++)
	  if (i != m)
	    delta *= nAct(lnk->coeff[i].body.act);
	temp += lnk->coeff[m].body.z
	  * lnk->coeff[m].body.act*(1.0-lnk->coeff[m].body.act)
	    * nDXDx(lnk->coeff[m].body.act) * delta;
      }

      for (delta = 1.0, i = 0; i < lnk->n; i++)
	delta *= nAct(lnk->coeff[i].body.act);
      *res += lnk->sub * lnk->confuse
	* (literal->z * literal->act*(1.0-literal->act)
	   * nD2XDx2(literal->act) * (nAct(ltrl->act)-negP(ltrl->imp)) * delta
	   + nDXDx(literal->act) * (nAct(ltrl->act)-negP(ltrl->imp)) * temp);
    }
  *res *= - literal->coeff.assm;
}


/*
 *   `link' and `nth' specify e_ab_m.
 *   Result = Sigma_n z_e_ab_n x_e_ab_n(1-x_e_ab_n)
 *                             d^2 E_assm / dx_e_ab_m dx_e_ab_n
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_e_ab_m
 *       = - A s_ab (nDXDx_e_ab_m) (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                                                    Pi_{i!=m} X_e_ab_i
 *   ==> d^2 E_assm / dx_e_ab_m^2
 *       = - A s_ab (nD2XDx2_e_ab_m) (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                                                      Pi_{i!=m} X_e_ab_i
 *
 *       d^2 E_assm / dx_e_ab_m dx_e_ab_n
 *       = - A s_ab (nDXDx_e_ab_m) (nDXDx_e_ab_n)
 *                  (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi_{i!=m,n} X_e_ab_i
 *
 *   Result = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *               [ z_e_ab_m x_e_ab_m(1-x_e_ab_m) (nD2XDx2_e_ab_m)
 *                                                 Pi_{i!=m} X_e_ab_i
 *               + Sigma_n z_n x_n(1-x_n) (nDXDx_e_ab_m) (nDXDx_e_ab_n)
 *                                                 Pi_{i!=m,n} X_e_ab_i ]
 */
void lnklnk_assimilation_estimation(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  litrlrec *ltrl0, *ltrl1;
  int n, i;
  double temp, delta;

  ltrl0 = link->ptr[0]->joint->ltrl;
  ltrl1 = link->ptr[1]->joint->ltrl;
  for (temp = 0.0, n = 0; n < link->n; n++) {
    for (delta = 1.0, i = 0; i < link->n; i++)
      if (i != nth && i != n)
	delta *= nAct(link->coeff[i].body.act);
    temp += link->coeff[n].body.z
      * link->coeff[n].body.act * (1.0-link->coeff[n].body.act)
	* nDXDx(link->coeff[nth].body.act) * nDXDx(link->coeff[n].body.act)
	  * delta;
  }

  for (delta = 1.0, i = 0; i < link->n; i++)
    if (i != nth)
      delta *= nAct(link->coeff[i].body.act);
  *res = - ltrl0->coeff.assm * link->sub * link->confuse
    * (nAct(ltrl0->act)-negP(ltrl0->imp)) * (nAct(ltrl1->act)-negP(ltrl1->imp))
      * (link->coeff[nth].body.z
	 * link->coeff[nth].body.act *(1.0-link->coeff[nth].body.act)
	 * nD2XDx2(link->coeff[nth].body.act) * delta + temp);
}


/*
 *   `link' and `nth' specify e_ab_m
 *   Result = z_a x_a(1-x_a) d^2 E_assm / dx_e_ab_m dx_a
 *          + z_b x_b(1-x_b) d^2 E_assm / dx_e_ab_m dx_b
 *   ------------------------------------------------------------
 *   E_assm = - A s_ab (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b)) Pi X_e_ab
 *   --> d   E_assm / dx_e_ab_m
 *       = - A s_ab (nDXDx_e_ab_m) (X_a - S(dP/ds_a)) (X_b - S(dP/ds_b))
 *                                                    Pi_{i!=m} X_e_ab_i
 *   ==> d^2 E_assm / dx_e_ab_m dx_a
 *       = - A s_ab (nDXDx_e_ab_m) (nDXDx_a) (X_b - S(dP/ds_b))
 *                                           Pi_{i!=m} X_e_ab_i
 *
 *   Result = - A s_ab (nDXDx_e_ab_m)
 *                [ z_a x_a(1-x_a) (nDXDx_a) (x_b - S(dP/ds_b))
 *                + z_b x_b(1-x_b) (nDXDx_b) (x_a - S(dP/ds_a)) ]
 *              Pi_{i!=m} X_e_ab_i
 */
void lnkafm_assimilation_estimation(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  litrlrec *ltrl0, *ltrl1;
  int i;
  double temp, delta;

  ltrl0 = link->ptr[0]->joint->ltrl;
  ltrl1 = link->ptr[1]->joint->ltrl;
  for (delta = 1.0, i = 0; i < link->n; i++)
    if (i != nth)
      delta *= nAct(link->coeff[i].body.act);

  *res = - ltrl0->coeff.assm * link->sub * link->confuse
    * nDXDx(link->coeff[nth].body.act)
      * (ltrl0->z * ltrl0->act * (1.0-ltrl0->act)
	 * nDXDx(ltrl0->act) * (nAct(ltrl1->act)-negP(ltrl0->imp))
	 + ltrl1->z * ltrl1->act * (1.0-ltrl1->act)
	 * nDXDx(ltrl1->act) * (nAct(ltrl0->act)-negP(ltrl1->imp))) * delta;
}


/*
 *   `link' and `nth' specify l.
 *   Result = Sigma_m z_e_m x_e_m(1-x_e_m) d^2 E_eqlz / dx_e_l dx_e_m
 *   ------------------------------------------------------------
 *   E_eqlz = - X_e_0 Sigma_{i>0} X_e_i
 *   --> d   E_eqlz / dx_e_0    = - (nDXDx_e_0) Sigma_{i>0} X_e_i
 *       d   E_eqlz / dx_e_j    = - X_e_0 (nDXDx_e_j)
 *   ==> d^2 E_eqlz / dx_e_0^2      = - (nD2XDx2_e_0) Sigma_{i>0} X_e_i
 *       d^2 E_eqlz / dx_e_0 dx_e_j = - (nDXDx_e_0) (nDXDx_e_j)
 *       d^2 E_eqlz / dx_e_j^2      = - X_e_0 (nD2XDx2_e_j)
 *       d^2 E_eqlz / dx_e_j dx_e_k = 0
 *
 *   Result = ...
 *      case (1) l == 0
 *            z_e_0 x_e_0(1-x_e_0) d^2 E_eqlz / dx_e_0^2
 *          + Sigma_{j>0} z_e_j x_e_j(1-x_e_j) d^2 E_eqlz / dx_e_0 dx_e_j
 *      = - z_e_0 x_e_0(1-x_e_0) (nD2XDx2_e_0) Sigma_{i>0} X_e_i
 *        - (nDXDx_e_0) Sigma_{j>0} z_e_j x_e_j(1-x_e_j) (nDXDx_e_j)
 *      case (2) l != 0
 *           z_e_0 x_e_0(1-x_e_0) d^2 E_eqlz / de_j de_0
 *         + z_e_j x_e_j(1-x_e_j) d^2 E_eqlz / de_j^2
 *      = - z_e_0 x_e_0(1-x_e_0) (nDXDx_e_j) (nDXDx_e_0)
 *        - z_e_j x_e_j(1-x_e_j) X_e_0 (nD2XDx2_e_j)
 */
void equalization_estimation(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  int i;
  double sum, temp;

  if (link->tag != INFERENCE ||
      link->ptr[0]->joint->ltrl->tag == CONSTRAINT) {
    *res = 0.0;
    return;
  }
  if (nth == link->n - 1) {
    /*=== case (1) ===*/
    for (sum = 0.0, i = 0; i < link->n-1; i++)
      sum += nAct(link->coeff[i].body.act);
    temp = link->coeff[nth].body.z
      * link->coeff[nth].body.act * (1.0-link->coeff[nth].body.act)
	* nD2XDx2(link->coeff[nth].body.act) * sum;
    for (sum = 0.0, i= 0; i < link->n-1; i++)
      sum += link->coeff[i].body.z
	* link->coeff[i].body.act * (1.0-link->coeff[i].body.act)
	  * nDXDx(link->coeff[i].body.act);
    *res = - temp - nDXDx(link->coeff[nth].body.act) * sum;
  } else
    /*=== case (2) ===*/
    *res = - link->coeff[link->n-1].body.z
      * link->coeff[link->n-1].body.act * (1.0-link->coeff[link->n-1].body.act)
	* nDXDx(link->coeff[nth].body.act) * nDXDx(link->coeff[link->n-1].body.act)
	  - link->coeff[nth].body.z
	    * link->coeff[nth].body.act * (1.0-link->coeff[nth].body.act)
	      * nAct(link->coeff[link->n-1].body.act)
		* nD2XDx2(link->coeff[nth].body.act);
}


/*
 *   `link' and `nth' specify j.
 *   Result = Sigma_loops Sigma_k z_k x_k(1-x_k) d^2 E_trns / de_j de_k
 *   ------------------------------------------------------------
 *   E_trns = - t Pi (X_e_i - theta)
 *   --> d   E_trns / dx_e_j        = - t (nDXDx_e_j) Pi_{i!=j} (X_e_i - theta)
 *   ==> d^2 E_trns / dx_e_j^2      = - t (nD2XDx2_e_j) Pi_{i!=j} (X_e_i - theta)
 *       d^2 E_trns / dx_e_j dx_e_k = - t (nDXDx_e_j) (nDXDx_e_k)
 *                                                    Pi_{i!=j,k} (X_e_i - theta)
 *
 *   result = - t Sigma_loops [
 *                  z_e_j x_e_j(1-x_e_j) (nD2XDx2_e_j) Pi_{i!=j} (X_e_i - theta)
 *                + Sigma_{k!=j} z_k x_e_k(1-x_e_k)
 *                      (nDXDx_e_j) (nDXDx_e_k) Pi_{i!=j,k} (X_e_i - theta)
 *                ]
 */
void transitivity_estimation(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  index *loop, *elem;
  double first, second, df;
  linkact_cell *az;

  *res = 0.0;
  for (loop = Gcontrol.loops; loop != NULL; loop = loop->others)
    /*=== check if this `loop' includes the given `nth of link' ===*/
    if (check_threshold(loop, link, nth)) {
      transitivity_diff_aux(&first, loop, link, nth, NULL);
      first *= link->coeff[nth].body.z
	* link->coeff[nth].body.act * (1.0-link->coeff[nth].body.act)
	  * nD2XDx2(link->coeff[nth].body.act);
      for (second = 0.0, elem = loop; elem != NULL; elem = elem->next)
	if (elem->link == link && (elem->link->tag == EQUATION ||
				   (elem->nth == PSTENTRY && nth == 0) ||
				   (elem->nth == LEFTHAND && nth == link->n-1) ||
				   elem->nth == nth))
	  continue;
	else {
	  transitivity_diff_aux(&df, loop, link, nth, elem);
	  az = nth_act_z(elem->link, elem->nth);
	  second += az->body.z * az->body.act * (1.0-az->body.act)
	    * nDXDx(link->coeff[nth].body.act) * nDXDx(az->body.act) * df;
	}
      *res -= first + second;
    }
}


Boolean check_threshold(loop, link, nth)
     index *loop;
     linkrec *link;
     int nth;
{
  index *elem;
  linkact_cell *az;
  int count;
  Boolean member;

  count = 0, member = FALSE;
  for (elem = loop; elem != NULL; elem = elem->next) {
    if (elem->link == link &&
	(elem->link->tag == EQUATION ||
	 (elem->nth == PSTENTRY && nth == 0) ||
	 (elem->nth == LEFTHAND && nth == link->n-1) ||
	 elem->nth == nth))
      member = TRUE;
    az = nth_act_z(elem->link, elem->nth);
    if (nAct(az->body.act)-Gparams.theta <= Gparams.epsilon) {
      if (count > 1)
	return FALSE;
      count++;
    }
  }
  return member;
}


/*
 *   `link', `nth' specify j and `path' specifies k.
 *   Result = Pi_{i!=j,k} (X_e_i - theta)
 *
 *   Notes:
 *	This routine is depend on routine detect_loops(). (loop.c)
 *	Though detect_loops() can detect loops including weak links,
 *	it cannot detect all possible loops.
 */
void transitivity_diff_aux(res, loop, link, nth, elem)
     double *res;
     index *loop;
     linkrec *link;
     int nth;
     index *elem;
{
  index *e;
  linkact_cell *az;

  *res = 1.0;
  for (e = loop; e != NULL; e = e->next)
    if (e != elem && ! (e->link == link &&
			(e->link->tag == EQUATION ||
			 (e->nth == PSTENTRY && nth == 0) ||
			 (e->nth == LEFTHAND && nth == link->n-1) ||
			 e->nth == nth))) {
      az = nth_act_z(e->link, e->nth);
      *res *= nAct(az->body.act) - Gparams.theta;
    }
}


/*
 *   `literal' specifies p.
 *   Result = z_p x_p(1-x_p) d^2 E_dpnd / dx_p^2
 *            + z_q x_q(1-x_q) d^2 E_dpnd / dx_p dx_q
 *            + Sigma_j z_e_j x_e_j(1-x_e_j) d^2 E_dpnd / dx_p dx_e_j
 *   ------------------------------------------------------------
 *   E_dpnd = - X_p X_q Pi s_i X_e_i
 *   ==> dE_dpnd / dx_p   = - F_prss = - (nDXDx_p) X_q Pi s_i X_e_i
 *       dE_dpnd / dx_e_j = - X_p X_q s_j (nDXDx_e_j) Pi_{i!=j} s_i X_e_i
 *   --> d^2 E_dpnd / dx_p^2      = - (nD2XDx2_p) X_q Pi s_i X_e_i
 *       d^2 E_dpnd / dx_p dx_q   = - (nDXDx_p) (nDXDx_q) Pi s_i X_e_i
 *       d^2 E_dpnd / dx_p dx_e_j = - (nDXDx_p) X_q s_j (nDXDx_e_j)
 *                                                      Pi_{i!=j} s_i X_e_i
 *
 *   result = - [ z_p x_p(1-x_p) (nD2XDx2_p) X_q
 *              + z_q x_q(1-x_q) (nDXDx_p) (nDXDx_q) ] Pi s_i X_e_i
 *            - (nDxDx_p) Sigma_j z_e_j x_e_j(1-x_e_j)
 *                                s_j (nDXDx_e_j) X_q Pi_{i!=j} s_i X_e_i
 */
void aform_dependency_estimation(res, literal)
     double   *res;
     litrlrec *literal;
{
  double first, second;
  linkact_cell *az;
  press *prss;

  if (literal->path == NULL) {
    *res = 0.0;
    return;
  }
  first = literal->z * literal->act * (1.0-literal->act)
    * nD2XDx2(literal->act) * nAct(literal->path->orgn->act)
      + literal->path->orgn->z
	* literal->path->orgn->act * (1.0-literal->path->orgn->act)
	  * nDXDx(literal->act) * nDXDx(literal->path->orgn->act);
  for (second = 0.0, prss = literal->path;
       prss != NULL; prss = prss->path.nxt) {
    az = nth_act_z(prss->link, prss->nth);
    second += az->body.z * az->body.act * (1.0-az->body.act)
      * prss->link->sub * nDXDx(az->body.act)
	* literal->dfrc / nAct(az->body.act);
  }
  *res = - first * literal->dfrc / nAct(literal->path->orgn->act)
    - nDXDx(literal->act) * second;
}


/*
 *   `link' and `nth' specifies j.
 *   Result = Sigma_p z_p x_p(1-x_p) d^2 E_dpnd / dx_e_j dx_p
 *          + Sigma_i z_e_i x_e_i(1-x_e_i) d^2 E_dpnd / dx_e_j dx_e_i
 *   ------------------------------------------------------------
 *   E_dpnd = - X_p X_q Pi s_i X_e_i
 *   --> dE_dpnd / dx_p = - F_prss = - (nDXDx_p) X_q Pi s_i X_e_i
 *       dE_dpnd / dx_e_j = - X_p X_q s_j (nDXDx_e_j) Pi_{i!=j} s_i X_e_i
 *   ==> d^2 E_dpnd / dx_e_j dx_p
 *       = - (nDXDx_p) X_q s_j (nDXDx_e_j) Pi_{i!=j} s_i X_e_i
 *       d^2 E_dpnd / dx_e_j^2
 *       = - X_p X_q s_j (nD2XDx2_e_j) Pi_{i!=j} s_i X_e_i
 *       d^2 E_dpnd / dx_e_j dx_e_k
 *       = - X_p X_q s_j s_k (nDXDx_e_j) (nDXDx_e_k) Pi_{i!=j,k} s_i X_e_i
 *
 *   result = - [ z_p x_p(1-x_p) (nDXDx_p) X_q
 *              + z_q x_q(1-x_q) X_p (nDXDx_q) ] s_j (nDXDx_e_j) Pi_{i!=j} s_i X_e_i
 *            - z_e_j x_e_j(1-x_e_j) s_j (nD2XDx2_e_j) X_p X_q Pi_{i!=j} s_i X_e_i
 *            - s_j (nDXDx_e_j)
 *                  Sigma_{k!=j} z_e_k x_e_k(1-x_e_k) s_k (nDXDx_e_k)
 *                                                    X_p X_q Pi_{i!=j,k} s_i X_e_i
 */
void link_dependency_estimation(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  litrlrec *binding, *origin;
  press *prss;
  double first, second, third;
  linkact_cell *az;

  *res = 0.0;
  for (binding = Gcontrol.unsigned_preds;
       binding != NULL;
       binding = binding->ctrl.nxt) {
    /*=== check if this dependency path includes `nth of link' ===*/
    for (origin = NULL, prss = binding->path;
	 prss != NULL;
	 prss = prss->path.nxt)
      if (prss->link == link && (prss->link->tag == EQUATION ||
				 (prss->nth == PSTENTRY && nth == 0) ||
				 (prss->nth == LEFTHAND && nth == link->n-1) ||
				 prss->nth == nth)) {
	origin = prss->orgn;
	break;
      }
    if (origin == NULL)
      continue;
    first = (origin->z * origin->act * (1.0-origin->act)
	     * nDXDx(origin->act) * nAct(binding->act)
	     + binding->z * binding->act * (1.0-binding->act)
	     * nAct(origin->act) * nDXDx(binding->act))
      * link->sub * link->confuse * nDXDx(link->coeff[nth].body.act)
	* binding->dfrc  / (nAct(origin->act) * nAct(link->coeff[nth].body.act));
    second = link->coeff[nth].body.z
      * link->coeff[nth].body.act * (1.0-link->coeff[nth].body.act)
	* link->sub * link->confuse * nD2XDx2(link->coeff[nth].body.act)
	  * nAct(binding->act) * binding->dfrc / nAct(link->coeff[nth].body.act);
    for (third = 0.0, prss = binding->path;
	 prss != NULL;
	 prss = prss->path.nxt) {
      if (prss->link == link && (prss->link->tag == EQUATION ||
				 (prss->nth == PSTENTRY && nth == 0) ||
				 (prss->nth == LEFTHAND && nth == link->n-1) ||
				 prss->nth == nth))
	continue;
      az = nth_act_z(prss->link, prss->nth);
      third += az->body.z * az->body.act * (1.0-az->body.act)
	* prss->link->sub * prss->link->confuse * nDXDx(az->body.act)
	  * nAct(binding->act) * binding->dfrc
	    / (nAct(link->coeff[nth].body.act) * nAct(az->body.act));
    }
    *res -= first
      + second
	+ link->sub * link->confuse * nDXDx(link->coeff[nth].body.act) * third;
  }
}


/*
 *   H = D (1 - r_{+true} X_{+true})
 *   ==> dH / dx_j = - D r_{+true} nDXDx_{+true}	: x_j is +true()
 *       dH / dx_k = 0					: otherwise
 */
void aform_estimation_diff_backp(res, literal)
     double *res;
     litrlrec *literal;
{
  if (literal->pol == POSITIVE &&
      literal->tag == CONSTRAINT &&
      strcmp(literal->body.afm.name, "true") == 0)
    *res = - literal->coeff.disj * literal->coeff.nOmega
      * literal->rel * nDXDx(literal->act);
  else
    *res = 0.0;
}


/*
 *   H = E_disj_of_top = D (1 - r_{+true} X_{+true})
 *   ==> dH / de_ab_j = 0
 */
void link_estimation_diff_backp(res, link, nth)
     double *res;
     linkrec *link;
     int nth;
{
  *res = 0.0;
}


linkact_cell *nth_act_z(link, nth)
     linkrec *link;
     int nth;
{
  if (link->tag == EQUATION)
    return &(link->coeff[0]);
  else if (nth == PSTENTRY)
    return &(link->coeff[0]);
  else if (nth == LEFTHAND)
    return &(link->coeff[link->n-1]);
  else
    return &(link->coeff[nth]);
}
