#include <stdio.h>
/* #include <stdlib.h>*/
#include <string.h>
#include <fcntl.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <pwd.h>

#include "qxt.h"
#include "qshell.h"

static char *progname;
char *prompt;

int session;
int interactive = 0;

CommandEntry *ss_command_table;
CommandEntry *db_command_table;
CommandEntry *ts_command_table;
CommandEntry *is_command_table;

CommandEntry *command_table;

struct m_c_entry {
  char *command_name;
  void (* func)();
};

static struct m_c_entry meta_command_table[] =
{
  {"help", qshell_help},
  {"lcd", qshell_lcd},
  {"quit", qshell_quit},
  {"shell", qshell_shell_command},
  {"take", qshell_take},
  {"timer-off", qshell_timer_off},
  {"timer-on", qshell_timer_on},
};

int use_timer;

#define NUM_OF_META_COMMANDS (sizeof(meta_command_table) / sizeof(struct m_c_entry))

char *qshell_directory;
char *qshell_config_file_name;

char *server;

static int mce_cmp (a,b)
     char *a, *b;
{
  return strcmp (((struct m_c_entry *)a)->command_name,
		 ((struct m_c_entry *)b)->command_name);
}

int qshell_main_loop ();

main (argc, argv)
     int argc;
     char *argv[];
{
  progname = argv[0];

  qshell_directory        = QSHELL_DIRECTORY;
  qshell_config_file_name = QSHELL_CONFIG_FILE_NAME;
  use_timer = 0;
  putenv ("QXTTIMER=OFF");

  while (*(++argv) && **argv == '-')
    {
      switch ((*argv)[1])
	{
	case 'd':
	  qshell_directory = *(++argv);
	  break;
	case 'f':
	  qshell_config_file_name = *(++argv);
	  break;
	default:
	  fprintf
	    (stderr,
	     "Usage: %s [-d qshell_dir] [-f config_file_name] [server]\n", progname);
	  exit (1);
	  break;
	}
    }

  if (isatty (fileno (stdin)) &&
      isatty (fileno (stdout)))
    interactive = 1;

  if (*argv)
    server = *argv;
  else
    { /* get host name from environment variable */
      extern char *getenv ();

      server = getenv ("QXTSERVER");
      if (server == NULL)
	{
	  fprintf (stderr,  "Please specify server name by argument or environment variable `QXTSERVER'.", progname);
	  exit (1);
	}
    }

  init_connection (server);
  init_quixote ();
  if (interactive)
    init_readline ();	/* Bind our completer. */
  read_configuration ();
  init_yylex ();

  session = 1;
  command_table = ss_command_table;
  fflush (stdout);

  qshell_main_loop (stdin);

  close_connection ();
  exit (0);
}

int qshell_main_loop (stream)
     FILE *stream;
{
  int done = 0;

  while (!done)
    {
      char *line;
      int result;
      struct m_c_entry *mce;

      if (interactive)
	line = readline (prompt);
      else
	line = readline_stream (stream);

      if (!line)
	done = 1;		/* Encountered EOF at top level. */
      else
	{
	  stripwhite (line);
	  if (interactive)
	    {
	      if (*line)
		add_history (line);
	    }
	  else
	    {
	      if (use_timer)
		printf ("%s\n", line);
	    }

	  init_yylex_for_line (line);

	  /* Invoke YYPARSE to parse the input line, the result is
	     assigned in the variable COMMAND. */
	  if (yyparse ())
	    { /* syntax error */
	      exit (1);
	    }

	  if (command == NULL)
	    {}
	  else
	    {
	      struct m_c_entry m;
	      m.command_name = command->argv[0];
	      mce = (struct m_c_entry *)bsearch ((char *)&m,
						 (char *)meta_command_table,
						 NUM_OF_META_COMMANDS,
						 sizeof(struct m_c_entry),
						 mce_cmp);
	      if (mce)
		mce->func (command);
	      else
		do_work (command);

	      free_command_deep (command);
	    }

	  switch (session)
	    {
	    case 1:
	      prompt = "ss% ";
	      command_table = ss_command_table;
	      break;
	    case 2:
	      prompt = "db% ";
	      command_table = db_command_table;
	      break;
	    case 3:
	      prompt = "ts% ";
	      command_table = ts_command_table;
	      break;
	    case 4:
	      prompt = "is% ";
	      command_table = is_command_table;
	      break;
	    }
	  free (line);
	}
    }
}

CommandEntry *get_entry_in_table (command_name, command_table)
     char *command_name;
     CommandEntry *command_table;
{
  CommandEntry *table = command_table;

  if (table == NULL)
    return NULL;

  while (table->name)
    {
      if (strcmp (table->name, command_name) == 0)
	return table;
      table++;
    }

  return NULL;
}

do_work (command)
     COMMAND *command;
{
  CommandEntry *ce;
  int child_pid;
  int status;
  int stdin_fd=0, stdout_fd=0, stderr_fd=0;
  REDIRECT *redirect = command->redirect;

  while (redirect)
    {
      switch (redirect->mode)
	{
	case input:
	  stdin_fd = open (redirect->file_name, O_RDONLY);
	  if (stdin_fd < 0)
	    {
	      fprintf (stderr, "redirection failed: %s\n", redirect->file_name);
	      return;
	    }
	  break;
	case output:
	  stdout_fd = open (redirect->file_name, O_CREAT | O_TRUNC | O_WRONLY, 0666);
	  if (stdout_fd < 0)
	    {
	      fprintf (stderr, "redirection failed: %s\n", redirect->file_name);
	      return;
	    }
	  break;
	case output_append:
	  stdout_fd = open (redirect->file_name,
			    O_CREAT | O_APPEND | O_WRONLY, 0666);
	  if (stdout_fd < 0)
	    {
	      fprintf (stderr, "redirection failed: %s\n", redirect->file_name);
	      return;
	    }
	  break;
	default:
	  abort ();
	  break;
	}

      redirect = redirect->next;
    }

  ce = get_entry_in_table (command->argv[0], command_table);
  if (ce)
    {
      if (child_pid = fork ())
	{ /* parent */
	  if (child_pid < 0)
	    fprintf (stderr, "Can't fork.\n");
	  else
	    {
	      int next_session;

	      wait (&status);
	      if (status & 0xff)
		fprintf (stderr, "Abnormal exit status %d.\n", status & 0x0ff);
	      else
		{
		  next_session = (status & 0xff00) >> 8;
		  if (next_session >= 0x05)
		    {
		      fprintf (stderr, "exit status %d\n", next_session);
		      fatal ("fatal error in child\n", 1);
		    }
		  else if (next_session)
		    session = next_session;
		}
	    }
	}
      else
	{
	  if (stdin_fd)
	    {
	      close (0);
	      dup (stdin_fd);
	      close (stdin_fd);
	    }
	  if (stdout_fd)
	    {
	      close (1);
	      dup (stdout_fd);
	      close (stdout_fd);
	    }
	  if (stderr_fd)
	    {
	      close (2);
	      dup (stderr_fd);
	      close (stderr_fd);
	    }
	  execv (ce->path, command->argv);
	  fprintf (stderr, "Can't exec\n");
	  exit (0);
	}
    }
  else
    fprintf (stderr, "No such command (%s) in %smode\n",
	     command->argv[0], prompt);
}

init_quixote ()
{
  prompt = "ss% ";
}

#define MAXLINE 1024

FILE *netin, *netout;
int netin_fd, netout_fd;

init_connection (hostname)
  char *hostname;
{
  struct hostent *host;
  struct sockaddr_in serv_addr;
  struct passwd *pw;
  char *user_name;
  FILE *fdopen();
  char line[MAXLINE];

  host = gethostbyname (hostname);
  if (host == NULL)
    {
      fprintf (stderr, "Unknown host %s\n", hostname);
      exit (1);
    }

  bzero ((char *)&serv_addr, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  bcopy (host->h_addr, (char *) &serv_addr.sin_addr,
	 host->h_length);
  serv_addr.sin_port  = htons (QXT_PORT);
  netin_fd = socket (AF_INET, SOCK_STREAM, 0);
  if (netin_fd < 0)
    {
      fprintf (stderr, "Can't get socket\n");
      exit (1);
    }

  if (connect (netin_fd, (struct sockaddr *) &serv_addr,
	       sizeof (serv_addr)) < 0)
    {
      close (netin_fd);
      fprintf (stderr, "Connection failed\n");
      exit (1);
    }

  netout_fd = dup (netin_fd);
  netin = fdopen (netin_fd, "r");
  netout = fdopen (netout_fd, "w");
  fgets (line, MAXLINE, netin);
  printf ("netin= %d, netout= %d; %s", netin_fd, netout_fd, line);


  pw = (struct passwd *) getpwuid (getuid ());
  if (pw == 0)
    user_name = "unknown";
  else
    user_name = pw->pw_name;

  fprintf (netout, "USER %s\n", user_name);
  fflush (netout);
  fgets (line, MAXLINE, netin);
  printf ("Qserver> %s", line);
}

close_connection ()
{
  fprintf (netout, "QUIT\n");
  fflush (netout);
}

/* Strip whitespace from the start and end of STRING. */
stripwhite (string)
     char *string;
{
  register int i = 0;

  while (whitespace (string[i]))
    i++;

  if (i)
    strcpy (string, string + i);

  i = strlen (string) - 1;

  while (i > 0 && whitespace (string[i]))
    i--;

  string[++i] = '\0';
}

/* Tell the GNU Readline library how to complete.  We want to try to complete
   on command names if this is the first word in the line, or on filenames
   if not. */
init_readline ()
{
  char **qshell_completion ();

  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "Quixote";

  /* Tell the completer that we want a crack first. */
  rl_attempted_completion_function = (Function *)qshell_completion;
}

/* Attempt to complete on the contents of TEXT.  START and END show the
   region of TEXT that contains the word to complete.  We can use the
   entire line in case we want to do some simple parsing.  Return the
   array of matches, or NULL if there aren't any. */
char **
qshell_completion (text, start, end)
     char *text;
     int start, end;
{
  char **matches;
  char *command_generator ();
  char *filename_completion_function ();

  matches = (char **)NULL;

  /* If this word is at the start of the line, then it is a command
     to complete.  Otherwise it is the name of a file in the current
     directory. */
  if (start == 0)
    matches = completion_matches (text, command_generator);

  return (matches);
}

/* Generator function for command completion.  STATE lets us know whether
   to start from scratch; without any state (i.e. STATE == 0), then we
   start at the top of the list. */
char *
command_generator (text, state)
     char *text;
     int state;
{
  static int list_index, len;
  char *name;

  /* If this is a new word to complete, initialize now.  This includes
     saving the length of TEXT for efficiency, and initializing the index
     variable to 0. */
  if (!state)
    {
      list_index = 0;
      len = strlen (text);
    }

  /* Return the next name which partially matches from the command list. */
  while (name = command_table[list_index].name)
    {
      list_index++;

      if (strncmp (name, text, len) == 0)
	return newstring (name);
    }

  /* If no names matched, then return NULL. */
  return (char *)NULL;
}
