/*
  Copyright (C) 1993 Institute for New Computer Technology

 */

/* Yacc grammer for Qshell
   Author: Yutaka Niibe <gniibe@mri.co.jp>

   Well, I guess you may say:
         Huh?  You need to define Qshell grammer by Yacc, don't you?
	 I couldn't believe.

	 Sorry, I just want to try it.  And you know, the extensibility.
	 Yeah, extensibility. :-)
 */
%{
#include <stdio.h>
#include "qxt.h"
#include "qshell.h"
#define YYDEBUG 1

static int lineno;

static WORD_LIST *new_word_list ();
static REDIRECT  *new_redirect ();
static COMMAND   *new_command ();
static void free_word_list ();
static void free_redirect ();
       void free_command ();
static void free_all ();

COMMAND *command;

extern void *xmalloc ();
extern void *xrealloc ();
extern char *newstring ();

extern int interactive;

static int yylex();
static void free_word_list_deep ();
static void free_redirect_deep ();


%}
%union {
  char *word;
  WORD_LIST *word_list;
  REDIRECT  *redirect;
  COMMAND   *command;
}

%token <word>      QUERY
%token <word>      WORD
%token             GREATER_GREATER
%type  <command>   line
%type  <word_list> word_list
%type  <redirect>  redirection redirections
%%
line: word_list '\n'
{
  WORD_LIST *wl;
  int argc;
  char **argv;
  int i;

  argc = 0;
  for (wl = $1; wl; wl = wl->next)
    argc++;

  argv = (char **) xmalloc (sizeof(char *) * (argc + 1));
  for (i=argc-1,wl = $1; i >= 0; i--,wl = wl->next)
    argv[i] = newstring (wl->word);
  argv[argc] = NULL;

  free_word_list_deep ($1);

  $$ = new_command (argc, argv, NULL);
  command = $$;
  lineno++;
  YYACCEPT
}
  | word_list redirections '\n'
{
  WORD_LIST *wl;
  int argc;
  char **argv;
  REDIRECT *rdir1, *rdir2;
  int i;

  argc = 0;
  for (wl = $1; wl; wl = wl->next)
    argc++;

  argv = (char **) xmalloc (sizeof(char *) * (argc + 1));

  for (i=argc-1,wl = $1; i >= 0; i--,wl = wl->next)
    argv[i] = newstring (wl->word);
  argv[argc] = NULL;

  free_word_list_deep ($1);

  /* reverse the order of $2 */
  rdir2 = NULL;
  for (rdir1 = $2; rdir1; rdir1 = rdir1->next)
    rdir2 = new_redirect (rdir1->mode, rdir1->file_name, rdir2);
  free_redirect_deep (rdir1);

  $$ = new_command (argc, argv, rdir2);
  command = $$;
  lineno++;
  YYACCEPT
}
  | QUERY '\n'
{ /* query -e "${word}" */
  int argc;
  char **argv;

  argc = 3;
  argv = (char **) xmalloc (sizeof(char *) * (argc + 1));
  argv[0] = newstring ("query");
  argv[1] = newstring ("-e");
  argv[2] = newstring ($1);
  argv[argc] = NULL;

  $$ = new_command (argc, argv, NULL);
  command = $$;
  lineno++;
  YYACCEPT
}
  | QUERY redirections '\n'
{ /* query -e "${word}" */
  int argc;
  char **argv;
  REDIRECT *rdir1, *rdir2;

  argc = 3;
  argv = (char **) xmalloc (sizeof(char *) * (argc + 1));
  argv[0] = newstring ("query");
  argv[1] = newstring ("-e");
  argv[2] = newstring ($1);
  argv[argc] = NULL;

  /* reverse the order of $2 */
  rdir2 = NULL;
  for (rdir1 = $2; rdir1; rdir1 = rdir1->next)
    rdir2 = new_redirect (rdir1->mode, rdir1->file_name, rdir2);
  free_redirect_deep (rdir1);

  $$ = new_command (argc, argv, rdir2);
  command = $$;
  lineno++;
  YYACCEPT
}
  | '\n'
{
  command = NULL;
  lineno++;
  YYACCEPT
}
  | error '\n'
{
  command = NULL;
  free_all ();
  lineno++;
  if (interactive)
    YYACCEPT
  else
    YYABORT
}
  ;

word_list: WORD
{
  $$ = new_word_list ($1, NULL);
}
  | word_list WORD
{
  $$ = new_word_list ($2, $1);
}
  ;

redirection: '>' WORD
{
  $$ = new_redirect (output, $2, NULL);
}
  | '<' WORD
{
  $$ = new_redirect (input, $2, NULL);
}
  | GREATER_GREATER WORD
{
  $$ = new_redirect (output_append, $2, NULL);
}
  ;

redirections: redirection
  | redirections redirection
{
  $2->next = $1;
  $$ = $2;
}
  ;

%%

char *token = (char *) NULL;
int token_buffer_size = 0;

/****************

           Y   Y   L   E  X

  Context            Regep            Action/definition

                     [0-9a-zA-Z_-@:]+ word

  beginning of line  \?.+\n           QUERY
  non interactive    #.*\n            skip like white space
                     {word}           WORD
                     .                the token is the character itself
  

 ****************/

static char *token_string;
static int token_length;
static int token_string_allocated;
#define YYLEX_TOKEN_LENGTH 16

init_yylex ()
{
  token_string = xmalloc (YYLEX_TOKEN_LENGTH);
  token_string_allocated = YYLEX_TOKEN_LENGTH;
  lineno = 1;
}

#define is_word(c) (c >= 'a' && c <= 'z' || \
		    c >= 'A' && c <= 'Z' || \
		    c >= '0' && c <= '9' || \
		    c == '_' || c == '-' || \
		    c == '.' || c == '/' || \
		    c == ',' || c == '+' || \
		    c == '@' || c == ':')

static char *yy_line;
static char *yy_line_end;

init_yylex_for_line (line)
     char *line;
{
  yy_line = line;
  yy_line_end = yy_line + strlen (yy_line);
}

static int see_newline = 1;

static yygetchar ()
{
  int c;

  if (yy_line > yy_line_end)
    return EOF;

  if (yy_line == yy_line_end)
    c = '\n';
  else
    c = *yy_line;

  yy_line++;

  return c;
}

static yyungetchar (c)
     int c;
{
  if (c != EOF)
    yy_line--;
}

static yylex ()
{
  int c;

  token_length = 0;

 loop:
  c = yygetchar ();

  if (c == EOF)
    {
      see_newline = 0;
      return EOF;
    }
  if (c == ' ' || c == '\t')
    /* skip white space */
    goto loop;

  if (see_newline && c == '?')
    { /* Query */
      /* skip to next newline */
      while (1)
	{
	  token_string[token_length] = c;
	  token_length++;
	  if (token_string_allocated <= token_length)
	    {
	      token_string_allocated *= 2;
	      token_string = xrealloc (token_string, token_string_allocated);
	    }
	  c = yygetchar ();
	  if (c == '\n' || c == EOF)
	    break;
	}
      yyungetchar (c);
      token_string[token_length] = 0;
      yylval.word = newstring (token_string);

      see_newline = 0;
      return QUERY;
    }

  if (c == '#' && !interactive)
    { /* skip to next newline */
      while (1)
	{
	  c = yygetchar ();
	  if (c == '\n')
	    {
	      see_newline = 1;
	      return '\n';
	    }
	  else if (c == EOF)
	    {
	      see_newline = 0;
	      return EOF;
	    }
	}
    }

  if (c == '\"')
    {
      c = yygetchar ();

      while (1)
	{
	  if (c == '\"')
	    {
	      c = yygetchar ();
	      break;
	    }
	  else if (c == '\n')
	    {
	      fprintf (stderr, "String termnated by newsline.\n");
	      break;
	    }
	  token_string[token_length] = c;
	  token_length++;
	  if (token_string_allocated <= token_length)
	    {
	      token_string_allocated *= 2;
	      token_string = xrealloc (token_string, token_string_allocated);
	    }
	  c = yygetchar ();
	}
      yyungetchar (c);
      token_string[token_length] = 0;
      yylval.word = newstring (token_string);

      see_newline = 0;
      return WORD;
    }
  else if (is_word(c))
    {
      while (1)
	{
	  token_string[token_length] = c;
	  token_length++;
	  if (token_string_allocated <= token_length)
	    {
	      token_string_allocated *= 2;
	      token_string = xrealloc (token_string, token_string_allocated);
	    }
	  c = yygetchar ();
	  if (!is_word(c))
	    break;
	}
      yyungetchar (c);
      token_string[token_length] = 0;
      yylval.word = newstring (token_string);

      see_newline = 0;
      return WORD;
    }

  if (c == '>')
    {
      see_newline = 0;
      c = yygetchar ();
      if (c == '>')
	return GREATER_GREATER;
      else
	{
	  yyungetchar (c);
	  return '>';
	}
    }

  if (c == '\n')
    see_newline = 1;
  else
    see_newline = 0;

  return c;
}

static ANY head_content;
static ANY *head = &head_content;

static init_alloc ()
{
  head_content.prev_entry = NULL;
  head_content.next_entry = NULL;
}

static WORD_LIST *new_word_list (word, next)
     char *word;
     WORD_LIST *next;
{
  WORD_LIST *x;

  x = xmalloc (sizeof (WORD_LIST));
  x->type = TYPE_WORD_LIST;
  x->prev_entry = head;
  x->next_entry = head->next_entry;
  if (head->next_entry)
    head->next_entry->prev_entry = (ANY *)x;
  head->next_entry = (ANY *)x;

  x->word = word;
  x->next = next;

  return x;
}

static REDIRECT *new_redirect (mode, file_name, next)
     input_mode mode;
     char *file_name;
     REDIRECT * next;
{
  REDIRECT *x;

  x = xmalloc (sizeof (REDIRECT));
  x->type = TYPE_REDIRECT;
  x->prev_entry = head;
  x->next_entry = head->next_entry;
  if (head->next_entry)
    head->next_entry->prev_entry = (ANY *)x;
  head->next_entry = (ANY *)x;

  x->mode = mode;
  x->file_name = file_name;
  x->next = next;

  return x;
}

static COMMAND *new_command (argc, argv, redirect)
     int argc;
     char **argv;
     REDIRECT *redirect;
{
  COMMAND *x;

  x = xmalloc (sizeof (COMMAND));
  x->type = TYPE_COMMAND;
  x->prev_entry = head;
  x->next_entry = head->next_entry;
  if (head->next_entry)
    head->next_entry->prev_entry = (ANY *)x;
  head->next_entry = (ANY *)x;

  x->argc = argc;
  x->argv = argv;
  x->redirect = redirect;

  return x;
}

static void free_word_list(x)
     WORD_LIST *x;
{
  x->prev_entry->next_entry = x->next_entry;
  if (x->next_entry)
    x->next_entry->prev_entry = x->prev_entry;
  free (x->word);
  free (x);
}

static void free_word_list_deep (x)
     WORD_LIST *x;
{
  WORD_LIST *y;

  while (x)
    {
      y = x->next;
      free_word_list (x);
      x = y;
    }
}

static void free_redirect (x)
     REDIRECT *x;
{
  x->prev_entry->next_entry = x->next_entry;
  if (x->next_entry)
    x->next_entry->prev_entry = x->prev_entry;
  free (x->file_name);
  free (x);
}

static void free_redirect_deep (x)
     REDIRECT *x;
{
  REDIRECT *y;

  while (x)
    {
      y = x->next;
      free_redirect (x);
      x = y;
    }
}

void free_command (x)
     COMMAND *x;
{
  char **s;
  int i;

  x->prev_entry->next_entry = x->next_entry;
  if (x->next_entry)
    x->next_entry->prev_entry = x->prev_entry;
  for (s=x->argv; *s; s++)
    free (*s);
  free (x->argv);
  free (x);
}

void free_command_deep (x)
     COMMAND *x;
{
  free_redirect_deep (x->redirect);
  free_command (x);
}

static void free_all ()
{
  ANY *x;

  while (x = head->next_entry)
    {
      switch (x->type)
	{
	case TYPE_WORD_LIST:
	  free_word_list_deep ((WORD_LIST *)x);
	  break;
	case TYPE_REDIRECT:
	  free_redirect_deep ((REDIRECT *)x);
	  break;
	case TYPE_COMMAND:
	  free_command_deep ((COMMAND *)x);
	  break;
	default:
	  abort ();
	}
    }
}

yyerror (message)
     char *message;
{
  if (interactive)
    fprintf (stderr, "Qshell: %s\n", message);
  else
    fprintf (stderr, "Qshell: %s at line %d\n", message, lineno);
}
